////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Delphi Project Editor.                                                //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231208 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_delphiprojecteditor; // Form Delphi Project Editor

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_ListBoxSelection,
 Form_TextEditor, Form_CurveWindow,
 Form_EditDataAnalysisPluginArguments,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_syscal,
 _crw_crwapi, _crw_crwapiserver, _crw_dcc32, 
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormDelphiProjectEditor = class(TFormTextEditor)
    ActionEditCompile: TAction;
    ActionEditRun: TAction;
    ActionEditApiHelp: TAction;
    MenuEditCompile: TMenuItem;
    MenuEditRun: TMenuItem;
    MenuEditApiHelp: TMenuItem;
    MenuEditDeleteSeparator: TMenuItem;
    ToolButtonEditCompileSeparator: TToolButton;
    ToolButtonEditCompile: TToolButton;
    ToolButtonEditRun: TToolButton;
    ToolButtonEditApiHelp: TToolButton;
    PopupMenuEditCompile: TMenuItem;
    PopupMenuEditRun: TMenuItem;
    PopupMenuEditApiHelp: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure ActionEditCompileExecute(Sender: TObject);
    procedure ActionEditRunExecute(Sender: TObject);
    procedure ActionEditApiHelpExecute(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

function  NewDelphiProjectEditor(aProject:LongString):TFormDelphiProjectEditor;
function  FindDelphiProjectEditor(FileName:LongString;
                                  CreateNew:Boolean=false;
                                  BringToFront:Boolean=false):TFormDelphiProjectEditor;
procedure Kill(var TheObject:TFormDelphiProjectEditor); overload;

implementation

{$R *.lfm}

function NewDelphiProjectEditor(aProject:LongString):TFormDelphiProjectEditor;
begin
 Result:=nil;
 try
  aProject:=UnifyFileAlias(aProject,ua_RealPath);
  if FileExists(aProject) then begin
   Application.CreateForm(TFormDelphiProjectEditor, Result);
   if Result.Ok then begin
    Result.Open(aProject);
    Result.DefaultReadOnly;
   end;
  end else Echo(Format(RusEng('Не найден проект: "%s".',
                              'Not found project: "%s".'),[aProject]));
 except
  on E:Exception do BugReport(E,nil,'NewDelphiProjectEditor');
 end;
end;

type
 TFindRec = packed record
  Match    : TFormDelphiProjectEditor;
  PathName : LongString;
 end;

procedure FindForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 with TFindRec(Custom^) do
 if Form is TFormDelphiProjectEditor then
 if SameText(UnifyAlias((Form as TFormDelphiProjectEditor).PathName),UnifyAlias(PathName)) then begin
  Match:=(Form as TFormDelphiProjectEditor);
  Terminate:=true;
 end;
end;

function FindDelphiProjectEditor(FileName:LongString;
                                 CreateNew:Boolean=false;
                                 BringToFront:Boolean=false):TFormDelphiProjectEditor;
var FindRec:TFindRec;
begin
 Result:=nil;
 try
  FindRec:=Default(TFindRec);
  try
   FileName:=UnifyFileAlias(FileName);
   FindRec.Match:=nil;
   FindRec.PathName:=StringBuffer(FileName);
   SdiMan.ForEachChild(FindForm,@FindRec);
   Result:=FindRec.Match;
   if CreateNew and not Assigned(Result) then Result:=NewDelphiProjectEditor(FileName);
   if Assigned(Result) and BringToFront then begin
    Result.Show;
    Result.WindowState:=wsNormal;
    SdiMan.ActivateChild(Result);
   end;
  finally
   FindRec.PathName:='';
  end;
 except
  on E:Exception do BugReport(E,nil,'FindDelphiProjectEditor');
 end;
end;

procedure Kill(var TheObject:TFormDelphiProjectEditor); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

procedure TFormDelphiProjectEditor.FormCreate(Sender: TObject);
begin
 inherited;
 UpdateMenu(MenuEditCompile,
            RusEng('Компиляция','Compile'),
            RusEng('Компилировать Delphi программу.','Compile Delphi program source.'),
            ShortCut(VK_F9,[ssCtrl]));
 UpdateMenu(MenuEditRun,
            RusEng('Выполнение','Run'),
            RusEng('Выполнить Delphi программу.','Execute Delphi program.'),
            0);
 UpdateMenu(MenuEditApiHelp,
            RusEng('Справка по API','API help'),
            RusEng('Вызов справки по API.','Call API help window.'),
            ShortCut(VK_F1, [ssCtrl]));
end;

procedure TFormDelphiProjectEditor.ActionEditCompileExecute(Sender:TObject);
var s:LongString; m:TParseDcc32Message;
begin
 if Guard.CheckAction(ga_Root,ActionEditCompile)<0 then Exit;
 inherited;
 try
  if PerformModified then FileSave;
  if not SafeCompileDelphiProject(PathName) then begin
   if Dcc32.Ok then begin
    if (Dcc32.Fatals.Count>0)   then s:=Dcc32.Fatals[0]   else
    if (Dcc32.Errors.Count>0)   then s:=Dcc32.Errors[0]   else
    if (Dcc32.Warnings.Count>0) then s:=Dcc32.Warnings[0] else
    if (Dcc32.Hints.Count>0)    then s:=Dcc32.Hints[0]    else s:='';
    if (Length(s)>0) then begin
     m:=ParseDcc32Message(s);
     if (m.Status<>cmsNone) then GoToRowCol(m.LineNum,1);
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditCompileExecute');
 end;
end;

procedure TFormDelphiProjectEditor.ActionEditRunExecute(Sender: TObject);
var CrwApi:TCrwApiServer; WinStat:TWindowState;
var Argums,DllPath,ExePath:LongString;
var aName,aSource,aBinary:LongString;
begin
 if Guard.CheckAction(ga_User,ActionEditRun)<0 then Exit;
 inherited;
 Argums:='';
 WinStat:=WindowState;
 WindowState:=wsMinimized;
 try
  DllPath:=ForceExtension(PathName,'.dll');
  ExePath:=ForceExtension(PathName,'.exe');
  DllPath:=UnifyFileAlias(AdaptDllFileName(DllPath),ua_RealPath);
  ExePath:=UnifyFileAlias(AdaptExeFileName(ExePath),ua_RealPath);
  if PerformModified or not (FileExists(DllPath) or FileExists(ExePath))
  then ActionEditCompileExecute(Sender);
  if not ActiveCurveWindow.CheckIsDataProtected then begin
   if FileExists(DllPath) then begin
    aSource:=PathName; aBinary:=DllPath;
    aName:=LowerCase(ExtractBaseName(aSource));
    if ExecuteFormEditDataAnalysisArguments(Self,Argums)=mrOk then begin
     CrwApi:=NewCrwApiServer(ForDataAnalysis,aName,aSource,aBinary);
     CrwApi.ExecuteDataAnalysis(DllPath,ActiveCurveWindow,Argums);
     CrwApi.Free;
    end;
   end else
   if FileExists(ExePath) then begin
    SendToMainConsole(Format('@silent @term /ca task "%s"'+EOL,[ExePath]));
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditRunExecute');
 end;
 WindowState:=WinStat;
 Argums:='';
end;

procedure AddFileToList(const FileName:LongString; const FileDetails:TSearchRec;
                        SubDirLevel:Integer; var Terminate:Boolean; CustomData:Pointer);
begin
 if FileDetails.Attr and ({faVolumeID+}faDirectory)=0 then
 with TText(CustomData) do Addln(MakeRelativePath(FileName,Line[0]));
end;

procedure TFormDelphiProjectEditor.ActionEditApiHelpExecute(Sender: TObject);
var List:TText; Path:LongString; Win:TFormTextEditor; i:Integer;
begin
 if Guard.CheckAction(ga_Guest,ActionEditApiHelp)<0 then Exit;
 inherited;
 try
  Path:='';
  if ReadIniFilePath(SysIniFile,'[Dcc32.Exe]','CrwApiPath',HomeDir,Path) then
  if DirExists(AddBackSlash(Path)) then begin
   List:=NewText;
   try
    List.Addln(AddBackSlash(Path)+'*');
    ForEachFile(AddBackSlash(Path),'*.pas;*.asm;*.inc',AddFileToList,10,List);
    List.DelLn(0);
    if (List.Count>0) then List.Text:=SortTextLines(List.Text);
    if (List.Count>0) then begin
     for i:=0 to List.Count-1 do if IsSameText(List[i],'_crwapi.pas') then Break;
     if (i=List.Count) then i:=0;
     if (ListBoxSelection('CRW-DAQ API units','Unit list:',List.Text,i)=mrOk) then begin
      Path:=UnifyFileAlias(AddBackSlash(Path)+List[i],ua_RealPath);
      Win:=FindTextEditor(Path,true,true);
      if Win.Ok then begin
       Win.IsReadOnly:=true;
       Win.ActionEditReadOnly.Enabled:=false;
      end;
     end;
    end;
   finally
    Kill(List);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditApiHelpExecute');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_delphiprojecteditor;
begin
end;

procedure Free_form_delphiprojecteditor;
begin
end;

initialization

 Init_form_delphiprojecteditor;

finalization

 Free_form_delphiprojecteditor;

end.

//////////////
// END OF FILE
//////////////

