////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Unit Keyboard Layout.                                                      //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231208 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit unit_keyboardlayout; // Unit Keyboard Layout

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 lcltype, lclintf, LMessages,
 _crw_alloc, _crw_rtc, _crw_str;

{$IFnDEF WINDOWS}
type HKL=HANDLE; const KL_NAMELENGTH=9;
{$ENDIF ~WINDOWS}

 //
 // Windows may switch language on CTRL+SHIFT, ALT+SHIFT, LSHIFT/RSHIFT etc.
 // I like always use CTRL+SHIFT for language switch in my applications.
 // To do so:
 // call KeyBoardLayoutOnKeyDown in Application.OnHandle
 // call KeyBoardLayoutCheck     on timer to control KeyBoard layout state
 // call KeyBoardLayoutExt       on timer to show current KeyBoard layout state
 //

function  KeyBoardLayoutHKL:HKL;                    // Current KeyBoard layout handle
function  KeyBoardLayoutLCID:LCID;                  // Current KeyBoard layout locale ID
function  KeyBoardLayoutID:LongString;              // Current KeyBoard layout locale name like '$00000419'
function  KeyBoardLayoutExt:LongString;             // Current KeyBoard layout locale Ext like 'RUS', 'ENG', etc.
function  KeyBoardLayoutName:LongString;            // Current KeyBoard layout locale name like 'English (United States)'
procedure KeyBoardLayoutCheck;                      // To check KeyBoard layout on timer
procedure KeyBoardLayoutOnKeyDown(const Msg:TMsg);  // To switch KeyBoard layout on CTRL+SHIFT

implementation

{$IFDEF WINDOWS}

var
 myFixed : HKL = 0;
 myCount : Integer = 0;
 myItems : packed array[0..63] of HKL;

procedure InitKeyboardLayouts;
begin
 myFixed:=GetKeyBoardLayout(0);
 myCount:=GetKeyBoardLayoutList(High(myItems)+1,myItems);
end;

function KeyBoardLayoutHKL:HKL;
begin
 Result:=GetKeyBoardLayout(0);
end;

function KeyBoardLayoutLCID:LCID;
var i:Integer; buf:TParsingBuffer; // KL_NAMELENGTH at least
begin
 Result:=0;
 try
  buf[0]:='$';
  if GetKeyBoardLayoutName(@buf[1]) and atoi(@buf[0],i) then Result:=i;
 except
  on E:Exception do BugReport(E,nil,'KeyBoardLayoutLCID');
 end;
end;

function KeyBoardLayoutID:LongString;
var buf:TParsingBuffer;
begin
 Result:='';
 try
  if GetKeyBoardLayoutName(buf) then Result:=Format('$%s',[buf]);
 except
  on E:Exception do BugReport(E,nil,'KeyBoardLayoutID');
 end;
end;

 //
 // This function uses to minimize IndexOf(..) search.
 // Usually KeyBoardLayoutLCID is constant, so we may keep previouse language
 // index value to avoid IndexOf call. This decrease CPU load.
 //
function LanguageIndex(L:LCID):Integer;
const i:Integer=0;
begin
 {$IFDEF SKIP_DRAFT}
 with Languages do
 if not InRange(i,0,Count-1) or (LocaleID[i]<>L) then i:=IndexOf(L);
 {$ENDIF ~SKIP_DRAFT}
 Result:=i;
end;

function KeyBoardLayoutExt:LongString;
var i:Integer;
begin
 Result:='';
 try
  {$IFDEF SKIP_DRAFT}
  i:=LanguageIndex(KeyBoardLayoutLCID);
  if (i>=0) then Result:=Languages.Ext[i];
  {$ENDIF ~SKIP_DRAFT}
 except
  on E:Exception do BugReport(E,nil,'KeyBoardLayoutExt');
 end;
end;

function KeyBoardLayoutName:LongString;
var i:Integer;
begin
 Result:='';
 try
  {$IFDEF SKIP_DRAFT}
  i:=LanguageIndex(KeyBoardLayoutLCID);
  if (i>=0) then Result:=Languages.Name[i];
  {$ENDIF ~SKIP_DRAFT}
 except
  on E:Exception do BugReport(E,nil,'KeyBoardLayoutName');
 end;
end;

procedure KeyBoardLayoutCheck;
var i:Integer;
begin
 try
  if (myFixed<>GetKeyBoardLayout(0)) then begin
   myCount:=GetKeyBoardLayoutList(High(myItems)+1,myItems);
   for i:=0 to myCount-1 do
   if (myFixed=myItems[i]) then begin
    ActivateKeyBoardLayout(myFixed, KLF_ACTIVATE);
    Break;
   end;
   myFixed:=GetKeyBoardLayout(0);
  end;
 except
  on E:Exception do BugReport(E,nil,'KeyBoardLayoutCheck');
 end;
end;

procedure KeyBoardLayoutOnKeyDown(const Msg:TMsg);
var i,j:Integer;
begin
 if (Msg.Message=WM_KEYDOWN) then
 if (Word(Msg.WParam)=VK_SHIFT) or (Word(Msg.WParam)=VK_CONTROL) then
 try
  if ((Word(Msg.WParam)=VK_SHIFT)   and (GetKeyState(VK_CONTROL) < 0))
  or ((Word(Msg.WParam)=VK_CONTROL) and (GetKeyState(VK_SHIFT)   < 0)) then begin
   myCount:=GetKeyBoardLayoutList(High(myItems)+1,myItems);
   j:=-1;
   for i:=0 to myCount-1 do
   if (myFixed=myItems[i]) then begin
    j:=i;
    Break;
   end;
   if (j<0) then myFixed:=GetKeyBoardLayout(0) else begin
    j:=j+1;
    if (j>=myCount) then j:=0;
    myFixed:=myItems[j];
   end;
   //KeyBoardLayoutCheck; // It's better to call on timer
  end;
 except
  on E:Exception do BugReport(E,nil,'KeyBoardLayoutOnKeyDown');
 end;
end;
{$ENDIF ~WINDOWS}

{$IFDEF SKIP_DRAFT}
// TODO: keyboard layout for UNIX
{$ENDIF ~SKIP_DRAFT}

{$IFDEF UNIX}
procedure InitKeyboardLayouts;
begin
end;

function KeyBoardLayoutHKL:HKL;
begin
 Result:=0;
end;

function KeyBoardLayoutLCID:LCID;
begin
 Result:=0;
end;

function KeyBoardLayoutID:LongString;
begin
 Result:='';
end;

function LanguageIndex(L:LCID):Integer;
begin
 Result:=0;
end;

function KeyBoardLayoutExt:LongString;
begin
 Result:='';
end;

function KeyBoardLayoutName:LongString;
begin
 Result:='';
end;

procedure KeyBoardLayoutCheck;
begin
end;

procedure KeyBoardLayoutOnKeyDown(const Msg:TMsg);
begin
end;
{$ENDIF ~UNIX}

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_unit_keyboardlayout;
begin
 InitKeyboardLayouts;
end;

procedure Free_unit_keyboardlayout;
begin
end;

initialization

 Init_unit_keyboardlayout;

finalization

 Free_unit_keyboardlayout;

end.

//////////////
// END OF FILE
//////////////

