////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// List of CRW32 Windows messages.                                            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20040108 - Creation, deffered callback request support                     //
// 20040119 - RegisterCrwDaqDeferredCallback                                  //
// 20231202 - Modified for FPC (A.K.)                                         //
// 20250129 - Use TAtomicCounter                                              //
////////////////////////////////////////////////////////////////////////////////

unit unit_crwdaqmessages; // Unit CRW-DAQ Messages

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, LMessages,
 lcltype, lclintf,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_appforms, _crw_apptools, _crw_apputils;

 //
 // WM_CrwDaqUser is CRW32 user messages base.
 // All CrwDaq user messages defined relative to this base.
 //
const
 WM_CrwDaqUser = WM_User + $4000;
 LM_CrwDaqUser = WM_CrwDaqUser;

 //
 // WM_CrwDaqDeferredCallbackRequest - message to do deffered procedure call.
 // Some procedures need to be execute in main VCL thread. You may post this
 // message from any thread to do something in main VCL thread.
 // WParam = Callback:TCrwDaqDeferredCallbackRequestProc
 // LParam = Sender:TObject
 // Destination window must have message handler, something like
 // procedure HandleDCR(var msg:TMessage); message WM_CrwDaqDeferredCallbackRequest;
 // begin
 //  HandleCrwDaqDeferredCallbackRequest(msg);
 // end;
 //
const
 WM_CrwDaqDeferredCallbackRequest = WM_CrwDaqUser + 0;

 //
 // This procedure type uses for deferred callback request.
 //
type
 TCrwDaqDeferredCallbackRequestProc = procedure(Sender:TObject);

 //
 // This procedure uses to post WM_CrwDaqDeferredCallbackRequest message.
 //
function  PostCrwDaqDeferredCallbackRequest(hWnd:HWND; Callback:TCrwDaqDeferredCallbackRequestProc; Sender:TObject):Boolean;

 //
 // This procedure uses to handle WM_CrwDaqDeferredCallbackRequest message.
 //
procedure HandleCrwDaqDeferredCallbackRequest(var Msg:TLMessage);

 //
 // For safety reason, handle only registered callbacks.
 //
procedure RegisterCrwDaqDeferredCallback(Callback:TCrwDaqDeferredCallbackRequestProc);

var  // Crwdaq statistics on Deferred Procedure Callback messages
 CrwdaqStatDPC : record
  PostCount : TAtomicCounter; // Count DPC Post requests
  HandCount : TAtomicCounter; // Count DPC handled messages
 end = (
  PostCount:nil;
  HandCount:nil
 );

implementation

const DcpList : TObjectStorage = nil; // Deferred callback procedure list

procedure InitDpcCounters;
begin
 LockedInit(CrwdaqStatDPC.PostCount);
 LockedInit(CrwdaqStatDPC.HandCount);
end;

procedure FreeDpcCounters;
begin
 LockedFree(CrwdaqStatDPC.PostCount);
 LockedFree(CrwdaqStatDPC.HandCount);
end;

function PostCrwDaqDeferredCallbackRequest(hWnd:HWND; Callback:TCrwDaqDeferredCallbackRequestProc; Sender:TObject):Boolean;
begin
 if (hWnd<>0) and (hWnd<>LclType.INVALID_HANDLE_VALUE) and Assigned(Callback) and Assigned(Sender)
 then Result:=PostMessage(hWnd,WM_CrwDaqDeferredCallbackRequest,PointerToPtrInt(@CallBack),PointerToPtrInt(Sender))
 else Result:=False;
 if Result then LockedInc(CrwdaqStatDPC.PostCount);
end;

procedure HandleCrwDaqDeferredCallbackRequest(var Msg:TLMessage);
var Callback:TCrwDaqDeferredCallbackRequestProc; Sender:TObject;
begin
 try
  Sender:=TObject(PtrIntToPointer(Msg.LParam));
  Callback:=TCrwDaqDeferredCallbackRequestProc(PtrIntToPointer(Msg.WParam));
  if Assigned(Sender) and Assigned(CallBack) then
  if (DcpList.IndexOf(TObject(@Callback))>=0) then begin
   LockedInc(CrwdaqStatDPC.HandCount);
   Callback(Sender);
  end;
 except
  on E:Exception do BugReport(E,nil,'HandleCrwDaqDeferredCallbackRequest');
 end;
end;

procedure RegisterCrwDaqDeferredCallback(Callback:TCrwDaqDeferredCallbackRequestProc);
begin
 if (DcpList.IndexOf(TObject(@Callback))<0) then DcpList.Add(TObject(@Callback));
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_unit_crwdaqmessages;
begin
 InitDpcCounters;
 DcpList:=NewObjectStorage(false);
 DcpList.Master:=@DcpList;
end;

procedure Free_unit_crwdaqmessages;
begin
 Kill(DcpList);
 FreeDpcCounters;
end;

initialization

 Init_unit_crwdaqmessages;

finalization

 Free_unit_crwdaqmessages;

end.

//////////////
// END OF FILE
//////////////

