////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ Device Control.                                                   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231202 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqdevicecontrol; // Form DAQ Device Control

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_ListBoxSelection,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_daqsys, _crw_daqdev, _crw_curves, _crw_calib,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormDaqDeviceControl }

  TFormDaqDeviceControl = class(TMasterForm)
    ActionListButtons: TActionList;
    ActionButtonProperty: TAction;
    ActionButtonCommonDev: TAction;
    ActionButtonCalibration: TAction;
    ActionButtonCommonDaq: TAction;
    ActionButtonProblems: TAction;
    ActionButtonErrorView: TAction;
    ActionButtonErrorClear: TAction;
    ActionButtonTagList: TAction;
    ButtonCalibration: TButton;
    ButtonCommonDaq: TButton;
    ButtonCommonDev: TButton;
    ButtonErrorClear: TButton;
    ButtonErrorView: TButton;
    ButtonProblems: TButton;
    ButtonProperty: TButton;
    ButtonTagList: TButton;
    LabelDeviceList: TLabel;
    ListBoxDeviceList: TListBox;
    PanelButtons: TPanel;
    PanelDeviceList: TPanel;
    PanelDeviceProperty: TPanel;
    LabelDeviceProperty: TLabel;
    MemoDeviceProperty: TMemo;
    PanelDeviceTop: TPanel;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormDeactivate(Sender: TObject);
    procedure ListBoxDeviceListClick(Sender: TObject);
    procedure ListBoxDeviceListDblClick(Sender: TObject);
    procedure LabelDeviceListClick(Sender: TObject);
    procedure LabelDevicePropertyClick(Sender: TObject);
    procedure ActionButtonPropertyExecute(Sender: TObject);
    procedure ActionButtonCommonDevExecute(Sender: TObject);
    procedure ActionButtonCalibrationExecute(Sender: TObject);
    procedure ActionButtonCommonDaqExecute(Sender: TObject);
    procedure ActionButtonProblemsExecute(Sender: TObject);
    procedure ActionButtonErrorViewExecute(Sender: TObject);
    procedure ActionButtonErrorClearExecute(Sender: TObject);
    procedure ActionButtonTagListExecute(Sender: TObject);
  private
    { Private declarations }
    myTimer : TIntervalTimer;
    function  GetCurrentDevice:TDaqDevice;
  public
    { Public declarations }
    property  CurrentDevice:TDaqDevice read GetCurrentDevice;
    procedure UpdateControls(Flags:Cardinal);
    procedure Idle;
  end;

procedure OpenFormDaqDeviceControl(const Params:LongString='');
procedure HideFormDaqDeviceControl;
procedure IdleFormDaqDeviceControl;
procedure KillFormDaqDeviceControl;
procedure UpdateFormDaqDeviceControl(Flags:Cardinal);

implementation

{$R *.lfm}

const
  FormDaqDeviceControl : TFormDaqDeviceControl = nil;

procedure OpenFormDaqDeviceControl;
begin
 try
  if not Assigned(FormDaqDeviceControl) then begin
   Application.CreateForm(TFormDaqDeviceControl, FormDaqDeviceControl);
   FormDaqDeviceControl.Master:=@FormDaqDeviceControl;
  end;
  if FormDaqDeviceControl.Ok then with FormDaqDeviceControl do begin
   if IsNonEmptyStr(Params) then ApplyParams(Params);
   UpdateControls(1+2+4);
   Show;
   WindowState:=wsNormal;
   BringToFront;
  end;
 except
  on E:Exception do BugReport(E,nil,'OpenFormDaqDeviceControl');
 end;
end;

procedure HideFormDaqDeviceControl;
begin
 if FormDaqDeviceControl.Ok then FormDaqDeviceControl.Hide;
end;

procedure IdleFormDaqDeviceControl;
begin
 if FormDaqDeviceControl.Ok then FormDaqDeviceControl.Idle;
end;

procedure KillFormDaqDeviceControl;
begin
 Kill(TObject(FormDaqDeviceControl));
end;

procedure UpdateFormDaqDeviceControl(Flags:Cardinal);
begin
 if FormDaqDeviceControl.Ok then FormDaqDeviceControl.UpdateControls(Flags);
end;

function TFormDaqDeviceControl.GetCurrentDevice:TDaqDevice;
var aIndex:Integer; aName:LongString;
begin
 Result:=nil;
 if Ok then
 try
  aIndex:=ListBoxDeviceList.ItemIndex;
  if InRange(aIndex,0,ListBoxDeviceList.Items.Count-1) then begin
   aName:=ExtractWord(1, ListBoxDeviceList.Items[aIndex], ScanSpaces);
   Result:=FullDaqDeviceList.Find(aName);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetCurrentDevice');
 end;
end;

procedure TFormDaqDeviceControl.UpdateControls(Flags:Cardinal);
var aList:TText; aDevice:TDaqDevice; aTotErr,aDevErr:Integer;
var aName,aStatus:LongString;
begin
 if Ok then
 try
  if HasFlags(Flags,1) then begin
   aList:=NewText;
   try
    FullDaqDeviceList.GetDescriptionList(aList);
    SmartUpdate(ListBoxDeviceList, aList.Text);
   finally
    Kill(aList);
   end;
  end;
  if HasFlags(Flags,2+4) then begin
   aDevice:=CurrentDevice;
   if HasFlags(Flags,2) then begin
    if aDevice.Ok then begin
     aList:=NewText;
     try
      aDevice.GetProperty(aList);
      if SameText(aList[0],'['+Name+']') then aList.Delln(0);
      SmartUpdate(MemoDeviceProperty, aList.Text);
     finally
      Kill(aList);
     end;
    end else begin
     SmartUpdate(MemoDeviceProperty, '');
    end;
   end;
   if HasFlags(Flags,4) then begin
    aTotErr:=FullDaqDeviceList.ErrorsTotal;
    if Daq.Ok and Daq.AcqTimer.IsStart
    then aStatus:=RusEng('набор','started')
    else aStatus:=RusEng('останов','stopped');
    SmartUpdate(LabelDeviceList, Format(RusEng('[%s] Статус:%s Ошибок:%d',
                                               '[%s] Status:%s Errors:%d'),
                                               ['DeviceList', aStatus, aTotErr]));
    if aDevice.Ok then begin
     aName:=aDevice.Name;
     aDevErr:=aDevice.ErrorsTotal;
     if aDevice.InquiryTimer.IsStart
     then aStatus:=RusEng('набор','started')
     else aStatus:=RusEng('останов','stopped');
    end else begin
     aName:='';
     aDevErr:=0;
     aStatus:='';
    end;
    SmartUpdate(LabelDeviceProperty,Format(RusEng('[%s] Статус:%s Ошибок:%d',
                                                  '[%s] Status:%s Errors:%d'),
                                                  [aName, aStatus, aDevErr]));
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateControls');
 end;
end;

procedure TFormDaqDeviceControl.Idle;
var i:Integer; TheCurrentDevice:TDaqDevice;
 procedure Handle(Device:TDaqDevice);
 var HaveToUpdate:Boolean;
 begin
  if Device.Ok then begin
   HaveToUpdate:=false;
   if Device.PropertyDialog.Ok and Device.UpdatePropertyDialog
   then HaveToUpdate:=true;
   if Device.CommonPropertyDialog.Ok and Device.UpdateCommonPropertyDialog
   then HaveToUpdate:=true;
   if Device<>TheCurrentDevice then HaveToUpdate:=false;
   if HaveToUpdate then UpdateControls(2+4);
  end;
 end;
begin
 if Ok then
 try
  if myTimer.Event then begin
   UpdateControls(4);
   TheCurrentDevice:=CurrentDevice;
   for i:=0 to FullDaqDeviceList.Count-1 do Handle(FullDaqDeviceList[i]);
  end;
 except
  on E:Exception do BugReport(E,Self,'Idle');
 end;
end;

procedure TFormDaqDeviceControl.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 LocateFormToCenterOfScreen(Self);
 Caption:=RusEng('DAQ - УСТРОЙСТВА','DAQ - DEVICES');
 ActionButtonProperty.Caption:=RusEng('Свойства','Property');
 ActionButtonCommonDev.Caption:=RusEng('Общие','Common dev');
 ActionButtonCalibration.Caption:=RusEng('Калибровки','Calibration');
 ActionButtonCommonDaq.Caption:=RusEng('Общие DAQ','Common DAQ');
 ActionButtonProblems.Caption:=RusEng('Проблемы','Warnings');
 ActionButtonErrorView.Caption:=RusEng('Ошибки','Errors');
 ActionButtonErrorClear.Caption:=RusEng('Очистить','Clear');
 ActionButtonTagList.Caption:=RusEng('ТЕГИ','TAGS');
 myTimer:=NewIntervalTimer(tmStart+tmCyclic,NewIntervalMs(250,1,nil));
end;

procedure TFormDaqDeviceControl.FormDestroy(Sender: TObject);
begin
 Kill(myTimer);
end;

procedure TFormDaqDeviceControl.FormClose(Sender: TObject; var Action: TCloseAction);
begin
 Action:=caHide;
end;

procedure TFormDaqDeviceControl.FormDeactivate(Sender: TObject);
begin
 if Ok then
 try
  Hide;
 except
  on E:Exception do BugReport(E,Self,'FormDeactivate');
 end;
end;

procedure TFormDaqDeviceControl.ListBoxDeviceListClick(Sender: TObject);
begin
 UpdateControls(2+4);
end;

procedure TFormDaqDeviceControl.ListBoxDeviceListDblClick(Sender: TObject);
begin
 ActionButtonProperty.Execute;
end;

procedure TFormDaqDeviceControl.LabelDeviceListClick(Sender: TObject);
begin
 SmartFocus(ListBoxDeviceList);
end;

procedure TFormDaqDeviceControl.LabelDevicePropertyClick(Sender: TObject);
begin
 SmartFocus(MemoDeviceProperty);
end;

procedure TFormDaqDeviceControl.ActionButtonPropertyExecute(Sender: TObject);
var aDevice:TDaqDevice;
begin
 if Guard.CheckAction(ga_Root,ActionButtonProperty)<0 then Exit;
 if Ok then
 try
  aDevice:=CurrentDevice;
  if aDevice.Ok then begin
   aDevice.OpenPropertyDialog;
   if aDevice.PropertyDialog=nil
   then Warning(RusEng('Устройство ','Device ')+aDevice.Name+
                RusEng(' не имеет интерактивной настройки свойств.',' has no property editor.')+EOL+
                RusEng('Для изменения свойств отредактируйте и перегрузите файл конфигурации.',
                       'To change properties, edit and reload configuration file.'));
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonPropertyExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonCommonDevExecute(Sender: TObject);
var aDevice:TDaqDevice;
begin
 if Guard.CheckAction(ga_Root,ActionButtonCommonDev)<0 then Exit;
 if Ok then
 try
  aDevice:=CurrentDevice;
  if aDevice.Ok then aDevice.OpenCommonPropertyDialog;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonCommonDevExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonCalibrationExecute(Sender: TObject);
var i,Key:Integer; aList:TText; aDevice:TDaqDevice;
 function GetCalibInfo(dev:TDaqDevice; i:Integer):LongString;
 var cal:TPolynomCalibration; crv:TCurve;
 var j,nc,nf,nx,ny,nz,ntx,nty:Integer;
 begin
  Result:='';
  if (i>=0) then
  if Assigned(dev) then
  if (Screen.Width>800) then
  if (i<dev.NumCalibrations) then begin
   cal:=dev.Calibration[i];
   crv:=dev.AnalogOutputCurve[i];
   if cal.Ok and crv.Ok then begin
    nc:=0; nf:=0; nx:=0; ny:=0; nz:=0; ntx:=0; nty:=0;
    for j:=0 to dev.NumAnalogOutputs-1 do nc:=Max(nc,Length(dev.AnalogOutputCurve[j].Name));
    for j:=0 to dev.NumCalibrations-1 do nf:=Max(nf,Length(Daq.FileRel(dev.Calibration[j].FileName)));
    for j:=0 to dev.NumCalibrations-1 do nx:=Max(nx,Length(dev.Calibration[j].NameX));
    for j:=0 to dev.NumCalibrations-1 do ny:=Max(ny,Length(dev.Calibration[j].NameY));
    for j:=0 to dev.NumCalibrations-1 do nz:=Max(nz,Length(dev.Calibration[j].NameZ));
    for j:=0 to dev.NumCalibrations-1 do ntx:=Max(ntx,Length(ExtractCalibAlias(dev.Calibration[j].TransformX)));
    for j:=0 to dev.NumCalibrations-1 do nty:=Max(nty,Length(ExtractCalibAlias(dev.Calibration[j].TransformY)));
    Result:=Format(' -  %-*s %-*s %-*s %-*s %-*s %-*s %-*s',
     [nc,crv.Name,nf,Daq.FileRel(cal.FileName),nx,cal.NameX,ny,cal.NameY,nz,cal.NameZ,
      ntx,ExtractCalibAlias(cal.TransformX),nty,ExtractCalibAlias(cal.TransformY)]);
   end;
  end;
 end;
begin
 if Guard.CheckAction(ga_User,ActionButtonCalibration)<0 then Exit;
 if Ok then
 try
  aDevice:=CurrentDevice;
  if aDevice.Ok then
  if NoProblem(aDevice.NumCalibrations>0,
               RusEng('Устройство ','Device ')+aDevice.Name+
               RusEng(' не имеет калибровок.',' has no calibrations.'))
  then begin
   if (aDevice.NumCalibrations>1) then begin
    aList:=NewText;
    try
     for i:=0 to aDevice.NumCalibrations-1 do
     aList.Addln(Format(RusEng('Калибровка %2d %s','Calibration %2d %s'),[i,GetCalibInfo(aDevice,i)]));
     Key:=ListBoxMenu(RusEng('Калибровка устройства ','Calibrate device ')+Name,
                      RusEng('Выбрать калибровку для правки:','Choose calibration to edit:'),
                      aList.Text);
    finally
     Kill(aList);
    end;
   end else Key:=0;
   if (Key>=0) then aDevice.EditCalibration(Key);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonCalibrationExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonCommonDaqExecute(Sender: TObject);
var aList:TText;
begin
 if Ok and Daq.Ok then
 try
  aList:=NewText;
  try
   aList.Addln(RusEng('Общие свойства DAQ:','Common DAQ property:'));
   aList.Addln(RusEng('*******************','********************'));
   Daq.CommonProperty(aList);
   SmartUpdate(MemoDeviceProperty, aList.Text);
  finally
   Kill(aList);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonCommonDaqExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonProblemsExecute(Sender: TObject);
begin
 if Ok and Daq.Ok then
 try
  MemoDeviceProperty.Text:=
   RusEng('Обнаруженные проблемы DAQ:','Known DAQ problems:')+EOL+
   RusEng('**************************','*******************')+EOL+
   Daq.WarningList.Text;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonProblemsExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonErrorViewExecute(Sender: TObject);
var aName:LongString; aDevice:TDaqDevice; aList:TText;
var aTotErr,aDevErr,i:Integer;
begin
 if Ok then
 if Daq.Ok and Daq.CheckSessionStarted then
 try
  aDevice:=CurrentDevice;
  if aDevice.Ok then aName:=aDevice.Name else aName:='';
  aList:=NewText;
  try
   aList.Addln(RusEng('Подробности об ошибках DAQ:','DAQ errors details:'));
   aList.Addln(RusEng('***************************','*******************'));
   aList.Addln(Format('%4s %14s %18s %s', [RusEng('Код','Code'), '[DeviceList]', '['+aName+']',
                      RusEng('Справка','Comment')]));
   for i:=0 to 255 do begin
    aTotErr:=FullDaqDeviceList.ErrorsCount[i];
    if aDevice.Ok then aDevErr:=aDevice.ErrorsCount[i] else aDevErr:=0;
    if (Max(aTotErr,aDevErr)>0) then
    aList.Addln(Format('%4d %14d %18d %s', [i, aTotErr, aDevErr, Daq.ErrMsg[i]]));
   end;
   SmartUpdate(MemoDeviceProperty,aList.Text);
  finally
   Kill(aList);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonErrorViewExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonErrorClearExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_User,ActionButtonErrorClear)<0 then Exit;
 if Ok then
 try
  FullDaqDeviceList.ClearErrors;
  UpdateControls(4);
 except
  on E:Exception do BugReport(E,Self,'ActionButtonErrorClearExecute');
 end;
end;

procedure TFormDaqDeviceControl.ActionButtonTagListExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Root,ActionButtonTagList)<0 then Exit;
 if Ok then
 try
  ViewTagList;
 except
  on E:Exception do BugReport(E,Self,'ActionButtonTagListExecute');
 end;
end;

procedure Init_form_daqdevicecontrol;
begin
end;

procedure Free_form_daqdevicecontrol;
begin
end;

initialization

 Init_form_daqdevicecontrol;

finalization

 Free_form_daqdevicecontrol;

end.

//////////////
// END OF FILE
//////////////

