////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form DAQ Curve Smoothing.                                                  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231201 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_daqcurvesmoothing; // Form DAQ Curve Smoothing

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Spin,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_CurveWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_spline, _crw_curves,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormDaqCurveSmoothing = class(TMasterForm)
    LabelSmooth: TLabel;
    ScrollBarSmooth: TScrollBar;
    CheckBoxNewAxis: TCheckBox;
    SpinEditNewAxis: TSpinEdit;
    PanelButtons: TPanel;
    BitBtnOk: TBitBtn;
    BitBtnCancel: TBitBtn;
    ButtonPreview: TButton;
    EditSmooth: TEdit;
    CheckBoxSmoothManually: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure CheckBoxNewAxisClick(Sender: TObject);
    procedure ButtonPreviewClick(Sender: TObject);
    procedure ScrollBarSmoothScroll(Sender: TObject; ScrollCode: TScrollCode; var ScrollPos: Integer);
  private
    { Private declarations }
  public
    { Public declarations }
    Win : TFormCurveWindow;
    procedure UpdateControls;
  end;

function FormDaqCurveSmoothingExecute(Win:TFormCurveWindow; const Params:LongString=''):Integer;

implementation

{$R *.lfm}

function FormDaqCurveSmoothingExecute(Win:TFormCurveWindow; const Params:LongString=''):Integer;
const
 TheForm : TFormDaqCurveSmoothing = nil;
begin
 Result:=mrCancel;
 if CanShowModal(TheForm) then
 if (Win is TFormCurveWindow) then
 try
  if not TheForm.Ok then begin
   Application.CreateForm(TFormDaqCurveSmoothing, TheForm);
   TheForm.Master:=@TheForm;
  end;
  if TheForm.Ok then begin
   if IsNonEmptyStr(Params)
   then TheForm.ApplyParams(Params)
   else TheForm.LocateToCenterOfScreen;
   TheForm.Top:=0;
   TheForm.SpinEditNewAxis.Value:=Win.Curves[0].Count;
   Win.WindowState:=wsMaximized;
   TheForm.Win:=Win;
   TheForm.UpdateControls;
   SdiMan.ActivateChild(Win);
   if (mrVoice(TheForm.ShowModal)=mrOk) then begin
    Result:=mrOk;
    TheForm.UpdateControls;
   end;
   TheForm.Win:=nil;
  end;
 except
  on E:Exception do BugReport(E,nil,'FormDaqCurveSmoothingExecute');
 end;
end;

function ScrollSmooth(Scroll:TScrollBar):Double;
begin
 with Scroll do
 if (Position=Min)
 then ScrollSmooth:=0
 else ScrollSmooth:=Power(10,30*2*((Position-Min)/(Max-Min)-0.5));
end;

procedure SmoothCurveWithSpline(C0,C1:TCurve; Smooth:Double; out AbsErr,RelErr:Double);
var i:Integer; p:TPoint2D; AveSqr:Double; Weight:PDoubleArray; Spline:TReinschSpline;
begin
 AveSqr:=0; AbsErr:=0; RelErr:=0;
 if C0.Ok and (C0.Count>3) and C1.Ok and (C1.Count>3) then
 try
  Weight:=Allocate(C0.Count*SizeOf(Double));
  try
   if (AllocSize(Weight)>0) then begin
    for i:=0 to C0.Count-1 do Weight[i]:=Smooth;
    Spline:=NewReinschSpline(C0.PX[0],C0.PY[0],Weight[0],C0.Count);
    try
     if Spline.Ok then begin
      for i:=0 to C0.Count-1 do begin
       p:=C0[i];
       AveSqr:=AveSqr+Sqr(p.y);
       AbsErr:=AbsErr+Sqr(p.y-Spline.Get(p.x));
      end;
      AveSqr:=Sqrt(AveSqr/C0.Count);
      AbsErr:=Sqrt(AbsErr/C0.Count);
      if (AveSqr>0) then RelErr:=100*AbsErr/AveSqr;
      for i:=0 to C1.Count-1 do begin
       p.x:=C1[i].X;
       p.y:=Spline.Get(p.x);
       C1[i]:=p;
      end;
     end else Echo(RusEng('Не могу вычислить сплайн!','Cannot evaluate spline!'));
    finally
     Kill(Spline);
    end;
   end;
  finally
   Deallocate(Pointer(Weight));
  end;
 except
  on E:Exception do BugReport(E,nil,'SmoothCurveWithSpline');
 end;
end;

procedure TFormDaqCurveSmoothing.UpdateControls;
var i:Integer; L:TRect2D; s,ae,re:Double; C0,C1:TCurve;
begin
 if Ok and Win.Ok then
 try
  C0:=Win.Curves[0];
  C1:=Win.Curves[1];
  if (C0.Count>3) then begin
   L:=C0.Limits;
   if CheckBoxNewAxis.Checked then begin
    C1.Count:=SpinEditNewAxis.Value;
    for i:=0 to C1.Count-1 do C1[i]:=Point2D(L.A.X+i*(L.B.X-L.A.X)/(C1.Count-1),0);
   end else begin
    C1.AssignData(C0.PX[0],C0.PY[0],C0.Count);
   end;
   s:=0;
   LabelSmooth.Caption:=RusEng('Гладкость ','Smoothing ');
   CheckBoxSmoothManually.Caption:=RusEng('Вручную','Manually');
   if not (CheckBoxSmoothManually.Checked and Str2Real(EditSmooth.Text,s)) then begin
    s:=ScrollSmooth(ScrollBarSmooth)*Power(abs(RectSizeX(L))/C0.Count,3);
    EditSmooth.Text:=Format('%.5g',[s]);
   end;
   SmoothCurveWithSpline(C0,C1,s,ae,re);
   Win.LockDraw;
   try
    Win.Legend:=^C+RusEng('Ошибка ','Error ')+Format('%.5g',[ae])+' ( '+Format('%.5g',[re])+' % )';
    Win.AutoRange;
   finally
    Win.UnlockDraw;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,RusEng('Ошибка при вычислении сплайна!','Error in spline evaluation!'));
 end;
end;

procedure TFormDaqCurveSmoothing.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Сглаживание кривой','Curve smoothing');
 SmartUpdate(BitBtnOk,mrCaption(mrOk));
 SmartUpdate(BitBtnCancel,mrCaption(mrCancel));
 SmartUpdate(ButtonPreview,RusEng('Просмотр','Preview'));
 LabelSmooth.Caption:=RusEng('Гладкость','Smoothing');
 CheckBoxNewAxis.Caption:=RusEng('Новая сетка # точек','New X-set of # points');
end;

procedure TFormDaqCurveSmoothing.CheckBoxNewAxisClick(Sender: TObject);
begin
 UpdateControls;
end;

procedure TFormDaqCurveSmoothing.ButtonPreviewClick(Sender: TObject);
begin
 UpdateControls;
end;

procedure TFormDaqCurveSmoothing.ScrollBarSmoothScroll(Sender: TObject; ScrollCode: TScrollCode; var ScrollPos: Integer);
begin
 if (ScrollCode=scEndScroll) then UpdateControls;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_daqcurvesmoothing;
begin
end;

procedure Free_form_daqcurvesmoothing;
begin
end;

initialization

 Init_form_daqcurvesmoothing;

finalization

 Free_form_daqcurvesmoothing;

end.

//////////////
// END OF FILE
//////////////

