////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Term Window - terminal for pipe input/output.                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231117 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_termwindow; // Form Term Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_ConsoleWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_bsencode,
 _crw_dynar, _crw_snd, _crw_guard, _crw_crypt, _crw_ef,
 _crw_pipeio, _crw_task, _crw_pipe, _crw_tcp, _crw_uart,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormTermWindow = class(TFormConsoleWindow)
    procedure FormDestroy(Sender: TObject);
  private
    { Private declarations }
    myPipe    : Integer;
    myCaption : LongString;
    myOptions : Integer;
    myClosing : Integer;
    myUpdate  : Double;
  public
    { Public declarations }
    procedure Poll;
    function  UsesEol:LongString;
  end;

const                          // Term options
 to_Oem2AnsiRx  = $00000001;   // a - Convert Rx from Oem to Ansi
 to_Ansi2OemTx  = $00000002;   // a - Convert Tx from Ansi to Oem
 to_Display     = $00000004;   // d - Display piped window
 to_Verbose     = $00000008;   // v - Verbose mode
 to_CheckSum    = $00000010;   // n - Use check sum DCON
 to_AutoClose   = $00000020;   // c - Auto close window on connection lost
 to_HexDump     = $00000040;   // x - Output as HEX dump
 to_UrlDump     = $00000080;   // l - Output as URL dump
 to_BShDump     = $00000100;   // s - Output as B\S dump
 to_HexConv     = $00000200;   // h - Input  as HEX encoded
 to_UrlConv     = $00000400;   // u - Input  as URL encoded
 to_BshConv     = $00000800;   // b - Input  as B\S encoded
 to_EolCrLf     = $00001000;   // w - Use EOL as CRLF (Windows like)
 to_EolLf       = $00002000;   // f - Use EOL as LF   (Unix like)
 to_EolCr       = $00004000;   // r - Use EOL as CR   (MAC like)
 to_EchoMode    = $10000000;   // e - Echo Input->Uutput
 to_Threshold   = 3000;        // Delay to auto close window

 // Term options -acendvhubxlswfr
function StringToTermWindowOptions(arg:LongString):Integer;

function  NewConsolePipeWindow(const aCaption   : LongString;
                                     aPipe      : Integer;
                                     aOptions   : Integer;
                                     aInpFifo   : Integer = 1024*8;
                                     aOutFifo   : Integer = 1024*16
                                              ) : TFormTermWindow;
procedure Kill(var TheObject:TFormTermWindow); overload;

implementation

{$R *.lfm}

function StringToTermWindowOptions(arg:LongString):Integer;
var opt:Integer; w1:LongString;
begin
 Result:=0;
 if IsNonEmptyStr(arg) then begin
  opt:=0;
  while IsOption(arg) do begin
   w1:=ExtractWord(1,arg,ScanSpaces);
   if HasChars(w1,'a') then opt:=opt or to_Oem2AnsiRx;
   if HasChars(w1,'a') then opt:=opt or to_Ansi2OemTx;
   if HasChars(w1,'c') then opt:=opt or to_AutoClose;
   if HasChars(w1,'e') then opt:=opt or to_EchoMode;
   if HasChars(w1,'n') then opt:=opt or to_CheckSum;
   if HasChars(w1,'d') then opt:=opt or to_Display;
   if HasChars(w1,'v') then opt:=opt or to_Verbose;
   if HasChars(w1,'h') then opt:=opt or to_HexConv;
   if HasChars(w1,'u') then opt:=opt or to_UrlConv;
   if HasChars(w1,'b') then opt:=opt or to_BshConv;
   if HasChars(w1,'x') then opt:=opt or to_HexDump;
   if HasChars(w1,'l') then opt:=opt or to_UrlDump;
   if HasChars(w1,'s') then opt:=opt or to_BshDump;
   if HasChars(w1,'w') then opt:=opt or to_EolCrLf;
   if HasChars(w1,'f') then opt:=opt or to_EolLf;
   if HasChars(w1,'r') then opt:=opt or to_EolCr;
   arg:=SkipWords(1,arg,ScanSpaces);
  end;
  Result:=opt;
 end;
end;

function AddAdamCheckSumm(const s:LongString):LongString;
var i,L:Integer; b:Byte;
begin
 Result:='';
 try
  b:=0; L:=Length(s);
  for i:=1 to L do
  case s[i] of
   ASCII_CR: begin Result:=Result+HexB(b)+Copy(s,i,L-i+1); Break; end;
   ASCII_LF: begin Result:=Result+HexB(b)+Copy(s,i,L-i+1); Break; end;
   else      begin Result:=Result+s[i]; b:=b+ord(s[i]); end;
  end;
 except
  on E:Exception do BugReport(E,nil,'AddAdamCheckSumm');
 end;
end;

function HexDump(const s:LongString):LongString;
var i,j,k:Integer;
 function GoodChar(c:Char):Char;
 begin
  if c<' ' then Result:=' ' else Result:=c;
 end;
begin
 Result:='';
 try
  for i:=1 to AdjustBufferSize(Length(s),16) div 16 do begin
   for j:=1 to 16 do begin
    k:=1+(i-1)*16+(j-1);
    if k<=Length(s)
    then Result:=Result+' '+HexB(ord(s[k]))
    else Result:=Result+'   ';
   end;
   Result:=Result+' | ';
   for j:=1 to 16 do begin
    k:=1+(i-1)*16+(j-1);
    if k<=Length(s)
    then Result:=Result+GoodChar(s[k])
    else Result:=Result+' ';
   end;
   Result:=Result+EOL;
  end;
 except
  on E:Exception do BugReport(E,nil,'HexDump');
 end;
end;

function UrlDump(const s:LongString):LongString;
begin
 Result:=url_packed(s);
end;

function BshDump(const s:LongString):LongString;
begin
 Result:=backslash_encode(s);
end;

function HexConv(const s:LongString):LongString;
begin
 Result:='';
 try
  Result:=hex_decode(s);
 except
  on E:Exception do BugReport(E,nil,'HexConv');
 end;
end;

function UrlConv(const s:LongString):LongString;
begin
 Result:='';
 try
  Result:=url_decode(s);
 except
  on E:Exception do BugReport(E,nil,'UrlConv');
 end;
end;

function BshConv(const s:LongString):LongString;
begin
 Result:='';
 try
  Result:=backslash_decode(s);
 except
  on E:Exception do BugReport(E,nil,'BshConv');
 end;
end;

procedure InputFilter(aConsole:TFormConsoleWindow; var aText:LongString);
var i,p,e:Integer; s:LongString;
begin
 try
  if Length(aText)>0 then
  if (aConsole is TFormTermWindow) then
  with TFormTermWindow(aConsole) do begin
   e:=0;
   s:=aText;
   s:=ValidateEol(s,0,UsesEol);
   if HasFlags(myOptions,to_UrlConv) then s:=UrlConv(s) else
   if HasFlags(myOptions,to_HexConv) then s:=HexConv(s) else
   if HasFlags(myOptions,to_BshConv) then s:=BshConv(s);
   if HasFlags(myOptions,to_Ansi2OemTx) then s:=StrAnsiToOem(s);
   if HasFlags(myOptions,to_CheckSum) then s:=AddAdamCheckSumm(s);
   for i:=0 to pipe_count(myPipe)-1 do begin
    p:=pipe_stream(myPipe,i);
    if pipe_connected(p)>0 then Inc(e,ord(pipe_send(p,s)<Length(s)));
   end;
   if (e>0) then aConsole.PutText('Send error!)'+EOL);
  end;
 except
  on E:Exception do BugReport(E,nil,'InputFilter');
 end;
end;

procedure OutputFilter(aConsole:TFormConsoleWindow; var aText:LongString);
begin
 try
  if (aConsole is TFormTermWindow) then begin
   if (aText<>'') then aText:=ValidateEol(aText);
  end;
 except
  on E:Exception do BugReport(E,nil,'OutputFilter');
 end;
end;

procedure termPolling(aConsole:TFormConsoleWindow);
begin
 if (aConsole is TFormTermWindow)
 then TFormTermWindow(aConsole).Poll;
end;

function  NewConsolePipeWindow(const aCaption   : LongString;
                                     aPipe      : Integer;
                                     aOptions   : Integer;
                                     aInpFifo   : Integer;
                                     aOutFifo   : Integer):TFormTermWindow;
begin
 Result:=nil;
 if Assigned(pipe_ref(aPipe)) then
 try
  Application.CreateForm(TFormTermWindow,Result);
  with Result do if Ok then begin
   LockDraw;
   try
    Caption:=aCaption;
    AssignFifo(NewFifo(aInpFifo), True, InputFilter,
               NewFifo(aOutFifo), True, OutputFilter);
    UpdateScrollBars;
    UpdateInputBox;
    myPipe:=aPipe;
    myCaption:=aCaption;
    myOptions:=aOptions;
    GuardInput:=ga_Root;
    if HasFlags(myOptions,to_Display) then pipe_ctrl(myPipe,'Display=1');
    Polling:=termPolling;
    StartMonitoring;
   finally
    UnlockDraw;
   end;
  end else Kill(Result);
 except
  on E:Exception do BugReport(E,nil,'NewConsolePipeWindow');
 end;
end;

procedure Kill(var TheObject:TFormTermWindow);
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

procedure TFormTermWindow.FormDestroy(Sender: TObject);
begin
 myCaption:='';
 if (myPipe<>0) then begin
  pipe_free(myPipe);
  myPipe:=0;
 end;
 inherited;
end;

function TFormTermWindow.UsesEol:LongString;
begin
 if HasFlags(myOptions,to_EolCrLf) then Result:=CRLF     else
 if HasFlags(myOptions,to_EolLf)   then Result:=ASCII_LF else
 if HasFlags(myOptions,to_EolCr)   then Result:=ASCII_CR else
 Result:=EOL;
end;

procedure TFormTermWindow.Poll;
var i,p,n:Integer; s:LongString;
begin
 if Assigned(Self) then
 try
  for i:=0 to pipe_count(myPipe)-1 do begin
   p:=pipe_stream(myPipe,i);
   s:=pipe_recv(p,pipe_rxcount(p));
   if (Length(s)>0) then begin
    if HasFlags(myOptions,to_Oem2AnsiRx) then s:=StrOemToAnsi(s);
    if HasFlags(myOptions,to_HexDump) then s:=HexDump(s) else
    if HasFlags(myOptions,to_UrlDump) then s:=UrlDump(s) else
    if HasFlags(myOptions,to_BshDump) then s:=BshDump(s);
    PutText(s);
   end;
  end;
  if (msecnow-myUpdate>TSocketPipe.DefTimeOut) then begin
   myUpdate:=msecnow;
   n:=pipe_connected(myPipe);
   s:=Format('%d',[n]);
   if (n>0) then begin
    if (pipe_ref(myPipe) is TTask) then begin
     s:=Format('task#%d',[pipe_pid(myPipe)]);
    end else
    if (pipe_ref(myPipe) is TSocketPipe) then begin
     if (n<4) then begin
      s:='';
      for i:=0 to pipe_count(myPipe)-1 do
      if (pipe_connected(pipe_stream(myPipe,i))>0) then begin
       if (Length(s)>0) then s:=s+',';
       s:=s+pipe_ctrl(pipe_stream(myPipe,i),'PeerIP');
      end;
     end;
    end;
   end;
   Caption:=Format('[Connected: %s] - %s',[s,myCaption]);
  end;
  if HasFlags(myOptions,to_AutoClose) then begin
   if (pipe_connected(myPipe)>0) then myClosing:=0 else Inc(myClosing);
   if (myClosing>(to_Threshold div 55)) then Close;
  end; 
 except
  on E:Exception do BugReport(E,Self,'Poll');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_termwindow;
begin
end;

procedure Free_form_termwindow;
begin
end;

initialization

 Init_form_termwindow;

finalization

 Free_form_termwindow;

end.

//////////////
// END OF FILE
//////////////

