////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Carib Dialog.                                                         //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231126 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_calibdialog; // Form Carib Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Spin,
 lcltype, lclintf,
 Form_CurveWindow,
 Form_TextEditDialog, Form_TextEditor,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_curves,
 _crw_calib, _crw_lsqpoly, _crw_task, _crw_assoc,
 _crw_sect,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormCalibDialog = class(TMasterForm)
    SaveDialog: TSaveDialog;
    OpenDialog: TOpenDialog;
    PageControlCalibration: TPageControl;
    TabSheetCalibrationPoints: TTabSheet;
    GroupBoxInput: TGroupBox;
    LabelX: TLabel;
    LabelY: TLabel;
    LabelW: TLabel;
    LabelZ: TLabel;
    EditX: TEdit;
    EditY: TEdit;
    EditW: TEdit;
    EditZ: TEdit;
    GroupBoxPoints: TGroupBox;
    LabelPoints: TLabel;
    ListBoxPoints: TListBox;
    ButtonInsert: TButton;
    ButtonDelete: TButton;
    ButtonClear: TButton;
    ButtonSave: TButton;
    ButtonLoad: TButton;
    ButtonNote: TButton;
    ButtonPlot: TButton;
    CheckBoxLinear: TCheckBox;
    CheckBoxErrors: TCheckBox;
    TabSheetCalibrationPreset: TTabSheet;
    GroupBoxPreset: TGroupBox;
    ButtonTextEditor: TButton;
    TabSheetCalibrationCalculator: TTabSheet;
    GroupBoxCalculator: TGroupBox;
    LabelCalculatorXNormal: TLabel;
    LabelCalculatorXLinear: TLabel;
    LabelCalculatorXScale: TLabel;
    LabelCalculatorYNormal: TLabel;
    LabelCalculatorYLinear: TLabel;
    LabelCalculatorYScale: TLabel;
    LabelCalculatorXYParam: TLabel;
    EditCalculatorXNormal: TEdit;
    EditCalculatorXLinear: TEdit;
    EditCalculatorYNormal: TEdit;
    EditCalculatorYLinear: TEdit;
    EditCalculatorXYParam: TEdit;
    ButtonCalculatorX2Normal: TButton;
    ButtonCalculatorX2Linear: TButton;
    ButtonCalculatorY2Normal: TButton;
    ButtonCalculatorY2Linear: TButton;
    ButtonCalculatorCopyPoint: TButton;
    ButtonCalculatorSavePoint: TButton;
    ImageCalculatorDn: TImage;
    ImageCalculatorUp: TImage;
    CheckBoxCalculatorXY: TCheckBox;
    BitBtnCalculatorY2X: TBitBtn;
    BitBtnCalculatorX2Y: TBitBtn;
    TabSheetCalibrationHelp: TTabSheet;
    GroupBoxCalibrationHelpInfo: TGroupBox;
    MemoCalibrationHelpInfo: TMemo;
    PanelPresetButtons: TPanel;
    ButtonApply: TButton;
    GroupBoxPresetTransformXY: TGroupBox;
    LabelScaleX: TLabel;
    ComboBoxScaleX: TComboBox;
    LabelTransformXEval: TLabel;
    EditTransformXEvalF: TEdit;
    LabelTransformXEvalF: TLabel;
    EditTransformXEvalI: TEdit;
    LabelTransformXEvalI: TLabel;
    EditTransformXRangA: TEdit;
    LabelTransformXRangeA: TLabel;
    EditTransformXRangB: TEdit;
    LabelScaleY: TLabel;
    ComboBoxScaleY: TComboBox;
    LabelTransformYEval: TLabel;
    EditTransformYEvalF: TEdit;
    LabelTransformYEvalF: TLabel;
    EditTransformYEvalI: TEdit;
    LabelTransformYEvalI: TLabel;
    EditTransformYRangA: TEdit;
    LabelTransformYRangeA: TLabel;
    EditTransformYRangB: TEdit;
    GroupBoxPresetNameXYZ: TGroupBox;
    LabelNameX: TLabel;
    EditNameX: TEdit;
    LabelNameY: TLabel;
    EditNameY: TEdit;
    LabelNameZ: TLabel;
    EditNameZ: TEdit;
    GroupBoxPresetPlot: TGroupBox;
    LabelBound: TLabel;
    EditBoundN: TEdit;
    LabelBoundN: TLabel;
    LabelBoundA: TLabel;
    EditBoundA: TEdit;
    LabelBoundB: TLabel;
    EditBoundB: TEdit;
    GroupBoxPresetPolynom: TGroupBox;
    LabelPower: TLabel;
    SpinEditPower: TSpinEdit;
    LabelCenter: TLabel;
    EditCenter: TEdit;
    LabelScale: TLabel;
    EditScale: TEdit;
    RadioGroupPresetPolynomFixed: TRadioGroup;
    LabelCoeff0: TLabel;
    LabelCoeff1: TLabel;
    LabelCoeff2: TLabel;
    LabelCoeff3: TLabel;
    LabelCoeff4: TLabel;
    LabelCoeff5: TLabel;
    LabelCoeff6: TLabel;
    LabelCoeff7: TLabel;
    LabelCoeff8: TLabel;
    LabelCoeff9: TLabel;
    LabelCoeff10: TLabel;
    EditCoeff0: TEdit;
    EditCoeff1: TEdit;
    EditCoeff2: TEdit;
    EditCoeff3: TEdit;
    EditCoeff4: TEdit;
    EditCoeff5: TEdit;
    EditCoeff6: TEdit;
    EditCoeff7: TEdit;
    EditCoeff8: TEdit;
    EditCoeff9: TEdit;
    EditCoeff10: TEdit;
    PanelCalibrationHelpButton: TPanel;
    ButtonCalibrationHelpHtm: TButton;
    PageControlCalibrationFormula: TPageControl;
    TabSheetCalibrationFormulaEng: TTabSheet;
    TabSheetCalibrationFormulaRus: TTabSheet;
    ImageCalibrationFormulaRus: TImage;
    ImageCalibrationFormulaEng: TImage;
    LabelCalculatorCpuOpCount: TLabel;
    PanelCalibrationStatus: TPanel;
    PanelCalibrationFileName: TPanel;
    TabSheetCalibrationEvalV: TTabSheet;
    PanelCalibrationEvalV: TPanel;
    MemoCalibrationEvalV: TMemo;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure LabelXClick(Sender: TObject);
    procedure LabelYClick(Sender: TObject);
    procedure LabelWClick(Sender: TObject);
    procedure LabelZClick(Sender: TObject);
    procedure LabelPointsClick(Sender: TObject);
    procedure LabelScaleXClick(Sender: TObject);
    procedure LabelScaleYClick(Sender: TObject);
    procedure LabelPowerClick(Sender: TObject);
    procedure LabelCenterClick(Sender: TObject);
    procedure LabelScaleClick(Sender: TObject);
    procedure LabelNameXClick(Sender: TObject);
    procedure LabelNameYClick(Sender: TObject);
    procedure LabelNameZClick(Sender: TObject);
    procedure LabelBoundAClick(Sender: TObject);
    procedure LabelBoundBClick(Sender: TObject);
    procedure ListBoxPointsDblClick(Sender: TObject);
    procedure ListBoxPointsKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure ButtonInsertClick(Sender: TObject);
    procedure ButtonDeleteClick(Sender: TObject);
    procedure ButtonClearClick(Sender: TObject);
    procedure ButtonSaveClick(Sender: TObject);
    procedure ButtonLoadClick(Sender: TObject);
    procedure ButtonApplyClick(Sender: TObject);
    procedure ButtonNoteClick(Sender: TObject);
    procedure ButtonPlotClick(Sender: TObject);
    procedure CheckBoxLinearClick(Sender: TObject);
    procedure CheckBoxErrorsClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure ButtonTextEditorClick(Sender: TObject);
    procedure ButtonCalculatorX2LinearClick(Sender: TObject);
    procedure ButtonCalculatorX2NormalClick(Sender: TObject);
    procedure ButtonCalculatorY2NormalClick(Sender: TObject);
    procedure ButtonCalculatorY2LinearClick(Sender: TObject);
    procedure ButtonCalculatorCopyPointClick(Sender: TObject);
    procedure ButtonCalculatorSavePointClick(Sender: TObject);
    procedure BitBtnCalculatorY2XClick(Sender: TObject);
    procedure BitBtnCalculatorX2YClick(Sender: TObject);
    procedure ButtonCalibrationHelpHtmClick(Sender: TObject);
    procedure SpinEditPowerChange(Sender: TObject);
    procedure TabSheetCalibrationCalculatorEnter(Sender: TObject);
    procedure TabSheetCalibrationPresetEnter(Sender: TObject);
    procedure TabSheetCalibrationPointsEnter(Sender: TObject);
    procedure TabSheetCalibrationHelpEnter(Sender: TObject);
  private
    { Private declarations }
    myCalib    : TPolynomCalibration;
    myOwns     : Boolean;
    myCurveExp : TCurve;
    myCurveCal : TCurve;
    myCoeff : array[TPolynomPower] of record Curr:Double; Edit:TEdit; Lab:TLabel; end;
    function  GetCalib:TPolynomCalibration;
    procedure OpenPlotter;
    procedure UpdateControls(Flags:Cardinal);
  public
    { Public declarations }
    property  Calib:TPolynomCalibration read GetCalib;
    procedure AssignCalib(aCalib:TPolynomCalibration; aOwnsCalib:Boolean);
  end;

const
  FormCalibDialog  : TFormCalibDialog = nil;
  FormCalibPlotter : TFormCurveWindow = nil;

const
  cdm_Default     = $00000000; // Default value
  cdm_OpenDialog  = $00000001; // Use OpenDialog on Load
  cdm_SaveDialog  = $00000002; // Use SaveDialog on Save
  cdm_EditTextCal = $00000004; // Can edit *.cal as Text
  cdm_EditNameXYZ = $00000008; // Edit NameX,NameY,NameZ
  cdm_EditScaleXY = $00000010; // Can edit ScaleX,ScaleY
  CalibDialogMode : Integer = cdm_Default;

function ReadCalibDialogMode(IniFile,Section:LongString; DefValue:Integer=cdm_Default):Integer;

procedure OpenFormCalibDialog(aCalib:TPolynomCalibration; aOwnsCalib:Boolean);
procedure OpenFormCalibDialogFromFile(aFileName:LongString);
procedure KillFormCalibDialog;

implementation

{$R *.lfm}

 {
 Update flags
 }
const
 uf_All           = $FFFFFFFF;
 uf_Status        = $00000001;
 uf_Captions      = $00000002;
 uf_ListBoxPoints = $00000004;
 uf_EditXYWZ      = $00000008;
 uf_Preset        = $00000010;
 uf_Plotter       = $00000020;

function ReadCalibDialogMode(IniFile,Section:LongString; DefValue:Integer):Integer;
begin
 Result:=0;
 IniFile:=UnifyFileAlias(IniFile);
 if not ReadIniFileInteger(IniFile,Section,'CalibDialogMode%i',Result)
 then Result:=DefValue;
end;

function iAnd(a,b:Integer):Integer;
begin
 Result:=a and b;
end;

procedure OpenFormCalibDialog(aCalib:TPolynomCalibration; aOwnsCalib:Boolean);
begin
 Kill(TObject(FormCalibDialog));
 if (aCalib is TPolynomCalibration) then
 try
  Application.CreateForm(TFormCalibDialog,FormCalibDialog);
  FormCalibDialog.Master:=@FormCalibDialog;
  FormCalibDialog.AssignCalib(aCalib,aOwnsCalib);
  FormCalibDialog.PageControlCalibration.ActivePage:=FormCalibDialog.TabSheetCalibrationPoints;
  FormCalibDialog.UpdateControls(uf_All);
  FormCalibDialog.LocateToCenterOfScreen;
  FormCalibDialog.Show;
 except
  on E:Exception do BugReport(E,nil,'OpenFormCalibDialog');
 end;
end;

procedure OpenFormCalibDialogFromFile(aFileName:LongString);
var Calib:TPolynomCalibration;
begin
 aFileName:=UnifyFileAlias(aFileName);
 Calib:=NewCalibrationSmartLoadFromFile(aFileName);
 if Calib.Ok
 then OpenFormCalibDialog(Calib,true)
 else begin
  Error(RusEng('Не могу прочитать ','Could not read ')+aFileName);
  Kill(Calib);
 end; 
end;

procedure KillFormCalibDialog;
begin
 Kill(TObject(FormCalibDialog));
end;

function TFormCalibDialog.GetCalib:TPolynomCalibration;
begin
 if Assigned(Self) then Result:=myCalib else Result:=nil;
end;

procedure TFormCalibDialog.OpenPlotter;
begin
 if FormCalibPlotter is TFormCurveWindow then begin
  FormCalibPlotter.WindowState:=wsNormal;
  FormCalibPlotter.BringToFront;
 end else begin
  FormCalibPlotter:=NewCurveWindow(RusEng('График калибровки ','Calibration plot ')+Calib.NameX+'-'+Calib.NameY,
                                   ^C^M^L+'   '+Calib.NameY,
                                   ^R+Calib.NameX+'   '^M^C);
  FormCalibPlotter.Master:=@FormCalibPlotter;
  FormCalibPlotter.IsDataProtected:=true;
  FormCalibPlotter.AddCurve(myCurveCal);
  FormCalibPlotter.AddCurve(myCurveExp);
  FormCalibPlotter.DefCurve:=nil;
 end;
end;

procedure TFormCalibDialog.AssignCalib(aCalib:TPolynomCalibration; aOwnsCalib:Boolean);
begin
 if Assigned(Self) then begin
  if myOwns then Kill(myCalib);
  myCalib:=aCalib;
  myOwns:=aOwnsCalib;
 end;
end;

procedure TFormCalibDialog.UpdateControls(Flags:Cardinal);
var i,n,m,nx,ny,nw,nz:Integer; relerr,abserr,norm,sum1,sum2,x,y,z:Double;
 procedure AdjustEditFont(Edit:TEdit);
 var tw:Integer;
 begin
  if Edit is TEdit then begin
   tw:=Canvas.TextWidth(Edit.Text);
   if (tw>Edit.Width*1.5) then Edit.Font.Name:=DefaultSansNarrowFont.Name else
   if (tw>Edit.Width*1.0) then Edit.Font.Name:=DefaultSansFont.Name else Edit.ParentFont:=true;
  end;
 end;
begin
 if Ok and (Calib is TPolynomCalibration) then
 try
  if Flags and uf_Status <> 0 then begin
   MemoCalibrationHelpInfo.Lines.Clear;
   if Calib.IsValid then begin
    sum1:=0;
    sum2:=0;
    norm:=0;
    for i:=0 to Calib.Count-1 do with Calib[i] do begin
     sum1:=sum1+sqr(y-Calib.GetY(x,z))*abs(w);
     sum2:=sum2+sqr(y)*abs(w);
     norm:=norm+abs(w);
    end;
    if norm>0 then begin
     abserr:=sqrt(sum1/norm);
     if sum2>0
     then relerr:=sqrt(sum1/sum2)*100
     else relerr:=_nan;
    end else begin
     abserr:=0;
     relerr:=0;
    end;
    PanelCalibrationStatus.Caption:=Format('OK. RMS=%1.3g (%1.3g %s)',[abserr,relerr,'%']);
   end else begin
    if Calib.Count<=Calib.Power
    then PanelCalibrationStatus.Caption:=RusEng('Недостаточно точек для вычисления полинома.',
                                     'Not enough points to evaluate polynom.')
    else PanelCalibrationStatus.Caption:=RusEng('Полином не пригоден для калибровки.',
                                     'Polynom is not valid for calibration.');
   end;
   PanelCalibrationFileName.Caption:=Calib.FileName;
   if Length(Calib.FileName)*8>PanelCalibrationFileName.Width-16
   then PanelCalibrationFileName.Font.Name:=DefaultSansNarrowFont.Name
   else PanelCalibrationFileName.ParentFont:=true;
   MemoCalibrationHelpInfo.Lines.Text:=Calib.GetText('');
  end;
  if Flags and uf_Captions <> 0 then begin
   Caption:=Format(RusEng('Калибровка','Calibration')+' %s - %s [ %s ]',[Calib.NameX,Calib.NameY,Calib.FileName]);
   TabSheetCalibrationPoints.Caption:=RusEng('Точки Калибровки','Calibration Points');
   TabSheetCalibrationPreset.Caption:=RusEng('Параметры Калибровки','Calibration Parameters');
   TabSheetCalibrationCalculator.Caption:=RusEng('Калькулятор','Calculator');
   GroupBoxInput.Caption:=RusEng('Точка калибровки','Calibration point');
   GroupBoxPoints.Caption:=RusEng('Список точек калибровки','Calibration points list');
   GroupBoxPreset.Caption:=RusEng('Параметры калибровки','Calibration preset');
   LabelX.Caption:=Format(RusEng('Аргумент','Argument')+' %s',[Calib.NameX]);
   LabelY.Caption:=Format(RusEng('Значение','Value')+' %s',[Calib.NameY]);
   LabelW.Caption:=Format(RusEng('Весовой фактор','Factor')+' %s',[Calib.NameW]);
   LabelZ.Caption:=Format(RusEng('Параметр','Parameter')+' %s',[Calib.NameZ]);
   LabelScaleX.Caption:=Format('%-*s %-14s %-14s',[MaxCalibNamesLeng,Calib.NameX,RusEng('От','From'),RusEng('До','To')]);
   LabelScaleY.Caption:=Format('%-*s %-14s %-14s',[MaxCalibNamesLeng,Calib.NameY,RusEng('От','From'),RusEng('До','To')]);
   LabelPower.Caption:=RusEng('Полином:Степень','Polynom:Power');
   LabelCenter.Caption:=RusEng('Центр','Center');
   LabelScale.Caption:=RusEng('Масштаб','Scale');
   LabelNameX.Caption:=RusEng('Имя X','Name X');
   LabelNameY.Caption:=RusEng('Имя Y','Name Y');
   LabelNameZ.Caption:=RusEng('Имя Z','Name Z');
   LabelBound.Caption:=RusEng('График','Plot');
   LabelBoundN.Caption:=RusEng('точек','points');
   LabelBoundA.Caption:=RusEng('От','From');
   LabelBoundB.Caption:=RusEng('До','To');
   ButtonInsert.Caption:=RusEng('Вставить','Insert');
   ButtonDelete.Caption:=RusEng('Удалить','Delete');
   ButtonClear.Caption:=RusEng('Очистка','Clear');
   ButtonSave.Caption:=RusEng('Запись','Save');
   ButtonLoad.Caption:=RusEng('Чтение','Load');
   ButtonApply.Caption:=RusEng('Принять внесенные изменения','Apply user changes');
   ButtonNote.Caption:=RusEng('Паспорт','Comment');
   ButtonPlot.Caption:=RusEng('График','Plot');
   CheckBoxLinear.Caption:=RusEng('Линейн.','Linear');
   CheckBoxErrors.Caption:=RusEng('Ошибки','Errors');
   GroupBoxCalculator.Caption:=RusEng('Калькулятор','Calculator');
   LabelCalculatorXNormal.Caption:=Format(RusEng('Аргумент','Argument')+' %s',[Calib.NameX]);
   LabelCalculatorYNormal.Caption:=Format(RusEng('Значение','Value')+' %s',[Calib.NameY]);
   LabelCalculatorXLinear.Caption:=RusEng('Преобразованное','Transformed');
   LabelCalculatorYLinear.Caption:=RusEng('Преобразованное','Transformed');
   LabelCalculatorXScale.Caption:=Format('%s',[ExtractCalibAlias(Calib.TransformX)]);
   LabelCalculatorYScale.Caption:=Format('%s',[ExtractCalibAlias(Calib.TransformY)]);
   LabelCalculatorXYParam.Caption:=Format(RusEng('Параметр','Parameter')+' %s',[Calib.NameZ]);
   ButtonCalculatorCopyPoint.Caption:=RusEng('Взять точку из калибровки','Copy point from calibration');
   ButtonCalculatorSavePoint.Caption:=RusEng('Внести точку в калибровку','Save point to calibration');
   CheckBoxCalculatorXY.Caption:=RusEng('Вычислить напрямую','Calculate directly');
   TabSheetCalibrationHelp.Caption:=RusEng('Справка','Help');
   GroupBoxCalibrationHelpInfo.Caption:=RusEng('Текст калибровки:','Calibration as text:');
   ButtonCalibrationHelpHtm.Caption:=RusEng('Справка по системе калибровок','Help on calibration system');
   GroupBoxPresetTransformXY.Caption:=RusEng('Калибровочные преобразования (тип шкалы)','Calibration trransformations (type of scale)');
   GroupBoxPresetNameXYZ.Caption:=RusEng('Наименование осей: X=Аргумент, Y=Значение, Z=Параметр','Axis naming: X=Argument, Y=Value, Z=Parameter');
   GroupBoxPresetPlot.Caption:=RusEng('Параметры графика','Plot parameters');
   GroupBoxPresetPolynom.Caption:=RusEng('Параметры полинома','Polynom parameters');
   RadioGroupPresetPolynomFixed.Caption:=RusEng('Метод расчета полинома','Polynom calculation method');
   RadioGroupPresetPolynomFixed.Items.Text:=RusEng('МНК (наименьших квадратов)'+EOL+'С фиксированными коэффициентами',
                                                   'LSM (least squares)'+EOL+'Use fixed coefficients');
   TabSheetCalibrationFormulaRus.Caption:='Формула';
   TabSheetCalibrationFormulaEng.Caption:='Formula';
   PageControlCalibrationFormula.ActivePageIndex:=Ord(RusEng('Rus','Eng')='Rus');
   for i:=Low(myCoeff) to High(myCoeff) do
   if Assigned(myCoeff[i].Lab) then myCoeff[i].Lab.Caption:=Format('Coeff[%d]',[i]);
   MemoCalibrationHelpInfo.Color:=clInfoBk;
   MemoCalibrationHelpInfo.ReadOnly:=true;
   if MemoCalibrationEvalV.Lines.Count=0 then MemoCalibrationEvalV.Text:=
    RusEng('Система калибровки позволяет задавать нелинейную шкалу в формульном виде. '
          +'Для этого надо задать тип шкалы EVAL(V), формулы для прямого:обратного преобразования '
          +'и допустимые нижние:верхние пределы аргумента, разделив их символом : (двоеточия). '+EOL
          +'Например, для описания сопротивления R термистора как функции '
          +' абсолютной температуры T используется уравнение Стейнхарта: '+EOL
          +'  1/T=a+b*ln(R)+c*ln(R)^2+d*ln(R)^3 '+EOL
          +'Это уравнение (при использовании температуры в °C) можно описать в виде полинома, '
          +'если использовать линеаризующие преобразования: '+EOL
          +'  TransformX = EVAL(V):LN(V):EXP(V):1E-100:1E100 '+EOL
          +'  TransformY = EVAL(V):1/(V+273.15):1/V-273.15:-273:1E100 '+EOL
          +'где EVAL(V) - признак формульного преобразования '+EOL
          +'    V,P     - аргумент V и параметр P '+EOL
          +'    LN(V)   - прямое (линеаризующее) преобразование шкалы X '+EOL
          +'    EXP(V)  - обратное преобразование шкалы X'' (в X) '+EOL
          +'    1E-100:1E100 - допустимые пределы шкалы X '+EOL
          +'    1/(V+273.15) - прямое (линеаризующее) преобразование шкалы Y '+EOL
          +'    1/V-273.15   - обратное преобразование шкалы Y'' (в Y)'+EOL
          +'    -273:1E100   - допустимые пределы шкалы Y '+EOL
          +'После линеаризации используется подгонка МНК полиномом. '+EOL
          +EOL,
           'Calibration system allow to use calibration transformation defined by formula evaluation. '
          +'Use EVAL(V) scale type, formulas for direct:inverse calibration fransformation '
          +'and lower:higher argument limits, delimited by : (colon). '+EOL
          +'For example, dependence of NTC sensor resistance R vs absolute temperature T '
          +' descibed by Stainhart equition: '+EOL
          +'  1/T=a+b*ln(R)+c*ln(R)^2+d*ln(R)^3 '+EOL
          +'This equition (with using temperature in °C) may be reduced to polinom, '
          +'with next scale transformation: '+EOL
          +'  TransformX = EVAL(V):LN(V):EXP(V):1E-100:1E100 '+EOL
          +'  TransformY = EVAL(V):1/(V+273.15):1/V-273.15:-273:1E100 '+EOL
          +'где EVAL(V) - sign of formula transformation '+EOL
          +'    V,P       - argument V and parameter P '+EOL
          +'    LN(V)     - direct scale transformation X '+EOL
          +'    EXP(V)    - inverse scale transformation '+EOL
          +'    1E-100:1E100 - lower:higher range of X '+EOL
          +'    1/(V+273.15) - direct scale transformation Y '+EOL
          +'    1/V-273.15   - inverse scale transformation '+EOL
          +'    -273:1E100   - lower:higher range of Y '+EOL
          +'After linearization least squares polynom is uses. '+EOL
          +EOL);
  end;
  if Flags and uf_ListBoxPoints <> 0 then begin
   nx:=Length(Calib.NameX); ny:=Length(Calib.NameY);
   nw:=Length(Calib.NameW); nz:=Length(Calib.NameZ);
   for i:=0 to Calib.Count-1 do nx:=max(nx,Length(Format('%g',[Calib.Items[i].x])));
   for i:=0 to Calib.Count-1 do ny:=max(ny,Length(Format('%g',[Calib.Items[i].y])));
   for i:=0 to Calib.Count-1 do nw:=max(nw,Length(Format('%g',[Calib.Items[i].w])));
   for i:=0 to Calib.Count-1 do nz:=max(nz,Length(Format('%g',[Calib.Items[i].z])));
   LabelPoints.Caption:=Format(' %-*s  %-*s  %-*s  %-*s  %s',[nx,Calib.NameX,ny,Calib.NameY,nw,Calib.NameW,nz,Calib.NameZ,RusEng('Отклонение','Difference')]);
   ListBoxPoints.Items.Clear;
   for i:=0 to Calib.Count-1 do with Calib[i] do
   ListBoxPoints.Items.Add(Format('%-*g  %-*g  %-*g  %-*g  %-1.3g',[nx,x,ny,y,nw,w,nz,z,y-Calib.GetY(x,z)]));
   ListBoxPoints.ItemIndex:=min(Calib.Count-1,max(0,ListBoxPoints.ItemIndex));
  end;
  if Flags and uf_EditXYWZ <> 0 then begin
   if InRange(ListBoxPoints.ItemIndex,0,ListBoxPoints.Items.Count-1) then begin
    EditX.Text:=ExtractWord(1,ListBoxPoints.Items[ListBoxPoints.ItemIndex],ScanSpaces);
    EditY.Text:=ExtractWord(2,ListBoxPoints.Items[ListBoxPoints.ItemIndex],ScanSpaces);
    EditW.Text:=ExtractWord(3,ListBoxPoints.Items[ListBoxPoints.ItemIndex],ScanSpaces);
    EditZ.Text:=ExtractWord(4,ListBoxPoints.Items[ListBoxPoints.ItemIndex],ScanSpaces);
   end;
  end;
  if Flags and uf_Preset <> 0 then begin
   ComboBoxScaleX.Text:=FormatCalibTransformItem(CalibTransformList.Find(Calib.TransformX));
   ComboBoxScaleX.Items.Text:=CalibTransformList.Text;
   ComboBoxScaleY.Text:=FormatCalibTransformItem(CalibTransformList.Find(Calib.TransformY));
   ComboBoxScaleY.Items.Text:=CalibTransformList.Text;
   EditTransformXEvalF.Text:=Trim(ExtractCalibEvalItem(2,Calib.TransformX)); AdjustEditFont(EditTransformXEvalF);
   EditTransformXEvalI.Text:=Trim(ExtractCalibEvalItem(3,Calib.TransformX)); AdjustEditFont(EditTransformXEvalI);
   EditTransformXRangA.Text:=Trim(ExtractCalibEvalItem(4,Calib.TransformX)); AdjustEditFont(EditTransformXRangA);
   EditTransformXRangB.Text:=Trim(ExtractCalibEvalItem(5,Calib.TransformX)); AdjustEditFont(EditTransformXRangB);
   EditTransformYEvalF.Text:=Trim(ExtractCalibEvalItem(2,Calib.TransformY)); AdjustEditFont(EditTransformYEvalF);
   EditTransformYEvalI.Text:=Trim(ExtractCalibEvalItem(3,Calib.TransformY)); AdjustEditFont(EditTransformYEvalI);
   EditTransformYRangA.Text:=Trim(ExtractCalibEvalItem(4,Calib.TransformY)); AdjustEditFont(EditTransformYRangA);
   EditTransformYRangB.Text:=Trim(ExtractCalibEvalItem(5,Calib.TransformY)); AdjustEditFont(EditTransformYRangB);
   SpinEditPower.Value:=Calib.Power;
   EditCenter.Text:=Format('%g',[Calib.Center]);
   EditScale.Text:=Format('%g',[Calib.Scale]);
   EditNameX.Text:=Calib.NameX;
   EditNameZ.Text:=Calib.NameZ;
   EditNameY.Text:=Calib.NameY;
   EditBoundA.Text:=Format('%g',[Calib.BoundA]);
   EditBoundB.Text:=Format('%g',[Calib.BoundB]);
   RadioGroupPresetPolynomFixed.ItemIndex:=Ord(Calib.IsFixed);
   for i:=Low(myCoeff) to High(myCoeff) do if Assigned(myCoeff[i].Edit) then begin
    myCoeff[i].Edit.Text:=Format('%g',[Calib.Coeff[i]*Ord(i<=Calib.Power)]);
    myCoeff[i].Edit.ReadOnly:=not Calib.IsFixed;
    myCoeff[i].Edit.Enabled:=(i<=Calib.Power);
   end;
  end;
  if Flags and uf_Plotter <> 0 then
  if FormCalibPlotter is TFormCurveWindow then begin
   FormCalibPlotter.Caption:=RusEng('График калибровки ','Calibration plot ')+Calib.NameX+'-'+Calib.NameY;
   m:=ord(CheckBoxLinear.Checked)+2*ord(CheckBoxErrors.Checked);
   case m of
    0 :
     begin
      FormCalibPlotter.Title:=^C+RusEng('Калибровка, нормальная шкала','Calibration, normal scale')+^M^L+'   '+Calib.NameY;
      FormCalibPlotter.Legend:=^R+Calib.NameX+'   '^M^C+PanelCalibrationStatus.Caption;
     end;
    1 :
     begin
      FormCalibPlotter.Title:=^C+RusEng('Калибровка, линеаризованная шкала','Calibration, linear scale')+^M^L+'   Linear('+Calib.NameY+')';
      FormCalibPlotter.Legend:=^R'Linear('+Calib.NameX+')   '^M^C+PanelCalibrationStatus.Caption;
     end;
    2 :
     begin
      FormCalibPlotter.Title:=^C+RusEng('Невязка, нормальная шкала','Residial, normal scale')+^M^L+'   '+Calib.NameY;
      FormCalibPlotter.Legend:=^R+Calib.NameX+'   '^M^C+PanelCalibrationStatus.Caption;
     end;
    3 :
     begin
      FormCalibPlotter.Title:=^C+RusEng('Невязка, линеаризованная шкала','Residial, linear scale')+^M^L+'   Linear('+Calib.NameY+')';
      FormCalibPlotter.Legend:=^R'Linear('+Calib.NameX+')   '^M^C+PanelCalibrationStatus.Caption;
     end;
   end;
   if Str2Int(EditBoundN.Text,n) then n:=Max(10,Min(1000000,n)) else n:=5000;
   myCurveExp.Count:=0;
   myCurveCal.Count:=0;
   for i:=0 to Calib.Count-1 do begin
    x:=Calib[i].x;
    y:=Calib[i].y;
    z:=Calib[i].z;
    case m of
     0 : ;
     1 : begin
          x:=Calib.LinearX(x,z,true);
          y:=Calib.LinearY(y,z,true);
         end;
     2 : y:=y-Calib.GetY(x,z);
     3 : begin
          y:=Calib.LinearY(y,z,true)-Calib.LinearY(Calib.GetY(x,z),z,true);
          x:=Calib.LinearX(x,z,true);
         end;
    end;
    myCurveExp.AddPoint(x,y);
   end;
   for i:=0 to n-1 do begin
    z:=Calib[0].z;
    x:=Calib.BoundA+(Calib.BoundB-Calib.BoundA)*i/(n-1);
    y:=Calib.GetY(x,z);
    case m of
     0 :;
     1 : begin
          x:=Calib.LinearX(x,z,true);
          y:=Calib.LinearY(y,z,true);
         end;
     2 : y:=0;
     3 : begin
          x:=Calib.LinearX(x,z,true);
          y:=0;
         end;
    end;
    myCurveCal.AddPoint(x,y);
   end;
   FormCalibPlotter.AutoRange;
  end;
 except
  on E:Exception do begin
   if (E is EConvertError)
   then Warning(RusEng('Ошибка формата!','Format error!'))
   else Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'UpdateControls');
  end;
 end;
end;

procedure TFormCalibDialog.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 myCalib:=nil;
 myOwns:=false;
 myCurveExp:=NewCurve(0,'CalibrationPoints',clBlack,$04);
 myCurveCal:=NewCurve(0,'CalibrationResult',clBlue, $10);
 myCoeff[0].Edit:=EditCoeff0; myCoeff[1].Edit:=EditCoeff1;
 myCoeff[2].Edit:=EditCoeff2; myCoeff[3].Edit:=EditCoeff3;
 myCoeff[4].Edit:=EditCoeff4; myCoeff[5].Edit:=EditCoeff5;
 myCoeff[6].Edit:=EditCoeff6; myCoeff[7].Edit:=EditCoeff7;
 myCoeff[8].Edit:=EditCoeff8; myCoeff[9].Edit:=EditCoeff9;
 myCoeff[10].Edit:=EditCoeff10;
 myCoeff[0].Lab:=LabelCoeff0; myCoeff[1].Lab:=LabelCoeff1;
 myCoeff[2].Lab:=LabelCoeff2; myCoeff[3].Lab:=LabelCoeff3;
 myCoeff[4].Lab:=LabelCoeff4; myCoeff[5].Lab:=LabelCoeff5;
 myCoeff[6].Lab:=LabelCoeff6; myCoeff[7].Lab:=LabelCoeff7;
 myCoeff[8].Lab:=LabelCoeff8; myCoeff[9].Lab:=LabelCoeff9;
 myCoeff[10].Lab:=LabelCoeff10;
end;

procedure TFormCalibDialog.FormDestroy(Sender: TObject);
begin
 Kill(FormCalibPlotter);
 Kill(myCurveExp);
 Kill(myCurveCal);
 AssignCalib(nil,false);
end;

procedure TFormCalibDialog.FormClose(Sender: TObject; var Action: TCloseAction);
begin
 Action:=caFree;
end;

procedure TFormCalibDialog.LabelXClick(Sender: TObject);
begin
 SmartFocus(EditX);
end;

procedure TFormCalibDialog.LabelYClick(Sender: TObject);
begin
 SmartFocus(EditY);
end;

procedure TFormCalibDialog.LabelWClick(Sender: TObject);
begin
 SmartFocus(EditW);
end;

procedure TFormCalibDialog.LabelZClick(Sender: TObject);
begin
 SmartFocus(EditZ);
end;

procedure TFormCalibDialog.LabelPointsClick(Sender: TObject);
begin
 SmartFocus(ListBoxPoints);
end;

procedure TFormCalibDialog.LabelScaleXClick(Sender: TObject);
begin
 SmartFocus(ComboBoxScaleX);
end;

procedure TFormCalibDialog.LabelScaleYClick(Sender: TObject);
begin
 SmartFocus(ComboBoxScaleY);
end;

procedure TFormCalibDialog.LabelPowerClick(Sender: TObject);
begin
 SmartFocus(SpinEditPower);
end;

procedure TFormCalibDialog.LabelCenterClick(Sender: TObject);
begin
 SmartFocus(EditCenter);
end;

procedure TFormCalibDialog.LabelScaleClick(Sender: TObject);
begin
 SmartFocus(EditScale);
end;

procedure TFormCalibDialog.LabelNameXClick(Sender: TObject);
begin
 SmartFocus(EditNameX);
end;

procedure TFormCalibDialog.LabelNameYClick(Sender: TObject);
begin
 SmartFocus(EditNameY);
end;

procedure TFormCalibDialog.LabelNameZClick(Sender: TObject);
begin
 SmartFocus(EditNameZ);
end;

procedure TFormCalibDialog.LabelBoundAClick(Sender: TObject);
begin
 SmartFocus(EditBoundA);
end;

procedure TFormCalibDialog.LabelBoundBClick(Sender: TObject);
begin
 SmartFocus(EditBoundB);
end;

procedure TFormCalibDialog.ListBoxPointsDblClick(Sender: TObject);
begin
 UpdateControls(uf_EditXYWZ);
end;

procedure TFormCalibDialog.ListBoxPointsKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
 if (Key = VK_RETURN) and (Shift=[]) then UpdateControls(uf_EditXYWZ);
end;

procedure TFormCalibDialog.ButtonInsertClick(Sender: TObject);
var p:TCalibPoint;
begin
 if Ok then
 try
  if Calib.Ok and
     Str2Real(EditX.Text,p.x) and
     Str2Real(EditY.Text,p.y) and
     Str2Real(EditW.Text,p.w) and
     Str2Real(EditZ.Text,p.z) and
     RectContainsPoint(Calib.TransformRange,Point2D(p.x,p.y)) and
     (p.w>0)
  then begin
   Calib.Insert(p);
   UpdateControls(uf_Status+uf_ListBoxPoints+uf_Plotter);
  end else Warning(RusEng('Недопустимый ввод!','Invalid input data!'));
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonInsertClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonDeleteClick(Sender: TObject);
begin
 if Ok then
 try
  if Calib.Ok and InRange(ListBoxPoints.ItemIndex,0,Calib.Count-1)
  then begin
   Calib.Delete(ListBoxPoints.ItemIndex);
   UpdateControls(uf_Status+uf_ListBoxPoints+uf_Plotter);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonDeleteClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonClearClick(Sender: TObject);
begin
 if Ok then
 try
  if Calib.Ok and (Calib.Count>0) then begin
   Calib.Clear;
   Calib.Update;
   UpdateControls(uf_Status+uf_Captions+uf_ListBoxPoints+uf_Plotter);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonClearClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonSaveClick(Sender: TObject);
var FName:LongString; SkipSaveDialog:Boolean;
begin
 if Ok then
 try
  SaveDialog.Title:=RusEng('Сохранить калибровку','Save calibration');
  SaveDialog.Filter:=RusEng('Калибровки (*.cal)|*.cal','Calibratuions (*.cal)|*.cal');
  SaveDialog.FileName:=Calib.FileName;
  if IsEmptyStr(SaveDialog.FileName)
  then OpenDialogSelectType(SaveDialog,'*.cal')
  else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
  SkipSaveDialog:=True;
  if myOwns then SkipSaveDialog:=False;
  if IsEmptyStr(SaveDialog.FileName) then SkipSaveDialog:=False;
  if IsWildCard(SaveDialog.FileName) then SkipSaveDialog:=False;
  if iAnd(CalibDialogMode,cdm_SaveDialog)<>0 then SkipSaveDialog:=False;
  if SkipSaveDialog or GuardOpenDialog(SaveDialog).Execute then begin
   FName:=UnifyFileAlias(DefaultExtension(SaveDialog.FileName,SaveDialog.DefaultExt));
   if FileExists(FName) then
   if (not SkipSaveDialog) and (YesNo(RusEng('Перезаписать файл ','Overwrite file ')+FName+'?')<>mrYes) then FName:='';
   if IsSameText(AddPathDelim(HomeDir),Copy(UnifyFileAlias(FName),1,Length(AddPathDelim(HomeDir)))) then begin
    Echo(RusEng('Нельзя менять системные калибровки!','Unable to overwrite system calibrations!'));
    FName:='';
   end;
   if IsNonEmptyStr(FName) then
   if Calib.SaveToFile(FName)
   then Echo(StdDateTimePrompt+RusEng('Сохранение калибровки.'+EOL+'Файл ',
                                      'Calibration was saved.'+EOL+'File ')+FName)
   else Error(RusEng('Не могу сохранить ','Could not save ')+FName);
  end;
  UpdateControls(uf_All);
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonSaveClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonLoadClick(Sender: TObject);
var FName:LongString; Cal:TPolynomCalibration; SkipOpenDialog:Boolean;
begin
 if Ok then
 try
  OpenDialog.Title:=RusEng('Загрузить калибровку','Load calibration');
  OpenDialog.Filter:=RusEng('Калибровки (*.cal)|*.cal','Calibratuions (*.cal)|*.cal');
  OpenDialog.FileName:=Calib.FileName;
  if IsEmptyStr(OpenDialog.FileName)
  then OpenDialogSelectType(OpenDialog,'*.cal')
  else OpenDialogSelectType(OpenDialog,OpenDialog.FileName);
  SkipOpenDialog:=True;
  if myOwns then SkipOpenDialog:=False;
  if IsEmptyStr(OpenDialog.FileName) then SkipOpenDialog:=False;
  if IsWildCard(OpenDialog.FileName) then SkipOpenDialog:=False;
  if iAnd(CalibDialogMode,cdm_OpenDialog)<>0 then SkipOpenDialog:=False;
  if SkipOpenDialog or GuardOpenDialog(OpenDialog).Execute then begin
   FName:=UnifyFileAlias(DefaultExtension(OpenDialog.FileName,OpenDialog.DefaultExt));
   Cal:=NewCalibrationLoadFromFile(Calib.NameX,Calib.NameY,Calib.NameZ,FName,'');
   if (not SkipOpenDialog) or (iAnd(CalibDialogMode,cdm_EditNameXYZ)<>0) then
   if not Cal.Ok then begin
    Kill(Cal);
    Cal:=NewCalibrationSmartLoadFromFile(FName);
   end;
   if Cal.Ok then begin
    Calib.Assign(Cal);
    if (not SkipOpenDialog) then Calib.FileName:=Cal.FileName;
    Echo(StdDateTimePrompt+RusEng('Загрузка калибровки.'+EOL+'Файл ',
                                  'Loading calibration.'+EOL+'File ')+FName)
   end else Warning(RusEng('Не могу загрузить ','Could not load ')+FName);
   Kill(Cal);
   UpdateControls(uf_All);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonLoadClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonApplyClick(Sender: TObject);
var aCenter,aScale,aBoundA,aBoundB:Double; i:Integer;
 function ReadEditCoff:Boolean;
 var  i:Integer;
 begin
  Result:=false;
  if Calib.IsFixed or Calib.IsValid then
  for i:=Low(myCoeff) to High(myCoeff) do begin
   if not Assigned(myCoeff[i].Edit) then Exit;
   if not Str2Real(myCoeff[i].Edit.Text,myCoeff[i].Curr) then Exit;
  end;
  Result:=true;
 end;
begin
 if Ok then
 try
  if Calib.Ok and
     IsNonEmptyStr(ComboBoxScaleX.Text) and
     IsNonEmptyStr(ComboBoxScaleY.Text) and
     IsNonEmptyStr(EditNameX.Text) and
     IsNonEmptyStr(EditNameY.Text) and
     Str2Real(EditCenter.Text,aCenter) and
     Str2Real(EditScale.Text,aScale) and
     Str2Real(EditBoundA.Text,aBoundA) and
     Str2Real(EditBoundB.Text,aBoundB) and
     (aScale>0) and (aBoundA<aBoundB) and
     ReadEditCoff
  then begin
   Calib.LockUpdate;
   Calib.NameX:=EditNameX.Text;
   Calib.NameY:=EditNameY.Text;
   Calib.NameZ:=EditNameZ.Text;
   if IsSameCalibAlias(ComboBoxScaleX.Text,id_CalibEvalAlias)
   then Calib.TransformX:=UpcaseStr(Format('%s:%s:%s:%s:%s',[id_CalibEvalAlias,
                             Trim(EditTransformXEvalF.Text),Trim(EditTransformXEvalI.Text),
                             Trim(EditTransformXRangA.Text),Trim(EditTransformXRangB.Text)]))
   else Calib.TransformX:=ExtractWord(1,ComboBoxScaleX.Text,ScanSpaces);
   if IsSameCalibAlias(ComboBoxScaleY.Text,id_CalibEvalAlias)
   then Calib.TransformY:=UpcaseStr(Format('%s:%s:%s:%s:%s',[id_CalibEvalAlias,
                             Trim(EditTransformYEvalF.Text),Trim(EditTransformYEvalI.Text),
                             Trim(EditTransformYRangA.Text),Trim(EditTransformYRangB.Text)]))
   else Calib.TransformY:=ExtractWord(1,ComboBoxScaleY.Text,ScanSpaces);
   Calib.Power:=SpinEditPower.Value;
   Calib.Center:=aCenter;
   Calib.Scale:=aScale;
   Calib.BoundA:=aBoundA;
   Calib.BoundB:=aBoundB;
   Calib.IsFixed:=(RadioGroupPresetPolynomFixed.ItemIndex>0);
   if Calib.IsFixed then
   for i:=Low(myCoeff) to High(myCoeff) do
   Calib.Coeff[i]:=myCoeff[i].Curr*Ord(i<=Calib.Power);
   Calib.UnlockUpdate;
   UpdateControls(uf_Status+uf_Captions+uf_Preset+uf_Plotter+uf_ListBoxPoints);
  end else Warning(RusEng('Недопустимый ввод!','Invalid input data!'));
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonApplyClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonNoteClick(Sender: TObject);
begin
 if Ok then
 try
  Calib.Note.Text:=TextEditDialog(RusEng('Редактирование комментария калибровки','Edit calibration comment'),
                                  RusEng('Отредактируйте:','Please, edit:'),Calib.Note.Text);
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonNoteClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonPlotClick(Sender: TObject);
begin
 if Ok then
 try
  OpenPlotter;
  UpdateControls(uf_Plotter);
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonPlotClick');
  end;
 end;
end;

procedure TFormCalibDialog.CheckBoxLinearClick(Sender: TObject);
begin
 UpdateControls(uf_Status+uf_Plotter);
end;

procedure TFormCalibDialog.CheckBoxErrorsClick(Sender: TObject);
begin
 UpdateControls(uf_Status+uf_Plotter);
end;

procedure TFormCalibDialog.FormShow(Sender: TObject);
begin
 if Ok then
 try
  PageControlCalibration.ActivePageIndex:=0;
  ButtonTextEditor.Enabled:=myOwns or (iAnd(CalibDialogMode,cdm_EditTextCal)<>0);
  ComboBoxScaleX.Enabled:=myOwns or (iAnd(CalibDialogMode,cdm_EditScaleXY)<>0);
  EditTransformXEvalF.Enabled:=ComboBoxScaleX.Enabled;
  EditTransformXEvalI.Enabled:=ComboBoxScaleX.Enabled;
  EditTransformXRangA.Enabled:=ComboBoxScaleX.Enabled;
  EditTransformXRangB.Enabled:=ComboBoxScaleX.Enabled;
  ComboBoxScaleY.Enabled:=myOwns or (iAnd(CalibDialogMode,cdm_EditScaleXY)<>0);
  EditTransformYEvalF.Enabled:=ComboBoxScaleY.Enabled;
  EditTransformYEvalI.Enabled:=ComboBoxScaleY.Enabled;
  EditTransformYRangA.Enabled:=ComboBoxScaleY.Enabled;
  EditTransformYRangB.Enabled:=ComboBoxScaleY.Enabled;
  EditNameX.Enabled:=myOwns or (iAnd(CalibDialogMode,cdm_EditNameXYZ)<>0);
  EditNameY.Enabled:=myOwns or (iAnd(CalibDialogMode,cdm_EditNameXYZ)<>0);
  EditNameZ.Enabled:=myOwns or (iAnd(CalibDialogMode,cdm_EditNameXYZ)<>0);
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'FormShow');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonTextEditorClick(Sender: TObject);
var TheEditor:TFormTextEditor;
begin
 if Ok then
 try
  if FileExists(Calib.FileName) then
  if not IsWildCard(Calib.FileName) then begin
   TheEditor:=FindTextEditor(Calib.FileName,true,true);
   if TheEditor.Ok then begin
    TheEditor.Open(Calib.FileName);
    if (Pos(AddPathDelim(UnifyFileAlias(HomeDir)),UnifyFileAlias(Calib.FileName))>0) then begin
     TheEditor.IsReadOnly:=true; TheEditor.ActionEditReadOnly.Enabled:=false;
    end;
   end;
   Close;
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonTextEditorClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalculatorX2NormalClick(Sender: TObject);
var x,y,z:Double; ops:Int64;
begin
 if Ok and Calib.Ok then
 try
  if Str2Real(EditCalculatorXYParam.Text,z) then
  if Str2Real(EditCalculatorXLinear.Text,y) then begin
   ops:=ReadTimeStampCounter;
   x:=Calib.LinearX(y,z,false);
   ops:=ReadTimeStampCounter-ops;
   EditCalculatorXNormal.Text:=Format('%g',[x]);
   LabelCalculatorCpuOpCount.Caption:=Format('CPU: %d op.',[ops]);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalculatorX2NormalClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalculatorX2LinearClick(Sender: TObject);
var x,y,z:Double; ops,ops1:Int64;
begin
 if Ok and Calib.Ok then
 try
  if Str2Real(EditCalculatorXYParam.Text,z) then
  if Str2Real(EditCalculatorXNormal.Text,x) then begin
   ops:=ReadTimeStampCounter;
   y:=Calib.LinearX(x,z,true);
   ops:=ReadTimeStampCounter-ops;
   EditCalculatorXLinear.Text:=Format('%g',[y]);
   if CheckBoxCalculatorXY.Checked then begin
    EditCalculatorYLinear.Text:=Format('%g',[y]);
    ops1:=ReadTimeStampCounter;
    x:=Calib.LinearY(y,z,false);
    ops1:=ReadTimeStampCounter-ops1;
    EditCalculatorYNormal.Text:=Format('%g',[x]);
    ops:=ops+ops1;
   end;
   LabelCalculatorCpuOpCount.Caption:=Format('CPU: %d op.',[ops]);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalculatorX2LinearClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalculatorY2NormalClick(Sender: TObject);
var x,y,z:Double; ops:Int64;
begin
 if Ok and Calib.Ok then
 try
  if Str2Real(EditCalculatorXYParam.Text,z) then
  if Str2Real(EditCalculatorYLinear.Text,y) then begin
   ops:=ReadTimeStampCounter;
   x:=Calib.LinearY(y,z,false);
   ops:=ReadTimeStampCounter-ops;
   EditCalculatorYNormal.Text:=Format('%g',[x]);
   LabelCalculatorCpuOpCount.Caption:=Format('CPU: %d op.',[ops]);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalculatorY2NormalClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalculatorY2LinearClick(Sender: TObject);
var x,y,z:Double; ops,ops1:Int64;
begin
 if Ok and Calib.Ok then
 try
  if Str2Real(EditCalculatorXYParam.Text,z) then
  if Str2Real(EditCalculatorYNormal.Text,x) then begin
   ops:=ReadTimeStampCounter;
   y:=Calib.LinearY(x,z,true);
   ops:=ReadTimeStampCounter-ops;
   EditCalculatorYLinear.Text:=Format('%g',[y]);
   if CheckBoxCalculatorXY.Checked then begin
    EditCalculatorXLinear.Text:=Format('%g',[y]);
    ops1:=ReadTimeStampCounter;
    x:=Calib.LinearX(y,z,false);
    ops1:=ReadTimeStampCounter-ops1;
    EditCalculatorXNormal.Text:=Format('%g',[x]);
    ops:=ops+ops1;
   end;
   LabelCalculatorCpuOpCount.Caption:=Format('CPU: %d op.',[ops]);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalculatorY2LinearClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalculatorCopyPointClick(Sender: TObject);
begin
 if Ok then
 try
  EditCalculatorXNormal.Text:=Trim(EditX.Text);
  EditCalculatorYNormal.Text:=Trim(EditY.Text);
  EditCalculatorXYParam.Text:=Trim(EditZ.Text);
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalculatorCopyPointClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalculatorSavePointClick(Sender: TObject);
begin
 if Ok then
 try
  EditX.Text:=Trim(EditCalculatorXNormal.Text);
  EditY.Text:=Trim(EditCalculatorYNormal.Text);
  EditZ.Text:=Trim(EditCalculatorXYParam.Text);
  PageControlCalibration.ActivePage:=TabSheetCalibrationPoints;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalculatorSavePointClick');
  end;
 end;
end;

procedure TFormCalibDialog.BitBtnCalculatorX2YClick(Sender: TObject);
var x,y,z:Double; ops:Int64;
begin
 if Ok and Calib.Ok then
 try
  if Str2Real(EditCalculatorXYParam.Text,z) then
  if Str2Real(EditCalculatorXNormal.Text,x) then begin
   ops:=ReadTimeStampCounter;
   y:=Calib.GetY(x,z);
   ops:=ReadTimeStampCounter-ops;
   EditCalculatorYNormal.Text:=Format('%g',[y]);
   LabelCalculatorCpuOpCount.Caption:=Format('CPU: %d op.',[ops]);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'BitBtnCalculatorX2YClick');
  end;
 end;
end;

procedure TFormCalibDialog.BitBtnCalculatorY2XClick(Sender: TObject);
var x,y,z:Double; ops:Int64;
begin
 if Ok and Calib.Ok then
 try
  if Str2Real(EditCalculatorXYParam.Text,z) then
  if Str2Real(EditCalculatorYNormal.Text,y) then begin
   ops:=ReadTimeStampCounter;
   x:=Calib.GetX(y,z);
   ops:=ReadTimeStampCounter-ops;
   EditCalculatorXNormal.Text:=Format('%g',[x]);
   LabelCalculatorCpuOpCount.Caption:=Format('CPU: %d op.',[ops]);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'BitBtnCalculatorY2XClick');
  end;
 end;
end;

procedure TFormCalibDialog.ButtonCalibrationHelpHtmClick(Sender: TObject);
var s:LongString; FName:LongString;
begin
 if Ok then
 try
  s:='';
  if ReadIniFilePath(SysIniFile,SectSystem,'HelpOnCalibrationSystem',HomeDir,s) then
  if IsNonEmptyStr(s) then begin
   FName:=UnifyFileAlias(s);
   if FileExists(FName) then if not ShellExecuteOpen(FName) then
   if FileExists(FName) then SendToMainConsole('@silent @run '+FName+EOL);
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'ButtonCalibrationHelpHtmClick');
  end;
 end;
end;

procedure TFormCalibDialog.SpinEditPowerChange(Sender: TObject);
var i:Integer;
begin
 if Ok then
 try
  for i:=Low(myCoeff) to High(myCoeff) do if Assigned(myCoeff[i].Edit) then begin
   myCoeff[i].Edit.Enabled:=(i<=SpinEditPower.Value);
   myCoeff[i].Edit.ReadOnly:=not Calib.IsFixed;
  end;
 except
  on E:Exception do begin
   Warning(RusEng('Ошибка калибровки!','Calibration error!'));
   BugReport(E,Self,'SpinEditPowerChange');
  end;
 end;
end;

procedure TFormCalibDialog.TabSheetCalibrationCalculatorEnter(Sender: TObject);
begin
 LabelCalculatorCpuOpCount.Caption:='';
 UpdateControls(uf_Status);
end;

procedure TFormCalibDialog.TabSheetCalibrationPresetEnter(Sender: TObject);
begin
 UpdateControls(uf_Preset);
end;

procedure TFormCalibDialog.TabSheetCalibrationPointsEnter(Sender: TObject);
begin
 UpdateControls(uf_Status+uf_ListBoxPoints+uf_EditXYWZ);
end;

procedure TFormCalibDialog.TabSheetCalibrationHelpEnter(Sender: TObject);
begin
 UpdateControls(uf_Status);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_calibdialog;
begin
end;

procedure Free_form_calibdialog;
begin
end;

initialization

 Init_form_calibdialog;

finalization

 Free_form_calibdialog;

end.

//////////////
// END OF FILE
//////////////

