////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для обработки спектров.
[]
[Manual.Eng]
This utility for spectr analysis.
[]
[Arguments.Rus]
[]
[Arguments.Eng]
[]
}

library rfa_1229;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

 ///////////////////////////////
 // General purpose constants //
 ///////////////////////////////
const
 swin                       = +1;               // Source window reference.
 twin                       = -1;               // Target window reference.
 cwin                       =  0;               // Clipboard window reference.
 CrwApi         : ICrwApi   = nil;              // Points to interface class.
 MaxNumRoi                  = 100;              // Maximal number of ROIs.
 EvalPrecision              = 1E-10;            // Precicion constant for evaluations
 rfa_AtomId_Air             = rfa_MaxAtomId+1;  // To identify AIR.
 alfa                       = 2.77258872223978123766892848583271;   // 4*ln(2).
 ViewBasisFunc  : boolean   = false;            // Create curves to view basis functions

 ///////////////////////////////
 // General purpose functions //
 ///////////////////////////////

 //
 // Procedures to kill object reference
 //
procedure Kill(var obj:TObject); overload; begin FreeAndNil(obj); end;

 //
 // Procedures to add element to vector
 //
procedure AddItem(Vector:ILongIntVectorWrap; Item:LongInt); overload;
begin
 Vector.Length:=Vector.Length+1;
 Vector[Vector.Length-1]:=Item;
end;

procedure AddItem(Vector:IDoubleVectorWrap; Item:Double); overload;
begin
 Vector.Length:=Vector.Length+1;
 Vector[Vector.Length-1]:=Item;
end;

 //
 // To be called from hard calculation procedures to process messages.
 //
procedure Refreshment(Delta:Integer);
const LastTicks : Cardinal = 0;
begin
 if LastTicks=0 then LastTicks:=GetTickCount;
 if abs(GetTickCount-LastTicks) > Delta then begin
  with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
  LastTicks:=GetTickCount;
 end;
end;

 //
 // Extract expression "Name = string" from Source string.
 //
function ExtractString(const Source,Name:LongString):LongString;
var s:ShortString;
begin
 if CrwApi.SysApi.ScanVar(svSpaces+svOrigin,PChar(Source),Name+'%s',s)<>nil then Result:=s else Result:='';
end;

 //
 // Extract expression "Name = integer" from Source string.
 //
function ExtractInteger(const Source,Name:LongString; var I:Integer):boolean;
begin
 Result:=CrwApi.SysApi.ScanVar(svSpaces+svOrigin,PChar(Source),Name+'%i',I)<>nil;
end;

 //
 // Extract expression "Name = double" from Source string.
 //
function ExtractDouble(const Source,Name:LongString; var D:Double):boolean;
begin
 Result:=CrwApi.SysApi.ScanVar(svSpaces+svOrigin,PChar(Source),Name+'%f',D)<>nil;
end;

 //
 // Extract expression "Name = string" from data window comment text.
 //
function GetArgAsString(const Arg:LongString):LongString;
begin
 with CrwApi,SysApi,DanApi do begin
  Result:=GetArgumentAsString(Arg);
  if Length(Result)=0 then Result:=ExtractString(WindowComment[swin],Arg);
 end;
end;

 //
 // Extract expression "Name = integer" from data window comment text.
 //
function GetArgAsInteger(const Arg:LongString; var I:Integer):boolean;
begin
 with CrwApi,SysApi,DanApi do begin
  Result:=GetArgumentAsInteger(Arg,I);
  if not Result then Result:=ExtractInteger(WindowComment[swin],Arg,I);
 end;
end;

 //
 // Extract expression "Name = double" from data window comment text.
 //
function GetArgAsDouble(const Arg:LongString; var D:Double):boolean;
begin
 with CrwApi,SysApi,DanApi do begin
  Result:=GetArgumentAsDouble(Arg,D);
  if not Result then Result:=ExtractDouble(WindowComment[swin],Arg,D);
 end;
end;

 //
 // Return atom symbol by given atom id.
 //
function RfaAtomSymbol(AtomId:Integer):ShortString;
begin
 with CrwApi,SysApi do
 case AtomId of
  rfa_MinAtomId..rfa_MaxAtomId: Result:=Rfa.AtomSymbol[AtomId];
  rfa_AtomId_Air:               Result:='AIR';
  else                          Result:='NONE';
 end;
end;

 //
 // Return atom id by given atom symbol.
 //
function RfaFindAtom(const AtomSymbol:ShortString):Integer;
begin
 with CrwApi,SysApi do
 if UnifyAlias(AtomSymbol)=UnifyAlias('AIR')
 then Result:=rfa_AtomId_Air
 else Result:=Rfa.FindAtom(AtomSymbol);
end;

 //
 // Find index of curve in source/target window by name.
 //
function CurveIndexByName(const Name:ShortString; win:Integer=swin):Integer;
var i:integer;
begin
 Result:=0;
 with CrwApi,SysApi,DanApi do
 case win of
  swin: for i:=1 to CurvesCount[swin] do if UnifyAlias(Name)=UnifyAlias(CurveName[+i]) then begin
         Result:=+i;
         Break;
        end;
  twin: for i:=1 to CurvesCount[twin] do if UnifyAlias(Name)=UnifyAlias(CurveName[-i]) then begin
         Result:=-i;
         Break;
        end;
 end;
end;

 //
 // Single Gauss peak with given amplitude.
 //
function GaussPeakByAmpl(Channel,Center,Ampl,FWHM:Double):Double;
var t : Extended;
begin
 t:=alfa*sqr((Channel-Center)/FWHM);
 if t>200 then Result:=0 else Result:=Ampl*exp(-t);
end;

 //
 // Single Gauss peak with given area.
 //
function GaussPeakByArea(Channel,Center,Area,FWHM:Double):Double;
var t : Extended;
begin
 t:=alfa*sqr((Channel-Center)/FWHM);
 if t>200 then Result:=0 else Result:=Area*sqrt(alfa/pi)*exp(-t)/FWHM;
end;

 //
 // Найти коэффициент флюоресцентного выхода "омега" (фактор эффективности) 8/5435, 1979, стр.7
 //
function RfaWhat_Omega(AtomId:Integer;LineId:TRfaFieldId; Energy:double):double;
begin
 Result:=0;
 with CrwApi,SysApi do
 if Rfa.Accessible[AtomId] then begin
  if LineId in rfa_K_Series then begin
   if Energy >= Rfa[AtomId,rfa_EKAB] then Result:=Rfa[AtomId,rfa_KEMISSION];
  end else
  if LineId in rfa_L2_Series then begin
   if (Rfa[AtomId,rfa_EL2AB] <= Energy) and (Energy < Rfa[AtomId,rfa_EL1AB]) then Result:=Rfa[AtomId,rfa_L22EMISSION];
   if (Rfa[AtomId,rfa_EL1AB] <= Energy) then Result:=Rfa[AtomId,rfa_L21EMISSION];
  end else
  if LineId in rfa_L3_Series then begin
   if (Rfa[AtomId,rfa_EL3AB] <= Energy) and (Energy < Rfa[AtomId,rfa_EL2AB]) then Result:=Rfa[AtomId,rfa_L33EMISSION];
   if (Rfa[AtomId,rfa_EL2AB] <= Energy) and (Energy < Rfa[AtomId,rfa_EL1AB]) then Result:=Rfa[AtomId,rfa_L32EMISSION];
   if (Rfa[AtomId,rfa_EL1AB] <= Energy) then Result:=Rfa[AtomId,rfa_L31EMISSION];
  end;
 end;
end;

 ////////////////////////////////////////////////////////////////////////////
 // ROI, Region Of Interest -> range of channels and polynom of background //
 ////////////////////////////////////////////////////////////////////////////
type
 TRoiItem = object
 public
  Left  : Integer;                          // Left  edge of ROI
  Right : Integer;                          // Right edge of ROI
  Power : Integer;                          // Polynom power
  Fixed : Integer;                          // Fixed polynom coefficients as bits
  Coeff : packed array[0..9] of Double;     // Polynom coefficients
  function IsFixed(n:Integer):Boolean;          // Is Coeff[n] fixed?
  function Contains(c:Double):Boolean;          // ROI contains given channel c?
  function Scaled(c:Double):Double;         // Evaluate scaled channel c
  function Ground(c:Double):Double;         // Evaluate ground polynom at channel c
  function GroundFixed(c:Double):Double;        // Evaluate ground polynom, fixed coefficients only
  function BasisFun(n:Integer;c:Double):Double; // Evaluate basis function
 end;

function TRoiItem.IsFixed(n:Integer):Boolean;
begin
 Result:=((Fixed and (1 shl n)) <> 0);
end;

function TRoiItem.Contains(c:Double):Boolean;
begin
 Result:=(c>=Left) and (c<=Right);
end;

function TRoiItem.Scaled(c:Double):Double;
begin
 Result:=(c-Left)/(Right-Left);
end;

function TRoiItem.Ground(c:Double):Double;
begin
 if Contains(c)
 then Result:=CrwApi.SysApi.EvaluatePolynomAt(Coeff[0],Power+1,Scaled(c))
 else Result:=0;
end;

function TRoiItem.GroundFixed(c:Double):Double;
var cf : packed array[0..9] of Double; i:integer;
begin
 if Contains(c) then begin
  for i:=0 to Power do cf[i]:=Coeff[i]*ord(IsFixed(i));
  Result:=CrwApi.SysApi.EvaluatePolynomAt(cf[0],Power+1,Scaled(c));
 end else Result:=0;
end;

function TRoiItem.BasisFun(n:Integer;c:Double):Double;
begin
 if Contains(c)
 then Result:=IntPower(Scaled(c),n)
 else Result:=0;
end;

 ////////////////////////////////////////////////////////////////////////////////////
 // Spectr definition -> curve in source window, En and Hw calibrations as polynom //
 ////////////////////////////////////////////////////////////////////////////////////
type
 TSpectrItem    = object
 public
  CurveIndex    : Integer;              // Index of spectrum curve in source window
  SourceFile    : ShortString;          // Source file to read spectrum from
  EnCal,HwCal   : packed record         // Energy and half width calibration
   Power        : Integer;              // Polynom power
   Center       : Double;               // Polynom center
   Scale        : Double;               // Polynom scale
   Coeff        : array[0..9] of Double;    // Polinom coefficients
  end;
  function  Length:Integer;             // Spectr length
  function  Energy(Chan:Integer):Double;    // Enegry, keV, in channel Chan
  function  Intens(Chan:Integer):Double;    // Intensity, counts, in channel Chan
  function  Weight(Chan:Integer):Double;    // Statistical weight factor 1/N, in channel Chan
  function  EnCalibr(Chan:Double):Double;   // Evaluate energy calibration
  function  HwCalibr(Chan:Double):Double;   // Evaluate half width calibration
  function  EnCalibrInv(En:Double):Double;  // Evaluate inverted energy calibration
  function  IsOkEnCalibr:Boolean;           // Check if Ok with energy calibration, must be Energy(i)=EnCalibr(i)
 end;

function TSpectrItem.Length:Integer;
begin
 if CurveIndex>0 then Result:=CrwApi.DanApi.CurveLength[CurveIndex] else Result:=0;
end;

function TSpectrItem.Energy(Chan:Integer):Double;
begin
 if CurveIndex>0 then Result:=CrwApi.DanApi.CurvePoint[CurveIndex,Chan].X else Result:=0;
end;

function TSpectrItem.Intens(Chan:Integer):Double;
begin
 if CurveIndex>0 then Result:=CrwApi.DanApi.CurvePoint[CurveIndex,Chan].Y else Result:=0;
end;

function TSpectrItem.Weight(Chan:Integer):Double;
begin
 if CurveIndex>0 then Result:=1/max(1,CrwApi.DanApi.CurvePoint[CurveIndex,Chan].Y) else Result:=0;
end;

function TSpectrItem.EnCalibr(Chan:Double):Double;
begin
 with EnCal do Result:=CrwApi.SysApi.EvaluatePolynomAt(Coeff[0],Power+1,(Chan-Center)/Scale);
end;

function TSpectrItem.HwCalibr(Chan:Double):Double;
begin
 with HwCal do Result:=CrwApi.SysApi.EvaluatePolynomAt(Coeff[0],Power+1,(Chan-Center)/Scale);
end;

type
 TInvEnCalibrRec = packed record
  Spectr : ^TSpectrItem;
  Energy : Double;
 end;

function EnCalibrInvObjective(x:Double; Custom:Pointer):Double;
begin
 with TInvEnCalibrRec(Custom^) do Result:=Spectr.EnCalibr(x)-Energy;
end;

function TSpectrItem.EnCalibrInv(En:Double):Double;
var R:TInvEnCalibrRec;
begin
 R.Energy:=En;
 R.Spectr:=@Self;
 Result:=CrwApi.SysApi.FindZero(EnCalibrInvObjective,0,Length-1,EvalPrecision,@R);
end;

function TSpectrItem.IsOkEnCalibr:Boolean;
var i:Integer;
begin
 Result:=true;
 for i:=0 to Length-1 do begin
  if abs(Energy(i)-EnCalibr(i))>EvalPrecision*(abs(Energy(i))+abs(EnCalibr(i))) then begin
   Result:=false;
   break;
  end;
 end;
end;

 ////////////////////////////////////////////
 // Rfa data sheet to keep main parameters //
 ////////////////////////////////////////////
type
 TRfaDataSheet = object
 public
  SelElement     : Integer;                     // Selected element
  SelElements    : TByteSet;                    // Set of selected elements
  ElementList    : packed record                // List of selected elements
   AtomId        : ILongIntVectorWrap;          // Atom identifier
   Content       : IDoubleVectorWrap;           // Atom mass concentration
   ContentError  : IDoubleVectorWrap;           // Atom mass concentration error eastimation
  end;                                          //
  ParamList      : packed record                // List of parameters type
   Elem          : ILongIntVectorWrap;          // Index of element in ElementList or  -1
   Roi           : ILongIntVectorWrap;          // Index of element in RoiList or  -1
   Pow           : ILongIntVectorWrap;          // Polynom power or -1
  end;
  SelLines       : TRfaFieldIdSet;              // Set of selected X ray lines
  Detector       : packed record                // Detector parameters
   Allow         : Integer;                     // Enable detector
   Element       : Integer;                     // Element of detector
   WorkThickness : Double;                      // Work thichness of detector
   DeadThickness : Double;                      // Dead thichness of detector
   AirDensity    : Double;                      // Air density
   Layer         : packed array[1..4] of packed record  // List of detector material layers
    Element      : Integer;                     // Element of layer
    Thickness    : Double;                      // Thithkness of layer
   end;                                         //
   Angles        : packed record                // Detector angles
    Alpha        : Double;                      // Hade angle
    Beta         : Double;                      // Reflexion angle
   end;                                         //
  end;                                          //
  RoiList        : array of TRoiItem;           // List of ROIs
  RoiNumByChan   : ILongIntVectorWrap;          // Contains ROI number by channel or -1 if out of ROI
  ExciSpectr     : TSpectrItem;                 // Excitation spectrum description
  TargSpectr     : array of TSpectrItem;        // Target spectrums desctiption
  Lsq            : packed record                // Least squares algorithm data
   Plane         : IDoubleMatrixWrap;           // Plane matrix for least squares algorithm
   RSide         : IDoubleVectorWrap;           // Need for least squares algorithm
   Weight        : IDoubleVectorWrap;           // Stat. weights for least squares algorithm
   A             : IDoubleMatrixWrap;           // Matrix for linear equition A*X=Y
   X             : IDoubleVectorWrap;           // Parameters to be found
   Y             : IDoubleVectorWrap;           // Right side vector for linear equition A*X=Y
   SVD           : ISVDWrap;                    // Singular decomposition of A
   B             : IDoubleMatrixWrap;           // Inverted matrix A
   NumFreeVars   : Integer;                     // Number of free variables
   Chi2          : Double;                      // Chi^2 criteria
   Chi2Crit      : Double;                      // Chi^2(0.95,NumFreeVars);
  end;
  procedure InitDataSheet;                      // Initialize, alocate memory
  procedure FreeDataSheet;                      // Finalize, free memory
  procedure ReadDataSheet;                      // Read all parameters stored in source window
  procedure EchoDataSheet(var F:Text);          // Echo parameters which was read by ReadDataSheet
  function  GetDetectorEfficiency(Energy:Double):Double;    // Find detector efficiency for given energy
  function  GetRfaXRayLine(AtomID:Integer; LineID:TRfaFieldId; EnMax:Double):TRfaXRayLine;  //
  function  CalcElemSpectr(TargetIndex,Channel,ElementIndex:Integer):Double;                // Calculate single element spectrum
  function  CalcBasisFun(TargetIndex,Channel,ParamIndex:Integer):Double;                // Calculate basis function
  procedure CalcLsq(TargetIndex:Integer);       // Calculate & solve least squares linear equiations
  procedure EchoLsq(TargetIndex:Integer);       // Echo result of calculations
  procedure MakeAnalysis;                       //
 end;

const
 RfaDataSheet : ^TRfaDataSheet = nil;

procedure TRfaDataSheet.InitDataSheet;
begin
 with CrwApi,SysApi do begin
  RfaDataSheet:=@Self;
  FillChar(Self,sizeof(Self),0);
  SelElement:=0;
  SelElements:=[];
  ElementList.AtomId:=CreateLongIntVector(0);
  ElementList.Content:=CreateDoubleVector(0);
  ElementList.ContentError:=CreateDoubleVector(0);
  ParamList.Elem:=CreateLongIntVector(0);
  ParamList.Roi:=CreateLongIntVector(0);
  ParamList.Pow:=CreateLongIntVector(0);
  SelLines:=[];
  Detector.Allow:=0;
  Detector.Element:=0;
  Detector.WorkThickness:=0;
  Detector.DeadThickness:=0;
  Detector.AirDensity:=0;
  Detector.Layer[1].Element:=0;
  Detector.Layer[1].Thickness:=0;
  Detector.Layer[2].Element:=0;
  Detector.Layer[2].Thickness:=0;
  Detector.Layer[3].Element:=0;
  Detector.Layer[3].Thickness:=0;
  Detector.Layer[4].Element:=0;
  Detector.Layer[4].Thickness:=0;
  Detector.Angles.Alpha:=0;
  Detector.Angles.Beta:=0;
  SetLength(RoiList,0);
  RoiNumByChan:=CreateLongIntVector(0);
  FillChar(ExciSpectr,sizeof(ExciSpectr),0);
  SetLength(TargSpectr,0);
  Lsq.Plane:=CreateDoubleMatrix(0,0);
  Lsq.RSide:=CreateDoubleVector(0);
  Lsq.Weight:=CreateDoubleVector(0);
  Lsq.A:=CreateDoubleMatrix(0,0);
  Lsq.X:=CreateDoubleVector(0);
  Lsq.Y:=CreateDoubleVector(0);
  Lsq.SVD:=nil;
  Lsq.B:=nil;
 end;
end;

procedure TRfaDataSheet.FreeDataSheet;
begin
 with CrwApi,SysApi do begin
  Kill(ElementList.AtomId);
  Kill(ElementList.Content);
  Kill(ElementList.ContentError);
  Kill(ParamList.Elem);
  Kill(ParamList.Roi);
  Kill(ParamList.Pow);
  SetLength(RoiList,0);
  Kill(RoiNumByChan);
  SetLength(TargSpectr,0);
  Kill(Lsq.Plane);
  Kill(Lsq.RSide);
  Kill(Lsq.Weight);
  Kill(Lsq.A);
  Kill(Lsq.X);
  Kill(Lsq.Y);
  Kill(Lsq.SVD);
  Kill(Lsq.B);
  RfaDataSheet:=nil;
 end;
end;

procedure TRfaDataSheet.ReadDataSheet;
var i,j,AtomId:Integer; LineId:TRfaFieldId;
begin
 with CrwApi,SysApi,GuiApi,DanApi do begin
  SelElement:=RfaFindAtom(GetArgAsString('SelectedElement'));
  SelElements:=[];
  for AtomId:=rfa_MinAtomId to rfa_MaxAtomId do
  if GetArgAsInteger('SelectedElements['+Rfa.AtomSymbol[AtomId]+']',j) and (j=1) then begin
   include(SelElements,AtomId);
   AddItem(ElementList.AtomId,AtomId);
   AddItem(ElementList.Content,0);
   AddItem(ElementList.ContentError,0);
   AddItem(ParamList.Elem,ElementList.AtomId.Length-1);
   AddItem(ParamList.Roi,-1);
   AddItem(ParamList.Pow,-1);
  end;
  if ElementList.AtomId.Length<2
  then Raise EDanApi.Create('Number of selected elements too small!');
  SelLines:=[];
  for LineId:=Low(LineId) to High(LineId) do
  if LineId in rfa_All_Series then
  if GetArgAsInteger('SelectedXRayLines['+Rfa.LineName[LineId]+']',j) and (j=1)
  then include(SelLines,LineId);
  GetArgAsInteger('Detector.Allow',Detector.Allow);
  Detector.Element:=RfaFindAtom(GetArgAsString('Detector.Element'));
  GetArgAsDouble('Detector.WorkThickness',Detector.WorkThickness);
  GetArgAsDouble('Detector.DeadThickness',Detector.DeadThickness);
  GetArgAsDouble('Detector.AirDensity',   Detector.AirDensity);
  Detector.Layer[1].Element:=RfaFindAtom(GetArgAsString('Detector.Layer[1].Element'));
  GetArgAsDouble('Detector.Layer[1].Thickness',Detector.Layer[1].Thickness);
  Detector.Layer[2].Element:=RfaFindAtom(GetArgAsString('Detector.Layer[2].Element'));
  GetArgAsDouble('Detector.Layer[2].Thickness',Detector.Layer[2].Thickness);
  Detector.Layer[3].Element:=RfaFindAtom(GetArgAsString('Detector.Layer[3].Element'));
  GetArgAsDouble('Detector.Layer[3].Thickness',Detector.Layer[3].Thickness);
  Detector.Layer[4].Element:=RfaFindAtom(GetArgAsString('Detector.Layer[4].Element'));
  GetArgAsDouble('Detector.Layer[4].Thickness',Detector.Layer[4].Thickness);
  GetArgAsDouble('Detector.Angles.Alpha', Detector.Angles.Alpha);
  GetArgAsDouble('Detector.Angles.Beta',  Detector.Angles.Beta);
  if GetArgAsInteger('ROI.Count',j) then begin
   SetLength(ROIList,j);
   for i:=0 to Length(ROIList)-1 do begin
    GetArgAsInteger(Format('ROI[%d].Left',[i]),RoiList[i].Left);
    GetArgAsInteger(Format('ROI[%d].Right',[i]),RoiList[i].Right);
    GetArgAsInteger(Format('ROI[%d].Power',[i]),RoiList[i].Power);
    GetArgAsInteger(Format('ROI[%d].Fixed',[i]),RoiList[i].Fixed);
    if(RoiList[i].Left<0) or (RoiList[i].Right<RoiList[i].Left)
    then Raise EDanApi.Create(Format('Invalid ROI[%d] range [%d,%d]',[i,RoiList[i].Left,RoiList[i].Right]));
    if(RoiList[i].Power<0) or (RoiList[i].Power>9)
    then Raise EDanApi.Create(Format('Invalid ROI[%d] power %d',[i,RoiList[i].Power]));
    for j:=0 to RoiList[i].Power do begin
     GetArgAsDouble(Format('ROI[%d].Coeff[%d]',[i,j]),RoiList[i].Coeff[j]);
     if not RoiList[i].IsFixed(j) then begin
      AddItem(ParamList.Elem,-1);
      AddItem(ParamList.Roi,i);
      AddItem(ParamList.Pow,j);
     end;
    end;
   end;
  end;
  Lsq.Plane.Cols:=ParamList.Elem.Length;
  with ExciSpectr do begin
   CurveIndex:=CurveIndexByName('ExcitationSpectrum');
   if CurveIndex=0 then Raise EDanApi.Create('ExcitationSpectrum curve not found in source window!');
   SourceFile:=ExtractString(CurveComment[CurveIndex],'SourceFile');
   ExtractInteger(CurveComment[CurveIndex],'EnergyCalibration.Power',EnCal.Power);
   ExtractDouble(CurveComment[CurveIndex],'EnergyCalibration.Center',EnCal.Center);
   ExtractDouble(CurveComment[CurveIndex],'EnergyCalibration.Scale',EnCal.Scale);
   for j:=0 to EnCal.Power do
   ExtractDouble(CurveComment[CurveIndex],Format('EnergyCalibration.Coeff[%d]',[j]),EnCal.Coeff[j]);
   ExtractInteger(CurveComment[CurveIndex],'HalfWidthCalibration.Power',HwCal.Power);
   ExtractDouble(CurveComment[CurveIndex],'HalfWidthCalibration.Center',HwCal.Center);
   ExtractDouble(CurveComment[CurveIndex],'HalfWidthCalibration.Scale',HwCal.Scale);
   for j:=0 to HwCal.Power do
   ExtractDouble(CurveComment[CurveIndex],Format('HalfWidthCalibration.Coeff[%d]',[j]),HwCal.Coeff[j]);
   if not IsOkEnCalibr then Raise EDanApi.Create('Invalid energy calibration of excitation spectrum curve.');
  end;
  for i:=0 to MaxInt do begin
   j:=CurveIndexByName(Format('TargetSpectrum[%d]',[i]));
   if j=0 then Break;
   SetLength(TargSpectr,Length(TargSpectr)+1);
   with TargSpectr[Length(TargSpectr)-1] do begin
    CurveIndex:=j;
    SourceFile:=ExtractString(CurveComment[CurveIndex],'SourceFile');
    ExtractInteger(CurveComment[CurveIndex],'EnergyCalibration.Power',EnCal.Power);
    ExtractDouble(CurveComment[CurveIndex],'EnergyCalibration.Center',EnCal.Center);
    ExtractDouble(CurveComment[CurveIndex],'EnergyCalibration.Scale',EnCal.Scale);
    for j:=0 to EnCal.Power do
    ExtractDouble(CurveComment[CurveIndex],Format('EnergyCalibration.Coeff[%d]',[j]),EnCal.Coeff[j]);
    ExtractInteger(CurveComment[CurveIndex],'HalfWidthCalibration.Power',HwCal.Power);
    ExtractDouble(CurveComment[CurveIndex],'HalfWidthCalibration.Center',HwCal.Center);
    ExtractDouble(CurveComment[CurveIndex],'HalfWidthCalibration.Scale',HwCal.Scale);
    for j:=0 to HwCal.Power do
    ExtractDouble(CurveComment[CurveIndex],Format('HalfWidthCalibration.Coeff[%d]',[j]),HwCal.Coeff[j]);
    RoiNumByChan.Length:=max(RoiNumByChan.Length,CurveLength[CurveIndex]);
    Lsq.Plane.Rows:=max(Lsq.Plane.Rows,CurveLength[CurveIndex]);
    Lsq.RSide.Length:=Lsq.Plane.Rows;
    Lsq.Weight.Length:=Lsq.Plane.Rows;
    if not IsOkEnCalibr then Raise EDanApi.Create(Format('Invalid energy calibration of target[%d] spectrum curve.',[i]));
   end;
  end;
  for i:=0 to RoiNumByChan.Length-1 do begin
   RoiNumByChan[i]:=-1;
   for j:=0 to Length(RoiList)-1 do
   if RoiList[j].Contains(i) then begin
    RoiNumByChan[i]:=j;
    Break;
   end;
  end;
  if Length(TargSpectr)=0 then Raise EDanApi.Create('TargetSpectrum curve(s) not found in source window!');
 end;
end;

procedure TRfaDataSheet.EchoDataSheet(var F:Text);
var i,j,AtomId:Integer; LineId:TRfaFieldId;
begin
 with CrwApi,SysApi,GuiApi,DanApi do begin
   Writeln(F,'[X-Ray Fluorescence Analysis Job parameters readback]');
   Writeln(F,Format('SelectedElement = %s',[RfaAtomSymbol(SelElement)]));
   for i:=0 to ElementList.AtomId.Length-1 do Writeln(F,Format('ElementList[%d] = %s',[i,RfaAtomSymbol(ElementList.AtomId[i])]));
   for AtomId:=rfa_MinAtomId to rfa_MaxAtomId do
   if AtomId in SelElements then Writeln(F,Format('SelectedElements[%s] = 1',[RfaAtomSymbol(AtomId)]));
   for LineId:=Low(LineId) to High(LineId) do
   if LineId in SelLines then Writeln(F,Format('SelectedXRayLines[%s] = 1',[Rfa.LineName[LineId]]));
   Writeln(F,Format('Detector.Allow = %d',[Detector.Allow]));
   Writeln(F,Format('Detector.Element = %s',[RfaAtomSymbol(Detector.Element)]));
   Writeln(F,Format('Detector.WorkThickness = %g',[Detector.WorkThickness]));
   Writeln(F,Format('Detector.DeadThickness = %g',[Detector.DeadThickness]));
   Writeln(F,Format('Detector.AirDensity = %g',[Detector.AirDensity]));
   Writeln(F,Format('Detector.Layer[1].Element = %s',[RfaAtomSymbol(Detector.Layer[1].Element)]));
   Writeln(F,Format('Detector.Layer[1].Thickness = %g',[Detector.Layer[1].Thickness]));
   Writeln(F,Format('Detector.Layer[2].Element = %s',[RfaAtomSymbol(Detector.Layer[2].Element)]));
   Writeln(F,Format('Detector.Layer[2].Thickness = %g',[Detector.Layer[2].Thickness]));
   Writeln(F,Format('Detector.Layer[3].Element = %s',[RfaAtomSymbol(Detector.Layer[3].Element)]));
   Writeln(F,Format('Detector.Layer[3].Thickness = %g',[Detector.Layer[3].Thickness]));
   Writeln(F,Format('Detector.Layer[4].Element = %s',[RfaAtomSymbol(Detector.Layer[4].Element)]));
   Writeln(F,Format('Detector.Layer[4].Thickness = %g',[Detector.Layer[4].Thickness]));
   Writeln(F,Format('Detector.Angles.Alpha = %g',[Detector.Angles.Alpha]));
   Writeln(F,Format('Detector.Angles.Beta = %g',[Detector.Angles.Beta]));
   Writeln(F,Format('ROI.Count = %d',[Length(RoiList)]));
   for i:=0 to Length(RoiList)-1 do begin
    Writeln(F,Format('ROI[%d].Left = %d',[i,RoiList[i].Left]));
    Writeln(F,Format('ROI[%d].Right = %d',[i,RoiList[i].Right]));
    Writeln(F,Format('ROI[%d].Power = %d',[i,RoiList[i].Power]));
    Writeln(F,Format('ROI[%d].Fixed = %d',[i,RoiList[i].Fixed]));
    for j:=0 to RoiList[i].Power do Writeln(F,Format('ROI[%d].Coeff[%d] = %g',[i,j,RoiList[i].Coeff[j]]));
   end;
   with ExciSpectr do begin
    Writeln(F,Format('ExcitationSpectrum = SourceCurve[%d]',[CurveIndex]));
    Writeln(F,Format('SourceFile = %s',[SourceFile]));
    Writeln(F,Format('EnCal.Power = %d',[EnCal.Power]));
    Writeln(F,Format('EnCal.Center = %g',[EnCal.Center]));
    Writeln(F,Format('EnCal.Scale = %g',[EnCal.Scale]));
    for j:=0 to EnCal.Power do
    Writeln(F,Format('EnCal.Coeff[%d] = %g',[j,EnCal.Coeff[j]]));
    Writeln(F,Format('HwCal.Power = %d',[HwCal.Power]));
    Writeln(F,Format('HwCal.Center = %g',[HwCal.Center]));
    Writeln(F,Format('HwCal.Scale = %g',[HwCal.Scale]));
    for j:=0 to HwCal.Power do
    Writeln(F,Format('HwCal.Coeff[%d] = %g',[j,HwCal.Coeff[j]]));
   end;
   for i:=0 to Length(TargSpectr)-1 do
   with TargSpectr[i] do begin
    Writeln(F,Format('TargetSpectrum[%d] = SourceCurve[%d]',[i,CurveIndex]));
    Writeln(F,Format('SourceFile = %s',[SourceFile]));
    Writeln(F,Format('EnCal.Power = %d',[EnCal.Power]));
    Writeln(F,Format('EnCal.Center = %g',[EnCal.Center]));
    Writeln(F,Format('EnCal.Scale = %g',[EnCal.Scale]));
    for j:=0 to EnCal.Power do
    Writeln(F,Format('EnCal.Coeff[%d] = %g',[j,EnCal.Coeff[j]]));
    Writeln(F,Format('HwCal.Power = %d',[HwCal.Power]));
    Writeln(F,Format('HwCal.Center = %g',[HwCal.Center]));
    Writeln(F,Format('HwCal.Scale = %g',[HwCal.Scale]));
    for j:=0 to HwCal.Power do
    Writeln(F,Format('HwCal.Coeff[%d] = %g',[j,HwCal.Coeff[j]]));
   end;
 end;
end;

function TRfaDataSheet.GetDetectorEfficiency(Energy:Double):Double;
var
 i : Integer;
 { найти затухание слоя из элемента AtomID толщины Thickness }
 function LayerFactor(AtomID:Byte; Thickness:Double):Double;
 var
  Mu      : Double;
  Lambda  : Double;
  Density : Double;
 begin
  Result:=1;
  with CrwApi,SysApi,DanApi do
  case AtomID of
   rfa_MinAtomId..rfa_MaxAtomId :
    if Rfa.Accessible[AtomID] then begin
     Mu:=Rfa.FindCrossSection(AtomID,rfa_TOTAL,Energy);
     Density:=Rfa[AtomID,rfa_ATOM_DENSITY];
     Lambda:= - Mu * Density * Thickness;
     if Lambda<-300 then Result:=0 else Result:=exp(Lambda);
    end;
   rfa_AtomId_Air:
    if Rfa.Accessible[ChemEl_N] and Rfa.Accessible[ChemEl_O] then begin
     Mu:=0.77*Rfa.FindCrossSection(ChemEl_N,rfa_TOTAL,Energy)+
         0.23*Rfa.FindCrossSection(ChemEl_O,rfa_TOTAL,Energy);
     Density:=Detector.AirDensity;
     Lambda:= - Mu * Density * Thickness;
     if Lambda<-300 then Result:=0 else Result:=exp(Lambda);
    end;
  end;
 end;
begin
 Result:=1;
 {если не надо учитывать эффективность,возвратим 1}
 with Detector do if Allow<>0 then begin
  {учтем затухание на каждом слое}
  for i:=Low(Layer) to High(Layer) do
  Result:=Result*LayerFactor(Layer[i].Element,Layer[i].Thickness);
  {учтем затухание и мертвую зону детектора}
  Result:=Result*(1-LayerFactor(Element,WorkThickness));
  Result:=Result*LayerFactor(Element,DeadThickness);
 end;
end;

function TRfaDataSheet.GetRfaXRayLine(AtomID:Integer; LineID:TRfaFieldId; EnMax:Double):TRfaXRayLine;
begin
 Result.Energy:=0;
 Result.Height:=0;
 with CrwApi,SysApi,DanApi do
 if Rfa.Accessible[AtomID] and (LineID in rfa_All_Series*SelLines) then begin
  Result:=Rfa.XRayLine[AtomID,LineID];
  Result.Height:=Result.Height*max(0,RfaWhat_Omega(AtomID,LineID,EnMax));
  Result.Height:=Result.Height*Rfa.FindCrossSection(AtomID,rfa_PHOTO,EnMax);
  Result.Height:=Result.Height*GetDetectorEfficiency(Result.Energy);
 end;
end;

function TRfaDataSheet.CalcElemSpectr(TargetIndex,Channel,ElementIndex:Integer):Double;
var AtomId,i:Integer; LineId:TRfaFieldId; Line:TRfaXRayLine;
    E1,E2,Ec,dE,S1,S2,Sc,Center,Fwhm,Area,Shape:Double;
begin
 Result:=0;
 if RoiNumByChan[Channel]>=0 then
 with CrwApi,SysApi,DanApi do begin
  for i:=0 to ExciSpectr.Length-2 do begin
   S1:=ExciSpectr.Intens(i);
   S2:=ExciSpectr.Intens(i+1);
   Sc:=S1+0.5*(S2-S1);
   if Sc>0 then begin
    E1:=ExciSpectr.Energy(i);
    E2:=ExciSpectr.Energy(i+1);
    Ec:=E1+0.5*(E2-E1);
    dE:=E2-E1;
    AtomId:=ElementList.AtomId[ElementIndex];
    for LineId:=Low(LineId) to High(LineId) do
    if LineID in rfa_All_Series*SelLines then begin
     Line:=GetRfaXRayLine(AtomId,LineId,Ec);
     if (Line.Energy>0) and (Line.Height>0) then begin
      Center:=TargSpectr[TargetIndex].EnCalibrInv(Line.Energy);
      Area:=Line.Height;
      Fwhm:=TargSpectr[TargetIndex].HwCalibr(Center);
      Shape:=GaussPeakByArea(Channel,Center,Area,Fwhm);
      if Shape>0 then Result:=Result+Shape*Sc*dE;
     end;
    end;
   end;
  end;
 end;
end;

function TRfaDataSheet.CalcBasisFun(TargetIndex,Channel,ParamIndex:Integer):Double;
begin
 Result:=0;
 if RoiNumByChan[Channel]>=0 then
 with CrwApi,SysApi,DanApi do begin
  if ParamList.Elem[ParamIndex]>=0
  then Result:=CalcElemSpectr(TargetIndex,Channel,ParamList.Elem[ParamIndex])
  else if ParamList.Roi[ParamIndex]>=0
  then Result:=RoiList[ParamList.Roi[ParamIndex]].BasisFun(ParamList.Pow[ParamIndex],Channel);
 end;
 Refreshment(10);
end;

procedure TRfaDataSheet.CalcLsq(TargetIndex:Integer);
var i,j,k,n,m,rn:Integer; Ei,Ni,Mi,Ti,Norm:Double;
begin
 with CrwApi,SysApi,GuiApi,DanApi do begin
  //
  // Calculate plane matrix and right side vector
  //
  for i:=0 to Lsq.Plane.Rows-1 do
  for j:=0 to Lsq.Plane.Cols-1 do Lsq.Plane[i,j]:=CalcBasisFun(TargetIndex,i,j);
  for i:=0 to Lsq.RSide.Length-1 do begin
   Lsq.RSide[i]:=0;
   rn:=RoiNumByChan[i];
   if rn>=0 then Lsq.RSide[i]:=TargSpectr[TargetIndex].Intens(i)-RoiList[rn].GroundFixed(i);
  end;
  for i:=0 to Lsq.Weight.Length-1 do Lsq.Weight[i]:=TargSpectr[TargetIndex].Weight(i);
  //
  // Prepare linear equations for least squares algorithm
  //
  Lsq.A.Cols:=Lsq.Plane.Cols;
  Lsq.A.Rows:=Lsq.A.Cols;
  Lsq.X.Length:=Lsq.A.Cols;
  Lsq.Y.Length:=Lsq.A.Cols;
  for i:=0 to Lsq.A.Rows-1 do begin
   for j:=0 to Lsq.A.Cols-1 do begin
    Lsq.A[i,j]:=0;
    for k:=0 to Lsq.Plane.Rows-1 do Lsq.A[i,j]:=Lsq.A[i,j]+Lsq.Plane[k,i]*Lsq.Plane[k,j]*Lsq.Weight[k];
   end;
   Lsq.Y[i]:=0;
   for k:=0 to Lsq.Plane.Rows-1 do Lsq.Y[i]:=Lsq.Y[i]+Lsq.Plane[k,i]*Lsq.RSide[k]*Lsq.Weight[k];
  end;
  //
  // Solve linear equations using singular values decomposition
  //
  Kill(Lsq.SVD);
  Lsq.SVD:=CreateSingularDecomposition(Lsq.A);
  if not Lsq.SVD.Solve(Lsq.X,Lsq.Y) then Raise EDanApi.Create('Could not solve linear equitions!');
  Kill(Lsq.B);
  Lsq.B:=Lsq.SVD.PInvert;
  //
  // Make corrections for unfixed ROI ground polynom coefficients
  //
  for i:=0 to ParamList.Roi.Length-1 do
  if (ParamList.Roi[i]>=0) and (ParamList.Pow[i]>=0) then RoiList[ParamList.Roi[i]].Coeff[ParamList.Pow[i]]:=Lsq.X[i];
  //
  // Calculate concentrations, normalize to 1
  //
  Norm:=0;
  for i:=0 to ElementList.AtomId.Length-1 do Norm:=Norm+Lsq.X[i];
  for i:=0 to ElementList.AtomId.Length-1 do ElementList.Content[i]:=Lsq.X[i]/Norm;
  //
  // Calculate theoretical spectrum fit curve and background curve
  //
  CurvesCount[twin]:=CurvesCount[twin]+1;
  n:=-CurvesCount[twin];
  CurvesCount[twin]:=CurvesCount[twin]+1;
  m:=-CurvesCount[twin];
  CurveLength[n]:=TargSpectr[TargetIndex].Length;
  CurveLength[m]:=TargSpectr[TargetIndex].Length;
  CurveStyle[n]:=$1F;
  CurveStyle[m]:=$1F;
  CurveColor[n]:=RGB(255,255,255);
  CurveColor[m]:=RGB(255,0,0);
  CurveName[n]:=Format('TheorySpectrum[%d]',[TargetIndex]);
  CurveName[m]:=Format('GroundSpectrum[%d]',[TargetIndex]);
  Lsq.Chi2:=0;
  Lsq.NumFreeVars:=0;
  for i:=0 to CurveLength[n]-1 do begin
   Ei:=TargSpectr[TargetIndex].Energy(i);
   Ti:=TargSpectr[TargetIndex].Intens(i);
   Ni:=0;
   Mi:=0;
   rn:=RoiNumByChan[i];
   if rn>=0 then begin
    for j:=0 to Lsq.Plane.Cols-1 do Ni:=Ni+Lsq.Plane[i,j]*Lsq.X[j];
    Ni:=Ni+RoiList[rn].GroundFixed(i);
    Mi:=RoiList[rn].Ground(i);
    Lsq.NumFreeVars:=Lsq.NumFreeVars+1;
    Lsq.Chi2:=Lsq.Chi2+sqr(Ti-Ni)*Lsq.Weight[i];
   end;
   CurvePoint[n,i]:=Point2d(Ei,Ni);
   CurvePoint[m,i]:=Point2d(Ei,Mi);
  end;
  Lsq.NumFreeVars:=Lsq.NumFreeVars-ParamList.Elem.Length;
  Lsq.Chi2Crit:=Chi2CumDistrInv(1-0.05,Lsq.NumFreeVars);
  for i:=0 to ElementList.AtomId.Length-1 do ElementList.ContentError[i]:=sqrt(Lsq.B[i,i]*Lsq.Chi2/Lsq.NumFreeVars)/Norm;
  //
  // Save result to TheorySpectrum curve comment
  //
  CurveComment[n]:='';
  CurveComment[n]:=CurveComment[n]+Format('[X-Ray Fluorescence Analysis Result - TargetSpectrum[%d]]',[TargetIndex]);
  CurveComment[n]:=CurveComment[n]+Format('SourceFile'#9#9'= %s',[TargSpectr[TargetIndex].SourceFile]);
  CurveComment[n]:=CurveComment[n]+Format('LinearEquitionsRank'#9'= %d',[Lsq.SVD.Rank]);
  CurveComment[n]:=CurveComment[n]+Format('LinearEquitionsDefect'#9'= %d',[Lsq.SVD.N-Lsq.SVD.Rank]);
  CurveComment[n]:=CurveComment[n]+Format('Chi2'#9#9#9'= %12.5f',[Lsq.Chi2]);
  CurveComment[n]:=CurveComment[n]+Format('Chi2(0.95,%d)'#9#9'= %12.5f',[Lsq.NumFreeVars,Lsq.Chi2Crit]);
  for i:=0 to ElementList.AtomId.Length-1 do
  CurveComment[n]:=CurveComment[n]+Format('Concentration[%s]'#9'= %12.5f %s +/- %8.5f %s',
   [RfaAtomSymbol(ElementList.AtomId[i]),ElementList.Content[i]*100,'%',ElementList.ContentError[i]*100,'%']);
  CurveComment[m]:=CurveComment[n];
 end;
end;

procedure TRfaDataSheet.EchoLsq(TargetIndex:Integer);
var i,j,n:Integer;
begin
 with CrwApi,SysApi,GuiApi,DanApi do begin
  //
  // Echo result stored in curve comment
  //
  n:=CurveIndexByName(Format('TheorySpectrum[%d]',[TargetIndex]),twin);
  if n<>0
  then Echo(CurveComment[n])
  else Echo(Format('Could not find TheorySpectrum[%d]',[TargetIndex]));
  //
  // Create curves to view basis functions, if need
  //
  if ViewBasisFunc then
  for j:=0 to Lsq.Plane.Cols-1 do begin
   CurvesCount[twin]:=CurvesCount[twin]+1;
   n:=-CurvesCount[twin];
   CurveLength[n]:=Lsq.Plane.Rows;
   CurveStyle[n]:=$1F;
   CurveColor[n]:=RGB(0,0,0);
   CurveName[n]:=Format('TheorySpectrum[%d].Basis[%d]',[TargetIndex,j]);
   for i:=0 to Lsq.Plane.Rows-1 do CurvePoint[n,i]:=Point2d(TargSpectr[TargetIndex].EnCalibr(i),Lsq.Plane[i,j]);
  end;
  //
  Refreshment(10);
 end;
end;

procedure TRfaDataSheet.MakeAnalysis;
var TargetIndex:Integer;
begin
 with CrwApi,SysApi,GuiApi,DanApi do begin
  for TargetIndex:=0 to Length(TargSpectr)-1 do begin
   Echo(Format(CRLF+CRLF+'Wait while fitting TargetSpectrum[%d]...',[TargetIndex]));
   CalcLsq(TargetIndex);
   EchoLsq(TargetIndex);
  end;
 end;
end;

 //////////////////////////
 // Main plugin function //
 //////////////////////////
/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
var
 DataSheet : ^TRfaDataSheet;
begin
 Result:=0;
 rfa_1229.CrwApi:=CrwApi;
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  if not WindowExists(swin)
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!','Source window not found!'));
  if not WindowExists(twin)
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!','Target window not found!'));
  if CurvesCount[swin]=0
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!','No input data curves found!'));
  if not Rfa.Accessible[1]
  then Raise EDanApi.Create(RusEng('База данных РФА недоступна!','XRay database is not accessible!'));
  if WindowCaption[swin]<>'X-Ray Fluorescence Analysis Job'
  then Raise EDanApi.CreateFmt(RusEng('Неверное окно "%s"!','Wrong window "%s"!'),[WindowCaption[swin]]);
  //
  DataSheet:=Allocate(sizeof(DataSheet^));
  with DataSheet^ do
  try
   InitDataSheet;
   ReadDataSheet;
   EchoDataSheet(Output);
   MakeAnalysis;
  finally
   FreeDataSheet;
   Deallocate(Pointer(DataSheet));
  end;
  SelectedCurve[twin]:=0;
  WindowCaption[twin]:='X-Ray Fluorescence Analysis Result';
  WindowTitle[twin]:=WindowTitle[swin];
  WindowLegend[twin]:=WindowLegend[swin];
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!'));
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////


