---

<b id="toc" class="big memo">Содержание</b>

[[toc]]

---

<a name="crwdaq_plugin_about"></a>

# Расширения crwdaq: плагины и утилиты …

Здесь содержится информация о расширениях пакета **[CrwDaq](crwdaq.htm)**.  

> Расширения (плагины, утилиты) могут быть весьма полезными, т.к. позволяют
  использовать высокую скорость и огромные возможности натурального кода,
  создаваемого с помощью компилятора **Free Pascal**.

Под расширениями здесь понимаются исполняемые программные модули пакета,
которые не входятв основной исполняемый файл пакета, а существуют в виде отдельных файлов.
Расширениями пакета считаются программные модули, специально разработанные для
работы в составе пакета, а не сторонние программы, которые разрабатывались
для других целей, но были включены в пакет ввиду их полезности.
Различие состоит в том, что расширения пакета делаются по определенным правилам,
с учетом возможности взаимодействия с основным пакетом.

Расширения делятся на два основных класса:  

1. **Плагины** (_plugin_) - динамически загружаемые модули в виде библиотек **`*.dll`** или **`lib*.so`**,
   которые загружаются по мере необходимости и работают в адресном пространстве
   основного исполняемого файла пакета.

2. **Утилиты** (_utility_) - автономные программы (как правило, консольные), выполняющие четко
   определенную (утилитарную) задачу, запускаемые из среды пакета и работающие
   параллельно с пакетом для решения общей задачи управления.

Все расширения пишутся в среде пакета **crwdaq** на **Free Pascal**, по шаблонам.
Компилятор **Free Pascal** интегрирован с пакетом, хотя устанавливается отдельно.
Редактирование, компиляция и запуск расширений деляется из среды пакета **crwdaq**.

Подробнее см. **[типы расширений](#crwdaq_plugin_types)**.

<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---

<a name="crwdaq_plugin_types"></a>

## Типы расширений пакета …

Расширения пакета делятся на такие типы:  

- **`Плагины dan`** - загружаемые расширения для анализа данных (**dan** - _data analysis plugin_).  
  Эти плагины привязаны к окнам с данными (графиками кривых) и служат для обработки данных.  
  Их вызов происходит по команде пользователя.  
  
- **`Плагины daq`** - загружаемые расширения для сбора данных (**daq** - _data acquisition plugin_).  
  Эти плагины привязаны к устройствам (_device_) системы сбора данных **DAQ**.  
  Их загрузка и выгрузка происходит вместе с конфигурацией **DAQ**-системы.    
  Их вызов происходит в цикле опроса в процессе работы **DAQ**-системы.  
  За загрузку/выгрузку/опрос отвечает программа-владелец **DaqPascal**,  
  с помощью вызовов **DaqDllInit**, **DaqDllFree**, **DaqDllCall**.  

- **`Консольные драйверы`** - автономные **присоединенные** консольные программы.  
  Обычно это **драйверы** или **серверы**, работающие в составе **DAQ**-систем,  
  запускаемые клиентской программой **DaqPascal** и подключенные к этой клиентской  
  программе через анонимный канал связи. Через этот канал клиентская программа  
  **DaqPascal** передает команды и получает данные от сервера.  
  Можно образно сказать, что присоединенные консольные драйверы - это программы **на коротком поводке**.  

- **`Консольные утилиты`** - автономные **отсоединенные** программы разного назначения.  
  Обычно эти они работают в составе **DAQ**-систем, но могут быть полезными и для других целей.  
  Эти программы не подключаются через канал связи, и работают без прямого контроля со стороны пакета.  
  Они, например, могут запускаться автономно и работать в фоновом режиме для обработки данных.  
  При этом запускаемые из пакета утилиты сохряняют с ним некоторую связь, т.к. наследуют окружение.  
  Через это окружение они могут взаимодействовать с основным пакетом, посылая ему сообщения.  
  Например, они могут по завершении обработки уведомить пакет о результатах работы посылкой сообщения  
  командой **`unix send2crwdaq …`** в **Главную Консоль** пакета.  
  Можно образно сказать, что отсоединенные консольные утилиты - это программы **на свободном выпасе**.  

<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---

<a name="crwdaq_plugin_principal"></a>

## Плагины: общие принципы …

Плагины - это загружаемые динамические библиотеки с кодом, построенным по определенным правилам.

### Общие правила работы …

1. Плагин оформляется в виде динамической библиотеки **`*.dll`** или **`lib*.so`**.  
   Библиотека загружается по требованию пользователя и  создается на языке **Free Pascal**.  

2. Библиотека экспортирует одну **функцию плагина**:  
   **`function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;`**  

3. Функция принимает аргумент **`CrwApi`** - интерфейс **`ICrwApi`** для доступа к пакету.  
   Переданный экземпляр интерфейса содержит в себе большой список (около **1000**) функций.  
   В совокупности эти функции содержат в себе библиотеку для доступа к данным вызывающего пакета.  
   Интерфейс описан в модуле **[_crw_crwapi.pas](../../../crwlib/_crw_crwapi.pas)**.  

4. Всё взаимодействие плагина с вызывающим пакетом выполняется с помощью функций интерейса **`ICrwApi`**.  
   Ввод-вывод также обеспечивается функциями этого интерфейса.  

5. Функция плагина при вызове выполняет требуемую работу и возвращает как результат статус операции:  
   обычно **0** при успехе или ненулевой код ошибки при возникновении проблем.  

### Интерфейс ICrwApi

Интерфейс **ICrwApi** описан в модуле **[_crw_crwapi.pas](../../../crwlib/_crw_crwapi.pas)**.  
В основном он содержит функции идентификации и другие (предметные) интерфейсы.  
Некоторые важные методы **ICrwApi** приведены в следующей таблице:  

| Методы **ICrwApi**   | Описание                                                              |
|----------------------|-----------------------------------------------------------------------|
| **Version:Cardinal** | Версия **API** плагина в виде числа - даты вида **20241121**.         |
| **Target:Cardinal**  | Целевой контекст вызова: **ForDataAnalysis**, **ForDataAcquisition**. |
| **SysApi:ISysApi**   | Интерфейс с набором системных (_system_) функций общего назначения.   |
| **GuiApi:IGuiApi**   | Интерфейс с функциями графического интерфейса пользователя (_GUI_).   |
| **DanApi:IDanApi**   | Интерфейс с функциями анализа данных (_data analusis_) в окне с кривыми. |
| **DaqApi:IDaqApi**   | Интерфейс с функциями сбора данных (_data acquisition_) и управления. |
| **PluginName:LongString**           | Короткое имя плагина.                                  |
| **PluginSourceFileName:LongString** | Полное имя файла исходного кода **.lpr** плагина.      |
| **PluginBinaryFileName:LongString** | Полное имя исполняемого файла **`*.dll`**/**`lib*.so`** плагина. |

Метод **Target** служит для опрелеления целевого контекста вызова.
Если он равен **ForDataAnalysis**, то это вызов обработки данных, находящихся в окне с кривыми.
В этом случае надо использовать интерфейс **DanApi** для достура к данным окна.
Если же метод **Target** вернул **ForDataAcquisition**, то это вызов из **DAQ**-системы сбора данных.
В этом случае надо использовать интерфейс **DaqApi** для доступа к данным **DAQ**-системы.

Методы **Version**, **PluginName**, **PluginSourceFileName**, **PluginBinaryFileName**
служат для идентификации плагина, позволяя определять его точное расположение, название и т.д.
Это може помочь, например, для диагностики или для поиска дополнительных файлов или ресурсов.

Методы **CrwApi.SysApi**, **CrwApi.GuiApi** содержат большое число функций общего назначения,
которые можно использовать для решения задач, выполняемых плагином.

> Полный список и описание интерфейсов, классов, функций, переменных, констант
  модуля **`_crw_crwapi`** с описанием интерфейса **ICrwApi** подключения плагинов
  см. **[тут](../../../crwlib/manual/_crw_crwapi/index.html)**
  или **[там](../../../crwlib/crwlib.chm)**.

### Дисциплина при создании плагинов

При создании плагинов следует соблюдать высокую дисциплину программирования.
Дело в том, что плагины загружаются и работают в адресном пространстве вызывающего их пакета.
Это дает высокую скорость работы плагинов, но одновременно повышает ответственность программиста.
Все ошибки в коде плагина могут иметь непосредственное влияние на работу пакета.
После выгрузки плагина все не освобожденные им ресурсы становятся "потерянными",
т.е. возникает **утечка ресурсов**.

Поэтому программировать плагины надо очень аккуратно и ответственно.

- Все занятые ресурсы (память, файлы, объекты) надо вовремя освобождать.  
  Рекомендуется при выделении ресурсов использовать конструкцию  
  **try … finally … end;** для гарантированного освобождения ресурсов.  

- Стараться избегать длительных циклов ожидания, чтобы не "подвесить" основной пакет.  

- Не надо жалеть времени на проверки, позволяющие заранее избегать аварийных ситуаций.  
  Своевременная проверка аргументов и условий позволит предотвратить исключения
  до их возможного возникновения.

- Надо стараться перехватывать и обрабатывать исключения **try … except … end;**.  
  Хотя код вызова перехватывает исключения, все же лучше делать это в самом плагине.  

- По мере возможности следует ограничивать число используемых (**uses**) модулей.  
  Максимальное использование функций **ICrwApi** позволяет решать
  большое число задач без привлечения других модулей.
  Проблема модулей в том, что они могут содержать коды инициализации
  и завершения, после которых могут оставаться неосвобожденные ресурсы
  (после выгрузки плагина из памяти).


<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---

<a name="crwdaq_plugin_dan"></a>

## Плагины dan для обработки данных …

Плагины для обработки данных  **dan** - _data analysis plugin_ - это динамически загружаемые
библиотеки **`*.dll`** или **`lib*.so`**, загружаемые по команде пользователя для выполнения
обработки данных в окне с графиками кривых.
Эти плагины привязаны к окнам с данными (графиками кривых) и служат для обработки данных,
которые в этих окнах содержатся. Их вызов происходит по команде пользователя - обычно
по кнопке ![Exec](crwdaq-plugin_files/run.png).

Плагины обработки данных пишутся по определенному шаблону на языке **Free Pascal**.
Написанный или отредактированный в среде пакета плагин затем можно сразу скомпилировать,
а затем загрузить и исполнить.
Это делает пакет **crwdaq** саморасширяемой средой разработки и исполнения, позволяющей
прикладному программисту самостоятельно дополнять пакет разработанными средствами,
если имеющихся средств не хватает.

Плагины обработки данных находятся в папке **[resource/plugin/dan/](../../resource/plugin/dan/)**.

### Общая логика работы dan плагинов

- Плагины для обработки данных **dan** вызываются по команде пользователя.  
  Обычно это вызов команды ![run](crwdaq-plugin_files/run.png) запуска плагина в окне с графиками кривых.  
- Исходными данными для обработки служат кривые в окне с графиками кривых, из которого вызван плагин.  
- Плагин создает новое окно с графиками кривых и помещает туда кривые с результатами расчетов.  
- При проведении расчетов плагин использует переданный ему интерфейс **CrwApi**.  
  В него входит ряд других интерфейсов с общим набором более **1000** функций.  
  В частности, интерфейс **DanApi** содержит функции для анализа данных.  

### Пример плагина _curve_integral

Рассмотрим протейший плагин обработки данных
**[_curve_integral](../../resource/plugin/dan/_curve_integral/_curve_integral.lpr)**,
выполняющий интегрирование (вычисление первообразной):

<iframe src="../../resource/plugin/dan/_curve_integral/_curve_integral.lpr" width="95%" height="50%" align="left" class="pre code">
 Ваш браузер не поддерживает плавающие фреймы!

``` pascal
////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Численное интегрирование кривой.
Интегрирование кривой производится по методу трапеций.
Это дискретный аналог первообразной функции.
[]
[Manual.Eng]
This utility evaluate curve integral using trapezium rule.
That is discrete analog of function integration.
[]
[Arguments.Rus]
Имя окна = Результат:"Первообразная кривой"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
[]
[Arguments.Eng]
Caption = Result:"Curve integral"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
[]
}

library _curve_integral;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNotSortedX + cfDuplicatesX + cfNanInf;
var
 c1, c2, cFirst, cLast : Integer;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  for c1:=cFirst to cLast do begin
   if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для интегрирования!',
                                           'Curve %d is not convenient for integration!'),[c1]));
   c2:=CreateCurve(twin);
   CurveAssign(c2,c1);
   CurveIntegrate(c2);
   if CurveFlags(c2,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Ошибка интегрирования кривой %d !',
                                           'Error on curve %d integration !'),[c1]));
   Refreshment(100);
  end;
  if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;			
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
```

</iframe>

Разберем приведенный выше плагин подробнее.

---

### Структура плагина обработки данных

Плагин обработки данных должен иметь чёткую структуру, описанную ниже.

#### Блок справки и аргументов

В начале кода идет помещенный в комментариях блок секций Русской/Английской справки - **[Manual.Rus]**/**[Manual.Eng]**,
а также секций аргументов на Русском/Агнглийском **[Arguments.Rus]**/**[Arguments.Eng]**.

``` pascal
{
[Manual.Rus]
…
[Manual.Eng]
…
[Arguments.Rus]
Имя окна = …
Заголовок = …
Легенда = …
[Arguments.Eng]
Caption = …
Title = …
Legend = …
}
```

Справка отображаются в диалоге вызова плагина для пояснения того, что он делает.  
Аргументы - это редактируемые в диалоге параметры, нужные для проведения расчетов.  
Значения аргументов извлекаются вызовами типа **`GetArgumentAsString(RusEng('Имя окна','Caption'));`**.  

Примерный вид диалога вызова плагина обработки данных приведен на следующем рисунке:  

![crwdaq Run Plugin Dialog](crwdaq-plugin_files/dan-dialog.png)

| Команда | Действие | Комментарий                                                                       |
|---------|----------|-----------------------------------------------------------------------------------|
| ![Exec](crwdaq-plugin_files/cmd1.png)       | Запуск        | Запуск выбранного плагина на выполнение. |
| ![Edit](crwdaq-plugin_files/cmd2.png)       | Изменить      | Открыть окно редактирования плагина.     |
| ![Delete](crwdaq-plugin_files/cmd3.png)     | Удалить       | Удалить выбранный плагин.                |
| ![Clone](crwdaq-plugin_files/cmd4.png)      | Клонировать   | Клонировать выбранный плагин в новый.    |
| ![Compile](crwdaq-plugin_files/cmd5.png)    | Компилировать | Компилировать выбранный плагин.          |
| ![CompileAll](crwdaq-plugin_files/cmd6.png) | Компил-ть Все | Компилировать все доступные плагины.     |
| ![Exit](crwdaq-plugin_files/cmd7.png)       | Выход         | Закрыть диалог запуска плагинов.         |

#### Блок определений и режимов

После блока справки и аргументов идет объявление библиотеки **`library …;`**,
затем **обязательная** вставка общих определений **[{$I _crw_sysdef.inc}](../../../crwlib/_crw_sysdef.inc)**
и режима компиляции **[{$I _crw_sysmode.inc}](../../../crwlib/_crw_sysmode.inc)**.

``` pascal
library _curve_integral;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}
```

Этот блок кода **обязателен**, т.к. он задает необходимые режимы компиляции.

#### Блок uses

После определений и режимов компиляции **обязательно** идет блок **`uses`**:

``` pascal
uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;
```

Очень важно соблюдать следующие правила:  

1. Первым выражением **uses** идет вставка **[{$I _crw_uses_first.inc}](../../../crwlib/_crw_uses_first.inc)**.  
   Это обеспечивает правильную инициализацию менеджера памяти и библиотеки поддержки потоков.  
2. Затем идет список стандартных модулей, в который обычно входят:  
   **sysutils, classes, math, graphics**. Но могут добавляться и другие.  
3. Обязательно включение в **uses** модуля **`_crw_crwapi`**.  
   В этом модуле описан интерфейс для доступа к функциям пакета.  
4. Список модулей должен быть минимально необходимым для решения данной задачи.  
5. Функция плагина должна освобождать все (явно) занятые в процессе исполнения ресурсы.  

Минимизация списка используемых модулей необходима по следующей причине.  
Плагин должен освобождать все занятые ресурсы, т.к. библиотека с кодом плагина загружается
динамически и затем освобождается. Прикладной код должен освобождать все (явно) занятые
ресурсы, однако при использовании модулей при загрузке библиотек будут неявно (скрыто)
выполняться код **initialization** модуля при загрузке и **finalization** при выгрузке
библиотеки. Далеко не все библиотеки написаны **чисто** в смысле освобождения ресурсов.
Это связанос тем, что модули чаще всего ориентируются на исполняемые приложения **`*.exe`**,
а не на библиотеки. Освобождать все ресурсы в конце работы программы не столь необходимо,
т.к. программа всё равно уже завершается. А вот при выгрузке библиотек выполнение программы
продолжается. Всё, что не освободил любой из модулей библиотеки, будет **утечкой ресурсов**.
Поэтому число модулей **uses** нужно по возможности минимизировать.

Интерфейс **CrwApi**, переданный в плагин при его вызове, содержит в себе порядка **1000** функций,
дающих доступ к библиотеке **crwlib**, уже скомпилированных в ядре пакета. Поэтому для плагинов
крайне желательно по мере возможности ограничиться этим набором вместо подключения дополнительных модулей.
Это не только сократит объем кода плагина (т.к. вместо компиляции нового кода интерфейс ссылается на уже
скомпилированный код), но и сократит использование ресурсов (которые уже выделены в коде пакета).
А главное, будет меньше проблем с освобождением ресурсов, занятых модулями, а потому меньше
вероятность утечки памяти.

#### Блок декларации функции плагина

После **uses** необходим блок декларации **функции плагина**.

``` pascal
/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
```

Для обеспечения централизации и единства кода декларация делается с помощью
включаемого файла **[{$I _crw_plugin_declare.inc}](../../../crwlib/_crw_plugin_declare.inc)**.
Это позволяет управлять кодом декларации всех плагинов в одном файле.
А также обеспечивает правильность декларации во всех плагинах.
Заметим, что плагин использует соглашение о вызовах **ABI call** (_application binary interface_),
заданное в файле **[_crw_plugin_abicall.inc](../../../crwlib/_crw_plugin_abicall.inc)**.

#### Блок тела (содержания) плагина

После декларации **функции плагина** следует **тело плагина**, т.е. его содержательная часть.

В начале этого блока следует:

- Использовать интерфейсы **`with CrwApi,SysApi,GuiApi,DanApi do`** для доступа к методам библиотеки,  
- Выполнять код в блоке **try … except … end**.  
- Перенаправить ввод-вывод в **Главную Консоль** пакета вызовом **RedirectStdIn**, **RedirectStdOut**,  
- Проверить версию и тип плагина вызовом **VerifyPluginDate**, **VerifyPluginType**,  

Это выглядит примерно так:

``` pascal
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  …
  … код плагина …
  …
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;			
end;
```

В теле плагина используются данные (кривые) из активного окна и данные из буфера обмена.
Результат (новые кривые) помещается в новое окно. Кривые и окна идентифицируются целочисленными
номерами: положительные числа нумеруют кривые в окне источника (_source_),
отрицательные - приемника (_target_), ноль ссылается на буфер обмена (_clipboard_).

``` pascal
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
```

Например, вызов **WindowExists(twin)** проверяет наличие окна приемника,
а **CurvesCount[twin]** - возвращает число кривых в окне приемника.

Как правило, назначение функций и их аргументов понятно из названий.

> Полный список и описание интерфейсов, классов, функций, переменных, констант
  модуля **`_crw_crwapi`** с описанием интерфейса **ICrwApi** подключения плагинов
  см. **[тут](../../../crwlib/manual/_crw_crwapi/index.html)**
  или **[там](../../../crwlib/crwlib.chm)**.

#### Блок экспорта функции плагина

В конце, после завершения **функции плагина**, необходимо **экспортировать** её:

``` pascal
/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.
```

Экспорт выполняется вставкой **[{$I _crw_plugin_exports.inc}](../../../crwlib/_crw_plugin_exports.inc)**,
что обеспечивает централизованное управление кодом плагинов.

### Шаблоны плагинов dan

Пакет содержит большую библиотеку (более **30**) готовых плагинов обработки данных **[dan](../plugin/dan/)**, созданных по одному шаблону.
Средства клонирования (копирования и сохранения под другим именем) позволяют создавать новые плагины,
взяв существующие плагины в качестве образца (шаблона).

Плагины обработки данных, имя которых начинается с подчеркивания **`_`**, считаются "**защищенными**".
Их нельзя удалять и редактировать средствами пакета - на них стоит блокировка.
Это сделано для того, чтобы избежать нежелательного изменения хорошо отлаженных стандартных плагинов,
которые не требуют редактирования, а используются "как есть", в готовом виде.

<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---

<a name="crwdaq_plugin_daq"></a>

## Плагины daq для сбора данных …

Плагины для **сбора данных** - **daq** - _data acquisition plugin_ - это динамически загружаемые
библиотеки **`*.dll`** или **`lib*.so`**, загружаемые в составе **DAQ** системы и работающие
для решения задач сбора данных и управления физическими установками.
Эти плагины привязаны к устройствам **DAQ** системы **devices software program**, т.е. к программам
на языке **DaqPascal** и служат для сбора и обработки данных в режиме **online**.
Они вызываются и выполняются в контесте потока программы **DaqPascal**.

Плагины обработки данных пишутся по определенному шаблону на языке **Free Pascal**.
Написанный или отредактированный в среде пакета плагин затем можно сразу скомпилировать,
а затем загрузить и исполнить в составе конфигурации **DAQ**-системы.
Это делает пакет **crwdaq** саморасширяемой средой разработки и исполнения, позволяющей
прикладному программисту самостоятельно дополнять пакет разработанными средствами,
если имеющихся средств не хватает.

Шаблоны плагинов сбора данных находятся в папке **[resource/plugin/daq/](../../resource/plugin/daq/)**.
Также есть плагины сбора данных в демо конфигурациях **[demo_dll_min](../../demo/demo_dll_min/)**,
**[demo_dll_sin](../../demo/demo_dll_sin/)** и ряда других.

### Общая логика работы daq плагинов

- Код плагина выполняется в контексте прикладной программы **DaqPascal**.  
  Эту программу называют владельцем (_host_) или оберткой (_wrap_).  
- Программа вызывает плагин с помощью **`DaqDllCall(ref,cmd)`**, где:  
    + **ref** - ссылка на библиотеку плагина,  
    + **cmd** - команда  
        - **DAQ_CMD_INIT** (инициализация),  
        - **DAQ_CMD_POLL** (цикл опроса),  
        - **DAQ_CMD_FREE** (завершение).  
    + Плагин узнает команду вызовом **CrwApi.DaqApi.DaqCommand**.  
- При старте **DAQ**-системы:  
    + Программа берет имя файла плагина _DllFilePath_ из конфигурации.  
    + Программа загружает библиотеку плагина вызовом **`DaqDllInit(DllFilePath)`**.  
    + Программа вызывает **`DaqDllCall(…,DAQ_CMD_INIT)`**, чтобы плагин провел свою инициализацию.  
- В цикле опроса **DAQ**-системы:  
    + Программа вызывает **`DaqDllCall(…,DAQ_CMD_POLL)`**, чтобы плагин выполнил свою процедуру опроса.  
      Это основное действие плагина, по которому он должен выполнят содержательную работу.  
    + Для выполнения работы плагину передается интерфейс **CrwApi**, в котором есть и другие интерфейсы.  
      В совокупности эти интерфейсы содержат более **1000** функций для выполнения различных операций.  
      Интерфейс **DaqApi** отвечает за взаимодействие плагина с **DAQ** системой.  
- При остановке **DAQ**-системы:  
    + Программа вызывает **`DaqDllCall(…,DAQ_CMD_FREE)`**, чтобы плагин освободил занятые ресурсы.  
    + Программа выгружает библиотеку плагина вызовом **`DaqDllFree`**.  

### Вызов плагинов с помощью Стандартной Библиотеки

Загружать, освобождать и вызывать **daq** плагины можно в принципе из любой программы на языке **DaqPascal**.
Это удобно делать с помощью функций стандартной библиотеки. Если использовать стандартный шаблон программы,
достаточно вызвать **[STD_DLL_INIT](daqpascalapi.htm#std_dll_init)** при старте программы, инициировав
загрузку **DLL**.

``` pascal
procedure InitApplication;
begin
 …
 // Initialize DAQ plugin DLL:
 STD_DLL_INIT('DLL_FILE_PATH');
 …
end;
```

Дальше стандартная библиотека возьмет на себя вызов плагина в каждом цикле опроса и его освобождение
в конце работы **DAQ**-системы.

Стандартная библиотека после вызова **`STD_DLL_INIT`** выполняет вызов плагина к каждом цикле опроса.
Если нужна другая логика работы (например, вызов плагина только по событиям), то лучше использовать функции
**[DLL_INIT/FREE/POLL](daqpascalapi.htm#dll_init)**, которые позволяют вызывать плагин тогда, когда
требуется прикладному алгоритму. Но и следить за логикой работы плагина тогда придется самостоятельно.

### Программа _dllhost.pas - рекомендуемая оболочка для вызова плагина

Как уже было сказано, для вызова **daq** плагина требуется **DAQ**-программа на языке **DaqPascal**,
которая его вызывает, так как плагин выполняется в контексте вызывающей его **DAQ**-программы.
Код вызова **daq** плагина можно вставить в любую программу, однако в стандартной библиотеке есть
готовая программа **[_dllhost.pas](../../resource/daqsite/stdlib/daqpas/_dllhost.pas)**, которая
специально предназначена для вызова **daq** плагина.

<iframe src="../../resource/daqsite/stdlib/daqpas/_dllhost.pas" width="95%" height="25%" align="left" class="pre code">
 Ваш браузер не поддерживает плавающие фреймы!

``` pascal
 {
 ***********************************************************************
 Daq Pascal application program _dllhost.
 ***********************************************************************
 * The purpose of _dllhost program is to handle DAQ plugin DLL: load,
 * initialize, call plugin on DAQ polling loop, and finally unload.
 * Program based on Standard Library functions: calls STD_DLL_INIT
 * directly, also STD_DLL_POLL and STD_DLL_FREE will be called
 * automatically by library code when needed.
 * The program code looks "empty", because most code hidden inside the
 * Standard Library, so user's code needs only STD_DLL_INIT call.
 ***********************************************************************
 }
program _dllhost; { The HOST program to Load & Poll DAQ plugin DLL. }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 
type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types:      }
 
var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }

 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('','');
  StdIn_SetTimeouts(0,0,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  // Initialize DAQ plugin DLL:
  STD_DLL_INIT('DLL_FILE_PATH');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid:Integer;
  procedure Cleanup;
  begin
   cmd:=''; arg:='';
  end;
 begin
  Cleanup;
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
```

</iframe>

<br><br>

Программа - оболочка конфигурируется примерно так:

``` ini
[DeviceList]
&DEMO_DLL_HOST = device software program
[]
[&DEMO_DLL_HOST]
InquiryPeriod = 10                                              ; период опроса
DevicePolling = 100, tpNormal                                   ; период, приоритет потока
DLL_FILE_PATH = demo_dll_min_plugin\demo_dll_min_plugin.dll     ; расположение DLL плагина
ProgramSource = ~~/resource/daqsite/stdlib/daqpas/_dllhost.pas  ; программа-владелец(обертка)
StdIn_EnablePoll = 1 ; Разрешение опроса StdIn в коде DaqPascal, иначе опрос делает код плагина
…
… и другие параметры …
…
[]
```
В конфигурации задается ссылка **`DLL_FILE_PATH`** на бинарный файл плагина,
а также ссылка **`ProgramSource`** на программу-оболочку **`_dllhost.pas`**.
Также задается флаг **`StdIn_EnablePoll`**, который разрешает опрос
консольного ввода **StdIn** в программе **DaqPascal**.
Если он сброшен в ноль, то обрабатывать консольный ввод должен сам плагин.


Использование программы - оболочки упрощает использование **daq** плагинов, предоставляя
готовую клиентскую часть для подключения плагина к **DAQ** системе.

### Программа _dllwrap.pas - запасная (устаревшая) оболочка для вызова плагина

В стандартной библиотеке есть запасная (несколько устаревшая, но вполне рабочая)
готовая программа **[_dllwrap.pas](../../resource/daqsite/stdlib/daqpas/_dllwrap.pas)**, которая
специально предназначена для вызова **daq** плагина.

<iframe src="../../resource/daqsite/stdlib/daqpas/_dllwrap.pas" width="95%" height="25%" align="left" class="pre code">
 Ваш браузер не поддерживает плавающие фреймы!

``` pascal
 {
 *********************************************************************
 Назначение: Интерфейс для вызова DLL из DAQ Pascal.
 Пример конфигурирования:
 [DeviceList]
 &DRIVER = device software program
 [&DRIVER]
 ProgramSource	= ~~/resource/daqsite/stdlib/daqpas/_dllwrap.pas
 DLL_FILE_PATH	= ..\daqpas\driver.dll   ; DLL plugin file path
 .....                                   ; Other information
 *********************************************************************
 }
program _dllwrap;
var
 hDll:Integer; { Ссылка на DLL }
 {
 Процедура освобождения DLL
 }
 procedure DLL_FREE(var hDll:Integer);
 var b:Boolean;
 begin
  if hDll<>0 then begin
   b:=daqdllcall(hDll,Daq_Cmd_Free);
   b:=daqdllfree(hDll);
   hDll:=0;
  end;
 end;
 {
 Процедура пытается загрузить DLL с заданным именем файла.
 Возвращает ненулевую ссылку DLL или ноль при ошибке.
 }
 function DLL_INIT(DllPath:String):Integer;
 var b:Boolean; hDll:Integer;
 begin
  hDll:=daqdllinit(DllPath);
  writeln(DevName+' : DLL_FILE_PATH = '+DllPath);
  if hDll<>0 then if not daqdllcall(hDll,Daq_Cmd_Init) then DLL_FREE(hDll);
  if hDll=0 then begin
   DllPath:=ExtractFileName(DllPath)+ExtractFileExt(DllPath);
   b:=fixerror(registererr('Fail load '+DllPath+' !'));
  end;
  DLL_INIT:=hDll;
 end;
 {
 Процедура опроса DLL.
 }
 procedure DLL_POLL(hDll:Integer);
 var b:Boolean;
 begin
  if hDll<>0 then b:=daqdllcall(hDll,Daq_Cmd_Poll);
 end;
 {
 Получить имя DLL файла по имени параметра конфигурации.
 }
 function DLL_FILE_PATH:String;
 var s:String;
 begin
  s:=ReadIni('DLL_FILE_PATH');
  s:=AdaptDllFileName(s);
  s:=DaqFileRef(s,'');
  DLL_FILE_PATH:=s;
  s:='';
 end;
begin
 {
 При старте загружаем DLL
 }
 if runcount=1 then hDll:=DLL_INIT(DLL_FILE_PATH) else
 {
 При останове удаляем DLL
 }
 if isinf(runcount) then DLL_FREE(hDll) else
 {
 В цикле опроса вызываем DLL
 }
 if hDll<>0 then DLL_POLL(hDll);
end.
```

</iframe>

<br><br>

Программа - оболочка конфигурируется примерно так:

``` ini
[DeviceList]
&DEMO_DLL_MIN = device software program
[]
[&DEMO_DLL_MIN]
InquiryPeriod = 10                                              ; период опроса
DevicePolling = 100, tpNormal                                   ; период, приоритет потока
DLL_FILE_PATH = demo_dll_min_plugin\demo_dll_min_plugin.dll     ; распположение DLL
ProgramSource = ~~/resource/daqsite/stdlib/daqpas/_dllwrap.pas  ; программа-оболочка
…
… и другие параметры …
…
[]
```

В конфигурации задается ссылка **`DLL_FILE_PATH`** на бинарный файл плагина,
а также ссылка **`ProgramSource`** на программу-оболочку **`_dllwrap.pas`**.

Использование программы - оболочки упрощает использование **daq** плагинов, предоставляя
готовую клиентскую часть для подключения плагина к **DAQ** системе.

### Пример плагина demo_dll_min_plugin

Рассмотрим протейший плагин сбора данных
**[demo_dll_min_plugin](../../resource/plugin/daq/demo_dll_min_plugin/demo_dll_min_plugin.lpr)**,
просто выводящий в консоль устройства синусоиду - значение **`sin(omega*time)`**:

<iframe src="../../resource/plugin/daq/demo_dll_min_plugin/demo_dll_min_plugin.lpr" width="95%" height="50%" align="left" class="pre code">
 Ваш браузер не поддерживает плавающие фреймы!

``` pascal
////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq plugin for demo_dll_min DAQ config - how to use plugins in DAQ.     //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 202411120 - Sample created by A.K.                                         //
////////////////////////////////////////////////////////////////////////////////

library demo_dll_min_plugin;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
 ////////////////////////////
 // Comment on TDaqDataSheet:
 // Hold all DAQ data there!!
 // Init all on DAQ_CMD_INIT.
 // Free all on DAQ_CMD_FREE.
 // Use data on DAQ_CMD_POLL.
 ////////////////////////////
type ////////////////////////
 TDaqDataSheet = packed record
  Sinus    : Double;
  Omega    : Double;
  timeSec  : Double;
  dllName  : LongString;
 end; //// TDaqDataSheet /////
 ////////////////////////////
begin
 Result:=0;
 with CrwApi,SysApi,GuiApi,DaqApi do
 try
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAcquisition);
  ///////////////////////////////////////////////////////////////
  // Nota Bene: all driver data must be located in DaqDataSheet!!
  ///////////////////////////////////////////////////////////////
  with TDaqDataSheet(DaqDataSheet(SizeOf(TDaqDataSheet))^) do begin
   case DaqCommand of
    ////////////////////
    // Initialize driver
    ////////////////////
    DAQ_CMD_INIT: begin
     RedirectStdIn(Input);
     RedirectStdOut(Output);
     //////////////////////////////////////////////////
     // Initialize USER data located in DaqDataSheet //
     //////////////////////////////////////////////////
     dllName:=ExtractFileNameExt(PluginBinaryFileName);
     GuiApi.Echo(Format('Starting %s …',[dllName]));
     Sinus:=0;
     Omega:=1;
     timeSec:=0;
     //////////////////
     GuiApi.Echo('Ok');
    end;
    //////////////////
    // Finalize driver
    //////////////////
    DAQ_CMD_FREE: begin
     GuiApi.Echo(Format('Stopping %s …',[dllName]));
     //////////////////
     // Don`t forget to
     // Clear all data:
     //////////////////
     Sinus:=0;
     Omega:=0;
     timeSec:=0;
     dllName:='';
     //////////////////
     // Free DataSheet:
     //////////////////
     DaqDataSheet(0);//
     //////////////////
     GuiApi.Echo('Ok');
    end;
    ////////////////////
    // Main polling loop
    ////////////////////
    DAQ_CMD_POLL: begin
     /////////////////////////////
     // USER data processing code:
     /////////////////////////////
     timeSec:=time*timeunits*1e-3;
     Sinus:=sin(Omega*timeSec);
     Echo(Format('RunCount = %g',[RunCount]));
     writeln(Format('sin(%11.4f)=%10.6f',[timeSec,Sinus]));
     /////////////////////////////
     // USER data processing done.
     /////////////////////////////
    end;
   end; // case
  end; // with
 except
  on E:Exception do begin
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message);
   Result:=-1;
  end;
 end;
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////

```

</iframe>

Разберем приведенный выше плагин подробнее.

---

### Структура плагина сбора данных

Плагин сбора данных должен иметь чёткую структуру, описанную ниже.

#### Блок определений и режимов daq плагина

Первым идет объявление библиотеки **`library …;`**,
затем **обязательная** вставка общих определений **[{$I _crw_sysdef.inc}](../../../crwlib/_crw_sysdef.inc)**
и режима компиляции **[{$I _crw_sysmode.inc}](../../../crwlib/_crw_sysmode.inc)**.

``` pascal
library demo_dll_min_plugin;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}
```

Этот блок кода **обязателен**, т.к. он задает необходимые режимы компиляции.

#### Блок uses для daq плагина

После определений и режимов компиляции **обязательно** идет блок **`uses`**:

``` pascal
uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;
```

Очень важно соблюдать следующие правила:  

1. Первым выражением **uses** идет вставка **[{$I _crw_uses_first.inc}](../../../crwlib/_crw_uses_first.inc)**.  
   Это обеспечивает правильную инициализацию менеджера памяти и библиотеки поддержки потоков.  
2. Затем идет список стандартных модулей, в который обычно входят:  
   **sysutils, classes, math, graphics**. Но могут добавляться и другие.  
3. Обязательно включение в **uses** модуля **`_crw_crwapi`**.  
   В этом модуле описан интерфейс для доступа к функциям пакета.  
4. Список модулей должен быть минимально необходимым для решения данной задачи.  
5. Функция плагина должна освобождать все (явно) занятые в процессе исполнения ресурсы.  

Минимизация списка используемых модулей необходима по следующей причине.  
Плагин должен освобождать все занятые ресурсы, т.к. библиотека с кодом плагина загружается
динамически и затем освобождается. Прикладной код должен освобождать все (явно) занятые
ресурсы, однако при использовании модулей при загрузке библиотек будут неявно (скрыто)
выполняться код **initialization** модуля при загрузке и **finalization** при выгрузке
библиотеки. Далеко не все библиотеки написаны **чисто** в смысле освобождения ресурсов.
Это связанос тем, что модули чаще всего ориентируются на исполняемые приложения **`*.exe`**,
а не на библиотеки. Освобождать все ресурсы в конце работы программы не столь необходимо,
т.к. программа всё равно уже завершается. А вот при выгрузке библиотек выполнение программы
продолжается. Всё, что не освободил любой из модулей библиотеки, будет **утечкой ресурсов**.
Поэтому число модулей **uses** нужно по возможности минимизировать.

Интерфейс **CrwApi**, переданный в плагин при его вызове, содержит в себе порядка **1000** функций,
дающих доступ к библиотеке **crwlib**, уже скомпилированных в ядре пакета. Поэтому для плагинов
крайне желательно по мере возможности ограничиться этим набором вместо подключения дополнительных модулей.
Это не только сократит объем кода плагина (т.к. вместо компиляции нового кода интерфейс ссылается на уже
скомпилированный код), но и сократит использование ресурсов (которые уже выделены в коде пакета).
А главное, будет меньше проблем с освобождением ресурсов, занятых модулями, а потому меньше
вероятность утечки памяти.

#### Блок декларации функции daq плагина

После **uses** необходим блок декларации **функции плагина**.

``` pascal
/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
```

Для обеспечения централизации и единства кода декларация делается с помощью
включаемого файла **[{$I _crw_plugin_declare.inc}](../../../crwlib/_crw_plugin_declare.inc)**.
Это позволяет управлять кодом декларации всех плагинов в одном файле.
А также обеспечивает правильность декларации во всех плагинах.
Заметим, что плагин использует соглашение о вызовах **ABI call** (_application binary interface_),
заданное в файле **[_crw_plugin_abicall.inc](../../../crwlib/_crw_plugin_abicall.inc)**.

#### Блок определения области данных daq плагина

После объявления функции плагина надо определить **область данных daq** плагина,
обычно это тип записи **TDaqDataSheet**.

``` pascal
 ////////////////////////////
 // Comment on TDaqDataSheet:
 // Hold all DAQ data there!!
 // Init all on DAQ_CMD_INIT.
 // Free all on DAQ_CMD_FREE.
 // Use data on DAQ_CMD_POLL.
 ////////////////////////////
type ////////////////////////
 TDaqDataSheet = packed record
  Sinus    : Double;
  Omega    : Double;
  timeSec  : Double;
  dllName  : LongString;
 end; //// TDaqDataSheet /////
 ////////////////////////////
```

Запись **TDaqDataSheet** должна содержать **ВСЕ** данные, нужные в процессе работы **daq** плагина.
Необходимость области данных **TDaqDataSheet** связана с тем, что плагин **сбора данных**, в отличие от
плагина **обработки данных**, вызывается многократно - в цикле опроса потока **DAQ** устройства.
Данные между вызовами надо где-то хранить - но не в стеке, который освобождается после вызова.
Разумно использовать для хранения данных интерфейс **CrwApi**, время жизни которого глобально -
точнее, оно совпадает с временем жизни загруженного плагина.
Поэтому все данные **daq** плагина хранятся в буфере памяти, доступ к которому дает
функция интерфейса **CrwApi.DaqApi.DaqDataSheet(n)**, где **n** - размер буфера памяти.
При этом, если значение **n** меняется, происходит перераспределение памяти.
Поэтому важно, чтобы значение **n** было постоянным в процессе опроса.
Для этого обычно используется конструкция **`with TDaqDataSheet(DaqDataSheet(SizeOf(TDaqDataSheet))^) do …`**,
которая ипользует **`SizeOf(TDaqDataSheet)`** для выделения буфера
и приведение типа **`TDaqDataSheet(DaqDataSheet(…)^)`** для доступа к данным в буфере.

Подчеркнем еще раз - **все** данные плагина надо размещать в **TDaqDataSheet**.  
Затем эти данные надо:

- Инициализировать по команде **`DAQ_CMD_INIT`**,  
- Освобождать по команде **`DAQ_CMD_FREE`**,  
- Использовать для измерений/расчетов/управления по команде **`DAQ_CMD_POLL`**.  

Клиентская часть программы **DaqPascal** должна обеспечить вызов этих команд
в процессе работы **DAQ**-системы.

#### Блок тела (содержания) daq плагина

После декларации **функции плагина** следует **тело плагина**, т.е. его содержательная часть.

В этом блоке следует:

- Использовать интерфейсы **`with CrwApi,SysApi,GuiApi,DaqApi do`** для доступа к методам библиотеки,  
- Выполнять код в блоке **try … except … end**.  
- Проверить версию и тип плагина вызовом **VerifyPluginDate**, **VerifyPluginType**,  
- По команде **DaqCommand** = **`DAQ_CMD_INIT`** выполнить инициализацию полей области данных **DaqDataSheet**:  
    + Перенаправить ввод-вывод в консоль **DAQ**-устройства вызовом **RedirectStdIn**, **RedirectStdOut**,  
    + Выделить необходимую память, создать нужные объекты или задать значения полей **TDaqDataSheet**,  
- По команде **DaqCommand** = **`DAQ_CMD_FREE`** выполнить освобождение (очистку, удаление) данных:  
    + Освободить все ресурсы, выделенные пи вызове команды инициализации **`DAQ_CMD_INIT`**,  
    + В конце вызвать **DaqDataSheet(0);** для удаления области данных **DAQ**.
- По команде **DaqCommand** = **`DAQ_CMD_POLL`** выполнить работу по сбору или обработке данных:  
    + Использовать поля из области данных **DaqDataSheet** для хранения переменных,  
    + Использовать интерфейсы **ICrwApi**, **ISysApi**, **IGuiApi**, **IDaqApi** для доступа к функциям и данным пакета.  

На практике это выглядит примерно так:

``` pascal
begin
 Result:=0;
 with CrwApi,SysApi,GuiApi,DaqApi do
 try
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAcquisition);
  ///////////////////////////////////////////////////////////////
  // Nota Bene: all driver data must be located in DaqDataSheet!!
  ///////////////////////////////////////////////////////////////
  with TDaqDataSheet(DaqDataSheet(SizeOf(TDaqDataSheet))^) do begin
   case DaqCommand of
    ////////////////////
    // Initialize driver
    ////////////////////
    DAQ_CMD_INIT: begin
     RedirectStdIn(Input);
     RedirectStdOut(Output);
     //////////////////////////////////////////////////
     // Initialize USER data located in DaqDataSheet //
     //////////////////////////////////////////////////
     dllName:=ExtractFileNameExt(PluginBinaryFileName);
     GuiApi.Echo(Format('Starting %s …',[dllName]));
     …
     … Инициализация данных TDaqDataSheet
     …
     //////////////////
     GuiApi.Echo('Ok');
    end;
    //////////////////
    // Finalize driver
    //////////////////
    DAQ_CMD_FREE: begin
     GuiApi.Echo(Format('Stopping %s …',[dllName]));
     //////////////////
     // Don`t forget to
     // Clear all data:
     //////////////////
     …
     … Освобождение данных TDaqDataSheet
     …
     //////////////////
     // Free DataSheet:
     //////////////////
     DaqDataSheet(0);//
     //////////////////
     GuiApi.Echo('Ok');
    end;
    ////////////////////
    // Main polling loop
    ////////////////////
    DAQ_CMD_POLL: begin
     /////////////////////////////
     // USER data processing code:
     /////////////////////////////
     …
     … Цикл сбора и обработки данных TDaqDataSheet
     …
     /////////////////////////////
     // USER data processing done.
     /////////////////////////////
    end;
   end; // case
  end; // with
 except
  on E:Exception do begin
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message);
   Result:=-1;
  end;
 end;
end;
```

Для выполнения полезной работы по сбору и обработке данных используются
интерфейсы **ICrwApi**, **ISysApi**, **IGuiApi**, **IDaqApi**,
содержащие большой набор функций для доступа к данным пакета.
Например, для доступа к последней точке **(x,y)** кривой, подключенной к аналоговому входу (_analog input_)
используются функции **getai_xn(i)**, **getai_yn(i)**, где i - номер аналогового входа.

Как правило, назначение функций и их аргументов понятно из названий.
Многие функции интерфейсов **ICrwApi**, **ISysApi**, **IDaqApi** совпадают
или имеют аналоги 

> Полный список и описание интерфейсов, классов, функций, переменных, констант
  модуля **`_crw_crwapi`** с описанием интерфейса **ICrwApi** подключения плагинов
  см. **[тут](../../../crwlib/manual/_crw_crwapi/index.html)**
  или **[там](../../../crwlib/crwlib.chm)**.

#### Блок экспорта функции daq плагина

В конце, после завершения **функции плагина**, необходимо **экспортировать** её:

``` pascal
/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.
```

Экспорт выполняется вставкой **[{$I _crw_plugin_exports.inc}](../../../crwlib/_crw_plugin_exports.inc)**,
что обеспечивает централизованное управление кодом плагинов.

### Шаблоны плагинов daq

Пакет содержит библиотеку образцов плагинов сбора данных **[daq](../plugin/daq/)**, созданных по одному шаблону.
Средства клонирования (копирования и сохранения под другим именем) позволяют создавать новые плагины,
взяв эти образцы плагинов в качестве шаблона.

Шаблоны уже содержат необходимую структуру для работы плагинов сбора данных.
Прикладному программисту нужно дополнить шаблон своим кодом, решающим требуемую задачу.
Большинство потребностей прикладного программирования закрывает интерфейс **ICrwApi**,
в состав которого входит еще ряд предметных интерфейсов:  

- **ISysApi** - общесистемные функции,  
- **IGuiApi** - функции графического интерфейса,  
- **IDaqApi** - функции для работы с **DAQ** системой.  

> Полный список и описание интерфейсов, классов, функций, переменных, констант
  модуля **`_crw_crwapi`** с описанием интерфейса **ICrwApi** подключения плагинов
  см. **[тут](../../../crwlib/manual/_crw_crwapi/index.html)**
  или **[там](../../../crwlib/crwlib.chm)**.

<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---

## Консольные утилиты

Используйте шаблон **[hello_world.lpr](../../resource/sample/lpr/hello_world/hello_world.lpr)**.  
А также модули библиотеки **[crwlib](../../../crwlib/)**.

> Полный список и описание интерфейсов, классов, функций, переменных, констант библиотеки **`crwlib`**
  см. **[тут](../../../crwlib/manual/index.html)**
  или **[там](../../../crwlib/crwlib.chm)**.

<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---

## Консольные драйверы

Используйте шаблон **[daq_user_task.lpr](../../resource/sample/lpr/daq_user_task/daq_user_task.lpr)**.  
А также модули библиотеки **[crwlib](../../../crwlib/)**.

> Полный список и описание интерфейсов, классов, функций, переменных, констант библиотеки **`crwlib`**
  см. **[тут](../../../crwlib/manual/index.html)**
  или **[там](../../../crwlib/crwlib.chm)**.

<a href="#toc" class="bold memo">Перейти к Содержанию</a>

---
