////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWKIT.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// _crw_fpqui is library to work with tooltip windows via FP-QUI package.     //
// FP-QUI is nice tooltip notification system made by Florian Pollak.         //
// FP-QUI Copyright 2010-2017 Florian Pollak (bfourdev@gmail.com).            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20170323 - Creation                                                        //
// 20170324 - First tested release                                            //
// 20190918 - ReadFpQuiEnv() to read FPQUI_DIR environment variable           //
// 20240129 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_fpqui;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 interfaces, lclintf, lmessages,
 {$IFDEF WINDOWS} jwatlhelp32, {$ENDIF}
 _crw_alloc, _crw_cmdargs, _crw_str, _crw_base64, _crw_fio, _crw_fifo,
 _crw_rtc, _crw_polling, _crw_task, _crw_wmctrl, _crw_ascio, _crw_az;

const                                           // Expected FP-QUI target dispatcher window parameters
 fpQuiClass     = 'AutoIt v3 GUI';              // Expected FP-QUI window class
 fpQuiTitle     = 'FP-QUI/dispatcherWindow';    // Expected FP-QUI window title
 fpQuiExeId     = 'FP-QUICore.exe';             // Expected FP-QUI EXE filename
 fpQuiMagic     = $21495551;                    // Uses to identify WM_COPYDATA = 558454097 = dump('QUI!')
 fpQuiTimeOut   = 5000;                         // Default timeout to run FP-QUI
 fpQuiDemoDelay = 1000;                         // Default delay for FP-QUI demo
 
const                                           // FpQuiManager.RunFpQuiTipExe result codes
 EcfpQuiSuccess = 0;                            // Success
 EcFpQuiNoFound = 1;                            // Target window not found
 EcFpQuiFailExe = 2;                            // Unexpected exe filename
 EcFpQuiBadData = 3;                            // Nothing to send, empty data
 EcFpQuiNotSent = 4;                            // Send operation failed or refused
 EcFpQuiBadArgs = 5;                            // Invalid parameters was specified
 EcfpQuiGenFail = 6;                            // General failure

{$IFDEF WINDOWS}

type
 TFpQuiManager = class(TObject)
 private
  myTargetClass  : LongString;
  myTargetTitle  : LongString;
  myTargetExeId  : LongString;
  myTargetMagic  : Int64;
 private
  function    GetTargetClass:LongString;    procedure SetTargetClass(const aClass:LongString);
  function    GetTargetTitle:LongString;    procedure SetTargetTitle(const aTitle:LongString);
  function    GetTargetExeId:LongString;    procedure SetTargetExeId(const aExeId:LongString);
  function    GetTargetMagic:Int64;         procedure SetTargetMagic(const aMagic:Int64);
 public
  // Target FP-QUI window Class, Title, Exe name ID and Magic command ID
  property    TargetClass:LongString        read GetTargetClass     write SetTargetClass;
  property    TargetTitle:LongString        read GetTargetTitle     write SetTargetTitle;
  property    TargetExeId:LongString        read GetTargetExeId     write SetTargetExeId;
  property    TargetMagic:Int64             read GetTargetMagic     write SetTargetMagic;
 private
  myActualWin     : HWND;
  myActualPid     : DWORD;
  myActualExe     : LongString;
  myActualExePath : LongString;
  myActualWorkDir : LongString;
 private
  function    GetActualWin:HWND;
  function    GetActualPid:DWORD;
  function    GetActualExe:LongString;
  function    GetActualExePath:LongString;
  function    GetActualWorkDir:LongString;
 public
  // Actual (found) target FP-QUI window handle, Process ID, Exe name
  property    ActualWin:HWND                read GetActualWin;
  property    ActualPid:DWORD               read GetActualPid;
  property    ActualExe:LongString          read GetActualExe;
  property    ActualExePath:LongString      read GetActualExePath;
  property    ActualWorkDir:LongString      read GetActualWorkDir;
 public
  // Find actual target window, process, exe file
  function    FindFpQuiWindow(const aClass,aTitle,aExeId:LongString):HWND;
  function    FindActualTarget(AllowRun:Boolean):Boolean;
 private
  myTimeOut       : Cardinal;
  myDemoDelay     : Cardinal;
  myErrno         : Integer;
 private
  function    GetTimeOut:Cardinal;          procedure SetTimeOut(aTimeOut:Cardinal);
  function    GetDemoDelay:Cardinal;        procedure SetDemoDelay(aDemoDelay:Cardinal);
  function    GetErrno:Integer;             procedure SetErrno(aErrno:Integer);
 public
  // Run timeout, demo delay, last error code
  property    TimeOut:Cardinal              read GetTimeOut         write SetTimeOut;
  property    DemoDelay:Cardinal            read GetDemoDelay       write SetDemoDelay;
  property    errno:Integer                 read GetErrno           write SetErrno;
 private
  myLog          : LongString;
  myVerbose      : Boolean;
  myProgramId    : LongString;
 private
  function    GetTheLog:LongString;         procedure SetTheLog(const aLog:LongString);
  function    GetVerbose:Boolean;           procedure SetVerbose(const aVerbose:Boolean);
  function    GetProgramId:LongString;      procedure SetProgramId(const aProgramId:LongString);
 public
  // Printing and logging features
  property    theLog:LongString             read GetTheLog          write SetTheLog;
  property    verbose:Boolean               read GetVerbose         write SetVerbose;
  property    ProgramId:LongString          read GetProgramId       write SetProgramId;
  procedure   Print(const S:LongString);
  function    PrintLog(const S:LongString):Boolean;
  function    Failure(code:Integer; const msg:LongString):Integer;
  function    Success(code:Integer; const msg:LongString):Integer;
  function    Usage(code:Integer=0):Integer;
 private
  par            : record
   xml           : LongString;
   avi           : LongString;
   text          : LongString;
   verbose       : LongString;
   trans         : LongString;
   font          : LongString;
   fontSize      : LongString;
   bkColor       : LongString;
   delay         : LongString;
   untilClickAny : LongString;
   ico           : LongString;
   audio         : LongString;
   textColor     : LongString;
   noDouble      : LongString;
   onClick       : LongString;
   progress      : LongString;
   guid          : LongString;
   delete        : LongString;
   sure          : LongString; // synonym createIfNotVisible
   run           : LongString;
   button        : LongString;
   btn1,cmd1     : LongString;
   btn2,cmd2     : LongString;
   btn3,cmd3     : LongString;
   btn4,cmd4     : LongString;
   btn5,cmd5     : LongString;
   btn6,cmd6     : LongString;
   btn7,cmd7     : LongString;
   btn8,cmd8     : LongString;
   btn9,cmd9     : LongString;
  end;
 private
  procedure   ClearAll;
  procedure   SetDefaults;
  procedure   PresetStd(const s1,s2,s3,s4,s5,s6,s7:LongString);
  procedure   ParseArgumentPair(const s1,s2:LongString);
  procedure   PresetParams(const s1:LongString);
  procedure   ParseButtons;
  procedure   ColorCheck;
  function    CheckParameters:Integer;
  function    ComposeMessage:LongString;
  function    RunFpQuiCoreExe(aTimeOut:DWORD):DWORD;
 public
  // General functions
  function    GetCmdLineArguments:LongString;                                           // Read data from command line
  function    ReadFromStdIn(aMaxLeng:Cardinal=1024*64):LongString;                      // Read data from StdIn stream
  function    ReadCmdLineOrStdIn(aMaxLeng:Cardinal=1024*64):LongString;                 // Read data from Cmd or StdIn
  function    ParseCommandLine(const CmdLine:LongString; Skip:Integer=0):LongString;    // Main command parser to XML
  function    SendMessage(const data:LongString; AllowRun:Boolean=true):Integer;        // Send message to FP-QUI
  function    RunFpQuiTipExe(arg:LongString=''):Integer;                                // Read data, parse and send
  function    RunDemo(aDelay:Cardinal=1000):Integer;                                    // Run list of demo commands
  procedure   Cleanup;                                                                  // Cleanup after data sent
 private
  mySearchPath    : LongString;
 public
  function    GetSearchPath:LongString;
  procedure   ClearSearchPath;
 public
  // Registry and file search routines
  function    ReadFpQuiEnv(const Name:LongString):LongString;       // Read FPQUI_NAME environment string
  function    ReadFpQuiReg(const Name:LongString):LongString;       // Read HKEY_LOCAL_MACHINE\SOFTWARE\FP-QUI\Name
  function    FindFile(const aFileName:LongString):LongString;      // Find a file in FP-QUI search locations
  function    FindFpQuiCoreExe:LongString;                          // Find FP-QUICore.exe
  function    FindFpQuiExe:LongString;                              // Find FP-QUI.exe
 public
  // Constructor/destructor
  constructor Create;
  destructor  Destroy; override;
  procedure   BeforeDestruction; override;
 end;

function FpQuiManager:TFpQuiManager;
procedure Kill(var TheObject:TFpQuiManager); overload;

const // Virtual console and exception handler
 TheFpQuiEchoProcedure : procedure(const Msg:LongString) = nil;
 TheFpQuiExceptionHandlerProcedure : procedure(E:Exception) = nil;

{$ENDIF ~WINDOWS}
 
implementation

{$IFDEF WINDOWS}

///////////////////
// Utility routines
///////////////////

const
 EnvBufferSize                  = 1024*32;
 myFpQuiManager : TFpQuiManager = nil;

procedure FpQuiExceptionHandler(E:Exception);
begin
 if Assigned(E) then
 if Assigned(TheFpQuiExceptionHandlerProcedure) then
 try
  TheFpQuiExceptionHandlerProcedure(E);
 except
 end;
end;

procedure Echo(const Msg:LongString; const LineBreak:LongString=EOL);
begin
 if Assigned(TheFpQuiEchoProcedure) then
 try
  TheFpQuiEchoProcedure(Msg+LineBreak);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function FpQuiManager:TFpQuiManager;
begin
 if not Assigned(myFpQuiManager) then
 try
  myFpQuiManager:=TFpQuiManager.Create;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
 Result:=myFpQuiManager;
end;

procedure Kill(var TheObject:TFpQuiManager); overload;
var P:TObject;
begin
 if Assigned(TheObject) then
 try
  P:=TheObject; TheObject:=nil; P.Free;  // clear the reference before destroying the object
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

///////////////////////////
// General purpose routines
///////////////////////////

 // Convert integer to string with leading zeros to given width
function IntToStrZ(i,w:Integer):LongString;
begin
 Result:=IntToStr(i);
 while Length(Result)<w do Result:='0'+Result;
end;

 // Convert string to integer or return default on error
function iValDef(const S:LongString; Def:Int64):Int64;
var code:Integer;
begin
 Val(S,Result,code); if code<>0 then Result:=Def;
end;

// Delete leading & trailing spaces & quotes
function TrimChars(const S:LongString; Spaces:TCharSet):LongString;
var I,L:Integer;
begin
 L:=Length(S); I:=1;
 while (I<=L) and (S[I] in Spaces) do Inc(I);
 if I>L then Result:='' else begin
  while S[L] in Spaces do Dec(L);
  Result:=Copy(S,I,L-I+1);
 end;
end;

// Delete leading & trailing spaces & quotes
function TrimSpacesQuotes(const S:LongString):LongString;
begin
 Result:=TrimChars(S,[#0..' ','"','''']);
end;

 // Attach a char to tail of string if one not present yet
function AttachTailChar(const S:LongString; c:Char):LongString;
begin
 Result:=S;
 if Length(Result)>0 then
 if Result[Length(Result)]<>c then Result:=Result+c;
end;

 // Return file base name without path & extension
function ExtractFileBaseName(const FileName:LongString):LongString;
begin
 Result:=ChangeFileExt(ExtractFileName(TrimSpacesQuotes(FileName)),'');
end;

 // Command line parser
function GetNextParamStr(P:PChar; var Param:LongString):PChar;
var nSize,Len:DWORD; Buffer:PChar;
begin
 if Assigned(P) then begin
  nSize:=EnvBufferSize;
  GetMem(Buffer,nSize);
  try
   while True do begin
    while (P[0]<>#0) and (P[0]<=' ') do Inc(P);
    if (P[0]='"') and (P[1]='"') then Inc(P,2) else Break;
   end;
   Len:=0;
   while (P[0]>' ') and (Len<nSize) do
   if P[0]='"' then begin
    Inc(P);
    while (P[0]<>#0) and (P[0]<>'"') do begin
     Buffer[Len]:=P[0];
     Inc(Len);
     Inc(P);
    end;
    if P[0]<>#0 then Inc(P);
   end else begin
    Buffer[Len]:=P[0];
    Inc(Len);
    Inc(P);
   end;
   SetString(Param,Buffer,Len);
  finally
   FreeMem(Buffer);
  end;
 end;
 Result:=P;
end;

//////////////////
// WinApi routines
//////////////////

 // Get window title
function GetWindowTitle(hWin:HWND):LongString;
begin
 Result:=wmctrl.WindowTitle(hWin);
end;

 // Get window class
function GetWindowClass(hWin:HWND):LongString;
begin
 Result:=wmctrl.WindowClass(hWin);
end;

 // Get process ID by window handle
function GetWindowProcessId(hWin:HWND):DWORD;
begin
 Result:=wmctrl.WindowPid(hWin);
end;

 // Read a string from registry
function ReadReqistryString(RootKey:DWORD; const Path,Name:LongString):LongString;
var key:HKEY; DataType,BufSize:DWORD; Buffer:LongString;
begin
 Result:=''; key:=0;
 if RegOpenKeyEx(RootKey,PChar(Path),0,KEY_READ,Key)=ERROR_SUCCESS then
 try
  Buffer:=StringBuffer(OS_PIPE_BUF); BufSize:=Length(Buffer);
  if RegQueryValueEx(Key,PChar(Name),nil,@DataType,PByte(Buffer),@BufSize)=ERROR_SUCCESS then
  if (DataType=REG_SZ) and InRange(BufSize,1,Length(Buffer)) then Result:=StrPas(PChar(Buffer));
 finally
  RegCloseKey(Key);
 end;
end;

 // Get exe filename by process ID
function GetExeNameByPid(pid:DWORD):LongString;
var NextProc:Boolean; SnapHandle:THandle; ProcEntry:TProcessEntry32;
begin
 Result:='';
 if pid<>0 then begin
  SnapHandle:=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if (SnapHandle<>0) and (SnapHandle<>INVALID_HANDLE_VALUE) then
  try
   ProcEntry.dwSize:=SizeOf(ProcEntry);
   NextProc:=Process32First(SnapHandle,ProcEntry);
   while NextProc do begin
    if ProcEntry.th32ProcessID=pid then begin
     Result:=TrimSpacesQuotes(ProcEntry.szExeFile);
     Break;
    end;
    NextProc:=Process32Next(SnapHandle,ProcEntry);
   end;
  finally
   CloseHandle(SnapHandle);
  end;
 end;
end;

 // Find process ID (first of them) by exe filename
function FindProcessPid(const exe:LongString):DWORD;
var NextProc:Boolean; SnapHandle:THandle; ProcEntry:TProcessEntry32;
begin
 Result:=0;
 if Length(exe)>0 then begin
  SnapHandle:=CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS,0);
  if (SnapHandle<>0) and (SnapHandle<>INVALID_HANDLE_VALUE) then
  try
   ProcEntry.dwSize:=SizeOf(ProcEntry);
   NextProc:=Process32First(SnapHandle,ProcEntry);
   while NextProc do begin
    if SameText(exe,ProcEntry.szExeFile) then begin
     Result:=ProcEntry.th32ProcessID;
     Break;
    end;
    NextProc:=Process32Next(SnapHandle,ProcEntry);
   end;
  finally
   CloseHandle(SnapHandle);
  end;
 end;
end;

 // Append directory dir\subdir list to search path
 // Parameter subrirs is semicolon separated list of subdirectories
procedure AppendDirToSearchPath(var path:LongString; dir,subdirs:LongString);
var p:Integer; subdir:LongString;
begin
 dir:=Trim(dir); subdirs:=Trim(subdirs);
 if (Length(dir)>0) and DirectoryExists(dir) then begin
  repeat
   p:=Pos(';',subdirs);
   if p>0 then begin
    subdir:=Trim(Copy(subdirs,1,p-1));
    subdirs:=Trim(Copy(subdirs,p+1,Length(subdirs)-p));
   end else begin
    subdir:=Trim(subdirs);
    subdirs:='';
   end;
   if (Length(subdir)>0) then begin
    subdir:=ExpandFileName(AttachTailChar(dir,'\')+subdir);
    if DirectoryExists(subdir) then begin
     p:=Pos(subdir,path);
     if p>1 then if path[p-1]<>';' then p:=0;
     if p>0 then if p+Length(subdir)<=Length(path) then if path[p+Length(subdir)]<>';' then p:=0;
     if p=0 then path:=AttachTailChar(path,';')+subdir;
    end;
   end;
  until Length(subdirs)=0;
 end;
 path:=AttachTailChar(path,';');
end;

function SHGetSpecialFolderPath(hwndOwner:HWND; lpszPath:PChar;  nFolder:Integer; fCreate:BOOL):BOOL; stdcall;
external 'shell32.dll' name 'SHGetSpecialFolderPathA';

const                               // shlobj.h
 CSIDL_APPDATA          = $001A;    // C:\Documents and Settings\%User%\Application Data
 CSIDL_LOCAL_APPDATA    = $001C;    // C:\Documents and Settings\%User%\Local Settings\Application Data
 CSIDL_COMMON_APPDATA   = $0023;    // C:\Documents and Settings\All Users\Application Data
 CSIDL_COMMON_DOCUMENTS = $002E;    // C:\Documents and Settings\All Users\Documents
 
 // Get shecial folder path by CSIDL
function GetSpecialShellFolderPath(CSIDL:Word):LongString;
var Buff:TMaxPathBuffer;
begin
 if SHGetSpecialFolderPath(0,Buff,CSIDL,True) then Result:=Buff else Result:='';
end;

 // Get temporary directory
function GetTempDir:LongString;
var Buff:TMaxPathBuffer;
begin
 SetString(Result,Buff,GetTempPath(SizeOf(Buff),Buff));
end;

 // Return shared work directory - Common Documents or (as fallback) Common Application Data or TEMP
function GetSharedWorkDir:LongString;
begin
 Result:='';
 if (Length(Result)=0) or not DirectoryExists(Result) then Result:=GetSpecialShellFolderPath(CSIDL_COMMON_DOCUMENTS);
 if (Length(Result)=0) or not DirectoryExists(Result) then Result:=GetSpecialShellFolderPath(CSIDL_COMMON_APPDATA);
 if (Length(Result)=0) or not DirectoryExists(Result) then Result:=GetEnv('TEMP');
 if (Length(Result)=0) or not DirectoryExists(Result) then Result:=GetTempDir;
 Result:=TrimSpacesQuotes(Result);
 if Length(Result)>0 then if Result[Length(Result)] in ['\','/'] then Result:=Copy(Result,1,Length(Result)-1);
end;

 // Send WM_COPYDATA message to window and return result from target handler
function wmCopyDataSend(hWin:HWND; Data:PChar; Size:Cardinal; aMagic:Cardinal):LRESULT;
var DataRec:TCopyDataStruct;
begin
 Result:=0;
 if Assigned(Data) and (Size>0) and IsWindow(hWin) then begin
  DataRec.dwData:=aMagic;
  DataRec.cbData:=Size;
  DataRec.lpData:=Data;
  Result:=SendMessage(hWin,WM_COPYDATA,GetConsoleWindow,PointerToPtrInt(@DataRec));
 end;
end;

/////////////////////////////////////
// TFpQuiManager class implementation
/////////////////////////////////////
constructor TFpQuiManager.Create;
begin
 inherited Create;
 SetDefaults;
 myTimeOut:=fpQuiTimeOut;
 myDemoDelay:=fpQuiDemoDelay;
 ProgramId:=ExtractFileBaseName(ParamStr(0));
end;

destructor TFpQuiManager.Destroy;
begin
 ClearAll;
 inherited Destroy;
end;

procedure TFpQuiManager.BeforeDestruction;
begin
 if Self=myFpQuiManager then myFpQuiManager:=nil;
 inherited BeforeDestruction;
end;

function TFpQuiManager.GetTargetClass:LongString;
begin
 if Assigned(Self) then Result:=myTargetClass else Result:='';
end;

procedure TFpQuiManager.SetTargetClass(const aClass:LongString);
begin
 if Assigned(Self) then myTargetClass:=aClass;
end;

function TFpQuiManager.GetTargetTitle:LongString;
begin
 if Assigned(Self) then Result:=myTargetTitle else Result:='';
end;

procedure TFpQuiManager.SetTargetTitle(const aTitle:LongString);
begin
 if Assigned(Self) then myTargetTitle:=aTitle;
end;

function TFpQuiManager.GetTargetExeId:LongString;
begin
 if Assigned(Self) then Result:=myTargetExeId else Result:='';
end;

procedure TFpQuiManager.SetTargetExeId(const aExeId:LongString);
begin
 if Assigned(Self) then myTargetExeId:=aExeId;
end;

function TFpQuiManager.GetTargetMagic:Int64;
begin
 if Assigned(Self) then Result:=myTargetMagic else Result:=0;
end;

procedure TFpQuiManager.SetTargetMagic(const aMagic:Int64);
begin
 if Assigned(Self) then myTargetMagic:=aMagic;
end;

function TFpQuiManager.GetErrno:Integer;
begin
 if Assigned(Self) then Result:=myErrno else Result:=0;
end;

procedure TFpQuiManager.SetErrno(aErrno:Integer);
begin
 if Assigned(Self) then myErrno:=aErrno;
end;

function TFpQuiManager.GetTimeOut:Cardinal;
begin
 if Assigned(Self) then Result:=myTimeOut else Result:=0;
end;

procedure TFpQuiManager.SetTimeOut(aTimeOut:Cardinal);
begin
 if Assigned(Self) then myTimeOut:=aTimeOut;
end;

function TFpQuiManager.GetDemoDelay:Cardinal;
begin
 if Assigned(Self) then Result:=myDemoDelay else Result:=0;
end;

procedure TFpQuiManager.SetDemoDelay(aDemoDelay:Cardinal);
begin
 if Assigned(Self) then myDemoDelay:=aDemoDelay;
end;

function TFpQuiManager.GetTheLog:LongString;
begin
 if Assigned(Self) then Result:=myLog else Result:='';
end;

procedure TFpQuiManager.SetTheLog(const aLog:LongString);
begin
 if Assigned(Self) then myLog:=TrimSpacesQuotes(aLog);
end;

function TFpQuiManager.GetVerbose:Boolean;
begin
 if Assigned(Self) then Result:=myVerbose else Result:=false;
end;

procedure TFpQuiManager.SetVerbose(const aVerbose:Boolean);
begin
 if Assigned(Self) then myVerbose:=aVerbose;
end;

function TFpQuiManager.GetProgramId:LongString;
begin
 if Assigned(Self) then Result:=myProgramId else Result:='';
end;

function TFpQuiManager.GetActualWin:HWND;
begin
 if Assigned(Self) then Result:=myActualWin else Result:=0;
end;

function TFpQuiManager.GetActualPid:DWORD;
begin
 if Assigned(Self) then Result:=myActualPid else Result:=0;
end;

function TFpQuiManager.GetActualExe:LongString;
begin
 if Assigned(Self) then Result:=myActualExe else Result:='';
end;

function TFpQuiManager.GetActualExePath:LongString;
begin
 if Assigned(Self) then Result:=myActualExePath else Result:='';
end;

function TFpQuiManager.GetActualWorkDir:LongString;
begin
 if Assigned(Self) then Result:=myActualWorkDir else Result:='';
end;

procedure TFpQuiManager.SetProgramId(const aProgramId:LongString);
begin
 if Assigned(Self) then myProgramId:=aProgramId;
end;

 // Print time stamp and message to log file if one specified
function TFpQuiManager.PrintLog(const S:LongString):Boolean;
var F:Text; t:TSystemTime;
begin
 Result:=false;
 if Length(S)>0 then
 if Assigned(Self) then
 if Length(myLog)>0 then
 try
  Assign(F,myLog);
  try
   IOResult;
   SafeFillChar(t,SizeOf(t),0); GetSystemTime(t);
   if FileExists(myLog) then Append(F) else Rewrite(F);
   if IOResult=0 then writeln(F,IntToStrZ(t.wYear,4)+'.'+IntToStrZ(t.wMonth,2)+'.'+IntToStrZ(t.wDay,2)+'-'
                               +IntToStrZ(t.wHour,2)+':'+IntToStrZ(t.wMinute,2)+':'+IntToStrZ(t.wSecond,2)+'=> '+S);
  finally
   Close(F);
   IOResult;
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Print message to virtual console and log file
procedure TFpQuiManager.Print(const S:LongString);
begin
 if Assigned(Self) then
 try
  if Length(myLog)>0 then PrintLog(S);
  Echo(S);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Report error and set error code
function TFpQuiManager.Failure(code:Integer; const msg:LongString):Integer;
begin
 Result:=code;
 if Assigned(Self) then begin
  if Length(msg)>0 then Print('Failure: '+msg);
  errno:=code;
 end;
end;

 // Report success and set process error code
function TFpQuiManager.Success(code:Integer; const msg:LongString):Integer;
begin
 Result:=code;
 if Assigned(Self) then begin
  if Length(msg)>0 then Print('Success: '+msg);
  errno:=code;
 end;
end;

 // Show help screen end exit process
function TFpQuiManager.Usage(code:Integer=0):Integer;
var sAbout:LongString;
begin
 Result:=code;
 if Assigned(Self) then
 try
  sAbout        := ProgramId+' Copyright(c) Alexey Kuryakin, 2019 <kouriakine@mail.ru>.'+EOL
                  +ProgramId+' - program to show tooltip notifications via FP-QUI.'+EOL
                  +'Data to show may come from stdin or may be specified as parameter.'+EOL
                  +'FP-QUI is nice tooltip notification system made by Florian Pollak.'+EOL
                  +'FP-QUI Copyright(c) 2010-2017 Florian Pollak (bfourdev@gmail.com).'+EOL
                  +'Usage: '+EOL
                  +' '+ProgramId+ ' [-o [p]] [d]'+EOL
                  +' -o - option identifier'+EOL
                  +' p  - option parameter'+EOL
                  +' d  - data to send'+EOL
                  +'Options:'+EOL
                  +' -h, --help     - show help screen'+EOL
                  +' -v, --verbose  - set verbose mode   (default OFF)'+EOL
                  +' -c, --class c  - set window class c (default '+fpQuiClass+')'+EOL
                  +' -t, --title t  - set window title t (default '+fpQuiTitle+')'+EOL
                  +' -e, --exe   f  - set exe filename f (default '+fpQuiExeId+')'+EOL
                  +' -m, --magic n  - set magic number n (default '+Format('$%x',[fpQuiMagic])+')'+EOL
                  +' -d, --data  d  - set data to send d (specified in XML format)'+EOL
                  +' -l, --log   l  - set log filename l (default OFF)'+EOL
                  +'Data [d] specify message to send. '+EOL
                  +'That is list of pairs NAME VALUE which both must be non-empty.'+EOL
                  +' NAME           VALUE           - Comment'+EOL
                  +' preset         xxx             - Preset predefined parameters named `xxx`, see below:'+EOL
                  +'                xxx             - stdOk,stdNo,stdHelp,stdStop,stdDeny,stdAbort,stdError,stdFails,stdSiren,'+EOL
                  +'                                - stdAlarm,stdAlert,stdBreak,stdCancel,stdNotify,stdTooltip,stdSuccess,'+EOL
                  +'                                - stdWarning,stdQuestion,stdException,stdAttention,stdInformation,'+EOL
                  +'                                - stdExclamation'+EOL
                  +'                                - NB: preset xxx should be first in the list of parameters'+EOL
                  +' verbose        1               - Verbose mode  (default = 0 is silent execution)'+EOL
                  +' noDouble       1               - noDouble mode (default = 0 is enable double text)'+EOL
                  +' text           "Hello world"   - Text to display'+EOL
                  +' delay          15000           - Close after specified time ms (default 86400000 ms = day)'+EOL
                  +' audio          notify.wav      - Play sound (*.wav file)'+EOL
                  +' wav            notify.wav      - Synonym of audio'+EOL
                  +' ico            notify.ico      - Show icon file (*.ico,*.exe,*.dll)'+EOL
                  +' avi            alert.avi       - Show avi 32x32 icon file (*.avi)'+EOL
                  +' bkColor        red             - Background color (default violet) '+EOL
                  +' textColor      0xFF0000        - Text color (default black)'+EOL
                  +'                                - Colors: black,white,red,green,blue,purple,orange,violet'+EOL
                  +'                                -  or Hex RGB as  0xRRGGBB, for example violet=0xBC8BDA'+EOL
                  +' trans          255             - Transparency 0..255 (default 255)'+EOL
                  +' font           "PT Mono"       - Font family name (default Tahoma)'+EOL
                  +' fontSize       16              - Font size, pt (default 16)'+EOL
                  +' onClick        "cmd arg"       - On click run command "cmd arg"'+EOL
                  +' cmd0           "cmd arg"       - Symonym of onClick'+EOL
                  +' untilClickAny  1               - Close on any click (default 1)'+EOL
                  +' btn1..btn9     "Button label"  - Button 1..9 label text'+EOL
                  +' cmd1..cmd9     "cmd arg"       - Button 1..9 command'+EOL
                  +' guid           "{..}"          - Assign target window GUID which uses'+EOL
                  +'                                - to refresh updateable windows content'+EOL
                  +' sure           0/1             - Synonym of createIfNotVisible flag'+EOL
                  +'                                - Uses to be sure that GUID window appears'+EOL
                  +' createIfNotVisible 0/1         - 0/1=not/create window if one not visible'+EOL
                  +'                                - Uses only with guid option; default=1'+EOL
                  +' delete         "{..guid..}"    - delete message window with given GUID'+EOL
                  +' progress       p               - Show progress bar, p=0..100 percent'+EOL
                  +'                                - Uses with GUID to update progress'+EOL
                  +' run            "cmd arg"       - Run command on message received'+EOL
                  +'                                - Command runs only if window visible'+EOL
                  +' xml            "<x>..</x>"     - Append xml expression to message'+EOL
                  +'Exit codes:'+EOL
                  +' 0 - data sent successfully'+EOL
                  +' 1 - target window not found'+EOL
                  +' 2 - unexpected exe filename'+EOL
                  +' 3 - nothing to send, i.e. empty data'+EOL
                  +' 4 - send operation failed or refused'+EOL
                  +' 5 - invalid parameters was specified'+EOL
                  +' 6 - general fault'+EOL
                  +'Note:'+EOL
                  +' '+ProgramId+' search FP-QUI directory:'+EOL
                  +' 1) In FPQUI_DIR environment variable'+EOL
                  +' 2) In HKEY_LOCAL_MACHINE\SOFTWARE\FP-QUI\dir'+EOL
                  +' 3) In ProgramFiles,CommonProgramFiles\FP-QUI'+EOL
                  +'Examples:'+EOL
                  +' 1) Help: '+ProgramId+' --help '+EOL
                  +' 2) Args: '+ProgramId+' -d "<text>Hello world.</text>"'+EOL
                  +' 3) Pipe: cmd /c echo xml "<text>Hello world.</text>" | '+ProgramId+EOL
                  +' 4) Logs: '+ProgramId+' -v -l %temp%\'+ProgramId+'.log xml "<text>Hello world.</text>"'+EOL
                  +' 5) Data: '+ProgramId+' text "Hello world." preset stdTooltip delay 15000'+EOL
                  +'';
  Print(sAbout);
  errno:=code;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Run FP-QUI.exe and wait initialization
function TFpQuiManager.RunFpQuiCoreExe(aTimeOut:DWORD):DWORD;
var si:STARTUPINFOA; pi:PROCESS_INFORMATION; tick:QWORD;
begin 
 Result:=0;
 if Assigned(Self) then
 try
  if (Length(ActualWorkDir)=0) or not DirectoryExists(ActualWorkDir)
  then myActualWorkDir:=GetSharedWorkDir;
  if (Length(ActualExePath)=0) or not FileExists(ActualExePath)
  then myActualExePath:=TrimSpacesQuotes(FindFpQuiCoreExe);
  if (Length(ActualExePath)>0) and FileExists(ActualExePath) then begin
   ZeroMemory(@pi,sizeof(pi));
   ZeroMemory(@si,sizeof(si)); si.cb:=sizeof(si);
   si.dwFlags:=STARTF_USESHOWWINDOW; si.wShowWindow:=SW_SHOW;
   if CreateProcess(PChar(ActualExePath),nil,nil,nil,FALSE,0,nil,PChar(ActualWorkDir),si,pi) then begin
    Result:=pi.dwProcessId; WaitForInputIdle(pi.hProcess,aTimeOut);
    tick:=GetTickCount64;
    while (GetTickCount64<tick+aTimeOut)
    and (WaitForSingleObject(pi.hProcess,0)=WAIT_TIMEOUT)
    and not IsWindow(FindFpQuiWindow(TargetClass,TargetTitle,TargetExeId))
    do Sleep(10);
    CloseHandle(pi.hProcess); CloseHandle(pi.hThread);
   end;
   if verbose then if Result<>0
   then Print('Launch "'+ActualExePath+'" PID '+IntToStr(Result))
   else Print('Failed "'+ActualExePath+'"');
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;


procedure TFpQuiManager.ClearAll;
begin
 if Assigned(Self) then
 try
  myTargetClass:='';
  myTargetTitle:='';
  myTargetExeId:='';
  myTargetMagic:=0;
  myActualWin:=0;
  myActualPid:=0;
  myActualExe:='';
  myActualExePath:='';
  myActualWorkDir:='';
  myLog:='';
  myErrno:=0;
  myTimeOut:=0;
  myDemoDelay:=0;
  myVerbose:=false;
  myProgramId:='';
  par.xml:='';
  par.avi:='';
  par.text:='';
  par.verbose:='';
  par.trans:='';
  par.font:='';
  par.fontSize:='';
  par.bkColor:='';
  par.delay:='';
  par.untilClickAny:='';
  par.ico:='';
  par.audio:='';
  par.textColor:='';
  par.noDouble:='';
  par.onClick:='';
  par.progress:='';
  par.guid:='';
  par.delete:='';
  par.sure:=''; // synonym createIfNotVisible
  par.run:='';
  par.button:='';
  par.btn1:='';
  par.cmd1:='';
  par.btn2:='';
  par.cmd2:='';
  par.btn3:='';
  par.cmd3:='';
  par.btn4:='';
  par.cmd4:='';
  par.btn5:='';
  par.cmd5:='';
  par.btn6:='';
  par.cmd6:='';
  par.btn7:='';
  par.cmd7:='';
  par.btn8:='';
  par.cmd8:='';
  par.btn9:='';
  par.cmd9:='';
  mySearchPath:='';
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

procedure TFpQuiManager.SetDefaults;
begin
 if Assigned(Self) then
 try
  myTargetClass:=fpQuiClass;
  myTargetTitle:=fpQuiTitle;
  myTargetExeId:=fpQuiExeId;
  myTargetMagic:=fpQuiMagic;
  myLog:='';
  myErrno:=0;
  myVerbose:=false;
  par.xml:='';
  par.avi:='';
  par.text:='?';
  par.verbose:='0';
  par.trans:='255';
  par.font:='Tahoma';
  par.fontSize:='16';
  par.bkColor:='0xBC8BDA';
  par.delay:='86400000';
  par.untilClickAny:='1';
  par.ico:='default.ico';
  par.audio:='default.wav';
  par.textColor:='0x000000';
  par.noDouble:='0';
  par.onClick:='?';
  par.progress:='';
  par.guid:='';
  par.delete:='';
  par.sure:='1'; // synonym createIfNotVisible
  par.run:='';
  par.button:='';
  par.btn1:='';
  par.cmd1:='';
  par.btn2:='';
  par.cmd2:='';
  par.btn3:='';
  par.cmd3:='';
  par.btn4:='';
  par.cmd4:='';
  par.btn5:='';
  par.cmd5:='';
  par.btn6:='';
  par.cmd6:='';
  par.btn7:='';
  par.cmd7:='';
  par.btn8:='';
  par.cmd8:='';
  par.btn9:='';
  par.cmd9:='';
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

procedure TFpQuiManager.Cleanup;
begin
 if Assigned(Self) then SetDefaults;
end;

procedure TFpQuiManager.PresetStd(const s1,s2,s3,s4,s5,s6,s7:LongString);
begin
 if Assigned(Self) then
 try
  par.font:=TrimSpacesQuotes(s1);
  par.fontSize:=TrimSpacesQuotes(s2);
  par.bkColor:=TrimSpacesQuotes(s3);
  par.ico:=TrimSpacesQuotes(s4);
  par.audio:=TrimSpacesQuotes(s5);
  par.avi:=TrimSpacesQuotes(s6);
  par.noDouble:=TrimSpacesQuotes(s7);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

procedure TFpQuiManager.PresetParams(const s1:LongString);
begin
 if Assigned(Self) then
 try
  // Preset predefined standard parameters         font            fontSize bkColor   ico                audio              avi                noDouble
  if SameText(s1,'stdOk') then          PresetStd( 'PT Mono Bold', '16',    'green',  'ok.ico',          'ok.wav',          'ok.avi',          '1' ) else
  if SameText(s1,'stdNo') then          PresetStd( 'PT Mono Bold', '16',    'red',    'no.ico',          'no.wav',          'no.avi',          '1' ) else
  if SameText(s1,'stdHelp') then        PresetStd( 'PT Mono Bold', '16',    'blue',   'help.ico',        'help.wav',        'help.avi',        '1' ) else
  if SameText(s1,'stdStop') then        PresetStd( 'PT Mono Bold', '16',    'red',    'stop.ico',        'stop.wav',        'stop.avi',        '1' ) else
  if SameText(s1,'stdDeny') then        PresetStd( 'PT Mono Bold', '16',    'red',    'deny.ico',        'deny.wav',        'deny.avi',        '1' ) else
  if SameText(s1,'stdAbort') then       PresetStd( 'PT Mono Bold', '16',    'red',    'abort.ico',       'abort.wav',       'abort.avi',       '1' ) else
  if SameText(s1,'stdError') then       PresetStd( 'PT Mono Bold', '16',    'red',    'error.ico',       'error.wav',       'error.avi',       '1' ) else
  if SameText(s1,'stdFails') then       PresetStd( 'PT Mono Bold', '16',    'red',    'fails.ico',       'fails.wav',       'fails.avi',       '1' ) else
  if SameText(s1,'stdSiren') then       PresetStd( 'PT Mono Bold', '16',    'red',    'siren.ico',       'siren.wav',       'siren.avi',       '1' ) else
  if SameText(s1,'stdAlarm') then       PresetStd( 'PT Mono Bold', '16',    'red',    'alarm.ico',       'alarm.wav',       'alarm.avi',       '1' ) else
  if SameText(s1,'stdAlert') then       PresetStd( 'PT Mono Bold', '16',    'red',    'alert.ico',       'alert.wav',       'alert.avi',       '1' ) else
  if SameText(s1,'stdBreak') then       PresetStd( 'PT Mono Bold', '16',    'red',    'break.ico',       'break.wav',       'break.avi',       '1' ) else
  if SameText(s1,'stdCancel') then      PresetStd( 'PT Mono Bold', '16',    'red',    'cancel.ico',      'cancel.wav',      'cancel.avi',      '1' ) else
  if SameText(s1,'stdNotify') then      PresetStd( 'PT Mono Bold', '16',    'green',  'notify.ico',      'notify.wav',      'notify.avi',      '1' ) else
  if SameText(s1,'stdTooltip') then     PresetStd( 'PT Mono Bold', '16',    'violet', 'tooltip.ico',     'tooltip.wav',     'tooltip.avi',     '1' ) else
  if SameText(s1,'stdSuccess') then     PresetStd( 'PT Mono Bold', '16',    'green',  'success.ico',     'success.wav',     'success.avi',     '1' ) else
  if SameText(s1,'stdWarning') then     PresetStd( 'PT Mono Bold', '16',    'yellow', 'warning.ico',     'warning.wav',     'warning.avi',     '1' ) else
  if SameText(s1,'stdQuestion') then    PresetStd( 'PT Mono Bold', '16',    'blue',   'question.ico',    'question.wav',    'question.avi',    '1' ) else
  if SameText(s1,'stdException') then   PresetStd( 'PT Mono Bold', '16',    'red',    'exception.ico',   'exception.wav',   'exception.avi',   '1' ) else
  if SameText(s1,'stdAttention') then   PresetStd( 'PT Mono Bold', '16',    'yellow', 'attention.ico',   'attention.wav',   'attention.avi',   '1' ) else
  if SameText(s1,'stdInformation') then PresetStd( 'PT Mono Bold', '16',    'blue',   'information.ico', 'information.wav', 'information.avi', '1' ) else
  if SameText(s1,'stdExclamation') then PresetStd( 'PT Mono Bold', '16',    'yellow', 'exclamation.ico', 'exclamation.wav', 'exclamation.avi', '1' );
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

procedure TFpQuiManager.ParseArgumentPair(const s1,s2:LongString);
begin
 if Assigned(Self) then
 try
  if Length(s1)=0 then exit;
  if Length(s2)=0 then exit;
  if SameText(s1,'preset') then                                  PresetParams(s2)         else
  if SameText(s1,'verbose') then     begin   par.verbose         :=  TrimSpacesQuotes(s2); verbose:=SameText(par.verbose,'1') end else
  if SameText(s1,'xml') then                 par.xml := par.xml  +   TrimSpacesQuotes(s2) else
  if SameText(s1,'ico') then                 par.ico             :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'text') then                par.text            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'font') then                par.font            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'delay') then               par.delay           :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'audio') then               par.audio           :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'wav') then                 par.audio           :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'avi') then                 par.avi             :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'trans') then               par.trans           :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'bkColor') then             par.bkColor         :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'fontSize') then            par.fontSize        :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'textColor') then           par.textColor       :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'untilClickAny') then       par.untilClickAny   :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'noDouble') then            par.noDouble        :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'onClick') then             par.onClick         :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'progress') then            par.progress        :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'guid') then                par.guid            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'delete') then              par.delete          :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'createIfNotVisible') then  par.sure            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'sure') then                par.sure            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'run') then                 par.run             :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn0') then                par.text            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd0') then                par.onClick         :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn1') then                par.btn1            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd1') then                par.cmd1            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn2') then                par.btn2            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd2') then                par.cmd2            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn3') then                par.btn3            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd3') then                par.cmd3            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn4') then                par.btn4            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd4') then                par.cmd4            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn5') then                par.btn5            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd5') then                par.cmd5            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn6') then                par.btn6            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd6') then                par.cmd6            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn7') then                par.btn7            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd7') then                par.cmd7            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn8') then                par.btn8            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd8') then                par.cmd8            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'btn9') then                par.btn9            :=  TrimSpacesQuotes(s2) else
  if SameText(s1,'cmd9') then                par.cmd9            :=  TrimSpacesQuotes(s2);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

procedure TFpQuiManager.ColorCheck;
begin
 if Assigned(Self) then
 try
  if SameText(par.bkColor,'default')   then par.bkColor:='violet';
  if SameText(par.textColor,'default') then par.textColor:='black';
  if SameText(par.bkColor,'violet')    then par.bkColor:='0xBC8BDA';
  if SameText(par.textColor,'violet')  then par.textColor:='0xBC8BDA';
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function XmlTag(tag,value:LongString):LongString;
begin
 tag:=TrimSpacesQuotes(tag);
 value:=TrimSpacesQuotes(value);
 if (Length(tag)>0) and (Length(value)>0)
 then Result:='<'+tag+'>'+value+'</'+tag+'>'
 else Result:='';
end;

procedure TFpQuiManager.ParseButtons;
begin
 if Assigned(Self) then with par do
 try
  if Length(btn1)>0 then if Length(cmd1)>0 then button:=button+XmlTag('ID1',XmlTag('label',btn1)+XmlTag('cmd',cmd1)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn2)>0 then if Length(cmd2)>0 then button:=button+XmlTag('ID2',XmlTag('label',btn2)+XmlTag('cmd',cmd2)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn3)>0 then if Length(cmd3)>0 then button:=button+XmlTag('ID3',XmlTag('label',btn3)+XmlTag('cmd',cmd3)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn4)>0 then if Length(cmd4)>0 then button:=button+XmlTag('ID4',XmlTag('label',btn4)+XmlTag('cmd',cmd4)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn5)>0 then if Length(cmd5)>0 then button:=button+XmlTag('ID5',XmlTag('label',btn5)+XmlTag('cmd',cmd5)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn6)>0 then if Length(cmd6)>0 then button:=button+XmlTag('ID6',XmlTag('label',btn6)+XmlTag('cmd',cmd6)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn7)>0 then if Length(cmd7)>0 then button:=button+XmlTag('ID7',XmlTag('label',btn7)+XmlTag('cmd',cmd7)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn8)>0 then if Length(cmd8)>0 then button:=button+XmlTag('ID8',XmlTag('label',btn8)+XmlTag('cmd',cmd8)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
  if Length(btn9)>0 then if Length(cmd9)>0 then button:=button+XmlTag('ID9',XmlTag('label',btn9)+XmlTag('cmd',cmd9)+XmlTag('font',font)+XmlTag('fontSize',fontSize));
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function TFpQuiManager.ComposeMessage:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  if SameText(par.text,'?') then par.text:='';
  if SameText(par.sure,'0') then par.sure:='';
  if SameText(par.onClick,'?') then par.onClick:='';
  if SameText(par.noDouble,'0') then par.noDouble:='';
  if SameText(par.untilClickany,'0') then par.untilClickany:='';
  if Length(par.guid)>0 then
  Result:=Result+XmlTag('guid',                 par.guid)
                +XmlTag('createIfNotVisible',   par.sure);
  Result:=Result+XmlTag('delete',               par.delete)
                +XmlTag('progress',             par.progress)
                +XmlTag('ico',                  par.ico)
                +XmlTag('bkColor',              par.bkColor)
                +XmlTag('textColor',            par.textColor)
                +XmlTag('trans',                par.trans)
                +XmlTag('font',                 par.font)
                +XmlTag('fontSize',             par.fontSize)
                +XmlTag('untilClick',           XmlTag('any',   par.untilClickany))
                +XmlTag('onClick',              XmlTag('any',   par.onClick))
                +XmlTag('audio',                XmlTag('path',  par.audio))
                +XmlTag('run',                  XmlTag('cmd',   par.run))
                +XmlTag('delay',                par.delay)
                +XmlTag('noDouble',             par.noDouble)
                +XmlTag('text',                 par.text)
                +XmlTag('button',               par.button)
                +XmlTag('avi',                  par.avi);
  if Length(par.xml)>0 then Result:=Result+par.xml;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Parse command line parameters
function TFpQuiManager.ParseCommandLine(const CmdLine:LongString; Skip:Integer=0):LongString;
var Index:Integer; Ident,Param:LongString; P:PChar;
 procedure NextParam(var P:PChar; var Param:LongString; var Index:Integer);
 begin
  P:=GetNextParamStr(P,Param); Inc(Index);
 end;
begin
 Result:='';
 P:=PChar(CmdLine);
 if Assigned(P) then
 if Assigned(Self) then
 try
  Index:=0;
  Param:='';
  SetDefaults;
  while true do begin
   NextParam(P,Param,Index);
   if Index<=Skip then continue;
   if Length(Param)=0 then Break;
   // Option -v, --verbose : Set verbose mode
   if SameText(Param,'-v') or SameText(Param,'--verbose') then begin
    par.verbose:='1';
    myVerbose:=true;
    continue;
   end;
   // Option -l, --log : Set log filename
   if SameText(Param,'-l') or SameText(Param,'--log') then begin
    NextParam(P,Param,Index);
    theLog:=Param;
    continue;
   end;
   // Option -d, --data : Assign data to send
   if SameText(Param,'-d') or SameText(Param,'--data') then begin
    NextParam(P,Param,Index);
    ParseArgumentPair('xml',Param);
    continue;
   end;
   // Option -c, --class : Assign window class
   if SameText(Param,'-c') or SameText(Param,'--class') then begin   
    NextParam(P,Param,Index);
    TargetClass:=Param;
    continue;
   end;
   // Option -t, --title : Assign window title
   if SameText(Param,'-t') or SameText(Param,'--title') then begin
    NextParam(P,Param,Index);
    TargetTitle:=Param;
    continue;
   end;
   // Option -e, --exe : Assign exe filename
   if SameText(Param,'-e') or SameText(Param,'--exe') then begin
    NextParam(P,Param,Index);
    TargetExeId:=Param;
    continue;
   end;
   // Option -m, --magic : Assign magic number
   if SameText(Param,'-m') or SameText(Param,'--magic') then begin
    NextParam(P,Param,Index);
    TargetMagic:=iValDef(Param,-1);
    continue;
   end;
   Ident:=Param; NextParam(P,Param,Index);
   if Length(Ident)>0 then if Length(Param)>0 then ParseArgumentPair(Ident,Param);
  end;
  if Length(theLog)>0 then theLog:=ExpandFileName(ExpEnv(theLog));
  if Length(par.audio)>0 then par.audio:=FindFile(par.audio);
  if Length(par.ico)>0 then par.ico:=FindFile(par.ico);
  if Length(par.avi)>0 then par.avi:=FindFile(par.avi);
  ParseButtons;
  ColorCheck;
  Result:=ComposeMessage;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Check parameters is valid
function TFpQuiManager.CheckParameters:Integer;
begin
 Result:=EcfpQuiGenFail;
 if Assigned(Self) then
 try
  if verbose then begin
   Print('--class='+TargetClass);
   Print('--title='+TargetTitle);
   Print('--exe='+TargetExeId);
   Print('--magic='+IntToStr(TargetMagic));
  end;
  // Check window class
  if Length(TargetClass)=0 then begin
   Result:=Failure(EcFpQuiBadArgs,'Invalid window class. Help: '+ProgramId+' --help.');
   exit;
  end;
  // Check window title
  if Length(TargetTitle)=0 then begin
   Result:=Failure(EcFpQuiBadArgs,'Invalid window title. Help: '+ProgramId+' --help.');
   exit;
  end;
  // Check exe filename
  if Length(TargetExeId)=0 then begin
   Result:=Failure(EcFpQuiBadArgs,'Invalid exe filename. Help: '+ProgramId+' --help.');
   exit;
  end;
  // Check magic number
  if TargetMagic<0 then begin
   Result:=Failure(EcFpQuiBadArgs,'Invalid magic number. Help: '+ProgramId+' --help.');
   exit;
  end;
  // Check message data
  if Length(par.text)+Length(par.xml)+Length(par.delete)=0 then begin
   Result:=Failure(EcFpQuiBadArgs,'Invalid call syntax. Help: '+ProgramId+' --help.');
   exit;
  end;
  Result:=EcFpQuiSuccess;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Read FP-QUI parameter from environment string FPQUI_NAME
function TFpQuiManager.ReadFpQuiEnv(const Name:LongString):LongString;
begin
 Result:='';
 if Length(Name)>0 then
 if Assigned(Self) then
 try
  Result:=GetEnv('FPQUI_'+UpperCase(Name));
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Read FP-QUI parameter from registry string HKEY_LOCAL_MACHINE\SOFTWARE\FP-QUI\Name
function TFpQuiManager.ReadFpQuiReg(const Name:LongString):LongString;
begin
 Result:='';
 if Length(Name)>0 then
 if Assigned(Self) then
 try
  Result:=ReadFpQuiEnv(Name); if (Result<>'') then exit;
  Result:=ReadReqistryString(HKEY_LOCAL_MACHINE,'SOFTWARE\FP-QUI',Name);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function TFpQuiManager.GetSearchPath:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  if (mySearchPath='') then begin
   AppendDirToSearchPath(mySearchPath,ReadFpQuiReg('dir'),'.;gui');
   AppendDirToSearchPath(mySearchPath,GetEnv('ProgramFiles'),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetEnv('ProgramFiles(x86)'),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetEnv('CommonProgramFiles'),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetEnv('CommonProgramFiles(x86)'),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetSpecialShellFolderPath(CSIDL_APPDATA),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetSpecialShellFolderPath(CSIDL_LOCAL_APPDATA),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetSpecialShellFolderPath(CSIDL_COMMON_APPDATA),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetSpecialShellFolderPath(CSIDL_COMMON_DOCUMENTS),'FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,GetEnv('UnixRoot'),'add\bin;add\fp-qui;fp-qui\gui');
   AppendDirToSearchPath(mySearchPath,'.','.;FP-QUI;FP-QUI\gui');
   AppendDirToSearchPath(mySearchPath,'..','.;FP-QUI;FP-QUI\gui');
   mySearchPath:=AttachTailChar(AttachTailChar(mySearchPath,';')+GetEnv('PATH'),';');
  end;
  Result:=SysUtils.Trim(mySearchPath);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

procedure TFpQuiManager.ClearSearchPath;
begin
 if Assigned(Self) then mySearchPath:='';
end;

function TFpQuiManager.FindFile(const aFileName:LongString):LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Result:=Trim(aFileName);
  if Length(Result)>0 then
  if not FileExists(Result) then
  if IsRelativePath(Result) then begin
   Result:=FileSearch(Result,GetSearchPath);
  end;
  Result:=Trim(Result);
  if (Length(Result)>0) and FileExists(Result) then Result:=ExpandFileName(Result) else Result:='';
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Get FP-QUI.exe file name. Find in registry or in usual locations.
function TFpQuiManager.FindFpQuiExe:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Result:=FindFile(ReadFpQuiReg('exe'));
  if (Length(Result)=0) or not FileExists(Result) then Result:=FindFile('FP-QUI.exe');
  if (Length(Result)>0) and FileExists(Result) then Result:=ExpandFileName(Result) else Result:='';
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Get FP-QUICore.exe file name. Find in registry or in usual locations.
function TFpQuiManager.FindFpQuiCoreExe:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Result:=FindFile(ReadFpQuiReg('coreExe'));
  if (Length(Result)=0) or not FileExists(Result) then Result:=FindFile(fpQuiExeId);
  if (Length(Result)>0) and FileExists(Result) then Result:=ExpandFileName(Result) else Result:='';
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

type
 PSearchWindowRec = ^TSearchWindowRec;
 TSearchWindowRec = record wClass,wTitle,wExeId:PChar; hWin:HWND; end;

function SearchFpQuiWindow(Handle:HWND; Info:Pointer):BOOL; stdcall;
begin
 Result:=True;
 try
  if Handle<>0 then
  if Assigned(Info) then
  with PSearchWindowRec(Info)^ do
  if GetWindowClass(Handle)=wClass then
  if GetWindowTitle(Handle)=wTitle then
  if SameText(wExeId,GetExeNameByPid(GetWindowProcessId(Handle))) then begin
   Result:=false; hWin:=Handle;
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function FindWindowByClassTitleExe(aClass,aTitle,aExeId:LongString):HWND;
var R:TSearchWindowRec; Iter:Integer; const MaxIter=32;
begin
 Result:=0;
 try
  aClass:=Trim(aClass); if Length(aClass)=0 then exit;
  aTitle:=Trim(aTitle); if Length(aTitle)=0 then exit;
  aExeId:=Trim(aExeId); if Length(aExeId)=0 then exit;
  // Search via FindWindowEx
  if Result=0 then begin
   for Iter:=1 to MaxIter do begin
    Result:=FindWindowEx(0,Result,PChar(aClass),PChar(aTitle)); if Result=0 then break;
    if SameText(aExeId,GetExeNameByPid(GetWindowProcessId(Result))) then break;
   end;
  end;
  // Search via EnumWindows
  if Result=0 then begin
   R.wClass:=PChar(aClass);
   R.wTitle:=PChar(aTitle);
   R.wExeId:=PChar(aExeId);
   R.hWin:=0;
   EnumWindows(@SearchFpQuiWindow,PointerToPtrInt(@R));
   Result:=R.hWin;
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Find FP-QUI window by Class,Title,Exe file name
function TFpQuiManager.FindFpQuiWindow(const aClass,aTitle,aExeId:LongString):HWND;
begin
 Result:=0;
 if Assigned(Self) then
 try
  Result:=FindWindowByClassTitleExe(aClass,aTitle,aExeId);
  if Result=0 then Result:=FindWindowEx(0,0,PChar(aClass),PChar(aTitle)); // Fallback
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Find actual FP-QUI target window, process etc
function TFpQuiManager.FindActualTarget(AllowRun:Boolean):Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  if (Length(ActualWorkDir)=0) or not DirectoryExists(ActualWorkDir)
  then myActualWorkDir:=GetSharedWorkDir;
  if (Length(myActualExePath)=0) or not FileExists(myActualExePath)
  then myActualExePath:=TrimSpacesQuotes(FindFpQuiCoreExe);
  if (myActualWin<>0) and IsWindow(myActualWin) then begin
   if GetWindowTitle(myActualWin)<>TargetTitle then myActualWin:=0 else
   if GetWindowClass(myActualWin)<>TargetClass then myActualWin:=0;
  end;
  if (myActualWin=0) or not IsWindow(myActualWin)
  then myActualWin:=FindFpQuiWindow(TargetClass,TargetTitle,TargetExeId);
  if AllowRun then
  if (myActualWin=0) or not IsWindow(myActualWin) then begin
   if FindProcessPid(fpQuiExeId)=0 then RunFpQuiCoreExe(TimeOut);
   myActualWin:=FindFpQuiWindow(TargetClass,TargetTitle,TargetExeId);
  end;
  if (myActualWin<>0) and IsWindow(myActualWin) then begin
   myActualPid:=GetWindowProcessId(myActualWin);
   myActualExe:=GetExeNameByPid(myActualPid);
   Result:=true;
  end else begin
   myActualPid:=0;
   myActualExe:='';
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Send WM_COPYDATA message to FP-QUI; return status
function TFpQuiManager.SendMessage(const data:LongString; AllowRun:Boolean=true):Integer;
begin
 Result:=EcfpQuiGenFail;
 if Assigned(Self) then
 try
  if FindActualTarget(AllowRun) then begin
   if not SameText(ActualExe,TargetExeId) then begin
    Result:=Failure(EcFpQuiFailExe,'Unexpected EXE name.');
    exit;
   end;
   if Length(data)=0 then begin
    Result:=Failure(EcFpQuiBadData,'Nothing to send.');
    exit;
   end;
   if wmCopyDataSend(myActualWin,PChar(data),Length(data)+1,TargetMagic)>0 then begin
    Result:=Success(EcFpQuiSuccess,'Sent char['+IntToStr(Length(data)+1)+'] message to '+myActualExe+' PID '+IntToStr(myActualPid));
   end else begin
    Result:=Failure(EcFpQuiNotSent,'Lost char['+IntToStr(Length(data)+1)+'] message to '+myActualExe+' PID '+IntToStr(myActualPid));
   end;
  end else begin
   Result:=Failure(EcFpQuiNoFound,'FP-QUI window not found.');
   exit;
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Get arguments of Command Line
function TFpQuiManager.GetCmdLineArguments:LongString;
var s,CmdLine:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  CmdLine:=GetCommandLine; s:='';
  Result:=Trim(StrPas(GetNextParamStr(PChar(CmdLine),s)));
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Try to read StdIn
function TFpQuiManager.ReadFromStdIn(aMaxLeng:Cardinal=1024*64):LongString;
var stdin:THandle; buff:LongString; dwLen,Count:DWORD; s:LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  dwLen:=0;
  if IsConsole then begin
   buff:=StringBuffer(OS_PIPE_BUF);
   stdin:=GetStdHandle(STD_INPUT_HANDLE); Count:=0; s:='';
   while ReadFile(stdin,PChar(buff)^,Length(buff),dwLen,nil) and (Count<aMaxLeng) do begin
    if (dwLen>0) then SetString(s,PChar(buff),dwLen) else break;
    inc(Count,dwLen); Result:=Result+s;
   end;
   Result:=Trim(Result);
  end;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

 // Get data from Command Line argiments or from StdIn
function TFpQuiManager.ReadCmdLineOrStdIn(aMaxLeng:Cardinal=1024*64):LongString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Result:=GetCmdLineArguments;
  if (Length(Result)=0) and IsConsole then
  Result:=ReadFromStdIn(aMaxLeng);
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function TFpQuiManager.RunFpQuiTipExe(arg:LongString=''):Integer;
var data:LongString;
begin
 Result:=EcfpQuiGenFail;
 if Assigned(Self) then
 try
  arg:=Trim(arg);
  if Length(arg)=0 then
  arg:=ReadCmdLineOrStdIn;
  // Option --demo : Run demo
  if SameText(arg,'--demo') then begin
   Result:=RunDemo(DemoDelay);
   exit;
  end;
  // Option -h, --help : Show help
  if SameText(arg,'-h') or SameText(arg,'--help')
  or SameText(arg,'/?') or SameText(arg,'-?') then begin
   Result:=Usage(0);
   exit;
  end;
  data:=ParseCommandLine(arg);
  if verbose then Print('--data='+data);
  if CheckParameters=EcFpQuiSuccess then SendMessage(data);
  Result:=errno;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

function TFpQuiManager.RunDemo(aDelay:Cardinal=1000):Integer;
var dpi,dpn,dpp:Integer; presets,s:LongString; P:PChar;
const dpiguid='{909BDAF7-B24E-438F-9542-6C4FFC19AA31}';
begin
 Result:=0;
 if Assigned(Self) then
 try
  dpi:=0; dpn:=0; s:='';
  presets:='stdOk stdNo stdHelp stdStop stdDeny stdAbort stdError stdFails stdSiren '
          +'stdAlarm stdAlert stdBreak stdCancel stdNotify stdTooltip stdSuccess '
          +'stdWarning stdQuestion stdException stdAttention stdInformation stdExclamation';
  RunFpQuiTipExe(Format('verbose 1 guid %s text "FP-QUI - Demo progress %d%s" preset stdTooltip progress %d',[dpiguid,0,'%',0]));
  RunFpQuiTipExe(Format('verbose 1 text "Defaults" delay 15000',[]));
  Print('');
  P:=PChar(presets);
  while true do begin
   P:=GetNextParamStr(P,s);
   if Length(s)=0 then break;
   Inc(dpn);
  end;;
  P:=PChar(presets);
  while true do begin
   P:=GetNextParamStr(P,s);
   if length(s)=0 then break;
   Inc(dpi); dpp:=100*dpi div dpn;
   RunFpQuiTipExe(Format('verbose 1 guid %s sure 0 text "FP-QUI - Demo progress %d%s" preset stdTooltip progress %d',[dpiguid,dpp,'%',dpp]));
   RunFpQuiTipExe(Format('verbose 1 text "FP-QUI - Demo preset %s" preset %s delay 15000 btn1 "cmd" cmd1 "cmd /c start cmd"',[s,s]));
   if aDelay>0 then Sleep(aDelay);
   Print('');
  end;
  RunFpQuiTipExe(Format('verbose 1 guid %s text "FP-QUI - Demo progress %d%s" preset stdTooltip progress %d delay 15000',[dpiguid,100,'%',100]));
  Result:=errno;
 except
  on E:Exception do FpQuiExceptionHandler(E);
 end;
end;

{$ENDIF ~WINDOWS}

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_fpqui;
begin
end;

procedure Free_crw_fpqui;
begin
 {$IFDEF WINDOWS}
 Kill(myFpQuiManager);
 {$ENDIF ~WINDOWS}
end;

initialization

 Init_crw_fpqui;

finalization

 Free_crw_fpqui;

end.

//////////////
// END OF FILE
//////////////

