////////////////////////////////////////////////////////////////
// Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru      //
// Utility to get DIM information and poll rate statistics.   //
// Look https://dim.web.cern.ch/ to get help on DIM system.   //
////////////////////////////////////////////////////////////////

#include <dim.h>
#include <dic.hxx>
#include <dis.hxx>
#include <ctime>
#include <cstring>
#include <cstdlib>
#include <climits>
#include <iostream>
#include <algorithm>
#include <map>

using namespace std;

typedef unsigned int uint;
typedef unsigned long long int counter_t;           // type uses for counters
enum sortby_t { byNONE, byNAME, byPOLLRATE, byBYTERATE, byPOLLCOUNT, byBYTECOUNT };

//////////////////////////////////////////////////////
// Program common types, constants and variables
//////////////////////////////////////////////////////

const char *SPACES                  = "  ";         // spaces uses to separate table columns
const char *DIMSTAT                 = "dimStat";    // the program name
const uint  KILOBYTE                = 1024;         // base 2 info unit
const uint  MEGABYTE                = 1024*1024;    // base 2 info unit
const uint  NAMESIZE                = 256;          // name buffer size
const uint  LINESIZE                = KILOBYTE;     // line buffer size
const uint  STDOUTBUFFERSIZE        = KILOBYTE*256; // stdout buffer size
const uint  MAXIMALBUFFERSIZE       = MEGABYTE*4;   // maximal buffer size
const char *FORMATPC                = "%-20llu";    // format for poll count
const char *FORMATBC                = "%-20llu";    // format for byte count
const char *FORMATPR                = "%-14.3f";    // format for poll rate
const char *FORMATBR                = "%-14.3f";    // format for byte rate
const int   DEFTIMEOUT              = 5;            // default timeout, sec
const int   NOLINK                  = 0xDEADDEAD;   // nolink marker

int  opt_top                        = INT_MAX;      // option - number of top rows to show
int  opt_stat                       = 0;            // option - stat period, sec; 0 means no stat
int  opt_loop                       = 1;            // option - loop count; 0 = forever
int  opt_sort                       = byNONE;       // option - sort table items by name or rate
bool opt_reverse_name               = false;        // option - sort by name in reverse order
bool opt_reverse_rate               = false;        // option - sort by rate in reverse order
bool opt_nohead                     = false;        // option - don`t print table headers
bool opt_nostderr                   = false;        // option - don`t use stdout to print errors
int  opt_timeout                    = DEFTIMEOUT;   // option - timeout for DIM operations, seconds
bool opt_verbose                    = false;        // option - verbose output (for debug)
bool opt_running                    = false;        // option - check DNS is running
bool opt_servers                    = false;        // option - print DIM servers
char opt_services[NAMESIZE]         = "";           // option - wildcard of services
char opt_serverclients[NAMESIZE]    = "";           // option - server name to get clients
char opt_serverservices[NAMESIZE]   = "";           // option - server name to get services

char dns_node[NAMESIZE]             = "";           // current DIM_DNS_NODE

time_t start_time                   = 0;            // time of start

//////////////////////////////////////////////////////
// General purpose routines
//////////////////////////////////////////////////////

// compare a,b values of scalar types
template <typename TA, typename TB> inline bool EQL(const TA a, const TB b) { return ( a == b ); }
template <typename TA, typename TB> inline bool NEQ(const TA a, const TB b) { return ( a != b ); }
template <typename TA, typename TB> inline bool LSS(const TA a, const TB b) { return ( a <  b ); }
template <typename TA, typename TB> inline bool LEQ(const TA a, const TB b) { return ( a <= b ); }
template <typename TA, typename TB> inline bool GTR(const TA a, const TB b) { return ( a >  b ); }
template <typename TA, typename TB> inline bool GEQ(const TA a, const TB b) { return ( a >= b ); }

// check arg is empty string
bool IsEmptyStr(const char *arg) {
    if(arg) for(int i=0; arg[i]; i++) if(arg[i]>' ') return false;
    return true;
}

// check a,b is same string
bool SameStr(const char *a, const char *b) {
    return EQL(strcmp(a,b),0);
}

// check arg is option like -h
bool IsOption(const char *arg) {
    if(arg) return EQL(arg[0],'-');
    return false;
}

// check arg is option specified by opt
bool IsOption(const char *arg, const char *opt) {
    return (IsOption(arg)&&SameStr(arg,opt));
}

// check arg is option specified by opt1,opt2
bool IsOption(const char *arg, const char *opt1, const char *opt2) {
    return (IsOption(arg)&&(SameStr(arg,opt1)||SameStr(arg,opt2)));
}

// go to end of string
char *strend(char *src) {
    if(src) while(src[0]) src++;
    return src;
}

// time elapsed since program start, sec
double uptime() {
    return difftime(time(NULL),start_time);
}

// flush standard i/o streams
void flushstd() {
    fflush(stdout);
    fflush(stderr);
}

// return error stream (stderr or stdout)
FILE *outerr() {
    if(opt_nostderr) return stdout;
    return stderr;
}

// check if file is a pipe
bool ispipe(FILE *file) {
    return !isatty(fileno(file));
}

// print horizontal line of n chars c
void printhorline(int n, char c) { 
    if(n>0) if(c) printf("%s\n",string(n,c).c_str());
}

// scan command line to get --buffer-size kb for stdout
uint wanted_stdout_size(int argc, char **argv, uint defsize) {
    uint size=defsize;
    for(int i=1; i<argc; i++) {
        char *arg=argv[i];
        if(IsOption(arg,"--buffer-size")&&(i+1<argc)) {
            uint n=atol(argv[i+1])*KILOBYTE;
            if(n>=KILOBYTE) size=n;
        }
    }
    return max(KILOBYTE,min(MAXIMALBUFFERSIZE,size));
}

// buffer for stdout stream
static char  *stdout_buffer=NULL;
static uint stdout_buffer_size=0;

// set (big) buffer of stdout stream for better performance
uint set_stdout_buffer(uint size) {
    flushstd();
    if(size<KILOBYTE) return 0;
    stdout_buffer=new char[size];
    if(EQL(stdout_buffer,(char *)NULL)) return 0;
    if(setvbuf(stdout,stdout_buffer,_IOFBF,size)) return 0;
    stdout_buffer_size=size;
    return size;
}

// fallback (compatible) version of clear screen
void CLS(int n=100){
    if(n>0) printf(string(n,'\n').c_str());
}

#ifdef WIN32
    #include <windows.h>
    bool cls() { // see https://www.cplusplus.com/articles/4z18T05o/
        HANDLE hStdOut; CONSOLE_SCREEN_BUFFER_INFO csbi; DWORD count; DWORD cellCount; COORD homeCoords={0,0};
        hStdOut=GetStdHandle(STD_OUTPUT_HANDLE); if(EQL(hStdOut,INVALID_HANDLE_VALUE)) return false; // Console handle
        if(!GetConsoleScreenBufferInfo(hStdOut,&csbi)) return false;  // Get the number of cells in the current buffer
        cellCount=csbi.dwSize.X*csbi.dwSize.Y; // Calculate the number of cells in the current buffer 
        // Fill the entire buffer with spaces, with the current colors and attributes
        if(!FillConsoleOutputCharacter(hStdOut,(TCHAR)' ',cellCount,homeCoords,&count)) return false;
        if(!FillConsoleOutputAttribute(hStdOut,csbi.wAttributes,cellCount,homeCoords,&count)) return false;
        SetConsoleCursorPosition(hStdOut,homeCoords); // Move the cursor home
        return true;
    }
#else
    #define uses_ncurses
    #ifdef uses_ncurses
        #include <unistd.h>
        #include <term.h>
        // this version may need link -lncurses library
        bool cls() {
            if (!cur_term) {
                int result;
                setupterm(NULL,STDOUT_FILENO,&result);
                if (result<=0) return false;
            }
            putp(tigetstr("clear"));
            return true;
        }
    #else
        // It's not fast but looks compatible for POSIX systems
        bool cls(){
            return !system("clear");
        } 
    #endif
#endif

// clear console screen, see https://www.cplusplus.com/articles/4z18T05o/
void ClearScreen(int method=0, int n=100) {
    switch (method) {
        case 0  :   if(ispipe(stdout)||!cls()) CLS(n);  break;
        case 1  :   CLS(n);                             break;
        case 2  :   system("@cls||clear");              break;
        case 3  :   if(system("@cls")) system("clear"); break;
        default :   CLS(n);                             break;
    }
}

// cat current time (YYYY.MM.DD-hh:mm:ss) with prompt to the end of line
char *cattimestamp(char *line, const char *prompt) {
    char *p=line; time_t now=time(NULL);
    p=strend(p); strftime(p,20,"%Y.%m.%d-%H:%M:%S",localtime(&now));
    p=strend(p); if(strlen(prompt)) strcat(p,prompt);
    return line;
}

// compose command line as it's recognized
char *catcmdline(char *line) {
    int loop=opt_loop; if(EQL(loop,INT_MAX)) loop=0;
    char *p=line;
    p=strend(p); sprintf(p,"%s",DIMSTAT);
    p=strend(p); if(opt_verbose) sprintf(p," --verbose");
    p=strend(p); if(opt_nostderr) sprintf(p," --nostderr");
    p=strend(p); if(strlen(dns_node)) sprintf(p," --dns %s",dns_node);
    p=strend(p); if(NEQ(opt_timeout,DEFTIMEOUT)) sprintf(p," --timeout %d",opt_timeout);
    p=strend(p); if(NEQ(stdout_buffer_size,STDOUTBUFFERSIZE)) sprintf(p," --buffer-size %u",stdout_buffer_size/KILOBYTE);
    p=strend(p); if(opt_running) sprintf(p," --running");
    p=strend(p); if(opt_nohead) sprintf(p," --nohead");
    p=strend(p); if(EQL(opt_sort,byNAME)) sprintf(p," --sort-by-name");
    p=strend(p); if(EQL(opt_sort,byPOLLRATE)) sprintf(p," --sort-by-pollrate");
    p=strend(p); if(EQL(opt_sort,byBYTERATE)) sprintf(p," --sort-by-byterate");
    p=strend(p); if(EQL(opt_sort,byPOLLCOUNT)) sprintf(p," --sort-by-pollcount");
    p=strend(p); if(EQL(opt_sort,byBYTECOUNT)) sprintf(p," --sort-by-bytecount");
    p=strend(p); if(opt_reverse_rate) sprintf(p," --reverse-rate");
    p=strend(p); if(opt_reverse_name) sprintf(p," --reverse-name");
    p=strend(p); if(opt_stat) sprintf(p," --stat %d",opt_stat);
    p=strend(p); if(NEQ(loop,1)) sprintf(p," --loop %d",loop);
    p=strend(p); if(NEQ(opt_top,INT_MAX)) sprintf(p," --top %d",opt_top);
    p=strend(p); if(opt_servers) sprintf(p," --servers");
    p=strend(p); if(strlen(opt_services)) sprintf(p," --services %s",opt_services);
    p=strend(p); if(strlen(opt_serverservices)) sprintf(p," --serverservices %s",opt_serverservices);
    p=strend(p); if(strlen(opt_serverclients)) sprintf(p," --serverclients %s",opt_serverclients);
    return line;
}

// print command line as it's recognized
void print_cmdline(const char *prefix) {
    char line[LINESIZE]=""; char *p=line;
    p=strend(p); sprintf(p,"%s",prefix);
    p=strend(p); catcmdline(p);
    printf("%s\n",line);
}

//////////////////////////////////////////////////////
// Routines to sort maps by value (rate) and by name 
//////////////////////////////////////////////////////

struct index_name { int index; const char *name; double rate; }; 

index_name *new_index_name(int n) {
    index_name *data=NULL; if(n<=0) return data;
    data=new index_name[n]; if(!data) return data;
    for(int i=0; i<n; i++) { data[i].index=0; data[i].name=0; data[i].rate=0.0; }
    return data;
}

int valcmp(double a, double b) {
    if(a>b) return +1;
    if(a<b) return -1;
    return 0;
}

int compare_index_name(const void *a, const void *b) {
    index_name *ina=(index_name *)a;
    index_name *inb=(index_name *)b;
    int c1=valcmp(ina->rate,inb->rate); // primary   key to sort by rate
    int c2=strcmp(ina->name,inb->name); // secondary key to sort by name
    c1=-c1;                             // by default use rate order from large to smaller
    if(opt_reverse_rate) c1=-c1;
    if(opt_reverse_name) c2=-c2;
    if(c1) return c1;
    return c2;
}

void sort_index_name(index_name *data, uint n) {
    if(data) if(n>0) qsort(data,n,sizeof(data[0]),compare_index_name);
}

//////////////////////////////////////////////////////
// DIM Statictics Counters and Timer classes for stat
//////////////////////////////////////////////////////

class DimStatCounters {                             // counters to accumulate statistics
    counter_t cb,cp;    // current  bytes/polls
    counter_t pb,pp;    // previous bytes/polls
    counter_t db,dp;    // delta of bytes/polls
    int dt;             // delta of time i.e. period
public:
    counter_t       byte_count()        { return cb; }
    counter_t       poll_count()        { return cp; }
    double          byte_rate()         { if(!dt) return 0.0; return (double)db/(double)dt; }
    double          poll_rate()         { if(!dt) return 0.0; return (double)dp/(double)dt; }
    void            poll(uint n)        { cb+=n; cp++; }
    virtual void    add(dim_long ref)   { ref=ref; } // to avoid warnings
    virtual void    reset()             { cb=0; cp=0; pb=0; pp=0; db=0; dp=0; dt=0; }
    virtual void    stop(int period)    { db=cb-pb; pb=cb; dp=cp-pp; pp=cp; dt=period; }
    DimStatCounters()                   { reset(); }
    virtual ~DimStatCounters()          { reset(); }
};

DimStatCounters *stat_list=NULL;                    // total statistics counters container
DimTimer        *stat_timer=NULL;                   // total statistics timer

class StatService : public DimInfo {                // DIM INFO service to count polls/bytes
    DimStatCounters *mystat;
public :
    DimStatCounters *stat() { return mystat; }
    virtual void infoHandler() {
        if(!stat_timer) return;
        if(!stat_timer->runningFlag) return;
        int size = getSize();
        if(mystat) mystat->poll(size);
        if(stat_list) stat_list->poll(size);
    }
    StatService(const char *name) : DimInfo(name,(char *)"--") {
        mystat = new DimStatCounters();
        if(stat_list) stat_list->add((dim_long)this);
    }
    ~StatService() { delete mystat; }
};

class DimStatCountersList : public DimStatCounters {
    map <int,dim_long> myitems;
    index_name *mybyname;
    index_name *mybyrate;
    int nm,nn,nf,nbr,nbc,npr,npc;
public:
    int size() { return myitems.size(); }
    StatService *items(int i) { return (StatService *)myitems[i]; }
    int bynone(int i) { return i; }
    int byname(int i) { if(mybyname) return mybyname[i].index; return i; }
    int byrate(int i) { if(mybyrate) return mybyrate[i].index; return i; }
    void markup_byname(bool head) {
        int nsize=size();
        char tmp[NAMESIZE];
        sprintf(tmp,FORMATBC,100); nbc=strlen(tmp);
        sprintf(tmp,FORMATBR,1e2); nbr=strlen(tmp);
        sprintf(tmp,FORMATPC,100); npc=strlen(tmp);
        sprintf(tmp,FORMATPR,1e2); npr=strlen(tmp);
        sprintf(tmp,"%d",nsize+1); nm=strlen(tmp);
        if(head) nm=max(nm,(int)strlen("Number"));
        nn=1; if(head) nn=max(strlen("Name"),strlen("TOTAL"));
        nf=1; if(head) nf=strlen("Format");
        for(int i=0; i<nsize; i++) nn=max(nn,(int)strlen(items(i)->getName()));
        for(int i=0; i<nsize; i++) nf=max(nf,(int)strlen(items(i)->getFormat()));
        mybyname=new_index_name(nsize);
        for(int i=0; i<nsize; i++) mybyname[i].index=i;
        for(int i=0; i<nsize; i++) mybyname[i].name=items(i)->getName();
        sort_index_name(mybyname,nsize);
    }
    void sort_byrate(int sortby) {
        int nsize=size();
        if(!nsize) return;
        if(!mybyrate) mybyrate=new_index_name(nsize); 
        for(int i=0; i<nsize; i++) mybyrate[i].index=i;
        for(int i=0; i<nsize; i++) mybyrate[i].name=items(i)->getName();
        switch (sortby) {
            case byPOLLRATE  : for(int i=0; i<nsize; i++) mybyrate[i].rate=items(i)->stat()->poll_rate();  break;
            case byBYTERATE  : for(int i=0; i<nsize; i++) mybyrate[i].rate=items(i)->stat()->byte_rate();  break;
            case byPOLLCOUNT : for(int i=0; i<nsize; i++) mybyrate[i].rate=items(i)->stat()->poll_count(); break;
            case byBYTECOUNT : for(int i=0; i<nsize; i++) mybyrate[i].rate=items(i)->stat()->byte_count(); break;
            default          : break;
        }
        sort_index_name(mybyrate,nsize);
    }
    void header(int loop) {
        char line[LINESIZE]=""; char *p=line;
        p=strend(p); cattimestamp(p," => ");
        p=strend(p); sprintf(p,"%s loop %d. Time elapsed since start: %.0f sec.",DIMSTAT,loop,uptime());
        p=strend(p); strcat(p,"\n");
        p=strend(p); catcmdline(p);
        printf("%s\n",line);
    }
    void report(int loop, int head, int total, int sortby, int maxrows) {
        if(!mybyname) markup_byname(head);
        sort_byrate(sortby);
        int ntail=0;
        if(head) if(size()) {
            header(loop);
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*s","Service ",nm,"Number");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,"Name");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,"Format");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,npr,"Rate,Poll/s");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nbr,"Rate,Byte/s");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,npc,"PollCounter");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nbc,"ByteCounter");
            p=strend(p); int n=(int)(p-line);
            printhorline(n,'=');
            printf("%s\n",line);
            printhorline(n,'-');
            ntail=n;
        }
        if(total) {
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*s","Service ",nm,"-");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,"TOTAL");
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,"-");
            p=strend(p); sprintf(p,"%s%-*.3f",SPACES,npr,poll_rate());
            p=strend(p); sprintf(p,"%s%-*.3f",SPACES,nbr,byte_rate());
            p=strend(p); sprintf(p,"%s%-*llu",SPACES,npc,poll_count());
            p=strend(p); sprintf(p,"%s%-*llu",SPACES,nbc,byte_count());
            printf("%s\n",line);
        }
        int ntop=min(size(),maxrows);
        for(int i=0; i<ntop; i++) {
            int j=i;
            switch(sortby) {
                case byNAME         : j=byname(i); break;
                case byPOLLRATE     : j=byrate(i); break;
                case byBYTERATE     : j=byrate(i); break;
                case byPOLLCOUNT    : j=byrate(i); break;
                case byBYTECOUNT    : j=byrate(i); break;
                default             : j=bynone(i); break;
            }
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*d","Service ",nm,i+1);
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,items(j)->getName());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,items(j)->getFormat());
            p=strend(p); sprintf(p,"%s%-*.3f",SPACES,npr,items(j)->stat()->poll_rate());
            p=strend(p); sprintf(p,"%s%-*.3f",SPACES,nbr,items(j)->stat()->byte_rate());
            p=strend(p); sprintf(p,"%s%-*llu",SPACES,npc,items(j)->stat()->poll_count());
            p=strend(p); sprintf(p,"%s%-*llu",SPACES,nbc,items(j)->stat()->byte_count());
            printf("%s\n",line);
        }
        if(ntail) printhorline(ntail,'-');
        flushstd();
    }
    virtual void reset() {
        DimStatCounters::reset();
        for(int i=0; i<size(); i++) items(i)->stat()->reset();
    }
    virtual void stop(int period){
        DimStatCounters::stop(period);
        for(int i=0; i<size(); i++) items(i)->stat()->stop(period);
    }
    virtual void add(dim_long ref) {
        int i=myitems.size();
        myitems[i]=ref;
    }
    DimStatCountersList() : DimStatCounters() {
        reset(); mybyname=NULL; mybyrate=NULL;
        nm=0; nn=0; nf=0; nbc=0; nbr=0; npc=0; npr=0;
    }
    ~DimStatCountersList() {
        delete mybyname;
        delete mybyrate;
    }
};

class DimStatTimer : public DimTimer {
    int the_time;
public:
    DimStatTimer() : DimTimer() { the_time=0; }
    void start(int time) { DimTimer::start(time); the_time=time; }
    virtual void timerHandler() { if(stat_list) stat_list->stop(the_time); }
};

//////////////////////////////////////////////////////
// Routines to handle command line options
//////////////////////////////////////////////////////

void setOptVerbose(bool opt) {
    opt_verbose=opt;
    if(opt_verbose) printf("Set option --verbose\n");
}

void setOptBufferSize(const char *arg) {
    // stdout_buffer_size should be set in preliminary scan
    if(opt_verbose) if(arg) printf("Set option --buffer-size %u\n",stdout_buffer_size/KILOBYTE);
}

void setOptRunning(bool opt) {
    opt_running=opt;
    if(opt_verbose) if(opt_running) printf("Set option --running\n");
}

void setOptServers(bool opt) {
    opt_servers=opt;
    if(opt_verbose) if(opt_servers) printf("Set option --servers\n");
}

void setOptNostderr(bool opt) {
    opt_nostderr=opt;
    if(opt_verbose) if(opt_nostderr) printf("Set option --nostderr\n");
}

void setOptNohead(bool opt) {
    opt_nohead=opt;
    if(opt_verbose) if(opt_nohead) printf("Set option --nohead\n");
}

void setOptReverseName(bool opt) {
    opt_reverse_name=opt;
    if(opt_verbose) if(opt_reverse_name) printf("Set option --reverse-name\n");
}

void setOptReverseRate(bool opt) {
    opt_reverse_rate=opt;
    if(opt_verbose) if(opt_reverse_rate) printf("Set option --reverse-rate\n");
}

void setOptSort(int sortby) {
    opt_sort=sortby;
    switch (opt_sort) {
        case byNAME         : if(opt_verbose) printf("Set option --sort-by-name\n");       break;
        case byPOLLRATE     : if(opt_verbose) printf("Set option --sort-by-pollrate\n");   break;
        case byBYTERATE     : if(opt_verbose) printf("Set option --sort-by-byterate\n");   break;
        case byPOLLCOUNT    : if(opt_verbose) printf("Set option --sort-by-pollcount\n");  break;
        case byBYTECOUNT    : if(opt_verbose) printf("Set option --sort-by-bytecount\n");  break;
        default             : break;
    }
}

void setOptTimeout(const char *arg) {
    opt_timeout=atol(arg);
    if(opt_verbose) printf("Set option --timeout %d\n",opt_timeout);
}

void setOptDns(const char *arg){
    dim_set_dns_node(arg);
    if(opt_verbose) printf("Set option --dns %s\n",arg);
}

void setOptServices(const char *arg){
    strcpy(opt_services,arg);
    if(opt_verbose) printf("Set option --services %s\n",arg);
}

void setOptServerServices(const char *arg){
    strcpy(opt_serverservices,arg);
    if(opt_verbose) printf("Set option --serverservices %s\n",arg);
}

void setOptServerClients(const char *arg){
    strcpy(opt_serverclients,arg);
    if(opt_verbose) printf("Set option --serverclients %s\n",arg);
}

void setOptStat(const char *arg){
    opt_stat=atol(arg);
    if(opt_stat<0) opt_stat=INT_MAX;
    if(opt_verbose) printf("Set option --stat %d\n",opt_stat);
}

void setOptLoop(const char *arg){
    opt_loop=atol(arg);
    if(opt_loop<=0) opt_loop=INT_MAX;
    if(opt_verbose) printf("Set option --loop %d\n",opt_loop);
}

void setOptTop(const char *arg){
    opt_top=atol(arg);
    if(opt_top<=0) opt_top=INT_MAX;
    if(opt_verbose) printf("Set option --top %d\n",opt_top);
}

int dnsVersionNumber(int timeout=DEFTIMEOUT, int nolink=NOLINK) {
   DimCurrentInfo dns("DIS_DNS/VERSION_NUMBER",timeout,nolink);
   return dns.getInt();
}

bool dnsRunning(int version, int nolink) {
    if(version) return NEQ(version,nolink);
    return false;
}

void PrintVersion(const char *name) {
    printf("%s version 1.0\n",name);
    flushstd();
}

void PrintHelp(const char *name) {
    PrintVersion(name);
    printf("Copyright (c) 2021 Alexey Kuryakin kouriakine@mail.ru\n");
    printf("Utility to get DIM information and poll rate statistics.\n");
    printf("Look https://dim.web.cern.ch/ to get help on DIM system.\n");
    printf("Usage: \n");
    printf("    %s [-options] [arguments]\n",name);
    printf("Options:\n");
    printf("    --                  - end of options.\n");
    printf("    --version           - print the version.\n");
    printf("    -h,--help           - print this help screen.\n");
    printf("    -v,--verbose        - switch on verbose mode.\n");
    printf("    -m,--mutely         - switch off verbose mode.\n");
    printf("    -d,--dns node       - set DIM_DNS_NODE (node).\n");
    printf("    -t,--timeout t      - set timeout t; default=%d sec.\n",DEFTIMEOUT);
    printf("    --buffer-size n     - set stdout buffer size n kb; default=%d kb.\n",STDOUTBUFFERSIZE/KILOBYTE);
    printf("    -r,--running        - check DIM DNS is running now.\n");
    printf("    -n,--nohead         - don`t use head to print tables.\n");
    printf("    --nostderr          - don`t use stderr to print errors.\n");
    printf("    -s,--sort           - sort by name when print tables.\n");
    printf("    --sort-by-name      - sort by name when print tables.\n");
    printf("    --reverse-name      - sort by name in reverse order.\n");
    printf("    --reverse-rate      - sort by rate in reverse order.\n");
    printf("    --sort-by-pollrate  - sort statistics by poll rate.\n");
    printf("    --sort-by-byterate  - sort statistics by byte rate.\n");
    printf("    --sort-by-pollcount - sort statistics by poll count.\n");
    printf("    --sort-by-bytecount - sort statistics by byte count.\n");
    printf("    --top n             - show only n top lines of statistics.\n");
    printf("    --stat n            - print statistics with period n sec.\n");
    printf("    --loop n            - repeat statistics n times; 0=forever.\n");
    printf("    --servers           - print list of DIM servers in this DNS.\n");
    printf("    --services s        - print list of DIM services by wildcard (s).\n");
    printf("    --serverclients s   - print list of DIM clients  by server name (s).\n");
    printf("    --serverservices s  - print list of DIM services by server name (s).\n");
    printf("Notes:\n");
    printf("    1) short (-h) and long (--help) options are both allowed.\n");
    printf("    2) some options has mandatory parameters which must be specified.\n");
    printf("    3) DIM DNS node can be specified by DIM_DNS_NODE or --dns option.\n");
    printf("    4) Statistics accumulates during period set by option (--stat n).\n");
    printf("    5) Statistics can repeat n times by option (--loop n); 0=forever.\n");
    printf("    6) Use --services, --serverservices to set target for statistics.\n");
    printf("    7) Normal name sort order is alphabet, see --reverse-name option.\n");
    printf("    8) Normal rate sort order is larger->smaller, see --reverse-rate.\n");
    printf("Example:\n");
    printf("    %s --help                                : print help screen\n",name);
    printf("    %s --version                             : print program version\n",name);
    printf("    %s --running --nostderr                  : check DNS is running; use DNS from environs\n",name);
    printf("    %s --dns localhost --running             : check DNS is running; use DNS how specified\n",name);
    printf("    %s -v -r -d localhost -t 10              : check DNS is running; verbose; timeout; DNS\n",name);
    printf("    %s --verbose --dns localhost --running   : check DNS is running; long options used\n",name);
    printf("    %s --servers --timeout 10 --nohead       : print list of DIM servers; use timeout 10 s\n",name);
    printf("    %s --services DEMO/* --timeout 10 --sort : print list of DIM services by given wildcard\n",name);
    printf("    %s --serverclients DEMO/SERVER           : print list of DIM clients  by given DIM server name\n",name);
    printf("    %s --serverservices DEMO/SERVER          : print list of DIM services by given DIM server name\n",name);
    printf("    %s --dns localhost --sort-by-byterate --stat 5 --loop 5 --top 20 --services BENCH*\n",name);
    printf("    %s --dns localhost --sort-by-pollrate --stat 5 --loop 0 --top 20 --services *\n",name);
    flushstd();
}

//////////////////////////////////////////////////////
// Main program
//////////////////////////////////////////////////////

int main(int argc, char *argv[]) {

    if(argc<2) {
        PrintHelp(DIMSTAT);
        return 0;
    }

    dim_init();

    start_time=time(NULL);

    //
    // Some options required preliminary command line scan.
    //
    if(argc>2) opt_verbose=IsOption(argv[1],"-v","--verbose");
    if(opt_verbose) printf("Command Line preliminary scan:\n");
    if(argc>2) if(IsOption(argv[1],"-v","--verbose")) setOptVerbose(true);
    uint sbsize=set_stdout_buffer(wanted_stdout_size(argc,argv,STDOUTBUFFERSIZE));
    if(opt_verbose) printf("Set option --buffer-size %u\n",sbsize/KILOBYTE);

    //
    // Command line parsing.
    //
    if(opt_verbose) printf("Command Line general parsing:\n");
    int optnum=0;
    int parnum=0;
    char *opt=NULL;
    bool isopt=true;
    for(int i=1; i<argc; i++) {
        char *arg=argv[i];
        if(IsOption(arg) && isopt && IsEmptyStr(opt)){
            optnum++; // arg is option
            if(opt_verbose) printf("opt[%d]: %s\n",optnum,arg);
            if(IsOption(arg,"--"))                  {   isopt=false;                continue;   } // assume next arguments are parameters
            if(IsOption(arg,"--version"))           {   PrintVersion(DIMSTAT);      return 0;   }
            if(IsOption(arg,"-h","--help"))         {   PrintHelp(DIMSTAT);         return 0;   }
            if(IsOption(arg,"-v","--verbose"))      {   setOptVerbose(true);        continue;   }
            if(IsOption(arg,"-m","--mutely"))       {   setOptVerbose(false);       continue;   }
            if(IsOption(arg,"-r","--running"))      {   setOptRunning(true);        continue;   }
            if(IsOption(arg,"-n","--nohead"))       {   setOptNohead(true);         continue;   }
            if(IsOption(arg,"--nostderr"))          {   setOptNostderr(true);       continue;   }
            if(IsOption(arg,"-s","--sort"))         {   setOptSort(byNAME);         continue;   }
            if(IsOption(arg,"--sort-by-name"))      {   setOptSort(byNAME);         continue;   }
            if(IsOption(arg,"--sort-by-pollrate"))  {   setOptSort(byPOLLRATE);     continue;   }
            if(IsOption(arg,"--sort-by-byterate"))  {   setOptSort(byBYTERATE);     continue;   }
            if(IsOption(arg,"--sort-by-pollcount")) {   setOptSort(byPOLLCOUNT);    continue;   }
            if(IsOption(arg,"--sort-by-bytecount")) {   setOptSort(byBYTECOUNT);    continue;   }
            if(IsOption(arg,"--reverse-name"))      {   setOptReverseName(true);    continue;   }
            if(IsOption(arg,"--reverse-rate"))      {   setOptReverseRate(true);    continue;   }
            if(IsOption(arg,"--servers"))           {   setOptServers(true);        continue;   }
            if(IsOption(arg,"--buffer-size"))       {   opt=arg;                    continue;   }
            if(IsOption(arg,"-t","--timeout"))      {   opt=arg;                    continue;   }
            if(IsOption(arg,"-d","--dns"))          {   opt=arg;                    continue;   }
            if(IsOption(arg,"--top"))               {   opt=arg;                    continue;   }
            if(IsOption(arg,"--stat"))              {   opt=arg;                    continue;   }
            if(IsOption(arg,"--loop"))              {   opt=arg;                    continue;   }
            if(IsOption(arg,"--services"))          {   opt=arg;                    continue;   }
            if(IsOption(arg,"--serverclients"))     {   opt=arg;                    continue;   }
            if(IsOption(arg,"--serverservices"))    {   opt=arg;                    continue;   }
            // other options is unrecognized i.e. marked as unknown, it's an error
            fprintf(outerr(),"Error: unknown option %s. See %s --help to get help.\n",arg,DIMSTAT);
            flushstd();
            return 1;
        } else {
            if(IsEmptyStr(opt)){
                parnum++; // arg is parameter
                if(opt_verbose) printf("par[%d]: %s\n",parnum,arg);
                // this program don`t use positional parameters but only options
                continue;
            } else {
                if(opt_verbose) printf("val[%d]: %s\n",optnum,arg);
            }
            if(IsOption(opt,"-d","--dns"))          {   setOptDns(arg);                     }
            if(IsOption(opt,"--buffer-size"))       {   setOptBufferSize(arg);              }
            if(IsOption(opt,"-t","--timeout"))      {   setOptTimeout(arg);                 }
            if(IsOption(opt,"--top"))               {   setOptTop(arg);                     }
            if(IsOption(opt,"--stat"))              {   setOptStat(arg);                    }
            if(IsOption(opt,"--loop"))              {   setOptLoop(arg);                    }
            if(IsOption(opt,"--services"))          {   setOptServices(arg);                }
            if(IsOption(opt,"--serverclients"))     {   setOptServerClients(arg);           }
            if(IsOption(opt,"--serverservices"))    {   setOptServerServices(arg);          }
            opt=NULL;
        }
    }

    if(opt_verbose) print_cmdline("Recognized Command Line is: ");

    //
    // Check DNS node is specified.
    //
    if(!dim_get_dns_node(dns_node)) strcpy(dns_node,"");
    if(opt_verbose) printf("DIM_DNS_NODE=%s\n",dns_node);
    if(IsEmptyStr(dns_node)) {
        fprintf(outerr(),"Error: DIM_DNS_NODE is not defined.\n");
        flushstd();
        return 1;
    }

    if(opt_timeout<1){
        fprintf(outerr(),"Error: Wrong --timeout option value.\n");
        flushstd();
        return 1;
    }

    //
    // Check DNS is running.
    //
    if(opt_running){
        int version = dnsVersionNumber(opt_timeout,NOLINK);
        if(dnsRunning(version,NOLINK)) {
            printf("RUNNING - DNS %s Version %d is ONLINE.\n",dns_node,version);
            flushstd();
            return 0;
        } else {
            printf("STOPPED - DNS %s is DEAD.\n",dns_node);
            flushstd();
            return 1;
        }
    }

    //
    // Preparing statistics counters & timer.
    //
    map <int,string> statservlist; // services to stat of
    DimStatCountersList *stat = new DimStatCountersList();
    DimStatTimer *timer = new DimStatTimer();
    stat_timer=timer;
    stat_list=stat;
    flushstd();

    //
    // Print list of DIM servers available with DNS.
    //
    if(opt_servers){
        char *task; char *node; int pid; int nt=0; int nn=0; int np=0;
        map <int,string> tasks; map <int,string> nodes; map <int,string> pids;
        DimBrowser br;
        br.getServers(opt_timeout);
        for(int i=0; br.getNextServer(task,node,pid); i++){
            if(opt_verbose) printf("Server[%d]: %s %s %d\n",i+1,task,node,pid);
            char spid[NAMESIZE]; sprintf(spid,"%d",pid);
            tasks[i]=task; nt=max(nt,(int)strlen(task));
            nodes[i]=node; nn=max(nn,(int)strlen(node));
            pids[i]=spid;  np=max(np,(int)strlen(spid));
        }
        int ntail=0;
        int nsize=tasks.size();
        char sm[NAMESIZE]; sprintf(sm,"%d",nsize);
        int nm=max(strlen("Number"),strlen(sm)); if(opt_nohead) nm=strlen(sm);
        if(!opt_nohead) printf("%d DIM Server(s) found with DNS %s\n",nsize,dns_node);
        index_name *indxs=new_index_name(nsize);
        for(int i=0; i<nsize; i++){ indxs[i].index=i; indxs[i].name=tasks[i].c_str(); }
        if(indxs) if(opt_sort) sort_index_name(indxs,nsize);
        for(int i=0; i<nsize; i++){
            if(!i && !opt_nohead){
                char line[LINESIZE]=""; char *p=line;
                p=strend(p); sprintf(p,"%s%-*s","Server ",nm,"Number");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nt,"Task");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,"Node");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,np,"PID");
                p=strend(p); int n=(int)(p-line);
                printhorline(n,'=');
                printf("%s\n",line);          
                printhorline(n,'-');
                ntail=n;
            }
            int j=i; if(indxs) j=indxs[i].index;
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*d","Server ",nm,i+1);
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nt,tasks[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,nodes[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,np,pids[j].c_str());
            printf("%s\n",line);
        }
        if(ntail) printhorline(ntail,'-');
        flushstd();
        if(indxs) delete indxs;
        if(!nsize) return 1;
        return 0;
    }

    //
    // Print list of services specified by wildcard (opt_services).
    //
    if(!IsEmptyStr(opt_services)){
        char *name; char *form; int nn=strlen("Name"); int nf=strlen("Format"); int nt=strlen("Type");
        map <int,string> names; map <int,string> forms; map <int,string> types;
        DimBrowser br; int ntype=0;
        br.getServices(opt_services,opt_timeout);
        for(int i=0; (ntype=br.getNextService(name,form)); i++){
            char type[NAMESIZE]="";
            switch (ntype) {
                case DimRPC     :   strcpy(type,"RPC");     break;
                case DimCOMMAND :   strcpy(type,"CMD");     break;
                case DimSERVICE :   strcpy(type,"INFO");    break;
                default         :   strcpy(type,"????");    break;
            }
            if(opt_stat>0) if(EQL(ntype,DimSERVICE)) statservlist[statservlist.size()]=name;
            if(opt_verbose) printf("Service[%d]: %s %s %s\n",i+1,name,form,type);
            names[i]=name; nn=max(nn,(int)strlen(name));
            forms[i]=form; nf=max(nf,(int)strlen(form));
            types[i]=type; nt=max(nt,(int)strlen(type));
        }
        int ntail=0;
        int nsize=names.size();
        char sm[NAMESIZE]; sprintf(sm,"%d",nsize);
        int nm=max(strlen("Number"),strlen(sm)); if(opt_nohead) nm=strlen(sm);
        if(!opt_nohead) printf("%d DIM Service(s) found by Name(s) %s with DNS %s\n",nsize,opt_services,dns_node);
        index_name *indxs=new_index_name(nsize);
        for(int i=0; i<nsize; i++){ indxs[i].index=i; indxs[i].name=names[i].c_str(); }
        if(indxs) if(opt_sort) sort_index_name(indxs,nsize);
        for(int i=0; i<nsize; i++){
            if(!i && !opt_nohead){
                char line[LINESIZE]=""; char *p=line;
                p=strend(p); sprintf(p,"%s%-*s","Service ",nm,"Number"); 
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,"Name");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,"Format");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nt,"Type");
                p=strend(p); int n=(int)(p-line);
                printhorline(n,'=');
                printf("%s\n",line);
                printhorline(n,'-');
                ntail=n;
            }
            int j=i; if(indxs) j=indxs[i].index;
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*d","Service ",nm,i+1);  
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,names[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,forms[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nt,types[j].c_str());
            printf("%s\n",line); 
        }
        if(ntail) printhorline(ntail,'-');
        flushstd();
        if(indxs) delete indxs;
        if(!opt_stat) if(!nsize) return 1;
        if(!opt_stat) return 0;
    }

    //
    // Print list of services specified by server name (opt_serverservices).
    //
    if(!IsEmptyStr(opt_serverservices)){
        char *name; char *form; int nn=strlen("Name"); int nf=strlen("Format"); int nt=strlen("Type");
        map <int,string> names; map <int,string> forms; map <int,string> types;
        DimBrowser br; int ntype=0;
        br.getServerServices(opt_serverservices,opt_timeout);
        for(int i=0; (ntype=br.getNextServerService(name,form)); i++){
            char type[NAMESIZE]="";
            switch (ntype) {
                case DimRPC     :   strcpy(type,"RPC");     break;
                case DimCOMMAND :   strcpy(type,"CMD");     break;
                case DimSERVICE :   strcpy(type,"INFO");    break;
                default         :   strcpy(type,"????");    break;
            }
            if(opt_stat>0) if(EQL(ntype,DimSERVICE)) statservlist[statservlist.size()]=name;
            if(opt_verbose) printf("Service[%d]: %s %s %s\n",i+1,name,form,type);
            names[i]=name; nn=max(nn,(int)strlen(name));
            forms[i]=form; nf=max(nf,(int)strlen(form));
            types[i]=type; nt=max(nt,(int)strlen(type));
        }
        int ntail=0;
        int nsize=names.size();
        char sm[NAMESIZE]; sprintf(sm,"%d",nsize);
        int nm=max(strlen("Number"),strlen(sm)); if(opt_nohead) nm=strlen(sm);
        if(!opt_nohead) printf("%d DIM Service(s) found on Server %s with DNS %s\n",nsize,opt_serverservices,dns_node);
        index_name *indxs=new_index_name(nsize);
        for(int i=0; i<nsize; i++){ indxs[i].index=i; indxs[i].name=names[i].c_str(); }
        if(indxs) if(opt_sort) sort_index_name(indxs,nsize);
        for(int i=0; i<nsize; i++){
            if(!i && !opt_nohead){
                char line[LINESIZE]=""; char *p=line;
                p=strend(p); sprintf(p,"%s%-*s","Service ",nm,"Number");   
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,"Name");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,"Format");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nt,"Type");
                p=strend(p); int n=(int)(p-line);
                printhorline(n,'=');
                printf("%s\n",line);
                printhorline(n,'-');
                ntail=n;
            }
            int j=i; if(indxs) j=indxs[i].index;
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*d","Service ",nm,i+1);
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,names[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nf,forms[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nt,types[j].c_str());
            printf("%s\n",line);
        }
        if(ntail) printhorline(ntail,'-');
        flushstd();
        if(indxs) delete indxs;
        if(!opt_stat) if(!nsize) return 1;
        if(!opt_stat) return 0;
    }

    //
    // Print list of clients specified by server name (opt_serverclients).
    //
    if(!IsEmptyStr(opt_serverclients)){
        char *name; char *node; int nn=0; int nd=0;
        map <int,string> names; map <int,string> nodes;
        DimBrowser br;
        br.getServerClients(opt_serverclients,opt_timeout);
        for(int i=0; br.getNextServerClient(name,node); i++){
            if(opt_verbose) printf("Client[%d]: %s %s\n",i+1,name,node);
            names[i]=name; nn=max(nn,(int)strlen(name));
            nodes[i]=node; nd=max(nd,(int)strlen(node));
        }
        int ntail=0;
        int nsize=names.size();
        char sm[NAMESIZE]; sprintf(sm,"%d",nsize);
        int nm=max(strlen("Number"),strlen(sm)); if(opt_nohead) nm=strlen(sm);
        if(!opt_nohead) printf("%d DIM Client(s) found on Server %s with DNS %s\n",nsize,opt_serverclients,dns_node);
        index_name *indxs=new_index_name(nsize);
        for(int i=0; i<nsize; i++){ indxs[i].index=i; indxs[i].name=names[i].c_str(); }
        if(indxs) if(opt_sort) sort_index_name(indxs,nsize);
        for(int i=0; i<nsize; i++){
            if(!i && !opt_nohead){
                char line[LINESIZE]=""; char *p=line;
                p=strend(p); sprintf(p,"%s%-*s","Client ",nm,"Number");           
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,"PID");
                p=strend(p); sprintf(p,"%s%-*s",SPACES,nd,"Node");
                p=strend(p); int n=(int)(p-line);
                printhorline(n,'=');
                printf("%s\n",line);
                printhorline(n,'-');
                ntail=n;
            }
            int j=i; if(indxs) j=indxs[i].index;
            char line[LINESIZE]=""; char *p=line;
            p=strend(p); sprintf(p,"%s%-*d","Client ",nm,i+1);   
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nn,names[j].c_str());
            p=strend(p); sprintf(p,"%s%-*s",SPACES,nd,nodes[j].c_str());
            printf("%s\n",line);
        }
        if(ntail) printhorline(ntail,'-');
        flushstd();
        if(indxs) delete indxs;
        if(!nsize) return 1;
        return 0;
    }

    //
    // Apply statistics loop.
    //
    if((opt_stat>0)&&(opt_loop>0)&&(statservlist.size()>0)){
        for(uint i=0; i<statservlist.size(); i++){
            if(opt_verbose) printf("Add stat service %s\n",statservlist[i].c_str());
            StatService *item=new StatService(statservlist[i].c_str());
	    if(!item) fprintf(outerr(),"Could not stat service %s\n",statservlist[i].c_str());
        }
        if(opt_verbose) printf("DIM Stat %d service(s) during %d seconds...\n",(int)statservlist.size(),opt_stat);
        ClearScreen();
        stat->report(0,!opt_nohead,true,opt_sort,opt_top);
        for(int i=0; i<opt_loop; i++){
            timer->start(opt_stat);
            while(timer->runningFlag) { sleep(1); };
            ClearScreen();
            stat->report(i+1,!opt_nohead,true,opt_sort,opt_top);
        }
        flushstd();
    }

    return 0;
}
/////////////////
// END OF FILE //
/////////////////
