// *****************************************************************************
// Головной файл c объявлениями констант, структур, функций и т.д. для работы
// с модулями производства компании ЗАО "L-Kard" в среде Borland Delphi 6.0.
// Поддерживаемые модули: E14-140, E14-440 и E20-10.
// *****************************************************************************

unit Lusbapi;

interface

uses
	windows;

// Общие константы
const
	NAME_LINE_LENGTH_LUSBAPI				=	25;
	COMMENT_LINE_LENGTH_LUSBAPI			=	256;
	ADC_CALIBR_COEFS_QUANTITY_LUSBAPI	=	128;
	DAC_CALIBR_COEFS_QUANTITY_LUSBAPI	=	128;

// Общие структуры и типы данных
type
	pBYTE			= ^BYTE;
	pSHORT		= ^SHORT;
	pWORD			= ^WORD;
	pDWORD		= ^DWORD;
	pOVERLAPPED = ^OVERLAPPED;

	// структура, содержащая информацию о версии драйвера DSP
	IO_REQUEST_LUSBAPI = packed record
		Buffer : pSHORT;			 							//	указатель на буфер данных
		NumberOfWordsToPass : DWORD;			 			// кол-во отсчётов, которые требуется передать
		NumberOfWordsPassed : DWORD;						// реальное кол-во переданных отсчётов
		Overlapped : pOVERLAPPED;							// для асинхроннного запроса - указатель на структура типа OVERLAPPED
		TimeOut : DWORD;										// для синхронного запроса - таймаут в мс
	end;
	pIO_REQUEST_LUSBAPI = ^IO_REQUEST_LUSBAPI;

	// структура с информацией об последней ошибке выполнения библиотеки
	LAST_ERROR_INFO_LUSBAPI = packed record
		ErrorString : array [0..255] of BYTE;			// строка с последней ошибкой
		ErrorNumber : DWORD;	  								// номер последней ошибки
	end;
	pLAST_ERROR_INFO_LUSBAPI = ^LAST_ERROR_INFO_LUSBAPI;

	// информация о ПО, работающем в испольнительном устройстве: MCU, DSP, PLD и т.д.
	VERSION_INFO_LUSBAPI = packed record
		Version : array [0..9] of BYTE;				  	// версия ПО для испольнительного устройства
		Date : array [0..13] of BYTE;					  	// дата сборки ПО
		Manufacturer : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;	// производитель ПО
		Author : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;	  		// автор ПО
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о ПО MCU, которая включает в себя информацию о прошивках
	// как основной программы, так и загрузчика
	MCU_VERSION_INFO_LUSBAPI = packed record
		FwVersion : VERSION_INFO_LUSBAPI;				// информация о версии прошивки основной программы 'Приложение'(Application) микроконтроллера
		BlVersion : VERSION_INFO_LUSBAPI;				// информация о версии прошивки 'Загрузчика'(BootLoader) микроконтроллера
	end;

	// общая информация о модуле
	MODULE_INFO_LUSBAPI = packed record
		CompanyName : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// название фирмы-изготовителя изделия
		DeviceName : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE; 		// название изделия
		SerialNumber : array [0..15] of BYTE;			// серийный номер изделия
		Revision : BYTE;										// ревизия изделия
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о DSP
	DSP_INFO_LUSBAPI = packed record
		Active : BOOL;											// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;				// название DSP
		ClockRate : double;									// тактовая частота работы DSP в кГц
		Version : VERSION_INFO_LUSBAPI;					// информация о драйвере DSP
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о микроконтроллере (без 'Загрузчика')
	MCU_INFO_LUSBAPI = packed record
		Active : BOOL;										  	// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;				// название микроконтроллера
		ClockRate : double;								  	// тактовая частота работы микроконтроллера в кГц
		Version : VERSION_INFO_LUSBAPI;				  	// информация о версии прошивки микроконтроллера
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о микроконтроллере (включая 'Загрузчик')
	MCU1_INFO_LUSBAPI = packed record
		Active : BOOL;										  	// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;				// название микроконтроллера
		ClockRate : double;								  	// тактовая частота работы микроконтроллера в кГц
		Version : MCU_VERSION_INFO_LUSBAPI;		 	  	// информация о версии как самой прошивки микроконтроллера, так и прошивки 'Загрузчика'
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о ПЛИС (PLD)
	PLD_INFO_LUSBAPI = packed record				 		// PLD - Programmable Logic Device
		Active : BOOL;									 		// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;		  		// название ПЛИС
		ClockRate : double;							 		// тактовая чатота работы ПЛИС в кГц
		Version : VERSION_INFO_LUSBAPI;			  		// информация о версии прошивки ПЛИС
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о АЦП
	ADC_INFO_LUSBAPI = packed record
		Active : BOOL;									 		// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;				// название микросхемы АЦП
		OffsetCalibration : array [0..(ADC_CALIBR_COEFS_QUANTITY_LUSBAPI-1)] of double;	// корректировочные коэффициенты смещения нуля
		ScaleCalibration : array [0..(ADC_CALIBR_COEFS_QUANTITY_LUSBAPI-1)] of double; 	// корректировочные коэффициенты масштаба
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о ЦАП
	DAC_INFO_LUSBAPI = packed record
		Active : BOOL;									 		// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;				// название микросхемы ЦАП
		OffsetCalibration : array [0..(DAC_CALIBR_COEFS_QUANTITY_LUSBAPI-1)] of double;	// корректировочные коэффициенты
		ScaleCalibration : array [0..(DAC_CALIBR_COEFS_QUANTITY_LUSBAPI-1)] of double;  	// корректировочные коэффициенты
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о цифрового ввода-вывода
	DIGITAL_IO_INFO_LUSBAPI = packed record
		Active : BOOL;											// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;				// название цифровой микросхемы
		InLinesQuantity : WORD;	 							// кол-во входных линий
		OutLinesQuantity : WORD; 							// кол-во выходных линий
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	// информация о используемого интерфейса для доступа к модулю
	INTERFACE_INFO_LUSBAPI = packed record
		Active : BOOL;										 	// флаг достоверности остальных полей структуры
		Name : array [0..(NAME_LINE_LENGTH_LUSBAPI-1)] of BYTE;			 	// название
		Comment : array [0..(COMMENT_LINE_LENGTH_LUSBAPI-1)] of BYTE;		// строка комментария
	end;

	
// объявления интерфейса Lusbbase
// *****************************************************************************
// предок всех классов для USB устройств от L-Card
// *****************************************************************************

type
	Lusbbase = class
		// функции общего назначения для работы с USB устройствами
		Function OpenLDevice(VirtualSlot : WORD) : BOOL; virtual; stdcall; abstract;
		Function CloseLDevice : BOOL; virtual; stdcall; abstract;
		Function ReleaseLInstance : BOOL; virtual; stdcall; abstract;
		// получение дескриптора устройства USB
		Function GetModuleHandle : THandle; virtual; stdcall; abstract;
		// получение названия используемого модуля
		Function GetModuleName(ModuleName : pCHAR) : BOOL; virtual; stdcall; abstract;
		// получение текущей скорости работы шины USB
		Function GetUsbSpeed(UsbSpeed : pBYTE) : BOOL; virtual; stdcall; abstract;
		// функция выдачи строки с последней ошибкой
		Function GetLastErrorInfo(LastErrorInfo : pLAST_ERROR_INFO_LUSBAPI) : BOOL; virtual; stdcall; abstract;
	end;

	
// объявления интерфейса для модуля E14-140
// *****************************************************************************
// ******************  М о д у л ь    E 1 4 - 1 4 0  ***************************
// *****************************************************************************

// Константы
const
	// индексы доступных диапазонов входного напряжения модуля E14-440
	ADC_INPUT_RANGE_10000mV_E140			=	($00);
	ADC_INPUT_RANGE_2500mV_E140			=	($01);
	ADC_INPUT_RANGE_625mV_E140				=	($02);
	ADC_INPUT_RANGE_156mV_E140				=	($03);
	INVALID_ADC_INPUT_RANGE_E140			=	($04);

	// индексы доступных источников тактовых импульсов для АЦП
	INT_ADC_CLOCK_E140						=	($00);
	EXT_ADC_CLOCK_E140						=	($01);
	INVALID_ADC_CLOCK_E140				  	=	($02);

	// доступные индексы управления трансляцией тактовых импульсов АЦП
	// на линию SYN внешнего цифрового разъёма (только при внутреннем
	// источнике тактовых импульсоц АЦП)
	ADC_CLOCK_TRANS_DISABLED_E140		  	=	($00);
	ADC_CLOCK_TRANS_ENABLED_E140		  	=	($01);
	INVALID_ADC_CLOCK_TRANS_E140		  	=	($02);

	// индексы возможных типов синхронизации модуля E14-440
	NO_SYNC_E140								=	($00);
	TTL_START_SYNC_E140						=	($01);
	TTL_KADR_SYNC_E140						=	($02);
	ANALOG_SYNC_E140							=	($03);
	INVALID_SYNC_E140							=	($04);

	// индексы возможных опций наличия микросхемы ЦАП
	DAC_INACCESSIBLED_E140					=	($00);
	DAC_ACCESSIBLED_E140						=	($01);
	INVALID_DAC_OPTION_E140					=	($02);

	// индексы доступных ревизий модуля E14-440
	REVISION_A_E140 							=	($00);
	INVALID_REVISION_E140					=	($01);

	// константы для работы с модулем
	MAX_CONTROL_TABLE_LENGTH_E140			=	(128);
	ADC_INPUT_RANGES_QUANTITY_E140		=	(INVALID_ADC_INPUT_RANGE_E140);
	ADC_CALIBR_COEFS_QUANTITY_E140		=	(ADC_INPUT_RANGES_QUANTITY_E140);
	DAC_CHANNELS_QUANTITY_E140				=	($02);
	DAC_CALIBR_COEFS_QUANTITY_E140		=	(DAC_CHANNELS_QUANTITY_E140);
	TTL_LINES_QUANTITY_E140					=	($10);							// кол-во цифровых линий
	REVISIONS_QUANTITY_E140 				=	(INVALID_REVISION_E140);	// кол-во ревизий (модификаций) модуля

	// диапазоны входного напряжения АЦП в В
	ADC_INPUT_RANGES_E140 : array [0..(ADC_INPUT_RANGES_QUANTITY_E140-1)]  of double =	( 10.0, 10.0/4.0, 10.0/16.0, 10.0/64.0 );
	// диапазоны выходного напряжения ЦАП в В
	DAC_OUTPUT_RANGE_E140					=	5.0;
	// доступные ревизии модуля
	REVISIONS_E140 : array [0..(REVISIONS_QUANTITY_E140-1)] of char = ( 'A' );


// Типы
type
	// структура с информацией об модуле E14-440
	MODULE_DESCRIPTION_E140 = packed record
		Module 		: MODULE_INFO_LUSBAPI;		  	// общая информация о модуле
		DevInterface	: INTERFACE_INFO_LUSBAPI;	// информация об используемом устройстром интерфейсе
		Mcu			: MCU_INFO_LUSBAPI;				// информация о микроконтроллере
		Adc			: ADC_INFO_LUSBAPI;				// информация о АЦП
		Dac			: DAC_INFO_LUSBAPI;				// информация о ЦАП
		DigitalIo	: DIGITAL_IO_INFO_LUSBAPI;		// информация о цифровом вводе-выводе
	end;
	pMODULE_DESCRIPTION_E140 = ^MODULE_DESCRIPTION_E140;

	// структура параметров работы АЦП для модуля E14-440
	ADC_PARS_E140 = packed record
		ClkSource : WORD;			  							// источник тактовых импульсов для запуска АПП
		EnableClkOutput : WORD; 							// разрешение трансляции тактовых импульсов запуска АЦП
		InputMode : WORD;										// режим ввода даных с АЦП
		SynchroAdType : WORD;								// тип аналоговой синхронизации
		SynchroAdMode : WORD; 								// режим аналоговой сихронизации
		SynchroAdChannel : WORD;  							// канал АЦП при аналоговой синхронизации
		SynchroAdPorog : SHORT;								// порог срабатывания АЦП при аналоговой синхронизации
		ChannelsQuantity : WORD;							// число активных логических каналов
		ControlTable : array [0..(MAX_CONTROL_TABLE_LENGTH_E140-1)] of WORD;	// управляющая таблица с активными логическими каналами
		AdcRate : double;	  			  						// тактовая частота АЦП в кГц
		InterKadrDelay : double;		  					// межкадровая задержка в мс
		KadrRate : double;									// частота ввода кадра в кГц
	end;
	pADC_PARS_E140 = ^ADC_PARS_E140;

// интерфейс модуля E14-440
ILE140 = class(Lusbbase)
  public
		// функции для работы с АЦП
		Function GET_ADC_PARS(ap : pADC_PARS_E140) : BOOL; virtual; stdcall; abstract;
		Function SET_ADC_PARS(ap : pADC_PARS_E140) : BOOL; virtual; stdcall; abstract;
		Function START_ADC : BOOL; virtual; stdcall; abstract;
		Function STOP_ADC : BOOL; virtual; stdcall; abstract;
		Function ADC_KADR(Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function ADC_SAMPLE(AdcData : pSHORT; AdcChannel : WORD) : BOOL; virtual; stdcall; abstract;
		Function ReadData(ReadRequest : pIO_REQUEST_LUSBAPI) : BOOL; virtual; stdcall; abstract;

		// функции для работы с ЦАП
		Function DAC_SAMPLE(DacData : pSHORT; DacChannel : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с цифровыми линиями
		Function ENABLE_TTL_OUT(EnableTtlOut : BOOL) : BOOL; virtual; stdcall; abstract;
		Function TTL_IN(TtlIn : pWORD) : BOOL; virtual; stdcall; abstract;
		Function TTL_OUT(TtlOut : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с пользовательской информацией ППЗУ
		Function ENABLE_FLASH_WRITE(IsFlashWriteEnabled : BOOL) : BOOL; virtual; stdcall; abstract;
		Function READ_FLASH_WORD(FlashAddress : WORD; FlashWord : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function WRITE_FLASH_WORD(FlashAddress : WORD; FlashWord : SHORT) : BOOL; virtual; stdcall; abstract;

		// функции для работы со служебной информацией ППЗУ
		Function GET_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E140) : BOOL; virtual; stdcall; abstract;
		Function SAVE_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E140) : BOOL; virtual; stdcall; abstract;

	end;
	pILE140 = ^ILE140;

	
// объявления интерфейса для модуля E-154
// *****************************************************************************
// ******************  М о д у л ь    E - 1 5 4      ***************************
// *****************************************************************************

// Константы
const
	// индексы доступных диапазонов входного напряжения модуля E-154
	ADC_INPUT_RANGE_5000mV_E154			=	($00);
	ADC_INPUT_RANGE_1600mV_E154			=	($01);
	ADC_INPUT_RANGE_500mV_E154				=	($02);
	ADC_INPUT_RANGE_160mV_E154				=	($03);
	INVALID_ADC_INPUT_RANGE_E154			=	($04);

	// индексы доступных источников тактовых импульсов для АЦП
	INT_ADC_CLOCK_E154						=	($00);
	EXT_ADC_CLOCK_E154						=	($01);
	INVALID_ADC_CLOCK_E154				  	=	($02);

	// доступные индексы управления трансляцией тактовых импульсов АЦП
	// на линию SYN внешнего цифрового разъёма (только при внутреннем
	// источнике тактовых импульсоц АЦП)
	ADC_CLOCK_TRANS_DISABLED_E154		  	=	($00);
	ADC_CLOCK_TRANS_ENABLED_E154		  	=	($01);
	INVALID_ADC_CLOCK_TRANS_E154		  	=	($02);

	// индексы возможных типов синхронизации модуля E-154
	NO_SYNC_E154								=	($00);
	TTL_START_SYNC_E154						=	($01);
	TTL_KADR_SYNC_E154						=	($02);
	ANALOG_SYNC_E154							=	($03);
	INVALID_SYNC_E154							=	($04);

	// индексы возможных опций наличия микросхемы ЦАП
	DAC_INACCESSIBLED_E154					=	($00);
	DAC_ACCESSIBLED_E154						=	($01);
	INVALID_DAC_OPTION_E154					=	($02);

	// индексы доступных ревизий модуля E-154
	REVISION_A_E154 							=	($00);
	INVALID_REVISION_E154					=	($01);

	// константы для работы с модулем
	MAX_CONTROL_TABLE_LENGTH_E154			=	(16);
	ADC_INPUT_RANGES_QUANTITY_E154		=	(INVALID_ADC_INPUT_RANGE_E154);
	ADC_CALIBR_COEFS_QUANTITY_E154		=	(ADC_INPUT_RANGES_QUANTITY_E154);
	DAC_CHANNELS_QUANTITY_E154				=	($01);
	DAC_CALIBR_COEFS_QUANTITY_E154		=	(DAC_CHANNELS_QUANTITY_E154);
	TTL_LINES_QUANTITY_E154					=	($10);							// кол-во цифровых линий
	REVISIONS_QUANTITY_E154 				=	(INVALID_REVISION_E154);	// кол-во ревизий (модификаций) модуля

	// диапазоны входного напряжения АЦП в В
	ADC_INPUT_RANGES_E154 : array [0..(ADC_INPUT_RANGES_QUANTITY_E154-1)]  of double =	( 5.0, 1.6, 0.5, 0.16);
	// диапазоны выходного напряжения ЦАП в В
	DAC_OUTPUT_RANGE_E154					=	5.0;
	// доступные ревизии модуля
	REVISIONS_E154 : array [0..(REVISIONS_QUANTITY_E154-1)] of char = ( 'A' );


// Типы
type
	// структура с информацией об модуле E-154
	MODULE_DESCRIPTION_E154 = packed record
		Module 		: MODULE_INFO_LUSBAPI;		  	// общая информация о модуле
		DevInterface	: INTERFACE_INFO_LUSBAPI;	// информация об используемом устройстром интерфейсе
		Mcu			: MCU_INFO_LUSBAPI;				// информация о микроконтроллере
		Adc			: ADC_INFO_LUSBAPI;				// информация о АЦП
		Dac			: DAC_INFO_LUSBAPI;				// информация о ЦАП
		DigitalIo	: DIGITAL_IO_INFO_LUSBAPI;		// информация о цифровом вводе-выводе
	end;
	pMODULE_DESCRIPTION_E154 = ^MODULE_DESCRIPTION_E154;

	// структура параметров работы АЦП для модуля E-154
	ADC_PARS_E154 = packed record
		ClkSource : WORD;			  							// источник тактовых импульсов для запуска АПП
		EnableClkOutput : WORD; 							// разрешение трансляции тактовых импульсов запуска АЦП
		InputMode : WORD;										// режим ввода даных с АЦП
		SynchroAdType : WORD;								// тип аналоговой синхронизации
		SynchroAdMode : WORD; 								// режим аналоговой сихронизации
		SynchroAdChannel : WORD;  							// канал АЦП при аналоговой синхронизации
		SynchroAdPorog : SHORT;								// порог срабатывания АЦП при аналоговой синхронизации
		ChannelsQuantity : WORD;							// число активных логических каналов
		ControlTable : array [0..(MAX_CONTROL_TABLE_LENGTH_E154-1)] of WORD;	// управляющая таблица с активными логическими каналами
		AdcRate : double;	  			  						// тактовая частота АЦП в кГц
		InterKadrDelay : double;		  					// межкадровая задержка в мс
		KadrRate : double;									// частота ввода кадра в кГц
	end;
	pADC_PARS_E154 = ^ADC_PARS_E154;

// интерфейс модуля E-154
ILE154 = class(Lusbbase)
  public
		// функции для работы с АЦП
		Function GET_ADC_PARS(ap : pADC_PARS_E154) : BOOL; virtual; stdcall; abstract;
		Function SET_ADC_PARS(ap : pADC_PARS_E154) : BOOL; virtual; stdcall; abstract;
		Function START_ADC : BOOL; virtual; stdcall; abstract;
		Function STOP_ADC : BOOL; virtual; stdcall; abstract;
		Function ADC_KADR(Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function ADC_SAMPLE(AdcData : pSHORT; AdcChannel : WORD) : BOOL; virtual; stdcall; abstract;
		Function ReadData(ReadRequest : pIO_REQUEST_LUSBAPI) : BOOL; virtual; stdcall; abstract;
		Function ProcessArray(src : pSHORT; dest : pDOUBLE; size : DWORD; calibr : BOOL; volt : BOOL) : BOOL; virtual; stdcall; abstract;
		Function ProcessOnePoint(src : SHORT; dest : pDOUBLE; channel : DWORD; calibr : BOOL; volt : BOOL) : BOOL; virtual; stdcall; abstract;
		Function FIFO_STATUS(FifoOverflowFlag : pDWORD; FifoMaxPercentLoad : pDOUBLE; FifoSize : pDWORD; MaxFifoBytesUsed : pDWORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с ЦАП
		Function DAC_SAMPLE(DacData : pSHORT; DacChannel : WORD) : BOOL; virtual; stdcall; abstract;
 		Function DAC_SAMPLE_VOLT(DacData : DOUBLE; calibr : BOOL) : BOOL; virtual; stdcall; abstract;

		// функции для работы с цифровыми линиями
		Function ENABLE_TTL_OUT(EnableTtlOut : BOOL) : BOOL; virtual; stdcall; abstract;
		Function TTL_IN(TtlIn : pWORD) : BOOL; virtual; stdcall; abstract;
		Function TTL_OUT(TtlOut : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с пользовательской информацией ППЗУ
		Function ENABLE_FLASH_WRITE(IsFlashWriteEnabled : BOOL) : BOOL; virtual; stdcall; abstract;
 		Function READ_FLASH_ARRAY(UserFlash : pBYTE) : BOOL; virtual; stdcall; abstract;
		Function WRITE_FLASH_ARRAY(UserFlash : pBYTE) : BOOL; virtual; stdcall; abstract;

		// функции для работы со служебной информацией ППЗУ
		Function GET_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E154) : BOOL; virtual; stdcall; abstract;
		Function SAVE_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E154) : BOOL; virtual; stdcall; abstract;

		// функции для прямого досупа к ARM
		Function GetArray (Buffer : pSHORT; Size : WORD; Address : WORD)  : BOOL; virtual; stdcall; abstract;
  		Function PutArray(Buffer : pSHORT; Size : WORD; Address : WORD)  : BOOL; virtual; stdcall; abstract;

	end;
	pILE154 = ^ILE154;
	
	
// объявления интерфейса для модуля E14-440
// *****************************************************************************
// ******************  М о д у л ь    E 1 4 - 4 4 0  ***************************
// *****************************************************************************

// Константы
const
	// адрес начала сегмента данных LBIOS ( располагается в памяти программ DSP модуля)
	VarsBaseAddress_E440 					= $30;

	// адреса переменных штатного LBIOS для модуля E14-440 (раполагаются в памяти программ DSP модуля)
	L_PROGRAM_BASE_ADDRESS_E440		  	= 	(VarsBaseAddress_E440 + $0);
	L_READY_E440   							= 	(VarsBaseAddress_E440 + $1);
	L_TMODE1_E440   							= 	(VarsBaseAddress_E440 + $2);
	L_TMODE2_E440	   						= 	(VarsBaseAddress_E440 + $3);
	L_TEST_LOAD_E440	 						= 	(VarsBaseAddress_E440 + $4);
	L_COMMAND_E440		 						= 	(VarsBaseAddress_E440 + $5);

	L_DAC_SCLK_DIV_E440 						= 	(VarsBaseAddress_E440 + $7);
	L_DAC_RATE_E440	 						= 	(VarsBaseAddress_E440 + $8);
	L_ADC_RATE_E440	 						= 	(VarsBaseAddress_E440 + $9);
	L_ADC_ENABLED_E440 						= 	(VarsBaseAddress_E440 + $A);
	L_ADC_FIFO_BASE_ADDRESS_E440			= 	(VarsBaseAddress_E440 + $B);
	L_CUR_ADC_FIFO_LENGTH_E440				= 	(VarsBaseAddress_E440 + $C);
	L_ADC_FIFO_LENGTH_E440					= 	(VarsBaseAddress_E440 + $E);
	L_CORRECTION_ENABLED_E440				= 	(VarsBaseAddress_E440 + $F);

	L_ADC_SAMPLE_E440							= 	(VarsBaseAddress_E440 + $11);
	L_ADC_CHANNEL_E440	 					= 	(VarsBaseAddress_E440 + $12);
	L_INPUT_MODE_E440							= 	(VarsBaseAddress_E440 + $13);

	L_SYNCHRO_AD_CHANNEL_E440				= 	(VarsBaseAddress_E440 + $16);
	L_SYNCHRO_AD_POROG_E440					= 	(VarsBaseAddress_E440 + $17);
	L_SYNCHRO_AD_MODE_E440 					= 	(VarsBaseAddress_E440 + $18);
	L_SYNCHRO_AD_TYPE_E440 					= 	(VarsBaseAddress_E440 + $19);

	L_CONTROL_TABLE_LENGHT_E440			= 	(VarsBaseAddress_E440 + $1B);
	L_FIRST_SAMPLE_DELAY_E440				= 	(VarsBaseAddress_E440 + $1C);
	L_INTER_KADR_DELAY_E440					= 	(VarsBaseAddress_E440 + $1D);

	L_DAC_SAMPLE_E440							= 	(VarsBaseAddress_E440 + $20);
	L_DAC_ENABLED_E440					 	= 	(VarsBaseAddress_E440 + $21);
	L_DAC_FIFO_BASE_ADDRESS_E440			= 	(VarsBaseAddress_E440 + $22);
	L_CUR_DAC_FIFO_LENGTH_E440				= 	(VarsBaseAddress_E440 + $24);
	L_DAC_FIFO_LENGTH_E440					= 	(VarsBaseAddress_E440 + $25);

	L_FLASH_ENABLED_E440 					= 	(VarsBaseAddress_E440 + $26);
	L_FLASH_ADDRESS_E440 					= 	(VarsBaseAddress_E440 + $27);
	L_FLASH_DATA_E440 						= 	(VarsBaseAddress_E440 + $28);

	L_ENABLE_TTL_OUT_E440					= 	(VarsBaseAddress_E440 + $29);
	L_TTL_OUT_E440								= 	(VarsBaseAddress_E440 + $2A);
	L_TTL_IN_E440								= 	(VarsBaseAddress_E440 + $2B);

	L_SCALE_E440								= 	(VarsBaseAddress_E440 + $30);
	L_ZERO_E440									= 	(VarsBaseAddress_E440 + $34);

	L_CONTROL_TABLE_E440						=	($80);

	L_DSP_INFO_STUCTURE_E440				=	($200);

	// индексы доступных состояний сброса модуля E14-440
	INIT_E440									=	($00);
	RESET_E440									=	($01);
	INVALID_RESET_TYPE_E440					=	($02);

	// индексы доступных диапазонов входного напряжения модуля E14-440
	ADC_INPUT_RANGE_10000mV_E440			=	($00);
	ADC_INPUT_RANGE_2500mV_E440			=	($01);
	ADC_INPUT_RANGE_625mV_E440				=	($02);
	ADC_INPUT_RANGE_156mV_E440				=	($03);
	INVALID_ADC_INPUT_RANGE_E440			=	($04);

	// индексы возможных типов синхронизации модуля E14-440
	NO_SYNC_E440								=	($00);
	TTL_START_SYNC_E440						=	($01);
	TTL_KADR_SYNC_E440						=	($02);
	ANALOG_SYNC_E440							=	($03);
	INVALID_SYNC_E440							=	($04);

	// индексы возможных опций наличия микросхемы ЦАП
	DAC_INACCESSIBLED_E440					=	($00);
	DAC_ACCESSIBLED_E440						=	($01);
	INVALID_DAC_OPTION_E440					=	($02);

	// индексы возможных типов DSP (сейчас только ADSP-2185)
	ADSP2184_E440								=	($00);
	ADSP2185_E440								=	($01);
	ADSP2186_E440								=	($02);
	INVALID_DSP_TYPE_E440					=	($03);

	// индексы возможных тактовых частот модудя (сейчас только 24000 кГц)
	F14745_E440									=	($00);
	F16667_E440									=	($01);
	F20000_E440									=	($02);
	F24000_E440									=	($03);
	INVALID_QUARTZ_FREQ_E440				=	($04);

	// индексы доступных ревизий модуля E14-440
	REVISION_A_E440	  						=	($00);
	REVISION_B_E440	  						=	($01);
	REVISION_C_E440	  						=	($02);
	REVISION_D_E440							=	($03);
	REVISION_E_E440							=	($04);
	INVALID_REVISION_E440					=	($05);

	// константы для работы с модулем
	MAX_CONTROL_TABLE_LENGTH_E440			=	(128);
	ADC_INPUT_RANGES_QUANTITY_E440		=	(INVALID_ADC_INPUT_RANGE_E440);
	ADC_CALIBR_COEFS_QUANTITY_E440		=	(ADC_INPUT_RANGES_QUANTITY_E440);
	MAX_ADC_FIFO_SIZE_E440					=	($3000);			// 12288
	DAC_CHANNELS_QUANTITY_E440				=	($02);
	DAC_CALIBR_COEFS_QUANTITY_E440		=	(DAC_CHANNELS_QUANTITY_E440);
	MAX_DAC_FIFO_SIZE_E440					=	($0FC0);			// 4032
	TTL_LINES_QUANTITY_E440					=	($10);							// кол-во цифровых линий
	REVISIONS_QUANTITY_E440 				=	(INVALID_REVISION_E440);	// кол-во ревизий (модификаций) модуля

	// диапазоны входного напряжения АЦП в В
	ADC_INPUT_RANGES_E440 : array [0..(ADC_INPUT_RANGES_QUANTITY_E440-1)]  of double =	( 10.0, 10.0/4.0, 10.0/16.0, 10.0/64.0 );
	// диапазоны выходного напряжения ЦАП в В
	DAC_OUTPUT_RANGE_E440					=	5.0;
	// доступные ревизии модуля
	REVISIONS_E440 : array [0..(REVISIONS_QUANTITY_E440-1)] of char = ( 'A', 'B', 'C', 'D', 'E'  );

   
// Типы
type
	// структура с информацией об модуле E14-440
	MODULE_DESCRIPTION_E440 = packed record
		Module 		: MODULE_INFO_LUSBAPI;		  	// общая информация о модуле
		DevInterface	: INTERFACE_INFO_LUSBAPI;	// информация об используемом устройстром интерфейсе
		Mcu			: MCU_INFO_LUSBAPI;				// информация о микроконтроллере
		Dsp			: DSP_INFO_LUSBAPI;				// информация о DSP
		Adc			: ADC_INFO_LUSBAPI;				// информация о АЦП
		Dac			: DAC_INFO_LUSBAPI;				// информация о ЦАП
		DigitalIo	: DIGITAL_IO_INFO_LUSBAPI;		// информация о цифровом вводе-выводе
	end;
	pMODULE_DESCRIPTION_E440 = ^MODULE_DESCRIPTION_E440;

	// структура параметров работы АЦП для модуля E14-440
	ADC_PARS_E440 = packed record
		IsAdcEnabled : BOOL;			  						// флажок разрешение/запрещение ввода данных (только при чтении)
		IsCorrectionEnabled : BOOL;						// разрешение управление корректировкой данных на уровне драйвера DSP
		InputMode : WORD;										// режим ввода даных с АЦП
		SynchroAdType : WORD;								// тип аналоговой синхронизации
		SynchroAdMode : WORD; 								// режим аналоговой сихронизации
		SynchroAdChannel : WORD;  							// канал АЦП при аналоговой синхронизации
		SynchroAdPorog : SHORT;								// порог срабатывания АЦП при аналоговой синхронизации
		ChannelsQuantity : WORD;							// число активных логических каналов
		ControlTable : array [0..(MAX_CONTROL_TABLE_LENGTH_E440-1)] of WORD;	// управляющая таблица с активными логическими каналами
		AdcRate : double;	  			  						// тактовая частота АЦП в кГц
		InterKadrDelay : double;		  					// межкадровая задержка в мс
		KadrRate : double;									// частота кадра в кГц
		AdcFifoBaseAddress : WORD;							// базовый адрес FIFO буфера ввода
		AdcFifoLength : WORD;	  							// длина FIFO буфера ввода
		AdcOffsetCoefs : array [0..3] of double; 		// корректировочные коэф. смещение нуля для АЦП
		AdcScaleCoefs : array [0..3] of double; 		// корректировочные коэф. масштаба для АЦП
	end;
	pADC_PARS_E440 = ^ADC_PARS_E440;

	// структура параметров работы ЦАП модуля E14-440
	DAC_PARS_E440 = packed record
		DacEnabled : BOOL;									// разрешение/запрещение вывода данных
		DacRate : double;	  		  							// частота вывода данных в кГц
		DacFifoBaseAddress : WORD;  						// базовый адрес FIFO буфера вывода
		DacFifoLength : WORD;								// длина FIFO буфера вывода
	end;
	pDAC_PARS_E440 = ^DAC_PARS_E440;

// интерфейс модуля E14-440
ILE440 = class(Lusbbase)
  public
		// функции для работы с DSP модуля
		Function RESET_MODULE(ResetFlag : BYTE = INIT_E440) : BOOL; virtual; stdcall; abstract;
		Function LOAD_MODULE(FileName : pCHAR = nil) : BOOL; virtual; stdcall; abstract;
		Function TEST_MODULE : BOOL; virtual; stdcall; abstract;
		Function SEND_COMMAND(Command : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с АЦП
		Function GET_ADC_PARS(ap : pADC_PARS_E440) : BOOL; virtual; stdcall; abstract;
		Function SET_ADC_PARS(ap : pADC_PARS_E440) : BOOL; virtual; stdcall; abstract;
		Function START_ADC : BOOL; virtual; stdcall; abstract;
		Function STOP_ADC : BOOL; virtual; stdcall; abstract;
		Function ADC_KADR(Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function ADC_SAMPLE(AdcData : pSHORT; AdcChannel : WORD) : BOOL; virtual; stdcall; abstract;
		Function ReadData(ReadRequest : pIO_REQUEST_LUSBAPI) : BOOL; virtual; stdcall; abstract;

		// функции для работы с ЦАП
		Function GET_DAC_PARS(dm : pDAC_PARS_E440) : BOOL; virtual; stdcall; abstract;
		Function SET_DAC_PARS(dm : pDAC_PARS_E440) : BOOL; virtual; stdcall; abstract;
		Function START_DAC : BOOL; virtual; stdcall; abstract;
		Function STOP_DAC : BOOL; virtual; stdcall; abstract;
		Function WriteData(WriteRequest : pIO_REQUEST_LUSBAPI) : BOOL; virtual; stdcall; abstract;
		Function DAC_SAMPLE(DacData : pSHORT; DacChannel : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с цифровыми линиями
		Function ENABLE_TTL_OUT(EnableTtlOut : BOOL) : BOOL; virtual; stdcall; abstract;
		Function TTL_IN(TtlIn : pWORD) : BOOL; virtual; stdcall; abstract;
		Function TTL_OUT(TtlOut : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы ППЗУ
		Function ENABLE_FLASH_WRITE(EnableFlashWrite : BOOL) : BOOL; virtual; stdcall; abstract;
		Function READ_FLASH_WORD(FlashAddress : WORD; FlashWord : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function WRITE_FLASH_WORD(FlashAddress : WORD; FlashWord : SHORT) : BOOL; virtual; stdcall; abstract;

		// функции для работы со служебной информацией из ППЗУ
		Function GET_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E440) : BOOL; virtual; stdcall; abstract;
		Function SAVE_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E440) : BOOL; virtual; stdcall; abstract;

		// функции для работы с памятью DSP
		Function PUT_LBIOS_WORD(Address : WORD; Data : SHORT) : BOOL; virtual; stdcall; abstract;
		Function GET_LBIOS_WORD(Address : WORD; Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function PUT_DM_WORD(Address : WORD; Data : SHORT) : BOOL; virtual; stdcall; abstract;
		Function GET_DM_WORD(Address : WORD; Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function PUT_PM_WORD(Address : WORD; Data : DWORD) : BOOL; virtual; stdcall; abstract;
		Function GET_PM_WORD(Address : WORD; Data : pDWORD) : BOOL; virtual; stdcall; abstract;
		Function PUT_DM_ARRAY(BaseAddress, NPoints : WORD; Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function GET_DM_ARRAY(BaseAddress, NPoints : WORD; Data : pSHORT) : BOOL; virtual; stdcall; abstract;
		Function PUT_PM_ARRAY(BaseAddress, NPoints : WORD; Data : pDWORD) : BOOL; virtual; stdcall; abstract;
		Function GET_PM_ARRAY(BaseAddress, NPoints : WORD; Data : pDWORD) : BOOL; virtual; stdcall; abstract;
	end;
	pILE440 = ^ILE440;


// объявления интерфейса для модуля E20-10
// *****************************************************************************
// ******************  М о д у л ь    E 2 0 - 1 0  *****************************
// *****************************************************************************

// Константы
const
	// индексы доступных источников импульса старта сбора данных
	INT_ADC_START_E2010							=	($00);
	INT_ADC_START_WITH_TRANS_E2010			=	($01);
	EXT_ADC_START_ON_RISING_EDGE_E2010	 	=	($02);
	EXT_ADC_START_ON_FALLING_EDGE_E2010		=	($03);
	INVALID_ADC_START_E2010						=	($04);

	// индексы доступных источников тактовых импульсов для АЦП
	INT_ADC_CLOCK_E2010							=	($00);
	INT_ADC_CLOCK_WITH_TRANS_E2010			=	($01);
	EXT_ADC_CLOCK_ON_RISING_EDGE_E2010	 	=	($02);
	EXT_ADC_CLOCK_ON_FALLING_EDGE_E2010		=	($03);
	INVALID_ADC_CLOCK_E2010						=	($04);

	// возможные типы аналоговой синхронизации ввода данных (для Rev.B и выше)
	NO_ANALOG_SYNCHRO_E2010					 		=	($00); 	// отсутствие аналоговой синхронизации
	ANALOG_SYNCHRO_ON_RISING_CROSSING_E2010	=	($01);	// аналоговая синхронизация по переходу снизу-вверх
   ANALOG_SYNCHRO_ON_FALLING_CROSSING_E2010	=	($02);	// аналоговая синхронизация по переходу сверху-вниз
	ANALOG_SYNCHRO_ON_HIGH_LEVEL_E2010			=	($03);	// аналоговая синхронизация по уровню выше
   ANALOG_SYNCHRO_ON_LOW_LEVEL_E2010			=	($04);	// аналоговая синхронизация по уровню ниже
	INVALID_ANALOG_SYNCHRO_E2010					=	($05);

	// индексы доступных диапазонов входного напряжения модуля E20-10
	ADC_INPUT_RANGE_3000mV_E2010				=	($00);
	ADC_INPUT_RANGE_1000mV_E2010				=	($01);
	ADC_INPUT_RANGE_300mV_E2010				=	($02);
	INVALID_ADC_INPUT_RANGE_E2010				=	($03);

	// индексы возможных типов подключения входного тракта модуля E20-10
	ADC_INPUT_ZERO_E2010							=	($00);
	ADC_INPUT_SIGNAL_E2010					 	=	($01);
	INVALID_ADC_INPUT_E2010						=	($02);

	// возможные индексы для управления входным током смещения модуля E20-10 (для Rev.B и выше)
	INPUT_CURRENT_OFF_E2010						=	($00);
	INPUT_CURRENT_ON_E2010						=	($01);
	INVALID_INPUT_CURRENT_E2010				=	($02);

	// возможные режимы фиксации факта перегрузки входных каналов при сборе данных (только для Rev.A)
	CLIPPING_OVERLOAD_E2010		 				=	($00);
   MARKER_OVERLOAD_E2010		 				=	($01);
   INVALID_OVERLOAD_E2010		 				=	($02);

	// доступные номера битов ошибок при выполнении сбора данных с АЦП
		// поле BufferOverrun структуры DATA_STATE_E2010
		BUFFER_OVERRUN_E2010	 					=	($00);	// переполнение внутреннего буфера модуля
		// поле ChannelsOverFlow структуры DATA_STATE_E2010
		OVERFLOW_OF_CHANNEL_1_E2010	 		=	($00);	// локальные признаки переполнения
	   OVERFLOW_OF_CHANNEL_2_E2010			=	($01);	// разрядной сетки соответствующего канала
	   OVERFLOW_OF_CHANNEL_3_E2010	 		=	($03);	// за время выполнения одного запроса
	   OVERFLOW_OF_CHANNEL_4_E2010	 		=	($04);	// сбора данных ReadData()
		OVERFLOW_E2010	 							=	($07);	// глобальный признак факта переполнения разрядной сетки
																		// модуля за всё время сбора данных от момента START_ADC() до STOP_ADC()

	// доступные константы для задания тестовых режимов работы модуля E20-10 (для Rev.B и выше)
	// используются в функции TEST_MODULE()   
	NO_TEST_MODE_E2010	 						=	($00); 	// не задействовано никаких тестовых режимов
   TEST_MODE_1_E2010		 						=	($01); 	// тестовый режим, при котором после START_ADC() данные с АЦП
																		// подменяются обычным натуральным рядом: 0, 1, 2, 3, ...

	// индексы возможных опции наличия микросхемы ЦАП
	DAC_INACCESSIBLED_E2010						=	($00);
	DAC_ACCESSIBLED_E2010						=	($01);
	INVALID_DAC_OPTION_E2010					=	($02);

	// индексы доступных ревизий модуля E20-10
	REVISION_A_E2010								=	($00);
	REVISION_B_E2010								=	($01);
	INVALID_REVISION_E2010						=	($02);

	// константы для работы с модулем
	MAX_CONTROL_TABLE_LENGTH_E2010 			=	(256);
	ADC_CHANNELS_QUANTITY_E2010				=	($04);
	ADC_INPUT_RANGES_QUANTITY_E2010			=	(INVALID_ADC_INPUT_RANGE_E2010);
	ADC_INPUT_TYPES_QUANTITY_E2010			=	(INVALID_ADC_INPUT_E2010);
	ADC_CALIBR_COEFS_QUANTITY_E2010			=	(ADC_CHANNELS_QUANTITY_E2010 * ADC_INPUT_RANGES_QUANTITY_E2010);
	DAC_CHANNELS_QUANTITY_E2010				=	($02);
	DAC_CALIBR_COEFS_QUANTITY_E2010			=	(DAC_CHANNELS_QUANTITY_E2010);
	TTL_LINES_QUANTITY_E2010					=	($10);	// кол-во входных и выходных цифровых линий
	USER_FLASH_SIZE_E2010 						=	($200);	// размер области пользовательского ППЗУ в байтах
	REVISIONS_QUANTITY_E2010 					=	(INVALID_REVISION_E2010);	// кол-во ревизий (модификаций) модуля
	ADC_PLUS_OVERLOAD_MARKER_E2010			=	($5FFF);	// признак 'плюс' перегрузки отсчёта с АЦП (только для Rev.A)
	ADC_MINUS_OVERLOAD_MARKER_E2010			=	($A000);	// признак 'минус' перегрузки отсчёта с АЦП (только для Rev.A)

	// диапазоны входного напряжения АЦП в В
	ADC_INPUT_RANGES_E2010 : array [0..(ADC_INPUT_RANGES_QUANTITY_E2010-1)]  of double =	( 3.0, 1.0, 0.3 );
	// диапазоны выходного напряжения ЦАП в В
	DAC_OUTPUT_RANGE_E2010	  					=	5.0;
	// доступные ревизии модуля
	REVISIONS_E2010 : array [0..(REVISIONS_QUANTITY_E2010-1)] of char = ( 'A', 'B' );


// Типы
type
	// структура с информацией об модуле E20-10
	MODULE_DESCRIPTION_E2010 = packed record
		Module 		: MODULE_INFO_LUSBAPI;		  	// общая информация о модуле
		DevInterface	: INTERFACE_INFO_LUSBAPI;	// информация об используемом устройстром интерфейсе
		Mcu			: MCU1_INFO_LUSBAPI;				// информация о микроконтроллере (включая 'Загрузчик')
		Pld			: PLD_INFO_LUSBAPI;				// информация о ПЛИС
		Adc			: ADC_INFO_LUSBAPI;				// информация о АЦП
		Dac			: DAC_INFO_LUSBAPI;				// информация о ЦАП
		DigitalIo	: DIGITAL_IO_INFO_LUSBAPI;		// информация о цифровом вводе-выводе
	end;
	pMODULE_DESCRIPTION_E2010 = ^MODULE_DESCRIPTION_E2010;

	// структура пользовательского ППЗУ
	USER_FLASH_E2010 = packed record
		Buffer : array [0..(USER_FLASH_SIZE_E2010-1)] of BYTE;
	end;
	pUSER_FLASH_E2010 = ^USER_FLASH_E2010;

	// структура с параметрами синхронизации ввода данных с АЦП
	SYNCHRO_PARS_E2010 = packed record
		StartSource		: WORD;				  	// тип и источник сигнала начала сбора данных с АЦП (внутренний или внешний и т.д.)
		StartDelay		: DWORD;					// задержка старта сбора данных в кадрах отсчётов c АЦП (для Rev.B и выше)
		SynhroSource	: WORD;					// источник тактовых импульсов запуска АЦП (внутренние или внешние и т.д.)
		StopAfterNKadrs: DWORD;					// останов сбора данных после задаваемого здесь кол-ва собранных кадров отсчётов АЦП (для Rev.B и выше)
		SynchroAdMode	: WORD;   				// режим аналоговой сихронизации: переход или уровень (для Rev.B и выше)
		SynchroAdChannel	: WORD;				// физический канал АЦП для аналоговой синхронизации (для Rev.B и выше)
		SynchroAdPorog		: SHORT;				// порог срабатывания при аналоговой синхронизации (для Rev.B и выше)
		IsBlockDataMarkerEnabled : BYTE;		// маркирование начала блока данных (удобно, например, при аналоговой синхронизации ввода по уровню) (для Rev.B и выше)
	end;

	// структура параметров работы АЦП для модуля E20-10
	ADC_PARS_E2010 = packed record
		IsAdcCorrectionEnabled : BOOL;		// управление разрешением автоматической корректировкой получаемых данных на уровне модуля (для Rev.B и выше)
		OverloadMode : WORD;	  					// режим фиксации факта перегрузки входных каналов модуля (только для Rev.A)
		InputCurrentControl: WORD;				// управление входным током смещения (для Rev.B и выше)
		SynchroPars : SYNCHRO_PARS_E2010;	// параметры синхронизации ввода данных с АЦП
		ChannelsQuantity : WORD; 		// число активных логических каналов
		ControlTable : array [0..(MAX_CONTROL_TABLE_LENGTH_E2010-1)] of WORD;	// управляющая таблица с активными логическими каналами
		InputRange : array [0..(ADC_CHANNELS_QUANTITY_E2010-1)] of WORD;	// индекс диапазона входного напряжения: 3.0В, 1.0В или 0.3В
		InputSwitch : array [0..(ADC_CHANNELS_QUANTITY_E2010-1)] of WORD;	// индекс типа входа канала АЦП: земля или сигнал
		AdcRate : double;	  		 		// тактовая частота АЦП в кГц
		InterKadrDelay : double; 		// межкадровая задержка в мс
		KadrRate : double;		 		// частота кадра в кГц
		AdcOffsetCoefs : array [0..(ADC_INPUT_RANGES_QUANTITY_E2010-1)] of array[0..(ADC_CHANNELS_QUANTITY_E2010-1)] of double;	// корректировочные коэффициенты смещение	АЦП: (3 диапазона)*(4 канала) (для Rev.B и выше)
		AdcScaleCoefs	: array [0..(ADC_INPUT_RANGES_QUANTITY_E2010-1)] of array[0..(ADC_CHANNELS_QUANTITY_E2010-1)] of double;	// корректировочные коэффициенты масштаба АЦП: (3 диапазона)*(4 канала) (для Rev.B и выше)
	end;
	pADC_PARS_E2010 = ^ADC_PARS_E2010;

	// структура с информацией о текущем состоянии процесса сбора данных
	DATA_STATE_E2010 = packed record
		ChannelsOverFlow : BYTE;				// битовые признаки перегрузки входных аналоговых каналов модуля для Rev.B и выше
		BufferOverrun : BYTE;					// битовые признаки переполнения внутреннего буфера модуля
		CurBufferFilling : DWORD;				// заполненность внутреннего буфера модуля для Rev.B и выше, в отсчётах
		MaxOfBufferFilling : DWORD;			// за время сбора максимальная заполненность внутреннего буфера модуля для Rev.B и выше, в отсчётах
		BufferSize : DWORD;						// размер внутреннего буфера модуля для Rev.B и выше, в отсчётах
		CurBufferFillingPercent : double;	// текущая степень заполнения внутреннего буфера модуля для Rev.B и выше, в %
		MaxOfBufferFillingPercent : double;	// за всё время сбора максимальная степень заполнения внутреннего буфера модуля для Rev.B и выше, в %
	end;
	pDATA_STATE_E2010 = ^DATA_STATE_E2010;

// интерфейс модуля E20-10
ILE2010 = class(Lusbbase)
  public
		// загрузка ПЛИС модуля
		Function LOAD_MODULE(FileName : pCHAR = nil) : BOOL; virtual; stdcall; abstract;
		Function TEST_MODULE(TestModeMask : WORD = 0) : BOOL; virtual; stdcall; abstract;

		// функции для работы с АЦП
		Function GET_ADC_PARS(ap : pADC_PARS_E2010) : BOOL; virtual; stdcall; abstract;
		Function SET_ADC_PARS(ap : pADC_PARS_E2010) : BOOL; virtual; stdcall; abstract;
		Function START_ADC : BOOL; virtual; stdcall; abstract;
		Function STOP_ADC : BOOL; virtual; stdcall; abstract;
		Function GET_DATA_STATE(DataState : pDATA_STATE_E2010) : BOOL; virtual; stdcall; abstract;
		Function ReadData(ReadRequest : pIO_REQUEST_LUSBAPI) : BOOL; virtual; stdcall; abstract;

		// функции для работы с ЦАП
		Function DAC_SAMPLE(DacData : pSHORT; DacChannel : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с цифровыми линиями
		Function ENABLE_TTL_OUT(EnableTtlOut : BOOL) : BOOL; virtual; stdcall; abstract;
		Function TTL_IN(TtlIn : pWORD) : BOOL; virtual; stdcall; abstract;
		Function TTL_OUT(TtlOut : WORD) : BOOL; virtual; stdcall; abstract;

		// функции для работы с пользовательской информацией ППЗУ
		Function ENABLE_FLASH_WRITE(IsUserFlashWriteEnabled : BOOL) : BOOL; virtual; stdcall; abstract;
		Function READ_FLASH_ARRAY(UserFlash : pUSER_FLASH_E2010) : BOOL; virtual; stdcall; abstract;
		Function WRITE_FLASH_ARRAY(UserFlash : pUSER_FLASH_E2010) : BOOL; virtual; stdcall; abstract;

		// функции для работы со служебной информацией из ППЗУ
		Function GET_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E2010) : BOOL; virtual; stdcall; abstract;
		Function SAVE_MODULE_DESCRIPTION(ModuleDescription : pMODULE_DESCRIPTION_E2010) : BOOL; virtual; stdcall; abstract;
	end;
	pILE2010 = ^ILE2010;


const
	// текущая версия штатной библиотеки Lusbapi.dll
	VERSION_MAJOR_LUSBAPI 			=	($03);
	VERSION_MINOR_LUSBAPI 			=	($02);
	CURRENT_VERSION_LUSBAPI			=	((VERSION_MAJOR_LUSBAPI shl 16) or VERSION_MINOR_LUSBAPI);

	// возможные индексы скорости работы модуля на шине USB
	USB11_LUSBAPI						=	($00);
	USB20_LUSBAPI						=	($01);
	INVALID_USB_SPEED_LUSBAPI		=	($02);

	// максимально возможное кол-во опрашиваемых виртуальных слотов
	MAX_VIRTUAL_SLOTS_QUANTITY_LUSBAPI	= 127;

// объявление экспортируемых из Lusbapi.dll функций
Function GetDllVersion : DWORD; stdcall;
Function CreateLInstance(DeviceName : PChar) : Pointer; stdcall;

Function InitLusbapiDll(const name:String='Lusbapi.dll'):Boolean;
Function FreeLusbapiDll:Boolean;

implementation

const
 dll              : THandle = 0;
 _GetDllVersion   : Function : DWORD; stdcall = nil;
 _CreateLInstance : Function(DeviceName : PChar) : Pointer; stdcall = nil;

Function GetDllVersion : DWORD;
begin
 if dll<>0 then Result:=_GetDllVersion else Result:=0;
end;

Function CreateLInstance(DeviceName : PChar) : Pointer;
begin
 if dll<>0 then Result:=_CreateLInstance(DeviceName) else Result:=nil;
end;

Function InitLusbapiDll(const name:String='Lusbapi.dll'):Boolean;
begin
 if dll = 0 then begin
  dll:=LoadLibrary(PChar(name));
  if dll <> 0 then begin
   _GetDllVersion:=GetProcAddress(dll,'GetDllVersion');
   _CreateLInstance:=GetProcAddress(dll,'CreateLInstance');
   if not Assigned(_GetDllVersion) then FreeLusbapiDll;
   if not Assigned(_CreateLInstance) then FreeLusbapiDll;
  end;
 end;
 Result:=(dll<>0); 
end;

Function FreeLusbapiDll:Boolean;
begin
 if dll<>0
 then Result:=FreeLibrary(dll)
 else Result:=True;
 dll:=0;
end;

initialization

 InitLusbapiDll;
 
finalization
 
 FreeLusbapiDll;

end.
