 //////////////////////////////////////////////////////////////////////////////
 // L-CARD E14-140 driver for CRW-DAQ: main driver class
 // Last release: 20101112 by Kuruakin Alexey <kouriakine@mail.ru>
 //////////////////////////////////////////////////////////////////////////////
unit E140_API;

{$I _sysdef}

interface

uses
 SysUtils, Windows, Math, Classes, Forms,
 _alloc, _rtc, _sharm, _ascio, Lusbapi, E140_SHM;

 //
 // E14-140 driver class
 //
type
 E140_FAILURE = class(Exception);
 TE140_Reporter = procedure(const Msg:LongString); // Procedure to report errors
 TE140 = class(TMasterObject)
 private
  myDll       : DWORD;                     // Version of Lusbapi.dll
  myUsb       : BYTE;                      // USB speed (12/480 Mbps)
  myApi       : ILE140;	                   // E14-140 device interface
  myHand      : THandle;                   // Device handle identifier 
  myName      : LongString;                // Module name, must be E140
  mySharm     : TSharedMemory;             // Shared memory instance
  myShared    : PE140_SHARED_BUFFER;       // Shared memory pointer
  mySerNum    : LongString;                // L-CARD Serial number
  myRequest   : record                     // Data for I/O requests
   Number     : Integer;                   // 0..E140NumBuffers-1
   Blocks     : Int64;                     // Total number of blocks
   Samples    : Int64;                     // Total number of samples
   Index      : array[0..E140NumBuffers-1] of Integer;    // Shared memory index
   Offset     : array[0..E140NumBuffers-1] of Integer;    // Shared memory offset
   ReadOv     : array[0..E140NumBuffers-1] of OVERLAPPED; // Overlapped struct
   IoReq      : array[0..E140NumBuffers-1] of IO_REQUEST_LUSBAPI; // For ReatData
  end;
  myReporter  : TE140_Reporter;            // Error report routine
  myBuffTime  : Integer;                   // Buffer capacity in time
  myIsOem     : Boolean;                   // Convert messages to ОЕМ
 private
  procedure   SetReporter(aReporter:TE140_Reporter);
  procedure   PrintReport(const msg:LongString);
  function    GetIsOem:Boolean;
  procedure   SetIsOem(aOem:Boolean);
  procedure   Success(const s:LongString); 
  procedure   Failure(const s:LongString);
  procedure   BugReport(const E:Exception; const msg:LongString='');
  procedure   PrintDescr(const aDescr:MODULE_DESCRIPTION_E140);
  procedure   PrintADC(const aAdc:ADC_PARS_E140);
  function    FindBufferLength(aBuffTime:Integer; const aAdc:ADC_PARS_E140):Integer;
  procedure   ClearIoRequests(IoCancel:Boolean);
  function    GetSharedBuff:PE140_SHARED_BUFFER;
  function    GetSharedName:LongString;
  function    GetSerNum:LongString;
  procedure   SetSerNum(const aSerNum:LongString);
 protected
  function    ProcessBuffer(nWords:DWORD; nRequest:Integer; nSamples:Int64; nBlocks:Int64):DWORD; virtual;
 public
  constructor Create(aShare:LongString; aReporter:TE140_Reporter=nil; aAdc:pADC_PARS_E140=nil; aOpen:Boolean=True; aOem:Boolean=False; aBuffTime:Integer=E140DefBufTime);
  destructor  Destroy; override;
 public
  property Reporter     : TE140_Reporter      write SetReporter;
  property Report       : LongString          write PrintReport;
  property IsOem        : Boolean             read GetIsOem write SetIsOem;
  property SharedBuff   : PE140_SHARED_BUFFER read GetSharedBuff;
  property SharedName   : LongString          read GetSharedName;
  property SerNum       : LongString          read GetSerNum write SetSerNum;
 public
  function    ReadSerNum:LongString;
  function    Open(aAdc:pADC_PARS_E140=nil):Boolean;
  function    Opened:Boolean;
  procedure   Close;
  function    Start(aAdc:pADC_PARS_E140=nil):Double;
  function    Started:Double;
  function    Stop:Double;
  function    Polling:Int64;
  function    DAC_SAMPLE(DacData : pSHORT; DacChannel : WORD) : BOOL;
  function    ENABLE_TTL_OUT(EnableTtlOut : Boolean) : Boolean;
  function    TTL_OUT(TtlOut : WORD) : Boolean;
  function    TTL_IN(TtlIn : pWORD) : Boolean;
  function    GetLastErrorInfo:LongString;
 end;

function NewE140(aShare:LongString; aReporter:TE140_Reporter=nil; aAdc:pADC_PARS_E140=nil; aOpen:Boolean=True; aOem:Boolean=False; aBuffTime:Integer=E140DefBufTime):TE140;
procedure Kill(var TheObject:TE140); overload;

function TestE140AdcPars(nChan:Integer=4; nRange:Integer=ADC_INPUT_RANGE_2500mV_E140; Freq:Double=100.0):pADC_PARS_E140;

implementation

function TestE140AdcPars(nChan:Integer; nRange:Integer; Freq:Double):pADC_PARS_E140;
const Adc : ADC_PARS_E140 = ();
var i : Integer;
begin
 Result:=@Adc;
 with Adc do begin
  ClkSource        := INT_ADC_CLOCK_E140;
  EnableClkOutput  := ADC_CLOCK_TRANS_DISABLED_E140;
  InputMode        := NO_SYNC_E140;
  SynchroAdType    := 0;
  SynchroAdMode    := 0;
  SynchroAdChannel := 0;
  SynchroAdPorog   := 0;
  ChannelsQuantity := nChan;
  for i:=0 to ChannelsQuantity-1 do
  ControlTable[i]  := (i and 31) or (nRange shl $6);
  AdcRate          := Freq;
  InterKadrDelay   := 0;
  KadrRate         := 0;
 end; 
end;

procedure TE140.SetReporter(aReporter:TE140_Reporter);
begin
 if Assigned(Self) then myReporter:=aReporter;
end;

procedure TE140.PrintReport(const msg:LongString);
var
 oem : LongString;
begin
 if Assigned(Self) then
 if Assigned(myReporter) then
 try
  if myIsOem then begin
   SetLength(oem,Length(msg));
   Windows.CharToOem(PChar(msg),PChar(oem));
   myReporter(oem);
   Finalize(oem);
  end else begin
   myReporter(msg);
  end;
 except
  on E:Exception do LockedInc(StdIoErrorCount);
 end;
end;

function TE140.GetIsOem:Boolean;
begin
 if Assigned(Self)
 then Result:=myIsOem
 else Result:=False;
end;

procedure TE140.SetIsOem(aOem:Boolean);
begin
 if Assigned(Self) then myIsOem:=aOem;
end;

function TE140.GetSharedBuff:PE140_SHARED_BUFFER;
begin
 if Assigned(Self)
 then Result:=myShared
 else Result:=nil;
end;

function TE140.GetSharedName:LongString;
begin
 if Assigned(Self)
 then Result:=mySharm.Name
 else Result:='';
end;

function TE140.GetSerNum:LongString;
begin
 if Assigned(Self)
 then Result:=mySerNum
 else Result:='';
end;

procedure TE140.SetSerNum(const aSerNum:LongString);
begin
 if Assigned(Self) then mySerNum:=Trim(aSerNum);
end;

function TE140.ReadSerNum:LongString;
var Description:MODULE_DESCRIPTION_E140;
begin
 Result:='';
 if Assigned(Self) then
 try
  if Assigned(myApi) then
  if myApi.GET_MODULE_DESCRIPTION(@Description) then
  Result:=GetE140SerialNumber(Description);
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure TE140.Success(const s:LongString);
begin
 Report:='@E140.OnSuccess='+s;
end;
 
procedure TE140.Failure(const s:LongString);
begin
 Raise E140_Failure.Create(s);
end; 

procedure TE140.BugReport(const E:Exception; const msg:LongString);
begin
 if Assigned(E) then begin
  if Length(msg)>0
  then Report:=Format('@E140.OnException=%s,%s,"%s"',[E.ClassName,E.Message,msg])
  else Report:=Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]);
 end;
 LockedInc(StdIoErrorCount);
end;

constructor TE140.Create(aShare:LongString; aReporter:TE140_Reporter; aAdc:pADC_PARS_E140;
                         aOpen:Boolean; aOem:Boolean; aBuffTime:Integer);
begin
 Inherited Create;
 IsOem:=aOem;
 Reporter:=aReporter;
 myBuffTime:=aBuffTime;
 mySharm:=NewSharedMemory(aShare,Sizeof(myShared^));
 mySharm.Master:=mySharm;
 myShared:=mySharm[0];
 if aOpen then Open(aAdc);
end;

destructor TE140.Destroy;
begin
 if Started>0 then Stop;
 if Opened then Close;
 Kill(mySharm);
 Inherited Destroy;
end;

procedure TE140.PrintDescr(const aDescr:MODULE_DESCRIPTION_E140);
begin
 if Assigned(Self) then
 try
  Report:=GetE140Description(aDescr,'@E140.OnSuccess= ');
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure TE140.PrintADC(const aAdc:ADC_PARS_E140);
begin
 if Assigned(Self) then
 try
  Report:=GetE140AdcPars(aAdc,'@E140.OnSuccess= ');
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.FindBufferLength(aBuffTime:Integer; const aAdc:ADC_PARS_E140):Integer;
var
 Quantity, AdjustFactor, MaxBuffLen : Integer; BuffRate : Double;
begin
 Result:=0;
 if Assigned(Self) then begin
  if aBuffTime<=0 then Result:=E140DefBuffLen else with aAdc do begin
   MaxBuffLen:=SizeOf(myShared.Buffer) div SizeOf(myShared.Buffer[0]);
   BuffRate:=Max(AdcRate,KadrRate*ChannelsQuantity);
   AdjustFactor:=Max(ChannelsQuantity,1);
   while (AdjustFactor mod 2) = 0 do
   AdjustFactor:=AdjustFactor div 2;
   MaxBuffLen:=MaxBuffLen div E140NumBuffers;
   Quantity:=E140BuffAdjust*AdjustFactor;
   MaxBuffLen:=MaxBuffLen - Quantity;
   Result:=Round(aBuffTime*BuffRate);
   Result:=Min(MaxBuffLen,Result);
   Result:=Max(E140MinBuffLen,Result);
   Result:=AdjustBufferSize(Result,Quantity);
   if Result*E140NumBuffers*SizeOf(myShared.Buffer[0])>SizeOf(myShared.Buffer)
   then Failure(Format('Invalid buffer length %d',[Result]));
  end;
 end;
end;

procedure TE140.ClearIoRequests(IoCancel:Boolean);
var
 i : Integer;
begin
 if Assigned(Self) then
 try
  // 
  // Cancel asynchronous I/O request queue
  //
  if IoCancel then
  if myHand<>0 then
  if Windows.CancelIO(myHand)
  then Success('CancellIo');
  //
  // Reset all events and clear all myRequest data
  //
  for i:=0 to E140NumBuffers-1 do begin
   if myRequest.ReadOv[i].hEvent<>0 then begin
    if Windows.ResetEvent(myRequest.ReadOv[i].hEvent)
    then Success(Format('ResetEvent %d',[i]));
    if Windows.CloseHandle(myRequest.ReadOv[i].hEvent)
    then Success(Format('CloseHandle %d',[i]));
    myRequest.ReadOv[i].hEvent:=0;
   end;
   ZeroMemory(@myRequest.ReadOv[i],SizeOf(myRequest.ReadOv[i]));
   ZeroMemory(@myRequest.IoReq[i],SizeOf(myRequest.IoReq[i]));
   myRequest.Offset[i]:=0;
   myRequest.Index[i]:=0;
  end;
  myRequest.Number:=0;
  myRequest.Blocks:=0;
  myRequest.Samples:=0;
  myShared.StartTime:=0;
 except
  on E:Exception do BugReport(E);
 end;
end;

const
 IsClearHeader = False;
 IsClearBuffer = True;

procedure TE140.Close;
begin
 if Assigned(Self) then
 try
  if Started>0 then Stop;
  if Assigned(myApi) then begin
   myApi.STOP_ADC;
   myApi.CloseLDevice;
   myApi.ReleaseLInstance;
   myApi:=nil;
  end;
  myHand:=0;
  myName:='';
  myDll:=0;
  myUsb:=0;
  ClearIoRequests(False);
  if Assigned(myShared) then begin
   if IsClearHeader then ZeroMemory(@myShared.Header,SizeOf(myShared.Header));
   if IsClearBuffer then ZeroMemory(@myShared.Buffer,SizeOf(myShared.Buffer));
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.Open(aAdc:pADC_PARS_E140):Boolean;
var
 i,n  : Integer; 
 Buff : array[0..255] of Char;
begin
 Result:=False;
 if Assigned(Self) then
 try
  //
  // Check if one already started or opened
  //
  if Started>0 then Stop;
  if Opened then Close;
  //
  // Check DLL version
  //
  myDll := Lusbapi.GetDllVersion;
  Success(Format('GetDllVersion = %d.%d',[myDll shr 16, myDll and $FFFF]));
  if myDll <> LusbApi.CURRENT_VERSION_LUSBAPI
  then Failure(Format('Required Dll %d.%d',[CURRENT_VERSION_LUSBAPI shr 16, CURRENT_VERSION_LUSBAPI and $FFFF]));
  //
  // Try to get module interface pointer
  //
  myApi := LusbApi.CreateLInstance('E140');
  if not Assigned(myApi)
  then Failure('CreateLInstance E140')
  else Success('CreateLInstance E140');
  //
  // Try to find module in 1st MAX_VIRTUAL_SLOTS_QUANTITY_LUSBAPI virtual slots
  // Open device with SerialNumber if one specified
  //
  n:=0;
  for i := 0 to Lusbapi.MAX_VIRTUAL_SLOTS_QUANTITY_LUSBAPI-1 do
  if myApi.OpenLDevice(i) then begin
   Success(Format('OpenLDevice = %d, %s',[i,ReadSerNum]));
   if Length(SerNum)=0 then Inc(n) else
   if SameText(SerNum,ReadSerNum) then Inc(n);
   if n>0 then Break else myApi.CloseLDevice;
  end;
  if n=0 then Failure('OpenLDevice');
  //
  // Get module handle if one found
  //
  myHand := myApi.GetModuleHandle;
  if (myHand=0) or (myHand=INVALID_HANDLE_VALUE)
  then Failure('GetModuleHandle')
  else Success('GetModuleHandle');
  //
  // Read module name in current virtual slot
  //
  if not myApi.GetModuleName(@Buff)
  then Failure('GetModuleName')
  else Success('GetModuleName = '+Buff);
  myName := StrPas(Buff);
  //
  // Check if this module is E140
  //
  if not SameText(myName,'E140') then Failure('Device is not E14-140!');
  //
  // Try to read USB bus speed
  //
  if not myApi.GetUsbSpeed(@myUsb)
  then Failure('GetUsbSpeed')
  else Success(Format('GetUsbSpeed = %d Mbit/s',[12+468*Ord(myUsb<>USB11_LUSBAPI)]));
  //
  // Device reset
  //
  if not myApi.Stop_Adc
  then Failure('Stop_Adc')
  else Success('Stop_Adc'); 
  //
  // Check shared memory
  //
  if not Assigned(myShared)
  then Failure('SharedMemory')
  else Success('SharedMemory = '+IntToStr(mySharm.Size));
  ZeroMemory(myShared,mySharm.Size);
  //
  // Read device driver parameters
  //
  if not myApi.GET_MODULE_DESCRIPTION(@myShared.Description)
  then Failure('GET_MODULE_DESCRIPTION')
  else Success('GET_MODULE_DESCRIPTION');
  PrintDescr(myShared.Description);
  //
  // Read current ADC settings
  //
  if not myApi.GET_ADC_PARS(@myShared.AdcInited)
  then Failure('GET_ADC_PARS')
  else Success('GET_ADC_PARS');
  PrintAdc(myShared.AdcInited);
  //
  // Get wanted ADC settings if one specified
  //
  if Assigned(aAdc)
  then myShared.AdcWanted:=aAdc^
  else myShared.AdcWanted:=myShared.AdcInited;
  //
  // Write this wanted ADC settings to device
  //
  if not myApi.SET_ADC_PARS(@myShared.AdcWanted)
  then Failure('SET_ADC_PARS')
  else Success('SET_ADC_PARS');
  PrintAdc(myShared.AdcWanted);
  //
  // Read actual (accepted) ADC settings
  //
  if not myApi.GET_ADC_PARS(@myShared.AdcActual)
  then Failure('GET_ADC_PARS')
  else Success('GET_ADC_PARS');
  PrintAdc(myShared.AdcActual);
  //
  // Clear asynchronous I/O requests
  //
  ClearIoRequests(False);
  //
  // Done
  //
  Result:=Opened;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Stop:Double;
begin
 Result:=Started;
 if Result>0 then
 try
  //
  // Check interface
  //
  if not Assigned(myApi)
  then Failure('STOP')
  else Success('STOP'); 
  //
  // Reset device
  //
  if not myApi.Stop_Adc
  then Failure('Stop_Adc')
  else Success('Stop_Adc');
  // 
  // Clear asynchronous I/O requests
  //
  ClearIoRequests(True);
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Start(aAdc:pADC_PARS_E140):Double;
var
 i    : Integer;
 Leng : Integer;
 Rate : Double;
begin
 Result:=0;
 if Opened then
 try
  //
  // Stop if one already started
  //
  if Started>0 then Stop;
  //
  // Check interface
  //
  if not Assigned(myApi)
  then Failure('START')
  else Success('START'); 
  //
  // Check shared memory
  //
  if not Assigned(myShared)
  then Failure('SharedMemory')
  else Success('SharedMemory = '+IntToStr(mySharm.Size));
  //
  // Reset device
  //
  if not myApi.Stop_Adc
  then Failure('Stop_Adc')
  else Success('Stop_Adc');
  //
  // Write wanted ADC settings if one specified
  //
  if Assigned(aAdc) then begin
   myShared.AdcWanted:=aAdc^;
   if not myApi.SET_ADC_PARS(@myShared.AdcWanted)
   then Failure('SET_ADC_PARS')
   else Success('SET_ADC_PARS');
   PrintAdc(myShared.AdcWanted);
   if not myApi.GET_ADC_PARS(@myShared.AdcActual)
   then Failure('GET_ADC_PARS')
   else Success('GET_ADC_PARS');
   PrintAdc(myShared.AdcActual);
  end;
  // 
  // Clear asynchronous I/O requests
  //
  ClearIoRequests(True);
  //
  // Calculate wanted buffer size
  // It's near BuffTime ms of data taking
  //
  Leng:=FindBufferLength(myBuffTime,myShared.AdcActual);
  Success(Format('BuffTime = %d',[myBuffTime]));
  Success(Format('BuffLeng = %d',[Leng]));
  //
  // Prepare asynchronous I/O buffers
  //
  for i:=0 to E140NumBuffers-1 do begin
   // Initialize OVERLAPPED, asynchronous I/O event
   ZeroMemory(@myRequest.ReadOv[i],SizeOf(myRequest.ReadOv[i]));
   myRequest.ReadOv[i].hEvent := CreateEvent(nil, FALSE , FALSE, nil);
   // Prepare IoReq struct for ReadData requests
   ZeroMemory(@myRequest.IoReq[i],SizeOf(myRequest.IoReq[i]));
   myRequest.IoReq[i].Buffer := @myShared.Buffer[i*Leng];
   myRequest.IoReq[i].NumberOfWordsToPass := Leng;
   myRequest.IoReq[i].NumberOfWordsPassed := 0;
   myRequest.IoReq[i].Overlapped := @myRequest.ReadOv[i];
   Rate:=Max(myShared.AdcActual.AdcRate,myShared.AdcActual.KadrRate*myShared.AdcActual.ChannelsQuantity);
   myRequest.IoReq[i].TimeOut := Round(Int(Leng/Rate)) + Max(E140DefTimeout,myBuffTime);
   myRequest.Offset[i]:=SizeOf(myShared.Header)+i*Leng*SizeOf(myShared.Buffer[0]);
   myRequest.Index[i]:=i*Leng;
  end;
  // 
  // Start first asynchronous I/O request queue
  //
  myRequest.Number:=0;
  for i:=0 to E140NumBuffers-1 do
  if not myApi.ReadData(@myRequest.IoReq[i])
  then Failure(Format('ReadData %d',[i]))
  else Success(Format('ReadData %d',[i]));
  //
  // Now ready to start data acquisition
  //
  if not myApi.START_ADC
  then Failure('START_ADC')
  else Success('START_ADC');
  myShared.StartTime:=MSecNow;
  myRequest.Samples:=0;
  myRequest.Blocks:=0;
  Result:=Started;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.Started:Double;
begin
 if Assigned(Self) and Assigned(myShared)
 then Result:=myShared.StartTime
 else Result:=0;
end;

function TE140.Opened:Boolean;
begin
 if Assigned(Self)
 then Result:=Assigned(myApi)
 else Result:=False;
end;

function TE140.Polling:Int64;
var
 nBytes : DWORD;
 nWords : DWORD;
begin
 Result:=0;
 if Assigned(Self) then
 if Assigned(myApi) then
 if Assigned(myShared) then
 try
  if Started>0 then begin
   Result:=myRequest.Blocks;
   if GetOverlappedResult(myHand,myRequest.IoReq[myRequest.Number].Overlapped^,nBytes,False) then begin
    nWords:=nBytes div SizeOf(myShared.Buffer[0]);
    if nBytes mod SizeOf(myShared.Buffer[0]) <> 0
    then Failure(Format('Invalid nBytes = %d',[nBytes]));
    ProcessBuffer(nWords,myRequest.Number,myRequest.Samples,myRequest.Blocks);
    Inc(myRequest.Blocks); 
    Result:=myRequest.Blocks;
    Inc(myRequest.Samples,nWords);
    if not myApi.ReadData(@myRequest.IoReq[myRequest.Number])
    then Failure(Format('ReadData %d',[myRequest.Number]))
    else Success(Format('ReadData %d',[myRequest.Number]));
    myRequest.Number:=(myRequest.Number+1) mod E140NumBuffers;
   end else
   if GetLastError <> ERROR_IO_INCOMPLETE then begin
    Failure(Format('GetOverlappedResult=%d',[GetLastError]));
   end;
  end;
 except
  on E:Exception do begin
   BugReport(E,GetLastErrorInfo);
   Close;
  end;
 end;
end;

function TE140.ProcessBuffer(nWords:DWORD; nRequest:Integer; nSamples:Int64; nBlocks:Int64):DWORD;
begin
 Result:=0;
 if Assigned(Self) then
 try
  Report:=Format('@E140.OnReadout=%s,%d,%d,%d,%d,%d,%d',
                [mySharm.Name, nWords, myRequest.Offset[nRequest], myRequest.Index[nRequest],
                 nRequest, nBlocks, nSamples]);
  Result:=nWords;
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.DAC_SAMPLE(DacData : pSHORT; DacChannel : WORD) : BOOL;
begin
 Result:=False;
 if Opened then
 try
  Result:=myApi.DAC_SAMPLE(DacData, DacChannel);
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.ENABLE_TTL_OUT(EnableTtlOut : Boolean) : Boolean;
begin
 Result:=False;
 if Opened then
 try
  Result:=myApi.ENABLE_TTL_OUT(EnableTtlOut);
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.TTL_OUT(TtlOut : WORD) : Boolean;
begin
 Result:=False;
 if Opened then
 try
  Result:=myApi.TTL_OUT(TtlOut);
 except
  on E:Exception do BugReport(E);
 end;
end;

function TE140.TTL_IN(TtlIn : pWORD) : Boolean;
begin
 Result:=False;
 if Opened then
 try
  Result:=myApi.TTL_IN(TtlIn);
 except
  on E:Exception do BugReport(E);
 end;
end;
 
function TE140.GetLastErrorInfo:LongString;
begin
 Result:='E140 error: NIL reference';
 if Assigned(Self) then
 if Assigned(myShared) then
 try
  Result:='E140 error: NIL interface';
  if Assigned(myApi) then
  if myApi.GetLastErrorInfo(@myShared.LastError)
  then Result:=Format('E140 error %d: %s; WIN32 error %d: %s',
                     [myShared.LastError.ErrorNumber,PChar(@myShared.LastError.ErrorString),
                      Windows.GetLastError,SysErrorMessage(Windows.GetLastError)])
  else Result:=Format('E140 error %d: %s; WIN32 error %d: %s',
                     [0,'UNKNOWN',Windows.GetLastError,SysErrorMessage(Windows.GetLastError)]);
 except
  on E:Exception do Result:=E.Message;
 end;
end;

function NewE140(aShare:LongString; aReporter:TE140_Reporter; aAdc:pADC_PARS_E140; aOpen:Boolean; aOem:Boolean; aBuffTime:Integer):TE140;
begin
 Result:=nil;
 try
  Result:=TE140.Create(aShare, aReporter, aAdc, aOpen, aOem, aBuffTime);
  if not Assigned(Result.myShared) then Raise E140_FAILURE.Create('Shared memory fails!');
 except
  on E:Exception do begin
   Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
   Kill(Result);
  end;
 end; 
end;

procedure Kill(var TheObject:TE140); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do Echo(Format('@E140.OnException=%s,%s',[E.ClassName,E.Message]));
 end; 
end;

end.
