////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWDAQ.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Web server for &WebSrv DAQ device - CGI agent to  produce Web pages.       //
// CGI script to work with CRW-DAQ only.                                      //
// 1)Check REQUEST_METHOD to be GET/POST.                                     //
// 2)Read WEB4DAQ_PIPENAME,WEB4DAQ_FIFOSIZE,WEB4DAQ_TIMEOUT from environment  //
// 3)Create named pipe to interact with CRW-DAQ.                              //
// 4)Write CGI.RequestAsText to pipe. Use #0 as message terminator.           //
// 5)Wait reply from pipe or timeout. Use #0 as message terminator.           //
// 6)Check presence of HTTP and HTML headers.                                 //
// 7)Write reply to StdOut for Web server.                                    //
// 8)Done.                                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20061006 - Creation                                                        //
// 20061007 - First tested version                                            //
// 20061122 - Modified, use GetEnv instead of INI file to transfer parameters //
// 20240315 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

program websrv; // Web server, CGI agent to produce web pages in crwdaq

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 interfaces, lclintf, forms,
 _crw_alloc, _crw_cmdargs, _crw_environ, _crw_rtc, _crw_str, _crw_fio,
 _crw_fifo, _crw_ascio, _crw_az, _crw_proc, _crw_pipe,
 _crw_cgi;

 {
 HTTP page on error.
 }
 procedure Trouble(code:Integer; const msg:LongString);
 begin
  HTTP_ShowErrorPage('CGI script failure: '+msg);
  ExitCode:=code;
 end;
 {
 Check HTTP headers.
 Headers must delimited by empty line, i.e. EOL,EOL.
 Headers should contain "Content-Type:" or "Location:" items.
 }
 function GoodReply(const Reply:LongString):Boolean;
 var head:LongString; p:Integer;
 begin
  Result:=False;
  p:=Pos(#13#10#13#10,Reply);              // Find 1st empty line,
  if (p=0) then p:=Pos(#10#10,Reply);      // i.e. end of HTTP header.
  if (p=0) then p:=Pos(#13#13,Reply);      // Also test LF,LF and CR,CR.
  if (p=0) then Exit;                      // Empty line is not found.
  head:=Copy(Reply,1,p-1);                 // Extract HTTP header.
  if (PosI('CONTENT-TYPE:',head)=0) then   // Check Content-Type & Location.
  if (PosI('LOCATION:',head)=0) then Exit; // One of this items should present.
  Result:=True;
 end;
 {
 Declare variables
 }
var
 Pipe:TPipe=nil;
 PipeRxFifo:LongString='';
 PipeName:LongString='';
 PipeHost:LongString='';
 TimeOut:Cardinal=1000;
 PipeFactor:Integer=2;
 FifoSize:Integer=16;
 i,p,code:Integer;
 Tick:QWord=0;
 {
 Main program
 }
procedure Main;
begin
 try
  try
   {
   1)Check reqiest method.
   }
   if IsSameText(Cgi.REQUEST_METHOD,'GET')
   or IsSameText(Cgi.REQUEST_METHOD,'POST')
   then begin
    {
    2)Read environment variables
    WEB4DAQ_PIPENAME=.\demo
    WEB4DAQ_FIFOSIZE=16
    WEB4DAQ_TIMEOUT=1000
    }
    PipeName:=GetEnv('WEB4DAQ_PIPENAME');
    Val(GetEnv('WEB4DAQ_FIFOSIZE'),i,code);
    if (code=0) and (i>0) then FifoSize:=i;
    Val(GetEnv('WEB4DAQ_TIMEOUT'),i,code);
    if (code=0) and (i>0) then TimeOut:=i;
    {
    3)Create named pipe to interact with CRW-DAQ.
    }
    if Length(PipeName)=0 then begin
     Trouble(1,'Could not find pipe.');
     Exit;
    end;
    p:=LastDelimiter(['\','/'],PipeName);
    if (p>0) then begin
     PipeHost:=Copy(PipeName,1,p-1);
     PipeName:=Copy(PipeName,p+1,MaxInt);
    end;
    FifoSize:=Min(1024*64,Max(4,FifoSize))*1024;
    Pipe:=TPipe.Create(PipeHost,PipeName,nil,4,tpNormal,FifoSize*PipeFactor*KiloByte,FifoSize*KiloByte,TimeOut);
    Pipe.ListenPeriod:=10;
    {
    4)Put HTTP request to pipe. Use #0 as terminator.
    }
    Pipe.TxFifo.PutText(Cgi.RequestAsText+#0);
    {
    5)Polling loop, waiting for reply from pipe.
      Use #0 terminator to detect end of message.
    }
    Tick:=GetTickCount64;
    while true do begin
     if Pipe.Connected then begin
      PipeRxFifo:=PipeRxFifo+Pipe.Recv;
     end else begin
      if (GetTickCount64>Tick+TimeOut) then begin
       Trouble(1,'Could not open pipe.');
       Exit;
      end;
     end;
     if (Pipe.RxFifo.Lost+Pipe.TxFifo.Lost>0) then begin
      Trouble(1,'Pipe I/O errors.');
      Exit;
     end;
     if (GetTickCount64>Tick+TimeOut) then begin
      Trouble(1,'Timeout, no reply.');
      Exit;
     end;
     p:=Pos(#0,PipeRxFifo);
     if (p>0) then begin
      Delete(PipeRxFifo,p,MaxInt);
      Break;
     end;
     Sleep(4);
    end;
    {
    6)Check if reply is good enough.
    }
    if IsEmptyStr(PipeRxFifo) then begin
     Trouble(1,'Empty reply.');
     Exit;
    end;
    if not GoodReply(PipeRxFifo) then begin
     Trouble(1,'Invalid reply structure.');
     Exit;
    end;
    {
    7)Send result to StdOut.
    }
    Cgi.OutText:=PipeRxFifo;
   end else begin
    Trouble(1,'HTTP method don''t supported: '+Cgi.REQUEST_METHOD);
    Exit;
   end;
  finally
   Pipe.Free;
  end;
 except
  on E:Exception do Trouble(1,'Unknown exception '+E.ClassName+'.');
 end;
end;

begin
 Main;
end.

//////////////
// END OF FILE
//////////////

