#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

################################################################
## Utility to generate UniHeater's bitmap and circuit files.  ##
################################################################

function fatal(){
 unix tooltip-notifier text "$scriptname: $2" preset stdError delay 60000;
 1>&2 echo -ne "\n$2\n\n";
 exit $1;
};
function call(){
 if [ $# -gt 0 ]; then
  eval "$@";
 fi;
};
function rem(){
 return 0;
};

################################
# Current script identification.
################################
declare startupdir="$(pwd -LP)";
declare scriptfile="${BASH_SOURCE[0]}";
declare scriptname="$(basename $scriptfile)";
declare scriptbase="$(basename $scriptfile .sh)";
declare scripthome="$(dirname  $scriptfile)";
declare scriptFILE="$(realpath $scriptfile)";
declare scriptHOME="$(dirname  $scriptFILE)";

declare -x SelfExt=".sh";
declare -x SelfBase="$scriptbase";
declare -x SelfFile="$scriptFILE";
declare -x SelfName="$scriptname";
declare -x HomeDir="$startupdir";
declare -x SelfDir="$scriptHOME";

declare -i errors=0;

function uniheater_make_crc(){
 call SetErrors 0;
 call SetErrorLevel 0;
 #ParseOptions
 case $1 in
  -h|--help)  call PrintHelp; exit 0; ;;
  --version)  call PrintVersion; exit 0; ;;
 *) ;;
 esac;
 #ParseArguments
 call RequiredParamStr UH_PrefixName $1                     || exit 1       ; ### Prefix string to identify UniHeater
 call RequiredParamInt UH_NumHeaters $2                     || exit 1       ; ### Number of Unified Heaters, 1..32
 call RequiredParamInt UH_TileNumber $3                     || exit 1       ; ### Tile number per Row, 1..UH_NumHeaters
 call RequiredParamStr UH_CrcFolder  $4                     || exit 1       ; ### Target Directory for CRC files
 call RequiredParamStr UH_BmpFolder  $5                     || exit 1       ; ### Target Directory for BMP files
 #SetAddonVariables
 call GetPrefixFile UH_PrefixFile $UH_PrefixName                             ; ### File name prefix
 call Define UH_PrefixFile $(printenv UH_PrefixName | tr '.' '_')            ; ### File name prefix
 #CheckRequiredEntities
 call RequiredAdminRights                                   || exit 1;
 call RequiredUnixUtils gm help                             || exit 1;
 call RequiredParentCrwDaq                                  || exit 1;
 call RequiredDefined UH_PrefixName                         || exit 1;
 call RequiredDefined UH_PrefixFile                         || exit 1;
 call RequiredDefined UH_NumHeaters                         || exit 1;
 call RequiredDefined UH_TileNumber                         || exit 1;
 call RequiredIntRange $UH_NumHeaters 1 32                  || exit 1;
 call RequiredIntRange $UH_TileNumber 1 $UH_NumHeaters      || exit 1;
 call RequiredDefined UH_CrcFolder                          || exit 1;
 call RequiredDefined UH_BmpFolder                          || exit 1;
 call RequiredFile "$UH_CrcFolder/"                         || exit 1;
 call RequiredFile "$UH_BmpFolder/"                         || exit 1;
 call RequiredFile "$SelfDir/uniheater_ctrl_sample.bmp"     || exit 1;
 call RequiredFile "$SelfDir/uniheater_settings.bmp"        || exit 1;
 if pushd "$HomeDir" >/dev/null 2>&1; then
  call UniHeaterCircuits;
 fi;
 popd >/dev/null 2>&1;
 call HasNoErrors || call ErrorPrintLn "$errors error(s) found in $SelfName.";
 return $?;
};

function PrintVersion(){
  echo "$SelfBase version 1.0";
};

function  PrintHelp(){
 call PrintVersion;
 echo "Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Utility to generate UniHeater's bitmaps and circuits.";
 echo "Usage:";
 echo " $SelfBase [Options] [Arguments]";
 echo "Options:";
 echo " -h,--help  - print help";
 echo " --version  - print version";
 echo "Arguments:";
 echo " UH_PrefixName - Prefix name to identify Uniheater";
 echo " UH_NumHeaters - Number of Unified Heaters, 1..32";
 echo " UH_TileNumber - Tile number = columns per row, 1..UH_NumHeaters";
 echo " UH_CrcFolder  - Target directory for CRC files, like ../circuits";
 echo " UH_BmpFolder  - Target directory for BMP files, like ../bitmaps";
 echo "Example:";
 echo " call $SelfBase DEMO 16 8 ../circuits ../bitmaps";
};

function UniHeaterCircuits(){
 call UniHeaterEnviron;
 call UniHeaterBitmaps;
 call UniHeaterTable $UH_NumHeaters $UH_TileNumber  > "$UH_CrcFolder/${UH_PrefixFile}_uniheater_ctrl.crc";
 call UniHeaterSettings                             > "$UH_CrcFolder/${UH_PrefixFile}_uniheater_settings.crc";
};

function UniHeaterEnviron(){
 export UH_HomeRef="~~/resource/daqsite/uniheater"  ; ### UniHeater home directory reference
 export UH_PREF="$UH_PrefixName.UNIHEATER"          ; ### UniHeater name prefix
 export UH_FILE="$(printenv UH_PREF | tr '.' '_')"  ; ### UniHeater file prefix
 export UH_PATH="$(printenv UH_PREF | tr '.' '/')"  ; ### UniHeater path prefix
};

function UniHeaterBitmaps(){
 call TileBitmaps "$SelfDir/uniheater_ctrl_sample.bmp" "$UH_BmpFolder/${UH_PrefixFile}_uniheater_ctrl.bmp" $UH_NumHeaters $UH_TileNumber || call IncErrors 1;
 call CopyFile "$SelfDir/uniheater_settings.bmp" "$UH_BmpFolder/${UH_PrefixFile}_uniheater_settings.bmp" || call IncErrors 1;
};

function TileBitmaps(){
 local src="$1"; shift; ### 1: Source bitmap
 local dst="$1"; shift; ### 2: Destination bitmap
 let "num=$1";   shift; ### 3: Number of bitmaps to copy
 let "tile=$1";  shift; ### 4: Number of bitmaps per row
 local list="$src"; for i in $(seq 2 $num); do list="$list $src"; done;
 if [ -e $dst ]; then call Exec rm -f $dst; fi;
 call Exec unix gm montage -background silver -tile $tile -geometry +0+0 $list -colors 16 $dst;
};

function CopyFile(){
 if [ -z "$1" ]; then return; fi;
 if [ -z "$2" ]; then return; fi;
 if [ -e "$1" ]; then call Exec cp -fvu "$1" "$2"; fi;
};

function UniHeaterTable(){
 call GeneralMap $1;
 for n in $(seq 1 $1); do call UniHeater $n $2; done;
};

function GeneralMap(){
 echo "[Circuit]";
 echo "GeneralMap = ../bitmaps/${UH_PrefixFile}_uniheater_ctrl.bmp";
 echo "Hint = UniHeater: Панель Контроля Нагревателей (Control Panel of Unified Heaters).";
 echo "[]";
 echo "";
 echo "[ConfigFileList] ; Include the Painter GUI Library";
 echo "ConfigFile = ~~/resource/daqsite/default/painter.crc";
 echo "[]";
 echo "";
};

function UniHeater(){
 if [ -z "$1" ]; then return; fi;
 if [ -z "$2" ]; then return; fi;
 let "dx=( ( $1 - 1 ) % $2 ) * 125";
 let "dy=( ( $1 - 1 ) / $2 ) * 119";
 echo ";**********************";
 echo ";  Universal Heater $1";
 echo ";**********************";
 echo "";
 let "x=$dx+30";
 let "y=$dy+6";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PROP_$1";
 echo "[$UH_PREF.PROP_$1]";
 echo "Pos = $x, $y";
 echo "LED = 7, 0, 0, %7s, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp";
 echo "Hint = UniHeater: Наименование нагревателя $1 (heater $1 name).";
 echo "[]";
 echo "";
 let "x=$dx+4";
 let "y=$dy+30";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.BT_$1";
 echo "[$UH_PREF.BT_$1]";
 echo "Pos = $x, $y";
 echo "Painter(v) = (glBarColor=clSilver*ne(v,3)*ne(v,6)+clRed*eq(v,3)+clYellow*eq(v,6))";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBar]";
 echo "Painter(v) = (bt_$1=v)+(v=ne(v%4,0))+(glButtonBit=0)+(glButtonBevel=3)+(glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Painter(v) = (v=bt_$1)+(glUniHeaterBevel=3)+(glUniHeaterColor=clBlack*ne(v,2)+clMaroon*eq(v,2))";
 echo "Painter(v) = [Painter(v).DaqLib.UniHeater]";
 echo "Painter(v) = (glCrossBevel=3)+(glCrossWidth=3*ge(v,4))+(glCrossColor=clRed)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleCross]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_37_37_4_silver.bmp";
 echo "Hint = UniHeater: Кнопка включения нагревателя $1 (heater $1 ON/OFF button).";
 echo "[]";
 echo "[Circuit.StartupScript]";
 echo "bt_$1 = 0";
 echo "[]";
 echo "";
 let "x=$dx+59";
 let "y=$dy+31";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TNUM_$1";
 echo "[$UH_PREF.TNUM_$1]";
 echo "Pos = $x, $y";
 echo "LED = 2, 0, 0, %2.0f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_23_20_4_white.bmp";
 echo "Hint = UniHeater: Номер температурного канала нагревателя $1 (temperature sensor number).";
 echo "[]";
 echo "";
 let "x=$dx+44";
 let "y=$dy+52";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TNAME_$1";
 echo "[$UH_PREF.TNAME_$1]";
 echo "Pos = $x, $y";
 echo "LED = 4, 0, 0, *, Name:PT_Sans\Size:10\Style:[Normal]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_38_14_4_silver.bmp";
 echo "Hint = UniHeater: Имя температурного канала нагревателя $1 (temperature sensor name).";
 echo "[]";
 echo "";
 let "x=$dx+30";
 let "y=$dy+71";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TGOAL_$1";
 echo "[$UH_PREF.TGOAL_$1]";
 echo "Pos = $x, $y";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Целевая температура нагревателя $1, °C (goal temperature).";
 echo "[]";
 echo "";
 let "x=$dx+30";
 let "y=$dy+93";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TREAL_$1";
 echo "[$UH_PREF.TREAL_$1]";
 echo "Pos = $x, $y";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_aqua.bmp";
 echo "Hint = UniHeater: Реальная температура нагревателя $1, °C (actual temperature).";
 echo "[]";
 echo "";
 let "x=$dx+104";
 let "y=$dy+31";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.GATE_$1";
 echo "[$UH_PREF.GATE_$1]";
 echo "Pos = $x, $y";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clBlack)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_16_16_4_gray.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_16_16_4_maroon.bmp";
 echo "Hint = UniHeater: Индикатор сигнала на включение нагревателя $1 (heater gate indicator).";
 echo "[]";
 echo "";
 let "x=$dx+104";
 let "y=$dy+50";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.BLK_$1";
 echo "[$UH_PREF.BLK_$1]";
 echo "Pos = $x, $y";
 echo "TagEval(v) = ne(v,0)+and(v,ge(bt_$1,4))";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clBlack)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_16_16_4_gray.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_16_16_4_lime.bmp";
 echo "Tag#3 = 2, ~~/resource/daqsite/stdlib/bitmaps/barbmp_16_16_4_red.bmp";
 echo "Hint = UniHeater: Индикатор сигнала блокировки нагревателя $1 (heater blocking indicator).";
 echo "[]";
 echo "";
 let "x=$dx+6";
 let "y=$dy+7";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.HENABL_$1";
 echo "[$UH_PREF.HENABL_$1]";
 echo "Pos = $x, $y";
 echo "LED = 0, 0, 0, %s, Name:PT_Sans_Narrow\Size:10\Style:[Normal]";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clBlack)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_18_18_4_gray.bmp Off";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_18_18_4_lime.bmp On";
 echo "Hint = UniHeater: Разрешение работы нагревателя $1 (heater enable indicator).";
 echo "[]";
 echo "";
};

function UniHeaterSettings(){
 echo "[Circuit]";
 echo "GeneralMap = ../bitmaps/${UH_PrefixFile}_uniheater_settings.bmp";
 echo "Hint = UniHeater: Настройка нагревателей (heater settings).";
 echo "[]";
 echo "";
 echo "[ConfigFileList] ; Include the Painter GUI Library";
 echo "ConfigFile = ~~/resource/daqsite/default/painter.crc";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.HENABL_0";
 echo "[$UH_PREF.HENABL_0]";
 echo "Pos = 10, 15";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=0)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_18_18_4_silver.bmp";
 echo "Hint = UniHeater: Разрешение работы нагревателя (heater enable).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.HNAME_0";
 echo "[$UH_PREF.HNAME_0]";
 echo "Pos = 169, 15";
 echo "LED = 7, 0, 0, %7s, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp";
 echo "Hint = UniHeater: Наименование нагревателя (heater name).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PENABL_0";
 echo "[$UH_PREF.PENABL_0]";
 echo "Pos = 10, 55";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=0)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_18_18_4_silver.bmp";
 echo "Hint = UniHeater: Разрешение блокировки по давлению (enable blocking by pressure).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PBLK_0";
 echo "[$UH_PREF.PBLK_0]";
 echo "Pos = 36, 76";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Верхнее давление блокировки (pressure level of blocking).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PREAL_0";
 echo "[$UH_PREF.PREAL_0]";
 echo "Pos = 36, 98";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_aqua.bmp";
 echo "Hint = UniHeater: Реальное измеренное давление (actual pressure).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PNAME_0";
 echo "[$UH_PREF.PNAME_0]";
 echo "Pos = 36, 120";
 echo "LED = 5, 0, 0, *, Name:PT_Sans\Size:12\Style:[Normal]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_64_20_4_silver.bmp";
 echo "Hint = UniHeater: Имя датчика давления блокировки (pressure sensor name).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PNUM_0";
 echo "[$UH_PREF.PNUM_0]";
 echo "Pos = 102, 120";
 echo "LED = 2, 0, 0, *, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_23_20_4_white.bmp";
 echo "Hint = UniHeater: Номер датчика давления блокировки (pressure sensor number).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TENABL_0";
 echo "[$UH_PREF.TENABL_0]";
 echo "Pos = 10, 145";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=0)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_18_18_4_silver.bmp";
 echo "Hint = UniHeater: Разрешение блокировки по температуре (enable blocking by temperature).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TBLK_0";
 echo "[$UH_PREF.TBLK_0]";
 echo "Pos = 36, 166";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Верхняя тепература блокировки, °C (temperature level of blocking).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TREAL_0";
 echo "[$UH_PREF.TREAL_0]";
 echo "Pos = 36, 188";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_aqua.bmp";
 echo "Hint = UniHeater: Реальная (текущая) температура, °C (actual temperature).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TNAME_0";
 echo "[$UH_PREF.TNAME_0]";
 echo "Pos = 36, 210";
 echo "LED = 4, 0, 0, *, Name:PT_Sans\Size:12\Style:[Normal]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_64_20_4_silver.bmp";
 echo "Hint = UniHeater: Имя датчика температуры блокировки (temperature sensor name).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TNUM_0";
 echo "[$UH_PREF.TNUM_0]";
 echo "Pos = 102, 210";
 echo "LED = 2, 0, 0, *, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_23_20_4_white.bmp";
 echo "Hint = UniHeater: Номер датчика температуры блокировки (temperature sensor number).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.WENABL_0";
 echo "[$UH_PREF.WENABL_0]";
 echo "Pos = 10, 235";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=0)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_18_18_4_silver.bmp";
 echo "Hint = UniHeater: Разрешение блокировки по таймеру (enable blocking by watchdog timer).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.WDT_0";
 echo "[$UH_PREF.WDT_0]";
 echo "Pos = 69, 256";
 echo "LED = 4, 1, 0, %4.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_56_20_4_white.bmp";
 echo "Hint = UniHeater: Сторожевой таймер, сек (watchdog timer, sec).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.MODE_0";
 echo "[$UH_PREF.MODE_0]";
 echo "Pos = 142, 59";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_22_20_4_silver.bmp \"++\"";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_22_20_4_silver.bmp --";
 echo "Hint = UniHeater: Режим работы нагревателя (heater operation mode).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.MODE_N";
 echo "[$UH_PREF.MODE_N]";
 echo "Pos = 165, 59";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp Нагрев";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp Охлаждение";
 echo "Hint = UniHeater: Режим работы нагревателя (heater operation mode).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.CONF_0_0";
 echo "[$UH_PREF.CONF_0_0]";
 echo "Pos = 285, 225";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=0)+(glCheckBoxColor=clWhite)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_19_19_4_silver.bmp";
 echo "Hint = UniHeater:  Флаг запроса на ВКЛЮЧЕНИЕ нагревателя (flag to confirm heater ON).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.CONF_0_1";
 echo "[$UH_PREF.CONF_0_1]";
 echo "Pos = 285, 245";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=1)+(glCheckBoxColor=clWhite)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_19_19_4_silver.bmp";
 echo "Hint = UniHeater:  Флаг запроса на ВЫКЛЮЧЕНИЕ нагревателя (flag to confirm heater OFF).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.CONF_0_2";
 echo "[$UH_PREF.CONF_0_2]";
 echo "Pos = 285, 265";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=2)+(glCheckBoxColor=clWhite)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_19_19_4_silver.bmp";
 echo "Hint = UniHeater:  Флаг запроса на ЧТЕНИЕ/ЗАПИСЬ INI файла (flag to confirm read/write INI file).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TGOAL_0";
 echo "[$UH_PREF.TGOAL_0]";
 echo "Pos = 165, 110";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Целевая тепература стабилизации, °C (goal temperature).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.DT2_0";
 echo "[$UH_PREF.DT2_0]";
 echo "Pos = 165, 132";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: dT2 - Верхний порог температуры, °C (upper temperature edge).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.DT1_0";
 echo "[$UH_PREF.DT1_0]";
 echo "Pos = 165, 154";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: dT1 - Нижний порог температуры, °C (lower temperature edge).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.Q1_0";
 echo "[$UH_PREF.Q1_0]";
 echo "Pos = 165, 212";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Q1 - мощность начального нагрева (power of initial heat).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.Q2_0";
 echo "[$UH_PREF.Q2_0]";
 echo "Pos = 165, 234";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Q2 - мощность медленного нагрева dT1 (power of slow heat).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.Q3_0";
 echo "[$UH_PREF.Q3_0]";
 echo "Pos = 165, 256";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Q3 - мощность легкого подогрева dT2 (power of light heat).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.CLOCK";
 echo "[$UH_PREF.CLOCK]";
 echo "Pos = 780, 80";
 echo "LED = 19, 0, 0, %s, Name:PT_Mono\Size:9\Style:[Bold]";
 echo "Painter(v) = (glBarColor=linkedtagcolor()+clRed*eq(linkedtagcolor(),0))";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBar]";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_149_16_4_silver.bmp";
 echo "Hint = UniHeater: Дата и Время Сервера (Host Date and Time).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SERVID";
 echo "[$UH_PREF.SERVID]";
 echo "Pos = 780, 97";
 echo "LED = 19, 0, 0, %s, Name:PT_Sans_Narrow\Size:10\Style:[Normal]";
 echo "Painter(v) = (glBarColor=linkedtagcolor()+clRed*eq(linkedtagcolor(),0))";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBar]";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_149_16_4_silver.bmp";
 echo "Hint = UniHeater: Идентификатор Сервера (Server Identifier).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SOUNDS";
 echo "[$UH_PREF.SOUNDS]";
 echo "Pos = 10, 297";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCheckBoxBit=0)";
 echo "Painter(v) = [Painter(v).GuiLib.CheckBox]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_20_20_4_silver.bmp";
 echo "Hint = UniHeater: Разрешение звуковых эффектов (enable sound).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.ACCEPT";
 echo "[$UH_PREF.ACCEPT]";
 echo "Pos = 70, 295";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glButtonBevel=2)+(glCmdOkLineColor=clBlack)+(glCmdOkLineWidth=1)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.Ok]";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)+(glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_91_25_4_silver.bmp \"   Принять\"";
 echo "Hint = UniHeater: Применить внесенные изменения (apply changes).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.CANCEL";
 echo "[$UH_PREF.CANCEL]";
 echo "Pos = 168, 295";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glButtonBevel=2)+(glCmdCancelLineColor=clBlack)+(glCmdCancelLineWidth=1)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.Cancel]";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)+(glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_91_25_4_silver.bmp \"   Отмена\"";
 echo "Hint = UniHeater: Отменить внесенные изменения (cancel changes).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SAVEINI";
 echo "[$UH_PREF.SAVEINI]";
 echo "Pos = 70, 340";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCmdHomeColor0=clYellow)+(glCmdHomeColor1=clOlive)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.SaveIni]";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)+(glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_91_25_4_silver.bmp \"   Запись\"";
 echo "Hint = UniHeater: Записать параметры в INI файл (save INI file).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.LOADINI";
 echo "[$UH_PREF.LOADINI]";
 echo "Pos = 168, 340";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glCmdHomeColor0=clYellow)+(glCmdHomeColor1=clOlive)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.LoadIni]";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)+(glButtonMoveX=2)+(glButtonMoveY=2)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_91_25_4_silver.bmp \"   Чтение\"";
 echo "Hint = UniHeater: Прочитать параметры из INI файла (load INI file).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.MAKE.LM9";
 echo "[$UH_PREF.MAKE.LM9]";
 echo "Pos = 780, 10";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)";
 echo "Painter(v) = (glCmdSettingColor0=clYellow)+(glCmdSettingColor1=clOlive)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.Setting]";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_49_48_4_silver.bmp";
 echo "Hint = Конфигуратор UniHeater.";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.MANUAL";
 echo "[$UH_PREF.MANUAL]";
 echo "Pos = 830 10";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)";
 echo "Painter(v) = (glCmdHelpColor0=clAqua)+(glCmdHelpColor1=clTeal)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.Help]";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_49_48_4_silver.bmp";
 echo "Hint = UniHeater: Показать Справку (show Help).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.CLOSE";
 echo "[$UH_PREF.CLOSE]";
 echo "Pos = 880 10";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Size:10\Style:[Bold]";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=2)";
 echo "Painter(v) = [Painter(v).GuiLib.Cmd.Close]";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_49_48_4_silver.bmp";
 echo "Hint = UniHeater: Закрыть окно (Close window).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.PERIOD_0";
 echo "[$UH_PREF.PERIOD_0]";
 echo "Pos = 490, 25";
 echo "LED = 7, 0, 0, %7.0f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: Период меандра, миллисекунд (meander period, ms).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.QX_0";
 echo "[$UH_PREF.QX_0]";
 echo "Pos = 520, 230";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp";
 echo "Hint = UniHeater: QX - текущая мощность нагрева (current power of heat).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.QMEAN_0";
 echo "[$UH_PREF.QMEAN_0]";
 echo "Pos = 520, 252";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp";
 echo "Hint = UniHeater: QMEAN - средняя мощность нагрева (average power of heat).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TSTAT_0";
 echo "[$UH_PREF.TSTAT_0]";
 echo "Pos = 520, 274";
 echo "LED = 7, 0, 0, %7.0f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_white.bmp";
 echo "Hint = UniHeater: время накопления статистики, сек (time of statistics, sec).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TMEAN_0";
 echo "[$UH_PREF.TMEAN_0]";
 echo "Pos = 520, 296";
 echo "LED = 7, 1, 0, %7.1f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp";
 echo "Hint = UniHeater: TMEAN - средняя температура (average temperature).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.TRMSD_0";
 echo "[$UH_PREF.TRMSD_0]";
 echo "Pos = 520, 318";
 echo "LED = 7, 2, 0, %7.2f, Name:PT_Mono\Size:15\Style:[Bold]";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_89_20_4_silver.bmp";
 echo "Hint = UniHeater: TRMSD - разброс температуры (RMS defiation of temperature).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_0";
 echo "[$UH_PREF.SB_0_0]";
 echo "Pos = 820, 230";
 echo "TagEval(v) = isbit(v,0)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_lime.bmp";
 echo "Hint = UniHeater: SB.0 BT Состояние кнопки нагрева (StatusByte - Heater Button state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_1";
 echo "[$UH_PREF.SB_0_1]";
 echo "Pos = 820, 242";
 echo "TagEval(v) = isbit(v,1)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_maroon.bmp";
 echo "Hint = UniHeater: SB.1 GATE Состояние сигнала GATE (StatusByte - GATE signal).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_2";
 echo "[$UH_PREF.SB_0_2]";
 echo "Pos = 820, 254";
 echo "TagEval(v) = isbit(v,2)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.2 BLK Состояние общей блокировки (StatusByte - Blocking summary state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_3";
 echo "[$UH_PREF.SB_0_3]";
 echo "Pos = 820, 266";
 echo "TagEval(v) = isbit(v,3)+not(isbit(v,0))";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Tag#3 = 2, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_lime.bmp";
 echo "Hint = UniHeater: SB.3 BLK Сигнал блокировки (StatusByte - Blocking signal).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_4";
 echo "[$UH_PREF.SB_0_4]";
 echo "Pos = 820, 278";
 echo "TagEval(v) = isbit(v,4)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.4 EBLK Состояние внешней блокировки (StatusByte - External Blocking state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_5";
 echo "[$UH_PREF.SB_0_5]";
 echo "Pos = 820, 290";
 echo "TagEval(v) = isbit(v,5)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.5 ICTRL Состояние контроля датчика (StatusByte - gauge control state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_6";
 echo "[$UH_PREF.SB_0_6]";
 echo "Pos = 820, 302";
 echo "TagEval(v) = isbit(v,6)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.6 TWDT Состояние сторожевого таймера T (StatusByte - T WatchDogTimer state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_7";
 echo "[$UH_PREF.SB_0_7]";
 echo "Pos = 820, 314";
 echo "TagEval(v) = isbit(v,7)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.7 PWDT Состояние сторожевого таймера P (StatusByte - P WatchDogTimer state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_8";
 echo "[$UH_PREF.SB_0_8]";
 echo "Pos = 880, 230";
 echo "TagEval(v) = isbit(v,8)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.8 TBLK Состояние блокировки по T>TBLK (StatusByte - T>TBLK Blocking state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_9";
 echo "[$UH_PREF.SB_0_9]";
 echo "Pos = 880, 242";
 echo "TagEval(v) = isbit(v,9)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_red.bmp";
 echo "Hint = UniHeater: SB.9 PBLK Состояние блокировки по P>PBLK (StatusByte - P>PBLK Blocking state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_10";
 echo "[$UH_PREF.SB_0_10]";
 echo "Pos = 880, 254";
 echo "TagEval(v) = isbit(v,10)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_silver.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_maroon.bmp";
 echo "Hint = UniHeater: SB.10 QXON Состояние сигнала QX>0 (StatusByte - QX>0 signal state).";
 echo "[]";
 echo "";
 echo "[SensorList]";
 echo "Sensor = $UH_PREF.SB_0_11";
 echo "[$UH_PREF.SB_0_11]";
 echo "Pos = 880, 266";
 echo "TagEval(v) = isbit(v,11)";
 echo "Tag#1 = 0, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_lime.bmp";
 echo "Tag#2 = 1, ~~/resource/daqsite/stdlib/bitmaps/barbmp_10_10_4_yellow.bmp";
 echo "Hint = UniHeater: SB.11 NETW Состояние сетевой связи (StatusByte - state of Network connection).";
 echo "[]";
};

################################################################################
## Common library routines                                                    ##
################################################################################

function Define(){
 if [ -z "$1" ]; then return 0; fi; ### Environ Variable Name
 if [ -z "$2" ]; then return 0; fi; ### Environ Variable Value
 if [ -z "$(printenv $1)" ]; then
  export $1="$2";
 fi;
};

function GetPrefixFile(){
 if [ -z "$1" ]; then return; fi;
 if [ -z "$2" ]; then return; fi;
 export $1="$(echo "$2" | tr '[:upper:]' '[:lower:]')";
};

function RequiredParamStr(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then export $1="$2"; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName - missed string param $1. Call $SelfName --help for details.";
};

function RequiredParamInt(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then let "$1=$2"; export $1; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName : missed integer param $1. Call $SelfName --help for details.";
};

function RequiredAdminRights(){
 if unix sudoit -v >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: Access denied. Admin rights required.";
};

function RequiredUnixUtils(){
 if which unix >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: UnixUtils is not valid. Install UnixUtils first.";
};

function RequiredParentCrwDaq(){
 local pid="$CRW_DAQ_SYS_EXE_PID";
 local cmd="$CRW_DAQ_SYS_EXE_FILE";
 local exe="$(basename $cmd)";
 local lst="";
 if [ -n "$pid" ] && [ -n "$exe" ]; then
  lst="$(pgrep $exe | grep $pid)";
 fi;
 if [ -z "$lst" ]; then
  fatal 1 "Error: $SelfName: parent CRW-DAQ is not found. Run this script from CRW-DAQ.";
 fi;
};

function RequiredFile(){
 if [ -n "$1" ] && [ -e "$1" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: not found $1.";
};

function RequiredDefined(){
 if [ -n "$1" ] && [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: undefined variable $1. Call $SelfName --help for details.";
};

function RequiredIntRange(){
 if [ -n "$1" ] &&  [ -n "$2" ] &&  [ -n "$3" ] && [ $1 -ge $2 ] && [ $1 -le $3 ]; then return 0; fi;
 fatal 1 "Error: $SelfName: invalid value $1 (not in $2..$3 range). Call $SelfName --help for details.";
};

function SetErrors(){
 let "errors=$1";
};

function IncErrors(){
 let "errors+=$1";
};

function HasNoErrors(){
 if [ "$errors" = "0" ]; then return $?; fi;
 return 1;
};

function Exec(){
 echo "$@";
 call "$@";
};

function PrintLn(){
 echo "$@";
};

function ErrorPrint(){
 1>&2 echo -n "$@";
};

function ErrorPrintLn(){
 1>&2 echo "$@";
};

function SetErrorLevel(){
 return $1;
};

function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then
   rm -f "$1";
  fi;
  shift;
 done;
};


uniheater_make_crc "$@";

##############
## END OF FILE
##############
