#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

################################################################
## Utility to generate UniHeater's configuration.             ##
################################################################

function fatal(){
 unix tooltip-notifier text "$scriptname: $2" preset stdError delay 60000;
 1>&2 echo -ne "\n$2\n\n";
 exit $1;
};
function call(){
 if [ $# -gt 0 ]; then
  echo "$@";
  eval "$@";
 fi;
};
function rem(){
 return 0;
};

################################
# Current script identification.
################################
declare startupdir="$(pwd -LP)";
declare scriptfile="${BASH_SOURCE[0]}";
declare scriptname="$(basename $scriptfile)";
declare scriptbase="$(basename $scriptfile .sh)";
declare scripthome="$(dirname  $scriptfile)";
declare scriptFILE="$(realpath $scriptfile)";
declare scriptHOME="$(dirname  $scriptFILE)";

declare -x SelfExt=".sh";
declare -x SelfBase="$scriptbase";
declare -x SelfFile="$scriptFILE";
declare -x SelfName="$scriptname";
declare -x HomeDir="$startupdir";
declare -x SelfDir="$scriptHOME";

declare -i errors=0;


function uniheater_make(){
 call SetErrors 0;
 call SetErrorLevel 0;
 #ParseOptions
 case $1 in
  -h|--help)  call PrintHelp; exit 0; ;;
  --version)  call PrintVersion; exit 0; ;;
  --samples)  call CopySamples "$@"; exit 0; ;;
 *) ;;
 esac;
 #ParseArguments
 call RequiredParamStr UH_PrefixName $1                     || exit 1       ; ### Prefix string to identify UniHeater
 call RequiredParamInt UH_NumHeaters $2                     || exit 1       ; ### Number of Unified Heaters, 1..32
 call RequiredParamInt UH_TileNumber $3                     || exit 1       ; ### Tile number per Row, 1..UH_NumHeaters
 #SetAddonVariables
 call GetPrefixFile UH_PrefixFile $UH_PrefixName                            ; ### File name prefix
 call Define UH_PrefixFile $(printenv UH_PrefixName | tr '.' '_')           ; ### File name prefix
 declare -x UH_CfgFolder="$(realpath "$SelfDir/../config")"                 ; ### Target folder for CFG files
 declare -x UH_BmpFolder="$(realpath "$SelfDir/../bitmaps")"                ; ### Target folder for BMP files
 declare -x UH_CrcFolder="$(realpath "$SelfDir/../circuits")"               ; ### Target folder for CRC files
 call Define UH_HomeDir $CRW_DAQ_SYS_HOME_DIR/resource/daqsite/uniheater    ; ### Source directory of UniHeater
 #CheckRequiredEntities
 call RequiredUnixUtils                                     || exit 1;
 call RequiredAdminRights                                   || exit 1;
 call RequiredParentCrwDaq                                  || exit 1;
 call RequiredDefined UH_PrefixName                         || exit 1;
 call RequiredDefined UH_PrefixFile                         || exit 1;
 call RequiredDefined UH_NumHeaters                         || exit 1;
 call RequiredDefined UH_TileNumber                         || exit 1;
 call RequiredIntRange $UH_NumHeaters 1 32                  || exit 1;
 call RequiredIntRange $UH_TileNumber 1 $UH_NumHeaters      || exit 1;
 call RequiredDefined UH_CfgFolder                          || exit 1;
 call RequiredDefined UH_BmpFolder                          || exit 1;
 call RequiredDefined UH_CrcFolder                          || exit 1;
 call RequiredFile "$UH_CfgFolder/"                         || exit 1;
 call RequiredFile "$UH_BmpFolder/"                         || exit 1;
 call RequiredFile "$UH_CrcFolder/"                         || exit 1;
 call RequiredDefined UH_HomeDir                            || exit 1;
 call RequiredFile "$UH_HomeDir/"                           || exit 1;
 call RequiredFile "$UH_HomeDir/uniheater_make_crc.sh"      || exit 1;
 call RequiredFile "$UH_HomeDir/uniheater_make_cfg.sh"      || exit 1;
 if pushd "$HomeDir" >/dev/null 2>&1; then
  call UniHeaterConfig;
 fi;
 popd >/dev/null 2>&1;
 call HasNoErrors || call ErrorPrintLn "$errors error(s) found in $SelfName.";
 return $?;
};

function CopySamples(){
 call RequiredUnixUtils || exit 1;
 call RequiredAdminRights || exit 1;
 call RequiredParentCrwDaq || exit 1;
 call RequiredParamStr UH_PrefixName $2 || exit 1;
 call GetPrefixFile UH_PrefixFile $UH_PrefixName;
 call Define UH_PrefixFile $(printenv UH_PrefixName | tr '.' '_');
 call Define UH_HomeDir $CRW_DAQ_SYS_HOME_DIR/resource/daqsite/uniheater;
 for f in $UH_HomeDir/sample_uniheater_*.sh; do
  local fn="$(basename $f | sed "s/sample_/${UH_PrefixFile}_/")";
  cat $f | sed "s/SAMPLE/$UH_PrefixName/ig" > "$SelfDir/$fn";
  chmod ugo+x "$SelfDir/$fn";
 done;
};

function PrintVersion(){
  echo "$SelfBase version 1.0";
};

function  PrintHelp(){
 call PrintVersion;
 echo "Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Utility to make UniHeater's configuration.";
 echo "Usage:";
 echo " $SelfBase [Options] [Arguments]";
 echo "Options:";
 echo " -h,--help  - print help";
 echo " --version  - print version";
 echo " --samples  - create command files [init,make,zero] to generate config";
 echo "Arguments:";
 echo " UH_PrefixName - Prefix name to identify Uniheater";
 echo " UH_NumHeaters - Number of Unified Heaters, 1..32";
 echo " UH_TileNumber - Tile number = columns per row, 1..UH_NumHeaters";
 echo "Example:";
 echo " $SelfBase DEMO 16 8";
};

function UniHeaterConfig(){
 call Exec  "$UH_HomeDir/uniheater_make_cfg.sh" $UH_PrefixName $UH_NumHeaters $UH_TileNumber "$UH_CfgFolder" "$UH_CrcFolder" "$UH_BmpFolder" || call IncErrors 1;
};

################################################################################
## Common library routines                                                    ##
################################################################################

function Define(){
 if [ -z "$1" ]; then return 0; fi; ### Environ Variable Name
 if [ -z "$2" ]; then return 0; fi; ### Environ Variable Value
 if [ -z "$(printenv $1)" ]; then
  export $1="$2";
 fi;
};

function GetPrefixFile(){
 if [ -z "$1" ]; then return; fi;
 if [ -z "$2" ]; then return; fi;
 export $1="$(echo "$2" | tr '[:upper:]' '[:lower:]')";
};

function RequiredParamStr(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then export $1="$2"; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName - missed string param $1. Call $SelfName --help for details.";
};

function RequiredParamInt(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then let "$1=$2"; export $1; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName : missed integer param $1. Call $SelfName --help for details.";
};

function RequiredAdminRights(){
 if unix sudoit -v >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: Access denied. Admin rights required.";
};

function RequiredUnixUtils(){
 if which unix >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: UnixUtils is not valid. Install UnixUtils first.";
};

function RequiredParentCrwDaq(){
 local pid="$CRW_DAQ_SYS_EXE_PID";
 local cmd="$CRW_DAQ_SYS_EXE_FILE";
 local exe="$(basename $cmd)";
 local lst="";
 if [ -n "$pid" ] && [ -n "$exe" ]; then
  lst="$(pgrep $exe | grep $pid)";
 fi;
 if [ -z "$lst" ]; then
  fatal 1 "Error: $SelfName: parent CRW-DAQ is not found. Run this script from CRW-DAQ.";
 fi;
};

function RequiredFile(){
 if [ -n "$1" ] && [ -e "$1" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: not found $1.";
};

function RequiredDefined(){
 if [ -n "$1" ] && [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: undefined variable $1. Call $SelfName --help for details.";
};

function RequiredIntRange(){
 if [ -n "$1" ] &&  [ -n "$2" ] &&  [ -n "$3" ] && [ $1 -ge $2 ] && [ $1 -le $3 ]; then return 0; fi;
 fatal 1 "Error: $SelfName: invalid value $1 (not in $2..$3 range). Call $SelfName --help for details.";
};

function SetErrors(){
 let "errors=$1";
};

function IncErrors(){
 let "errors+=$1";
};

function HasNoErrors(){
 if [ "$errors" = "0" ]; then return $?; fi;
 return 1;
};

function Exec(){
 echo "$@";
 call "$@";
};

function PrintLn(){
 echo "$@";
};

function ErrorPrint(){
 1>&2 echo -n "$@";
};

function ErrorPrintLn(){
 1>&2 echo "$@";
};

function SetErrorLevel(){
 return $1;
};

function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then
   rm -f "$1";
  fi;
  shift;
 done;
};


uniheater_make "$@";

##############
## END OF FILE
##############
