////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWDAQ.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Speak server for &SpekSrv DAQ device.                                      //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20240213 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

program speaksrv; // Speak server

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math,
 interfaces, lclintf, forms,
 _crw_alloc, _crw_cmdargs, _crw_str, _crw_base64, _crw_fio, _crw_fifo,
 _crw_rtc, _crw_ascio, _crw_az, _crw_polling, _crw_task, _crw_sesman,
 _crw_spk, _crw_proc;

//
// General variables and constants
//
const
Terminated : Boolean  = false; // Program should be terminated
//
// Get EXE file version info.
//
function GetVersionInfo(const Name:LongString):LongString;
begin
 Result:=CookieScan(GetFileVersionInfoAsText(ProgName),Name);
end;

//
// Print greetings message.
//
procedure PrintGreetings;
begin
 StdOut.Put:=GetVersionInfo('ProductName')+' version '+GetVersionInfo('ProductVersion')+'.';
 StdOut.Put:=GetVersionInfo('LegalCopyright')+'.';
 StdOut.Put:=GetVersionInfo('ProductName')+' - '+GetVersionInfo('FileDescription');
 StdOut.Put:=GetVersionInfo('Comments');
 StdOut.Put:='Type @help to get Help.';
end;
//
// Request: @help
// Reply:   help info
// Comment: Show help.
//
procedure DoHelp(const cmnd,args:LongString);
begin
 StdOut.Put:=Format('%s',[cmnd]);
 StdOut.Put:='>> Command list:';
 StdOut.Put:='>> @help                 This help.';
 StdOut.Put:='>> @exit=n               Exit program with code n.';
 StdOut.Put:='>> @errors               Return error counter.';
 StdOut.Put:='>> @memory               Return memory status.';
 StdOut.Put:='>> @processpriority=n    Set process priority:Idle/Low/Normal/High/RealTime.';
 StdOut.Put:='>> @threadpriority=n     Set thread priority:tpIdle/tpLow/tpNormal/tpHigh/tpTimeCritical.';
 StdOut.Put:='>> @engine               Get current speech engine.';
 StdOut.Put:='>> @engine=n             Set current speech engine (kill if n=-1).';
 StdOut.Put:='>> @engines              Get list of engines.';
 StdOut.Put:='>> @speak=s              Speak message "s" with current speech engine.';
 StdOut.Put:='>> @stop                 Stop speech speaking.';
 StdOut.Put:='>> @wait                 Get current wait option.';
 StdOut.Put:='>> @wait=n               Set current wait option.';
 StdOut.Put:='>> @pause                Get current pause state.';
 StdOut.Put:='>> @pause=n              Set current pause state.';
 StdOut.Put:='>> @pitch                Get current speech pitch.';
 StdOut.Put:='>> @pitch=n              Set current speech speed.';
 StdOut.Put:='>> @speed                Get current speech speed.';
 StdOut.Put:='>> @speed=n              Set current speech speed.';
 StdOut.Put:='>> @volume               Get current speech volume.';
 StdOut.Put:='>> @volume=n             Set current speech volume.';
end;
//
// Request: @exit
// Request: @exit=n
// Reply:   @exit=n
// Comment: Terminate program with exit code n.
//
procedure DoExit(const cmnd,args:LongString);
begin
 Terminated:=true;
 System.ExitCode:=StrToIntDef(args,0);
 StdOut.Put:=Format('%s=%d',[cmnd,System.ExitCode]);
end;
//
// Request: @stop
// Reply:   @stop
// Comment: Stop speech.
//
procedure DoStop(const cmnd,args:LongString);
begin
 Speaker.Stop;
 StdOut.Put:=Format('%s',[cmnd]);
end;
//
// Request: @errors
// Request: @errors=n
// Reply:   @errors=n
// Comment: Return value n of error counter.
//
procedure DoErrors(const cmnd,args:LongString);
var
 n : LongInt;
begin
 if Str2Long(args,n)
 then n:=LockedExchange(StdIoErrorCount,n)
 else n:=LockedGet(StdIoErrorCount);
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @memory
// Request: @memory=n
// Comment: Return AllocMemSize.
//
procedure DoMemory(const cmnd,args:LongString);
begin
 StdOut.Put:=Format('%s=%d',[cmnd,GetAllocMemSize]);
end;
//
// Request: @ProcessPriority
// Request: @ProcessPriority=n
// Reply:   @ProcessPriority=n
// Comment: Set process priority class.
//
procedure DoProcessPriority(const cmnd,args:LongString);
var p:DWORD;
begin
 if not IsEmptyStr(args) then begin
  p:=GetPriorityClassByName(args);
  if (p>0) then SetPriorityClass(GetCurrentProcess,p);
 end;
 p:=ProcessPriorityToClass(GetProcessPriority);
 StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityClassName(p)]);
end;
//
// Request: @ThreadPriority
// Request: @ThreadPriority=n
// Reply:   @ThreadPriority=n
// Comment: Set thread priority class.
//
procedure DoThreadPriority(const cmnd,args:LongString);
var p:TThreadPriority;
begin
 if not IsEmptyStr(args) then begin
  p:=GetPriorityByName(args);
  StdIn.Priority:=p;
  StdOut.Priority:=p;
  {$IFDEF WINDOWS}
  case p of
   tpIdle         : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_IDLE);
   tpLowest       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_LOWEST);
   tpLower        : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_BELOW_NORMAL);
   tpNormal       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_NORMAL);
   tpHigher       : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_ABOVE_NORMAL);
   tpHighest      : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_HIGHEST);
   tpTimeCritical : SetThreadPriority(GetCurrentThread,THREAD_PRIORITY_TIME_CRITICAL);
  end;
  {$ENDIF ~WINDOWS}
 end;
 StdOut.Put:=Format('%s=%s',[cmnd,GetPriorityName(StdIn.Priority)]);
end;
//
// Request: @Speak=s
// Comment: Speak message from args.
//
procedure DoSpeak(const cmnd,args:LongString);
 function Speak(s:LongString):Boolean;
 begin
  Result:=false;
  try
   if (Speaker.Engine>=0) then begin
    if IsWindows then s:=DefToAnsiCp(s);
    if (Length(Trim(s))>0) then Speaker.Speak:=Trim(s);
    Result:=true;
   end;
  except
   on E:Exception do StdOut.Put:=E.Message;
  end;
 end;
begin
 StdOut.Put:=Format('%s=%s',[cmnd,IntToStr(Ord(Speak(args)))]);
end;
//
// Request: @wait
// Request: @wait=n
// Reply:   @wait=n
// Comment: Get\set Wait option of speech.
//
procedure DoWait(const cmnd,args:LongString);
var n:LongInt;
begin
 if Str2Long(args,n)
 then Speaker.Wait:=(n<>0)
 else n:=Ord(Speaker.Wait);
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @pause
// Request: @pause=n
// Reply:   @pause=n
// Comment: Get\set Pause state of speech.
//
procedure DoPause(const cmnd,args:LongString);
var n:LongInt;
begin
 if Str2Long(args,n)
 then Speaker.Pause:=(n<>0)
 else n:=Ord(Speaker.Pause);
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @speed
// Request: @speed=n
// Reply:   @speed=n
// Comment: Return speed value n of sound in percent.
//
procedure DoSpeed(const cmnd,args:LongString);
var n:LongInt;
begin
 n:=0;
 if (Speaker.Engine>=0) then begin
  if Str2Long(args,n) then Speaker.Speed:=n;
  n:=Speaker.Speed;
 end;
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @pitch
// Request: @pitch=n
// Reply:   @pitch=n
// Comment: Return pitch value n of sound in percent.
//
procedure DoPitch(const cmnd,args:LongString);
var n:LongInt;
begin
 n:=0;
 if Speaker.Engine>=0 then begin
  if Str2Long(args,n) then Speaker.Pitch:=n;
  n:=Speaker.Pitch;
 end;
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @volume
// Request: @volume=n
// Reply:   @volume=n
// Comment: Return volume value n of sound in percent.
//
procedure DoVolume(const cmnd,args:LongString);
var n:LongInt;
begin
 n:=0;
 if Speaker.Engine>=0 then begin
  if Str2Long(args,n) then Speaker.Volume:=n;
  n:=Speaker.Volume;
 end;
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @engine
// Request: @engine=n
// Reply:   @engine=n
// Comment: Get\set speech engine.
//
procedure DoEngine(const cmnd,args:LongString);
var n,i,j,k:LongInt;
begin
 if Str2Long(args,n) then Speaker.Engine:=n else if Length(args)>0 then begin
  j:=-1;
  k:=Speaker.Engine;
  if Speaker.Engine<0 then Speaker.Engine:=0;
  for i:=0 to Speaker.Engines.Count-1 do
  if IsSameText(Trim(args),Trim(Speaker.Engines[i])) then j:=i;
  if (j<0) then Speaker.Engine:=k else Speaker.Engine:=j;
 end;
 n:=Speaker.Engine;
 StdOut.Put:=Format('%s=%d',[cmnd,n]);
end;
//
// Request: @engines
// Reply:   @engines=n
// Comment: Get speech engines list.
//
procedure DoEngines(const cmnd,args:LongString);
var i:LongInt;
begin
 StdOut.Put:=Format('%s=%d',[cmnd,Speaker.Engines.Count]);
 for i:=0 to Speaker.Engines.Count-1 do
 StdOut.Put:=Format('@Engine[%d]=%s',[i,Speaker.Engines[i]]);
end;
//
// This callback handles unrecognized commands.
//
procedure DoSpecificCommands(const args:LongString);
begin
 if (Length(args)>0) then
 StdOut.Put:=Format('Could not recognize "%s"',[args]);
end;
//
// Application specific initialization.
//
procedure SpecificInitialization;
begin
 UseRunCommandEx(True);
 Application.Initialize;
 SystemEchoProcedure:=StdOutEcho;
 PrintGreetings; StdOut.Put:='';
 Speaker.Engine:=0;
 if Speaker.Engines.Count>0 then begin
  StdIn.Put:='@Engines';
  StdIn.Put:='@Engine';
  StdIn.Put:='@Volume';
  StdIn.Put:='@Pitch';
  StdIn.Put:='@Speed';
  StdIn.Put:='@Wait';
 end else begin
  StdOut.Put:='Could not find speech engines.';
  StdIn.Put:='@exit=13';
 end;
 //
 // Register user commands coming from StdIn.
 //
 StdIn.SpecHandler:=DoSpecificCommands;
 StdIn.AddCommand('@Help',            DoHelp);
 StdIn.AddCommand('@Exit',            DoExit);
 StdIn.AddCommand('@Stop',            DoStop);
 StdIn.AddCommand('@Errors',          DoErrors);
 StdIn.AddCommand('@Memory',          DoMemory);
 StdIn.AddCommand('@ProcessPriority', DoProcessPriority);
 StdIn.AddCommand('@ThreadPriority',  DoThreadPriority);
 StdIn.AddCommand('@Speak',           DoSpeak);
 StdIn.AddCommand('@Wait',            DoWait);
 StdIn.AddCommand('@Pause',           DoPause);
 StdIn.AddCommand('@Speed',           DoSpeed);
 StdIn.AddCommand('@Pitch',           DoPitch);
 StdIn.AddCommand('@Volume',          DoVolume);
 StdIn.AddCommand('@Engine',          DoEngine);
 StdIn.AddCommand('@Engines',         DoEngines);
end;
//
// Application specific finalization.
//
procedure SpecificFinalization;
begin
 Speaker.Free;
 Application.Terminate;
end;
//
// Application specific polling.
//
procedure SpecificPolling;
const LastTicks:QWord=0;
var CurrTicks:QWord;
begin
 CurrTicks:=GetTickCount64;
 if (CurrTicks>LastTicks+1000) then begin
  TTask.PollDetachedPids;
  LastTicks:=CurrTicks;
 end;
 Application.ProcessMessages;
 if BecameZombie(FILE_TYPE_PIPE,1000) then StdIn.Put:='@Exit';
end;
//
// Main program
//
begin
 try
  try
   SpecificInitialization;
   while not Terminated do begin
    while (StdIn.Count>0) do StdIn.Process(StdIn.Get);
    SpecificPolling;
    Sleep(TPolling.DefPollPeriod);
   end;
  finally
   SpecificFinalization;
  end;
 except
  on E:Exception do begin
   StdOut.Put:=E.Message;
   ExitCode:=1;
  end;
 end;
 Sleep(100);
 if BecameZombie(FILE_TYPE_PIPE,0) then ExitCode:=1;
end.

//////////////
// END OF FILE
//////////////

