@echo off
SetLocal EnableExtensions EnableDelayedExpansion

rem ****************************************************
rem Graphic designations. Pipeline accessories generator
rem ****************************************************
:: Graphic designation of valves       are defined in GOST 2.785
:: Graphic designation of valve drives are defined in GOST 2.721
:: Graphic designation of pumps        are defined in GOST 2.796
:: Graphic designation of vacuometers  are defined in GOST 2.796
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: GOST Elements table: Valves, Pumps, etc.
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Valve        :: Limit Switch  :: Valve Drive :: Description
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: ThruWayVer   :: LimSwNone     :: NoneOp      :: Through-way vertical valve
:: ThruWayHor   :: LimSwOpen     :: ManualOp    :: Through-way horizontal valve
:: AngUpLeft    :: LimSwClose    :: SolenoidOp  :: Angle Up-Left valve
:: AngUpRight   :: LimSwBoth     :: MotorOp     :: Angle Up-Right valve
:: AngDownLeft  ::               :: PneumaticOp :: Angle Down-Left valve
:: AngDownRight ::               ::             :: Angle Down-Right valve
:: 3WayDown     ::               ::             :: Three-Way Down valve
:: 3WayUp       ::               ::             :: Three-Way Up valve
:: 3WayLeft     ::               ::             :: Three-Way Left valve
:: 3WayRight    ::               ::             :: Three-Way Right valve
::              ::               :::::::::::::::::
:: RotGateVer   ::               :: NoneOp      :: Rotary gate vertical
:: RotGateHor   ::               ::             :: Rotary gate horizontal
:: VlvGateVer   ::               ::             :: Valve gate vertical
:: VlvGateHor   ::               ::             :: Valve gate horizontal
::================================================================================
:: Pump         :: Orientation   :: Description
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: RotSingle    :: Right         :: Rotary Single-Stage pump
:: RotDouble    :: Left          :: Rotary Double-Stage pump
:: Roots        :: Up            :: Roots pump
:: Turbomol     :: Down          :: Turbomolecular pump
::              :::::::::::::::::::
:: Diffusion    :: None          :: Diffusion pump
:: SputIon      ::               :: Sputter Ion pump
:: Sublimation  ::               :: Sublimation pump
::================================================================================
:: Vacuometer   :: Orientation   :: Description
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Penning      :: Right         :: Penning gauge
:: Ionization   :: Left          :: Ionization vacuum gauge
::              :: Up            ::
::              :: Down          ::
::              :::::::::::::::::::
:: Thermal      :: None          :: Thermal conductivity vacuum gauge
::================================================================================
:: LeakDetector :: -             :: Leak detector
::================================================================================
:: Compressor   :: Right         :: Compressor as is
::              :: Left          ::
::              :: Up            ::
::              :: Down          ::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::================================================================================
:: Draw valve and valve drive type
:: Used vertex:
:: Valve      ; Solenoid          ; Motorized         ; Pneumatic          ; Rotary gate     ;
::            ;       C___        ;       C___        ;        C___        ;     _______     ;
::   C__D     ;       |   |       ;       /   \       ;       K|_S_|L      ;    C|     |     ;
:: B  \/  E   ;       |___|D      ;       \___/D      ;        |_|_|       ; B___|   / |___  ;
:: |\ O  /|   ;  A___   |S ___    ;   A__   |S  __    ;  A_I_    | D __M_  ;  |  |\ /  |  |  ;
:: |/    \|   ;   | R|__|__|E |   ;   / R\__|__/E \   ;  |R|_|_ _| _E|_|T| ;  |  | O*  |  |  ;
:: A  /\  F   ;   |  | O| T|  |   ;   \  / O| T\  /   ;  | | |  O|   | | | ;  |  |  / \|  |  ;
::    --      ;   ---B  |  ---F   ;    --B  |   --F   ;  ----B   |   ----F ;  ---| /   |---F ;
::   H  G     ;       G----       ;       G---        ;    J   G---    N   ;     |     |     ;
::            ;       | U |       ;       / U \       ;       P|_|_|Q      ;     -------G    ;
::            ;       |___|H      ;       \___/H      ;        |_U_|       ;                 ;
::            ;                   ;                   ;            H       ;                 ;
:: For temporary variables it is necessary to use a prefix and postfix "_".
:: For Example: _x1_,_y1_,_x2_,_y2_ - Sensor size variables;
::              _Cl_                - Color variable;
::================================================================================

:Main
call :InitializeVariables
pushd "%~dp0" && call :GostLibGenerate
echo %count% section^(s^) created
Sleep 10
popd
goto :EOF

:InitializeVariables
set GostLib=%~n0.crc
call :SetValveCoordinates
set /a count=0
goto :EOF

:GostLibGenerate
call :DeleteFiles %GostLib%
call :StartupSection >> %GostLib%
set list_valve_types=ThruWayVer ThruWayHor AngUpLeft AngUpRight AngDownLeft AngDownRight 3WayDown 3WayUp 3WayLeft 3WayRight
set list_valve_limsw=LimSwNone LimSwOpen LimSwClose LimSwBoth
set list_valve_drive=NoneOp ManualOp SolenoidOp MotorOp PneumaticOp
for %%i in ( %list_valve_types% ) do (
 for %%j in ( %list_valve_limsw% ) do (
  for %%k in ( %list_valve_drive% ) do (
   call :ValvesGenerate %%i %%j %%k
  )
 )
)
set list_valve_types=RotGateVer RotGateHor
set list_valve_drive=NoneOp
for %%i in ( %list_valve_types% ) do (
 for %%j in ( %list_valve_limsw% ) do (
  for %%k in ( %list_valve_drive% ) do (
   call :RotGateGenerate %%i %%j %%k
  )
 )
)
set list_valve_types=VlvGateVer VlvGateHor
set list_valve_drive=NoneOp
for %%i in ( %list_valve_types% ) do (
 for %%j in ( %list_valve_limsw% ) do (
  for %%k in ( %list_valve_drive% ) do (
   call :VlvGateGenerate %%i %%j %%k
  )
 )
)
set list_pump_types=RotSingle RotDouble Turbomol Roots
set list_pump_orien=Up Down Left Right
for %%i in ( %list_pump_types% ) do (
 for %%j in ( %list_pump_orien% ) do (
  call :PumpGenerate %%i %%j
 )
)
set list_pump_types=Diffusion SputIon Sublimation
set list_pump_orien=None
for %%i in ( %list_pump_types% ) do (
 for %%j in ( %list_pump_orien% ) do (
  call :PumpGenerate %%i %%j
 )
)
set list_vacuometer_types=Penning Thermal Ionization
set list_vacuometer_orien=Up Down Left Right None
for %%i in ( %list_vacuometer_types% ) do (
 for %%j in ( %list_vacuometer_orien% ) do (
  call :VacuometerGenerate %%i %%j
 )
)
set list_compressor_orien=Up Down Left Right
for %%i in ( %list_compressor_orien% ) do (
 call :CompressorGenerate %%i %%j
)
call :LeakDetectorGenerate
goto :EOF

:StartupSection
echo [Circuit.StartupScript]
echo (glValveOpenBit=1)+(glValveCloseBit=2)+(glPumpSpeedBit=0)+(glPumpErrorBit=0)+(glPumpEdgeBevel=3)
echo (glVacuometerBevel=4)+(glVacuometerFillColor=clGray)+(glVacuometerLineColor=clBlack)
echo (glLeakDetBevel=3)+(glLeakDetFillColor=clGray)+(glLeakDetLineColor=clBlack)
echo (glCompressorBevel=4)+(glCompressorFillColor=clGray)+(glCompressorLineColor=clBlack)
echo []
goto :EOF

:ValvesGenerate
set GostElementType=Valve
set "vType=%~1" & shift
set "vSwch=%~1" & shift
set "vOper=%~1" & shift
echo [Painter(v).GostLib.%GostElementType%.%vType%.%vSwch%.%vOper%]
call :ValveOrientDetermine
call :GostLibValveSection >> %GostLib%
call :ValveColorCalculate >> %GostLib%
call :ExecuteValveDrawing >> %GostLib%
call :ValveDriveDrawing   >> %GostLib%
call :FinalValveSection   >> %GostLib%
set /a count+=1
goto :EOF

:RotGateGenerate
set GostElementType=Valve
set "vType=%~1" & shift
set "vSwch=%~1" & shift
set "vOper=%~1" & shift
echo [Painter(v).GostLib.%GostElementType%.%vType%.%vSwch%.%vOper%]
call :GostLibValveSection >> %GostLib%
call :ValveColorCalculate >> %GostLib%
call :RotGateDrawing      >> %GostLib%
call :FinalValveSection   >> %GostLib%
set /a count+=1
goto :EOF

:VlvGateGenerate
set GostElementType=Valve
set "vType=%~1" & shift
set "vSwch=%~1" & shift
set "vOper=%~1" & shift
echo [Painter(v).GostLib.%GostElementType%.%vType%.%vSwch%.%vOper%]
call :ValveOrientDetermine
call :GostLibValveSection >> %GostLib%
call :ValveColorCalculate >> %GostLib%
call :ExecuteValveDrawing >> %GostLib%
call :ValveGateDrawing    >> %GostLib%
call :FinalValveSection   >> %GostLib%
set /a count+=1
goto :EOF

:PumpGenerate
set GostElementType=Pump
set "pType=%~1" & shift
set "pOrien=%~1" & shift
echo [Painter(v).GostLib.%GostElementType%.%pType%.%pOrien%]
call :BeginPumpSection >> %GostLib%
call :ExecPumpGenerate >> %GostLib%
call :FinalPumpSection >> %GostLib%
set /a count+=1
goto :EOF

:VacuometerGenerate
set GostElementType=Vacuometer
set "vmType=%~1" & shift
set "vmOrien=%~1" & shift
if %vmType% == Penning (
 if %vmOrien% == None goto :EOF
)
if %vmType% == Ionization (
 if %vmOrien% == None goto :EOF
)
if %vmType% == Thermal (
 if %vmOrien% == Up goto :EOF
 if %vmOrien% == Down goto :EOF
 if %vmOrien% == Left goto :EOF
 if %vmOrien% == Right goto :EOF
)
echo [Painter(v).GostLib.%GostElementType%.%vmType%.%vmOrien%]
call :BeginVacuometerSection >> %GostLib%
call :ExecVacuometerGenerate >> %GostLib%
call :FinalVacuometerSection >> %GostLib%
set /a count+=1
goto :EOF

:CompressorGenerate
set GostElementType=Compressor
set "cmOrien=%~1" & shift
echo [Painter(v).GostLib.%GostElementType%.%cmOrien%]
call :BeginCompressorSection >> %GostLib%
call :ExecCompressorGenerate >> %GostLib%
call :FinalCompressorSection >> %GostLib%
set /a count+=1
goto :EOF

:LeakDetectorGenerate
set GostElementType=LeakDetector
echo [Painter(v).GostLib.%GostElementType%]
call :ExecLeakDetectorGenerate >> %GostLib%
set /a count+=1
goto :EOF

:: Determine valve orientation
:ValveOrientDetermine
set /a valveUp=0 & set /a valveDown=0 & set /a valveLeft=0 & set /a valveRight=0
set /a driveUp=0 & set /a driveDown=0 & set /a driveLeft=0 & set /a driveRight=0
if "%vType%" == "ThruWayVer"   ( set /a valveUp=1   & set /a valveDown=1  & set /a driveLeft=1  & goto :EOF)
if "%vType%" == "ThruWayHor"   ( set /a valveLeft=1 & set /a valveRight=1 & set /a driveUp=1    & goto :EOF)
if "%vType%" == "VlvGateVer"   ( set /a valveUp=1   & set /a valveDown=1  & set /a driveLeft=1  & goto :EOF)
if "%vType%" == "VlvGateHor"   ( set /a valveLeft=1 & set /a valveRight=1 & set /a driveUp=1    & goto :EOF)
if "%vType%" == "AngUpLeft"    ( set /a valveUp=1   & set /a valveLeft=1  & set /a driveRight=1 & goto :EOF)
if "%vType%" == "AngUpRight"   ( set /a valveUp=1   & set /a valveRight=1 & set /a driveLeft=1  & goto :EOF)
if "%vType%" == "AngDownLeft"  ( set /a valveDown=1 & set /a valveLeft=1  & set /a driveUp=1    & goto :EOF)
if "%vType%" == "AngDownRight" ( set /a valveDown=1 & set /a valveRight=1 & set /a driveLeft=1  & goto :EOF)
if "%vType%" == "3WayDown"     ( set /a valveLeft=1 & set /a valveRight=1 & set /a valveDown=1  & set /a driveUp=1    & goto :EOF)
if "%vType%" == "3WayUp"       ( set /a valveLeft=1 & set /a valveRight=1 & set /a valveUp=1    & set /a driveDown=1  & goto :EOF)
if "%vType%" == "3WayLeft"     ( set /a valveLeft=1 & set /a valveDown=1  & set /a valveUp=1    & set /a driveRight=1 & goto :EOF)
if "%vType%" == "3WayRight"    ( set /a valveUp=1   & set /a valveRight=1 & set /a valveDown=1  & set /a driveLeft=1  & goto :EOF)
goto :EOF

:: Set used vertex coordinates (See "Used vertex"). Ox,Oy          - Sensor center coordinates
:: vAx,vAy,vBx,vBy,vCx,vCy,vDx,vDy,vEx,vEy,vFx,vFy,vGx,vGy,vHx,vHy - Coordinates for valves. Prefix "v" (valve)
:: dAx,dAy,dBx,dBy,dCx,dCy,dDx,dDy,dEx,dEy,dFx,dFy,dGx,dGy,dHx,dHy - Coordinates for valve drive drawing.
:: dRx,dRy,dSx,dSy,dTx,dTy,dUx,dUy                                   Prefix "d" (drive)
:: dIx,dIy,dJx,dJy,dKx,dKy,dLx,dLy,dMx,dMy,dNx,dNy,dPx,dPy,dQx,dQy
:SetValveCoordinates
set Ox=_xc_
set Oy=_yc_
:: Set valve coordinates (prefix v - valve)
set vAx=%Ox%-2*_x5_
set vAy=%Oy%+_x5_
set vBx=%vAx%
set vBy=%Oy%-_x5_
set vCx=%Ox%-_x5_
set vCy=%Oy%-2*_x5_
set vDx=%Ox%+_x5_
set vDy=%vCy%
set vEx=%Ox%+2*_x5_
set vEy=%vBy%
set vFx=%vEx%
set vFy=%vAy%
set vGx=%vDx%
set vGy=%Oy%+2*_x5_
set vHx=%vCx%
set vHy=%vGy%
:: Set drive coordinates (prefix d - drive)
set dAx=%Ox%-_x6_-_x3_+_pw_
set dAy=%Oy%-_x6_
set dBx=%Ox%+_x6_-_x3_+_pw_
set dBy=%Oy%+_x6_
set dCx=%Ox%-_x6_
set dCy=%Oy%-_x6_-_x3_+_pw_
set dDx=%Ox%+_x6_
set dDy=%Oy%+_x6_-_x3_+_pw_
set dEx=%Ox%-_x6_+_x3_-_pw_
set dEy=%Oy%-_x6_
set dFx=%Ox%+_x6_+_x3_-_pw_
set dFy=%Oy%+_x6_
set dGx=%Ox%-_x6_
set dGy=%Oy%-_x6_+_x3_-_pw_
set dHx=%Ox%+_x6_
set dHy=%Oy%+_x6_+_x3_-_pw_
set dRx=%Ox%-_x3_
set dRy=%Oy%
set dSx=%Ox%
set dSy=%Oy%-_x3_
set dTx=%Ox%+_x3_
set dTy=%Oy%
set dUx=%Ox%
set dUy=%Oy%+_x3_
set dIx=%Ox%-_x3_
set dIy=%Oy%-_x6_
set dJx=%Ox%-_x3_
set dJy=%Oy%+_x6_
set dKx=%Ox%-_x6_
set dKy=%Oy%-_x3_
set dLx=%Ox%+_x6_
set dLy=%Oy%-_x3_
set dMx=%Ox%+_x3_-1
set dMy=%Oy%-_x6_
set dNx=%Ox%+_x3_-1
set dNy=%Oy%+_x6_
set dPx=%Ox%-_x6_
set dPy=%Oy%+_x3_-1
set dQx=%Ox%+_x6_
set dQy=%Oy%+_x3_-1
goto :EOF

::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Valves generation :::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:GostLibValveSection
echo [Painter(v).GostLib.%GostElementType%.%vType%.%vSwch%.%vOper%]
if "%vSwch%" NEQ "LimSwNone" ( echo var glValveOpenBit,glValveCloseBit)
echo (_x2_=min(sensorWidth(),sensorHeight()))+(_x2_=_x2_+eq(frac(_x2_/2),0))+(_y2_=_x2_)+(_pw_=2-lt(_x2_,26))+(_xc_=int(_x2_/2))+(_yc_=int(_y2_/2))+(_x3_=int(_x2_/3))+(_x5_=int(_x2_/5))+(_x6_=int(_x2_/6))
goto :EOF

:FinalValveSection
if "%vSwch%" NEQ "LimSwNone" echo (glValveOpenBit=1^)+(glValveCloseBit=2^)
echo []
goto :EOF

:: Valve color calculate
:ValveColorCalculate
if "%vSwch%" == "LimSwNone"  echo _Cl_=(not(isbit(v,glButtonBit))*clGray+isbit(v,glButtonBit)*clLime)& goto :EOF
if "%vSwch%" == "LimSwOpen"  echo (_Bt_=isbit(v,glButtonBit))+(_oBit_=isbit(v,glValveOpenBit))+(_cBit_=isbit(v,glValveCloseBit))+(_vOpn_=and(_Bt_,_oBit_))+(_vClo_=and(not(_Bt_),not(_oBit_)))+(_vInt_=and(not(_vOpn_),not(_vClo_)))+(_Cl_=_vOpn_*clLime+_vClo_*clGray+_vInt_*clYellow)& goto :EOF
if "%vSwch%" == "LimSwClose" echo (_Bt_=isbit(v,glButtonBit))+(_oBit_=isbit(v,glValveOpenBit))+(_cBit_=isbit(v,glValveCloseBit))+(_vOpn_=and(_Bt_,not(_cBit_)))+(_vClo_=and(not(_Bt_),_cBit_))+(_vInt_=and(not(_vOpn_),not(_vClo_)))+(_Cl_=_vOpn_*clLime+_vClo_*clGray+_vInt_*clYellow)& goto :EOF
if "%vSwch%" == "LimSwBoth" (
 echo (_Bt_=isbit(v,glButtonBit^)^)+(_oBit_=isbit(v,glValveOpenBit^)^)+(_cBit_=isbit(v,glValveCloseBit^)^)+(_vOpn_=and(and(_Bt_,_oBit_^),not(_cBit_^)^)^)+(_vClo_=and(and(not(_Bt_^),not(_oBit_^)^),_cBit_^)^)
 echo (_vErr_=and(_oBit_,_cBit_^)^)+(_vInt_=and(and(not(_vOpn_^),not(_vClo_^)^),not(_vErr_^)^)^)+(_Cl_=_vOpn_*clLime+_vClo_*clGray+_vInt_*clYellow+_vErr_*clRed^)
)
goto :EOF

:: Draw Valve
:ExecuteValveDrawing
if "%valveLeft%" == "1"  echo setPen(clBlack,psSolid,pmCopy,_pw_)+setBrush(_Cl_,bsSolid)+drawPoly(%Ox%,%Oy%,plAddPoint)+drawPoly(%vAx%,%vAy%,plAddPoint)+drawPoly(%vBx%,%vBy%,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)
if "%valveUp%" == "1"    echo setPen(clBlack,psSolid,pmCopy,_pw_)+setBrush(_Cl_,bsSolid)+drawPoly(%Ox%,%Oy%,plAddPoint)+drawPoly(%vCx%,%vCy%,plAddPoint)+drawPoly(%vDx%,%vDy%,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)
if "%valveRight%" == "1" echo setPen(clBlack,psSolid,pmCopy,_pw_)+setBrush(_Cl_,bsSolid)+drawPoly(%Ox%,%Oy%,plAddPoint)+drawPoly(%vEx%,%vEy%,plAddPoint)+drawPoly(%vFx%,%vFy%,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)
if "%valveDown%" == "1"  echo setPen(clBlack,psSolid,pmCopy,_pw_)+setBrush(_Cl_,bsSolid)+drawPoly(%Ox%,%Oy%,plAddPoint)+drawPoly(%vGx%,%vGy%,plAddPoint)+drawPoly(%vHx%,%vHy%,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)
goto :EOF

:: Draw Valve Drive
:ValveDriveDrawing
if "%vOper%" == "NoneOp" ( goto :EOF ) else (
 if "%vOper%" == "ManualOp"    call :DrawManualOp    & goto :EOF
 if "%vOper%" == "SolenoidOp"  call :DrawSolenoidOp  & goto :EOF
 if "%vOper%" == "MotorOp"     call :DrawMotorOp     & goto :EOF
 if "%vOper%" == "PneumaticOp" call :DrawPneumaticOp & goto :EOF
)
goto :EOF

:DrawManualOp
if "%driveLeft%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dRx%,%dRy%)+drawLine(%dIx%,%dIy%,%dJx%,%dJy%+1)& goto :EOF
if "%driveUp%"    == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dSx%,%dSy%)+drawLine(%dKx%,%dKy%,%dLx%+1,%dLy%)& goto :EOF
if "%driveRight%" == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dTx%,%dTy%)+drawLine(%dMx%,%dMy%,%dNx%,%dNy%+1)& goto :EOF
if "%driveDown%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dUx%,%dUy%)+drawLine(%dPx%,%dPy%,%dQx%+1,%dQy%)
goto :EOF

:DrawSolenoidOp
if "%driveLeft%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dRx%,%dRy%)+drawRect(%dAx%,%dAy%+1,%dBx%-1,%dBy%)& goto :EOF
if "%driveUp%"    == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dSx%,%dSy%)+drawRect(%dCx%+1,%dCy%,%dDx%,%dDy%-1)& goto :EOF
if "%driveRight%" == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dTx%,%dTy%)+drawRect(%dEx%+1,%dEy%+1,%dFx%,%dFy%)& goto :EOF
if "%driveDown%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dUx%,%dUy%)+drawRect(%dGx%+1,%dGy%+1,%dHx%,%dHy%)
goto :EOF

:DrawMotorOp
if "%driveLeft%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dRx%,%dRy%)+drawEllipse(%dAx%,%dAy%,%dBx%,%dBy%)& goto :EOF
if "%driveUp%"    == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dSx%,%dSy%)+drawEllipse(%dCx%,%dCy%,%dDx%,%dDy%)& goto :EOF
if "%driveRight%" == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dTx%,%dTy%)+drawEllipse(%dEx%,%dEy%,%dFx%,%dFy%)& goto :EOF
if "%driveDown%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawLine(%Ox%,%Oy%,%dUx%,%dUy%)+drawEllipse(%dGx%,%dGy%,%dHx%,%dHy%)
goto :EOF

:DrawPneumaticOp
if "%driveLeft%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawRect(%dAx%,%dAy%+1,%dBx%-1,%dBy%)+drawLine(%Ox%,%Oy%,%dRx%,%dRy%)+drawLine(%dIx%,%dIy%+1,%dJx%,%dJy%)& goto :EOF
if "%driveUp%"    == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawRect(%dCx%+1,%dCy%,%dDx%,%dDy%-1)+drawLine(%Ox%,%Oy%,%dSx%,%dSy%)+drawLine(%dKx%+1,%dKy%,%dLx%,%dLy%)& goto :EOF
if "%driveRight%" == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawRect(%dEx%+1,%dEy%+1,%dFx%,%dFy%)+drawLine(%Ox%,%Oy%,%dTx%,%dTy%)+drawLine(%dMx%,%dMy%+1,%dNx%,%dNy%)& goto :EOF
if "%driveDown%"  == "1" echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(clSilver,bsSolid)+drawRect(%dGx%+1,%dGy%+1,%dHx%,%dHy%)+drawLine(%Ox%,%Oy%,%dUx%,%dUy%)+drawLine(%dPx%+1,%dPy%,%dQx%,%dQy%)
goto :EOF

:: Rotary gate drawing
:RotGateDrawing
if "%vType%" == "RotGateVer" call :DrawRotGateVer & goto :EOF
if "%vType%" == "RotGateHor" call :DrawRotGateHor & goto :EOF
goto :EOF

:DrawRotGateVer
echo setPen(clBlack,psSolid,pmCopy,_pw_)+setBrush(_Cl_,bsSolid)+drawRect(%vCx%,%vCy%,%vGx%+_pw_,%vGy%+1)+drawPoint(%Ox%,%Oy%,clBlack,15)
echo (_x8_=int(_x2_/8))+setPen(clBlack,psSolid,pmCopy,1)+drawLine(%Ox%+_x8_,%Oy%-2*_x8_,%Ox%-_x8_,%Oy%+2*_x8_)
goto :EOF

:DrawRotGateHor
echo setPen(clBlack,psSolid,pmCopy,_pw_)+setBrush(_Cl_,bsSolid)+drawRect(%vBx%,%vBy%,%vFx%+1,%vFy%+_pw_)+drawPoint(%Ox%,%Oy%,clBlack,15)
echo (_x8_=int(_x2_/8))+setPen(clBlack,psSolid,pmCopy,1)+drawLine(%Ox%-2*_x8_,%Oy%-_x8_,%Ox%+2*_x8_,%Oy%+_x8_)
goto :EOF

:: Valve gate drawing
:ValveGateDrawing
if "%vType%" == "VlvGateVer" echo drawLine(%vCx%,%Oy%-_pw_+1,%vDx%,%Oy%)& goto :EOF
if "%vType%" == "VlvGateHor" echo drawLine(%Ox%,%vBy%,%Ox%,%vAy%)
goto :EOF

::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Pumps generation ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:BeginPumpSection
echo [Painter(v).GostLib.%GostElementType%.%pType%.%pOrien%]
echo var glPumpSpeedBit,glPumpErrorBit,glPumpEdgeBevel
echo (_x2_=min(sensorWidth(),sensorHeight()))+(_kc_=eq(frac(_x2_/2),0))+(_x2_=_x2_+eq(frac(_x2_/2),0))+(_y2_=_x2_)+(_xc_=int(_x2_/2))+(_yc_=int(_y2_/2))
echo (_x4_=int(_x2_/4))+(_pw_=2-lt(_x2_,24))+(_be_=glPumpEdgeBevel)+(_Bt_=isbit(v,glButtonBit))+(_Sp_=isbit(v,glPumpSpeedBit))+(_Er_=isbit(v,glPumpErrorBit))
echo _Cl_=(and(eq(gt(glPumpSpeedBit,0)+gt(glPumpErrorBit,0),0),_Bt_)*clLime)+(and(eq(gt(glPumpSpeedBit,0)+gt(glPumpErrorBit,0),0),not(_Bt_))*clGray)
echo _Cl_=_Cl_+(and(gt(glPumpSpeedBit,0),not(gt(glPumpErrorBit,0)))*(and(_Sp_,_Bt_)*clLime+and(not(_Sp_),not(_Bt_))*clGray+and(not(_Sp_),_Bt_)*clYellow+and(_Sp_,not(_Bt_))*clYellow))
echo _Cl_=_Cl_+(and(not(gt(glPumpSpeedBit,0)),gt(glPumpErrorBit,0))*(and(not(_Er_),_Bt_)*clLime+and(not(_Er_),not(_Bt_))*clGray+_Er_*clRed))
echo _Cl_=_Cl_+(and(gt(glPumpSpeedBit,0),gt(glPumpErrorBit,0))*(and(and(_Sp_,_Bt_),not(_Er_))*clLime+and(and(not(_Sp_),not(_Bt_)),not(_Er_))*clGray+and(and(not(_Sp_),_Bt_),not(_Er_))*clYellow+and(and(_Sp_,not(_Bt_)),not(_Er_))*clYellow+_Er_*clRed))
goto :EOF

:FinalPumpSection
echo (glPumpSpeedBit=0)+(glPumpErrorBit=0)+(glPumpEdgeBevel=3)
echo []
goto :EOF

:: Various pump types generation
:ExecPumpGenerate
echo setPen(clBlack,psSolid,pmCopy,1)+setBrush(_Cl_,bsSolid)+drawRect(_be_,_be_,_x2_-_be_-_kc_,_y2_-_be_-_kc_)+setPen(clBlack,psSolid,pmCopy,_pw_^)+setBrush(_Cl_,bsClear^)
if "%pType%" == "RotSingle" (
 call :PumpOrientation
 echo setPen(clGray25,psSolid,pmCopy,_pw_^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)+setPen(clBlack,psSolid,pmCopy,1^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)
 goto :EOF
)
if "%pType%" == "RotDouble" (
 call :PumpOrientation
 echo (_x7_=int(_x2_/7^)^)+setPen(clGray25,psSolid,pmCopy,_pw_^)+setBrush(_Cl_,bsClear^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)+drawEllipse(_xc_-_x7_,_yc_-_x7_,_xc_+_x7_,_yc_+_x7_^)
 echo setPen(clBlack,psSolid,pmCopy,1^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)+drawEllipse(_xc_-_x7_,_yc_-_x7_,_xc_+_x7_,_yc_+_x7_^)
 goto :EOF
)
if "%pType%" == "Turbomol" (
 call :PumpOrientation
 echo setPen(clGray25,psSolid,pmCopy,_pw_^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)+setPen(clBlack,psSolid,pmCopy,1^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)
 echo (_x10_=int(_x2_/10^)^)+(_x14_=int(_x2_/14^)^)+drawLine(_xc_,_yc_-_x10_,_xc_,_yc_+_x10_^)+drawLine(_xc_-_x14_,_yc_-_x10_,_xc_-_x14_,_yc_+_x10_^)
 echo drawLine(_xc_-2*_x14_,_yc_-_x10_,_xc_-2*_x14_,_yc_+_x10_^)+drawLine(_xc_+_x14_,_yc_-_x10_,_xc_+_x14_,_yc_+_x10_^)+drawLine(_xc_+2*_x14_,_yc_-_x10_,_xc_+2*_x14_,_yc_+_x10_^)
 goto :EOF
)
if "%pType%" == "Roots" (
 call :PumpOrientation
 echo setPen(clGray25,psSolid,pmCopy,_pw_^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)+setPen(clBlack,psSolid,pmCopy,1^)+drawEllipse(_xc_-_x4_,_yc_-_x4_,_xc_+_x4_,_yc_+_x4_^)+(_x10_=int(_x2_/10^)^)
 echo (_x14_=int(_x2_/14^)^)+setPen(clBlack,psSolid,pmCopy,_pw_^)+drawLine(_xc_,_yc_-_x10_-_pw_,_xc_,_yc_+_x10_-_pw_-1^)+drawLine(_xc_-_x10_-eq(_pw_,2^),_yc_+_x10_+_x14_-_pw_-1,_xc_+_x10_,_yc_+_x10_+_x14_-_pw_-eq(_pw_,1^)^)
 goto :EOF
)
if "%pType%" == "Diffusion" (
 echo (_x7_=int(_x2_/7^)^)+(_x5_=int(_x2_/5^)^)+drawLine(_xc_-_x7_,_yc_-_x7_,_xc_-_x7_,_yc_+_x4_^)+drawLine(_xc_+_x7_,_yc_-_x7_,_xc_+_x7_,_yc_+_x4_^)+(_x25_=int(_x2_/25^)^)+drawLine(_xc_-_x5_,_yc_-_x4_+_pw_-1-_x25_,_xc_+_x5_-eq(_pw_,2^),_yc_-_x4_-_x25_^)
 echo (_x8_=int(_x2_/8^)^)+drawLine(_xc_-_x5_,_yc_-_x4_-_x25_,_xc_-_x5_-_x8_,_yc_-_x4_+_x8_-_x25_^)+drawLine(_xc_+_x5_-eq(_pw_,2^),_yc_-_x4_-_x25_,_xc_+_x5_+_x8_-eq(_pw_,2^),_yc_-_x4_+_x8_-_x25_^)
 goto :EOF
)
if "%pType%" == "SputIon" (
 echo drawLine(_xc_-_x4_+1,_yc_-_x4_+1,_xc_-_x4_+1,_yc_+_x4_-1^)+drawLine(_xc_+_x4_-1,_yc_-_x4_+1,_xc_+_x4_-1,_yc_+_x4_-1^)+drawLine(_xc_-_x4_+1,_yc_-_x4_+_pw_,_xc_+_x4_-1,_yc_-_x4_+1^)
 echo setPen(clBlack,psSolid,pmCopy,1^)+setBrush(clBlack,bsSolid^)+drawRoundBar(_xc_-_pw_,_yc_+_x4_-_pw_,_xc_+_pw_,_yc_+_x4_+_pw_,_pw_,_pw_^)
 goto :EOF
)
if "%pType%" == "Sublimation" goto :EOF
goto :EOF

:: Pump outlet direction
:PumpOrientation
if "%pOrien%" == "Left"  echo (_x8_=int(_x2_/8^)^)+drawLine(_be_,_yc_,_be_+_x8_,_yc_-_x8_)+drawLine(_be_,_yc_,_be_+_x8_,_yc_+_x8_)& goto :EOF
if "%pOrien%" == "Up"    echo (_x8_=int(_x2_/8^)^)+drawLine(_xc_,_be_,_xc_-_x8_,_be_+_x8_)+drawLine(_xc_,_be_,_xc_+_x8_,_be_+_x8_)& goto :EOF
if "%pOrien%" == "Right" echo (_x8_=int(_x2_/8^)^)+(_cf_=not(eq(_kc_,1)))+drawLine(_x2_-_be_-_pw_+_cf_,_yc_,_x2_-_be_-_pw_+_cf_-_x8_,_yc_-_x8_)+drawLine(_x2_-_be_-_pw_+_cf_,_yc_,_x2_-_be_-_pw_+_cf_-_x8_,_yc_+_x8_)& goto :EOF
if "%pOrien%" == "Down"  echo (_x8_=int(_x2_/8^)^)+(_cf_=not(eq(_kc_,1)))+drawLine(_xc_,_y2_-_be_-_pw_+_cf_,_xc_-_x8_,_y2_-_be_-_pw_+_cf_-_x8_)+drawLine(_xc_,_y2_-_be_-_pw_+_cf_,_xc_+_x8_,_y2_-_be_-_pw_+_cf_-_x8_)
goto :EOF

::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:: Vacuometer generation :::::::::::::::::::::::::::::::::::::::::::::::::::::::
::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:BeginVacuometerSection
echo [Painter(v).GostLib.%GostElementType%.%vmType%.%vmOrien%]
echo var glVacuometerBevel,glVacuometerFillColor,glVacuometerLineColor
if "%vmType%" == "Penning"    call :VacuometerPenningBegin& goto :EOF
if "%vmType%" == "Thermal"    call :VacuometerThermalBegin& goto :EOF
if "%vmType%" == "Ionization" call :VacuometerIonizatBegin& goto :EOF
goto :EOF

:: Penning vacuometer
:VacuometerPenningBegin
echo (_x1_=glVacuometerBevel)+(_y1_=_x1_)+(_x2_=min(sensorWidth(),sensorHeight())-_x1_)+(_y2_=_x2_)+(_c1_=glVacuometerFillColor)+(_c2_=glVacuometerLineColor)
echo (_rd_=0.5*(_x2_-_x1_))+(_cx_=_x1_+_rd_)+(_cy_=_cx_)+(_f2_=pi-2*pi/2)+(_f4_=pi-4*pi/2)+(_fb_=pi*0.868)+(_be_=lt(_x2_,40-_x1_)*1.5+ge(_x2_,40-_x1_)*2.5)+(_xm_=min(sensorWidth(),sensorHeight()))+(_ym_=_xm_)
echo setPen(clGray,psSolid,pmMask,1)+setBrush(_c1_,bsClear)+drawEllipse(_x1_-1,_y1_-1,_x2_+1,_y2_+1)+setPen(clBlack,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawEllipse(_x1_,_y1_,_x2_,_y2_)
echo setPen(_c2_,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawBar(_cx_+_rd_*sin(_f2_-_fb_)-_be_,_cy_+_rd_*sin(_f2_-_fb_)-_be_,_cx_+_rd_*sin(_f4_-_fb_)+_be_+0.5,_cy_+_rd_*sin(_f4_-_fb_)+_be_+0.5)
goto :EOF

:: Ionization vacuometer
:VacuometerIonizatBegin
echo (_x1c_=glVacuometerBevel)+(_y1c_=_x1c_)+(_x2c_=min(sensorWidth(),sensorHeight())-_x1c_)+(_y2c_=_x2c_)+(_c1_=glVacuometerFillColor)+(_c2_=glVacuometerLineColor)
echo setPen(clGray,psSolid,pmMask,1)+setBrush(_c1_,bsClear)+drawEllipse(_x1c_-1,_y1c_-1,_x2c_+1,_y2c_+1)+setPen(clBlack,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawEllipse(_x1c_,_y1c_,_x2c_,_y2c_)
echo (_x2_=min(sensorWidth(),sensorHeight()))+(_x2_=_x2_+eq(frac(_x2_/2),0))+(_y2_=_x2_)+(_xc_=int(_x2_/2))+(_yc_=int(_y2_/2))+(_x12_=int(_x2_/12))+(_2x12_=2*_x12_)+(_y4_=int(_y2_/4))+setPen(_c2_,psSolid,pmCopy,2)
goto :EOF

:: Thermal vacuometer
:VacuometerThermalBegin
echo (_x1_=glVacuometerBevel)+(_y1_=_x1_)+(_x2_=min(sensorWidth(),sensorHeight())-_x1_)+(_y2_=_x2_)+(_c1_=glVacuometerFillColor)+(_c2_=glVacuometerLineColor)
echo setPen(clGray,psSolid,pmMask,1)+setBrush(_c1_,bsClear)+drawEllipse(_x1_-1,_y1_-1,_x2_+1,_y2_+1)+setPen(clBlack,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawEllipse(_x1_,_y1_,_x2_,_y2_)
echo (_x2_=min(sensorWidth(),sensorHeight()))+(_x2_=_x2_+eq(frac(_x2_/2),0))+(_y2_=_x2_)+(_xc_=int(_x2_/2))+(_yc_=int(_y2_/2))+(_x14_=int(_x2_/14))+setPen(_c2_,psSolid,pmCopy,2)
echo drawLine(_xc_+_x14_,_yc_-5*_x14_,_xc_+_x14_,_yc_-3*_x14_)+drawLine(_xc_-_x14_,_yc_-3*_x14_,_xc_+_x14_,_yc_-3*_x14_)+drawLine(_xc_-_x14_,_yc_-3*_x14_-1,_xc_-_x14_,_yc_-_x14_)
echo drawLine(_xc_-_x14_,_yc_-_x14_,_xc_+_x14_,_yc_-_x14_)+drawLine(_xc_+_x14_,_yc_-_x14_,_xc_+_x14_,_yc_+_x14_)+drawLine(_xc_-_x14_,_yc_+_x14_,_xc_+_x14_,_yc_+_x14_)
echo drawLine(_xc_-_x14_,_yc_+_x14_-1,_xc_-_x14_,_yc_+3*_x14_)+drawLine(_xc_-_x14_,_yc_+3*_x14_,_xc_+_x14_,_yc_+3*_x14_)+drawLine(_xc_+_x14_,_yc_+3*_x14_,_xc_+_x14_,_yc_+5*_x14_-1)
goto :EOF

:: Final vacuometer generation
:FinalVacuometerSection
if "%vmType%" == "Penning" (
 echo setPen(_c2_,psSolid,pmCopy,1^)+setBrush(_c2_,bsSolid^)+drawEllipse(_xm_*0.5-2,_ym_*0.5-2,_xm_*0.5+2,_ym_*0.5+2^)
)
echo (glVacuometerBevel=4)+(glVacuometerFillColor=clGray)+(glVacuometerLineColor=clBlack)
echo []
goto :EOF

:: Various Vacuometer types generation
:ExecVacuometerGenerate
if "%vmType%" == "Thermal" goto :EOF
if "%vmType%" == "Penning"    call :ExecPenningGenerate& goto :EOF
if "%vmType%" == "Ionization" call :ExecIonizationGenerate& goto :EOF
goto :EOF

:ExecIonizationGenerate
if "%vmOrien%" == "Up" (
 echo drawLine(_x1c_,_yc_-1,_x1c_+_2x12_,_yc_^)+drawLine(_xc_-_x12_-1,_yc_-1,_xc_+_x12_-1,_yc_^)+drawLine(_x2c_-_2x12_-2,_yc_-1,_x2c_-2,_yc_^)+(_x4_=int(_x2c_/4^)^)+(_x5_=int(_x2c_/5^)^)+(_x11_=int(_x2c_/11^)^)
 echo drawLine(_xc_-_x5_,_yc_-_x4_-1,_xc_+_x5_,_yc_-_x4_^)+drawArc(_xc_-_x4_,_yc_+_x4_-_x11_,_xc_+_x4_,_yc_+3*_x4_-_x11_,_xc_+_x4_,_yc_+_x4_-_x11_,_xc_-_x4_,_yc_+_x4_-_x11_^)
 goto :EOF
)
if "%vmOrien%" == "Down" (
 echo drawLine(_x1c_,_yc_-1,_x1c_+_2x12_,_yc_^)+drawLine(_xc_-_x12_-1,_yc_-1,_xc_+_x12_-1,_yc_^)+drawLine(_x2c_-_2x12_-2,_yc_-1,_x2c_-2,_yc_^)+(_x4_=int(_x2c_/4^)^)+(_x5_=int(_x2c_/5^)^)+(_x11_=int(_x2c_/11^)^)
 echo drawLine(_xc_-_x5_,_yc_+_x4_-1,_xc_+_x5_,_yc_+_x4_^)+drawArc(_xc_-_x4_,_yc_-3*_x4_+_x11_,_xc_+_x4_,_yc_-_x4_+_x11_,_xc_-_x4_,_yc_-_x4_+_x11_,_xc_+_x4_,_yc_-_x4_+_x11_^)
 goto :EOF
)
if "%vmOrien%" == "Right" (
 echo drawLine(_xc_,_y1c_,_xc_,_y1c_+_2x12_^)+drawLine(_xc_,_yc_-_x12_-1,_xc_,_yc_+_x12_-1^)+drawLine(_xc_,_y2c_-_2x12_-2,_xc_,_y2c_-2^)+(_x4_=int(_x2c_/4^)^)+(_x5_=int(_x2c_/5^)^)+(_x11_=int(_x2c_/11^)^)
 echo drawLine(_xc_+_x4_,_yc_-_x5_,_xc_+_x4_,_yc_+_x5_^)+drawArc(_xc_-3*_x4_+_x11_,_yc_-_x4_,_xc_-_x4_+_x11_,_yc_+_x4_,_xc_-_x4_+_x11_,_yc_+_x4_,_xc_-_x4_+_x11_,_yc_-_x4_^)
 goto :EOF
)
if "%vmOrien%" == "Left" (
 echo drawLine(_xc_,_y1c_,_xc_,_y1c_+_2x12_^)+drawLine(_xc_,_yc_-_x12_-1,_xc_,_yc_+_x12_-1^)+drawLine(_xc_,_y2c_-_2x12_-2,_xc_,_y2c_-2^)+(_x4_=int(_x2c_/4^)^)+(_x5_=int(_x2c_/5^)^)+(_x11_=int(_x2c_/11^)^)
 echo drawLine(_xc_-_x4_,_yc_-_x5_,_xc_-_x4_,_yc_+_x5_^)+drawArc(_xc_+_x4_-_x11_,_yc_-_x4_,_xc_+3*_x4_-_x11_,_yc_+_x4_,_xc_+_x4_-_x11_,_yc_-_x4_,_xc_+_x4_-_x11_,_yc_+_x4_^)
 goto :EOF
)
goto :EOF

:ExecPenningGenerate
if "%vmOrien%" == "Up"    echo setPen(_c1_,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawBar(_cx_+_rd_*sin(_f2_-_fb_)-_be_,_y2_/2+_y2_*0.1,_cx_+_rd_*sin(_f4_-_fb_)+_be_+0.5,_cy_+_rd_*sin(_f4_-_fb_)+_be_+0.5)& goto :EOF
if "%vmOrien%" == "Down"  echo setPen(_c1_,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawBar(_cx_+_rd_*sin(_f2_-_fb_)-_be_,_cy_+_rd_*sin(_f2_-_fb_)-_be_,_cx_+_rd_*sin(_f4_-_fb_)+_be_+0.5,_y2_/2+_be_+0.5)& goto :EOF
if "%vmOrien%" == "Right" echo setPen(_c1_,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawBar(_cx_+_rd_*sin(_f2_-_fb_)-_be_,_cy_+_rd_*sin(_f2_-_fb_)-_be_,_x2_/2+_be_+0.5,_cy_+_rd_*sin(_f4_-_fb_)+_be_+0.5)& goto :EOF
if "%vmOrien%" == "Left"  echo setPen(_c1_,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawBar(_x2_/2+_x2_*0.1,_cy_+_rd_*sin(_f2_-_fb_)-_be_,_cx_+_rd_*sin(_f4_-_fb_)+_be_+0.5,_cy_+_rd_*sin(_f4_-_fb_)+_be_+0.5)& goto :EOF
goto :EOF

::::::::::::::::::::::::
:: Compressor generation
::::::::::::::::::::::::
:BeginCompressorSection
echo [Painter(v).GostLib.%GostElementType%.%cmOrien%]
echo var glCompressorBevel,glCompressorFillColor,glCompressorLineColor
echo (_x1b_=glCompressorBevel)+(_y1b_=_x1b_)+(_x2b_=min(sensorWidth(),sensorHeight())-_x1b_)+(_y2b_=_x2b_)+(_c1_=glCompressorFillColor)+(_c2_=glCompressorLineColor)
echo setPen(clGray,psSolid,pmMask,1)+setBrush(_c1_,bsClear)+drawEllipse(_x1b_-1,_y1b_-1,_x2b_+1,_y2b_+1)+setPen(clBlack,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawEllipse(_x1b_,_y1b_,_x2b_,_y2b_)
echo (_x2_=min(sensorWidth(),sensorHeight()))+(_x2_=_x2_+eq(frac(_x2_/2),0))+(_y2_=_x2_)+(_xc_=int(_x2_/2))+(_yc_=int(_y2_/2))+(_x5_=int(_x2_/5))+(_x10_=int(_x2_/10))+setPen(_c2_,psSolid,pmCopy,2)
goto :EOF
:: Final Compressor generation
:FinalCompressorSection
echo (glCompressorBevel=4)+(glCompressorFillColor=clGray)+(glCompressorLineColor=clBlack)
echo []
goto :EOF
:: Various Compressor orientation
:ExecCompressorGenerate
if "%cmOrien%" == "Up"    echo drawPoly(_xc_,_y1b_,plAddPoint)+drawPoly(_xc_-_x10_-1,_y1b_+_x5_+1,plAddPoint)+drawPoly(_xc_+_x10_+1,_y1b_+_x5_,plAddPoint)+drawPoly(_xc_,_y1b_,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)& goto :EOF
if "%cmOrien%" == "Down"  echo drawPoly(_xc_,_y2b_-1,plAddPoint)+drawPoly(_xc_-_x10_-1,_y2b_-_x5_-2,plAddPoint)+drawPoly(_xc_+_x10_+1,_y2b_-_x5_-1,plAddPoint)+drawPoly(_xc_,_y2b_-1,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)& goto :EOF
if "%cmOrien%" == "Right" echo drawPoly(_x2b_-1,_yc_,plAddPoint)+drawPoly(_x2b_-_x5_-1,_yc_-_x10_-1,plAddPoint)+drawPoly(_x2b_-_x5_-1,_yc_+_x10_+1,plAddPoint)+drawPoly(_x2b_-1,_yc_,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)& goto :EOF
if "%cmOrien%" == "Left"  echo drawPoly(_x1b_,_yc_,plAddPoint)+drawPoly(_x1b_+_x5_+1,_yc_-_x10_-1,plAddPoint)+drawPoly(_x1b_+_x5_+1,_yc_+_x10_+1,plAddPoint)+drawPoly(_x1b_,_yc_,plAddPoint)+drawPoly(0,0,plPolygon)+drawPoly(0,0,plClear)& goto :EOF
goto :EOF

::::::::::::::::::::::::::
:: LeakDetector generation
::::::::::::::::::::::::::
:ExecLeakDetectorGenerate
echo [Painter(v).GostLib.%GostElementType%]
echo var glLeakDetBevel,glLeakDetFillColor,glLeakDetLineColor
echo (_x1_=glLeakDetBevel)+(_y1_=_x1_)+(_x2_=min(sensorWidth(),sensorHeight())-_x1_)+(_y2_=_x2_)+(_c1_=glLeakDetFillColor)+(_c2_=glLeakDetLineColor)
echo (_a_=2/5*_x2_)+(_b_=_x2_/5)+(_c_=(_x2_-_a_)/2)+(_xm_=min(sensorWidth(),sensorHeight()))
echo setPen(_c2_,psSolid,pmCopy,2)+setBrush(_c1_,bsSolid)+drawRect(_x1_+1,_y1_+1,_x2_,_y2_)+setBrush(_c1_,bsClear)+drawRect(_x1_+1,_y1_+_c_-1,_x1_+1+_b_,_y2_-_c_+1)+drawRect(_x2_-_b_,_y1_+_c_-1,_x2_,_y2_-_c_+1)
echo setPen(_c2_,psSolid,pmCopy,1+ge(_xm_,40))+drawLine(_x1_+_b_,_y1_+_c_-1,_x1_,_y2_-_c_)+drawLine(_x2_-1,_y1_+_c_-1,_x2_-_b_-1,_y2_-_c_)+drawArrow(_xm_/2,_y2_-_c_,_xm_/2,_y1_+_c_/2,_c_/2,3)
echo (glLeakDetBevel=3)+(glLeakDetFillColor=clGray)+(glLeakDetLineColor=clBlack)
echo []
goto :EOF

::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
:DeleteFiles
if "%~1" == "" goto :EOF
if exist "%~1" del /f /q "%~1"
shift & goto :DeleteFiles
goto :EOF
