 {
 ***********************************************************************
 OpcuaSrv program - wrapper to integrate server opcuasrv.py to crwdaq.
 ***********************************************************************
 Next text uses by @Help command. Do not remove it.
 ***********************************************************************
[@Help]
|StdIn Command list: "@cmd=arg" or "@cmd arg"
|********************************************************
| @OpcServName n   - get/set server name (n), like OpcuaSrv
| @OpcEndpoint u   -  get/set server endpoint URL (u), like opc.tcp://0.0.0.0:4840/daqgroup/crwdaq/server
| @OpcXmlModel x   -  get/set server XML model (x), like ./opcuasrv.xml
| @OpcBooking p    - data change booking for nodes in pattern p
| @OpcServing      - start OPCUA data serving
| @OpcInform t     - print information on topic t
| @OpcNodeRead n   - send request to read node n
| @OpcNodeSend n v - send request to write v to node n
| @OpcNodePoke n s - poke node n from source s(:t|:y|:x)
| @OpcSubInter p   - set subscription interval
|********************************************************
[]
 }
program opcuasrv;                { OPC UA Server program            }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 {$I _con_StdPyApp}              { <-- Standard Python Application  }
 {------------------------------------------------------------------}
 patShortNodeId = '/^ns(\d+)(i\d+|s\S+|g\S+)$/i';       {ShortNodeId}
 patCanonNodeId = '/^ns=(\d+);(i=\d+|s=\S+|g=\S+)$/i';  {CanonNodeId}
 patStrngNodeId = '/^ns=?(\d+);?(s|g)/i'; { NodeId is (string|guid) }
 patFiltTooltip = '/^(info:|information:|warn:|warning:|error:)\s/i';
 patFiltConsole = '/^(info:|information:|warn:|warning:|error:)\s/i';
 procStatesList = 'DEAD STARTING PREPARING RUNNING STOPPED RESTARTING';

type
 {------------------------------}{ Declare uses program types:      }
 {$I _typ_StdLibrary}            { Include all Standard types,      }
 {------------------------------}{ And add User defined types:      }

var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 {$I _var_StdPyApp}              { <-- Standard Python Application  }
 {------------------------------------------------------------------}
 cmd_OpcServName   : Integer;    { @OpcServName                     }
 cmd_OpcEndpoint   : Integer;    { @OpcEndpoint                     }
 cmd_OpcXmlModel   : Integer;    { @OpcXmlModel                     }
 cmd_OpcNamespace  : Integer;    { @OpcNamespace                    }
 cmd_OpcNspArray   : Integer;    { @OpcNspArray                     }
 cmd_OpcNspIndex   : Integer;    { @OpcNspIndex                     }
 cmd_OpcBooking    : Integer;    { @OpcBooking                      }
 cmd_OpcServing    : Integer;    { @OpcServing                      }
 cmd_OpcNodeSend   : Integer;    { @OpcNodeSend                     }
 cmd_OpcNodeRead   : Integer;    { @OpcNodeRead                     }
 cmd_OpcNodePoke   : Integer;    { @OpcNodePoke                     }
 cmd_OpcWritable   : Integer;    { @OpcWritable                     }
 cmd_OpcInform     : Integer;    { @OpcInform                       }
 cmd_Error         : Integer;    { @Error                           }
 cmd_Warning       : Integer;    { @Warning                         }
 cmd_OpcSubInter   : Integer;    { @OpcSubInter                     }
 cmd_OpcDataVary   : Integer;    { @OpcDataVary                     }
 cmd_OpcDataRead   : Integer;    { @OpcDataRead                     }
 cmd_OpcDataSend   : Integer;    { @OpcDataSend                     }
 cmd_OpcDataBook   : Integer;    { @OpcDataBook                     }
 rexShortNodeId    : Integer;    { RegExp to detect short NodeId    }
 rexCanonNodeId    : Integer;    { RegExp to detect canon NodeId    }
 rexStrngNodeId    : Integer;    { RegExp to detect string NodeId   }
 srvTrigger        : String;     { StartServingTrigger              }
 srvSection        : String;     { StartServingSection              }
 srvPattern        : String;     { StartServingPattern              }
 rexPattern        : Integer;    { RegExp for StartServing commands }
 rexFiltTooltip    : Integer;    { RegExp for Filter Tooltip        }
 rexFiltConsole    : Integer;    { RegExp for Filter Console        }
 strFiltTooltip    : String;     { RegExp for Filter Tooltip        }
 strFiltConsole    : String;     { RegExp for Filter Console        }
 srvTriggerTime    : Real;       { Time to run serving trigger      }
 srvTriggerDelay   : Integer;    { Serving trigger delay, ms        }
 mapAliasToNode    : Integer;    { Map aliases to nodes             }
 mapNodeToAlias    : Integer;    { Map nodes to aliases             }
 mapNodeToIdent    : Integer;    { Map nodes to identifiers         }
 mapNodeLinks      : Integer;    { Map for node links               }
 mapNodeReads      : Integer;    { Map for node data reads          }
 mapNodeSends      : Integer;    { Map for node data writes         }
 mapNodeBooks      : Integer;    { Map for node bookings            }
 statReadRate      : Integer;    { Read rate - reads  per second    }
 statSendRate      : Integer;    { Send rate - writes per second    }
 tagOpcuaSrvStat   : Integer;    { tagOpcuaSrvStat string tag       }
 tipInfoDelay      : Integer;    { Tooltip information delay, ms    }
 tipWarnDelay      : Integer;    { Tooltip warning delay, ms        }
 tipErroDelay      : Integer;    { Tooltip error delay, ms          }
 procStateText     : String;     { ProcessState: see procStatesList }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_StdPyApp}              { <-- Standard Python Application  }

 {
 Check nodeId is short/canon/valid.
 ShortId: ns2i2003       - see patShortNodeId.
 CanonId: ns=2;i=2003    - see patCanonNodeId.
 ValidId: canon or short.
 }
 function IsShortNodeId(var s:String):Boolean;
 begin
  IsShortNodeId:=IsLexeme(s,rexShortNodeId);
 end;
 function IsCanonNodeId(var s:String):Boolean;
 begin
  IsCanonNodeId:=IsLexeme(s,rexCanonNodeId);
 end;
 function IsStrngNodeId(var s:String):Boolean;
 begin
  IsStrngNodeId:=IsLexeme(s,rexStrngNodeId);
 end;
 function IsValidNodeId(var s:String):Boolean;
 var cond:Boolean;
 begin
  if IsLexeme(s,rexShortNodeId) then cond:=True else
  if IsLexeme(s,rexCanonNodeId) then cond:=True else cond:=False;
  IsValidNodeId:=cond;
 end;
 function IsValidAlias(var s:String):Boolean;
 begin
  IsValidAlias:=IsLexeme(s,lex_name);
 end;
 procedure CanonToShortNodeId(var s:String);
 begin
  if IsCanonNodeId(s) then begin
   s:=StringReplace(s,'=','',rfReplaceAll);
   s:=StringReplace(s,';','',rfReplaceAll);
  end;
 end;
 {
 Get node linked attribute 'how' as lead char,
 i.e. m/p/t/r=monitor/polling/trigger/refresh.
 That is trigger mode - "how to monitor data".
 }
 function GetNodeAttrHow(var node:String):Char;
 var how:Integer;
 begin
  how:=hashlist_getlink(mapNodeLinks,node);
  GetNodeAttrHow:=LoCase(Chr(how));
 end;
 {
 Translate NodeId to Identifier.
 }
 procedure TranslateNodeIdToIdent(var nodeId,Ident:String);
 begin
  if IsStrngNodeId(nodeId)
  then Ident:=hashlist_getpara(mapNodeToIdent,nodeId)
  else Ident:=nodeId;
 end;
 {
 Translate alias to nodeId and back.
 }
 procedure TranslateAliasToNodeId(var s:String);
 begin
  if IsValidAlias(s) then
  if not IsShortNodeId(s) then begin
   if (hashlist_indexof(mapAliasToNode,s)>=0)
   then s:=hashlist_getpara(mapAliasToNode,s);
  end;
 end;
 procedure TranslateNodeIdToAlias(var s:String);
 begin
  if IsValidNodeId(s) then begin
   if IsCanonNodeId(s) then CanonToShortNodeId(s);
   if (hashlist_indexof(mapNodeToAlias,s)>=0)
   then s:=hashlist_getpara(mapNodeToAlias,s);
  end;
 end;
 {
 Add alias of node to mapping tables.
 }
 function AddAliasOfNode(alias,node:String):Boolean;
 var status:Boolean;
 begin
  status:=false;
  if IsValidAlias(alias) then begin
   if IsValidNodeId(node) then begin
    if IsCanonNodeId(node) then CanonToShortNodeId(node);
    if (hashlist_indexof(mapAliasToNode,alias)<0) then begin
     if (hashlist_indexof(mapNodeToAlias,node)<0) then begin
      if hashlist_setpara(mapAliasToNode,alias,node) then
      if hashlist_setpara(mapNodeToAlias,node,alias) then
      if hashlist_setpara(mapNodeToIdent,node,node)  then status:=true;
      if status
      then Details('Alias '+alias+' = '+node)
      else Trouble('Alias '+alias+' = '+node);
     end else Trouble('Duplicate node: '+node);
    end else Trouble('Duplicate alias: '+alias);
   end else Trouble('Invalid nodeId: '+node);
  end else Trouble('Invalid alias: '+alias);
  AddAliasOfNode:=status;
 end;
 {
 Read alias to node mapping from section (sect).
 [sect]
 Alias WaveSin = ns2i2003
 Alias WaveCos = ns2i2003
 []
 }
 procedure ParseAliasNodeSection(sect:String);
 var lines,mode,i,n,m:Integer; line,sn,sv:String;
  procedure Cleanup;
  begin
   line:=''; sn:=''; sv:='';
  end;
 begin
  Cleanup;
  if (sect='') then sect:=StrFmt('[%s]',DevName);
  if IsLexeme(sect,lex_section) then begin
   mode:=risModeTrimL+risModeTrimR+risModeRemComm; n:=0; m:=0;
   lines:=ReadIniSection(text_new,mode,ParamStr('DaqConfigFile'),sect);
   for i:=0 to text_numln(lines)-1 do begin
    line:=text_getln(lines,i);
    if IsSameText(ExtractWord(1,line),'Alias') then
    if ExtractNameValuePair(SkipWords(1,line),sn,sv,'=',3)>0 then
    if IsValidAlias(sn) then if IsValidNodeId(sv) then
    if AddAliasOfNode(sn,sv) then begin
     n:=n+1;
    end;
    m:=m+1;
   end;
   FreeAndZero(lines);
   Success(StrFmt('Alias map: %d alias(es) from ',n)+StrFmt('%d line(s)',m));
  end;
  Cleanup;
 end;
 {
 Parse Link node ... expression
 Link node WaveSin  with AnalogInput   0 tag WAVE.AMP refresh 5000 message @DevPost &DEMO @OpcData $alias $data
 Link node WaveCos  with AnalogOutput  1 tag WAVE.SIN trigger 5000 message @DevPost &DEMO @OpcData $alias $data
 Link node ns2i2006 with AnalogOutput  3 tag WAVE.SIN monitor 5000 message @DevPost &DEMO @OpcData $node $data
 Link node ns2i2007 with DigitalOutput 3 tag WAVE.SIN polling 5000 message @DevPost &DEMO @OpcData $node $data
 monitor - readonly, subscribe data change or @OpcDataRead by timer
 polling - readonly, @OpcDataRead by timer, readonly
 trigger - writable, @OpcDataRead by timer or by data change trigger
 refresh - writable, @OpcDataSend by timer
 }
 function ParseLinkNodeLine(line:String):String;
 var iai,nai,idi,ndi,iao,nao,ido,ndo,itg,tag,imo,nmo,itr,ntr,ire,nre,ipo,npo,ims,nms,per,h0w,sep:Integer;
     atr,msg,nod,ali:String; fil:Real; how,ttr:Char;
  procedure Cleanup;
  begin
   iai:=0; nai:=-1; idi:=0; ndi:=-1; iao:=0; nao:=-1; ido:=0; ndo:=-1; itg:=0; tag:=-1;
   imo:=0; nmo:=-1; itr:=0; ntr:=-1; ire:=0; nre:=-1; ipo:=0; npo:=-1; ims:=0; nms:=-1; per:=0;
   atr:=''; msg:=''; nod:=''; ali:=''; how:='?'; ttr:='?';
  end;
 begin
  Cleanup;
  nod:=LoCaseStr(ExtractWord(3,line)); TranslateAliasToNodeId(nod);
  ali:=LoCaseStr(ExtractWord(3,line)); TranslateNodeIdToAlias(ali);
  if IsSameText(ExtractWord(1,line),'Link') then
  if IsSameText(ExtractWord(2,line),'node') then
  if IsSameText(ExtractWord(4,line),'with') then
  if IsValidNodeId(nod) then begin
   iai:=WordIndex('AnalogInput',line);   if (iai>0) then nai:=iValDef(ExtractWord(iai+1,line),nai);
   idi:=WordIndex('DigitalInput',line);  if (idi>0) then ndi:=iValDef(ExtractWord(idi+1,line),ndi);
   iao:=WordIndex('AnalogOutput',line);  if (iao>0) then nao:=iValDef(ExtractWord(iao+1,line),nao);
   ido:=WordIndex('DigitalOutput',line); if (ido>0) then ndo:=iValDef(ExtractWord(ido+1,line),ndo);
   itg:=WordIndex('Tag',line);           if (itg>0) then tag:=FindTag(ExtractWord(itg+1,line));
   imo:=WordIndex('Monitor',line);       if (imo>0) then nmo:=iValDef(ExtractWord(imo+1,line),nmo);
   itr:=WordIndex('Trigger',line);       if (itr>0) then ntr:=iValDef(ExtractWord(itr+1,line),ntr);
   ire:=WordIndex('Refresh',line);       if (ire>0) then nre:=iValDef(ExtractWord(ire+1,line),nre);
   ipo:=WordIndex('Polling',line);       if (ipo>0) then npo:=iValDef(ExtractWord(ipo+1,line),npo);
   ims:=WordIndex('Message',line);       if (ims>0) then nms:=WordCount(line)-ims;
   if (nmo>=0) then begin atr:=atr+StrFmt('how=monitor %d;',nmo); per:=nmo; end else
   if (ntr>=0) then begin atr:=atr+StrFmt('how=trigger %d;',ntr); per:=ntr; end else
   if (nre>=0) then begin atr:=atr+StrFmt('how=refresh %d;',nre); per:=nre; end else
   if (npo>=0) then begin atr:=atr+StrFmt('how=polling %d;',npo); per:=npo; end else atr:='';
   how:=LoCase(StrFetch(CookieScan(atr,'how',sep),1));
   if (Pos(how,'mptr')=0) then begin
    Trouble('Bad trigger: '+how+' in line '+line);
    atr:='';
   end;
   if (atr<>'') then begin
    if (Pos(how,'mp')>0) then begin
     if (nai>=0) then begin Problem('Warning: AI for readonly '+nod); nai:=-1; end;
     if (ndi>=0) then begin Problem('Warinng: DI for readonly '+nod); ndi:=-1; end;
    end;
    if (Pos(how,'tr')>0) then begin
     if (nai>=0) and (ndi>=0) then Problem('Warinng: both AI,DI linked to '+nod);
     if (nai>=0) or (ndi>=0) then ttr:='y';
     if (itr>0) then begin
      case WordIndex(ExtractWord(itr+2,line),':x :y :t') of
       1: if (nai>=0) or (ndi>=0) then ttr:='x';
       2: if (nai>=0) or (ndi>=0) then ttr:='y';
       3: if (tag>0) then ttr:='t';
      end;
     end else
     if (ire>0) then begin
      case WordIndex(ExtractWord(ire+2,line),':x :y :t') of
       1: if (nai>=0) or (ndi>=0) then ttr:='x';
       2: if (nai>=0) or (ndi>=0) then ttr:='y';
       3: if (tag>0) then ttr:='t';
      end;
     end;
     if (Pos(ttr,'xyt')>0) then atr:=atr+StrFmt('ttr=%s;',ttr);
    end;
   end;  
   if (atr<>'') then begin
    msg:=Trim(SkipWords(ims,line));
    if (nai>=0) then atr:=atr+StrFmt('nai=%d;',nai);
    if (nao>=0) then atr:=atr+StrFmt('nao=%d;',nao);
    if (ndi>=0) then atr:=atr+StrFmt('ndi=%d;',ndi);
    if (ndo>=0) then atr:=atr+StrFmt('ndo=%d;',ndo);
    if (tag>0)  then atr:=atr+StrFmt('tag=%d;',tag);
   end;
   if (atr<>'') then begin
    ali:=LoCaseStr(ali);
    if IsValidAlias(ali) then
    if not IsSameText(ali,nod) then
    atr:=atr+StrFmt('ali=%s;',ali);
    atr:=atr+StrFmt('nod=%s;',nod);
   end;
   if (atr<>'') then begin
    if (nms>0) then
    if IsLexeme(msg,lex_atcmnd) then
    atr:=atr+StrFmt('msg=%s;',msg);
   end;
   if (atr<>'') then begin
    sep:=Ord(';'); fil:=MaxReal; h0w:=Ord(how);
    bNul(hashlist_setpara(mapNodeLinks,nod,atr)); // attributes
    bNul(hashlist_setlink(mapNodeLinks,nod,h0w)); // how to poll
    bNul(hashlist_setdata(mapNodeLinks,nod,fil)); // trigger stamp
    bNul(hashlist_setlink(mapNodeReads,nod,per)); // period of reads
    bNul(hashlist_setdata(mapNodeReads,nod,0.0)); // time stamp of reads
    bNul(hashlist_setlink(mapNodeSends,nod,per)); // period of writes
    bNul(hashlist_setdata(mapNodeSends,nod,0.0)); // time stamp of writes
    Details('Link node '+ExtractWord(3,line)+': '+atr);
   end else begin
    Trouble('Bad link: '+line);
   end;  
  end;
  ParseLinkNodeLine:=atr;
  Cleanup;
 end;
 procedure ParseLinkNodeSection(sect:String);
 var lines,mode,i,n,m:Integer; line,link:String;
  procedure Cleanup;
  begin
   line:=''; link:='';
  end;
 begin
  Cleanup;
  if (sect='') then sect:=StrFmt('[%s]',DevName);
  if IsLexeme(sect,lex_section) then begin
   mode:=risModeTrimL+risModeTrimR+risModeRemComm; n:=0; m:=0;
   lines:=ReadIniSection(text_new,mode,ParamStr('DaqConfigFile'),sect);
   for i:=0 to text_numln(lines)-1 do begin
    line:=text_getln(lines,i);
    link:=ParseLinkNodeLine(line);
    if (link<>'') then begin
     n:=n+1;
    end;
    m:=m+1;
   end;
   FreeAndZero(lines);
   Success(StrFmt('LinkNodes: %d link(s) from ',n)+StrFmt('%d line(s)',m));
  end;
  Cleanup;
 end;
 {
 Translate @OpcServing  pattern to set of @PyAsk .. commands.
 Translate @OpcEndpoint pattern to set of @PyAsk .. commands.
 Translate @OpcXmlModel pattern to set of @PyAsk .. commands.
 Translate @OpcBooking  pattern to set of @PyAsk .. commands.
 }
 function TranslateOpcCommands(var arg:String; prefix:Boolean):Boolean;
 var cond:Boolean; map,rexp,iw,ik:Integer; line,expr,ali,nod,cmd,url,atr,idn,per,xml,nsp:String;
  procedure Cleanup;
  begin
   line:=''; expr:=''; ali:=''; nod:=''; cmd:=''; url:=''; atr:=''; idn:=''; per:=''; xml:=''; nsp:='';
  end;
  procedure AddCmd(var arg:String; cmd:String);
  begin
   if prefix then cmd:='@PyAsk '+cmd;
   arg:=arg+Trim(cmd)+EOL;
  end;
 begin
  Cleanup;
  cond:=False;
  if (WordCount(arg)>0) then begin
   map:=hashlist_init(0);
   cmd:=ExtractWord(1,arg);
   if IsSameText(cmd,'@OpcServing') then begin
     arg:=''; cond:=true;
     AddCmd(arg,'@OpcStartSrv');
     AddCmd(arg,'@UseShortKey 1');
   end else
   if IsSameText(cmd,'@OpcServName') then begin
    arg:=''; cond:=true;
    idn:=ExtractWord(2,arg);
    AddCmd(arg,'@OpcServName '+idn);
    AddCmd(arg,'@UseShortKey 1');
   end else
   if IsSameText(cmd,'@OpcEndpoint') then begin
    url:=ExtractWord(2,arg);
    if not IsEmptyStr(url) then begin
     arg:=''; cond:=true;
     AddCmd(arg,'@OpcEndpoint '+url);
     AddCmd(arg,'@UseShortKey 1');
    end;
   end else
   if IsSameText(cmd,'@OpcNamespace') then begin
    nsp:=ExtractWord(2,arg);
    if not IsEmptyStr(nsp) then begin
     arg:=''; cond:=true;
     AddCmd(arg,'@OpcNamespace '+nsp);
     AddCmd(arg,'@UseShortKey 1');
    end;
   end else
   if IsSameText(cmd,'@OpcXmlModel') then begin
    xml:=ExtractWord(2,arg);
    if not IsEmptyStr(xml) then begin
     arg:=''; cond:=true;
     if (Pos('$',xml)+Pos('%',xml)>0)
     then xml:=AdaptFileName(ExpEnv(xml));
     xml:=DaqFileRef(xml,'.xml');
     AddCmd(arg,'@OpcXmlModel '+xml);
     AddCmd(arg,'@UseShortKey 1');
    end;
   end else
   if IsSameText(cmd,'@OpcSubInter') then begin
    per:=ExtractWord(2,arg);
    if not IsEmptyStr(per) then begin
     arg:=''; cond:=true;
     AddCmd(arg,'@OpcSubInter '+per);
     AddCmd(arg,'@UseShortKey 1');
    end;
   end else
   if IsSameText(cmd,'@OpcBooking') then begin
    for iw:=2 to WordCountDelims(arg,', ') do begin
     expr:=ExtractWordDelims(iw,arg,', ');
     rexp:=regexp_init(0,expr);
     for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
      ali:=hashlist_getkey(mapAliasToNode,ik);
      nod:=hashlist_getpara(mapAliasToNode,ali);
      if IsLexeme(ali,rexp) then bNul(hashlist_setlink(map,nod,1));
      if IsLexeme(nod,rexp) then bNul(hashlist_setlink(map,nod,1));
      if Pos('=',expr)>0 then begin
       atr:=hashlist_getpara(mapNodeLinks,nod);
       if IsLexeme(atr,rexp) then bNul(hashlist_setlink(map,nod,1));
      end;
     end;
     if (hashlist_count(map)>0) then begin
      arg:=''; cond:=True;
      for ik:=0 to hashlist_count(map)-1 do begin
       nod:=hashlist_getkey(map,ik);
       TranslateNodeIdToIdent(nod,idn);
       AddCmd(arg,'@OpcDataBook '+idn);
      end;
     end;
     FreeAndZero(rexp);
    end;
   end;
   FreeAndZero(map);
  end;
  TranslateOpcCommands:=cond;
  Cleanup;
 end;
 {
 Update OpcuaSrv Stat tag.
 }
 procedure UpdateOpcuaSrvStatTag;
 var atr:String;
 begin
  atr:='';
  if (TypeTag(tagOpcuaSrvStat)=tag_type_string) then begin
   atr:=StrFmt('State=%s;',procStateText)
       +StrFmt('ReadRate=%d;',statReadRate)
       +StrFmt('SendRate=%d;',statSendRate)
       +StrFmt('TimeStamp=%g;',mSecNow);
   bNul(sSetTag(tagOpcuaSrvStat,atr));
  end;
  atr:='';
 end;
 {
 Update process state.
 }
 procedure UpdateProcState(aState:String);
 var up:Boolean;
 begin
  up:=false;
  if (StdPyApp.Tid=0) then begin
   if (procStateText<>'DEAD') then begin
    statReadRate:=0; statSendRate:=0;
    procStateText:='DEAD';
    up:=True;
   end;
  end else begin
   if task_wait(StdPyApp.Tid,0) then begin
    if (aState<>'') then begin
     if (WordIndex(aState,procStatesList)>0) then begin
      if (procStateText<>aState) then begin
       procStateText:=aState;
       up:=True;
      end;
     end;
    end;
   end else begin
    if (procStateText<>'STOPPED') then begin
     procStateText:='STOPPED';
     up:=True;
    end;
   end;
  end;
  if up then UpdateOpcuaSrvStatTag;
 end;
 {
 Read StartServing actions from section (sect) and execute.
 }
 procedure DoStartServingActions(sect:String);
 var lines,mode,i,n,m:Integer; line:String; UseTranslation:Boolean;
  procedure Cleanup;
  begin
   line:='';
  end;
 begin
  Cleanup;
  UseTranslation:=false;
  if IsLexeme(sect,lex_section) then begin
   mode:=risModeTrimL+risModeTrimR+risModeRemComm; n:=0; m:=0;
   lines:=ReadIniSection(text_new,mode,ParamStr('DaqConfigFile'),sect);
   if UseTranslation then
   for i:=0 to text_numln(lines)-1 do begin
    line:=text_getln(lines,i);
    if TranslateOpcCommands(line,true)
    then bNul(text_putln(lines,i,Trim(line)));
   end;
   iNul(TextAssignString(lines,TextToString(lines)));
   for i:=0 to text_numln(lines)-1 do begin
    line:=text_getln(lines,i);
    if IsLexeme(line,lex_atcmnd) then
    if IsLexeme(line,rexPattern) then begin
     DevPostCmdLocal(line);
     n:=n+1;
    end;
    m:=m+1;
   end;
   FreeAndZero(lines);
   UpdateProcState('RUNNING');
   Success(StrFmt('StartServing: %d action(s) from ',n)+StrFmt('%d line(s)',m));
  end;
  Cleanup;
 end;
 {
 OPCUASRV - Perform Serving: monitor/trigger/refresh/polling.
 }
 procedure OPCUASRV_Serving;
 var i,sep,per,nai,ndi,tag:Integer; ms,tim,num,upd,x,y:Real; nod,atr,dat,idn:String; how,ttr:Char;
  procedure Cleanup;
  begin
   nod:=''; atr:=''; dat:=''; idn:='';
  end;
 begin
  Cleanup;
  if StdPyApp_Running then begin
   ms:=msecnow; sep:=Ord(';');
   if (srvTriggerTime>0) and (ms>srvTriggerTime) then begin
    Success('ServingStart trigger detected.');
    DoStartServingActions(srvSection);
    srvTriggerTime:=MaxReal;
   end;
   for i:=0 to hashlist_count(mapNodeLinks)-1 do begin
    nod:=hashlist_getkey(mapNodeLinks,i);
    if (nod<>'') then begin
     how:=GetNodeAttrHow(nod);
     if (Ord(how)>0) then begin
      case how  of
       'm','p': begin // monitor,polling
        per:=hashlist_getlink(mapNodeReads,nod);
        tim:=hashlist_getdata(mapNodeReads,nod);
        if (per>0) and (ms>=tim+per) then begin
         TranslateNodeIdToIdent(nod,idn);
         DevPostCmdLocal('@PyAsk @OpcDataRead '+idn);
         bNul(hashlist_setdata(mapNodeReads,nod,ms));
        end;
       end;
       't','r': begin // trigger,refresh
        per:=hashlist_getlink(mapNodeSends,nod);
        tim:=hashlist_getdata(mapNodeSends,nod);
        atr:=hashlist_getpara(mapNodeLinks,nod);
        num:=_Nan; x:=_Nan; y:=_Nan; ttr:='?';
        if (how='t') then begin
         if (Pos('ttr=',atr)>0) then begin
          ttr:=StrFetch(CookieScan(atr,'ttr',sep),1);
          ttr:=LoCase(ttr);
         end else ttr:='y';
        end else ttr:='y';
        if IsNan(num) then begin
         if (Pos('nai=',atr)>0) then begin
          nai:=iValDef(CookieScan(atr,'nai',sep),-1);
          if (nai>=0) then begin
           x:=getai_xn(nai); if (ttr='x') then num:=x;
           y:=getai_yn(nai); if (ttr='y') then num:=y;
           if not IsNan(num) then dat:=str(y);
          end;
         end;
        end;
        if IsNan(num) then begin
         if (Pos('ndi=',atr)>0) then begin
          ndi:=iValDef(CookieScan(atr,'ndi',sep),-1);
          if (ndi>=0) then begin
           x:=getdi_xn(ndi); if (ttr='x') then num:=x;
           y:=getdi_yn(ndi); if (ttr='y') then num:=y;
           if not IsNan(num) then dat:=str(y);
          end;
         end;
        end;
        if IsNan(num) then begin
         if (Pos('tag=',atr)>0) then begin
          tag:=FindTag(CookieScan(atr,'tag',sep));
          if (tag=0) then tag:=Val(CookieScan(atr,'tag',sep));
          if (ttr='t') then
          case TypeTag(tag) of
           tag_type_int:    begin num:=iGetTag(tag); dat:=str(num); end;
           tag_type_real:   begin num:=rGetTag(tag); dat:=str(num); end;
           tag_type_string: begin dat:=sGetTag(tag); num:=hashindexof(dat,0,0); end;
          end;
         end;
        end;
        if not IsNan(num) then begin
         if (how='t') then begin
          upd:=hashlist_getdata(mapNodeLinks,nod);
          if ShouldRefresh(upd,num)>0 then begin
           bNul(hashlist_setdata(mapNodeLinks,nod,upd));
           tim:=0; per:=1;
          end;
         end;
        end;
        if not IsNan(num) then
        if (per>0) and (ms>=tim+per) then begin
         TranslateNodeIdToIdent(nod,idn);
         DevPostCmdLocal('@PyAsk @OpcDataSend '+idn+' '+dat);
         bNul(hashlist_setdata(mapNodeSends,nod,ms));
        end;
        per:=hashlist_getlink(mapNodeReads,nod);
        tim:=hashlist_getdata(mapNodeReads,nod);
        if (per>0) and (ms>=tim+per) then begin
         TranslateNodeIdToIdent(nod,idn);
         DevPostCmdLocal('@PyAsk @OpcDataRead '+idn);
         bNul(hashlist_setdata(mapNodeReads,nod,ms));
        end;
       end;
      end;
     end;
    end;
   end;
  end;
  UpdateProcState('');
  if (SysTimer_Pulse(1000)>0) then begin
   UpdateOpcuaSrvStatTag;
   statReadRate:=0;
   statSendRate:=0;
  end;
  Cleanup;
 end;
 {
 OPCUASRV - handle @OpcDataVary/@OpcDataRead notification on node data changed.
 }
 procedure OPCUASRV_OpcDataVary(var arg:String);
 var cmd,nod,ali,dat,atr,msg:String; nao,ndo,tag,sep:Integer; how,cvr:Char; ms:Real;
  procedure Cleanup;
  begin
   cmd:=''; nod:=''; ali:=''; dat:=''; atr:=''; nao:=-1; ndo:=-1;
  end;
 begin
  Cleanup;
  if IsLexeme(arg,lex_atcmnd) then begin
   cmd:=ExtractWord(1,arg);
   cvr:=LoCase(StrFetch(cmd,9));
   if IsSameText(cmd,'@OpcDataVary')
   or IsSameText(cmd,'@OpcDataRead') then begin
    nod:=ExtractWord(2,arg); CanonToShortNodeId(nod);
    dat:=SkipWords(2,arg); ms:=msecnow; sep:=Ord(';');
    if (hashlist_indexof(mapNodeLinks,nod)>=0) then begin
     how:=GetNodeAttrHow(nod);
     case how of
      'm','p','t','r': begin // monitor,polling,trigger/refresh
       atr:=hashlist_getpara(mapNodeLinks,nod);
       if Pos(' ',dat)>0 then dat:=Trim(dat);
       if (Pos('tag=',atr)>0) then begin
        tag:=Val(CookieScan(atr,'tag',sep));
        case TypeTag(tag) of
         tag_type_int:    if IsLexeme(dat,lex_iparam) then bNul(iSetTag(tag,Val(dat)));
         tag_type_real:   if IsLexeme(dat,lex_fparam) then bNul(rSetTag(tag,rVal(dat)));
         tag_type_string: bNul(sSetTag(tag,dat));
        end;
       end;
       if IsLexeme(dat,lex_fparam) then begin
        if (Pos('nao=',atr)>0) then begin
         nao:=iValDef(CookieScan(atr,'nao',sep),-1);
         if (nao>=0) then UpdateAo(nao,time,rVal(dat));
        end;
        if (Pos('ndo=',atr)>0) then begin
         ndo:=iValDef(CookieScan(atr,'ndo',sep),-1);
         if (ndo>=0) then UpdateDo(ndo,time,rVal(dat));
        end;
       end;
       if (Pos('msg=',atr)>0) then begin
        msg:=Trim(CookieScan(atr,'msg',sep));
        if IsLexeme(msg,lex_atcmnd) then begin
         ali:=nod; TranslateNodeIdToAlias(ali);
         if (Pos('$',msg)>0) then begin
          msg:=StringReplace(msg,'$alias',ali,rfReplaceAll+rfIgnoreCase);
          msg:=StringReplace(msg,'$node',nod,rfReplaceAll+rfIgnoreCase);
          msg:=StringReplace(msg,'$data',dat,rfReplaceAll+rfIgnoreCase);
         end;
         DevPostCmdLocal(msg);
        end;
       end;
       statReadRate:=statReadRate+1;
       bNul(hashlist_setdata(mapNodeReads,nod,ms));
      end;
     end;
    end;
   end;
  end;  
  Cleanup;
 end;
 {
 OPCUASRV - handle @OpcDataBook - data change booking confirmation.
 }
 procedure OPCUASRV_OpcDataBook(var arg:String);
 var cmd,nod:String;
  procedure Cleanup;
  begin
   cmd:=''; nod:='';
  end;
 begin
  Cleanup;
  if IsLexeme(arg,lex_atcmnd) then begin
   cmd:=ExtractWord(1,arg);
   if IsSameText(cmd,'@OpcDataBook') then begin
    nod:=ExtractWordDelims(2,arg,' '+EOL); CanonToShortNodeId(nod);
    bNul(hashlist_setlink(mapNodeBooks,nod,0));
   end;
  end;
  Cleanup;
 end;
 {
 OPCUASRV - handle @OpcDataSend - data send confirmation.
 }
 procedure OPCUASRV_OpcDataSend(var arg:String);
 var cmd,nod:String;
  procedure Cleanup;
  begin
   cmd:=''; nod:='';
  end;
 begin
  Cleanup;
  if IsLexeme(arg,lex_atcmnd) then begin
   cmd:=ExtractWord(1,arg);
   if IsSameText(cmd,'@OpcDataSend') then begin
    statSendRate:=statSendRate+1;
   end;
  end;
  Cleanup;
 end;
 {
 OPCUASRV - send alias/node data value.
 }
 procedure OPCUASRV_OpcNodeSend(var arg:String);
 var nod,dat,idn:String;
  procedure Cleanup;
  begin
   nod:=''; dat:=''; idn:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   nod:=ExtractWord(1,arg);
   dat:=Trim(SkipWords(1,arg));
   TranslateAliasToNodeId(nod);
   if IsValidNodeId(nod) then begin
    case GetNodeAttrHow(nod) of
     'm','p': begin
      Trouble('Deny OpcNodeSend: '+arg);
     end;
     't','r': begin
      TranslateNodeIdToIdent(nod,idn);
      DevPostCmdLocal(Trim(StrFmt('@PyAsk @OpcDataSend %s ',idn)+dat));
      bNul(hashlist_setdata(mapNodeSends,nod,mSecNow));
     end;
    end;
   end else Problem('Bad node: '+arg);
  end;
  Cleanup;
 end;
 {
 OPCUASRV - poke alias/node data value.
 Get data value from linked tag or AI/DI and send to server. 
 }
 procedure OPCUASRV_OpcNodePoke(var arg:String);
 var nod,dat,idn,atr,src:String; tag,ndi,nai,sep:Integer; x,y,num:Real; ttr:Char;
  procedure Cleanup;
  begin
   nod:=''; dat:=''; idn:=''; atr:=''; src:='';
   tag:=0; ndi:=-1; nai:=-1; sep:=Ord(';');
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   nod:=ExtractWord(1,arg);
   src:=TrimDef(SkipWords(1,arg),':t');
   ttr:=StrFetch(src,2); num:=_Nan;
   TranslateAliasToNodeId(nod);
   if IsValidNodeId(nod) then begin
    case GetNodeAttrHow(nod) of
     'm','p': begin
      Trouble('Deny OpcNodeSend: '+arg);
     end;
     't','r': begin
      atr:=hashlist_getpara(mapNodeLinks,nod);
      case ttr of
       't': begin
        if (Pos('tag=',atr)>0) then begin
         tag:=FindTag(CookieScan(atr,'tag',sep));
         if (tag=0) then tag:=Val(CookieScan(atr,'tag',sep));
         if (ttr='t') then
         case TypeTag(tag) of
          tag_type_int:    begin num:=iGetTag(tag); dat:=str(num); end;
          tag_type_real:   begin num:=rGetTag(tag); dat:=str(num); end;
          tag_type_string: begin dat:=sGetTag(tag); num:=hashindexof(dat,0,0); end;
         end;
        end;
       end;
       'x','y': begin
        if IsNan(num) then begin
         if (Pos('nai=',atr)>0) then begin
          nai:=iValDef(CookieScan(atr,'nai',sep),-1);
          if (nai>=0) then begin
           x:=getai_xn(nai); if (ttr='x') then num:=x;
           y:=getai_yn(nai); if (ttr='y') then num:=y;
           if not IsNan(num) then dat:=str(y);
          end;
         end;
        end;
        if IsNan(num) then begin
         if (Pos('ndi=',atr)>0) then begin
          ndi:=iValDef(CookieScan(atr,'ndi',sep),-1);
          if (ndi>=0) then begin
           x:=getdi_xn(ndi); if (ttr='x') then num:=x;
           y:=getdi_yn(ndi); if (ttr='y') then num:=y;
           if not IsNan(num) then dat:=str(y);
          end;
         end;
        end;
       end;
      end;
      if not IsNan(num) then begin
       TranslateNodeIdToIdent(nod,idn);
       DevPostCmdLocal(Trim(StrFmt('@PyAsk @OpcDataSend %s ',idn)+dat));
       bNul(hashlist_setdata(mapNodeSends,nod,mSecNow));
      end else Problem('Bad command format: '+arg);
     end;
    end;
   end else Problem('Bad node: '+arg);
  end;
  Cleanup;
 end;
 {
 OPCUASRV - send request to read node.
 }
 procedure OPCUASRV_OpcNodeRead(var arg:String);
 var nod,idn:String;
  procedure Cleanup;
  begin
   nod:=''; idn:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   nod:=ExtractWord(1,arg);
   TranslateAliasToNodeId(nod);
   if IsValidNodeId(nod) then begin
    case GetNodeAttrHow(nod) of
     'm','p','t','r': begin
      TranslateNodeIdToIdent(nod,idn);
      DevPostCmdLocal(StrFmt('@PyAsk @OpcDataRead %s',idn));
      bNul(hashlist_setdata(mapNodeReads,nod,mSecNow));
     end;
    end;
   end else Problem('Bad node: '+arg);
  end;
  Cleanup;
 end;
 {
 OPCUASRV - send request to make node writable.
 }
 procedure OPCUASRV_OpcWritable(var arg:String);
 var nod,idn:String;
  procedure Cleanup;
  begin
   nod:=''; idn:='';
  end;
 begin
  Cleanup;
  if (arg<>'') then begin
   nod:=ExtractWord(1,arg);
   TranslateAliasToNodeId(nod);
   if IsValidNodeId(nod) then begin
    case GetNodeAttrHow(nod) of
     'm','p','t','r': begin
      TranslateNodeIdToIdent(nod,idn);
      DevPostCmdLocal(StrFmt('@PyAsk @OpcWritable %s',idn));
     end;
    end;
   end else Problem('Bad node: '+arg);
  end;
  Cleanup;
 end;
 {
 OPCUASRV - send request to set subscription update interval.
 }
 procedure OPCUASRV_OpcSubInter(var arg:String);
 begin
  StdPyApp_Send(Trim('@OpcSubInter '+Trim(arg))+EOL);
 end;
 {
 OPCUASRV Information.
 }
 procedure OPCUASRV_OpcInform(var arg:String);
 var w1,ali,nod,lnk,how,tst:String; wc,iw,ik,l1:Integer; ms,dt:Real; cow:Char;
  procedure Cleanup;
  begin
   w1:=''; ali:=''; nod:=''; lnk:=''; how:=''; tst:='';
  end;
 begin
  Cleanup;
  if IsSameText(arg,'*') then begin
   arg:='Alias Links Books How';
  end;
  wc:=WordCount(arg);
  if (wc=0) then begin
   Problem('No topics specified.');
   Success('Available topics are:');
   Success(' Links - table of node links.');
   Success(' Alias - alias to node table.');
   Success(' Books - nodes booking table.');
   Success(' How   - node trig how table.');
   Success(' *     - all listed topics.  ');
  end;
  for iw:=1 to wc do begin
   w1:=ExtractWord(iw,arg);
   if IsSameText(w1,'Alias') then begin
    Success('Alias to Node table:'); l1:=1;
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     l1:=imax(l1,length(ali));
    end;   
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     nod:=hashlist_getpara(mapAliasToNode,ali);
     writeln(' '+RightPad(ali,l1,' ')+' = '+nod);
    end;
   end;
   if IsSameText(w1,'Links') then begin
    Success('Table of Node links:'); l1:=1;
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     l1:=imax(l1,length(ali));
    end;
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     nod:=hashlist_getpara(mapAliasToNode,ali);
     lnk:=hashlist_getpara(mapNodeLinks,nod);
     writeln(' '+RightPad(ali,l1,' ')+' - '+lnk);
    end;
   end;
   if IsSameText(w1,'Books') then begin
    Success('Nodes Booking table:'); l1:=1;
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     l1:=imax(l1,length(ali));
    end;   
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     nod:=hashlist_getpara(mapAliasToNode,ali);
     if (hashlist_indexof(mapNodeBooks,nod)>=0) then
     writeln(' '+RightPad(ali,l1,' ')+' = Booked');
    end;
   end;
   if IsSameText(w1,'How') then begin
    Success('Node Trig How table:'); l1:=1;
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     l1:=imax(l1,length(ali));
    end;   
    for ik:=0 to hashlist_count(mapAliasToNode)-1 do begin
     ali:=hashlist_getkey(mapAliasToNode,ik);
     nod:=hashlist_getpara(mapAliasToNode,ali);
     case GetNodeAttrHow(nod) of
      'm': how:='monitor';
      'p': how:='polling';
      't': how:='trigger';
      'r': how:='refresh';
     end;
     writeln(' '+RightPad(ali,l1,' ')+' = '+how);
    end;
   end;
   if IsSameText(w1,'Benchmarks') then begin
    Success('Benchmarks:');
    dt:=rValDef(ExtractWord(2,arg),100);
    ik:=0; ms:=mSecNow;
    ali:=hashlist_getkey(mapAliasToNode,ik);
    nod:=hashlist_getpara(mapAliasToNode,ali);
    while (mSecNow<ms+dt) do begin
     cow:=GetNodeAttrHow(nod); cow:=GetNodeAttrHow(nod);
     cow:=GetNodeAttrHow(nod); cow:=GetNodeAttrHow(nod);
     cow:=GetNodeAttrHow(nod); cow:=GetNodeAttrHow(nod);
     cow:=GetNodeAttrHow(nod); cow:=GetNodeAttrHow(nod);
     cow:=GetNodeAttrHow(nod); cow:=GetNodeAttrHow(nod);
     ik:=ik+10;
    end;
    ms:=mSecNow-ms;
    Success(StrFmt('GetNodeAttrHow takes %1.3f mks/op',1e3*ms/ik));
    ik:=0; ms:=mSecNow;
    ali:=hashlist_getkey(mapAliasToNode,ik);
    nod:=hashlist_getpara(mapAliasToNode,ali);
    tst:='@OpcDataVary '+nod+' 0';
    while (mSecNow<ms+dt) do begin
     OPCUASRV_OpcDataVary(tst); OPCUASRV_OpcDataVary(tst);
     OPCUASRV_OpcDataVary(tst); OPCUASRV_OpcDataVary(tst);
     OPCUASRV_OpcDataVary(tst); OPCUASRV_OpcDataVary(tst);
     OPCUASRV_OpcDataVary(tst); OPCUASRV_OpcDataVary(tst);
     OPCUASRV_OpcDataVary(tst); OPCUASRV_OpcDataVary(tst);
     ik:=ik+10;
    end;
    ms:=mSecNow-ms;
    Success(StrFmt('OpcDataVary takes %1.3f mks/op',1e3*ms/ik));
   end;
  end;
  Cleanup;
 end;
 {
 OPCUASRV clear strings
 }
 procedure OPCUASRV_CLEAR;
 begin
  rexShortNodeId:=0;
  rexCanonNodeId:=0;
  rexStrngNodeId:=0;
  srvTrigger:='';
  srvSection:='';
  srvPattern:='';
  rexPattern:=0;
  rexFiltTooltip:=0;
  rexFiltConsole:=0;
  strFiltTooltip:='';
  strFiltConsole:='';
  mapAliasToNode:=0;
  mapNodeToAlias:=0;
  mapNodeToIdent:=0;
  mapNodeLinks:=0;
  mapNodeReads:=0;
  mapNodeSends:=0;
  mapNodeBooks:=0;
  srvTriggerTime:=0;
  srvTriggerDelay:=0;
  statReadRate:=0;
  statSendRate:=0;
  tagOpcuaSrvStat:=0;
  tipInfoDelay:=0;
  tipWarnDelay:=0;
  tipErroDelay:=0;
  procStateText:='';
 end;
 {
 OPCUASRV initialization
 }
 procedure OPCUASRV_INIT;
 begin
  InitTag(tagOpcuaSrvStat,ReadIni('tagOpcuaSrvStat'),-3);
  rexShortNodeId:=regexp_init(0,patShortNodeId);
  rexCanonNodeId:=regexp_init(0,patCanonNodeId);
  rexStrngNodeId:=regexp_init(0,patStrngNodeId);
  srvTrigger:=ReadIniVar('StartServingTrigger',risModeTrimL+risModeTrimR+risModeRemComm);
  srvSection:=ReadIniVar('StartServingSection',risModeTrimL+risModeTrimR+risModeRemComm);
  srvPattern:=ReadIniVar('StartServingPattern',risModeTrimL+risModeTrimR+risModeRemComm);
  srvTriggerDelay:=iValDef(ReadIni('DelayServingTrigger'),1000);
  if IsEmptyStr(srvSection) then srvSection:=StrFmt('[%s.StartServing]',DevName);
  if IsEmptyStr(srvTrigger) then srvTrigger:='Welcome to opcuasrv program.';
  if IsEmptyStr(srvPattern) then srvPattern:='/^@PyAsk\s.*$/i';
  Success('StartServingTrigger = '+srvTrigger); 
  Success('StartServingSection = '+srvSection); 
  Success('StartServingPattern = '+srvPattern);
  Success('DelayServingTrigger = '+Str(srvTriggerDelay)); 
  strFiltTooltip:=TrimDef(ReadIniVar('FilterTooltipPattern',risModeTrimL+risModeTrimR+risModeRemComm),patFiltTooltip);
  strFiltConsole:=TrimDef(ReadIniVar('FilterConsolePattern',risModeTrimL+risModeTrimR+risModeRemComm),patFiltConsole);
  Success('FilterTooltipPattern = '+strFiltTooltip);
  Success('FilterConsolePattern = '+strFiltConsole);
  rexPattern:=regexp_init(0,srvPattern);
  rexFiltTooltip:=regexp_init(0,strFiltTooltip);
  rexFiltConsole:=regexp_init(0,strFiltConsole);
  mapAliasToNode:=hashlist_init(0);
  mapNodeToAlias:=hashlist_init(0);
  mapNodeToIdent:=hashlist_init(0);
  mapNodeLinks:=hashlist_init(0);
  mapNodeReads:=hashlist_init(0);
  mapNodeSends:=hashlist_init(0);
  mapNodeBooks:=hashlist_init(0);
  tipInfoDelay:=iValDef(ReadIni('tipInfoDelay'),15000);
  tipWarnDelay:=iValDef(ReadIni('tipWarnDelay'),30000);
  tipErroDelay:=iValDef(ReadIni('tipErroDelay'),60000);
  Success('tipInfoDelay = '+Str(tipInfoDelay));
  Success('tipWarnDelay = '+Str(tipWarnDelay));
  Success('tipErroDelay = '+Str(tipErroDelay));
  ParseAliasNodeSection('');
  ParseLinkNodeSection('');
  UpdateProcState('DEAD');
 end;
 {
 OPCUASRV finalization
 }
 procedure OPCUASRV_FREE;
 begin
  UpdateProcState('DEAD');
  FreeAndZero(rexShortNodeId);
  FreeAndZero(rexCanonNodeId);
  FreeAndZero(rexStrngNodeId);
  FreeAndZero(mapAliasToNode);
  FreeAndZero(mapNodeToAlias);
  FreeAndZero(mapNodeToIdent);
  FreeAndZero(mapNodeLinks);
  FreeAndZero(mapNodeReads);
  FreeAndZero(mapNodeSends);
  FreeAndZero(mapNodeBooks);
  FreeAndZero(rexPattern);
  FreeAndZero(rexFiltTooltip);
  FreeAndZero(rexFiltConsole);
  strFiltTooltip:='';
  strFiltConsole:='';
  srvTriggerDelay:=0;
  srvTriggerTime:=0;
  srvTrigger:='';
  srvSection:='';
  srvPattern:='';
  procStateText:='';
 end;
 {
 OPCUASRV polling
 }
 procedure OPCUASRV_POLL;
 begin
  OPCUASRV_Serving;
 end;
 {
 StdPyApp - User specific Hook for StdIn commands.
 Handle only @PyAns, @PyInf for Application specific stuff.
 }
 function StdPyApp_UserHookStdIn(var Data:String):Boolean;
 var cmd,arg,subcmd,subarg,msg,severity:String; cmdid,subcmdid:Integer; Hook:Boolean; wi,p:Integer;
  procedure Cleanup;
  begin
   cmd:=''; arg:=''; subcmd:=''; subarg:=''; msg:=''; severity:='';
  end;
  procedure HookFilterTooltip(var arg:String);
  begin
   if IsLexeme(arg,rexFiltTooltip) then
   if (regexp_exec(rexFiltTooltip,arg)>0) then begin
    msg:=GetDateTime(msecnow)+' - '+ProgName+': '+arg;
    msg:=StringReplace(msg,'<','{',rfReplaceAll);
    msg:=StringReplace(msg,'>','}',rfReplaceAll);
    severity:=LoCaseStr(regexp_matchstr(rexFiltTooltip,1,1));
    if (tipInfoDelay>0) then begin
     p:=Pos('info',severity);
     if (p>0) then ShowTooltip('text '+AnsiQuotedStr(msg,QuoteMark)+' preset stdInformation delay '+str(tipInfoDelay));
    end;
    if (tipWarnDelay>0) then begin
     p:=Pos('warn',severity);
     if (p>0) then ShowTooltip('text '+AnsiQuotedStr(msg,QuoteMark)+' preset stdWarning delay '+str(tipWarnDelay));
    end;
    if (tipErroDelay>0) then begin
     p:=Pos('error',severity)+Pos('fail',severity)+Pos('critical',severity);
     p:=p+Pos('fatal',severity)+Pos('alarm',severity)+Pos('alert',severity);
     p:=p+Pos('except',severity);
     if (p>0) then ShowTooltip('text '+AnsiQuotedStr(msg,QuoteMark)+' preset stdError delay '+str(tipErroDelay));
    end;
    Hook:=True;
   end;
  end;
  procedure HookFilterConsole(var arg:String);
  begin
   if IsLexeme(arg,rexFiltConsole) then begin
    Success(arg);
    Hook:=True;
   end;
  end;
 begin
  Cleanup;
  Hook:=False; cmdid:=0; subcmdid:=0;
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   if (cmdid=cmd_Std_PyAns) then begin
    if GotCommandId(arg,subcmd,subarg,subcmdid) then begin
     if (subcmdid=cmd_OpcDataVary) then begin
      OPCUASRV_OpcDataVary(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcDataRead) then begin
      OPCUASRV_OpcDataVary(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcDataBook) then begin
      OPCUASRV_OpcDataBook(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcDataSend) then begin
      OPCUASRV_OpcDataSend(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcSubInter) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcServName) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcEndpoint) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcXmlModel) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcNamespace) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcNspArray) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcNspIndex) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcSubInter) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_OpcWritable) then begin
      Success(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_Warning) then begin
      HookFilterTooltip(arg);
      Problem(arg);
      Hook:=True;
     end else
     if (subcmdid=cmd_Error) then begin
      HookFilterTooltip(arg);
      Trouble(arg);
      Hook:=True;
     end;
    end else
    if IsSameText(arg,srvTrigger) then begin
     if (srvTriggerTime>0) then begin
      Success(StrFmt('Start serving after %d ms',srvTriggerDelay));
      srvTriggerTime:=mSecNow+srvTriggerDelay;
      UpdateProcState('PREPARING');
     end;
     Hook:=True;
    end else begin
     HookFilterTooltip(arg);
     HookFilterConsole(arg);
    end;
   end else
   if (cmdid=cmd_Std_PyInf) then begin
    // On (started,stopped) show tooltip
    // After start send parameters to server
    wi:=WordIndex(ExtractWord(1,arg),'Started,Stopped');
    if (wi>0) then begin
     case wi of
      1: begin // Started
       Success('Started serving process.');
       UpdateProcState('STARTING');
       srvTriggerTime:=MaxReal;
      end;
      2: begin // Stopped
       Success('Stopped serving process.');
       UpdateProcState('STOPPED');
       srvTriggerTime:=0;
      end;
     end;
     ShowTooltip('text "PyApp: '+arg+'" preset stdNotify delay 15000');
     Success('PyApp: '+arg);
     Hook:=True;
    end;
   end;
  end;
  StdPyApp_UserHookStdIn:=Hook;
  Cleanup;
 end; 
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  ClearStdPyApp;
  OPCUASRV_CLEAR;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  StdIn_SetScripts('@StartupScript','@FinallyScript');
  StdIn_SetTimeouts(60000,60000,0,MaxInt);
  iNul(ClickFilter(ClickFilter(1)));
  iNul(ClickAwaker(ClickAwaker(1)));
  InitStdPyApp;
  OPCUASRV_INIT;
  if Val(ReadIni('CustomIniAutoLoad'))=1 then DevPostCmdLocal('@LoadIni');
  cmd_OpcServName:=RegisterStdInCmd('@OpcServName','');
  cmd_OpcEndpoint:=RegisterStdInCmd('@OpcEndpoint','');
  cmd_OpcXmlModel:=RegisterStdInCmd('@OpcXmlModel','');
  cmd_OpcNamespace:=RegisterStdInCmd('@OpcNamespace','');
  cmd_OpcNspArray:=RegisterStdInCmd('@OpcNspArray','');
  cmd_OpcNspIndex:=RegisterStdInCmd('@OpcNspIndex','');
  cmd_OpcBooking:=RegisterStdInCmd('@OpcBooking','');
  cmd_OpcServing:=RegisterStdInCmd('@OpcServing','');
  cmd_OpcNodeSend:=RegisterStdInCmd('@OpcNodeSend','');
  cmd_OpcNodePoke:=RegisterStdInCmd('@OpcNodePoke','');
  cmd_OpcNodeRead:=RegisterStdInCmd('@OpcNodeRead','');
  cmd_OpcWritable:=RegisterStdInCmd('@OpcWritable','');
  cmd_OpcInform:=RegisterStdInCmd('@OpcInform','');
  cmd_Error:=RegisterStdInCmd('@Error','');
  cmd_Warning:=RegisterStdInCmd('@Warning','');
  cmd_OpcSubInter:=RegisterStdInCmd('@OpcSubInter','');
  cmd_OpcDataVary:=RegisterStdInCmd('@OpcDataVary','');
  cmd_OpcDataRead:=RegisterStdInCmd('@OpcDataRead','');
  cmd_OpcDataSend:=RegisterStdInCmd('@OpcDataSend','');
  cmd_OpcDataBook:=RegisterStdInCmd('@OpcDataBook','');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  if Val(ReadIni('CustomIniAutoSave'))=1 then DevPostCmdLocal('@SaveIni');
  OPCUASRV_FREE;
  FreeStdPyApp;
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  OPCUASRV_POLL;
  PollStdPyApp;
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(var Data:String);
 var cmd,arg:String; cmdid:Integer;
  procedure Cleanup;
  begin
   cmd:=''; arg:='';
  end;
 begin
  Cleanup;
  if DebugFlagEnabled(dfViewImp)
  then ViewImp('CON: '+Data);
  {
  User PyApp Hook to handle:
  @PyApp Start
  @PyAsk @PollCount
  @PyAns @PollCount 123
  @PyInf Started ...
  }
  if StdPyApp_UserHookStdIn(Data) then begin
   Data:=''; // Data was hooked and handled
  end else
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   @OpcServName OpcuaSrv
   }
   if (cmdid=cmd_OpcServName) then begin
    if TranslateOpcCommands(Data,False)
    then StdPyApp_Send(Data);
    Data:='';
   end else
   {
   @OpcEndpoint opc.tcp://0.0.0.0:4840
   }
   if (cmdid=cmd_OpcEndpoint) then begin
    if TranslateOpcCommands(Data,False)
    then StdPyApp_Send(Data);
    Data:='';
   end else
   {
   @OpcXmlModel ../daqpas/datamodel.xml
   }
   if (cmdid=cmd_OpcXmlModel) then begin
    if TranslateOpcCommands(Data,False)
    then StdPyApp_Send(Data);
    Data:='';
   end else
   {
   @OpcNamespace http//freeopcua/defaults/modeler
   }
   if (cmdid=cmd_OpcNamespace) then begin
    if TranslateOpcCommands(Data,False)
    then StdPyApp_Send(Data);
    Data:='';
   end else
   {
   @OpcBooking  /.*/i
   }
   if (cmdid=cmd_OpcBooking) then begin
    if TranslateOpcCommands(Data,False)
    then StdPyApp_Send(Data);
    Data:='';
   end else
   {
   @OpcServing
   }
   if (cmdid=cmd_OpcServing) then begin
    if TranslateOpcCommands(Data,False)
    then StdPyApp_Send(Data);
    Data:='';
   end else
   {
   @OpcNodeSend WaveAplitude 2.0
   }
   if (cmdid=cmd_OpcNodeSend) then begin
    OPCUASRV_OpcNodeSend(arg);
    Data:='';
   end else
   {
   @OpcNodePoke WaveAplitude :t
   }
   if (cmdid=cmd_OpcNodePoke) then begin
    OPCUASRV_OpcNodePoke(arg);
    Data:='';
   end else
   {
   @OpcNodeRead WaveAplitude
   }
   if (cmdid=cmd_OpcNodeRead) then begin
    OPCUASRV_OpcNodeRead(arg);
    Data:='';
   end else
   {
   @OpcWritable WaveAplitude
   }
   if (cmdid=cmd_OpcWritable) then begin
    OPCUASRV_OpcWritable(arg);
    Data:='';
   end else
   {
   @OpcSubInter 100
   }
   if (cmdid=cmd_OpcSubInter) then begin
    OPCUASRV_OpcSubInter(arg);
    Data:='';
   end else
   {
   @OpcInform
   }
   if (cmdid=cmd_OpcInform) then begin
    OPCUASRV_OpcInform(arg);
    Data:='';
   end else
   {
   Default PyApp Hook to handle:
   @PyApp Start
   @PyAsk @PollCount
   @PyAns @PollCount 123
   @PyInf Started ...
   }
   if StdPyApp_DefaultHookStdIn(Data) then begin
    Data:=''; // Data was hooked and handled
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  Cleanup;
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
