##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################

##########################################################
# wave-demo-server.py - demo server to provide opcua data.
# Generate sin & cos wave for demo purposes with specified
# amplitude, frequency and noise level - just for testing.
##########################################################

# pylint: disable=invalid-name,superfluous-parens,consider-using-f-string
# pylint: disable=multiple-imports,unused-import,wildcard-import,unused-wildcard-import
# pylint: disable=missing-module-docstring,missing-function-docstring,missing-class-docstring
# pylint: disable=unused-argument,broad-exception-caught,broad-exception-raised,duplicate-code
# pylint: disable=too-many-public-methods,too-many-instance-attributes,too-many-arguments,line-too-long

import sys, os, time, math, asyncio, logging, argparse, random
from asyncua import ua, uamethod, Server
from datetime import datetime
from pycrwkit import *

EOL = os.linesep

_logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.WARN)

###################################
# python script self identification
###################################
scriptfile = __file__                     # python script file name
scriptname = ExtractFileName(scriptfile)  # python script base name with extension
scriptbase = ExtractBaseName(scriptname)  # python script base name without extension
scriptFILE = RealPath(scriptfile)         # absolute dereferenced script file name
scriptHOME = ExtractDirName(scriptFILE)   # absolute dereferenced script directory

###########################
# get OPC UA node reference
# with namespace index (ns)
# with local node index (i)
###########################
def opcuaNodeRef(ns,i):
    return "ns={0};i={1}".format(ns,i)

def varFloat(val):
    return ua.Variant(float(val), ua.VariantType.Float)
def varDouble(val):
    return ua.Variant(float(val), ua.VariantType.Double)
def varInt32(val):
    return ua.Variant(int(val), ua.VariantType.Int32)
def varInt64(val):
    return ua.Variant(int(val), ua.VariantType.Int64)

#########################################################################
# Subscription Handler. To receive events from server for a subscription.
#########################################################################
class SubHandler(object):
    def datachange_notification(self, node, val, data):
        _logger.warning("Python: New data change event %s %s", node, val)
    def event_notification(self, event):
        _logger.warning("Python: New event %s", event)

serverUrl = "opc.tcp://localhost:4840/wave/demo/server/"

###############################
# Parse command line parameters
###############################
def ParseCommandLineParams():
    parser = argparse.ArgumentParser(prog="wave-demo-server.py",
                  description='OPC UA Server test application.',
                  epilog='OPCUA server to publish data.')
    parser.add_argument('-u', '--url')
    params = parser.parse_args()
    if isinstance(params.url,str):
        global serverUrl
        serverUrl = params.url.strip()

##################
# wave demo server
##################
async def wave_demo_server():
    ###############
    # create server
    ###############
    server = Server()
    await server.init()
    serverName = "Wave Demo Server"
    server.set_server_name(serverName)
    ParseCommandLineParams()
    server.set_endpoint(serverUrl)
    print("Server:",serverName)
    print("Endpoint:",serverUrl)
    # set all possible endpoint policies for clients to connect through
    server.set_security_policy(
        [
            ua.SecurityPolicyType.NoSecurity,
            ua.SecurityPolicyType.Basic256Sha256_SignAndEncrypt,
            ua.SecurityPolicyType.Basic256Sha256_Sign,
        ]
    )
    '''
    #########################
    # setup our own namespace
    #########################
    #uri = "http//freeopcua/defaults/modeler"
    #idx = await server.register_namespace(uri)
    ##############################
    # populating our address space
    ##############################
    # waveObj = await server.nodes.objects.add_object(idx, "WaveObject")
    # waveSin = await waveObj.add_variable(idx, "WaveSin", 0.0)
    # waveCos = await waveObj.add_variable(idx, "WaveCos", 1.0)
    # waveAmp = await waveObj.add_variable(idx, "WaveAmpitude", 1.0)
    # waveFre = await waveObj.add_variable(idx, "WaveFrequency", 0.5)
    # waveNoi = await waveObj.add_variable(idx, "WaveNoise", 0.1)
    '''
    ############################
    # import uses nodes from xml
    ############################
    namespace = "http//freeopcua/defaults/modeler"
    xml = os.path.join(scriptHOME,scriptbase+".xml")
    if not os.path.exists(xml):
        print("Error: not found",xml)
        os.exit(1)
    await server.import_xml(xml)
    ###################################
    # detect namespace index for future
    ###################################
    na = await server.get_namespace_array()
    ns = await server.get_namespace_index(namespace)
    print("@OpcServerNameSpaces",", ".join(na))
    print("@TargetNameSpaceIndex",ns)
    waveDir = server.get_node(opcuaNodeRef(ns,2001))
    waveObj = server.get_node(opcuaNodeRef(ns,2002))
    waveSin = server.get_node(opcuaNodeRef(ns,2003))
    waveCos = server.get_node(opcuaNodeRef(ns,2004))
    waveAmp = server.get_node(opcuaNodeRef(ns,2005))
    waveFre = server.get_node(opcuaNodeRef(ns,2006))
    waveNoi = server.get_node(opcuaNodeRef(ns,2007))
    waveBtn = server.get_node(opcuaNodeRef(ns,2008))
    testNum = 20
    testPer = []
    for i in range(0,testNum):
        testPer.append(server.get_node(opcuaNodeRef(ns,2010+i)))
    testTrg = server.get_node(opcuaNodeRef(ns,2031))
    testRfr = server.get_node(opcuaNodeRef(ns,2032))
    ########################################
    # write initial values to uses varianles
    ########################################
    await server.write_attribute_value(waveSin.nodeid, ua.DataValue(varDouble(0.0)))
    await server.write_attribute_value(waveCos.nodeid, ua.DataValue(varDouble(1.0)))
    await server.write_attribute_value(waveAmp.nodeid, ua.DataValue(varDouble(1.0)))
    await server.write_attribute_value(waveFre.nodeid, ua.DataValue(varDouble(0.5)))
    await server.write_attribute_value(waveNoi.nodeid, ua.DataValue(varDouble(0.1)))
    await server.write_attribute_value(waveBtn.nodeid, ua.DataValue(varInt32(1)))
    for i in range(0,len(testPer)):
        await server.write_attribute_value(testPer[i].nodeid, ua.DataValue(varDouble(0.0)))
    await server.write_attribute_value(testTrg.nodeid, ua.DataValue(varDouble(0.0)))
    await server.write_attribute_value(testRfr.nodeid, ua.DataValue(varDouble(0.0)))
    #####################################
    # make some nodes writeble by clients
    #####################################
    wlist = [waveAmp, waveFre, waveNoi, waveBtn]
    for i in range(0,len(testPer)):
        wlist.append(testPer[i])
    wlist.append(testTrg)
    wlist.append(testRfr)
    for node in wlist:
        await node.set_writable(True)
    ###########
    # starting!
    ###########
    async with server:
        # print("Available loggers are: ", logging.Logger.manager.loggerDict.keys())
        # write_attribute_value is a server side method which is faster
        # than using write_value but than methods has less checks
        while True:
            await asyncio.sleep(0.1)
            wbtn = await waveBtn.read_value()
            ampl = await waveAmp.read_value()
            freq = await waveFre.read_value()
            nois = await waveNoi.read_value()
            if (wbtn == 0):
                continue
            wsin = ampl*math.sin(freq*time.time())+nois*random.uniform(-1,1)
            wcos = ampl*math.cos(freq*time.time())+nois*random.uniform(-1,1)
            await server.write_attribute_value(waveSin.nodeid, ua.DataValue(varDouble(wsin)))
            await server.write_attribute_value(waveCos.nodeid, ua.DataValue(varDouble(wcos)))

######
# MAIN
######

if __name__ == "__main__":
    asyncio.run(wave_demo_server())

#############
# END OF FILE
#############
