 {
 ***********************************************************************
 Daq Pascal application program CronSrv.
 CRON server to launch actions at specified time.
 ***********************************************************************
 [@Help]
 |Command list: StdIn "@cmd=arg" or "@cmd arg"
 |******************************************************
 | @cron.tab j t   - Add new cron job with name j crontab t.
 | @cron.job j l   - Add line l to cron job j.
 | @cron.pul j p c - Add pulser to cron job j: period ms, counter.
 | @cron.run j     - Run cron job j.
 | @cron.del j     - Delete cron job j.
 | @cron.enb j f   - Enable/disable cron job.
 | @cron.see j     - See cron job j.
 | @cron.see       - See list of all cron jobs.
 | @cron.cpu n     - See CPU(%) load report with flags n (default 3).
 | @cron.cpu start - Restart CPU statistics.
 | @cron.test      - Test Cron functionality
 | @WinHide w      - Hide window w.
 | @WinShow w      - Show window w.
 | @WinDraw w|o    - Draw window w with options o.
 | @WinSelect w    - Show and select window w.
 | @ShowMainToolBar n      - Show\hide (1\0) main ToolBar.
 | @ShowMainStatusBar n    - Show\hide (1\0) main StatusBar.
 | @ShowMainDaqControl n   - Show\hide (1\0) main DAQ control window.
 | @Async cmd      - Execute command cmd asynchronously.
 | @Run cmd        = @eval @system @async @run cmd
 | @Pid arg        = @eval @system @async @pid arg
 | @Browse s       - Open site s in Browser program.
 | @Speak s        - Speak text s with speech engine.
 | @Voice s        - Play sound s with wave sound player.
 | @Error s        - Open Error dialog with URL-encoded text s.
 | @Warning s      - Open Warning dialog with URL-encoded text s.
 | @Information s  - Open Information dialog with URL-encoded text s.
 | @Echo s         - Echo print message s to console.
 | @Shutdown Daq Exit Bye  - Exit    DAQ system, sound Bye
 | @Shutdown Daq Restart   - Restart DAQ system, no sound
 | @Shutdown Crw Exit Bye  - Exit    CRW-DAQ program, say Bye
 | @Shutdown Win Exit Bye  - Exit    Windows, say Bye
 | @Shutdown Win Logout    - Logout  Windows
 | @Shutdown Win Restart   - Restart Windows
 | @If e c                 - Run command c if expression e is nonzero.
 | @IfComputerName n c     - Run command c if computer name equal to n.
 | @IfNotComputerName n c  - Run command c if computer name is not = n.
 | @IfProcessExists e c    - Run command c if process e exists.
 | @IfNotProcessExists e c - Run command c if process e not exists.
 | @IfFileExists f c       - Run command c if file f exists.
 | @IfNotFileExists f c    - Run command c if file f not exists.
 | @IfDirExists d c        - Run command c if directory d exists.
 | @IfNotDirExists d c     - Run command c if directory d not exists.
 | @MkDir d                - Make directory d (URL-encoded,DAQ-relative).
 | @FileErase f            - Delete file f (URL,DAQ).
 | @FileCopy s d           - Copy file s to file d (URL,DAQ).
 | @FileRename s d         - Rename file s to d (URL,DAQ).
 | @FileOpenDialog f       - File open dialog with start file name f.
 | @FileWriteln f l        - write line l to file f.
 | @DevMsg &d s            - Send message s to device &d and awake.
 | @DevSend &d s           - Send message s to device &d and awake.
 | @DevPost &d s           - Post message s to device &d not awake.
 | @DevSendMsg &d s        - Send message s to device &d and awake.
 | @DevPostMsg &d s        - Post message s to device &d not awake.
 | @Guard.AppName e        - Set Guard (postmortal) application name.
 | @Guard.CmdLine c        - Set Guard (postmortal) command line params.
 | @Guard.HomeDir e        - Set Guard (postmortal) home directory.
 | @Guard.Period p         - Set Guard polling period, ms.
 | @Guard.Display d        - Set Guard display mode 0/1.
 | @Guard.Start            - Start Guard process.
 | @Guard.Stop             - Stop Guard process.
 | @Guard.View             - View Guard parameters.
 | @SilentEval n           - 0/1=silent/verbose evaluation mode.
 | @task_init n            - initialize task with name (n).
 | @task_free n            - stop, free task with name (n).
 | @task_ctrl n p=v        - task_ctrl(n,p=v) for task (n).
 | @task_view n            - view=print task with name (n).
 | @task_run n             - run task with name (n).
 | @task_kill n h c t      - kill task name n, method h, exitcode c, timeout t.
 | @task_poll n p g        - setup poll period (p) and recovery guard period (g) for task (n).
 | @task_send n m          - send text messsage (m) to stdin of task (n).
 | @task_recv n с          - set command pattern (c) to process stdout of task (n). Uses $* substitution.
 | @MenuExitOpen           - open Menu Exit dialog.
 |******************************************************
 | Example:
 | [DeviceList]
 | &CronSrv = device software program
 | [&CronSrv]
 | Comment       = CRON server to launch actions at specified time.
 | InquiryPeriod = 1
 | DevicePolling = 10, tpHighest
 | ProgramGuard  = ~~\resource\daqsite\cronserver\crongrd.exe
 | ProgramSource = ~~\resource\daqsite\cronserver\cronsrv.pas
 | StartupScript = [&CronSrv.StartupScript]
 | FinallyScript = [&CronSrv.FinallyScript]
 | StopTimeOut   = 2000
 | StdInFifo     = 512
 | StdOutFifo    = 512
 | OpenConsole   = 2
 | DebugFlags    = 3
 | ...
 | if DevSend(RefFind('Device &CronSrv'),
 |    '@cron.del daily'+EOL
 |   +'@cron.tab daily 0 8'+EOL
 |   +'@cron.job daily @speak Wake up, now 8:00!'+EOL
 |   +'@cron.enb daily 1'+EOL)>0
 | then Success('Success!')
 | else Trouble('Failure!');
 |******************************************************
 []
 }
{
********************************************************
[Compiler.Options]
Compiler.dtabmax = 1024*20 ; Data segment
Compiler.stabmax = 1024*16 ; String table
Compiler.dtabmin = 1024*1  ; Min stack  space
Compiler.stabmin = 1024*1  ; Min string space
[]
********************************************************
}
program CRONSRV;                 { CRON server to run jobs by time  }
const
 {------------------------------}{ Declare uses program constants:  }
 {$I _con_StdLibrary}            { Include all Standard constants,  }
 {------------------------------}{ And add User defined constants:  }
 MaxCount          = 1024;       { Max number of CronTab jobs       }
 MaxLeng           = 16384;      { Max string length                }
 SendTimeOut       = 100;        { timeout on message send          }
 GuardTimerPeriod  = 1000;       { Guard.Timer poll period          }
 CronPollPeriod    = 1000;       { Period of CronTab polling, ms    }
 DefStopTimeOut    = 2000;       { Default StopTimeOut, ms          }
 ProfilerPeriod    = 1000;       { Profiler polling period, ms      }
 DefProfilerMode   = 3;          { Default Profiler mode            }
 TaskerIMin        = 1;          { Min Tasker index                 }
 TaskerIMax        = 128;        { Max Tasker index                 }

var
 {------------------------------}{ Declare uses program variables:  }
 {$I _var_StdLibrary}            { Include all Standard variables,  }
 {------------------------------}{ And add User defined variables:  }
 TheCron           : record      { Cron table                       }
  id               : array[1..MaxCount] of String; { Job name       }
  tab              : array[1..MaxCount] of String; { Cron table     }
  job              : array[1..MaxCount] of String; { Job text       }
  tim              : array[1..MaxCount] of Real;   { Cron timer     }
  pul              : array[1..MaxCount] of Real;   { Pulser time    }
  per              : array[1..MaxCount] of Integer;{  Period,ms     }
  cnt              : array[1..MaxCount] of Integer;{  Counter       }
 end;
 CronStartMs       : Real;       { Time of CronSrv start            }
 CronTabPoll       : Real;       { Time of last TheCron.tab polling }
 Profiler          : record      { Record to profile performance    }
  Init,Last,Curr,Peak,Rate  : record   {                            }
   Kern,User,Vdpm,Time : Real;   { Kernel,User,VirtDaqPasMachine,ms }
  end;
 end;
 Guard             : record      { Cron_Guard to run postmortal exe }
  Exe              : String;     { Postmortal executable name       }
  Tid              : Integer;    { Postmortal task id               }
  Line             : String;     { For temporary data               }
  Timer            : Integer;    { Guard Timer                      }
  Display,Period   : Integer;    { Display,Period                   }
  App,Cmd,Dir,Buf  : String;     { AppName,CmdLine,HomeDir,Buffer   }
 end;
 Tasker            : record      { @task_xxxx command handler       }
  hlist            : Integer;    { hashlist for task search         }
  ulist            : String;     { uses index list for fast poll    }
  flist            : String;     { free index list for fast init    }
  Items            : array[TaskerIMin..TaskerIMax] of record {      }
   tid             : Integer;    { task id                          }
   Name            : String;     { task name                        }
   CmdLine         : String;     { command line to execute          }
   HomeDir         : String;     { home directory                   }
   Line            : String;     { Temporary buffer for data read   }
   Buff            : String;     { Temporary buffer for data read   }
   TxConv,RxConv   : String;     { Converter for Tx,Rx              }
   OnRecv          : String;     { Command on task_recv             }
   CodePage        : Integer;    { CodePage code                    }
   Display         : Integer;    { DisplayMode                      }
   Priority        : Integer;    { Process priority: 4,6,8,10,13,24 }
   StdInPipeSize   : Integer;    { StdInPipeSize  bytes buffer      }
   StdOutPipeSize  : Integer;    { StdOutPipeSize bytes buffer      }
   GuardPeriod     : Integer;    { Guard (recovery) Period, ms      }
   GuardTimer      : Real;       { Guard timer to check/recover     }
   PollPeriod      : Integer;    { Polling period, ms               }
   PollTimer       : Real;       { Polling timer, ms                }
  end;                           {                                  }
  cpAnsi,cpOEM     : Integer;    { System ANSI, OEM codepage        }
 end;                            {                                  }
 SilentEval        : Boolean;    { Silent SysEval mode              }
 cmd_crontab       : Integer;    { Command @cron.tab                }
 cmd_cronjob       : Integer;    { Command @cron.job                }
 cmd_cronpul       : Integer;    { Command @cron.pul                }
 cmd_cronrun       : Integer;    { Command @cron.run                }
 cmd_crondel       : Integer;    { Command @cron.del                }
 cmd_cronenb       : Integer;    { Command @cron.enb                }
 cmd_cronsee       : Integer;    { Command @cron.see                }
 cmd_croncpu       : Integer;    { Command @cron.cpu                }
 cmd_crontest      : Integer;    { Command @cron.test               }
 cmd_WinHide       : Integer;    { Command @WinHide                 }
 cmd_WinShow       : Integer;    { Command @WinShow                 }
 cmd_WinDraw       : Integer;    { Command @WinDraw                 }
 cmd_WinSelect     : Integer;    { Command @WinSelect               }
 cmd_ShowMainToolBar : Integer;  { Command @ShowMainToolBar         }
 cmd_ShowMainStatusBar :Integer; { Command @ShowMainStatusBar       }
 cmd_ShowMainDaqControl:Integer; { Command @ShowMainDaqControl      }
 cmd_Async         : Integer;    { Command @Async                   }
 cmd_Eval          : Integer;    { Command @Eval                    }
 cmd_Run           : Integer;    { Command @Run                     }
 cmd_Pid           : Integer;    { Command @Pid                     }
 cmd_Browse        : Integer;    { Command @Browse                  }
 cmd_Speak         : Integer;    { Command @Speak                   }
 cmd_Voice         : Integer;    { Command @Voice                   }
 cmd_Error         : Integer;    { Command @Error                   }
 cmd_Warning       : Integer;    { Command @Warning                 }
 cmd_Information   : Integer;    { Command @Information             }
 cmd_Echo          : Integer;    { Command @Echo                    }
 cmd_Shutdown      : Integer;    { Command @Shutdown                }
 cmd_If            : Integer;    { Command @If                      }
 cmd_IfComputerName : Integer;   { Command @IfComputerName          }
 cmd_IfNotComputerName :Integer; { Command @IfNotComputerName       }
 cmd_IfProcessExists : Integer;  { Command @IfProcessExists         }
 cmd_IfNotProcessExists:Integer; { Command @IfNotProcessExists      }
 cmd_IfFileExists  : Integer;    { Command @IfFileExists            }
 cmd_IfNotFileExists : Integer;  { Command @IfNotFileExists         }
 cmd_IfDirExists     : Integer;  { Command @IfDirExists             }
 cmd_IfNotDirExists  : Integer;  { Command @IfNotDirExists          }
 cmd_MkDir           : Integer;  { Command @MkDir                   }
 cmd_FileErase       : Integer;  { Command @FileErase               }
 cmd_FileCopy        : Integer;  { Command @FileCopy                }
 cmd_FileRename      : Integer;  { Command @FileRename              }
 cmd_FileOpenDialog  : Integer;  { Command @FileOpenDialog          }
 cmd_FileWriteln     : Integer;  { Command @FileWriteln             }
 cmd_DevMsg          : Integer;  { Command @DevMsg                  }
 cmd_DevSend         : Integer;  { Command @DevSend                 }
 cmd_DevPost         : Integer;  { Command @DevPost                 }
 cmd_DevSendMsg      : Integer;  { Command @DevSendMsg              }
 cmd_DevPostMsg      : Integer;  { Command @DevPostMsg              }
 cmd_GuardAppName    : Integer;  { Command @Guard.AppName           }
 cmd_GuardCmdLine    : Integer;  { Command @Guard.CmdLine           }
 cmd_GuardHomeDir    : Integer;  { Command @Guard.HomeDir           }
 cmd_GuardPeriod     : Integer;  { Command @Guard.Period            }
 cmd_GuardDisplay    : Integer;  { Command @Guard.Display           }
 cmd_GuardStart      : Integer;  { Command @Guard.Start             }
 cmd_GuardStop       : Integer;  { Command @Guard.Stop              }
 cmd_GuardView       : Integer;  { Command @Guard.View              }
 cmd_SilentEval      : Integer;  { Command @SilentEval              }
 cmd_task_init       : Integer;  { Command @task_init               }
 cmd_task_free       : Integer;  { Command @task_free               }
 cmd_task_ctrl       : Integer;  { Command @task_ctrl               }
 cmd_task_view       : Integer;  { Command @task_view               }
 cmd_task_run        : Integer;  { Command @task_run                }
 cmd_task_kill       : Integer;  { Command @task_kill               }
 cmd_task_poll       : Integer;  { Command @task_poll               }
 cmd_task_send       : Integer;  { Command @task_send               }
 cmd_task_recv       : Integer;  { Command @task_recv               }
 cmd_MenuExitOpen    : Integer;  { @MenuExitOpen                    }

 {------------------------------}{ Declare procedures & functions:  }
 {$I _fun_StdLibrary}            { Include all Standard functions,  }
 {------------------------------}{ And add User defined functions:  }
 {$I _fun_StdMenuTools}          { Standard Menu Tools  functions,  }

 {
 Menu EXIT Starter to start editing.
 }
 procedure MenuExitStarter;
 var i,n:Integer;
 begin
  if EditStateReady then begin
   //////////////////////////////////////////
   n:=0+EditAddOpening(LoCaseStr(GetEnv('CRW_DAQ_SYS_SESSION_HEAD'))+' - '+RusEngStr('Команда "Завершить" …','Command "Exit" …'));
   n:=n+EditAddInputLn(RusEngStr('Что выбираете:','Please Choose:'));
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Продолжить работу текущего сеанса DAQ','Continue running of current DAQ session'));
   n:=n+EditAddConfirm('');
   n:=n+EditAddCommand('@tooltip text "'+RusEngStr('Желаю успешной работы','I wish you successful work')+'" preset stdNotify delay 15000');
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Завершить сеанс DAQ и закрыть программу','Stop DAQ session and close program'));
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @Shutdown Crw Exit');
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Завершить сеанс DAQ и продолжить работу','Exit DAQ session and continue work'));
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @Shutdown Daq Exit');
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Перезагрузить сеанс DAQ и начать заново','Restart DAQ session'));
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @Shutdown Daq Restart');
   //////////////////////////////////////////
   if False then
   for i:=1 to WordCount(EditGetWellKnownDevices(DevName)) do
   if (RefFind('device '+ExtractWord(i,EditGetWellKnownDevices(DevName)))<>0) then begin
    n:=n+EditAddInputLn(RusEngStr('Перезапустить сервер ','Restart server ')+ExtractWord(i,EditGetWellKnownDevices(DevName)));
    n:=n+EditAddConfirm(EditGetLastInputLn);
    n:=n+EditAddCommand('@SysEval @Daq Compile '+ExtractWord(i,EditGetWellKnownDevices(DevName)));
   end;
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Завершить сеанс ','Exit session of ')+ParamStr('System Os Platform'));
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @Shutdown Win Logout');
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Перезагрузить компьютер ','Restart computer ')+ParamStr('HostName'));
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @Shutdown Win Restart');
   //////////////////////////////////////////
   n:=n+EditAddInputLn(RusEngStr('Выключить     компьютер ','Switch OFF computer ')+ParamStr('HostName'));
   n:=n+EditAddConfirm(EditGetLastInputLn);
   n:=n+EditAddCommand('@Cron @Shutdown Win Exit');
   //////////////////////////////////////////
   n:=n+EditAddSetting('@set ListBox.Font Size:14\Style:[Bold]');
   {
   n:=n+EditAddSetting('@set Form.Left 530 relative '+Copy(DevName,2)+' PaintBox');
   n:=n+EditAddSetting('@set Form.Top  0   relative '+Copy(DevName,2)+' PaintBox');
   }
   //////////////////////////////////////////
   n:=n+EditAddClosing('MenuList',EditGetUID('MENU_EXIT'),'');
   if (n>0) then Problem('Error initializing MenuList!');
  end else Problem('Cannot edit right now!');
 end;
 {
 Menu EXIT Handler to handle editing.
 }
 procedure MenuExitHandler;
 begin
  EditMenuDefaultHandler(EditGetUID('MENU_EXIT'));
 end;
 {
 Tasker - task manager to start/stop/control user processes by name.
 }
 function CharSequence(a,b:Char):String;
 var s:String; c:Char;
 begin
  s:='';
  for c:=a to b do s:=s+c;
  CharSequence:=s;
  s:='';
 end;
 procedure StrExcludeChar(var s:String; c:Char);
 begin
  s:=StringReplace(s,c,'',rfReplaceAll);
 end;
 procedure StrIncludeChar(var s:String; c:Char);
 begin
  StrExcludeChar(s,c); s:=s+c;
 end;
 function TaskerValidIndex(i:Integer):Boolean;
 begin
  if ((i>=TaskerIMin) and (i<=TaskerIMax))
  then TaskerValidIndex:=true
  else TaskerValidIndex:=false;
 end;
 function TaskerUsesIndex(i:Integer):Boolean;
 begin
  if ((i>=TaskerIMin) and (i<=TaskerIMax))
  then TaskerUsesIndex:=(Length(Tasker.Items[i].Name)>0)
  else TaskerUsesIndex:=false;
 end;
 function TaskerItemName(i:Integer):String;
 begin
  if ((i>=TaskerIMin) and (i<=TaskerIMax))
  then TaskerItemName:=Tasker.Items[i].Name
  else TaskerItemName:='';
 end;
 function TaskerItemTid(i:Integer):Integer;
 begin
  if ((i>=TaskerIMin) and (i<=TaskerIMax))
  then TaskerItemTid:=Tasker.Items[i].tid
  else TaskerItemTid:=0;
 end;
 procedure TaskerClearItem(i:Integer);
 begin
  if TaskerValidIndex(i) then begin
   Tasker.Items[i].tid:=0;
   Tasker.Items[i].Name:='';
   Tasker.Items[i].CmdLine:='';
   Tasker.Items[i].HomeDir:='';
   Tasker.Items[i].Line:='';
   Tasker.Items[i].Buff:='';
   Tasker.Items[i].TxConv:='';
   Tasker.Items[i].RxConv:='';
   Tasker.Items[i].OnRecv:='';
   Tasker.Items[i].Display:=0;
   Tasker.Items[i].Priority:=0;
   Tasker.Items[i].StdInPipeSize:=0;
   Tasker.Items[i].StdOutPipeSize:=0;
   Tasker.Items[i].GuardPeriod:=0;
   Tasker.Items[i].GuardTimer:=0;
   Tasker.Items[i].PollPeriod:=0;
   Tasker.Items[i].PollTimer:=0;
  end;
 end;
 procedure TaskerClear;
 var i:Integer;
 begin
  Tasker.hlist:=0; Tasker.ulist:=''; Tasker.flist:='';
  for i:=TaskerIMin to TaskerIMax do TaskerClearItem(i);
  Tasker.cpAnsi:=0; Tasker.cpOem:=0;
 end;
 procedure TaskerInit;
 begin
  Tasker.hlist:=hashlist_init(0);
  Tasker.ulist:='';
  Tasker.flist:=CharSequence(Chr(TaskerIMin),Chr(TaskerIMax));
  Tasker.cpAnsi:=Val(ParamStr('AnsiCodePage'));
  Tasker.cpOem:=Val(ParamStr('OemCodePage'));
 end;
 function TaskerFindItem(name:String):Integer;
 var i:Integer;
 begin
  if (name<>'') and (Tasker.hlist<>0)
  then TaskerFindItem:=hashlist_getlink(Tasker.hlist,name)
  else TaskerFindItem:=0;
 end;
 function TaskerFlistCount:Integer;
 begin
  TaskerFlistCount:=Length(tasker.flist);
 end;
 function TaskerFlistItem(i:Integer):Integer;
 begin
  TaskerFlistItem:=Ord(StrFetch(tasker.flist,i));
 end;
 function TaskerUlistCount:Integer;
 begin
  TaskerUlistCount:=Length(Tasker.ulist);
 end;
 function TaskerUlistItem(i:Integer):Integer;
 begin
  TaskerUlistItem:=Ord(StrFetch(Tasker.ulist,i));
 end;
 procedure TaskerInitItem(name,args:String);
 var i:Integer;
 begin
  name:=Trim(name);
  args:=Trim(args);
  if (name<>'') then begin
   i:=TaskerFindItem(name);
   if (i=0) then begin
    i:=Ord(StrFetch(tasker.flist,1));
    if TaskerValidIndex(i) then begin
     TaskerClearItem(i);
     Tasker.Items[i].Name:=name;
     Tasker.Items[i].Display:=1;
     Tasker.Items[i].Priority:=8;
     bNul(hashlist_setlink(Tasker.hlist,name,i));
     StrIncludeChar(Tasker.ulist,Chr(i));
     StrExcludeChar(Tasker.flist,Chr(i));
     Tasker.Items[i].tid:=task_init('');
     if (args<>'') then begin
      Tasker.Items[i].CmdLine:=args;
      sNul(task_ctrl(Tasker.Items[i].tid,'CmdLine='+args));
     end;
    end;
   end else Trouble('Task already exist: '+name);
  end;
 end;
 procedure TaskerViewItem(i:Integer; mode:Integer);
 var tid,pid,live,code:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i); pid:=task_pid(tid); live:=Ord(task_wait(tid,0)); code:=task_result(tid);
   Success('Task['+Str(i)+'] Name: '+TaskerItemName(i)+', TID: '+Str(tid)+', PID: '+Str(pid)
          +', State: '+ExtractWord(1+live,'Stopped,Running')+', Code: '+Str(code));
   if HasFlags(mode,1) then begin
    Success('Preset:');
    Success(' CmdLine  = '+Tasker.Items[i].CmdLine);
    Success(' HomeDir  = '+Tasker.Items[i].HomeDir);
    Success(' PipeIn   = '+Str(Tasker.Items[i].StdInPipeSize));
    Success(' PipeOut  = '+Str(Tasker.Items[i].StdOutPipeSize));
    Success(' Display  = '+Str(Tasker.Items[i].Display));
    Success(' Priority = '+Str(Tasker.Items[i].Priority));
    Success(' CodePage = '+Str(Tasker.Items[i].CodePage));
    Success(' Polling  = '+Str(Tasker.Items[i].PollPeriod)+', '+Str(Tasker.Items[i].GuardPeriod));
    Success(' OnRecv   = '+Tasker.Items[i].OnRecv);
   end;
   if HasFlags(mode,2) and (tid<>0) then begin
    Success('Actual:');
    Success(' Status   = '+ExtractWord(1+Ord(task_wait(tid,0)),'Stopped,Running'));
    Success(' tid,pid  = '+str(tid)+', '+str(task_pid(tid)));
    Success(' [i],ref  = '+str(i)+', '+str(task_ref(tid)));
    Success(' CmdLine  = '+task_ctrl(tid,'CmdLine'));
    Success(' HomeDir  = '+task_ctrl(tid,'HomeDir'));
    Success(' Display  = '+task_ctrl(tid,'Display'));
    Success(' Priority = '+task_ctrl(tid,'ProcessPriority'));
    Success(' PipeIn   = '+task_ctrl(tid,'StdInPipeSize'));
    Success(' PipeOut  = '+task_ctrl(tid,'StdOutPipeSize'));
   end;
  end;
 end;
 procedure TaskerViewArg(arg:String);
 var i:Integer;
 begin
  if (Trim(arg)='') then begin
   Success(StrFmt('%d task(s) found.',TaskerUlistCount));
  end else
  if IsSameText(Trim(arg),'*') then begin
   Success(StrFmt('%d task(s) found.',TaskerUlistCount));
   for i:=1 to TaskerUlistCount do TaskerViewItem(TaskerUlistItem(i),0);
  end else
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if (i>0) then TaskerViewItem(i,3) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 procedure TaskerSetItemCodePage(i:Integer; cp:Integer);
 begin
  if (cp>0) then
  if TaskerUsesIndex(i) then begin
   Tasker.Items[i].CodePage:=cp;
   if (Tasker.cpAnsi>0) and (Tasker.cpOem=cp) then begin
    Tasker.Items[i].TxConv:='ansi-oem';
    Tasker.Items[i].RxConv:='oem-ansi';
   end;
  end;
 end;
 procedure TaskerCtrlArg(arg:String);
 var i,tid:Integer; s,n,v:String;
 begin
  s:=''; n:=''; v:='';
  if (WordCount(arg)>1) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if TaskerUsesIndex(i) then begin
    n:=ExtractWord(2,arg);
    v:=Trim(SkipWords(2,arg));
    if IsSameText(n,'Priority') then n:='ProcessPriority';
    if IsSameText(n,'CmdLine') and (v<>'') then v:=ExpEnv(v);
    if IsSameText(n,'HomeDir') and (v<>'') then v:=ExpEnv(v);
    if IsSameText(n,'CmdLine') and (v<>'') then Tasker.Items[i].CmdLine:=v;
    if IsSameText(n,'HomeDir') and (v<>'') then Tasker.Items[i].HomeDir:=v;
    if IsSameText(n,'Display') and (v<>'') then Tasker.Items[i].Display:=iValDef(v,0);
    if IsSameText(n,'ProcessPriority') and (v<>'') then Tasker.Items[i].Priority:=iValDef(v,0);
    if IsSameText(n,'StdInPipeSize') and (v<>'') then Tasker.Items[i].StdInPipeSize:=iEvalDef(v,16*1024);
    if IsSameText(n,'StdOutPipeSize') and (v<>'') then Tasker.Items[i].StdOutPipeSize:=iEvalDef(v,16*1024);
    if IsSameText(n,'CodePage') and (v<>'') then TaskerSetItemCodePage(i,Val(v));
    tid:=TaskerItemTid(i);
    if (task_ref(tid)<>0) then begin
     if (v<>'') then s:=n+'='+v else s:=n;
     s:=task_ctrl(tid,s);
     if (v='') then Success(n+'='+s);
    end;
   end else
   Problem('Task not found: '+ExtractWord(1,arg));
  end;
  s:=''; n:=''; v:='';
 end;
 procedure TaskerKillItem(i:Integer; arg:String);
 var tid,pid,how,cod,tot:Integer;
  procedure TryKillItem(i,tid,pid,how,code,timeout:Integer);
  begin
   if (tid<>0) then if task_wait(tid,0) then if task_kill(tid,how,code,timeout) and not task_wait(tid,0) then
   Success(StrFmt('Killed[%d] ',how)+StrFmt('Task[%d] ',i)+StrFmt('PID[%d] ',pid)+StrFmt('with CODE[%d].',task_result(tid)));
   rNul(wdt_reset(true));
  end;
 begin
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i);
   if (tid<>0) then begin
    pid:=task_pid(tid);
    how:=iValDef(ExtractWord(1,arg),0);
    cod:=iValDef(ExtractWord(2,arg),0);
    tot:=iValDef(ExtractWord(3,arg),1000);
    TryKillItem(i,tid,pid,how,cod,tot);
   end;
  end;
 end;
 procedure TaskerKillArg(arg:String);
 var i:Integer;
 begin
  if IsSameText(ExtractWord(1,arg),'*') then begin
   for i:=TaskerUlistCount downto 1 do TaskerKillItem(TaskerUlistItem(i),SkipWords(1,arg));
  end else
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if (i>0) then TaskerKillItem(i,SkipWords(1,arg)) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 procedure TaskerStopItem(i:Integer; FreeTid:Boolean);
 var tid:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i);
   if (tid<>0) then begin
    TaskerKillItem(i,'1,0,1000');
    TaskerKillItem(i,'0,0,1000');
    if FreeTid then begin
     bNul(task_free(tid));
     Tasker.Items[i].tid:=0;
     Tasker.Items[i].Line:='';
     Tasker.Items[i].Buff:='';
    end;
   end;
  end;
 end;
 procedure TaskerFreeItem(i:Integer);
 begin
  if TaskerValidIndex(i) then begin
   bNul(hashlist_delete(Tasker.hlist,TaskerItemName(i)));
   StrExcludeChar(Tasker.ulist,Chr(i));
   StrIncludeChar(Tasker.flist,Chr(i));
   TaskerStopItem(i,true);
   TaskerClearItem(i);
  end;
 end;
 procedure TaskerFreeArg(arg:String);
 var i:Integer;
 begin
  if IsSameText(Trim(arg),'*') then begin
   for i:=TaskerUlistCount downto 1 do TaskerFreeItem(TaskerUlistItem(i));
  end else
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if (i>0) then TaskerFreeItem(i) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 procedure TaskerFree;
 begin
  TaskerFreeArg(Dump('*'));
  if (Tasker.hlist<>0) then begin
   bNul(hashlist_free(Tasker.hlist));
   Tasker.hlist:=0;
  end;
  Tasker.ulist:='';
  Tasker.flist:='';
 end;
 procedure TaskerRunItem(i:Integer);
 var tid,pid:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i);
   if (tid<>0) then begin
    if task_wait(tid,0) then begin
     pid:=task_pid(tid);
     Success(StrFmt('Info: Task %d is already running, ',i)+StrFmt('PID %d',pid));
    end else begin
     if (task_pid(tid)<>0) then begin
      bNul(task_free(tid));
      tid:=task_init('');
      Tasker.Items[i].Tid:=tid;
     end;
     sNul(task_ctrl(tid,'CmdLine='+Trim(Tasker.Items[i].CmdLine)));
     sNul(task_ctrl(tid,'HomeDir='+Trim(Tasker.Items[i].HomeDir)));
     sNul(task_ctrl(tid,'Display='+Str(Tasker.Items[i].Display)));
     sNul(task_ctrl(tid,'ProcessPriority='+Str(Tasker.Items[i].Priority)));
     sNul(task_ctrl(tid,'StdInPipeSize='+Str(Tasker.Items[i].StdInPipeSize)));
     sNul(task_ctrl(tid,'StdOutPipeSize='+Str(Tasker.Items[i].StdOutPipeSize)));
     if task_run(tid) then begin
      pid:=task_pid(tid);
      Success(StrFmt('Run task %d, ',i)+StrFmt('PID %d.',pid));
     end else begin
      Trouble(StrFmt('Error: could not run task[%d]',i));
     end;
    end;
   end;
  end;
 end;
 procedure TaskerRunArg(arg:String);
 var i:Integer;
 begin
  arg:=Trim(arg);
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if (i>0) then TaskerRunItem(i) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 function SubstRecvLine(pattern,line:String):String;
 begin
  SubstRecvLine:=StringReplace(pattern,'$*',line,0);
 end;
 procedure TaskerProcessItem(i:Integer; var line:String);
 var tid,pid:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i); pid:=task_pid(tid);
   if DebugFlagEnabled(dfViewImp) then ViewImp(StrFmt('PID[%d] < ',pid)+line);
   if LooksLikeCommand(Tasker.Items[i].OnRecv) then DevPostCmdLocal(SubstRecvLine(Tasker.Items[i].OnRecv,line));
  end;
 end;
 procedure TaskerReadItem(i:Integer);
 var tid:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i);
   if (tid<>0) then
   if (task_rxcount(tid)>0) then
   while task_readln(tid,Tasker.Items[i].Line,Tasker.Items[i].Buff) do begin
    if (Length(Tasker.Items[i].RxConv)>0)
    then Tasker.Items[i].Line:=StrConv(Tasker.Items[i].RxConv,Tasker.Items[i].Line);
    TaskerProcessItem(i,Tasker.Items[i].Line);
   end;
  end;
 end;
 procedure TaskerPollItem(i:Integer);
 var tid,pid:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   if (Tasker.Items[i].PollPeriod>0) then
   if (msElapsedSinceMarker(Tasker.Items[i].PollTimer)>Tasker.Items[i].PollPeriod) then begin
    Tasker.Items[i].PollTimer:=mSecNow;
    tid:=TaskerItemTid(i);
    if (task_rxcount(tid)>0) then TaskerReadItem(i);
    if (Tasker.Items[i].GuardPeriod>0) then
    if (msElapsedSinceMarker(Tasker.Items[i].GuardTimer)>Tasker.Items[i].GuardPeriod) then begin
     Tasker.Items[i].GuardTimer:=mSecNow;
     if not task_wait(tid,0) then begin
      pid:=task_pid(tid);
      if (pid<>0) then Trouble(StrFmt('Detected dead task[%d] ',i)+StrFmt('PID[%d]. ',pid)+'Restart it.');
      TaskerRunItem(i);
     end;
    end;
   end;
  end;
 end;
 procedure TaskerPollItemSetup(i,p,g:Integer);
 var tid:Integer;
 begin
  if TaskerUsesIndex(i) then begin
   Tasker.Items[i].PollPeriod:=imax(0,p);
   Tasker.Items[i].GuardPeriod:=imax(0,g);
   if (Tasker.Items[i].PollPeriod>0) then Tasker.Items[i].PollTimer:=mSecNow;
   if (Tasker.Items[i].GuardPeriod>0) then Tasker.Items[i].GuardTimer:=mSecNow;
  end;
 end;
 procedure TaskerPollArg(arg:String);
 var i,p,g:Integer;
 begin
  arg:=Trim(arg);
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg)); // Name
   p:=iValDef(ExtractWord(2,arg),0);      // Polling period
   g:=iValDef(ExtractWord(3,arg),0);      // Guard (recovery) period
   if (i>0) then TaskerPollItemSetup(i,p,g) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 procedure TaskerPolling;
 var i:Integer;
 begin
  for i:=1 to TaskerUlistCount do TaskerPollItem(TaskerUlistItem(i));
 end;
 procedure TaskerSendItem(i:Integer; s:String);
 var tid,pid,n:Integer;
 begin
  if (Length(s)>0) then
  if TaskerUsesIndex(i) then begin
   tid:=TaskerItemTid(i); pid:=task_pid(tid);
   if (task_txspace(tid)>Length(s)) then begin
    if (Length(Tasker.Items[i].TxConv)>0) then s:=StrConv(Tasker.Items[i].TxConv,s);
    n:=task_send(tid,s);
    if (n<Length(s)) then Trouble(StrFmt('Error send to task[%d] ',i)+StrFmt('PID[%d].',pid));
   end; 
  end;
 end;
 procedure TaskerSendArg(arg:String);
 var i:Integer;
 begin
  arg:=Trim(arg);
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if (i>0) then TaskerSendItem(i,SkipWords(1,arg)+EOL) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 procedure TaskerRecvItem(i:Integer; s:String);
 begin
  s:=Trim(s);
  if LooksLikeCommand(s) then
  if TaskerUsesIndex(i) then begin
   Tasker.Items[i].OnRecv:=Trim(s);
   Success(StrFmt('Set Task[%d] ',i)+StrFmt('OnReceive paattern %s',s));
  end;
 end;
 procedure TaskerRecvArg(arg:String);
 var i:Integer;
 begin
  arg:=Trim(arg);
  if (WordCount(arg)>0) then begin
   i:=TaskerFindItem(ExtractWord(1,arg));
   if (i>0) then TaskerRecvItem(i,SkipWords(1,arg)) else Problem('Task not found: '+ExtractWord(1,arg));
  end;
 end;
 function TaskerHandler(cmdid:Integer; arg:String):Boolean;
 var done:Boolean;
 begin
  done:=false;
  if (cmdid>0) then begin
   //
   // @task_init demo
   //
   if (cmdid=cmd_task_init) then begin
    if (WordCount(arg)>0) then begin
     TaskerInitItem(ExtractWord(1,arg),SkipWords(1,arg));
     done:=true;
    end;
   end else
   //
   // @task_free demo
   //
   if (cmdid=cmd_task_free) then begin
    if (WordCount(arg)>0) then begin
     TaskerFreeArg(arg);
     done:=true;
    end;
   end else
   //
   // @task_ctrl demo CmdLine = cmd.exe /c dir
   //
   if (cmdid=cmd_task_ctrl) then begin
    TaskerCtrlArg(arg);
    done:=true;
   end else
   //
   // @task_view demo
   //
   if (cmdid=cmd_task_view) then begin
    TaskerViewArg(arg);
    done:=true;
   end else
   //
   // @task_run demo
   //
   if (cmdid=cmd_task_run) then begin
    TaskerRunArg(arg);
    done:=true;
   end else
   //
   // @task_kill demo 1 0 1000
   //
   if (cmdid=cmd_task_kill) then begin
    TaskerKillArg(arg);
    done:=true;
   end else
   //
   // @task_poll demo 100 10000
   //
   if (cmdid=cmd_task_poll) then begin
    TaskerPollArg(arg);
    done:=true;
   end else
   //
   // @task_send demo exit
   //
   if (cmdid=cmd_task_send) then begin
    TaskerSendArg(arg);
    done:=true;
   end else
   //
   // @task_recv demo @devpost &Demo @received $*
   //
   if (cmdid=cmd_task_recv) then begin
    TaskerRecvArg(arg);
    done:=true;
   end else
   done:=false;
  end;
  TaskerHandler:=done;
 end;
 {
 Evaluate expression via system calculator in Main Console.
 }
 function SysEval(expr:String):Real;
 begin
  if SilentEval
  then SysEval:=Eval('@system @async @silent '+Trim(expr))
  else SysEval:=Eval('@system @async '        +Trim(expr));
 end;
 {
 Extract word with space and tab delimeters.
 }
 function ExtractWordSpaces(Num:Integer;Data:String):String;
 begin
  ExtractWordSpaces:=ExtractWordDelims(Num,Data,Dump(' ')+Dump(Chr(9)));
 end;
 {
 Dialog with Warning message
 }
 procedure WarningDlg(msg:String);
 var i,t:Integer;
 begin
  if Length(Trim(msg))>0 then begin
   Success('WARNING:');
   if EditState=0 then begin
    t:=StringToText(msg);
    if Text_NumLn(t)>0 then begin 
     for i:=0 to Text_NumLn(t)-1 do begin
      if i=0
      then msg:=Edit('('+Text_GetLn(t,i))
      else msg:=Edit(' '+Text_GetLn(t,i));
      Success(' '+Text_GetLn(t,i));
     end; 
     msg:=Edit(')Warning ___Warning___');
    end;
    bNul(Text_Free(t));
   end else bNul(Echo('WARNING:'+EOL+msg));
  end;
 end;
 {
 Dialog with Error message
 }
 procedure ErrorDlg(msg:String);
 var i,t:Integer;
 begin
  if Length(Trim(msg))>0 then begin
   Success('ERROR:');
   if EditState=0 then begin
    t:=StringToText(msg);
    if Text_NumLn(t)>0 then begin 
     for i:=0 to Text_NumLn(t)-1 do begin
      if i=0
      then msg:=Edit('('+Text_GetLn(t,i))
      else msg:=Edit(' '+Text_GetLn(t,i));
      Success(' '+Text_GetLn(t,i));
     end; 
     msg:=Edit(')Error ___Error___');
    end;
    bNul(Text_Free(t));
   end else bNul(Echo('ERROR:'+EOL+msg));
  end;
 end;
 {
 Dialog with Information message
 }
 procedure InformationDlg(msg:String);
 var i,t:Integer;
 begin
  if Length(Trim(msg))>0 then begin
   Success('INFORMATION:');
   if EditState=0 then begin
    t:=StringToText(msg);
    if Text_NumLn(t)>0 then begin 
     for i:=0 to Text_NumLn(t)-1 do begin
      if i=0
      then msg:=Edit('('+Text_GetLn(t,i))
      else msg:=Edit(' '+Text_GetLn(t,i));
      Success(' '+Text_GetLn(t,i));
     end; 
     msg:=Edit(')Information ___Information___');
    end;
    bNul(Text_Free(t));
   end else bNul(Echo('INFORMATION:'+EOL+msg));
  end;
 end;
 {
 Speak, i.e. send message to speech server.
 }
 function SpeakMsg(msg:String):Integer;
 var n:Integer;
 begin
  n:=0;
  msg:=Trim(msg);
  if Length(msg)>0 then
  if devSpeakSrv<>0 then
  n:=Round(DevSend(devSpeakSrv,'@speak='+msg+EOL));
  if n>0 then Details('Speak: '+msg);
  SpeakMsg:=n;
 end;
 {
 File open dialog...
 }
 function FileOpenDialog(FileName:String):Integer;
 var n:Integer;
 begin
  n:=0;
  if Length(Trim(FileName))>0 then
  //if DirExists(ExtractFilePath(FileName)) then
  n:=Round(SysEval('@Open -d '+FileName));
  FileOpenDialog:=n;
 end;
 {
 File Writeln...
 }
 function FileWriteln(FileName,Data:String):Integer;
 var n,status:Integer; size:Real;
 begin
  n:=0;
  if Length(Trim(FileName))>0 then
  if DirExists(ExtractFilePath(FileName)) then begin
   if FileExists(FileName)
   then status:=f_Reset(FileName,1)
   else status:=f_Rewrite(FileName,1);
   if status<>0 then Trouble('Error '+Str(status)+' on open file '+FileName) else begin
    size:=f_Size;
    if size>=0 then
    if f_Seek(size)=size then n:=f_Write(Data+EOL);
   end;
   bNul(f_Close);
   status:=IoResult;
   if status<>0 then Trouble('Error '+Str(status)+' on write file '+FileName);
  end;
  FileWriteln:=n;
 end;
 {
 Enforce a List to be Delim separated: "1;2,3 4" -> "1<delim>2<delim>3<delim>4".
 }
 function MendDelimList(List:String;Delim:Char):String;
 begin
  if WordCount(List)>1
  then List:=ExtractWord(1,List)+Delim+MendDelimList(SkipWords(1,List),Delim)
  else List:=ExtractWord(1,List);
  MendDelimList:=List;
 end;
 {
 Enforce a List to be comma separated: "1;2,3 4" -> "1,2,3,4".
 }
 function MendCommaList(List:String):String;
 begin
  if WordCount(List)>1
  then List:=ExtractWord(1,List)+','+MendCommaList(SkipWords(1,List))
  else List:=ExtractWord(1,List);
  MendCommaList:=List;
 end;
 {
 Enforce a List of week days to be numerical: "Mo,Fr" -> "1,5".
 }
 function MendDayOfWeek(List:String):String;
 var i:Integer;
 begin
  i:=Pos('-',List);
  if i>0 then List:=MendDayOfWeek(Copy(List,1,i-1))+'-'+MendDayOfWeek(Copy(List,i+1)) else
  if WordCount(List)>1
  then List:=MendDayOfWeek(ExtractWord(1,List))+','+MendDayOfWeek(SkipWords(1,List))
  else begin
   i:=0;
   if i=0 then i:=WordIndex(List,DayOfWeekList('En'));
   if i=0 then i:=WordIndex(List,DayOfWeekList('Ru'));
   if i=0 then i:=WordIndex(List,DayOfWeekList('Eng'));
   if i=0 then i:=WordIndex(List,DayOfWeekList('Rus'));
   if i=0 then i:=WordIndex(List,DayOfWeekList('English'));
   if i=0 then i:=WordIndex(List,DayOfWeekList('Russian'));
   if i>0 then List:=Str(i) else List:=ExtractWord(1,List);
  end;
  MendDayOfWeek:=List;
 end;
 {
 Enforce a List of month to be numerical: "Jan,May" -> "1,5".
 }
 function MendMonthName(List:String):String;
 var i:Integer;
 begin
  i:=Pos('-',List);
  if i>0 then List:=MendMonthName(Copy(List,1,i-1))+'-'+MendMonthName(Copy(List,i+1)) else
  if WordCount(List)>1
  then List:=MendMonthName(ExtractWord(1,List))+','+MendMonthName(SkipWords(1,List))
  else begin
   i:=0;
   if i=0 then i:=WordIndex(List,MonthList('Eng'));
   if i=0 then i:=WordIndex(List,MonthList('Rus'));
   if i=0 then i:=WordIndex(List,MonthList('English'));
   if i=0 then i:=WordIndex(List,MonthList('Russian'));
   if i>0 then List:=Str(i) else List:=ExtractWord(1,List);
  end;
  MendMonthName:=List;
 end;
 {
 Normalize Cron table.
 Replace empty strings to star *.
 Replace @ to current  minute,hour,day,month,week,year.
 Replace ? to startup  minute,hour,day,month,week,year.
 Replace # to TimeBase minute,hour,day,month,week,year.
 Replace alfabet month and week days to numerical.
 1 2 3 4 5 6    Word index
 * * * * * *
 | | | | | | 
 | | | | | +--- Years      (range: 1-9999)
 | | | | +----- Week days  (range: 1-7)
 | | | +------- Month      (range: 1-12)
 | | +--------- Month days (range: 1-31)
 | +----------- Hours      (range: 0-23)
 +------------- Minutes    (range: 0-59)
 }
 function MendCronTab(CronCmd:String):String;
  procedure Joker(var s:String; c:Char; ms:Real; k:Integer);
  var m:Integer;
  begin
   if Pos(Dump(c),s)=0 then m:=-1  else
   if k=1 then m:=ms2min(ms)       else
   if k=2 then m:=ms2hour(ms)      else
   if k=3 then m:=ms2day(ms)       else
   if k=4 then m:=ms2month(ms)     else
   if k=5 then m:=ms2DayOfWeek(ms) else
   if k=6 then m:=ms2year(ms)      else m:=-1;
   if m>=0 then s:=StrReplace(s,Dump(c),Str(m),3);
  end;
  function Star(s:String;k:Integer):String;
  begin
   if Length(s)=0 then s:='*';
   Joker(s,'?',CronStartMs,k);
   Joker(s,'#',TimeBase,k);
   Joker(s,'@',mSecNow,k);
   Star:=s;
  end;
 begin
  MendCronTab:=Star(MendCommaList(ExtractWordSpaces(1,CronCmd)),1)+' '+
               Star(MendCommaList(ExtractWordSpaces(2,CronCmd)),2)+' '+
               Star(MendCommaList(ExtractWordSpaces(3,CronCmd)),3)+' '+
               Star(MendMonthName(ExtractWordSpaces(4,CronCmd)),4)+' '+
               Star(MendDayOfWeek(ExtractWordSpaces(5,CronCmd)),5)+' '+
               Star(MendCommaList(ExtractWordSpaces(6,CronCmd)),6);
 end;
 {
 Read line from Guard task stdout pipe with CR terminator and LF ignore.
 }
 function Guard_Readln(var s:String):boolean;
 var p,q:integer;
 begin
  s:='';
  Guard_Readln:=false;
  if task_pid(Guard.Tid)<>0 then begin
   if Length(Guard.Buf)<MaxLeng
   then Guard.Buf:=Guard.Buf+task_recv(Guard.Tid,MaxLeng-Length(Guard.Buf));
   p:=Pos(chr(13),Guard.Buf);
   if p>0 then begin
    Guard_Readln:=true;
    if p>1 then s:=Copy(Guard.Buf,1,p-1);
    if Length(s)>0 then begin
     q:=Pos(chr(10),s);
     if q>0 then s:=Copy(s,q+1,MaxLeng);
    end;
    Guard.Buf:=Copy(Guard.Buf,p+1,MaxLeng);
    if Length(Guard.Buf)>0 then
    if Copy(Guard.Buf,1,1)=chr(10) then Guard.Buf:=Copy(Guard.Buf,2,MaxLeng);
   end else begin
    if Length(Guard.Buf)=MaxLeng then begin
     Trouble('Received line is too long!');
     Guard.Buf:='';
    end;
   end;
  end;
 end;
 {
 Send message to Guard task.
 Wait for some time if transmitter FIFO is over.
 }
 procedure Guard_Send(msg:string);
 var ms:real;
 begin
  if Guard.Tid<>0 then
  if Length(msg)>0 then begin
   if task_txspace(Guard.Tid)<Length(msg)+2 then begin
    ms:=msecnow;
    while(msecnow-ms<SendTimeOut) and (task_txspace(Guard.Tid)<Length(msg)+2) do bNul(Sleep(1));
   end;
   if task_send(Guard.Tid,msg+EOL)>0 then ViewExp('GUARD: '+msg) else Trouble('GUARD: '+msg);
  end;
 end;
 {
 Analyse data coming from Guard task stdout.
 }
 procedure Guard_Process(var Data:string);
 var cmd,arg:String;
 begin
  ViewImp('GUARD: '+Data);
  {
  "@cmd=arg" or "@cmd args" commands:
  }
  cmd:='';
  arg:='';
  if strFetch(Data,1)='@' then begin
   cmd:=ExtractWord(1,Data);
   arg:=Copy(Data,Pos(cmd,Data)+Length(cmd)+1);
   {
   OnException=class-name,comment
   crongrd send this message on exception occured.
   Standard handling is to send @Exit to restart EXE.
   }
   if IsSameText(cmd,'@OnException') then begin
    Guard_Send('@Exit');
    Data:='';
   end else
   {
   @Exit=n
   crongrd send this message on EXIT.
   }
   if IsSameText(cmd,'@Exit') then begin
    Success('Exit with code '+Trim(arg));
    Data:='';
   end else
   {
   Any other command
   }
   if Length(Data)>0 then begin
    Data:='';
   end;
  end;
  Data:='';
  cmd:='';
  arg:='';
 end;
 {
 Stop Guard task if one started.
 }
 procedure Guard_Stop;
 var ms:Real;
 begin
  if Guard.Tid>0 then begin
   if task_wait(Guard.Tid,0) then begin
    Success('crongrd termination will take some time.');
    Success('You should wait up to 10 sec...');
    Guard_Send('@Exit');
    ms:=msecnow;
    while (msecnow-ms<10000) and task_wait(Guard.Tid,100) do Guard_Send(EOL);
    if task_wait(Guard.Tid,0) then bNul(task_kill(Guard.Tid,0,1,0));
    if task_rxcount(Guard.Tid)>0 then
    while Guard_Readln(Guard.Line) do Guard_Process(Guard.Line);
    Success('crongrd exit code = '+str(task_result(Guard.Tid)));
   end;
   bNul(task_free(Guard.Tid));
  end;
  Guard.Tid:=0;
  Guard.Buf:='';
 end;
 {
 Kill Guard task if one started.
 }
 procedure Guard_Kill;
 var ms:Real;
 begin
  if Guard.Tid>0 then begin
   if task_wait(Guard.Tid,0) then begin
    Success('crongrd killing will take some time.');
    Success('You should wait up to 10 sec...');
    ms:=msecnow;
    Guard_Send('@Exit');
    while (msecnow-ms<10000) and task_wait(Guard.Tid,100) do Guard_Send(EOL);
    if task_wait(Guard.Tid,0) then bNul(task_kill(Guard.Tid,0,1,0));
    if task_rxcount(Guard.Tid)>0 then
    while Guard_Readln(Guard.Line) do Guard_Process(Guard.Line);
    Success('crongrd exit code = '+str(task_result(Guard.Tid)));
   end;
  end;
 end;
 {
 Clear Guard.
 }
 procedure Guard_Clear(ForceFree:Boolean);
 begin
  if ForceFree then begin
   if Guard.Timer<>0 then bNul(tm_free(Guard.Timer));
  end;
  Guard.Tid:=0;
  Guard.Timer:=0;
  Guard.Display:=0;
  Guard.Period:=15000;
  Guard.Line:='';
  Guard.Exe:='';
  Guard.App:='';
  Guard.Cmd:='';
  Guard.Dir:='';
  Guard.Buf:='';
 end;
 {
 Finalize Guard.
 }
 procedure Guard_Free;
 begin
  Guard_Stop;
  Guard_Clear(True);
 end;
 {
 Initialize Guard.
 }
 procedure Guard_Init;
 begin
  Guard_Clear(False);
  {---Find Guard executable---}
  Guard.Exe:=DaqFileRef(AdaptExeFileName(ReadIni('ProgramGuard')),'');
  if FileExists(Guard.Exe)
  then Success('ProgramGuard='+Guard.Exe)
  else Problem('Could not find GuardExe: '+Guard.Exe);
  {---Initialize timer---}
  Guard.Timer:=tm_new;
  if not tm_addint(Guard.Timer,GuardTimerPeriod) then Trouble('tm_addint fails.');
 end;
 {
 Start Guard server if one not started.
 }
 procedure Guard_Start;
 var i,j,t:Integer;
 begin
  if Guard.Tid=0 then begin
   {
   Initialize separate user task, run it invisible...
   }
   Guard.Tid:=task_init(Guard.Exe);
   if pos('?',task_ctrl(Guard.Tid,'HomeDir='+ExtractFilePath(Guard.Exe))
             +task_ctrl(Guard.Tid,'StdOutPipeSize=32')
             +task_ctrl(Guard.Tid,'StdInPipeSize=32')
             +task_ctrl(Guard.Tid,'Display=0')
          )>0
   then begin
    Trouble('User task setup error!');
    Guard_Stop;
   end;
   {
   Run task if one was created...
   }
   if Guard.Tid>0 then
   if task_run(Guard.Tid) then begin
    Success('TaskId  = '+str(Guard.Tid));
    Success('TaskPid = '+str(task_pid(Guard.Tid)));
    Success('TaskRef = '+str(task_ref(Guard.Tid)));
    Success('CmdLine = '+task_ctrl(Guard.Tid,'CmdLine'));
    Success('HomeDir = '+task_ctrl(Guard.Tid,'HomeDir'));
    Success('PipeIn  = '+task_ctrl(Guard.Tid,'StdInPipeSize'));
    Success('PipeOut = '+task_ctrl(Guard.Tid,'StdOutPipeSize'));
    Success('Display = '+task_ctrl(Guard.Tid,'Display'));
   end else begin
    Trouble('Could not start crongrd!');
    Guard_Stop;
   end;
   {
   Is it Ok with user task? Send preset parameters.
   Send all commands to self StdIn for future processing.
   }
   if Guard.Tid>0 then
   if task_wait(Guard.Tid,0) then begin
    Guard_Send('@AppName='+Guard.App);
    Guard_Send('@CmdLine='+Guard.Cmd);
    Guard_Send('@HomeDir='+Guard.Dir);
    Guard_Send('@Period='+Str(Guard.Period));
    Guard_Send('@Display='+Str(Guard.Display));
   end else Trouble('Unexpected termination!');
  end;
 end;
 {
 Guard polling.
 }
 procedure Guard_Poll;
 begin
  {
  If Guard process is not still running,
  try to start Guard process periodically.
  }
  if Guard.Tid=0 then
  if tm_event(Guard.Timer) then Guard_Start;
  {
  Communicate with Guad process if one still running...
  }
  if Guard.Tid>0 then
  if task_wait(Guard.Tid,0) then begin
   if task_rxcount(Guard.Tid)>0 then
   while Guard_Readln(Guard.Line) do Guard_Process(Guard.Line);
  end else begin
   Trouble('crongrd terminated, exit code = '+Str(task_result(Guard.Tid)));
   Guard_Stop;
  end;
 end;
 {
 Clear Cron server.
 }
 procedure Cron_Clear;
 var i:Integer;
 begin
  CronStartMs:=0;
  CronTabPoll:=0;
  for i:=1 to MaxCount do TheCron.cnt[i]:=0;
  for i:=1 to MaxCount do TheCron.per[i]:=0;
  for i:=1 to MaxCount do TheCron.pul[i]:=0;
  for i:=1 to MaxCount do TheCron.tim[i]:=0;
  for i:=1 to MaxCount do TheCron.job[i]:='';
  for i:=1 to MaxCount do TheCron.tab[i]:='';
  for i:=1 to MaxCount do TheCron.id[i]:='';
  Guard_Clear(False);
  TaskerClear;
 end;
 {
 Initialize Cron server.
 }
 procedure Cron_Init;
 begin
  Cron_Clear;
  CronStartMs:=mSecNow;
  CronTabPoll:=mSecNow;
  SilentEval:=False;
  Guard_Init;
  TaskerInit;
 end;
 {
 Free Cron server.
 }
 procedure Cron_Free;
 begin
  TaskerFree;
  Cron_Clear;
  Guard_Free;
 end;
 {
 Find Cron job.
 }
 function Cron_Find(id:String):Integer;
 var i,n:Integer;
 begin
  i:=1; n:=0;
  id:=Trim(id);
  if Length(id)>0 then
  while (i<=MaxCount) and (n=0) do begin
   if Length(TheCron.id[i])=0 then i:=MaxCount else
   if IsSameText(TheCron.id[i],id) then n:=i;
   i:=i+1;
  end;
  Cron_Find:=n;
 end;
 {
 Add item id to cron table. 
 }
 function Cron_Tab(id,tab:String):Integer;
 var i,n:Integer;
 begin
  i:=1; n:=0;
  id:=Trim(id);
  tab:=Trim(tab);
  if Length(id)>0 then
  while (i<=MaxCount) and (n=0) do begin
   if Length(TheCron.id[i])=0 then n:=i else
   if IsSameText(TheCron.id[i],id) then n:=i;
   i:=i+1;
  end;
  if n>0 then begin
   TheCron.id[n]:=id;
   TheCron.tab[n]:=tab;
   TheCron.job[n]:='';
   TheCron.tim[n]:=0;
   TheCron.pul[n]:=0;
   TheCron.per[n]:=0;
   TheCron.cnt[n]:=0;
  end;
  Cron_Tab:=n;
 end;
 {
 Add job text to cron table item id. 
 }
 function Cron_Job(id,job:String):Integer;
 var n:Integer;
 begin
  id:=Trim(id);
  job:=Trim(job);
  n:=Cron_Find(id);
  if n>0 then
  if Length(TheCron.job[n])+Length(EOL)+Length(job)>MaxLeng then n:=0 else
  if Length(TheCron.job[n])=0 then TheCron.job[n]:=job else TheCron.job[n]:=TheCron.job[n]+EOL+job;
  Cron_Job:=n;
 end;
 {
 Add pulser to cron table item id. 
 }
 function Cron_Pul(id:String; per,cnt:Integer):Integer;
 var n:Integer;
 begin
  id:=Trim(id);
  if per<=0 then per:=0;
  if cnt<=-1 then cnt:=-1;
  n:=Cron_Find(id);
  if n>0 then begin
   TheCron.pul[n]:=mSecNow;
   TheCron.per[n]:=per;
   TheCron.cnt[n]:=cnt;
  end;
  Cron_Pul:=n;
 end;
 {
 Delete cron table item id. 
 }
 function Cron_Del(id:String):Integer;
 var i,n:Integer;
 begin
  id:=Trim(id);
  n:=Cron_Find(id);
  if n>0 then begin
   i:=n;
   while i<=MaxCount do begin
    TheCron.id[i]:='';
    TheCron.tab[i]:='';
    TheCron.job[i]:='';
    TheCron.tim[i]:=0;
    TheCron.pul[i]:=0;
    TheCron.per[i]:=0;
    TheCron.cnt[i]:=0;
    i:=i+1;
    if i<=MaxCount then
    if Length(TheCron.id[i])=0 then i:=MaxCount+1 else begin
     TheCron.id[i-1]:=TheCron.id[i];
     TheCron.tab[i-1]:=TheCron.tab[i];
     TheCron.job[i-1]:=TheCron.job[i];
     TheCron.tim[i-1]:=TheCron.tim[i];
     TheCron.pul[i-1]:=TheCron.pul[i];
     TheCron.per[i-1]:=TheCron.per[i];
     TheCron.cnt[i-1]:=TheCron.cnt[i];
    end;
   end;
  end;
  Cron_Del:=n;
 end;
 {
 Run cron table item id. 
 }
 function Cron_Run(id:String):Integer;
 var n,dev:Integer;
 begin
  id:=Trim(id);
  n:=Cron_Find(id);
  if n>0 then
  if Length(TheCron.job[n])>0
  then n:=Round(DevPost(devMySelf,TheCron.job[n]+EOL))
  else n:=0;
  Cron_Run:=n;
 end;
 {
 Enable/disable job item id. 
 }
 function Cron_Enb(id,enb:String):Integer;
 var n,m:Integer;
 begin
  id:=Trim(id);
  enb:=Trim(enb);
  n:=Cron_Find(id);
  if n>0 then begin
   if Length(enb)>0 then
   if Val(enb)<>0 then TheCron.tim[n]:=Max(0,TheCron.tim[n]) else TheCron.tim[n]:=-1;
   m:=Sign(TheCron.tim[n]+0.5);
  end else m:=0;
  Cron_Enb:=n*m;
 end;
 {
 See Cron jobs.
 }
 function Cron_See(id:String):Integer;
 var i,j,n,t:Integer;
 begin
  i:=1; n:=0;
  id:=Trim(id);
  while i<=MaxCount do begin
   if Length(TheCron.id[i])=0 then i:=MaxCount else
   if IsSameText(id,'') then Success(StrFix(i,5,0)+' : @cron.tab '+TheCron.id[i]+' '+TheCron.tab[i]) else
   if IsSameText(TheCron.id[i],id) or IsSameText(id,'*') then begin
    Success(';-------- CronTab['+Str(i)+']: '+ExtractWord(1+Ord(TheCron.tim[i]>=0),'Disabled,Enabled'));
    Success('@cron.tab '+TheCron.id[i]+' '+TheCron.tab[i]);
    Success('@cron.pul '+TheCron.id[i]+' '+Str(TheCron.per[i])+' '+Str(TheCron.cnt[i]));
    t:=StringToText(TheCron.job[i]);
    for j:=0 to Text_NumLn(t)-1 do
    Success('@cron.job '+TheCron.id[i]+' '+Text_Getln(t,j));
    j:=Ord(Text_Free(t));
    n:=i;
   end;
   i:=i+1;
  end;
  Cron_See:=n;
 end;
 {
 Return true if Value corresponds to Template.
 Template may be:
  *        - any value
  */2      - each 2-nd
  3/2      - each 2-nd starting from 3
  1-3      - range from 1 to 3
  1,2,3    - from list
 }
 function Cron_Match(Template:String; Value:Integer):Boolean;
 var p,wc:Integer; Match:Boolean;
  function MatchRange(Value:Integer;r1,r2:Real;Step:Integer):Boolean;
  var Match:Boolean; n1,n2:Integer;
  begin
   Match:=False;
   if Step>0 then if Value>=0 then
   if not IsNan(r1) then if not IsInf(r1) then
   if not IsNan(r2) then if not IsInf(r2) then begin
    n1:=Round(r1); n2:=Round(r2);
    if n1<=Value then if Value<=n2 then
    if ((Value-n1) mod Step)=0 then Match:=True;
   end;
   MatchRange:=Match;
  end;
  function MatchValue(Template:String;Value,Step:Integer):Boolean;
  var p:Integer;
  begin
   p:=Pos('-',Template);
   if p>0
   then MatchValue:=MatchRange(Value,rVal(Copy(Template,1,p-1)),rVal(Copy(Template,p+1)),Step)
   else if (Length(Template)=0) or IsSameText(Template,'*')
   then MatchValue:=MatchRange(Value,0,MaxInt,Step)
   else MatchValue:=MatchRange(Value,rVal(Template),rVal(Template)*Ord(Step=1)+MaxInt*Ord(Step>1),Step);
  end;
 begin
  Match:=False;
  wc:=WordCount(Template);
  if wc>1 then begin
   p:=1;
   while (p<=wc) and not match do begin
    match:=Cron_Match(ExtractWord(p,Template),Value);
    p:=p+1;
   end; 
  end else begin
   p:=Pos('/',Template);
   if p>0
   then Match:=MatchValue(Copy(Template,1,p-1),Value,Val(Copy(Template,p+1)))
   else Match:=MatchValue(Template,Value,1);
  end;
  Cron_Match:=Match;
 end;
 {
 Return list of items which match the Template in (a,b) range.
 }
 function Match_List(Template:String; a,b:Integer):String;
 var list:String; i:Integer;
 begin
  list:='';
  for i:=a to b do
  if Cron_Match(Template,i) then
  if Length(list)=0 then list:=Str(i) else list:=list+','+Str(i);
  Match_List:=list;
  list:='';
 end; 
 {
 Cron timer.
 CronCmd = minute hour day month weekday year
 Minute(0-59), hour(0-23), day(1-31), month(1-12), weekday(1-7), year (1-9999)
 may be a number (n), asterisk (*), or divider (n/m). Number n means specified value,
 asterisk means any value, divider means "each m starting from n".
 Examples: * * * * * *     - each minute
           */2             - each 2 minutes (even)
           1/2             - each 2 minutes (odd)
           0 12            - 12:00 each day
           0 12 * * 5      - 12:00 each Friday
           0 12 * * 1 2011 - 12:00 each Monday 2011   
 }
 function Cron_Timer(var msec:Real; CronCmd:String):Boolean;
 var ms:Real; y,n,d,w,h,m:Integer;
 begin
  if msec<0 then Cron_Timer:=False else begin
   ms:=mSecNow;
   y:=ms2year(ms); n:=ms2month(ms); d:=ms2day(ms);
   w:=ms2DayOfWeek(ms); h:=ms2hour(ms); m:=ms2min(ms);
   ms:=datetime2ms(y,n,d,h,m,0,0);
   if msec=ms then Cron_Timer:=False else begin
    if msec<=0 then Cron_Timer:=False else
    if not Cron_Match(ExtractWordSpaces(1,CronCmd),m) then Cron_Timer:=False else
    if not Cron_Match(ExtractWordSpaces(2,CronCmd),h) then Cron_Timer:=False else
    if not Cron_Match(ExtractWordSpaces(3,CronCmd),d) then Cron_Timer:=False else
    if not Cron_Match(ExtractWordSpaces(4,CronCmd),n) then Cron_Timer:=False else
    if not Cron_Match(ExtractWordSpaces(5,CronCmd),w) then Cron_Timer:=False else
    if not Cron_Match(ExtractWordSpaces(6,CronCmd),y) then Cron_Timer:=False else
    Cron_Timer:=True;
    msec:=ms;
   end;
  end;
 end;
 {
 Cron pulser.
 }
 function Cron_Pulser(var pul:Real; per:Integer; var cnt:Integer):Boolean;
 var ms:Real; Pulse:Boolean;
 begin
  Pulse:=False;
  if per>0 then
  if cnt<>0 then begin
   ms:=mSecNow;
   if ms>=pul+per then begin
    if cnt>0 then cnt:=cnt-1;
    Pulse:=True;
    pul:=ms;
   end;
  end;
  Cron_Pulser:=Pulse;
 end;
 {
 Poll Cron server.
 }
 procedure Cron_Poll;
 var i,run:Integer; PollCronTab,CronTimer,CronPulser:Boolean;
 begin
  {
  Execute CRON actions...
  }
  PollCronTab:=False;
  if mSecNow>=CronTabPoll+CronPollPeriod then begin
   CronTabPoll:=mSecNow;
   PollCronTab:=True;
  end;
  i:=1;
  while i<=MaxCount do begin
   if Length(TheCron.id[i])=0 then i:=MaxCount else begin
    CronTimer:=False; CronPulser:=False;
    if PollCronTab then CronTimer:=Cron_Timer(TheCron.tim[i],TheCron.tab[i]); 
    if TheCron.tim[i]>=0 then CronPulser:=Cron_Pulser(TheCron.pul[i],TheCron.per[i],TheCron.cnt[i]);
    if CronTimer or CronPulser then begin
     run:=Cron_Run(TheCron.id[i]);
     Success('Launch '+TheCron.id[i]+' ('+Str(run)+') at '+GetDateTime(mSecNow));
    end;
   end;
   i:=i+1;
  end;
  {
  Execute Guard actions...
  }
  Guard_Poll;
  {
  Execute Tasker actions...
  }
  TaskerPolling;
  {
  Edit...
  }
  if EditStateDone then begin
   if EditTestResultName('___Error___') then EditReset;
   if EditTestResultName('___Warning___') then EditReset;
   if EditTestResultName('___Information___') then EditReset;
   MenuExitHandler;
  end;
  if EditStateDone then begin
   Problem('Unhandled edit detected!');
   EditReset;
  end else
  if EditStateError then begin
   Problem('Edit error detected!');
   EditReset;
  end;
 end;
 {
 Test Cron routines.
 }
 function Cron_Test(num:Integer):Integer;
 var nerr,nstr:Integer;
  procedure TestItem(Item,Argument,Result,Expect:String);
  begin
   if IsSameText(Result,Expect) then begin
    Success('Success '+Item+'('+Argument+') = "'+Result+'", expected "'+Expect+'".');
   end else begin
    Problem('Failure '+Item+'('+Argument+') = "'+Result+'", expected "'+Expect+'".');
    nerr:=nerr+1;
   end;
  end;
 begin
  nerr:=0;
  nstr:=MaxAvail;
  TestItem('DayOfWeekStr','Mo',DayOfWeekStr(1,'En'),'Mo');
  TestItem('DayOfWeekStr','Пн',DayOfWeekStr(1,'Ru'),'Пн');
  TestItem('DayOfWeekStr','Mon',DayOfWeekStr(1,'Eng'),'Mon');
  TestItem('DayOfWeekStr','Пнд',DayOfWeekStr(1,'Rus'),'Пнд');
  TestItem('MendList','1;2 3,4',MendCommaList('1;2 3,4'),'1,2,3,4');
  TestItem('MendDayOfWeek','Пн,Вт,Ср,Чт,Пт,Сб,Вс',MendDayOfWeek('Пн,Вт,Ср,Чт,Пт,Сб,Вс'),'1,2,3,4,5,6,7');
  TestItem('MendDayOfWeek','Пнд,Втр,Срд,Чтв,Птн,Сбт,Вск',MendDayOfWeek('Пнд,Втр,Срд,Чтв,Птн,Сбт,Вск'),'1,2,3,4,5,6,7');
  TestItem('MendCronTab','* * * * Mo,Fr',MendCronTab('* * * * Mo,Fr'),'* * * * 1,5 *');
  TestItem('MendCronTab','* * * * Mo-Fr',MendCronTab('* * * * Mo-Fr'),'* * * * 1-5 *');
  TestItem('MendCronTab','* * * Jan,May',MendCronTab('* * * Jan,May'),'* * * 1,5 * *');
  TestItem('MendCronTab','* * * Jan-May',MendCronTab('* * * Jan-May'),'* * * 1-5 * *');
  TestItem('Match_List','1-5',Match_List('1-5',0,59),'1,2,3,4,5');
  TestItem('Match_List','1,2,3,4,5',Match_List('1,2,3,4,5',0,59),'1,2,3,4,5');
  TestItem('Match_List','1-5,10-15',Match_List('1-5,10-15',0,59),'1,2,3,4,5,10,11,12,13,14,15');
  TestItem('Match_List','*/5',Match_List('*/5',0,59),'0,5,10,15,20,25,30,35,40,45,50,55');
  TestItem('Match_List','41,42,43',Match_List('41,42,43',0,59),'41,42,43');
  TestItem('Match_List','41/5',Match_List('41/5',0,59),'41,46,51,56');
  TestItem('Match_List','41',Match_List('41',0,59),'41');
  TestItem('Match_List','*/15,40-50/3,50/2',Match_List('*/15,40-50/3,50/2',0,59),
                        '0,15,30,40,43,45,46,49,50,52,54,56,58');
  Success('Now '+ms2MonthStr(mSecNow,'Eng')+','+ms2MonthStr(mSecNow,'English')
            +','+ms2MonthStr(mSecNow,'Rus')+','+ms2MonthStr(mSecNow,'Russian')
            +','+ms2DayOfWeekStr(mSecNow,'En')+','+ms2DayOfWeekStr(mSecNow,'Eng')
            +','+ms2DayOfWeekStr(mSecNow,'English')
            +','+ms2DayOfWeekStr(mSecNow,'Ru')+','+ms2DayOfWeekStr(mSecNow,'Rus')
            +','+ms2DayOfWeekStr(mSecNow,'Russian'));
  nstr:=MaxAvail-nstr;
  Success('String Manager delta = '+Str(nstr));
  Cron_Test:=nerr;
 end;
 {
 Start performance profiler.
 }
 procedure Profiler_Start;
 begin
  Profiler.Init.Kern:=rVal(ExtractWord(7,ParamStr('System Process Times')));
  Profiler.Init.User:=rVal(ExtractWord(8,ParamStr('System Process Times')));
  Profiler.Init.Vdpm:=VDPM_OpCount;
  Profiler.Init.Time:=mSecNow;
  Profiler.Last.Kern:=Profiler.Init.Kern;
  Profiler.Last.User:=Profiler.Init.User;
  Profiler.Last.Vdpm:=Profiler.Init.Vdpm;
  Profiler.Last.Time:=Profiler.Init.Time;
  Profiler.Curr.Kern:=0;
  Profiler.Curr.User:=0;
  Profiler.Curr.Vdpm:=0;
  Profiler.Curr.Time:=0;
  Profiler.Peak.Kern:=0;
  Profiler.Peak.User:=0;
  Profiler.Peak.Vdpm:=0;
  Profiler.Peak.Time:=0;
  Profiler.Rate.Kern:=0;
  Profiler.Rate.User:=0;
  Profiler.Rate.Vdpm:=0;
  Profiler.Rate.Time:=0;
 end;
 {
 Poll performance profiler.
 }
 procedure Profiler_Poll(Period:Integer);
 var dt:Real;
 begin
  if mSecNow>Profiler.Last.Time+Period then begin
   Profiler.Curr.Kern:=rVal(ExtractWord(7,ParamStr('System Process Times')));
   Profiler.Curr.User:=rVal(ExtractWord(8,ParamStr('System Process Times')));
   Profiler.Curr.Vdpm:=VDPM_OpCount;
   Profiler.Curr.Time:=mSecNow;
   dt:=1e-3*(Profiler.Curr.Time-Profiler.Last.Time);
   if dt>0 then begin
    Profiler.Rate.Kern:=1e-5*(Profiler.Curr.Kern-Profiler.Last.Kern)/dt;
    Profiler.Rate.User:=1e-5*(Profiler.Curr.User-Profiler.Last.User)/dt;
    Profiler.Rate.Vdpm:=1e-3*(Profiler.Curr.Vdpm-Profiler.Last.Vdpm)/dt;
    Profiler.Rate.Time:=Profiler.Rate.Kern+Profiler.Rate.User;
    Profiler.Peak.Kern:=Max(Profiler.Peak.Kern,Profiler.Rate.Kern);
    Profiler.Peak.User:=Max(Profiler.Peak.User,Profiler.Rate.User);
    Profiler.Peak.Vdpm:=Max(Profiler.Peak.Vdpm,Profiler.Rate.Vdpm);
    Profiler.Peak.Time:=Max(Profiler.Peak.Time,Profiler.Rate.Time);
   end;
   Profiler.Last.Kern:=Profiler.Curr.Kern;
   Profiler.Last.User:=Profiler.Curr.User;
   Profiler.Last.Vdpm:=Profiler.Curr.Vdpm;
   Profiler.Last.Time:=Profiler.Curr.Time;
  end;
 end;
 {
 Print performance profiler report.
 1:  Header,Date,Time
 2:  Average CPU load
 }
 procedure Profiler_Report(mode:Integer);
 var dt:Real;
 begin
  dt:=1e-3*(Profiler.Last.Time-Profiler.Init.Time);
  if iAnd(mode,1)>0 then begin
   Success('Profiler report at '+GetDateTime(mSecNow)+': '+StrFix(dt,8,0)+' sec.work');
  end;
  if iAnd(mode,2)*dt>0 then begin
   Success(' ---------+----------+----------+----------+-------------');
   Success(' CPU Load | Kernel,% |   User,% | Thread,% | VDPM,kOp/sec');
   Success(' ---------+----------+----------+----------+-------------');
   Success('  Average | '+StrFix(1e-5*(Profiler.Last.Kern-Profiler.Init.Kern)/dt,8,3)+' | '
                         +StrFix(1e-5*(Profiler.Last.User-Profiler.Init.User)/dt,8,3)+' | '
                         +StrFix(1e-5*(Profiler.Last.Kern+Profiler.Last.User-
                                      (Profiler.Init.Kern+Profiler.Init.User))/dt,8,3)+' | '
                         +StrFix(1e-3*(Profiler.Last.Vdpm-Profiler.Init.Vdpm)/dt,12,3));
   Success('  LastSec | '+StrFix(Profiler.Rate.Kern,8,3)+' | '
                         +StrFix(Profiler.Rate.User,8,3)+' | '
                         +StrFix(Profiler.Rate.Time,8,3)+' | '
                         +StrFix(Profiler.Rate.Vdpm,12,3));
   Success('  Peaking | '+StrFix(Profiler.Peak.Kern,8,3)+' | '
                         +StrFix(Profiler.Peak.User,8,3)+' | '
                         +StrFix(Profiler.Peak.Time,8,3)+' | '
                         +StrFix(Profiler.Peak.Vdpm,12,3));
   Success(' ---------+----------+----------+----------+-------------');
  end;
 end;
 {
 Exit DAQ system
 }
 procedure DoExitDaq(Sound:String);
 begin
  rNul(SysEval('SaveGuard=@guard'));
  rNul(SysEval('@guard root'));
  rNul(SysEval('_Daq_Force_Stop_=1'));
  rNul(SysEval('_Daq_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@view norm FormDaqControlDialog'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqStop'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqDone'));
  rNul(SysEval('@guard %SaveGuard'));
  rNul(SysEval('SaveGuard='));
  bNul(Voice(Sound));
 end;
 {
 Exit CRW system
 }
 procedure DoExitCrw(Sound:String);
 begin
  rNul(SysEval('SaveGuard=@guard'));
  rNul(SysEval('@guard root'));
  rNul(SysEval('_Daq_Force_Stop_=1'));
  rNul(SysEval('_Daq_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@view norm FormDaqControlDialog'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqStop'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqDone'));
  rNul(SysEval('_Crw_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@menu run FormCrwDaq.ActionFileExit'));
  rNul(SysEval('@guard %SaveGuard'));
  rNul(SysEval('SaveGuard='));
  bNul(Voice(Sound));
 end;
 {
 Construct shutdown command (system dependent).
 }
 procedure get_shutdown_command(var cmd:String; How:Char; Delay:Integer);
 var minutes:Integer;
 begin
  cmd:='';
  if IsUnix then begin
   cmd:='unix sudoit shutdown';
   minutes:=(Delay+59) div 60;
   if (How='r') then cmd:=cmd+' -r'; 
   if (How='s') then cmd:=cmd+' -h';
   if (cmd<>'') and (Delay>=0) then cmd:=cmd+' +'+Str(minutes);
   if (How='l') then cmd:='unix session-logout '+Str(delay);
  end;
  if IsWindows then begin
   cmd:=GetComSpec+' /c ';
   cmd:=cmd+AddPathDelim(ExtractFilePath(GetComSpec))+'shutdown.exe';
   cmd:=cmd+' -'+How+' -t '+Str(Delay);
  end;
 end;
 {
 Exit Windows after Delay seconds.
 How = l/s/r = logout/stop/restart
 }
 procedure DoExitWin(How:Char; Sound:String; Delay:Integer);
 var cmd:String;
 begin
  cmd:='';
  rNul(SysEval('SaveGuard=@guard'));
  rNul(SysEval('@guard root'));
  rNul(SysEval('_Daq_Force_Stop_=1'));
  rNul(SysEval('_Daq_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@view norm FormDaqControlDialog'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqStop'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqDone'));
  get_shutdown_command(cmd,How,Delay);
  rNul(SysEval('@run -hide '+cmd));
  rNul(SysEval('_Crw_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@menu run FormCrwDaq.ActionFileExit'));
  rNul(SysEval('@guard %SaveGuard'));
  rNul(SysEval('SaveGuard='));
  bNul(Voice(Sound));
  cmd:='';
 end;
 {
 Restart DAQ, enforce start when ForceStart=true
 }
 procedure DoRestartDaq(cfg:String; Sound:String; Delay:Integer; ForceStart:Boolean);
 begin
  rNul(SysEval('SaveGuard=@guard'));
  rNul(SysEval('@guard root'));
  rNul(SysEval('_Daq_Force_Stop_=1'));
  rNul(SysEval('_Daq_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@view norm FormDaqControlDialog'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqStop'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqDone'));
  rNul(SysEval('_Daq_Force_Start_='+Str(Ord(ForceStart))));
  rNul(SysEval('@run '+ParamStr('ProgName')+' '+cfg));
  rNul(SysEval('@sleep '+Str(Delay*1000)));
  rNul(SysEval('@async @silent @guard %SaveGuard'));
  rNul(SysEval('@async @silent SaveGuard='));
  bNul(Voice(Sound));
 end;
 {
 Another version of restart DAQ, enforce start when ForceStart=true
 }
 procedure DoReloadDaq(cfg:String; Sound:String; Delay:Integer; ForceStart:Boolean);
 begin
  rNul(SysEval('SaveGuard=@guard'));
  rNul(SysEval('@guard root'));
  rNul(SysEval('_Daq_Force_Stop_=1'));
  rNul(SysEval('_Daq_Force_Exit_=1'));
  rNul(SysEval('@view norm FormCrwDaq'));
  rNul(SysEval('@view norm FormDaqControlDialog'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqStop'));
  rNul(SysEval('@menu run FormDaqControlDialog.ActionDaqDone'));
  if ForceStart then begin
   rNul(SysEval('@run '+ParamStr('ProgName')+' '+cfg));
   rNul(SysEval('@sleep '+Str(Delay*1000)));
   rNul(SysEval('@async @silent @view norm FormCrwDaq'));
   rNul(SysEval('@async @silent @view norm FormDaqControlDialog'));
   rNul(SysEval('@async @silent @menu run FormDaqControlDialog.ActionDaqStart'));
  end;
  rNul(SysEval('@async @silent @guard %SaveGuard'));
  rNul(SysEval('@async @silent SaveGuard='));
  bNul(Voice(Sound));
 end;
 {
 Show\hide main ToolBar
 }
 function ShowMainToolBar(Visible:Boolean):Integer;
 begin
  if Visible
  then ShowMainToolBar:=Round(SysEval('@view show FormCrwDaq.ToolBar'))
  else ShowMainToolBar:=Round(SysEval('@view hide FormCrwDaq.ToolBar'));
 end;
 {
 Show\hide main StatusBar
 }
 function ShowMainStatusBar(Visible:Boolean):Integer;
 begin
  if Visible
  then ShowMainStatusBar:=Round(SysEval('@view show FormCrwDaq.StatusBar'))
  else ShowMainStatusBar:=Round(SysEval('@view hide FormCrwDaq.StatusBar'));
 end;
 {
 Show\hide DAQ SYSTEM control
 }
 function ShowMainDaqControl(Visible:Boolean):Integer;
 begin
  if Visible
  then ShowMainDaqControl:=Round(SysEval('@view norm FormDaqControlDialog'))
  else ShowMainDaqControl:=Round(SysEval('@view min  FormDaqControlDialog'));
 end;
 {
 Process data coming from standard input...
 }
 procedure StdIn_Processor(Data:String);
 var cmd,arg:String; cmdid:Integer; scl,swd:String; i,j,n:Integer; r:Real;
  function FitCmdLine(s:String):String;
  begin
   s:=StrReplace(s,'%*',Trim(ParamStr('1 ')+' '+ParamStr('2 ')+' '+ParamStr('3 ')
                        +' '+ParamStr('4 ')+' '+ParamStr('5 ')+' '+ParamStr('6 ')
                        +' '+ParamStr('7 ')+' '+ParamStr('8 ')+' '+ParamStr('9 ')),3);
   s:=StrReplace(s,'%1',ParamStr('1 '),3);
   s:=StrReplace(s,'%2',ParamStr('2 '),3);
   s:=StrReplace(s,'%3',ParamStr('3 '),3);
   s:=StrReplace(s,'%4',ParamStr('4 '),3);
   s:=StrReplace(s,'%5',ParamStr('5 '),3);
   s:=StrReplace(s,'%6',ParamStr('6 '),3);
   s:=StrReplace(s,'%7',ParamStr('7 '),3);
   s:=StrReplace(s,'%8',ParamStr('8 '),3);
   s:=StrReplace(s,'%9',ParamStr('9 '),3);
   if (Pos('$',s)>0) or (Pos('%',s)>0) then s:=ExpEnv(s);
   s:=Trim(s);
   if IsNonEmptyStr(s) and (Pos(StrFetch(s,1),'\/-')=0) then
   if (Pos('.\',s)=1) or (Pos('..\',s)=1) or (Pos('~\',s)=1) or (Pos('~~\',s)=1)
   or (Pos('./',s)=1) or (Pos('../',s)=1) or (Pos('~/',s)=1) or (Pos('~~/',s)=1)
   then s:=DaqFileRef(s,'');
   FitCmdLine:=s; s:='';
  end;
 begin
  if DebugFlagEnabled(dfViewImp) then ViewImp('CON: '+Data);
  {
  Handle "@cmd=arg" or "@cmd arg" commands:
  }
  scl:='';
  swd:='';
  cmd:='';
  arg:='';
  if GotCommandId(Data,cmd,arg,cmdid) then begin
   {
   Example: @cron.tab daily 0 12 * * * *
   }
   if (cmdid = cmd_crontab) then begin
    n:=Cron_Tab(ExtractWord(1,arg),MendCronTab(SkipWords(1,arg)));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.job daily @echo Hello, now 12:00.
   }
   if (cmdid = cmd_cronjob) then begin
    n:=Cron_Job(ExtractWord(1,arg),SkipWords(1,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.pul EverySecond  1000
   Example: @cron.pul Launch5Times 1000 5
   }
   if (cmdid = cmd_cronpul) then begin
    i:=iValDef(ExtractWord(2,arg),0);
    j:=iValDef(ExtractWord(3,arg),-1);
    n:=Cron_Pul(ExtractWord(1,arg),i,j*Ord(i>0));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.run daily
   }
   if (cmdid = cmd_cronrun) then begin
    n:=Cron_Run(ExtractWord(1,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.del daily
   }
   if (cmdid = cmd_crondel) then begin
    n:=Cron_Del(ExtractWord(1,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.enb daily 0
   }
   if (cmdid = cmd_cronenb) then begin
    n:=Cron_Enb(ExtractWord(1,arg),ExtractWord(2,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.see
   }
   if (cmdid = cmd_cronsee) then begin
    n:=Cron_See(ExtractWord(1,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.cpu
   Example: @cron.cpu start
   }
   if (cmdid = cmd_croncpu) then begin
    n:=iValDef(ExtractWord(1,arg),DefProfilerMode);
    if IsSameText(Trim(arg),'Start')
    then Profiler_Start
    else Profiler_Report(n);
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @cron.test
   }
   if (cmdid = cmd_crontest) then begin
    n:=Cron_Test(Val(ExtractWord(1,arg)));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @WinHide DEMO.GUI
   }
   if (cmdid = cmd_WinHide) then begin
    n:=Ord(WinHide(Trim(arg)));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @WinShow DEMO.GUI
   }
   if (cmdid = cmd_WinShow) then begin
    n:=Ord(WinShow(Trim(arg)));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @WinDraw DEMO.GUI
   }
   if (cmdid = cmd_WinDraw) then begin
    n:=Ord(WinDraw(Trim(arg)));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @WinSelect DEMO.GUI
   }
   if (cmdid = cmd_WinSelect) then begin
    n:=Ord(WinSelect(Trim(arg)));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @ShowMainToolBar 0
   }
   if (cmdid = cmd_ShowMainToolBar) then begin
    n:=ShowMainToolBar(Val(Trim(arg))=1);
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @ShowMainStatusBar 0
   }
   if (cmdid = cmd_ShowMainStatusBar) then begin
    n:=ShowMainStatusBar(Val(Trim(arg))=1);
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @ShowMainDaqControl 0
   }
   if (cmdid = cmd_ShowMainDaqControl) then begin
    n:=ShowMainDaqControl(Val(Trim(arg))=1);
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @Async @WinSelect DEMO.GUI
   }
   if (cmdid = cmd_Async) then begin
    r:=DevPost(devMySelf,Trim(arg)+EOL);
    Success(cmd+'='+Str(r));
    Data:='';
   end else
   {
   Example: @Eval @System @Async @View Hide Crw32.ToolBar
   }
   if (cmdid = cmd_Eval) then begin
    r:=Eval(Trim(arg));
    Success(cmd+'='+Str(r));
    Data:='';
   end else
   {
   Example: @Run cmd.exe
   Same as @Eval @System @Async @Run cmd.exe
   }
   if (cmdid = cmd_Run) then begin
    r:=SysEval('@Run '+Trim(arg));
    Success(cmd+'='+Str(r));
    Data:='';
   end else
   {
   Example: @Pid kill dns.exe
   Same as @Eval @System @Async @Pid kill dns.exe
   }
   if (cmdid = cmd_Pid) then begin
    r:=SysEval('@Pid '+Trim(arg));
    Success(cmd+'='+Str(r));
    Data:='';
   end else
   {
   Example: @Browse http://mail.ru/
   }
   if (cmdid = cmd_Browse) then begin
    if devWebSrv=0 then WebBrowser(arg) else
    if DevPost(devWebSrv,'@Browse '+arg+EOL)=0 then WebBrowser(arg);
    Data:='';
   end else
   {
   Example: @Speak Hello, world!
   }
   if (cmdid = cmd_Speak) then begin
    if Length(Trim(arg))>0 then n:=SpeakMsg(Trim(arg)) else n:=0;
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @Voice Siren
   }
   if (cmdid = cmd_Voice) then begin
    if Length(Trim(arg))>0 then n:=Ord(Voice(Trim(arg))) else n:=0;
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @Error Error message! %0D%0A %0D%0A Message 1. %0D%0A Message 2.
   }
   if (cmdid = cmd_Error) then begin
    ErrorDlg(Trim(URL_Decode(arg)));
    Data:='';
   end else
   {
   Example: @Warning Warning message! %0D%0A %0D%0A Message 1. %0D%0A Message 2.
   }
   if (cmdid = cmd_Warning) then begin
    WarningDlg(Trim(URL_Decode(arg)));
    Data:='';
   end else
   {
   Example: @Information Information message! %0D%0A %0D%0A Message 1. %0D%0A Message 2.
   }
   if (cmdid = cmd_Information) then begin
    InformationDlg(Trim(URL_Decode(arg)));
    Data:='';
   end else
   {
   Example: @Echo msg
   }
   if (cmdid = cmd_Echo) then begin
    writeln(DevName+' : '+arg);
    Data:='';
   end else
   {
   Example: @Shutdown Daq Exit Bye       - Exit DAQ with sound Bye 
            @Shutdown Daq Restart Hi 1 1 - Restart DAQ, with sound Hi, delay 1 sec, force start
            @Shutdown Crw Exit Bye       - Exit CRW, with sound Bye
            @Shutdown Win Exit Bye 30    - Exit Windows, with sound Bye, delay 30 sec
            @Shutdown Win Logout Bye 30  - Logout Windows, with sound Bye, delay 30 sec
            @Shutdown Win Restart Bye 30 - Restart Windows, with sound Bye, delay 30 sec
   }
   if (cmdid = cmd_Shutdown) then begin
    if IsSameText('Daq',ExtractWord(1,arg)) then begin
     i:=iValDef(ExtractWord(4,arg),1);
     n:=iValDef(ExtractWord(5,arg),1);
     if IsSameText('Exit',    ExtractWord(2,arg)) then DoExitDaq(ExtractWord(3,arg));
     if IsSameText('Restart', ExtractWord(2,arg)) then DoRestartDaq(ParamStr('DaqConfigFile'),
                                                                    ExtractWord(3,arg),i,n>0);
    end;
    if IsSameText('Crw',ExtractWord(1,arg)) then begin
     if IsSameText('Exit',    ExtractWord(2,arg)) then DoExitCrw(ExtractWord(3,arg));
    end;
    if IsSameText('Win',ExtractWord(1,arg)) then begin
     i:=iValDef(ExtractWord(4,arg),30);
     if IsSameText('Exit',    ExtractWord(2,arg)) then DoExitWin('s',ExtractWord(3,arg),i);
     if IsSameText('Logout',  ExtractWord(2,arg)) then DoExitWin('l',ExtractWord(3,arg),i);
     if IsSameText('Restart', ExtractWord(2,arg)) then DoExitWin('r',ExtractWord(3,arg),i);
    end;
    Data:='';
   end else
   {
   Example: @If IsWindows @Echo Running OS is Windows.
   }
   if (cmdid = cmd_If) then begin
    if (WordCountDelims(arg,' '+EOL)>1) then begin
     r:=Eval(ExtractWordDelims(1,arg,' '+EOL));
     if (r<>0) and not IsNan(r) then begin
      if IsSameText(ExtractWordDelims(2,arg,' '+EOL),'then')
      then StdIn_Processor(SkipWordsDelims(2,arg,' '+EOL))
      else StdIn_Processor(SkipWordsDelims(1,arg,' '+EOL));
     end;
    end;
    Data:='';
   end else
   {
   Example: @IfComputerName alidcscom252 @Warning It's PHOS cooling plant server!
   }
   if (cmdid = cmd_IfComputerName) then begin
    if WordCount(arg)>1 then
    if IsSameText(ExtractWord(1,arg),ParamStr('ComputerName'))
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfNotComputerName alidcscom252 @Warning It's not PHOS cooling plant server!
   }
   if (cmdid = cmd_IfNotComputerName) then begin
    if WordCount(arg)>1 then
    if not IsSameText(ExtractWord(1,arg),ParamStr('ComputerName'))
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfProcessExists Crw32.exe @Warning Process Crw32.exe exists!
   }
   if (cmdid = cmd_IfProcessExists) then begin
    if WordCount(arg)>1 then
    if PidCounter(URL_Decode(ExtractWord(1,arg)))>0
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfNotProcessExists cmd.exe @Warning Process cmd.exe not exists!
   }
   if (cmdid = cmd_IfNotProcessExists) then begin
    if WordCount(arg)>1 then
    if PidCounter(URL_Decode(ExtractWord(1,arg)))=0
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfFileExists ..\Data\Demo.txt @Warning File ..\Data\Demo.txt exists!
   }
   if (cmdid = cmd_IfFileExists) then begin
    if WordCount(arg)>1 then
    if FileExists(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''))
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfNotFileExists ..\Data\Demo.txt @Warning File ..\Data\Demo.txt is not exists!
   }
   if (cmdid = cmd_IfNotFileExists) then begin
    if WordCount(arg)>1 then
    if not FileExists(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''))
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfDirExists ..\Data @Warning Directory ..\Data exists!
   }
   if (cmdid = cmd_IfDirExists) then begin
    if WordCount(arg)>1 then
    if DirExists(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''))
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @IfNotDirExists ..\Data @Warning Directory ..\Data is not exists!
   }
   if (cmdid = cmd_IfNotDirExists) then begin
    if WordCount(arg)>1 then
    if not DirExists(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''))
    then StdIn_Processor(SkipWords(1,arg));
    Data:='';
   end else
   {
   Example: @MkDir ..\Data
   }
   if (cmdid = cmd_MkDir) then begin
    if Length(Trim(arg))=0 then n:=0 else
    n:=Ord(MkDir(DaqFileRef(URL_Decode(Trim(arg)),'')));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @FileErase ..\Data\Temp.txt
   }
   if (cmdid = cmd_FileErase) then begin
    if Length(Trim(arg))=0 then n:=0 else
    n:=Ord(FileErase(DaqFileRef(URL_Decode(Trim(arg)),'')));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @FileCopy ..\Data\Temp.txt ..\Data\Test.txt
   }
   if (cmdid = cmd_FileCopy) then begin
    if Length(Trim(arg))=0 then n:=0 else
    n:=Ord(FileCopy(URL_Packed(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''))+' '+
                    URL_Packed(DaqFileRef(URL_Decode(ExtractWord(2,arg)),''))));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @FileRename ..\Data\Temp.txt ..\Data\Test.txt
   }
   if (cmdid = cmd_FileRename) then begin
    if Length(Trim(arg))=0 then n:=0 else
    n:=Ord(FileRename(URL_Packed(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''))+' '+
                      URL_Packed(DaqFileRef(URL_Decode(ExtractWord(2,arg)),''))));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @FileOpenDialog ..\Data\*.dat
   }
   if (cmdid = cmd_FileOpenDialog) then begin
    if Length(Trim(arg))=0 then n:=0 else
    n:=FileOpenDialog(DaqFileRef(URL_Decode(Trim(arg)),''));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @FileWriteln ..\Data\demo.txt Hello, world.
   }
   if (cmdid = cmd_FileWriteln) then begin
    if Length(Trim(arg))=0 then n:=0 else
    n:=FileWriteln(DaqFileRef(URL_Decode(ExtractWord(1,arg)),''),SkipWords(1,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @DevMsg      &SpeakSrv @Speak=Hello
            @DevSend     &SpeakSrv @Speak=Hello
            @DevSendMsg  &SpeakSrv @Speak=Hello
   }
   if (cmdid = cmd_DevMsg)
   or (cmdid = cmd_DevSend)
   or (cmdid = cmd_DevSendMsg) then begin
    i:=RefFind('Device '+ExtractWord(1,arg));
    if i=0 then n:=0 else n:=Round(DevSend(i,Trim(SkipWords(1,arg))+EOL));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @DevPost     &SpeakSrv @Speak=Hello
            @DevPostMsg  &SpeakSrv @Speak=Hello
   }
   if (cmdid = cmd_DevPost)
   or (cmdid = cmd_DevPostMsg) then begin
    i:=RefFind('Device '+ExtractWord(1,arg));
    if i=0 then n:=0 else n:=Round(DevPost(i,Trim(SkipWords(1,arg))+EOL));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @Guard.AppName=~~\Crw32Go.exe
   }
   if (cmdid = cmd_GuardAppName) then begin
    if Length(Trim(arg))>0 then Guard.App:=DaqFileRef(AdaptExeFileName(arg),'') else Guard.Exe:='';
    Success(cmd+'='+Guard.App);
    Data:='';
   end else
   {
   Example: @Guard.CmdLine=%* или %1 %2 %3 %4 %5 %6 %7 или ..\Config\demo.cfg
   }
   if (cmdid = cmd_GuardCmdLine) then begin
    i:=1;
    scl:='';
    while i>0 do begin
     swd:=ExtractWordDelims(i,arg,'    ');
     if Length(swd)=0 then i:=0 else begin
      if Length(scl)>0 then scl:=scl+' ';
      scl:=scl+FitCmdLine(swd);
      i:=i+1;
     end;
    end;
    if WordCount(scl)>0 then Guard.Cmd:=scl else Guard.Cmd:='';
    Success(cmd+'='+Guard.Cmd);
    Data:='';
   end else
   {
   Example: @Guard.HomeDir=~~\
   }
   if (cmdid = cmd_GuardHomeDir) then begin
    if Length(Trim(arg))>0 then Guard.Dir:=DaqFileRef(AdaptFileName(arg),'') else Guard.Dir:='';
    Success(cmd+'='+Guard.Dir);
    Data:='';
   end else
   {
   Example: @Guard.Period=15000
   }
   if (cmdid = cmd_GuardPeriod) then begin
    if not IsNan(rVal(arg)) then Guard.Period:=Round(rVal(arg));
    Success(cmd+'='+Str(Guard.Period));
    Data:='';
   end else
   {
   Example: @Guard.Display=0
   }
   if (cmdid = cmd_GuardDisplay) then begin
    if not IsNan(rVal(arg)) then Guard.Display:=Round(rVal(arg));
    Success(cmd+'='+Str(Guard.Display));
    Data:='';
   end else
   {
   Example: @Guard.Start
   }
   if (cmdid = cmd_GuardStart) then begin
    if Guard.Tid=0 then Guard_Start;
    if not tm_start(Guard.Timer) then Trouble('tm_start fails.');
    Success(cmd+'='+Str(Task_Pid(Guard.Tid)));
    Data:='';
   end else
   {
   Example: @Guard.Stop
   }
   if (cmdid = cmd_GuardStop) then begin
    if Guard.Tid>0 then Guard_Stop;
    if not tm_stop(Guard.Timer) then Trouble('tm_stop fails.');
    Success(cmd+'='+Str(Task_Pid(Guard.Tid)));
    Data:='';
   end else
   {
   Example: @Guard.View
   }
   if (cmdid = cmd_GuardView) then begin
    Success('Guard.AppName = '+Guard.App);
    Success('Guard.CmdLine = '+Guard.Cmd);
    Success('Guard.HomeDir = '+Guard.Dir);
    Success('Guard.Display = '+Str(Guard.Display));
    Success('Guard.Period  = '+Str(Guard.Period));
    Success('Guard.Running = '+Str(Ord(Task_Wait(Guard.Tid,0))));
    Success('Guard.Pid     = '+Str(Task_Pid(Guard.Tid)));
    Data:='';
   end else
   {
   Example: @SilentEval 1
   }
   if (cmdid = cmd_SilentEval) then begin
    SilentEval:=(iValDef(Trim(Arg),Ord(SilentEval))<>0);
    Success(cmd+'='+Str(Ord(SilentEval)));
    Data:='';
   end else
   {
   Example: @task_init demo
   }
   if (cmdid = cmd_task_init) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_free demo
   }
   if (cmdid = cmd_task_free) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_ctrl demo CmdLine = cmd /c dir
   }
   if (cmdid = cmd_task_ctrl) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_view demo
   }
   if (cmdid = cmd_task_view) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_run demo
   }
   if (cmdid = cmd_task_run) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_kill demo 1 0 1000
   }
   if (cmdid = cmd_task_kill) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_poll demo 100 10000
   }
   if (cmdid = cmd_task_poll) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_send demo exit
   }
   if (cmdid = cmd_task_send) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   Example: @task_recv demo @devpost &Demo @received $*
   }
   if (cmdid = cmd_task_recv) then begin
    n:=Ord(TaskerHandler(cmdid,arg));
    Success(cmd+'='+Str(n));
    Data:='';
   end else
   {
   @MenuExitOpen
   }
   if (cmdid=cmd_MenuExitOpen) then begin
    MenuExitStarter;
    Data:='';
   end else
   {
   Handle other commands by default handler...
   }
   StdIn_DefaultHandler(Data,cmd,arg);
  end;
  Data:='';
  cmd:='';
  arg:='';
  scl:='';
  swd:='';
 end;
 {
 Clear user application strings...
 }
 procedure ClearApplication;
 begin
  Cron_Clear;
 end;
 {
 User application Initialization...
 }
 procedure InitApplication;
 begin
  Cron_Init;
  Profiler_Start;
  StdIn_SetScripts('@StartupScript','@FinallyScript');
  StdIn_SetTimeouts(0,iValDef(ReadIni('StopTimeOut'),DefStopTimeOut),MaxInt,0);
  SilentEval:=(iValDef(ReadIni('SilentEval'),Ord(SilentEval))<>0);
  cmd_crontab            := RegisterStdInCmd('@cron.tab',           '');
  cmd_cronjob            := RegisterStdInCmd('@cron.job',           '');
  cmd_cronpul            := RegisterStdInCmd('@cron.pul',           '');
  cmd_cronrun            := RegisterStdInCmd('@cron.run',           '');
  cmd_crondel            := RegisterStdInCmd('@cron.del',           '');
  cmd_cronenb            := RegisterStdInCmd('@cron.enb',           '');
  cmd_cronsee            := RegisterStdInCmd('@cron.see',           '');
  cmd_croncpu            := RegisterStdInCmd('@cron.cpu',           '');
  cmd_crontest           := RegisterStdInCmd('@cron.test',          '');
  cmd_WinHide            := RegisterStdInCmd('@WinHide',            '');
  cmd_WinShow            := RegisterStdInCmd('@WinShow',            '');
  cmd_WinDraw            := RegisterStdInCmd('@WinDraw',            '');
  cmd_WinSelect          := RegisterStdInCmd('@WinSelect',          '');
  cmd_ShowMainToolBar    := RegisterStdInCmd('@ShowMainToolBar',    '');
  cmd_ShowMainStatusBar  := RegisterStdInCmd('@ShowMainStatusBar',  '');
  cmd_ShowMainDaqControl := RegisterStdInCmd('@ShowMainDaqControl', '');
  cmd_Async              := RegisterStdInCmd('@Async',              '');
  cmd_Eval               := RegisterStdInCmd('@Eval',               '');
  cmd_Run                := RegisterStdInCmd('@Run',                '');
  cmd_Pid                := RegisterStdInCmd('@Pid',                '');
  cmd_Browse             := RegisterStdInCmd('@Browse',             '');
  cmd_Speak              := RegisterStdInCmd('@Speak',              '');
  cmd_Voice              := RegisterStdInCmd('@Voice',              '');
  cmd_Error              := RegisterStdInCmd('@Error',              '');
  cmd_Warning            := RegisterStdInCmd('@Warning',            '');
  cmd_Information        := RegisterStdInCmd('@Information',        '');
  cmd_Echo               := RegisterStdInCmd('@Echo',               '');
  cmd_Shutdown           := RegisterStdInCmd('@Shutdown',           '');
  cmd_If                 := RegisterStdInCmd('@If',                 '');
  cmd_IfComputerName     := RegisterStdInCmd('@IfComputerName',     '');
  cmd_IfNotComputerName  := RegisterStdInCmd('@IfNotComputerName',  '');
  cmd_IfProcessExists    := RegisterStdInCmd('@IfProcessExists',    '');
  cmd_IfNotProcessExists := RegisterStdInCmd('@IfNotProcessExists', '');
  cmd_IfFileExists       := RegisterStdInCmd('@IfFileExists',       '');
  cmd_IfNotFileExists    := RegisterStdInCmd('@IfNotFileExists',    '');
  cmd_IfDirExists        := RegisterStdInCmd('@IfDirExists',        '');
  cmd_IfNotDirExists     := RegisterStdInCmd('@IfNotDirExists',     '');
  cmd_MkDir              := RegisterStdInCmd('@MkDir',              '');
  cmd_FileErase          := RegisterStdInCmd('@FileErase',          '');
  cmd_FileCopy           := RegisterStdInCmd('@FileCopy',           '');
  cmd_FileRename         := RegisterStdInCmd('@FileRename',         '');
  cmd_FileOpenDialog     := RegisterStdInCmd('@FileOpenDialog',     '');
  cmd_FileWriteln        := RegisterStdInCmd('@FileWriteln',        '');
  cmd_DevMsg             := RegisterStdInCmd('@DevMsg',             '');
  cmd_DevSend            := RegisterStdInCmd('@DevSend',            '');
  cmd_DevPost            := RegisterStdInCmd('@DevPost',            '');
  cmd_DevSendMsg         := RegisterStdInCmd('@DevSendMsg',         '');
  cmd_DevPostMsg         := RegisterStdInCmd('@DevPostMsg',         '');
  cmd_GuardAppName       := RegisterStdInCmd('@Guard.AppName',      '');
  cmd_GuardCmdLine       := RegisterStdInCmd('@Guard.CmdLine',      '');
  cmd_GuardHomeDir       := RegisterStdInCmd('@Guard.HomeDir',      '');
  cmd_GuardPeriod        := RegisterStdInCmd('@Guard.Period',       '');
  cmd_GuardDisplay       := RegisterStdInCmd('@Guard.Display',      '');
  cmd_GuardStart         := RegisterStdInCmd('@Guard.Start',        '');
  cmd_GuardStop          := RegisterStdInCmd('@Guard.Stop',         '');
  cmd_GuardView          := RegisterStdInCmd('@Guard.View',         '');
  cmd_SilentEval         := RegisterStdInCmd('@SilentEval',         '');
  cmd_task_init          := RegisterStdInCmd('@task_init',          '');
  cmd_task_free          := RegisterStdInCmd('@task_free',          '');
  cmd_task_ctrl          := RegisterStdInCmd('@task_ctrl',          '');
  cmd_task_view          := RegisterStdInCmd('@task_view',          '');
  cmd_task_run           := RegisterStdInCmd('@task_run',           '');
  cmd_task_kill          := RegisterStdInCmd('@task_kill',          '');
  cmd_task_poll          := RegisterStdInCmd('@task_poll',          '');
  cmd_task_send          := RegisterStdInCmd('@task_send',          '');
  cmd_task_recv          := RegisterStdInCmd('@task_recv',          '');
  cmd_MenuExitOpen       := RegisterStdInCmd('@MenuExitOpen',       '');
 end;
 {
 User application Finalization...
 }
 procedure FreeApplication;
 begin
  Cron_Free;
  Profiler_Report(DefProfilerMode);
 end;
 {
 User application Polling...
 }
 procedure PollApplication;
 begin
  Cron_Poll;
  Profiler_Poll(ProfilerPeriod);
 end;

{***************************************************}
{***************************************************}
{***                                             ***}
{***  MMM    MMM        AAA   IIII   NNN    NN   ***}
{***  MMMM  MMMM       AAAA    II    NNNN   NN   ***}
{***  MM MMMM MM      AA AA    II    NN NN  NN   ***}
{***  MM  MM  MM     AA  AA    II    NN  NN NN   ***}
{***  MM      MM    AAAAAAA    II    NN   NNNN   ***}
{***  MM      MM   AA    AA   IIII   NN    NNN   ***}
{***                                             ***}
{***************************************************}
{$I _std_main}{*** Please never change this code ***}
{***************************************************}
