////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2026 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user program. //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Console program sample.                                                    //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241022 - Sample created by A.K.                                          //
// 20260222 - Modified for FPC/Win32/Linux by A.K.                            //
////////////////////////////////////////////////////////////////////////////////

program crwgo;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics, interfaces,
 _crw_alloc, _crw_cmdargs, _crw_environ, _crw_proc, _crw_rtc,
 _crw_str, _crw_utf8, _crw_fio, _crw_fifo, _crw_ef, _crw_regexp,
 _crw_wmctrl, _crw_task, _crw_crypt, _crw_uac, _crw_wine, _crw_sect,
 _crw_syslog, _crw_sesman, _crw_az;

const
 FSign_IV = '©Alexey.Kuryakin';
 FSign_PW = '©kouriakine@mail.ru,Sarov,Russia';
 DefCap   = 'cap_net_bind_service,cap_sys_nice+ep';
 ExeList  : array[0..3] of PChar = ('crwdaq.exe','..\crwdaq.exe','..\..\crwdaq.exe','..\..\..\crwdaq.exe');

var
 AppName : LongString = '';
 CmdLine : LongString = '';
 nShow   : Integer    = SW_SHOW;

type
 EHalt = class(Exception);

 {
 Utility functions
 }
procedure PrintReport(const Msg:LongString; sev:Integer=sev_Print);
begin
 if IsEmptyStr(Msg) then Exit;
 Writeln(StdErr,Msg);
 SysLog.Info(sev,Msg);
end;

procedure ErrorReport(const Msg:LongString);
var cmd,ans:LongString;
begin
 if IsEmptyStr(Msg) then Exit;
 Writeln(StdErr,Msg); ans:='';
 SysLog.Error(sev_Fail,Msg);
 if IsWindows and (detect_wine_number>0) then Exit;
 cmd:='unix tooltip-notifier preset stdError delay 60000 text '+QArg('crwgo: '+Msg);
 RunCommand(cmd,ans);
end;

procedure Halt(ECode:Integer);
begin
 ExitCode:=ECode;
 Raise EHalt.Create('Program terminated with code '+IntToStr(ECode));
end;

function GetCommandLineArguments:LongString;
begin
 Result:=SkipPhrases(1,GetCommandLine,JustSpaces);
end;

function ValidateArgs(const Args:LongString):LongString;
const opts='-session -number -r -e -d -icon=0 -icon=1 -icon=2 -icon=3 -icon=4 -icon=5 -icon=6';
var opt,bad:LongString; i:Integer;
begin
 If IsEmptyStr(Args) then Exit(Args);
 Result:=' '+Tab2Space(Args,1)+' ';
 for i:=1 to WordCount(opts,JustSpaces) do begin
  opt:=' '+ExtractWord(i,opts,JustSpaces)+' ';
  bad:=StringReplace(opt,'-','/',[rfReplaceAll]);
  if (PosI(bad,Result)>0) then Result:=StringReplace(Result,bad,opt,[rfReplaceAll,rfIgnoreCase]);
 end;
 Result:=Trim(Result);
end;

function IsGoodFileSign(const FileName:LongString):Boolean;
var s,md5,fn,pn,fs,cs,w1,w2:LongString; n:Int64; i:Integer; Lines:TStringList;
const nLimit=MegaByte*128;
begin
 Result:=False;
 try
  if not FileExists(FileName) then begin
   ErrorReport('File not found '+QArg(FileName));
   Exit;
  end;
  if not FileIsExecutable(FileName) then begin
   ErrorReport('File not executable '+QArg(FileName));
   Exit;
  end;
  n:=GetFileSize(FileName);
  if not InRange(n,1,nLimit) then begin
   ErrorReport('Invalid file size of '+QArg(FileName));
   Exit;
  end;
  s:=StringFromFile(FileName,n);
  if (Length(s)<>n) then begin
   ErrorReport('Could not read '+QArg(FileName));
   Exit;
  end;
  if (Pos(FSign_IV,s)=0) or (Pos(FSign_PW,s)=0) then begin
   ErrorReport('Invalid sign of '+QArg(FileName));
   Exit;
  end;
  md5:=GetMd5FromFile(FileName,df_Hex);
  fn:=AddPathDelim(ExtractFileDir(FileName))+'secure.md5';
  if not FileExists(fn) then fn:=AddPathDelim(ExtractFileDir(FileName))+'checksum.md5';
  if not FileExists(fn) then begin
   ErrorReport('Checksum file not found '+QArg(fn));
   Exit;
  end;
  cs:='';
  pn:=ExtractFileNameExt(FileName);
  fs:=StringFromFile(fn,GetFileSize(fn));
  if (fs='') then begin
   ErrorReport('Could not read '+QArg(fn));
   Exit;
  end;
  Lines:=TStringList.Create;
  try
   Lines.Text:=fs;
   for i:=0 to Lines.Count-1 do begin
    w1:=ExtractWord(1,Lines[i],ScanSpaces+['*']);
    w2:=ExtractWord(2,Lines[i],ScanSpaces+['*']);
    if SameText(w2,pn) then begin
     cs:=w1; break;
    end;
   end;
  finally
   Kill(Lines);
  end;
  if IsEmptyStr(cs) or not SameText(cs,md5) then begin
   ErrorReport('Invalid checksum '+QArg(FileName));
   Exit;
  end;
  Result:=True;
 except
  Result:=False;
 end;
end;

function FindFile(aFileName,aPath:LongString):LongString;
var Path:LongString;
begin
 Result:='';
 aPath:=Trim(aPath);
 aFileName:=Trim(aFileName);
 if Length(aFileName)>0 then begin
  Path:=Trim(GetEnv('PATH'));
  if Length(aPath)>0 then Path:=aPath+PathSeparator+Path;
  Result:=FileSearch(aFileName,Path);
 end;
end;

function GetAppName:LongString;
var i:Integer;
begin
 Result:='';
 for i:=Low(ExeList) to High(ExeList) do begin
  Result:=AddPathDelim(HomeDir)+AdaptExeFileName(ExeList[i]);
  Result:=UnifyFileAlias(Result,ua_FileLow);
  if FileExists(Result) then Exit;
 end;
 if not FileExists(Result) then Result:=AdaptExeFileName(ExeList[i]);
end;

function ClearParentCrwdaqPidFileIfNeed:Boolean;
var PidFile:LongString; CrwPid,RunPid:TPid; i,CrwSid:Integer;
begin
 Result:=False;
 CrwPid:=StrToIntDef(GetEnv('CRW_DAQ_SYS_EXE_PID'),0); if (CrwPid<=0) then Exit;
 CrwSid:=StrToIntDef(GetEnv('CRW_DAQ_SYS_SESSION_NB'),0); if (CrwSid<=0) then Exit;
 if IsNonEmptyStr(GetListOfProcesses(CrwPid,0,AdaptExeFileName(ExeList[0]))) then Exit;
 if not IsSameText(AdaptExeFileName(GetParentProcessExe),AdaptExeFileName('crongrd.exe')) then Exit;
 PidFile:=SessionManager.RunningPidFile;
 PidFile:=ExtractFileDir(PidFile);
 PidFile:=ExtractFileDir(PidFile);
 PidFile:=ExtractFileDir(PidFile);
 PidFile:=AddPathDelim(PidFile)+ExtractBaseName(ExeList[0]);
 PidFile:=AddPathDelim(PidFile)+'session_'+IntToStr(CrwSid);
 PidFile:=AddPathDelim(PidFile)+'running.pid';
 if not FileIsReadable(PidFile) then Exit;
 RunPid:=StrToIntDef(Trim(StringFromFile(PidFile,0)),0); if (RunPid<=0) then Exit;
 if IsNonEmptyStr(GetListOfProcesses(RunPid,0,AdaptExeFileName(ExeList[0]))) then Exit;
 Result:=FileErase(PidFile); if Result then PrintReport('File Erased: '+PidFile); 
end;

function SetCapabilities(AppName,DefCap:LongString):Boolean;
var ini,gc,wc,sc,cmd,ans:LongString;
begin
 Result:=False;
 ini:=''; gc:=''; wc:=''; sc:=''; cmd:=''; ans:='';
 if IsUnix then begin
  ini:=ForceExtension(AppName,'ini');
  if ReadIniFileAlpha(ini,SectSystem(1),'CrwDaqSetCapScript%a',cmd,efConfigNC) then begin
   if FileIsExecutable(cmd) and RunCommand(cmd,ans) then begin
    PrintReport('Exec: '+cmd+EOL+ans);
    Exit(True);
   end;
  end;
  if RunCommand(UnixSudoit('getcap '+AppName),gc) then begin
   gc:=ExtractWord(2,gc,JustSpaces);
   PrintReport('getcap: '+gc);
  end else begin
   ErrorReport('Could not getcap '+AppName);
   Exit;
  end;
  if ReadIniFileString(ini,SectSystem(1),'RequiredCapabilities%s',wc,efConfigNC) then begin
   wc:=LoCaseStr(TrimDef(wc,DefCap));
   PrintReport('inicap: '+wc);
  end else begin
   wc:=LoCaseStr(Trim(DefCap));
   PrintReport('defcap: '+wc);
  end;
  if RunCommand(UnixSudoit('setcap -v '+wc+' '+AppName),sc) then begin
   PrintReport('curcap: '+gc);
   Exit(True);
  end;
  if RunCommand(UnixSudoit('setcap '+wc+' '+AppName),sc) then begin
   if RunCommand(UnixSudoit('getcap '+AppName),sc)
   then sc:=ExtractWord(2,sc,JustSpaces) else sc:='';
   PrintReport('setcap: '+sc);
  end else begin
   ErrorReport('Could not setcap '+AppName);
   Exit;
  end;
 end;
end;

 //
 // Get EXE file version info.
 //
function GetVersionInfo(const Name:AnsiString):AnsiString;
var List:TStringList;
begin
 Result:='';
 if (Name<>'') then
 try
  List:=TStringList.Create;
  try
   List.Text:=GetFileVersionInfoAsText(ProgName);
   Result:=List.Values[Name];
  finally
   List.Free;
  end;
 except
  on E:Exception do Result:='';
 end;
end;
function DotEnding(const S:AnsiString):AnsiString;
const dot='.';
begin
 Result:=S;
 if (Result<>'') then if (Result[Length(Result)]<>dot) then Result:=Result+dot;
end;
procedure PrintVersionInfo(const Fallback:AnsiString);
begin
 if (GetVersionInfo('ProductName')<>'') then begin
  writeln(DotEnding(GetVersionInfo('ProductName')+' version '+GetVersionInfo('ProductVersion')));
  writeln(DotEnding(GetVersionInfo('FileDescription')));
  writeln(DotEnding(GetVersionInfo('LegalCopyright')));
 end else begin
  writeln(Fallback);
 end;
end;

procedure Launch(const AppName,CmdLine:LongString);
{$IFDEF WINDOWS} var ProcessInfo:TProcessInformation; StartupInfo:TStartupInfo; Starter:LongString; {$ENDIF}
{$IFDEF UNIX} var tid:Integer; {$ENDIF}
var AppDir,Msg:LongString;
begin
 PrintReport('Launch: '+CmdLine);
 AppDir:=ExtractFileDir(AppName);
 {$IFDEF WINDOWS}
 Starter:=''; 
 StartupInfo:=Default(TStartupInfo);
 ProcessInfo:=Default(TProcessInformation); 
 if IsNonEmptyStr(CmdLine) then begin
  ClearParentCrwdaqPidFileIfNeed;
  if IsElevated or not IsUACEnabled then begin
   ZeroMemory(@StartupInfo,SizeOf(StartupInfo));
   StartupInfo.cb:=SizeOf(StartupInfo);
   if CreateProcess(
    nil,                      // pointer to name of executable module
    ArgChar(CmdLine),         // pointer to command line string
    nil,                      // pointer to process security attributes
    nil,                      // pointer to thread security attributes
    False,                    // handle inheritance flag
    0,                        // creation flags
    nil,                      // pointer to new environment block
    ArgChar(AppDir),          // pointer to current directory name
    StartupInfo,              // pointer to STARTUPINFO
    ProcessInfo)              // pointer to PROCESS_INFORMATION
   then begin
    Msg:='Started PID '+IntToStr(ProcessInfo.dwProcessId);
    PrintReport(Msg,sev_Succeed);
   end else begin
    ErrorReport('Could not run '+QArg(AppName));
    Halt(4);
   end;
   CloseHandle(ProcessInfo.hProcess);
   CloseHandle(ProcessInfo.hThread);
  end else begin
   Starter:=FindFile('grun.exe',AddPathDelim(AppDir)+'resource\shell');
   if (Length(Starter)>0) and FileExists(Starter) then begin
    Starter:=Format('%s -%d',[QArg(Starter),nShow]);
    nShow:=SW_SHOWMINNOACTIVE;
   end else begin
    Starter:=Format('start "start %s"',[ExtractFileName(AppName)]);
   end;
   if RunElevatedCmd(Format('%s %s',[Starter,CmdLine]),AppDir,nShow) then begin
    Msg:='Started "run as" elevated process '+ExtractFileNameExt(AppName);
    PrintReport(Msg,sev_Succeed);
   end else begin
    ErrorReport('Could not run '+QArg(AppName));
    Halt(4);
   end;
  end;
 end;
 {$ENDIF ~WINDOWS}
 {$IFDEF UNIX}
 if IsNonEmptyStr(CmdLine) then begin
  ClearParentCrwdaqPidFileIfNeed;
  tid:=task_init(CmdLine);
  if (tid<>0) then
  try
   task_ctrl(tid,'HomeDir='+AppDir);
   task_ctrl(tid,'Display='+IntToStr(nShow));
   if task_run(tid) then begin
    Msg:='Started PID '+task_ctrl(tid,'Pid');
    PrintReport(Msg,sev_Succeed);
   end else begin
    ErrorReport('Could not run '+QArg(AppName));
    Halt(4);
   end;
  finally
   task_free(tid);
  end;
 end;
 {$ENDIF ~UNIX}
end;

begin

 try
  UseRunCommandEx(True);
  if (ParamCount=1) then begin
   if IsOption(ParamStr(1),'--version') then begin
    PrintVersionInfo('crwgo - crwdaq Launcher utility.');
   end;
   if IsOption(ParamStr(1),'-h','--help') then begin
    PrintVersionInfo('crwgo - crwdaq Launcher utility.');
    writeln('All rights reserved. This software is FREE for non-commercial use.');
    writeln('******************************************************************');
    writeln('* THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND *');
    writeln('*  EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE *');
    writeln('*   IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A   *');
    writeln('*      PARTICULAR PURPOSE. YOU MAY USE IT FOR OWN RISK.          *');
    writeln('******************************************************************');
    writeln('crwgo: Run (launch) crwdaq executable with an arguments specified.');
    writeln('Usage: crwgo args   (where args is crwdaq command line arguments).');
    writeln('Example: crwgo /daq/demo/test.cfg                                 ');
    Exit;
   end;
  end;
  // Set Log file Name
  if SessionManager.Start('? 100') then begin
   SysLog.LogFile:=SessionManager.VarTmpFile(ForceExtension(ApplicationName,'.log'));
   SysLog.TriggerLevel:=sev_Debug; SysLog.Polling.Enabled:=True;
   SysLog.Info(sev_Notify,'Enter '+ApplicationName);
   Writeln('Open Log file: '+SysLog.LogFile);
  end; 
  if not DirectoryExists(HomeDir) then begin
   ErrorReport('Could not find directory '+QArg(HomeDir));
   Halt(1);
  end;
  AppName:=GetAppName;
  if not FileExists(AppName) then begin
   ErrorReport('Could not find EXE file ('+AppName+').');
   Halt(2);
  end;
  if not IsGoodFileSign(AppName) then begin
   ErrorReport('Invalid file sign '+QArg(AppName));
   Halt(3);
  end;
  if IsWindows and (detect_wine_number>0) then begin
   ErrorReport('can not run in WINE '+QArg(AppName));
   Halt(4);
  end;
  if IsUnix then SetCapabilities(AppName,DefCap);
  CmdLine:=QArg(AppName)+' '+ValidateArgs(GetCommandLineArguments);
  Launch(AppName,CmdLine);
  SysLog.Info(sev_Notify,'Leave '+ApplicationName);
 except
  on H:EHalt do begin Writeln(H.Message); SysLog.Error(sev_Fail,H.Message); end;
  on E:Exception do BugReport(E,nil,'crwgo');
 end;
 if SysLog.Polling.Enabled then SysLog.Polling.SyncLoop(1000,1);

end.

//////////////
// END OF FILE
//////////////
