/////////////////////////////////////////////////////////////////////
// SafeRun Copyright(c) 2018 Alexey Kuryakin kouriakine@mail.ru
//
// Purposes:
// SafeRun - free Win32 utility to run program in Safer mode.
// Safer mode means restricted process privileges for safety.
// Can be used for safe Web surfing or high security systems.
// 
// Requirements:
// Required client Windows XP or Server 2003 at least to run.
// Uses:  kernel32.dll, user32.dll, advapi32.dll.
//
// License for use and distribution: GNU LGPL (see below).
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// See the GNU Lesser General Public License for more details.
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330,
// Boston, MA  02111-1307  USA
/////////////////////////////////////////////////////////////////////
program saferun;

{$APPTYPE GUI}

{$ALIGN             OFF}
{$BOOLEVAL          OFF}
{$ASSERTIONS        ON}
{$DEBUGINFO         ON}
{$DEFINITIONINFO    ON}
{$EXTENDEDSYNTAX    ON}
{$LONGSTRINGS       ON}
{$HINTS             ON}
{$IOCHECKS          OFF}
{$WRITEABLECONST    ON}
{$LOCALSYMBOLS      ON}
{$MINENUMSIZE       1}
{$OPENSTRINGS       ON}
{$OPTIMIZATION      ON}
{$OVERFLOWCHECKS    OFF}
{$RANGECHECKS       OFF}
{$REALCOMPATIBILITY OFF}
{$STACKFRAMES       OFF}
{$TYPEDADDRESS      OFF}
{$TYPEINFO          ON}
{$VARSTRINGCHECKS   OFF}
{$WARNINGS          ON}

{$RESOURCE saferun.res}

uses windows;

const CRLF = #13#10;
const About =
     'Help on SafeRun.exe:'+CRLF
    +''+CRLF
    +'Copyright(c) 2018 Alexey Kuryakin RU kouriakine@mail.ru.'+CRLF
    +'SafeRun - Win32 utility to run program in Safer mode,v1.5.'+CRLF
    +'Safer mode means restricted process privileges for safety.'+CRLF
    +'Required client Windows XP or Server 2003 at least to run.'+CRLF
    +'Use this program free under the terms of LGPL license.'+CRLF
    +''+CRLF
    +'Usage: saferun [-opt] cmdline'+CRLF
    +''+CRLF
    +'Options (started from - char):'+CRLF
    +' -v  - verbose mode: MessageBox on errors (default is silent)'+CRLF
    +'       it''s better to place this option as 1st option in list'+CRLF
    +' -w  - wait while cmdline running, default is no waits'+CRLF
    +'       in wait mode SafeRun return cmdline''s exit code'+CRLF
    +' -h  - run cmdline in hidden window (SW_HIDE), default is visible (SW_SHOW)'+CRLF
    +' -0..9 run with display mode assigned to 0=SW_HIDE(like -h), 1=SW_SHOWNORMAL,'+CRLF
    +'       2=SW_SHOWMINIMIZED, 3=SW_SHOWMAXIMIZED, 4=SW_SHOWNOACTIVATE,'+CRLF
    +'       5=SW_SHOW(default), 6=SW_MINIMIZE, 7=SW_SHOWMINNOACTIVE,'+CRLF
    +'       8=SW_SHOWNA, 9=SW_RESTORE'+CRLF
    +' -i  - run with IDLE_PRIORITY_CLASS         ( 4  lowest priority)'+CRLF
    +' -b  - run with BELOW_NORMAL_PRIORITY_CLASS ( 6  lower normal)'+CRLF
    +' -n  - run with NORMAL_PRIORITY_CLASS       ( 8  default)'+CRLF
    +' -a  - run with ABOVE_NORMAL_PRIORITY_CLASS ( 10 higher normal)'+CRLF
    +' -g  - run with HIGH_PRIORITY_CLASS         ( 13 greater normal)'+CRLF
    +' -r  - run with REALTIME_PRIORITY_CLASS     ( 24 highest)'+CRLF
    +' -t  - run cmdline in trusted mode, i.e. don''t use Safer'+CRLF
    +' -o  - run cmdline in Safer mode as normal user, by default'+CRLF
    +' -c  - run cmdline in Safer mode as constrained user'+CRLF
    +' -u  - run cmdline in Safer mode as untrusted user'+CRLF
    +' -f  - run cmdline in Safer mode as fully trusted user'+CRLF
    +' -?  - run this help screen'+CRLF
    +''+CRLF
    +'Exit code returned by SafeRun:'+CRLF
    +' 0   - OK, cmdline was successfully started'+CRLF
    +' 1   - invalid options specified (unsupported opt)'+CRLF
    +' 2   - no arguments specified (empty command line)'+CRLF
    +' 3   - could not create process (invalid cmdline or access)'+CRLF
    +' 4   - exception happened during execution'+CRLF
    +' n   - in wait mode (-w) SafeRun return cmdline''s exit code'+CRLF
    +''+CRLF
    +'Examples:'+CRLF
    +' saferun firefox.exe - run Web browser in Safer mode with normal user level'+CRLF
    +' saferun -c firefox  - run Web browser in Safer mode with constrained user level'+CRLF
    +' saferun -u firefox  - run Web browser in Safer mode with untrusted user level'+CRLF
    +' saferun -f firefox  - run Web browser in Safer mode  with fully trusted user level'+CRLF
    +' saferun -t firefox  - run Web browser in normal mode, i.e. don''t use Safer at all'+CRLF
    +' saferun -wv notepad - run notepad in Safer mode with normal user level, wait, verbose'+CRLF
    +' saferun -htw defrag - run defrag in fully trusted (i.e. normal) mode, hidden, wait result'+CRLF
    +' saferun -t7 defrag  - run deftag in trusted (i.e. normal) mode, minimized noactive, no wait'+CRLF
    ;
    
// Header derived from WinSafer.h to access advapi32.dll functions.
// Minimum supported client Windows XP, server Windows Server 2003.
///////////////////////////////////////////////////////////////////
type  SAFER_LEVEL_HANDLE            = THandle;
const SAFER_SCOPEID_MACHINE         = 1;
const SAFER_SCOPEID_USER            = 2;
const SAFER_LEVELID_FULLYTRUSTED    = $40000;
const SAFER_LEVELID_NORMALUSER      = $20000;
const SAFER_LEVELID_CONSTRAINED     = $10000;
const SAFER_LEVELID_UNTRUSTED       = $01000;
const SAFER_LEVELID_DISALLOWED      = $00000;
const SAFER_LEVEL_OPEN              = 1;
function SaferCloseLevel(hLevelHandle:SAFER_LEVEL_HANDLE):BOOL; stdcall; external 'advapi32.dll';
function SaferCreateLevel(dwScopeId:DWORD; dwLevelId:DWORD; OpenFlags:DWORD; var pLevelHandle:SAFER_LEVEL_HANDLE; lpReserved:Pointer):BOOL; stdcall; external 'advapi32.dll';
function SaferComputeTokenFromLevel(LevelHandle:SAFER_LEVEL_HANDLE; InAccessToken:THANDLE; OutAccessToken:PHANDLE; dwFlags:DWORD; lpReserved:Pointer):BOOL; stdcall; external 'advapi32.dll';

// Special chars:
/////////////////
const SpaceChar                      = ' ';
const TabChar                        = #9;
const DoubleQuota                    = '"';

// Utility functions for CmdLine parsing:
/////////////////////////////////////////
function IsSameChar(a,b:Char):Boolean; begin Result:=(a=b); end;
function IsSameCharOrNul(a,b:Char):Boolean; begin Result:=IsSameChar(a,b) or IsSameChar(a,#0); end;
function TabToSpace(c:Char):Char; begin if(c=TabChar) then Result:=SpaceChar else Result:=c; end;
function IsSpaceChar(c:Char):Boolean; begin Result:=IsSameChar(c,SpaceChar); end;
function IsSpaceCharOrTab(c:Char):Boolean; begin Result:=IsSpaceChar(TabToSpace(c)); end;
function IsSpaceCharOrNul(c:Char):Boolean; begin Result:=IsSameCharOrNul(c,SpaceChar); end;
function IsSpaceCharOrTabOrNul(c:Char):Boolean; begin Result:=IsSameCharOrNul(TabToSpace(c),SpaceChar); end;
function IsOptionChar(c:Char):Boolean; begin Result:=IsSameChar(c,'-') or IsSameChar(c,'/'); end;

// Exit codes returned by SafeRun:
//////////////////////////////////
const ecOK                          = 0;
const ecBadOpt                      = 1;
const ecNoArgs                      = 2;
const ecFailStart                   = 3;
const ecFailAuthz                   = 4;
const ecFailToken                   = 5;
const ecException                   = 4;

const BELOW_NORMAL_PRIORITY_CLASS   = $4000;     // Not defined in Windows.pas
const ABOVE_NORMAL_PRIORITY_CLASS   = $8000;     // Not defined in Windows.pas

function IntToStr(i:Integer):AnsiString;
begin
 Str(i,Result);
end;

function Max(a,b:Integer):Integer;
begin
 if(a>b) then Result:=a else Result:=b;
end;

// To be called on fatal errors:
////////////////////////////////
function Fatal(exitcode:Integer; verb:Boolean; Title:PChar; Content:PChar):Integer;
var LastError,len:Integer; msg,tmp:AnsiString;
begin
 if ( verb ) then
 case ( exitcode ) of
  ecOK        : MessageBox(0,Content,Title,MB_OK+MB_ICONINFORMATION);
  ecBadOpt    : MessageBox(0,Content,Title,MB_OK+MB_ICONWARNING);
  ecNoArgs    : MessageBox(0,Content,Title,MB_OK+MB_ICONWARNING);
  ecFailStart : begin
   LastError := GetLastError(); SetLength(tmp,16*1024);
   len := FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM or FORMAT_MESSAGE_ARGUMENT_ARRAY, nil, LastError, 0, PChar(tmp), Length(tmp), nil);
   while ( (len > 0) and (tmp[len-1] <= SpaceChar)) do Dec(len); SetLength(tmp,Max(0,len-1));
   if (Length(tmp)=0) and (LastError=193) then tmp:='That is not a valid Win32 application.'; if (Length(tmp)=0) then tmp:='???';
   msg:='Win32 error '+IntToStr(LastError)+' occured:'+CRLF+CRLF+'   "'+tmp+'"'+CRLF+CRLF+'in command line:'+CRLF+CRLF+'   '+GetCommandLine()+CRLF+CRLF+Content;
   MessageBox(0,PChar(msg),Title,MB_OK+MB_ICONERROR);
  end;
 end;
 Result:=exitcode;
end;

procedure Main;
var si:STARTUPINFO; pi:PROCESS_INFORMATION;
 WaitTimeOut,PriorityClass,DisplayMode:DWORD; CreateProcessResult,VerboseMode:BOOL; lpszCmdLine:PChar; stopchar:char;
 hToken:THANDLE; UsesSafer:BOOL; hAuthzLevel:SAFER_LEVEL_HANDLE; hSaferLevel:DWORD;
 procedure Initialize;
 begin
  ZeroMemory(@si,sizeof(si));
  ZeroMemory(@pi,sizeof(pi));
  WaitTimeOut         := 0;
  VerboseMode         := FALSE;
  stopchar            := SpaceChar;
  DisplayMode         := SW_SHOW;
  CreateProcessResult := FALSE;
  lpszCmdLine         := nil;
  PriorityClass       := 0;
  hToken              := 0;
  UsesSafer           := TRUE;    
  hAuthzLevel         := 0;
  hSaferLevel         := SAFER_LEVELID_NORMALUSER;
 end;
 procedure Cleanup;
 begin
  if ( hAuthzLevel <> 0 ) then SaferCloseLevel(hAuthzLevel);
  hAuthzLevel := 0;
 end;
begin
 try
  Initialize;
  try
    // Initialize.
    exitcode := ecOK;
   
    lpszCmdLine := GetCommandLine();

    // Skip leading space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);

    // Skip executable file name, maybe double quoted.
    if ( IsSameChar(lpszCmdLine[0],DoubleQuota) ) then begin stopchar := DoubleQuota; Inc(lpszCmdLine); end;
    while ( not IsSameCharOrNul(TabToSpace(lpszCmdLine[0]),stopchar) ) do Inc(lpszCmdLine);
    if ( IsSameChar(lpszCmdLine[0],stopchar) ) then Inc(lpszCmdLine);

    // Skip leading space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);

    // Check options /opt or -opt.
    while ( IsOptionChar(lpszCmdLine[0]) ) do begin
        Inc(lpszCmdLine);
        while ( not IsSpaceCharOrTabOrNul(lpszCmdLine[0]) ) do begin
            case ( lpszCmdLine[0] ) of
                'W' ,  // Set waiting timeout.
                'w' :  WaitTimeOut := INFINITE;
                'H' ,  // Set hidden mode.
                'h' :  DisplayMode := SW_HIDE;
                '0' :  DisplayMode := SW_HIDE;
                '1' :  DisplayMode := SW_SHOWNORMAL;
                '2' :  DisplayMode := SW_SHOWMINIMIZED;
                '3' :  DisplayMode := SW_SHOWMAXIMIZED;
                '4' :  DisplayMode := SW_SHOWNOACTIVATE;
                '5' :  DisplayMode := SW_SHOW;
                '6' :  DisplayMode := SW_MINIMIZE;
                '7' :  DisplayMode := SW_SHOWMINNOACTIVE;
                '8' :  DisplayMode := SW_SHOWNA;
                '9' :  DisplayMode := SW_RESTORE;
                'V' ,  // Set verbose mode.
                'v' :  VerboseMode := TRUE;
                'I' ,  // Set IDLE_PRIORITY_CLASS
                'i' :  PriorityClass := IDLE_PRIORITY_CLASS;
                'B' ,  // Set BELOW_NORMAL_PRIORITY_CLASS
                'b' :  PriorityClass := BELOW_NORMAL_PRIORITY_CLASS;
                'N' ,  // Set NORMAL_PRIORITY_CLASS
                'n' :  PriorityClass := NORMAL_PRIORITY_CLASS;
                'A' ,  // Set ABOVE_NORMAL_PRIORITY_CLASS
                'a' :  PriorityClass := ABOVE_NORMAL_PRIORITY_CLASS;
                'G' ,  // Set HIGH_PRIORITY_CLASS
                'g' :  PriorityClass := HIGH_PRIORITY_CLASS;
                'R' ,  // Set REALTIME_PRIORITY_CLASS
                'r' :  PriorityClass := REALTIME_PRIORITY_CLASS;
                'T' ,  // Set trusted mode,i.e. don't use safer.
                't' :  UsesSafer := FALSE;
                'F' ,  // Set safer level for fully trusted user.
                'f' :  hSaferLevel := SAFER_LEVELID_FULLYTRUSTED;
                'O' ,  // Set safer level for normal user.
                'o' :  hSaferLevel := SAFER_LEVELID_NORMALUSER;
                'C' ,  // Set safer level for constrained user.
                'c' :  hSaferLevel := SAFER_LEVELID_CONSTRAINED; 
                'U' ,  // Set safer level for untrusted user.
                'u' :  hSaferLevel := SAFER_LEVELID_UNTRUSTED;
                '?' :  begin ExitCode:=Fatal(ecOK,TRUE,'SafeRun -> Help.',About); Exit; end;
                else   begin ExitCode:=Fatal(ecBadOpt,VerboseMode,'SafeRun -> Invalid option found.',About); Exit; end;
            end;
            Inc(lpszCmdLine);
        end;
        // Skip space & tab chars.
        while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);
    end;

    // Skip space & tab chars.
    while ( IsSpaceCharOrTab(lpszCmdLine[0]) ) do Inc(lpszCmdLine);

    // No arguments?
    if (lpszCmdLine[0] = #0) then begin ExitCode:=Fatal(ecNoArgs,TRUE,'SafeRun -> No arguments found.',About); Exit; end;
    
    //  Generate the restricted token that we will use.
    if ( UsesSafer ) then begin
        if ( not SaferCreateLevel(SAFER_SCOPEID_USER, hSaferLevel, 0, hAuthzLevel, nil) ) then begin
            ExitCode:=Fatal(ecFailAuthz,VerboseMode,'SafeRun -> SaferCreateLevel failed.','');
            hAuthzLevel:=0;
            Exit;
        end;
        if ( not SaferComputeTokenFromLevel(hAuthzLevel, 0, @hToken, 0, nil) ) then begin
            ExitCode:=Fatal(ecFailToken,VerboseMode,'SafeRun -> SaferComputeTokenFromLevel failed.','');
            Exit;
        end;
    end else hSaferLevel := SAFER_LEVELID_FULLYTRUSTED;

    // Initialize startup info.
    ZeroMemory(@si, sizeof(si));
    si.cb := sizeof(si);
    si.lpDesktop := nil;
    si.dwFlags := STARTF_USESHOWWINDOW;
    si.wShowWindow := DisplayMode;

    // Create process with new console, with\without Safer mode.
    // Uses tail of original command line to create new process.
    if ( UsesSafer )
    then CreateProcessResult := CreateProcessAsUser( hToken, nil, lpszCmdLine, nil, nil, FALSE, CREATE_NEW_CONSOLE or PriorityClass, nil, nil, si, pi )
    else CreateProcessResult := CreateProcess(               nil, lpszCmdLine, nil, nil, FALSE, CREATE_NEW_CONSOLE or PriorityClass, nil, nil, si, pi );

    // Wait while child running (if /w option specified). Take child exit code.
    if( CreateProcessResult ) then begin
        if ( WaitForSingleObject(pi.hProcess, WaitTimeOut) = WAIT_OBJECT_0 )
            then GetExitCodeProcess(pi.hProcess, DWORD(exitcode));
        CloseHandle( pi.hProcess );
        CloseHandle( pi.hThread );
    end else begin
        ExitCode:=Fatal(ecFailStart,VerboseMode,'SafeRun -> CreateProcess failed.','');
        Exit;
    end;
    // Done!
  finally
   Cleanup;
  end;
 except
  ExitCode:=Fatal(ecException,VerboseMode,'SafeRun -> Exception happened.','');
 end;
end;

begin
 Main;
end.
