#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Configuration Generator for WJ300A devices.
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

###################################
### WJ300A devices config generator
###################################

######################
# initialize variables
######################
readonly hdwcfg="${scriptbase}_hdw.cfg"; # hardware   configuration
readonly simcfg="${scriptbase}_sim.cfg"; # simulator  configuration
readonly diccfg="${scriptbase}_dic.cfg"; # DIM client configuration
readonly discfg="${scriptbase}_dis.cfg"; # DIM server configuration
readonly impcfg="${scriptbase}_imp.cfg"; # DIM import configuration
readonly devmsg="devPostMsg";            # Method to send/post messages
readonly MAIN_CTRL="GENDC.MAIN.CTRL";    # Name of main control window
readonly PARAM_PLOT="GENDC.PARAM.PLOT";  # Name of Param plot window
readonly STATE_PLOT="GENDC.STATE.PLOT";  # Name of State plot window
declare -i numhdwcfg=0;                  # Flag to include hardware config
declare -i TABCRC_TOP=20;
declare -i TABCRC_STEP=115;
declare fileprefix="";
declare mainprefix="";
declare TcpHostname="";                  # Pipe TCP connection ip
declare TcpPort="";                      # Pipe TCP connection port
declare PollPeriod="";                   # Poll Period
declare Simulator="";                    # Simulator mode
readonly tabcrc="../circuits/${scriptbase}_link.crc";
declare FACN="";                         # Facility name
declare IDEN="";                         # Device identifier name
declare FACP="";                         # Facility DIM  path
declare FACF="";                         # Facility file name
declare FACL="";                         # Facility name (lower case)
declare IDNL="";                         # Device identifier name (lower case)
declare IDNF="";                         # Device identifier name (lower case)

function call(){
 echo "$*" 1>&2;
 eval "$@";
};

function rem(){
 return 0;
};

function wj300a_devices(){
 call DeleteFiles $hdwcfg $simcfg $tabcrc $diccfg $discfg;
 rem =======================TCP CONNECTION====================================;
 rem  Device  Facility                    Identitier TcpHostname     TcpPort PollPeriod Simulator;
 call WJ300A  PRGK100.PRPI1.OSCS.LECROY   L1         172.21.4.125    1861    10         0;
 call WJ300A  PRGK100.PRPI1.OSCS.LECROY   L2         172.21.4.126    1861    10         1;
 call WJ300A  PRGK100.PRPI1.OSCS.LECROY   L3         172.21.4.127    1861    10         2;
 call WJ300A  PRGK100.PRPI1.OSCS.LECROY   L4         172.21.4.128    1861    10         3;
#call WJ300A  WJ300A                      D2         172.21.4.124    1861    10         1;
#call WJ300A  WJ300A                      D3         172.21.4.123    1861    10         1;
#call WJ300A  WJ300A                      D4         172.21.4.122    1861    10         1;
 rem =========================================================================;
 cat $hdwcfg $simcfg $diccfg $discfg;
};

function WJ300A(){
 call WJ300A_SET $*;
 call WJ300A_HDW $* >> $hdwcfg;
 call WJ300A_SIM $* >> $simcfg;
 call WJ300A_DIC $* >> $diccfg;
 call WJ300A_DIS $* >> $discfg;
 call WJ300A_TABCRC >> $tabcrc;
};

function WJ300A_SET(){
 FACN="$1"        ; shift; # FACN - Facility name
 IDEN="$1"        ; shift; # IDEN - Identifier
 TcpHostname="$1" ; shift; # Pipe TCP connection ip
 TcpPort="$1"     ; shift; # Pipe TCP connection port
 PollPeriod="$1"  ; shift; # Poll Period
 Simulator="$1"   ; shift; # Simulator mode
 FACP="$(echo "$FACN" | tr '.' '/')"; ## Facility DIM  path, i.e. FACN name with . to / replacement
 FACF="$(echo "$FACN" | tr '.' '_')"; ## Facility file name, i.e. FACN name with . to _ replacement
 FACL="$(echo "$FACN" | tr '[:upper:]' '[:lower:]')"; ## Facility name to lower case
 FACF="$(echo "$FACF" | tr '[:upper:]' '[:lower:]')"; ## Facility file name to lower case
 IDNF="$(echo "$IDEN" | tr '[:upper:]' '[:lower:]')";
 IDNL="$(echo "$IDEN" | tr '[:upper:]' '[:lower:]')";
};

function WJ300A_HDW(){
 echo ";****************************";
 echo "; WJ300A Driver for $FACN.$IDEN";
 echo ";****************************";
 echo "[DeviceList]";
 echo "&$FACN.$IDEN.DRV = device software program";
 echo "[&$FACN.$IDEN.DRV]";
 echo "Comment         = Hardware driver for LeCroy WJ300A Series Oscilloscope";
 echo "InquiryPeriod   = 10";
 echo "DevicePolling   = 10, tpHigher";
 echo "ProgramSource   = ../daqpas/${fileprefix}wj300a_drv.pas";
 echo "DigitalFifo     = 1024";
 echo "AnalogFifo      = 1024";
 echo "DebugFlags      = 3";
 echo "OpenConsole     = 3";
 echo "Simulator       = $Simulator";
 echo "PipeTimeOut     = 1000";
 echo "PipeTimeGap     = 10";
 echo "PollPeriod      = $PollPeriod";
 echo "DefaultDirBase   = ../../${FACL}_data/experiment/common/";
 echo "DefaultDirExp    = misc/${FACL}_$IDNL/";
 echo "filePrefix       = ${FACL}_$IDNL";
 echo "tagPrefix       = $FACN.$IDEN";
 echo "TcpHostname     = $TcpHostname";
 echo "TcpPort         = $TcpPort";
 echo "HelpFile        = ../help/wj300a.md";
 echo "EncryptMethod  = RC6";
 echo "TrustedUsers   = [TrustedUsers]";
 echo "CalcDevName      = &$FACN.$IDEN.PROCESSING";
 echo "AnalogOutputs  = 16";
 echo "Link AnalogOutput 0  with curve $FACN.$IDEN.POLL_RATE history 1000";
 echo "Link AnalogOutput 1  with curve $FACN.$IDEN.ERROR_CNT history 1000";
 echo "Link AnalogOutput 2  with curve $FACN.$IDEN.ESR       history 1000";
 echo "Link AnalogOutput 3  with curve $FACN.$IDEN.OFST_C1   history 1000";
 echo "Link AnalogOutput 4  with curve $FACN.$IDEN.OFST_C2   history 1000";
 echo "Link AnalogOutput 5  with curve $FACN.$IDEN.OFST_C3   history 1000";
 echo "Link AnalogOutput 6  with curve $FACN.$IDEN.OFST_C4   history 1000";
 echo "Link AnalogOutput 7  with curve $FACN.$IDEN.OFST_M1   history 1000";
 echo "Link AnalogOutput 8  with curve $FACN.$IDEN.TDIV      history 1000";
 echo "Link AnalogOutput 9  with curve $FACN.$IDEN.TLVL      history 1000";
 echo "Link AnalogOutput 10 with curve $FACN.$IDEN.TRDL      history 1000";
 echo "Link AnalogOutput 11 with curve $FACN.$IDEN.VDIV_C1   history 1000";
 echo "Link AnalogOutput 12 with curve $FACN.$IDEN.VDIV_C2   history 1000";
 echo "Link AnalogOutput 13 with curve $FACN.$IDEN.VDIV_C3   history 1000";
 echo "Link AnalogOutput 14 with curve $FACN.$IDEN.VDIV_C4   history 1000";
 echo "Link AnalogOutput 15 with curve $FACN.$IDEN.VDIV_M1   history 1000";
 echo ";------ Save custom parameters to INI file";
 echo "CustomIniTagList  = [$FACN.$IDEN.CustomParameters.TagList]";
 echo "CustomIniFileRef  = ../data/${FACF}_${IDNF}_custom.ini";
 echo "CustomIniSection  = [$FACN.$IDEN.CustomParameters]";
 echo "CustomIniBackups  = ../../${mainprefix}_data/$FACF/custom";
 echo "CustomIniAutoLoad = 1";
 echo "CustomIniAutoSave = 0";
 echo "[]";
 echo "";
 echo "[DeviceList]";
 echo "&$FACN.$IDEN.PROCESSING = device software program";
 echo "[&$FACN.$IDEN.PROCESSING]";
 echo "ProgramSource    = ../daqpas/wj300a_processing";
 echo "MainProg         = &$FACN.$IDEN.CTRL";
 echo "DebugFlags       = 3";
 echo "OpenConsole      = 1";
 echo "WatchDogDeadline = 150000";
 echo "tagPrefix        = $FACN.$IDEN";
 echo "AnalogOutputs    = 5";
 echo "Link AnalogOutput 0  with curve $FACN.$IDEN.WAVFORM_C1 history 1000";
 echo "Link AnalogOutput 1  with curve $FACN.$IDEN.WAVFORM_C2 history 1000";
 echo "Link AnalogOutput 2  with curve $FACN.$IDEN.WAVFORM_C3 history 1000";
 echo "Link AnalogOutput 3  with curve $FACN.$IDEN.WAVFORM_C4 history 1000";
 echo "Link AnalogOutput 4  with curve $FACN.$IDEN.WAVFORM_M1 history 1000";
 echo "[]";
 echo "";
 echo "[ConfigFileList]";
 echo "ConfigFile = ../data/${FACF}_${IDNF}_custom.ini";
 echo "[]";
 echo "";
 echo "[TagList]";
 echo "$FACN.$IDEN.DIMGUICLICK = String ?  ; The DIM click sensor data";
 echo "$FACN.$IDEN.POLL_ENABLE = Integer 1 ; Enable polling flag";
 echo "$FACN.$IDEN.POLL_RATE   = Real    0 ; Polling rate, poll/sec";
 echo "$FACN.$IDEN.ERROR_CNT   = Real    0 ; Device error counter";
 echo "$FACN.$IDEN.ACQ         = String ?";
 echo "$FACN.$IDEN.CPL_C1      = String ?";
 echo "$FACN.$IDEN.CPL_C2      = String ?";
 echo "$FACN.$IDEN.CPL_C3      = String ?";
 echo "$FACN.$IDEN.CPL_C4      = String ?";
 echo "$FACN.$IDEN.DATE        = String ?";
 echo "$FACN.$IDEN.DTBORD      = String ?";
 echo "$FACN.$IDEN.DTFORM      = String ?";
 echo "$FACN.$IDEN.DTSTART     = Integer 0";
 echo "$FACN.$IDEN.DTPOINTS    = Integer 0";
 echo "$FACN.$IDEN.DTWAVE      = Integer 0";
 echo "$FACN.$IDEN.ESR         = Integer 0";
 echo "$FACN.$IDEN.IDN         = String ?";
 echo "$FACN.$IDEN.MLEN        = String ?";
 echo "$FACN.$IDEN.OFST_C1     = Real 0";
 echo "$FACN.$IDEN.OFST_C2     = Real 0";
 echo "$FACN.$IDEN.OFST_C3     = Real 0";
 echo "$FACN.$IDEN.OFST_C4     = Real 0";
 echo "$FACN.$IDEN.OFST_M1     = Real 0";
 echo "$FACN.$IDEN.TCPL        = String ?";
 echo "$FACN.$IDEN.TDIV        = Real 0";
 echo "$FACN.$IDEN.TESR        = Integer 0";
 echo "$FACN.$IDEN.TLVL        = Real 0";
 echo "$FACN.$IDEN.TRA_C1      = Integer 0";
 echo "$FACN.$IDEN.TRA_C2      = Integer 0";
 echo "$FACN.$IDEN.TRA_C3      = Integer 0";
 echo "$FACN.$IDEN.TRA_C4      = Integer 0";
 echo "$FACN.$IDEN.TRA_M1      = Integer 0";
 echo "$FACN.$IDEN.TRDL        = Real 0";
 echo "$FACN.$IDEN.TRMD        = Integer 0";
 echo "$FACN.$IDEN.TSLP        = String ?";
 echo "$FACN.$IDEN.TSRC        = String ?";
 echo "$FACN.$IDEN.TTYP        = String ?";
 echo "$FACN.$IDEN.VDIV_C1     = Real 0";
 echo "$FACN.$IDEN.VDIV_C2     = Real 0";
 echo "$FACN.$IDEN.VDIV_C3     = Real 0";
 echo "$FACN.$IDEN.VDIV_C4     = Real 0";
 echo "$FACN.$IDEN.VDIV_M1     = Real 0";
 echo "$FACN.$IDEN.WAVFORM_C1  = String ?";
 echo "$FACN.$IDEN.WAVFORM_C2  = String ?";
 echo "$FACN.$IDEN.WAVFORM_C3  = String ?";
 echo "$FACN.$IDEN.WAVFORM_C4  = String ?";
 echo "$FACN.$IDEN.WAVFORM_M1  = String ?";
 echo "$FACN.$IDEN.WAVESRC     = String ?";
 echo "$FACN.$IDEN.SAVESTAT    = Integer 0";
 echo "$FACN.$IDEN.DIRBASE     = String  0 ; Base directory";
 echo "$FACN.$IDEN.DIREXP      = String  0 ; Experiment directory";
 echo "$FACN.$IDEN.DEFAULTDIR  = String  0 ; Default directory";
 echo "$FACN.$IDEN.EXTRACTNUM  = String  0 ; Extraction number";
 echo "[]";
 echo "";
 echo "[DataStorage]";
 echo "$FACN.$IDEN.POLL_RATE  = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.ERROR_CNT  = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.ESR        = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.OFST_C1    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.OFST_C2    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.OFST_C3    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.OFST_C4    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.OFST_M1    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.TDIV       = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.TLVL       = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.TRDL       = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.VDIV_C1    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.VDIV_C2    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.VDIV_C3    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.VDIV_C4    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.VDIV_M1    = Curve 0 10 Black 15 1";
 echo "$FACN.$IDEN.WAVFORM_C1 = Curve 0 10 Yellow 15 1";
 echo "$FACN.$IDEN.WAVFORM_C2 = Curve 0 10 Magenta 15 1";
 echo "$FACN.$IDEN.WAVFORM_C3 = Curve 0 10 Cyan 15 1";
 echo "$FACN.$IDEN.WAVFORM_C4 = Curve 0 10 Green 15 1";
 echo "$FACN.$IDEN.WAVFORM_M1 = Curve 0 10 Red 15 1";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.DRV = Circuit_Window";
 echo "[$FACN.$IDEN.DRV]";
 echo "Circuit = ../circuits/${fileprefix}wj300a_ctrl.crc";
 echo "Link sensor POLL_ENABLE    with tag $FACN.$IDEN.POLL_ENABLE device &$FACN.$IDEN.DRV";
 echo "Link sensor TOOLS          with                             device &$FACN.$IDEN.DRV";
 echo "Link sensor ERROR_CNT      with tag $FACN.$IDEN.ERROR_CNT   device &$FACN.$IDEN.DRV";
 echo "Link sensor POLL_RATE      with tag $FACN.$IDEN.POLL_RATE   device &$FACN.$IDEN.DRV";
 echo "Link sensor DATE           with tag $FACN.$IDEN.DATE        device &$FACN.$IDEN.DRV";
 echo "Link sensor ACQ            with tag $FACN.$IDEN.ACQ         device &$FACN.$IDEN.DRV";
 echo "Link sensor C1:CPL         with tag $FACN.$IDEN.CPL_C1      device &$FACN.$IDEN.DRV";
 echo "Link sensor C2:CPL         with tag $FACN.$IDEN.CPL_C2      device &$FACN.$IDEN.DRV";
 echo "Link sensor C3:CPL         with tag $FACN.$IDEN.CPL_C3      device &$FACN.$IDEN.DRV";
 echo "Link sensor C4:CPL         with tag $FACN.$IDEN.CPL_C4      device &$FACN.$IDEN.DRV";
 echo "Link sensor CLS            with                             device &$FACN.$IDEN.DRV";
 echo "Link sensor DATE           with tag $FACN.$IDEN.DATE        device &$FACN.$IDEN.DRV";
 echo "Link sensor DTSTART        with tag $FACN.$IDEN.DTSTART     device &$FACN.$IDEN.DRV";
 echo "Link sensor DTPOINTS       with tag $FACN.$IDEN.DTPOINTS    device &$FACN.$IDEN.DRV";
 echo "Link sensor WAV.DATA       with tag $FACN.$IDEN.DTWAVE      device &$FACN.$IDEN.DRV";
 echo "Link sensor ESR.0          with curve $FACN.$IDEN.ESR       device &$FACN.$IDEN.DRV";
 echo "Link sensor ESR.2          with curve $FACN.$IDEN.ESR       device &$FACN.$IDEN.DRV";
 echo "Link sensor ESR.3          with curve $FACN.$IDEN.ESR       device &$FACN.$IDEN.DRV";
 echo "Link sensor ESR.4          with curve $FACN.$IDEN.ESR       device &$FACN.$IDEN.DRV";
 echo "Link sensor ESR.5          with curve $FACN.$IDEN.ESR       device &$FACN.$IDEN.DRV";
 echo "Link sensor ESR.7          with curve $FACN.$IDEN.ESR       device &$FACN.$IDEN.DRV";
 echo "Link sensor ID_IDN         with tag $FACN.$IDEN.IDN         device &$FACN.$IDEN.DRV";
 echo "Link sensor MLEN           with tag $FACN.$IDEN.MLEN        device &$FACN.$IDEN.DRV";
 echo "Link sensor C1:OFST        with tag $FACN.$IDEN.OFST_C1     device &$FACN.$IDEN.DRV";
 echo "Link sensor C2:OFST        with tag $FACN.$IDEN.OFST_C2     device &$FACN.$IDEN.DRV";
 echo "Link sensor C3:OFST        with tag $FACN.$IDEN.OFST_C3     device &$FACN.$IDEN.DRV";
 echo "Link sensor C4:OFST        with tag $FACN.$IDEN.OFST_C4     device &$FACN.$IDEN.DRV";
 echo "Link sensor M1:OFST        with tag $FACN.$IDEN.OFST_M1     device &$FACN.$IDEN.DRV";
 echo "Link sensor RUN            with tag $FACN.$IDEN.TRMD        device &$FACN.$IDEN.DRV";
 echo "Link sensor STOP           with tag $FACN.$IDEN.TRMD        device &$FACN.$IDEN.DRV";
 echo "Link sensor TRG            with tag $FACN.$IDEN.TRMD        device &$FACN.$IDEN.DRV";
 echo "Link sensor TRMD           with tag $FACN.$IDEN.TRMD        device &$FACN.$IDEN.DRV";
 echo "Link sensor TCPL           with tag $FACN.$IDEN.TCPL        device &$FACN.$IDEN.DRV";
 echo "Link sensor TDIV           with tag $FACN.$IDEN.TDIV        device &$FACN.$IDEN.DRV";
 echo "Link sensor TESR           with tag $FACN.$IDEN.TESR        device &$FACN.$IDEN.DRV";
 echo "Link sensor TLVL           with tag $FACN.$IDEN.TLVL        device &$FACN.$IDEN.DRV";
 echo "Link sensor C1:TRA         with tag $FACN.$IDEN.TRA_C1      device &$FACN.$IDEN.DRV";
 echo "Link sensor C2:TRA         with tag $FACN.$IDEN.TRA_C2      device &$FACN.$IDEN.DRV";
 echo "Link sensor C3:TRA         with tag $FACN.$IDEN.TRA_C3      device &$FACN.$IDEN.DRV";
 echo "Link sensor C4:TRA         with tag $FACN.$IDEN.TRA_C4      device &$FACN.$IDEN.DRV";
 echo "Link sensor M1:TRA         with tag $FACN.$IDEN.TRA_M1      device &$FACN.$IDEN.DRV";
 echo "Link sensor TRDL           with tag $FACN.$IDEN.TRDL        device &$FACN.$IDEN.DRV";
 echo "Link sensor TSLP           with tag $FACN.$IDEN.TSLP        device &$FACN.$IDEN.DRV";
 echo "Link sensor TSRC           with tag $FACN.$IDEN.TSRC        device &$FACN.$IDEN.DRV";
 echo "Link sensor TTYP           with tag $FACN.$IDEN.TTYP        device &$FACN.$IDEN.DRV";
 echo "link sensor C1:VDIV        with tag $FACN.$IDEN.VDIV_C1     device &$FACN.$IDEN.DRV";
 echo "link sensor C2:VDIV        with tag $FACN.$IDEN.VDIV_C2     device &$FACN.$IDEN.DRV";
 echo "link sensor C3:VDIV        with tag $FACN.$IDEN.VDIV_C3     device &$FACN.$IDEN.DRV";
 echo "link sensor C4:VDIV        with tag $FACN.$IDEN.VDIV_C4     device &$FACN.$IDEN.DRV";
 echo "link sensor M1:VDIV        with tag $FACN.$IDEN.VDIV_M1     device &$FACN.$IDEN.DRV";
 echo "Link sensor BASE.DIR       with tag $FACN.$IDEN.DIRBASE     device &$FACN.$IDEN.DRV";
 echo "Link sensor EXP.DIR        with tag $FACN.$IDEN.DIREXP      device &$FACN.$IDEN.DRV";
 echo "Link sensor WAVE.SAVE.STATUS.0 with tag $FACN.$IDEN.SAVESTAT  device &$FACN.$IDEN.PROCESSING";
 echo "Link sensor WAVE.SAVE.STATUS.1 with tag $FACN.$IDEN.SAVESTAT  device &$FACN.$IDEN.PROCESSING";
 echo "Link sensor WAVE.SAVE.STATUS.2 with tag $FACN.$IDEN.SAVESTAT  device &$FACN.$IDEN.PROCESSING";
 echo "Link sensor WAVE.SAVE.STATUS.3 with tag $FACN.$IDEN.SAVESTAT  device &$FACN.$IDEN.PROCESSING";
 echo "Link sensor WAVE.SAVE.STATUS.4 with tag $FACN.$IDEN.SAVESTAT  device &$FACN.$IDEN.PROCESSING";
 echo "Link sensor DRAW_CURVE                                        device &$FACN.$IDEN.PROCESSING";
 echo "Link sensor WAVE_COPY                                         device &$FACN.$IDEN.PROCESSING";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.DRV.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.DRV.HOME @WinShow $FACN.$IDEN.DRV";
#echo "@cron.job $FACN.$IDEN.DRV.HOME @WinDraw $FACN.$IDEN.DRV|Left=600|Top=0|Width=945|Height=437";
 echo "@cron.job $FACN.$IDEN.DRV.HOME @WinDraw $FACN.$IDEN.DRV|Left=600|Top=0|Width=945|Height=450";
 echo "@cron.job $FACN.$IDEN.DRV.HOME @WinDraw $FACN.$IDEN.DRV|Options=-Min,-Max,+Close,-VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.DRV.HOME @WinDraw $FACN.$IDEN.DRV|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.DRV.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.DRV.SHOW @WinShow $FACN.$IDEN.DRV";
 echo "@cron.tab $FACN.$IDEN.DRV.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.DRV.HIDE @WinHide $FACN.$IDEN.DRV";
 echo "@cron.tab $FACN.$IDEN.DRV.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.DRV.SELECT @WinSelect $FACN.$IDEN.DRV";
 echo "@cron.job WJ300A.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.DRV.HOME";
 echo "@cron.job WJ300A.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.DRV.HIDE";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.PARAM.PLOT = Curve_Window";
 echo "[$FACN.$IDEN.PARAM.PLOT]";
 echo "AxisX = ^R{Секунд}___^N^CВремя, 0, 0.25";
 echo "AxisY = ^CПараметры^N^L___{Вольт}, 0, 150";
 echo "CurveList = $FACN.$IDEN.POLL_RATE";
 echo "CurveList = $FACN.$IDEN.ERROR_CNT";
 echo "CurveList = $FACN.$IDEN.ESR";
 echo "CurveList = $FACN.$IDEN.OFST_C1";
 echo "CurveList = $FACN.$IDEN.OFST_C2";
 echo "CurveList = $FACN.$IDEN.OFST_C3";
 echo "CurveList = $FACN.$IDEN.OFST_C4";
 echo "CurveList = $FACN.$IDEN.OFST_M1";
 echo "CurveList = $FACN.$IDEN.TDIV";
 echo "CurveList = $FACN.$IDEN.TLVL";
 echo "CurveList = $FACN.$IDEN.TRDL";
 echo "CurveList = $FACN.$IDEN.VDIV_C1";
 echo "CurveList = $FACN.$IDEN.VDIV_C2";
 echo "CurveList = $FACN.$IDEN.VDIV_C3";
 echo "CurveList = $FACN.$IDEN.VDIV_C4";
 echo "CurveList = $FACN.$IDEN.VDIV_M1";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinShow $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Left=167|Top=120|Height=600|Width=1000";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Options=-Min,-Max,+Close,+VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HOME @WinDraw $FACN.$IDEN.PARAM.PLOT|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.SHOW @WinShow $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.HIDE @WinHide $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.tab $FACN.$IDEN.PARAM.PLOT.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.PARAM.PLOT.SELECT @WinSelect $FACN.$IDEN.PARAM.PLOT";
 echo "@cron.job WJ300A.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.PLOT.HOME";
 echo "@cron.job WJ300A.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.PARAM.PLOT.HIDE";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "$FACN.$IDEN.WAVEFORM.PLOT = Curve_Window";
 echo "[$FACN.$IDEN.WAVEFORM.PLOT]";
 echo "AxisX = ^R{Секунд}___^N^CВремя, 0, 0.25";
 echo "AxisY = ^CПараметры^N^L___{Вольт}, 0, 150";
 echo "CurveList = $FACN.$IDEN.WAVFORM_C1";
 echo "CurveList = $FACN.$IDEN.WAVFORM_C2";
 echo "CurveList = $FACN.$IDEN.WAVFORM_C3";
 echo "CurveList = $FACN.$IDEN.WAVFORM_C4";
 echo "CurveList = $FACN.$IDEN.WAVFORM_M1";
 echo "[]";
 echo "";
 echo "[&CronSrv.StartupScript]";
 echo "@cron.tab $FACN.$IDEN.WAVEFORM.PLOT.HOME 0 0 0";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.HOME @WinShow $FACN.$IDEN.WAVEFORM.PLOT";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.HOME @WinDraw $FACN.$IDEN.WAVEFORM.PLOT|Left=167|Top=120|Height=600|Width=1000";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.HOME @WinDraw $FACN.$IDEN.WAVEFORM.PLOT|Options=-Min,-Max,+Close,+VScroll,-HScroll,+StatusBar";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.HOME @WinDraw $FACN.$IDEN.WAVEFORM.PLOT|Options=+Top,+Left,-Width,-Height";
 echo "@cron.tab $FACN.$IDEN.WAVEFORM.PLOT.SHOW 0 0 0";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.SHOW @WinShow $FACN.$IDEN.WAVEFORM.PLOT";
 echo "@cron.tab $FACN.$IDEN.WAVEFORM.PLOT.HIDE 0 0 0";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.HIDE @WinHide $FACN.$IDEN.WAVEFORM.PLOT";
 echo "@cron.tab $FACN.$IDEN.WAVEFORM.PLOT.SELECT 0 0 0";
 echo "@cron.job $FACN.$IDEN.WAVEFORM.PLOT.SELECT @WinSelect $FACN.$IDEN.WAVEFORM.PLOT";
 echo "@cron.job WJ300A.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.WAVEFORM.PLOT.HOME";
 echo "@cron.job WJ300A.MAIN.CTRL.HOME @cron.run $FACN.$IDEN.WAVEFORM.PLOT.HIDE";
 echo "[]";
 echo "";
 echo "[&DatSrv]";
 echo "CurveList = $FACN.$IDEN.POLL_RATE";
 echo "CurveList = $FACN.$IDEN.ERROR_CNT";
 echo "CurveList = $FACN.$IDEN.ESR";
 echo "CurveList = $FACN.$IDEN.OFST_C1";
 echo "CurveList = $FACN.$IDEN.OFST_C2";
 echo "CurveList = $FACN.$IDEN.OFST_C3";
 echo "CurveList = $FACN.$IDEN.OFST_C4";
 echo "CurveList = $FACN.$IDEN.OFST_M1";
 echo "CurveList = $FACN.$IDEN.TDIV";
 echo "CurveList = $FACN.$IDEN.TLVL";
 echo "CurveList = $FACN.$IDEN.TRDL";
 echo "CurveList = $FACN.$IDEN.VDIV_C1";
 echo "CurveList = $FACN.$IDEN.VDIV_C2";
 echo "CurveList = $FACN.$IDEN.VDIV_C3";
 echo "CurveList = $FACN.$IDEN.VDIV_C4";
 echo "CurveList = $FACN.$IDEN.VDIV_M1";
 echo "[]";
 echo "";
 echo "[WJ300A.MAIN.CTRL]";
 echo "Link sensor @CRON+@CRON.RUN+$FACN.$IDEN.DRV.SELECT        with device &WJ300A.MAIN.CTRL";
 echo "Link sensor $FACN.$IDEN.C1:OFST  with tag $FACN.$IDEN.OFST_C1 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C2:OFST  with tag $FACN.$IDEN.OFST_C2 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C3:OFST  with tag $FACN.$IDEN.OFST_C3 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C4:OFST  with tag $FACN.$IDEN.OFST_C4 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C1:VDIV  with tag $FACN.$IDEN.VDIV_C1 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C2:VDIV  with tag $FACN.$IDEN.VDIV_C2 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C3:VDIV  with tag $FACN.$IDEN.VDIV_C3 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C4:VDIV  with tag $FACN.$IDEN.VDIV_C4 device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C1:TRA   with tag $FACN.$IDEN.TRA_C1  device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C2:TRA   with tag $FACN.$IDEN.TRA_C2  device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C3:TRA   with tag $FACN.$IDEN.TRA_C3  device &$FACN.$IDEN.DRV";
 echo "Link sensor $FACN.$IDEN.C4:TRA   with tag $FACN.$IDEN.TRA_C4  device &$FACN.$IDEN.DRV";
 echo "[]";
 echo "";
};

function WJ300A_SIM(){
 echo "[&$FACN.$IDEN.DRV]";
 echo "Simulator = 1";
 echo "[]";
 rem Include hardware config if one not done yet;
 if [[ "$numhdwcfg" == "0" ]]; then let "numhdwcfg=numhdwcfg+1"; else return 0; fi;
 echo "[ConfigFileList]";
 echo "ConfigFile = $hdwcfg";
 echo "[]";
};

function WJ300A_TABCRC(){
 let "YPOS=TABCRC_TOP+60";
 echo "[SensorList]";
 echo "Sensor = @CRON+@CRON.RUN+$FACN.$IDEN.DRV.SELECT";
 echo "[@CRON+@CRON.RUN+$FACN.$IDEN.DRV.SELECT]";
 echo "Pos = 10, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:14\Color:Black";
 echo "Painter(v) = (glBorderBevel=2)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_150_30_24_lightsteelblue1.bmp $FACN.$IDEN";
 echo "Hint = $FACN.$IDEN.DRV: Открыть окно управления";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.STATE";
 echo "[$FACN.$IDEN.STATE]";
 echo "Pos = 165, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:12\Color:Black";
 echo "Painter(v) = (glBorderBevel=2)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_125_30_24_lime.bmp OK";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_125_30_24_red.bmp TIMEOUT";
 echo "Tag#3 = 2, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_125_30_24_red.bmp REFUSE";
 echo "Tag#4 = 3, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_125_30_24_yellow.bmp СИМУЛЯТОР";
 echo "Tag#5 = 4, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_125_30_24_silver.bmp ОТКЛЮЧЕН";
 echo "Hint = $FACN.$IDEN.STATE: Индикатор состояния";
 echo "[]";
 echo "[SensorList]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C1:TRA";
 echo "[$FACN.$IDEN.C1:TRA]";
 echo "Pos = 440, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=1)+(glButtonMoveX=1)+(glButtonMoveY=1)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_silver.bmp Канал+С1";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_lime.bmp Канал+С1";
 echo "Hint = Sensor Button.c354";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C1:OFST";
 echo "[$FACN.$IDEN.C1:OFST]";
 echo "Pos = 525, $YPOS";
 echo "LED = 12, 0, 0, %3.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C1:OFST";
 echo "Hint = TRDL";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C1:VDIV";
 echo "[$FACN.$IDEN.C1:VDIV]";
 echo "Pos = 650, $YPOS";
 echo "LED = 12, 0, 0, %1.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C1:VDIV";
 echo "Hint = TRDL";
 echo "[]";
 let "YPOS=YPOS+25";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C2:TRA";
 echo "[$FACN.$IDEN.C2:TRA]";
 echo "Pos = 440, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=1)+(glButtonMoveX=1)+(glButtonMoveY=1)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_silver.bmp Канал+С2";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_lime.bmp Канал+С2";
 echo "Hint = Sensor Button.c354";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C2:OFST";
 echo "[$FACN.$IDEN.C2:OFST]";
 echo "Pos = 525, $YPOS";
 echo "LED = 12, 0, 0, %3.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C2:OFST";
 echo "Hint = TRDL";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C2:VDIV";
 echo "[$FACN.$IDEN.C2:VDIV]";
 echo "Pos = 650, $YPOS";
 echo "LED = 12, 0, 0, %1.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C2:VDIV";
 echo "Hint = TRDL";
 echo "[]";
 let "YPOS=YPOS+25";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C3:TRA";
 echo "[$FACN.$IDEN.C3:TRA]";
 echo "Pos = 440, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=1)+(glButtonMoveX=1)+(glButtonMoveY=1)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_silver.bmp Канал+С3";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_lime.bmp Канал+С3";
 echo "Hint = Sensor Button.c354";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C3:OFST";
 echo "[$FACN.$IDEN.C3:OFST]";
 echo "Pos = 525, $YPOS";
 echo "LED = 12, 0, 0, %3.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C3:OFST";
 echo "Hint = TRDL";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C3:VDIV";
 echo "[$FACN.$IDEN.C3:VDIV]";
 echo "Pos = 650, $YPOS";
 echo "LED = 12, 0, 0, %1.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C3:VDIV";
 echo "Hint = TRDL";
 echo "[]";
 let "YPOS=YPOS+25";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C4:TRA";
 echo "[$FACN.$IDEN.C4:TRA]";
 echo "Pos = 440, $YPOS";
 echo "LED = 0, 0, 0, *, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glButtonBit=0)+(glButtonBevel=1)+(glButtonMoveX=1)+(glButtonMoveY=1)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleButton]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_silver.bmp Канал+С4";
 echo "Tag#2 = 1, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_80_20_24_lime.bmp Канал+С4";
 echo "Hint = Sensor Button.c354";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C4:OFST";
 echo "[$FACN.$IDEN.C4:OFST]";
 echo "Pos = 525, $YPOS";
 echo "LED = 12, 0, 0, %3.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C4:OFST";
 echo "Hint = TRDL";
 echo "[]";
 echo "[SensorList]";
 echo "Sensor = $FACN.$IDEN.C4:VDIV";
 echo "[$FACN.$IDEN.C4:VDIV]";
 echo "Pos = 650, $YPOS";
 echo "LED = 12, 0, 0, %1.2f V, Name:PT_Mono\Style:[Bold]\Size:10\Color:Black";
 echo "Painter(v) = (glBorderBevel=1)+(glBorderColor=clGray)";
 echo "Painter(v) = [Painter(v).GuiLib.SimpleBorder]";
 echo "Tag#1 = 0, ~~\Resource\DaqSite\StdLib\Bitmaps\barbmp_120_20_24_white.bmp C4:VDIV";
 echo "Hint = TRDL";
 echo "[]";
 let "TABCRC_TOP+=TABCRC_STEP";
};

function WJ300A_DIS(){
 unix dimcfg \
  -n section "[&$FACN.$IDEN.DRV]" \
  -n print    DimServerMode = 1 \
  -n end \
  -n dis_cmnd   $FACP/$IDEN/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n $devmsg "&$FACN.$IDEN.DRV @DIMGUICLICK=%**" \
  -n end \
  -n dic_cmnd   $FACP/$IDEN/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dis_info   $FACP/$IDEN/ACQ \
  -n tag        $FACN.$IDEN.ACQ \
  -n end \
  -n dis_info   $FACP/$IDEN/CPL_C1+CPL_C2+CPL_C3+CPL_C4 \
  -n tag        $FACN.$IDEN.CPL_C1 \
  -n tag        $FACN.$IDEN.CPL_C2 \
  -n tag        $FACN.$IDEN.CPL_C3 \
  -n tag        $FACN.$IDEN.CPL_C4 \
  -n end \
  -n dis_info   $FACP/$IDEN/DATE \
  -n tag        $FACN.$IDEN.DATE \
  -n end \
  -n dis_info   $FACP/$IDEN/DTBORD\
  -n tag        $FACN.$IDEN.DTBORD \
  -n end \
  -n dis_info   $FACP/$IDEN/DTFORM\
  -n tag        $FACN.$IDEN.DTFORM \
  -n end \
  -n dis_info   $FACP/$IDEN/DTSTART\
  -n tag        $FACN.$IDEN.DTSTART \
  -n end \
  -n dis_info   $FACP/$IDEN/DTPOINTS\
  -n tag        $FACN.$IDEN.DTPOINTS \
  -n end \
  -n dis_info   $FACP/$IDEN/DTWAVE \
  -n tag        $FACN.$IDEN.DTWAVE \
  -n end \
  -n dis_info   $FACP/$IDEN/ESR \
  -n tag        $FACN.$IDEN.ESR \
  -n end \
  -n dis_info   $FACP/$IDEN/IDN \
  -n tag        $FACN.$IDEN.IDN \
  -n end \
  -n dis_info   $FACP/$IDEN/MLEN \
  -n tag        $FACN.$IDEN.MLEN \
  -n end \
  -n dis_info   $FACP/$IDEN/OFST_C1+OFST_C2+OFST_C3+OFST_C4+OFST_M1 \
  -n tag        $FACN.$IDEN.OFST_C1 \
  -n tag        $FACN.$IDEN.OFST_C2 \
  -n tag        $FACN.$IDEN.OFST_C3 \
  -n tag        $FACN.$IDEN.OFST_C4 \
  -n tag        $FACN.$IDEN.OFST_M1 \
  -n end \
  -n dis_info   $FACP/$IDEN/TCPL \
  -n tag        $FACN.$IDEN.TCPL \
  -n end \
  -n dis_info   $FACP/$IDEN/TDIV \
  -n tag        $FACN.$IDEN.TDIV \
  -n end \
  -n dis_info   $FACP/$IDEN/TESR \
  -n tag        $FACN.$IDEN.TESR \
  -n end \
  -n dis_info   $FACP/$IDEN/TLVL \
  -n tag        $FACN.$IDEN.TLVL \
  -n end \
  -n dis_info   $FACP/$IDEN/TRA_C1+TRA_C2+TRA_C3+TRA_C4+TRA_M1 \
  -n tag        $FACN.$IDEN.TRA_C1 \
  -n tag        $FACN.$IDEN.TRA_C2 \
  -n tag        $FACN.$IDEN.TRA_C3 \
  -n tag        $FACN.$IDEN.TRA_C4 \
  -n tag        $FACN.$IDEN.TRA_M1 \
  -n end \
  -n dis_info   $FACP/$IDEN/TRDL \
  -n tag        $FACN.$IDEN.TRDL \
  -n end \
  -n dis_info   $FACP/$IDEN/TRG \
  -n tag        $FACN.$IDEN.TRG \
  -n end \
  -n dis_info   $FACP/$IDEN/TRMD \
  -n tag        $FACN.$IDEN.TRMD \
  -n end \
  -n dis_info   $FACP/$IDEN/TSLP \
  -n tag        $FACN.$IDEN.TSLP \
  -n end \
  -n dis_info   $FACP/$IDEN/TSRC \
  -n tag        $FACN.$IDEN.TSRC \
  -n end \
  -n dis_info   $FACP/$IDEN/TTYP \
  -n tag        $FACN.$IDEN.TTYP \
  -n end \
  -n dis_info   $FACP/$IDEN/VDIV_C1+VDIV_C2+VDIV_C3+VDIV_C4+VDIV_M1 \
  -n tag        $FACN.$IDEN.VDIV_C1 \
  -n tag        $FACN.$IDEN.VDIV_C2 \
  -n tag        $FACN.$IDEN.VDIV_C3 \
  -n tag        $FACN.$IDEN.VDIV_C4 \
  -n tag        $FACN.$IDEN.VDIV_M1 \
  -n end \
  -n dis_info   $FACP/$IDEN/WAVESRC \
  -n tag        $FACN.$IDEN.WAVESRC \
  -n end \
  -n dis_info   $FACP/$IDEN/POLL/ENABLE \
  -n tag        $FACN.$IDEN.POLL_ENABLE \
  -n end \
  -n dis_info   $FACP/$IDEN/POLL/RATE \
  -n tag        $FACN.$IDEN.POLL_RATE \
  -n end \
  -n dis_info   $FACP/$IDEN/ERROR/CNT \
  -n tag        $FACN.$IDEN.ERROR_CNT \
  -n end \
  -n dis_info   $FACP/$IDEN/DIRBASE \
  -n tag        $FACN.$IDEN.DIRBASE \
  -n end \
  -n dis_info   $FACP/$IDEN/DIREXP \
  -n tag        $FACN.$IDEN.DIREXP \
  -n end \
  -n dis_info   $FACP/$IDEN/WFM_SAVE_STATUS \
  -n tag        $FACN.$IDEN.SAVESTAT \
  -n end \
  -n ;
};

function WJ300A_DIC(){
  unix dimcfg \
  -n section "[&$FACN.$IDEN.DRV]" \
  -n print    DimClientMode = 1 \
  -n end \
  -n dic_cmnd   $FACP/$IDEN/DIMGUICLICK \
  -n tag        $FACN.$IDEN.DIMGUICLICK \
  -n end \
  -n dic_info   $FACP/$IDEN/ACQ \
  -n tag        $FACN.$IDEN.ACQ \
  -n end \
  -n dic_info   $FACP/$IDEN/CPL_C1+CPL_C2+CPL_C3+CPL_C4 \
  -n tag        $FACN.$IDEN.CPL_C1 \
  -n tag        $FACN.$IDEN.CPL_C2 \
  -n tag        $FACN.$IDEN.CPL_C3 \
  -n tag        $FACN.$IDEN.CPL_C4 \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CPL_C1" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CPL_C2" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CPL_C3" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.CPL_C4" \
  -n end \
  -n dic_info   $FACP/$IDEN/DATE \
  -n tag        $FACN.$IDEN.DATE \
  -n end \
  -n dic_info   $FACP/$IDEN/DTBORD \
  -n tag        $FACN.$IDEN.DTBORD \
  -n end \
  -n dic_info   $FACP/$IDEN/DTFORM \
  -n tag        $FACN.$IDEN.DTFORM \
  -n end \
  -n dic_info   $FACP/$IDEN/DTSTART \
  -n tag        $FACN.$IDEN.DTSTART \
  -n end \
  -n dic_info   $FACP/$IDEN/DTPOINTS \
  -n tag        $FACN.$IDEN.DTPOINTS \
  -n end \
  -n dic_info   $FACP/$IDEN/DTWAVE \
  -n tag        $FACN.$IDEN.DTWAVE \
  -n end \
  -n dic_info   $FACP/$IDEN/ESR \
  -n tag        $FACN.$IDEN.ESR \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ESR" \
  -n end \
  -n dic_info   $FACP/$IDEN/IDN \
  -n tag        $FACN.$IDEN.IDN \
  -n end \
  -n dic_info   $FACP/$IDEN/MLEN \
  -n tag        $FACN.$IDEN.MLEN \
  -n end \
  -n dic_info   $FACP/$IDEN/OFST_C1+OFST_C2+OFST_C3+OFST_C4+OFST_M1 \
  -n tag        $FACN.$IDEN.OFST_C1 \
  -n tag        $FACN.$IDEN.OFST_C2 \
  -n tag        $FACN.$IDEN.OFST_C3 \
  -n tag        $FACN.$IDEN.OFST_C4 \
  -n tag        $FACN.$IDEN.OFST_M1 \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.OFST_C1" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.OFST_C2" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.OFST_C3" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.OFST_C4" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.OFST_M1" \
  -n end \
  -n dic_info   $FACP/$IDEN/TCPL \
  -n tag        $FACN.$IDEN.TCPL \
  -n end \
  -n dic_info   $FACP/$IDEN/TDIV \
  -n tag        $FACN.$IDEN.TDIV \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.TDIV" \
  -n end \
  -n dic_info   $FACP/$IDEN/TESR \
  -n tag        $FACN.$IDEN.TESR \
  -n end \
  -n dic_info   $FACP/$IDEN/TLVL \
  -n tag        $FACN.$IDEN.TLVL \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.TLVL" \
  -n end \
  -n dic_info   $FACP/$IDEN/TRA_C1+TRA_C2+TRA_C3+TRA_C4+TRA_M1 \
  -n tag        $FACN.$IDEN.TRA_C1 \
  -n tag        $FACN.$IDEN.TRA_C2 \
  -n tag        $FACN.$IDEN.TRA_C3 \
  -n tag        $FACN.$IDEN.TRA_C4 \
  -n tag        $FACN.$IDEN.TRA_M1 \
  -n end \
  -n dic_info   $FACP/$IDEN/TRDL \
  -n tag        $FACN.$IDEN.TRDL \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.TRDL" \
  -n end \
  -n dic_info   $FACP/$IDEN/TRG \
  -n tag        $FACN.$IDEN.TRG \
  -n end \
  -n dic_info   $FACP/$IDEN/TRMD \
  -n tag        $FACN.$IDEN.TRMD \
  -n end \
  -n dic_info   $FACP/$IDEN/TSLP \
  -n tag        $FACN.$IDEN.TSLP \
  -n end \
  -n dic_info   $FACP/$IDEN/TSRC \
  -n tag        $FACN.$IDEN.TSRC \
  -n end \
  -n dic_info   $FACP/$IDEN/TTYP \
  -n tag        $FACN.$IDEN.TTYP \
  -n end \
  -n dic_info   $FACP/$IDEN/VDIV_C1+VDIV_C2+VDIV_C3+VDIV_C4+VDIV_M1 \
  -n tag        $FACN.$IDEN.VDIV_C1 \
  -n tag        $FACN.$IDEN.VDIV_C2 \
  -n tag        $FACN.$IDEN.VDIV_C3 \
  -n tag        $FACN.$IDEN.VDIV_C4 \
  -n tag        $FACN.$IDEN.VDIV_M1 \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VDIV_C1" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VDIV_C2" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VDIV_C3" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VDIV_C4" \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.VDIV_M1" \
  -n end \
  -n dic_info   $FACP/$IDEN/WAVESRC \
  -n tag        $FACN.$IDEN.WAVESRC \
  -n end \
  -n dic_info   $FACP/$IDEN/POLL/ENABLE \
  -n tag        $FACN.$IDEN.POLL_ENABLE \
  -n end \
  -n dic_info   $FACP/$IDEN/POLL/RATE \
  -n tag        $FACN.$IDEN.POLL_RATE \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.POLL_RATE" \
  -n end \
  -n dic_info   $FACP/$IDEN/ERROR/CNT \
  -n tag        $FACN.$IDEN.ERROR_CNT \
  -n $devmsg "&$FACN.$IDEN.DRV @DimTagUpdate=$FACN.$IDEN.ERROR_CNT" \
  -n end \
  -n dic_info   $FACP/$IDEN/DIRBASE \
  -n tag        $FACN.$IDEN.DIRBASE \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.DIRBASE" \
  -n end \
  -n dic_info   $FACP/$IDEN/DIREXP \
  -n tag        $FACN.$IDEN.DIREXP \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.DIREXP" \
  -n end \
  -n dic_info   $FACP/$IDEN/WFM_SAVE_STATUS \
  -n tag        $FACN.$IDEN.SAVESTAT \
  -n $devmsg "&$FACN.$IDEN.CTRL @DimTagUpdate=$FACN.$IDEN.SAVESTAT" \
  -n end \
  -n ;
};

##############
# delete files
##############
function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then rm -f "$1"; fi;
  shift;
 done;
};

################
# main function.
################
function main(){
 if pushd $scriptHOME >/dev/null 2>&1; then
  wj300a_devices;
 fi;
 popd >/dev/null 2>&1;
 local code=$?;
 if [ $code -eq 0 ]; then
  local msg="$(langstr ru "конфигурация успешно создана." en "configuration created.")";
  unix tooltip-notifier preset stdOk delay 7000 text "$scriptname: $msg";
 else
  local msg="$(langstr ru "ошибка создания конфигурации." en "failed on create configuration.")";
  fatal $code "$scriptname: $msg";
 fi;
 return $code;
};

main "$@";

##############
## END OF FILE
##############
