#!/bin/bash

##########################################################
# Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru
##########################################################

################################################################
## Utility to generate UniHeater's simulator configuration.   ##
################################################################

function fatal(){
 unix tooltip-notifier text "$scriptname: $2" preset stdError delay 60000;
 1>&2 echo -ne "\n$2\n\n";
 exit $1;
};
function call(){
 if [ $# -gt 0 ]; then
  echo "$@";
  eval "$@";
 fi;
};
function rem(){
 return 0;
};

################################
# Current script identification.
################################
declare startupdir="$(pwd -LP)";
declare scriptfile="${BASH_SOURCE[0]}";
declare scriptname="$(basename $scriptfile)";
declare scriptbase="$(basename $scriptfile .sh)";
declare scripthome="$(dirname  $scriptfile)";
declare scriptFILE="$(realpath $scriptfile)";
declare scriptHOME="$(dirname  $scriptFILE)";

declare -x SelfExt=".sh";
declare -x SelfBase="$scriptbase";
declare -x SelfFile="$scriptFILE";
declare -x SelfName="$scriptname";
declare -x HomeDir="$startupdir";
declare -x SelfDir="$scriptHOME";

declare -i errors=0;

function duh_sumulator_make(){
 call SetErrors 0;
 call SetErrorLevel 0;
 #ParseOptions
 case $1 in
  -h|--help)  call PrintHelp; exit 0; ;;
  --version)  call PrintVersion; exit 0; ;;
 *) ;;
 esac;
 #ParseArguments
 call RequiredParamInt UH_NumHeaters $1                     || exit 1       ; ### Number of Unified Heaters, 1..32
 #SetAddonVariables
 source "$SelfDir/demo_uniheater_init.sh"                                   ; ### Initialize environ
 call GetPrefixFile UH_PrefixFile $UH_PrefixName                            ; ### File name prefix
 call Define UH_PrefixFile $(printenv UH_PrefixName | tr '.' '_')           ; ### File name prefix
 #CheckRequiredEntities
 call RequiredUnixUtils                                     || exit 1;
 call RequiredAdminRights                                   || exit 1;
 call RequiredParentCrwDaq                                  || exit 1;
 call RequiredDefined UH_PrefixName                         || exit 1;
 call RequiredDefined UH_PrefixFile                         || exit 1;
 call RequiredDefined UH_NumHeaters                         || exit 1;
 call RequiredDefined UH_TileNumber                         || exit 1;
 call RequiredDefined UH_PrefixTSen                         || exit 1;
 call RequiredDefined UH_PrefixPSen                         || exit 1;
 call RequiredDefined UH_NumberTSen                         || exit 1;
 call RequiredDefined UH_NumberPSen                         || exit 1;
 call RequiredDefined UH_NickTSen                           || exit 1;
 call RequiredDefined UH_NickPSen                           || exit 1;
 call RequiredIntRange $UH_NumHeaters 1 32                  || exit 1;
 call RequiredIntRange $UH_TileNumber 1 $UH_NumHeaters      || exit 1;
 if pushd "$HomeDir" >/dev/null 2>&1; then
  call MakeSim;
 fi;
 popd >/dev/null 2>&1;
 call HasNoErrors || call ErrorPrintLn "$errors error(s) found in $SelfName.";
 return $?;
};

function PrintVersion(){
  echo "$SelfBase version 1.0";
};

function  PrintHelp(){
 call PrintVersion;
 echo "Copyright (c) 2001-2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "Utility to generate UniHeater's simulator configuration.";
 echo "Usage:";
 echo " $SelfBase [Options] [Arguments]";
 echo "Options:";
 echo " -h,--help  - print help";
 echo " --version  - print version";
 echo " --samples  - create command files [init,make,zero] to generate config";
 echo "Arguments:";
 echo " UH_NumHeaters - Number of Unified Heaters, 1..32";
 echo "Example:";
 echo " $SelfBase 16";
};



function MakeSim(){
 call SimEnviron;
 call SimHead      >  duh_simulator.cfg;
 call SimTags      >> duh_simulator.cfg;
 call SimDatSrv    >> duh_simulator.cfg;
 call SimMainCtrl  >> duh_simulator.cfg;
 call SimScript    >> duh_simulator.cfg;
 call ./demo_uniheater_make.sh;
};

function SimEnviron(){
 ###  Define Name               Value               ; ### Comment:
 call Define UH_PrefixName      DEMO                ; ### Prefix for UniHeater
 call Define UH_NumHeaters      16                  ; ### Number of Heaters
 call Define UH_TileNumber      8                   ; ### Number of columns
 call Define UH_PrefixTSen      DUH.SIM.T           ; ### Prefix for T sensors
 call Define UH_PrefixPSen      DUH.SIM.P           ; ### Prefix for P sensors
 call Define UH_NumberTSen      16                  ; ### Number of T sensors to bind
 call Define UH_NumberPSen      16                  ; ### Number of P sensors to bind
 call Define UH_HistoryLen      5000                ; ### History for curves
 export UH_PREF="$UH_PrefixName.UNIHEATER";
};

function SimHead(){
 echo ";*****************************";
 echo "; Simulator for Demo_UniHeater";
 echo ";*****************************";
 echo "";
};


function SimTags(){
 echo "[DataStorage] ; Simulator curves T,P";
 for i in $(seq 1 $UH_NumberTSen); do echo "DUH.SIM.T$i  = Curve 0 10 Black  15 1"; done;
 for i in $(seq 1 $UH_NumberPSen); do echo "DUH.SIM.P$i  = Curve 0 10 Black  15 1"; done;
 echo "[]";
 echo "";
};


function SimDatSrv(){
 echo "[&DATSRV] ; Data Saver";
 for i in $(seq 1 $UH_NumberTSen); do echo "CurveList = DUH.SIM.T$i"; done;
 for i in $(seq 1 $UH_NumberPSen); do echo "CurveList = DUH.SIM.P$i"; done;
 for i in $(seq 1 $UH_NumHeaters); do echo "CurveList = $UH_PREF.TREAL_$i"; done;
 for i in $(seq 1 $UH_NumHeaters); do echo "CurveList = $UH_PREF.PREAL_$i"; done;
 for i in $(seq 1 $UH_NumHeaters); do echo "CurveList = $UH_PREF.GATE_$i"; done;
 for i in $(seq 1 $UH_NumHeaters); do echo "CurveList = $UH_PREF.BLK_$i"; done;
 echo "[]";
 echo "";
};


function SimMainCtrl(){
 echo "[DUH.MAIN.CTRL] ; Add sensors to main control window";
 for i in $(seq 1 3); do
  echo "Link sensor DEMO.UNIHEATER.BT_$i    with tag   $UH_PREF.ST_$i    device &$UH_PREF.CTRL";
  echo "Link sensor DEMO.UNIHEATER.TGOAL_$i with tag   $UH_PREF.TGOAL_$i device &$UH_PREF.CTRL";
  echo "Link sensor DEMO.UNIHEATER.TREAL_$i with curve $UH_PREF.TREAL_$i device &$UH_PREF.CTRL";
  echo "Link sensor DEMO.UNIHEATER.GATE_$i  with curve $UH_PREF.GATE_$i  device &$UH_PREF.CTRL";
  echo "Link sensor DEMO.UNIHEATER.TNAME_$i with tag   $UH_PREF.TNAME_$i device &$UH_PREF.CTRL";
  echo "Link sensor DEMO.UNIHEATER.PNAME_$i with tag   $UH_PREF.PNAME_$i device &$UH_PREF.CTRL";
  echo "Link sensor DEMO.UNIHEATER.PROP_$i  with device &$UH_PREF.CTRL";
 done;
 for i in $(seq 1 $UH_NumHeaters); do
  if [ $i -le 10 ]; then echo "Link sensor DUH.T$i with curve $UH_PREF.TREAL_$i device &$UH_PREF.CTRL"; fi;
  if [ $i -le 10 ]; then echo "Link sensor DUH.P$i with curve $UH_PREF.PREAL_$i device &$UH_PREF.CTRL"; fi;
 done;
 echo "Link sensor DEMO.UNIHEATER.MAKE.LM9 with device &$UH_PREF.CTRL";
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "DUH.T.PLOT = Curve_Window";
 echo "[DUH.T.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CT_Sensors^N^L___{C}, 0, 1000";
 for i in $(seq 1 $UH_NumberTSen); do echo "CurveList = DUH.SIM.T$i"; done;
 echo "[]";
 echo "";
 echo "[Windows]";
 echo "DUH.P.PLOT = Curve_Window";
 echo "[DUH.P.PLOT]";
 echo "AxisX = ^R{Hours}___^N^CTime, 0, 0.05";
 echo "AxisY = ^CP_Sensors^N^L___{Bar}, 0, 2500";
 for i in $(seq 1 $UH_NumberTSen); do echo "CurveList = DUH.SIM.P$i"; done;
 echo "[]";
 echo "";
};



function SimScript(){
 echo ";***********************************";
 echo "; Sumulator script to test UniHeater";
 echo ";***********************************";
 echo "[DeviceList]";
 echo "&DUH.SIM = device software script";
 echo "[&DUH.SIM]";
 echo "Comment       = Simulator for UniHeater.";
 echo "InquiryPeriod = 1";
 echo "DevicePolling = 100, tpNormal";
 echo "ScriptSection = &DUH.SIM.TEXT";
 call LinkAI $UH_NumberTSen;
 call LinkAO $UH_NumberTSen $UH_NumberPSen;
 call LinkDI $UH_NumberTSen;
 call SimScriptText $UH_NumberTSen;
};
function LinkAI(){
 echo "AnalogInputs  = $1";
 for i in $(seq 1 $1); do call LinkAIs $i; done;
};
function LinkAIs(){
 let "ai=$1-1";
 echo "Link AnalogInput $ai with curve DUH.SIM.T$1";
};
function LinkAO(){
 let "numao=$1+$2";
 echo "AnalogOutputs = $numao";
 for i in $(seq 1 $1); do call LinkAOsT $i; done;
 for i in $(seq 1 $1); do call LinkAOsP $i $1; done;
};
function LinkAOsT(){
 let "ao=$1-1";
 echo "Link AnalogOutput $ao with curve DUH.SIM.T$1 history $UH_HistoryLen";
};
function LinkAOsP(){
 let "ao=$1-1+$2";
 echo "Link AnalogOutput $ao with curve DUH.SIM.P$1 history $UH_HistoryLen";
};
function LinkDI(){
 let "numdi=$1*2";
 echo "DigitalInputs = $numdi";
 for i in $(seq 1 $1); do call LinkDIsBLK $i; done;
 for i in $(seq 1 $1); do call LinkDIsGAT $i $1; done;
};
function LinkDIsBLK(){
 let "di=$1-1";
 if [ $1 -gt $UH_NumHeaters ]; then return; fi;
 echo "Link DigitalInput $di with curve $UH_PREF.BLK_$1  inverted bit 0 ; Heater$1  POWER";
};
function LinkDIsGAT(){
 let "di=$1-1+$2";
 if [ $1 -gt $UH_NumHeaters ]; then return; fi;
 echo "Link DigitalInput $di with curve $UH_PREF.GATE_$1          bit 0 ; Heater$1  SSR";
};
function SimScriptText(){
 echo "[]";
 echo "[&DUH.SIM.TEXT]";
 echo "var runcount";
 echo "runcount=runcount+1";
 echo "Noise=0.0";
 echo "RoomTemp=20";
 echo "HeatRate=1e-1";
 echo "CoolRate=1e-4";
 echo "nchan=NumAis()";
 echo "i=0";
 echo "modes=vnew(nchan)";
 for i in $(seq 1 $1); do echo "var tagMode$i"; done;
 for i in $(seq 1 $1); do echo "if not(tagMode$i) then tagMode$i=@findtag $UH_PREF.MODE_$i"; done;
 for i in $(seq 1 $1); do echo "vput(modes,$i-1,iGetTag(tagMode$i))"; done;
 echo "Loop:";
 echo " j=i+nchan";
 echo " pow=eq(getdi_yn(i),0)";
 echo " ssr=ne(getdi_yn(j),0)";
 echo " temp=getai_yn(i)+RoomTemp*eq(runcount,1)";
 echo " if eq(vget(modes,i),0) then temp=temp+pow*ssr*HeatRate-(temp-RoomTemp)*CoolRate";
 echo " if eq(vget(modes,i),1) then temp=temp-pow*ssr*HeatRate-(temp-RoomTemp)*CoolRate";
 echo " temp=temp+Noise*random(-1,1)";
 echo " pres=temp/2";
 echo " putao(i, time(), temp)";
 echo " putao(j, time(), pres)";
 echo " i=i+1";
 echo "if lt(i,nchan) then goto Loop";
 echo "vfree(modes)";
 echo "[]";
 echo "";
 echo ";**********************************";
 echo ";*** Speak server for UniHeater ***";
 echo ";**********************************";
 echo "[&$UH_PREF.CTRL]";
 echo "UsesSpeakSrv = 1";
 echo "[]";
 echo "";
};


################################################################################
## Common library routines                                                    ##
################################################################################

function Define(){
 if [ -z "$1" ]; then return 0; fi; ### Environ Variable Name
 if [ -z "$2" ]; then return 0; fi; ### Environ Variable Value
 if [ -z "$(printenv $1)" ]; then
  export $1="$2";
 fi;
};

function GetPrefixFile(){
 if [ -z "$1" ]; then return; fi;
 if [ -z "$2" ]; then return; fi;
 export $1="$(echo "$2" | tr '[:upper:]' '[:lower:]')";
};

function RequiredParamStr(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then export $1="$2"; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName - missed string param $1. Call $SelfName --help for details.";
};

function RequiredParamInt(){
 if [ -z "$1" ]; then return 0; fi;
 if [ -n "$2" ]; then let "$1=$2"; export $1; fi;
 if [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName : missed integer param $1. Call $SelfName --help for details.";
};

function RequiredAdminRights(){
 if unix sudoit -v >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: Access denied. Admin rights required.";
};

function RequiredUnixUtils(){
 if which unix >/dev/null 2>&1; then return 0; fi;
 fatal 1 "Error: $SelfName: UnixUtils is not valid. Install UnixUtils first.";
};

function RequiredParentCrwDaq(){
 local pid="$CRW_DAQ_SYS_EXE_PID";
 local cmd="$CRW_DAQ_SYS_EXE_FILE";
 local exe="$(basename $cmd)";
 local lst="";
 if [ -n "$pid" ] && [ -n "$exe" ]; then
  lst="$(pgrep $exe | grep $pid)";
 fi;
 if [ -z "$lst" ]; then
  fatal 1 "Error: $SelfName: parent CRW-DAQ is not found. Run this script from CRW-DAQ.";
 fi;
};

function RequiredFile(){
 if [ -n "$1" ] && [ -e "$1" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: not found $1.";
};

function RequiredDefined(){
 if [ -n "$1" ] && [ -n "$(printenv $1)" ]; then return 0; fi;
 fatal 1 "Error: $SelfName: undefined variable $1. Call $SelfName --help for details.";
};

function RequiredIntRange(){
 if [ -n "$1" ] &&  [ -n "$2" ] &&  [ -n "$3" ] && [ $1 -ge $2 ] && [ $1 -le $3 ]; then return 0; fi;
 fatal 1 "Error: $SelfName: invalid value $1 (not in $2..$3 range). Call $SelfName --help for details.";
};

function SetErrors(){
 let "errors=$1";
};

function IncErrors(){
 let "errors+=$1";
};

function HasNoErrors(){
 if [ "$errors" = "0" ]; then return $?; fi;
 return 1;
};

function Exec(){
 echo "$@";
 call "$@";
};

function PrintLn(){
 echo "$@";
};

function ErrorPrint(){
 1>&2 echo -n "$@";
};

function ErrorPrintLn(){
 1>&2 echo "$@";
};

function SetErrorLevel(){
 return $1;
};

function DeleteFiles(){
 while [ -n "$1" ]; do
  if [ -e "$1" ]; then
   rm -f "$1";
  fi;
  shift;
 done;
};


duh_sumulator_make "$@";

##############
## END OF FILE
##############
