#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## Tiny Web server (emulated with busybox httpd).
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

function usage(){
 local name="$(basename ${BASH_SOURCE[0]})";
 echo "$name version 1.0";
 if [ "$1" = "0" ]; then return; fi;
 echo "Copyright (c) 2024 Alexey Kuryakin daqgroup@mail.ru";
 echo "TinyWeb: Web Server emulator (using busybox httpd).";
 echo "Usage:";
 echo " $name -h,--help       - show help";
 echo " $name --version       - show version";
 echo " $name -l,--list       - list running tinyweb servers";
 echo " $name --kill          - kill all tinyweb servers";
 echo " $name --kill port     - kill server by port";
 echo " $name homedir port    - run as process, wait";
 echo " $name homedir port -d - run as daemon, no wait";
 echo "Examples:";
 echo " $name /var/tmp/www 8080 -d";
 echo " $name /var/tmp/www 8000";
 echo " $name --list";
 echo " $name --kill 8080";
};

let fatal_notify_uses=0;
declare opt="";

function list_tinyweb(){
 local pattern="$1";
 pgrep -a busybox | grep "${pattern:=busybox httpd}";
};

function kill_tinyweb(){
 if [ -z "$1" ]; then
  local pids="$(list_tinyweb | awk '{print $1}')";
  if [ -n "$pids" ]; then
   echo "kill $pids";
   for pid in $pids; do kill $pid; done;
  fi;
 else
  local mask="busybox httpd .* -p $1";
  local pids="$(list_tinyweb "$mask" | awk '{print $1}')";
  if [ -n "$pids" ]; then
   echo "kill $pids";
   for pid in $pids; do kill $pid; done;
  fi;
 fi;
};

function run_tinyweb(){
 if [ -z "$2" ]; then $1; else exec $1 $2; fi;
};

function exec_tinyweb(){
 local home="$1"; shift;
 local port="$1"; shift;
 local mode="$1"; shift; 
 if [ -e "$home/" ]; then
  if is_number $port; then
   local cmnd="busybox httpd -h $home -p $port";
   if [ "$mode" = "-d" ]; then opt=""; fi;
   if run_tinyweb "$cmnd" "$opt"; then
    list_tinyweb "$cmnd";
   else
    list_tinyweb;
    fatal 1 "$scriptname: $(langstr ru "не могу запустить" en "could not run") busybox httpd";
   fi;
  else
   fatal 1 "$scriptname: $(langstr ru "недопустимый порт $port" en "invalid port $port")";
  fi;
 else
  fatal 1 "$scriptname: $(langstr ru "не найден каталог $home" en "invalid directory $home")";
 fi;
};

function main(){
 case $1 in
  --version) usage 0; exit 0; ;;
  -h|--help) usage 1; exit 0; ;;
  -l|--list) list_tinyweb; ;;
  --kill)    shift; kill_tinyweb "$@"; ;;
  '')        fatal 1 "Error: empty arguments. Type $(basename ${BASH_SOURCE[0]}) -h for help."; ;;
  -*)        fatal 1 "Error: wrong option $1. Type $(basename ${BASH_SOURCE[0]}) -h for help."; ;;
  *)         opt="-f -vv";  exec_tinyweb "$@"; ;;
 esac;
};

main "$@";

##############
## END OF FILE
##############
