#!/bin/bash

###########################################################
## Copyright (c) 2002-2024 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## lister - simulate Windows command (lister) on Linux
## lister uses to view text files (txt,pdf,djvu etc)
###########################################################

###########################################################
source $(crwkit which crwlib_base.sh); # Use base library #
source $(crwkit which crwlib_file.sh); # Use file library #
###########################################################

function execute(){
 if which $1 >/dev/null 2>&1; then
  exec "$@";
 else
  echo "$(basename ${BASH_SOURCE[0]}): could not find $1" 1>&2;
 fi;
};

function xterm_cat(){
 xterm -fa 'PT Mono' -fs 12 -fg white -bg black -sb -rightbar -hold -geometry 138x35+50+130 -title "cat $*" -e cat "$@";
};

function xterm_tail(){
 local opt="-n 1000 -F";
 xterm -fa 'PT Mono' -fs 12 -fg white -bg black -sb -rightbar -hold -geometry 138x35+50+130 -title "tail $opt $*" -e tail $opt "$@";
};

function xterm_mcview(){
 xterm -fa 'PT Mono' -fs 12 -fg white -bg black -sb -rightbar -hold -geometry 138x35+50+130 -title "mcview $*" -e mcview "$@";
};

function geany_viewer(){
 local cfgdir="$TMPDIR/daqgroup-$USER/geany-viewer";
 if [ ! -e $cfgdir/ ]; then mkdir -p $cfgdir; fi;
 if [ ! -e $cfgdir/geany.conf ]; then  cp -fv $scriptHOME/geany.conf $cfgdir/; fi;
 local opt="--read-only --config $cfgdir";
 opt="$opt --no-session"; # opt="$opt --new-instance";
 opt="$opt --no-plugins --no-ctags --no-msgwin --no-terminal";
 geany $opt "$@";
};

#########################
# Get viewer for file $1.
#########################
function get_viewer(){
 if [ -z "$1" ]; then unix getapppath geany okular; return; fi;
 local desc="$(file -b -L "$1")";
 ########################
 # PDF, DJVU, PS document
 ########################
 if file_desc_match_any "$desc" document && file_desc_match_any "$desc" PDF DjVu PostScript; then
  unix getapppath okular gv evince xpdf mupdf -t application/pdf firefox -t text/html cat;
  return;
 fi;
 ###############
 # HTML document
 ###############
 if file_desc_match_any "$desc" document && file_desc_match_any "$desc" HTML; then
  unix getapppath firefox chromium opera tor -t text/html okular mcview cat;
  return;
 fi;
 if file_desc_match_all "$desc" Internet shortcut text; then
  unix getapppath firefox chromium -t text/html okular mcview cat;
  return;
 fi;
 #################
 # ASCII text file
 #################
 if file_desc_match_any "$desc" text &&  file_desc_match_any "$desc" ASCII ISO-8859; then
  unix getapppath geany mcview okular -t text/plain firefox -t text/html cat;
  return;
 fi;
 ###################
 # Unicode text file
 ###################
 if file_desc_match_any "$desc" text && file_desc_match_any "$desc" UTF-8 Unicode; then
  unix getapppath geany mcview okular -t text/plain firefox -t text/html cat;
  return;
 fi;
 #########################################
 # any document which has MIME association
 #########################################
 if file_desc_match_any "$desc" document && [ -n "$(file -b -i $1)" ]; then
  unix getapppath xdg-open;
  return;
 fi;
 ################
 # any image file
 ################
 if file_desc_match_any "$desc" bitmap image; then
  unix getapppath gwenview fly-gallery -t image/png firefox -t text/html;
  return;
 fi;
 ################
 # any audio file
 ################
 if file_desc_match_any "$desc" audio; then
  unix getapppath play qmmp;
  return;
 fi;
 ################
 # any video file
 ################
 if file_desc_match_any "$desc" video; then
  unix getapppath vlc smplayer mplayer totem banshee xine mpv kodi kylin-video parole;
  return;
 fi;
 ###############
 # any text file
 ###############
 if file_desc_match_any "$desc" text; then
  unix getapppath geany okular mcview -t text/plain firefox -t text/html cat;
  return;
 fi;
 ####################################
 # Report failure on viewer not found
 ####################################
 fatal 1 "$(langstr ru "Не знаю как открыть файл $1" en "Could not find program to view $1")";
 return;
 # obsolete version:
 case $1 in
  *.desktop)                   unix getapppath okular mcview cat; ;;
  *.html|*.htm)                unix getapppath firefox chromium -t text/html okular mcview cat; ;;
  *.pdf|*.djv|*.djvu|*.ps)     unix getapppath okular gv evince -t application/pdf firefox -t text/html cat; ;;
  *.bmp|*.png|*.gif|*.ico)     unix getapppath gwenview fly-gallery -t image/png firefox -t text/html; ;;
  *.jpg|*.jpeg|*.tiff|*.pcx)   unix getapppath gwenview fly-gallery -t image/png firefox -t text/html; ;;
  *.txt|*.text|*.ascii)        unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.ini|*.cfg|*.crc|*.conf)    unix getapppath geany mcview -t text/plain firefox -t text/html cat; ;;
  *.pas|*.inc|*.lpr|*.dpr)     unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.lpi|*.lps|*.lpk|*.pp)      unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.sml|*.fsm|*.sobj)          unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.md|*.markdown)             unix getapppath markdownviewer geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.lst|*.list)                unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.sh|*.bash|*.dash|*.csh|*.ksh|*.tcsh) unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.xml)                       unix getapppath geany okular mcview -t text/plain firefox -t text/html cat; ;;
  *.log)                       unix getapppath geany okular mcview cat; ;;
  *)                           unix getapppath xdg-open cat; ;;
 esac;
};

function main(){
 local exe="$(get_viewer "$1")";
 if [ -z "$exe" ]; then
  1>&2 echo "Error: could not find viewer.";
  exit 1;
 fi;
 case $(basename $exe) in
  cat)    xterm_cat    "$@"; ;;
  tail)   xterm_tail   "$@"; ;;
  mcview) xterm_mcview "$@"; ;;
  geany)  geany_viewer "$@"; ;;
  *)      execute $exe "$@"; ;;
 esac;
};

main "$@";

##############
## END OF FILE
##############
