#!/bin/bash

###########################################################
## Copyright (c) 2002-2023 Alexey Kuryakin daqgroup@mail.ru
###########################################################

###########################################################
## grun - simulate Windows grun command from Unix Utils kit
###########################################################

function selfname(){
 echo "$(basename ${BASH_SOURCE[0]})";
};
function print_version(){
 echo "$(selfname) version 1.0 built 20230905";
};
function print_copyright(){
 echo "Copyright (c) 2002-2023 Alexey Kuryakin daqgroup@mail.ru";
};
function print_help(){
 print_version;
 print_copyright;
 echo "about:";
 echo " $(selfname) is 'general run' utility to run specified program.";
 echo " $(selfname) simulates Windows grun command from UnixUtils kit.";
 echo "usage:";
 echo " $(selfname) [options] [parameters]";
 echo "options:";
 echo " --version - print version";
 echo " -h,--help - print help screen";
 echo " -e,--elev - run elevated as root, i.e. apply sudo";
 echo " -w,--wait - wait for result (by default not wait)";
 echo "             if no wait, detach (disown) a process";
 echo "parameters:";
 echo " command line to run";
 echo "exit code:";
 echo " if not wait (default) return 0 or error (1)";
 echo " if wait (-w) return run command's exit code";
 echo "examples:";
 echo " $(selfname) --help";
 echo " $(selfname) okular demo.pdf";
 echo " $(selfname) -w okular demo.pdf";
};

# detect location of script (with symlink resolved)

readonly scriptfile="${BASH_SOURCE[0]}";          # caller location of script
readonly scriptFILE="$(realpath $scriptfile)";    # physic location of script
readonly scriptHOME="$(dirname  $scriptFILE)";    # physic location of folder

function isoption(){
 case $1 in
  -*) return 0; ;;
  *)  return 1; ;;
 esac;
};

function is_number(){
 case $1 in
  ''|*[!0-9]*)  return 1; ;;
  *)            return 0; ;;
 esac;
}; 

function trim(){
 echo "$*" | xargs;
};

function report(){
 if [ "$verb" != "0" ]; then
  echo "$1";
 fi;
};

function main(){
 local wait=0;
 local verb=0;
 local disp=1;
 local prio=8;
 local sudo="";
 while isoption $1; do
  case $1 in
   -v|--verb) verb=1;  ;;
   -w|--wait) wait=1;  ;;
   -h|--hide) disp=0;  ;;
   -0)        disp=0;  ;;
   -1)        disp=1;  ;;
   -2)        disp=2;  ;;
   -3)        disp=3;  ;;
   -4)        disp=4;  ;;
   -5)        disp=5;  ;;
   -6)        disp=6;  ;;
   -7)        disp=7;  ;;
   -8)        disp=8;  ;;
   -9)        disp=9;  ;;
   -i)        prio=4;  ;;
   -b)        prio=6;  ;;
   -n)        prio=8;  ;;
   -a)        prio=10; ;;
   -g)        prio=13; ;;
   -r)        prio=24; ;;
   -c)        ;;
   -k)        ;;
   -s)        ;;
   -o)        ;;
   -p)        ;;
   -m)        ;;
   -x)        ;;
   -y)        ;;
   -e|--elev) sudo="sudo -n"; ;;
   -?|--help) print_help; return 0; ;;
   --version) print_version; return 0; ;;
   -*)        1>&2 echo "$(selfname): invalid option $1"; exit 1; ;;
   *)         break; ;;
  esac;
  shift;
 done;
 if [ $# = 0 ]; then
  print_help;
  return 0;
 fi;
 if [ -e $1 ] || which $1 >/dev/null 2>&1; then
  if [ -e $1 ] && [ ! -x $1 ]; then
   echo "grun: not executable $1" 1>&2;
   exit 1;
  fi;
  local pid=""; local code=0;
  report "grun: $(trim $sudo $*)";
  if [ "$wait" = "0" ]; then
   $sudo "$@" </dev/null >/dev/null 2>&1 & disown;
   code="$?"; pid="$!";
  else
   $sudo "$@" </dev/null >/dev/null 2>&1;
   code="$?"; pid="$!";
  fi;
  if is_number $pid; then
   report "grun: start PID $pid";
  fi;
  if is_number $code; then
   report "grun: exit code $code";
  fi;
  return $code;
 else
  echo "grun: not found $1" 1>&2;
  exit 1;
 fi;
};

main "$@";

##############
## END OF FILE
##############
