#!/bin/bash

##########################################################
# Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru
##########################################################

##########################################################
# Utility to get the parentage directory list, i.e. chain.
# For example:
# > dirchain /usr/share/applications | xargs
#   /usr/share/applications /usr/share /usr
##########################################################

################################
# Current script identification.
################################
readonly startupdir="$(pwd -LP)";
readonly scriptfile="${BASH_SOURCE[0]}";
readonly scriptname="$(basename $scriptfile)";
readonly scriptbase="$(basename $scriptfile .sh)";
readonly scripthome="$(dirname  $scriptfile)";
readonly scriptFILE="$(realpath $scriptfile)";
readonly scriptHOME="$(dirname  $scriptFILE)";

##################################################
# Get directory depth, i.e. number of parent dirs.
# dir:depth example: /:0, /mnt:1, /mnt/data:2, etc
##################################################
function dir_depth(){
 local dir="$(realpath -ms "$1")";
 local dep=$(echo "$dir" | tr '/' '\n' | wc -l);
 if [[ $dep -gt 0 ]]; then let dep--; fi;
 echo $dep;
};

####################################
# Get parentage directory chain list
# For example: /opt/a/b /opt/a /opt.
####################################
function dir_chain(){
 local dir="$(realpath -ms "$1")"; local dep="$2";
 if [[ -L $dir ]]; then dir="$(realpath "$dir")"; fi;
 if [[ -f $dir ]]; then dir="$(dirname "$dir")"; fi;
 if [[ -z $dep ]]; then dep="$(dir_depth "$dir")"; fi;
 if [[ -d $dir ]]; then
  for i in $(seq 1 $dep 2>/dev/null); do
   if [[ $dir = '/' ]]; then break; fi;
   echo "$dir"; dir="$(dirname $dir)";
  done;
 else
  1>&2 echo "$scriptname: invalid directory $dir";
  return 1;
 fi;
};

function do_version(){
 echo "$scriptname version 1.0";
};

function do_print_help(){
 do_version;
cat<<EOF
$scriptname Copyright (c) 2025 Alexey Kuryakin daqgroup@mail.ru
Utility to get the parentage directory list, i.e. directory chain.
Usage:
 $scriptname [-options] [dir|file [depth]]
Options:
 --version  - print version
 -h,--help  - print help screen
 -t,--tac   - use tac, i.e. reverse order chain
 -d,--depth - print directory chain depth
Parameters:
 dir        - directory name or
 file       - regular file name
 depth      - max. chain depth
Examples:
 > $scriptname /usr/share/applications
   /usr/share/applications
   /usr/share
   /usr
 > $scriptname /usr/share/applications 2
   /usr/share/applications
   /usr/share
 > $scriptname -t /usr/share/applications
   /usr
   /usr/share
   /usr/share/applications
 > $scriptname -d /usr/share/applications
   3
EOF
};

function main(){
 case $1 in
  '')                echo "Usage: $scriptname [-h|--help|--version|-d|--depth|-t|--tac] [directory [depth]]"; ;;
  --version)         do_version; return 0; ;;
  -h|-help|--help)   do_print_help; return 0; ;;
  -t|-tac|--tac)     shift; dir_chain "$@" | tac; ;;
  -d|-depth|--depth) shift; dir_depth "$@"; ;;
  *)                 dir_chain "$@"; ;;
 esac;
};

######
# MAIN
######

main "$@";

##############
## END OF FILE
##############
