////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Text Replace Dialog.                                                  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231121 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_textreplacedialog; // Form Text Replace Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 _crw_alloc, _crw_str, _crw_eldraw, _crw_fio, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils,
 form_crwdaqsyschild, form_texteditor;

type

  { TFormTextReplaceDialog }

  TFormTextReplaceDialog = class(TMasterForm)
    PanelButtons: TPanel;
    BitBtnFind: TBitBtn;
    BitBtnClose: TBitBtn;
    PanelControls: TPanel;
    LabelFindText: TLabel;
    CheckBoxWholeWord: TCheckBox;
    CheckBoxMatchCase: TCheckBox;
    EditFindText: TEdit;
    LabelReplaceText: TLabel;
    EditReplaceText: TEdit;
    BitBtnReplace: TBitBtn;
    BitBtnReplaceAll: TBitBtn;
    CheckBoxPrompt: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var CloseAction: TCloseAction);
    procedure FormShow(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure BitBtnFindClick(Sender: TObject);
    procedure BitBtnReplaceClick(Sender: TObject);
    procedure BitBtnReplaceAllClick(Sender: TObject);
    procedure BitBtnCloseClick(Sender: TObject);
  private
    { Private declarations }
    myEditor : TFormTextEditor;
    function  ValidEditor:Boolean;
    procedure SelectWordUnderCursor;
  public
    { Public declarations }
  end;

function NewFormTextReplaceDialog(aEditor:TFormTextEditor; const aParams:LongString=''):TFormTextReplaceDialog;

implementation

{$R *.lfm}

function NewFormTextReplaceDialog(aEditor:TFormTextEditor; const aParams:LongString=''):TFormTextReplaceDialog;
var apFlags:Integer;
begin
 Result:=nil;
 try
  if aEditor.Ok then begin
   Application.CreateForm(TFormTextReplaceDialog, Result);
   Result.Caption:=Format(RusEng('Замена в "%s"','Replace in "%s"'),[ExtractFileName(aEditor.Caption)]);
   apFlags:=Result.ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then Result.LocateToCenterOfScreen;
   Result.myEditor:=aEditor;
  end;
 except
  on E:Exception do BugReport(E,nil,'NewFormTextReplaceDialog');
 end;
end;

procedure TFormTextReplaceDialog.FormCreate(Sender: TObject);
begin
 myEditor:=nil;
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 SmartUpdate(BitBtnFind,RusEng('Найти далее','Find next'));
 SmartUpdate(BitBtnReplace,RusEng('Заменить','Replace'));
 SmartUpdate(BitBtnReplaceAll,RusEng('Заменить все','Replace all'));
 SmartUpdate(BitBtnClose,RusEng('Закрыть','Close'));
 SmartUpdate(LabelFindText,RusEng('Текст, который надо найти:','Text to find:'));
 SmartUpdate(LabelReplaceText,RusEng('Текст, на который заменить:','Text to replace:'));
 SmartUpdate(CheckBoxWholeWord,RusEng('Искать целое слово','Search whole word'));
 SmartUpdate(CheckBoxMatchCase,RusEng('Учитывать регистр','Match character case'));
 SmartUpdate(CheckBoxPrompt,RusEng('Запрос на замену','Prompt on replace'));
end;

procedure TFormTextReplaceDialog.FormClose(Sender: TObject; var CloseAction: TCloseAction);
begin
 if ValidEditor then CloseAction:=caHide else CloseAction:=caFree;
end;

function TFormTextReplaceDialog.ValidEditor:Boolean;
begin
 if Assigned(Self)
 then Result:=SdiMan.IsChild(myEditor,sf_SdiTextEdit)
 else Result:=false;
end;

procedure TFormTextReplaceDialog.FormActivate(Sender: TObject);
begin
 if Ok then
 try
  ActiveControl:=EditFindText;
 except
  on E:Exception do BugReport(E,Self,'FormActivate');
 end;
end;

procedure TFormTextReplaceDialog.BitBtnFindClick(Sender: TObject);
begin
 if Ok then
 try
  if ValidEditor then
  if (EditFindText.Text<>'') then
  myEditor.FindText(EditFindText.Text,
                    fr_WholeWord * ord(CheckBoxWholeWord.Checked) +
                    fr_MatchCase * ord(CheckBoxMatchCase.Checked));
 except
  on E:Exception do BugReport(E,Self,'BitBtnFindClick');
 end;
end;

procedure TFormTextReplaceDialog.BitBtnReplaceClick(Sender: TObject);
begin
 if Ok then
 try
  if ValidEditor then
  if (EditFindText.Text<>'') then
  myEditor.ReplaceText(EditFindText.Text,EditReplaceText.Text,
                       fr_WholeWord * ord(CheckBoxWholeWord.Checked) +
                       fr_MatchCase * ord(CheckBoxMatchCase.Checked)+
                       fr_PromptOnReplace * ord(CheckBoxPrompt.Checked) +
                       fr_Replace);
 except
  on E:Exception do BugReport(E,Self,'BitBtnReplaceClick');
 end;
end;

procedure TFormTextReplaceDialog.BitBtnReplaceAllClick(Sender: TObject);
begin
 if Ok then
 try
  if ValidEditor then
  if (EditFindText.Text<>'') then
  myEditor.ReplaceText(EditFindText.Text,EditReplaceText.Text,
                       fr_WholeWord * ord(CheckBoxWholeWord.Checked) +
                       fr_MatchCase * ord(CheckBoxMatchCase.Checked)+
                       fr_PromptOnReplace * ord(CheckBoxPrompt.Checked) +
                       fr_ReplaceAll);
 except
  on E:Exception do BugReport(E,Self,'BitBtnReplaceAllClick');
 end;
end;

procedure TFormTextReplaceDialog.BitBtnCloseClick(Sender: TObject);
begin
 if Ok then Hide;
end;

procedure TFormTextReplaceDialog.FormShow(Sender: TObject);
begin
 if Ok then SelectWordUnderCursor;
end;

procedure TFormTextReplaceDialog.SelectWordUnderCursor;
var sw:WideString;
begin
 if Ok then
 try
  if ValidEditor then begin
   sw:=GetWordUnderCursor(myEditor.PerformWideText,myEditor.PerformSelStart);
   if (sw<>'') then EditFindText.Text:=WideToStr(sw);
  end;
 except
  on E:Exception do BugReport(E,Self,'SelectWordUnderCursor');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_textreplacedialog;
begin
end;

procedure Free_form_textreplacedialog;
begin
end;

initialization

 Init_form_textreplacedialog;

finalization

 Free_form_textreplacedialog;

end.

//////////////
// END OF FILE
//////////////

