////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Tab Window for data (curve/tag) tables.                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231118 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_tabwindow; // Form Tab Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Grids,
 lcltype, lclintf,
 Form_ListBoxSelection, Form_CrwDaqSysChild,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo, _crw_ef,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut, _crw_dynar,
 _crw_snd, _crw_guard, _crw_sort, _crw_curves, _crw_daqtags,
 _crw_appforms, _crw_apptools, _crw_apputils, _crw_colors, ptembed;

const
  MaxTabWindowRows = 1024*4;
  MaxTabWindowLeng = 127;

type
  TFormTabWindow = class;
  TTabWindowItem = class(TMasterObject)
  protected
   Curve   : TCurve;
   Tag     : Integer;
   Title   : String[MaxTabWindowLeng];
   Value   : String[MaxTabWindowLeng];
  public
   constructor Create(aCurve:TCurve; aTag:Integer; const aTitle:LongString);
   destructor  Destroy; override;
  end;
  TTabWindowItemList = class(TObjectStorage)
  private
    function   GetItem(i:Integer):TTabWindowItem;
    procedure  SetItem(i:Integer; aItem:TTabWindowItem);
  public
    property   Item[i:Integer]:TTabWindowItem read GetItem write SetItem; default;
  public
    function FindCurve(aName:LongString):TCurve;
    function FindTag(aName:LongString):Integer;
  end;
  TFormTabWindow = class(TFormCrwDaqSysChild)
    StringGridItems: TStringGrid;
    FontDialog: TFontDialog;
    ActionEditFont: TAction;
    ActionEditFormat: TAction;
    ActionEditCopy: TAction;
    MenuEdit: TMenuItem;
    MenuEditFont: TMenuItem;
    MenuEditFormat: TMenuItem;
    MenuEditCopy: TMenuItem;
    MainPopupMenuEditFont: TMenuItem;
    MainPopupMenuEditFormat: TMenuItem;
    MainPopupMenuEditCopy: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ActionEditFontExecute(Sender: TObject);
    procedure ActionEditFormatExecute(Sender: TObject);
    procedure ActionEditCopyExecute(Sender: TObject);
    procedure StringGridItemsDblClick(Sender: TObject);
  private
    { Private declarations }
    myItems      : TTabWindowItemList;
    myLedWidth   : Integer;
    myLedDigit   : Integer;
    myLedFmtCh   : Char; // f,g,e format
    myLedFormat  : packed array[1..3] of String[31];
    myPadding    : TPoint;
    myResizeNum  : Integer;
    procedure UpdateFormat;
    function  GetItems:TTabWindowItemList;
    function  GetLedWidth:Integer;
    procedure SetLedWidth(aWidth:Integer);
    function  GetLedDigit:Integer;
    procedure SetLedDigit(aDigit:Integer);
    function  GetLedFmtCh:Char;
    procedure SetLedFmtCh(aCh:Char);
    function  GetValue(i:Integer):LongString;
  public
    { Public declarations }
    property  Items    : TTabWindowItemList read GetItems;
    property  LedWidth : Integer read GetLedWidth write SetLedWidth;
    property  LedDigit : Integer read GetLedDigit write SetLedDigit;
    property  LedFmtCh : Char    read GetLedFmtCh write SetLedFmtCh;
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure DrawView; override;
    procedure StartMonitoring;
    procedure StopMonitoring;
    procedure Monitoring;
    procedure AddItem(aCurve:TCurve; aTag:Integer; const aCaption:LongString; aUpdate:Boolean=false);
    procedure Config(const IniFile,Section:LongString);
    function  GetColor(i:Integer):TColor;
    function  GetCellsAsText:LongString;
    procedure ShouldResize(n:Integer=1);
    procedure ResetColors;
    procedure ResetFont;
    procedure UpdateGrid;
    procedure UpdateSize;
   end;
  TTabWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormTabWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormTabWindow);
  public
    property   Window[i:Integer]:TFormTabWindow read GetWindow write SetWindow; default;
  end;

const
  FullTabWindowList : TTabWindowList = nil;
  TabWindowsMonitor : TTabWindowList = nil;

function  NewTabWindow(const aCaption:LongString):TFormTabWindow;
procedure Kill(var TheObject:TFormTabWindow); overload;
function  ActiveTabWindow:TFormTabWindow;

function  NewTabWindowList(aOwnsObjects : Boolean = true;
                           aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                           aStep        : LongInt = DefaultTObjectStorageStep
                                      ) : TTabWindowList;
procedure Kill(var TheObject:TTabWindowList); overload;

 {
 ********************
 TableWindowsProfiler
 ********************
 }
procedure TableWindowsProfiler_Clear;
procedure TableWindowsProfiler_Start;
procedure TableWindowsProfiler_Stop;
procedure TableWindowsProfiler_Poll;

var
 TableWindowsProfiler : record
  Curr, Prev, Rate    : record
   TickCount          : Int64;
   DrawView           : Int64;
   MonitorCall        : Int64;
   MonitorDraw        : Int64;
  end;
  RateFactor          : Double;
 end;

implementation

{$R *.lfm}

uses
 Form_TabWindowFormatDialog;

 {
 ***********
 TTabWinItem
 ***********
 }
constructor TTabWindowItem.Create(aCurve:TCurve; aTag:Integer; const aTitle:LongString);
begin
 inherited Create;
 Curve:=aCurve;
 Tag:=aTag;
 Title:=Trim(aTitle);
 if (Title='') and Assigned(aCurve) then Title:=Trim(aCurve.Name);
 if (Title='') and (TypeTag(aTag)>0) then Title:=Trim(NameTag(aTag));
 Value:='';
end;

destructor  TTabWindowItem.Destroy;
begin
 Title:='';
 Value:='';
 inherited Destroy;
end;

function   TTabWindowItemList.GetItem(i:Integer):TTabWindowItem;
begin
 if Assigned(Self)
 then Result:=TTabWindowItem(Items[i])
 else Result:=nil;
end;

procedure  TTabWindowItemList.SetItem(i:Integer; aItem:TTabWindowItem);
begin
 if Assigned(Self) then Items[i]:=aItem;
end;

function TTabWindowItemList.FindCurve(aName:LongString):TCurve;
var i:Integer;
begin
 Result:=nil;
 if Assigned(Self) then
 if IsNonEmptyStr(aName) then
 try
  for i:=0 to Count-1 do
  if Assigned(Item[i]) then
  if Assigned(Item[i].Curve) then
  if IsSameText(aName,Item[i].Curve.Name)
  then Exit(Item[i].Curve);
 except
  on E:Exception do BugReport(E,nil,'FindCurve');
 end;
end;

function TTabWindowItemList.FindTag(aName:LongString):Integer;
var i:Integer;
begin
 Result:=0;
 if Assigned(Self) then
 if IsNonEmptyStr(aName) then
 try
  for i:=0 to Count-1 do
  if Assigned(Item[i]) then
  if TypeTag(Item[i].Tag)>0 then
  if IsSameText(aName,NameTag(Item[i].Tag))
  then Exit(Item[i].Tag);
 except
  on E:Exception do BugReport(E,nil,'FindTag');
 end;
end;

 {
 ********************
 TableWindowsProfiler
 ********************
 }
procedure TableWindowsProfiler_Clear;
begin
 FillChar(TableWindowsProfiler, SizeOf(TableWindowsProfiler), 0);
end;

procedure TableWindowsProfiler_Start;
begin
 TableWindowsProfiler_Clear;
 SecondActions.Add(TableWindowsProfiler_Poll);
end;

procedure TableWindowsProfiler_Stop;
begin
 SecondActions.Remove(TableWindowsProfiler_Poll);
 TableWindowsProfiler_Clear;
end;

procedure TableWindowsProfiler_Poll;
begin
 with TableWindowsProfiler do begin
  Curr.TickCount     := GetTickCount64;
  Rate.TickCount     := Curr.TickCount    - Prev.TickCount;
  RateFactor:=IfThen(Rate.TickCount=0,0.0,1000.0/Rate.TickCount);
  Rate.DrawView      := Curr.DrawView     - Prev.DrawView;
  Rate.MonitorCall   := Curr.MonitorCall  - Prev.MonitorCall;
  Rate.MonitorDraw   := Curr.MonitorDraw  - Prev.MonitorDraw;
  Prev:=Curr;
 end;
end;

 {
 *****************************
 TTabWindowList implementation
 *****************************
 }
function   TTabWindowList.GetWindow(i:Integer):TFormTabWindow;
begin
 Result:=TFormTabWindow(Items[i]);
end;

procedure  TTabWindowList.SetWindow(i:Integer; aWindow:TFormTabWindow);
begin
 Items[i]:=aWindow;
end;

function  NewTabWindowList(aOwnsObjects : Boolean = true;
                           aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                           aStep        : LongInt = DefaultTObjectStorageStep
                                      ) : TTabWindowList;
begin
 Result:=nil;
 try
  Result:=TTabWindowList.Create(aOwnsObjects,aCapacity,aStep);
 except
  on E:Exception do BugReport(E,nil,'NewTabWindowList');
 end;
end;

procedure Kill(var TheObject:TTabWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

 {
 *******************************************************************************
 General purpose routines
 *******************************************************************************
 }
function NewTabWindow(const aCaption:LongString):TFormTabWindow;
begin
 Result:=nil;
 try
  Application.CreateForm(TFormTabWindow, Result);
  if Result.Ok then begin
   Result.Caption:=aCaption;
   Result.ShouldResize;
   Result.UpdateSize;
  end else Kill(Result);
 except
  on E:Exception do BugReport(E,nil,'NewTabWindow');
 end;
end;

procedure Kill(var TheObject:TFormTabWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

function ActiveTabWindow:TFormTabWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if SdiMan.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormTabWindow) then Result:=TFormTabWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'TFormTabWindow');
 end;
end;

 {
 *******************************************************************************
 TFormTabWindow implementation
 *******************************************************************************
 }
procedure TFormTabWindow.ShouldResize(n:Integer=1);
begin
 if Assigned(Self) and (n>0) then Inc(myResizeNum,n);
end;

procedure TFormTabWindow.ResetColors;
begin
 if Assigned(Self) then begin
  StringGridItems.FixedColor:=clBtnFace;
  StringGridItems.Color:=clBtnFace;
  Font.Color:=StandardFont.Color;
 end;
end;

procedure TFormTabWindow.ResetFont;
begin
 if Assigned(Self) then begin
  SetStandardFont(Self);
  ShouldResize;
 end;
end;

procedure TFormTabWindow.UpdateGrid;
var i:Integer;
begin
 if Ok then
 try
  StringGridItems.FixedCols:=1;
  StringGridItems.FixedRows:=0;
  if (myItems.Count>0) then begin
   StringGridItems.ColCount:=2;
   StringGridItems.RowCount:=myItems.Count;
   for i:=0 to myItems.Count-1 do if myItems[i].Ok then begin
    myItems[i].Value:=GetValue(i);
    if i>=StringGridItems.RowCount then continue;
    StringGridItems.Cells[0,i]:=myItems[i].Title;
    StringGridItems.Cells[1,i]:=myItems[i].Value;
   end;
  end else begin
   StringGridItems.ColCount:=2;
   StringGridItems.RowCount:=1;
   StringGridItems.Cells[0,0]:='';
   StringGridItems.Cells[1,0]:='';
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateGrid');
 end;
end;

procedure TFormTabWindow.UpdateFormat;
var w,d:Integer; f:Char;
begin
 if Ok then
 try
  w:=Max(2,Min(MaxTabWindowLeng,abs(LedWidth)))*Sgn(LedWidth);
  d:=Max(0,Min(MaxTabWindowLeng,abs(LedDigit)));
  f:=LedFmtCh; if (Pos(f,'fgeFGE')=0) then f:='f';
  if (f='f') and (LedDigit<0) then f:='g';
  myLedFormat[1]:= Format('%s%d.%d%s', ['%',w,0,'d']);
  myLedFormat[2]:= Format('%s%d.%d%s', ['%',w,d,f]);
  myLedFormat[3]:= Format('%s%d.%d%s', ['%',w,abs(w),'s']);
  _crw_str.Format(myLedFormat[1]+' '+myLedFormat[2]+' '+myLedFormat[3], [1, pi, 'Ok'], True);
 except
  on E:Exception do begin
   BugReport(E,Self,'UpdateFormat');
   myLedFormat[1]:='%d'; // Assign
   myLedFormat[2]:='%g'; // fallback
   myLedFormat[3]:='%s'; // formats
  end;
 end;
end;

function TFormTabWindow.GetItems:TTabWindowItemList;
begin
 if Assigned(Self) then Result:=myItems else Result:=nil;
end;

function TFormTabWindow.GetLedWidth:Integer;
begin
 if Ok then Result:=myLedWidth else Result:=0;
end;

procedure TFormTabWindow.SetLedWidth(aWidth:Integer);
begin
 if Ok then begin
  myLedWidth:=Max(2,Min(MaxTabWindowLeng,abs(aWidth)))*Sgn(aWidth);
  UpdateFormat;
  ShouldResize;
  UpdateSize;
 end;
end;

function TFormTabWindow.GetLedDigit:Integer;
begin
 if Ok then Result:=myLedDigit else Result:=0;
end;

procedure TFormTabWindow.SetLedDigit(aDigit:Integer);
begin
 if Ok then begin
  myLedDigit:=aDigit;
  UpdateFormat;
  ShouldResize;
  UpdateSize;
 end;
end;

function  TFormTabWindow.GetLedFmtCh:Char;
begin
 if Ok then Result:=LoCase(myLedFmtCh) else Result:='f';
end;

procedure TFormTabWindow.SetLedFmtCh(aCh:Char);
begin
 if Ok and (Pos(aCh,'fgeFGE')>0) then begin
  myLedFmtCh:=LoCase(aCh);
  UpdateFormat;
  ShouldResize;
  UpdateSize;
 end;
end;

function TFormTabWindow.GetValue(i:Integer):LongString;
begin
 Result:='';
 if Ok then
 if (i>=0) and (i<myItems.Count) then with myItems[i] do if Ok then
 try
  if Assigned(Curve)
  then Result:=Format(myLedFormat[2],[Curve.LastPoint.Y]) else
  case TypeTag(Tag) of
   1 : Result:=Format(myLedFormat[1],[iGetTag(Tag)]);
   2 : Result:=Format(myLedFormat[2],[rGetTag(Tag)]);
   3 : Result:=Format(myLedFormat[3],[sGetTag(Tag)]);
  end;
  if Length(Result)>MaxTabWindowLeng then
  SetLength(Result,Min(Length(Result),MaxTabWindowLeng));
 except
  on E:Exception do begin
   BugReport(E,Self,'GetValue');
   Result:='???';
  end;
 end;
end;

function TFormTabWindow.GetColor(i:Integer):TColor;
begin
 Result:=clBlack;
 if Ok then
 if (i>=0) and (i<myItems.Count) then with myItems[i] do if Ok then begin
  if Assigned(Curve)
  then Result:=Curve.Color
  else Result:=Font.Color;
 end;
end;

function TFormTabWindow.GetCellsAsText:LongString;
var i:Integer; List:TStringList;
begin
 Result:='';
 if Ok then
 try
  List:=TStringList.Create;
  try
   if StringGridItems.ColCount>1 then
   for i:=0 to StringGridItems.RowCount-1 do
   List.Add(StringGridItems.Cells[0,i]+ASCII_HT+StringGridItems.Cells[1,i]);
   Result:=List.Text;
  finally
   List.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'GetCellsAsText');
 end;
end;

procedure TFormTabWindow.UpdateCommands;
var Exposed:Boolean;
begin
 try
  inherited UpdateCommands;
  Exposed:=FormIsExposed(Self);
  ActionFileSave.Enabled:=false;
  ActionFileSaveAs.Enabled:=false;
  ActionFilePrint.Enabled:=Exposed;
  ActionEditFont.Enabled:=Exposed;
  ActionEditFormat.Enabled:=Exposed;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormTabWindow.DrawView;
begin
 if Ok then
 try
  if IsFormViewable then begin
   Inc(TableWindowsProfiler.Curr.DrawView);
   DebugLogReport_DrawView;
   inherited DrawView;
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawView');
 end;
end;

procedure TabWindowMonitor(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if aObject is TFormTabWindow then
 with aObject as TFormTabWindow do if MonitorEvent then Monitoring;
end;

procedure TabWindowsMonitoring;
begin
 TabWindowsMonitor.ForEach(TabWindowMonitor,nil);
end;

procedure TFormTabWindow.StartMonitoring;
begin
 if Ok and TabWindowsMonitor.Ok then begin
  if (TabWindowsMonitor.IndexOf(Self)<0) then begin
   TabWindowsMonitor.Add(Self);
   Tick55Actions.Add(TabWindowsMonitoring);
  end;
 end;
end;

procedure TFormTabWindow.StopMonitoring;
begin
 if Ok and TabWindowsMonitor.Ok then begin
  if (TabWindowsMonitor.IndexOf(Self)>=0) then begin
   TabWindowsMonitor.Remove(Self);
   if TabWindowsMonitor.Count=0 then
   Tick55Actions.Remove(TabWindowsMonitoring);
  end;
 end;
end;

procedure TFormTabWindow.Monitoring;
var i:Integer; s:LongString;
begin
 if Ok then
 try
  Inc(TableWindowsProfiler.Curr.MonitorCall);
  if not IsFormExposed then Exit;
  Inc(TableWindowsProfiler.Curr.MonitorDraw);
  if (myResizeNum<0) then myResizeNum:=0;
  if (myResizeNum>0) then begin
   Dec(myResizeNum);
   UpdateSize;
  end;
  if StringGridItems.ColCount>1 then
  for i:=0 to myItems.Count-1 do with myItems[i] do if Ok then begin
   s:=GetValue(i);
   if (s<>Value) then begin
    Value:=s;
    if i<StringGridItems.RowCount
    then StringGridItems.Cells[1,i]:=Value;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'Monitoring');
 end;
end;

procedure TFormTabWindow.AddItem(aCurve:TCurve; aTag:Integer; const aCaption:LongString; aUpdate:Boolean=false);
begin
 try
  if Ok and (Assigned(aCurve) xor (aTag>0)) and (myItems.Count<MaxTabWindowRows) then begin
   myItems.Add(TTabWindowItem.Create(aCurve,aTag,aCaption));
   if aUpdate then begin UpdateGrid; ShouldResize; end;
  end;
 except
  on E:Exception do BugReport(E,Self,'AddItem');
 end;
end;

procedure TFormTabWindow.Config(const IniFile,Section:LongString);
var
 afont  : packed record
  name  : PureString;
  sizex : LongInt;
  sizey : LongInt;
 end;
 afmt   : packed record
  width : LongInt;
  digit : LongInt;
  fmtch : PureString;
 end;
 acolors:LongString;
 FontPar:TFontParams;
begin
 if Ok then
 try
  //
  // Font = Name:PT_Mono\Color:Navy\Size:12\Style:[Bold]
  //
  afont.name:=Font.Name;
  afont.sizex:=Canvas.TextWidth('0');
  afont.sizey:=abs(Font.Height);
  ReadIniFileRecord(IniFile,Section,'Font%a;%d;%d',afont);
  if (Pos(':',afont.name)>0) then begin
   // New style Font = Name:PT_Mono\Color:Navy\Size:12\Style:[Bold]
   if ReadBufferedFont(FontPar,afont.name,true,StandardFont)
   then RestoreFont(Font,FontPar);
  end else begin
   // Old style Font = FixedSys 8 13
   Font.Name:=Trim(ReplaceString(afont.name,'_',' '));
   if (Screen.Fonts.IndexOf(Font.Name)<0) then Font.Name:=StandardFont.Name;
   Font.Height:=-Max(abs(afont.sizey),8);
  end;
  //
  // Format = width digit fmtchar ; fmtchar=(f,g,e)
  //
  afmt.width:=LedWidth;
  afmt.digit:=LedDigit;
  afmt.fmtch:=LedFmtCh;
  ReadIniFileRecord(IniFile,Section,'Format%d;%d;%a',afmt);
  LedWidth:=afmt.width;
  LedDigit:=afmt.digit;
  LedFmtCh:=LoCase(StrFetch(afmt.fmtch,1));
  //
  // Colors = Silver, White - colors of StringGridItems FixedColor, Color
  //
  acolors:='* * *';
  if ReadIniFileString(IniFile,Section,'Colors%s',acolors) then begin
   StringGridItems.FixedColor:=StringToColor(ExtractWord(1,acolors,ScanSpaces),StringGridItems.FixedColor);
   StringGridItems.Color:=StringToColor(ExtractWord(2,acolors,ScanSpaces),StringGridItems.Color);
   Font.Color:=StringToColor(ExtractWord(3,acolors,ScanSpaces),Font.Color);
  end;
  UpdateFormat;
  ShouldResize;
  UpdateSize;
 except
  on E:Exception do BugReport(E,Self,'Config');
 end;
end;

procedure TFormTabWindow.UpdateSize;
const xMin=16; yMin=8; xPad=8; yPad=8; xSpace=32; ySpace=32; xInner=150; yInner=150;
var i:Integer; p,p0,p1,pMax:TPoint; s:LongString;
begin
 if Ok then
 try
  LockDraw;
  try
   UpdateGrid;
   p0:=Point(0,0);
   if (StringGridItems.ColCount>0) then begin
    for i:=0 to StringGridItems.RowCount-1 do begin
     s:=StringGridItems.Cells[0,i];
     s:=WideToStr(Copy(StrToWide(s),1,MaxTabWindowLeng));
     p0.x:=Max(p0.x,StringGridItems.Canvas.TextWidth(s));
     p0.y:=Max(p0.y,StringGridItems.Canvas.TextHeight(s));
    end;
    StringGridItems.ColWidths[0]:=Max(xMin,p0.x+xPad);
   end;
   p1:=Point(0,0);
   if (StringGridItems.ColCount>1) then begin
    for i:=0 to StringGridItems.RowCount-1 do begin
     s:=StringGridItems.Cells[1,i];
     s:=WideToStr(Copy(StrToWide(s),1,MaxTabWindowLeng));
     p1.x:=Max(p1.x,StringGridItems.Canvas.TextWidth(s));
     p1.y:=Max(p1.y,StringGridItems.Canvas.TextHeight(s));
    end;
    StringGridItems.ColWidths[1]:=Max(xMin,p1.x+xPad);
   end;
   StringGridItems.DefaultRowHeight:=Max(yMin,Max(p0.y,p1.y)+yPad);
   p.x:=myPadding.x+xSpace+p0.x+p1.x;
   p.y:=myPadding.y+ySpace+(StringGridItems.DefaultRowHeight+StringGridItems.GridLineWidth)*StringGridItems.RowCount;
   pMax:=Point(Screen.DesktopWidth-xInner,Screen.DesktopHeight-yInner);
   if Assigned(Application) and Assigned(Application.MainForm) then
   if (Application.MainForm.FormStyle=fsMdiForm) and SdiMan.IsMdiMode then
   pMax:=Point(Application.MainForm.ClientWidth-xInner,Application.MainForm.ClientHeight-yInner);
   if (p.y>=pMax.y) then p.x:=p.x+GetSystemMetrics(SM_CYVSCROLL);
   Width:=Max(100,Min(p.x,pMax.x));
   Height:=Max(60,Min(p.y,pMax.y));
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateSize');
 end;
end;

procedure TFormTabWindow.FormCreate(Sender: TObject);
begin
 inherited;
 myItems:=TTabWindowItemList.Create(true);
 myItems.Master:=@myItems;
 DoubleBuffered:=true;
 ToolBar.Hide;
 StatusBar.Hide;
 try
  LockDraw;
  myLedWidth:=10;
  myLedDigit:=3;
  myLedFmtCh:='f';
  UpdateFormat;
  ResetColors;
  Width:=100;
  Height:=150;
 finally
  UnlockDraw;
 end;
 UpdateMenu(MenuEdit,
            RusEng('Правка','Edit')+MenuRightSpace,
            RusEng('Меню для редактирования.','Editor operations.'),
            0);
 UpdateMenu(MenuEditFont,
            RusEng('Фонт...','Font...'),
            RusEng('Диалог выбора фонта.','Choose font dialog.'),
            ShortCut(Word('F'),[ssAlt]));
 UpdateMenu(MenuEditFormat,
            RusEng('Цвет и формат','Color and font'),
            RusEng('Выбрать цвет и формат.','Choose color and format dialog.'),
            ShortCut(Word('C'),[ssAlt]));
 UpdateMenu(MenuEditCopy,
            RusEng('Копировать в Буфер','Copy to Clipboard'),
            RusEng('Копировать текст в Буфер Обмена.','Copy as Text to Clipboard.'),
            ShortCut(Word('C'),[ssCtrl]));
 MainPopupMenuEditFont.Caption:=MenuEditFont.Caption;     MainPopupMenuEditFont.Hint:=MenuEditFont.Hint;
 MainPopupMenuEditFormat.Caption:=MenuEditFormat.Caption; MainPopupMenuEditFormat.Hint:=MenuEditFormat.Hint;
 MainPopupMenuEditCopy.Caption:=MenuEditCopy.Caption;     MainPopupMenuEditCopy.Hint:=MenuEditCopy.Hint;
 myPadding:=Point(Width-ClientWidth,Height-ClientHeight);
 ShouldResize;
end;

procedure TFormTabWindow.FormDestroy(Sender: TObject);
begin
 FreeAndNil(myItems);
 inherited;
end;

procedure TFormTabWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullTabWindowList.Add(Self);
 AddonSdiFlags(sf_SdiTabWin);
end;

procedure TFormTabWindow.BeforeDestruction;
begin
 StopMonitoring;
 FullTabWindowList.Remove(Self);
 inherited BeforeDestruction;
end;

procedure TFormTabWindow.ActionEditFontExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditFont)<0 then Exit;
 if Ok then
 try
  inherited;
  ScreenFontsUpdate;
  FontDialog.Font.Assign(Font);
  if FontDialog.Execute then begin
   Font.Assign(FontDialog.Font);
   ShouldResize;
   UpdateSize; 
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionEditFontExecute');
 end;
end;

procedure TFormTabWindow.ActionEditFormatExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditFormat)<0 then Exit;
 if Ok then
 try
  inherited;
  FormTabWindowFormatDialogExecute(Self);
  ShouldResize;
 except
  on E:Exception do BugReport(E,Self,'ActionEditFormatExecute');
 end;
end;

procedure TFormTabWindow.ActionEditCopyExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionEditCopy)<0 then Exit;
 if Ok then
 try
  inherited;
  Clipboard.AsText:=GetCellsAsText;
 except
  on E:Exception do BugReport(E,Self,'ActionEditCopyExecute');
 end;
end;

procedure TFormTabWindow.StringGridItemsDblClick(Sender: TObject);
begin
 inherited;
 ShouldResize;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_tabwindow;
begin
 TableWindowsProfiler_Clear;
 FullTabWindowList:=NewTabWindowList(false);
 FullTabWindowList.Master:=@FullTabWindowList;
 TabWindowsMonitor:=NewTabWindowList(false);
 TabWindowsMonitor.Master:=@TabWindowsMonitor;
end;

procedure Free_form_tabwindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullTabWindowList.Count', FullTabWindowList.Count]));
 ResourceLeakageLog(Format('%-60s = %d',['TabWindowsMonitor.Count', TabWindowsMonitor.Count]));
 Kill(FullTabWindowList);
 Kill(TabWindowsMonitor);
end;

initialization

 Init_form_tabwindow;

finalization

 Free_form_tabwindow;

end.

//////////////
// END OF FILE
//////////////

