////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Surf Window Clip Dialog.                                              //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231125 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_surfwindowclipdialog; // Form Surf Window Clip Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 lcltype, lclintf,
 Form_SurfWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSurfToolsWindowClipDialog = class(TMasterForm)
    PanelImage: TPanel;
    Image: TImage;
    ScrollBarGamma: TScrollBar;
    PanelInfo: TPanel;
    LabelX1: TLabel;
    LabelY1: TLabel;
    LabelX2: TLabel;
    LabelY2: TLabel;
    PanelButtons: TPanel;
    ButtonClose: TButton;
    ButtonMaxArea: TButton;
    CheckBoxInvertX: TCheckBox;
    CheckBoxInvertY: TCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ImageMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ImageMouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer);
    procedure ImageMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure ButtonMaxAreaClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure OnChangeImage(Sender: TObject);
  private
    mySel   : TXorSelector;
    myLim   : TRect2D;
    myNum   : TPoint2I;
    myClip  : TRect2D;
    function  GetClip:TRect2D;
    procedure SetClip(const aClip:TRect2D);
    function  LocalToIndex(P:TPoint2I):TPoint2I;
    function  LocalToReal(P:TPoint2I):TPoint2D;
    procedure InitBitmap;
  private
    myActiveSurfWindow : TFormSurfWindow;
  public
    property  Clip:TRect2D read GetClip write SetClip;
 end;

procedure FormSurfToolsWindowClipDialogExecute(aForm:TForm; const aParams:LongString='');

implementation

{$R *.lfm}

uses
 Form_SurfWindowStyleDialog;

const
  FormSurfToolsWindowClipDialog : TFormSurfToolsWindowClipDialog = nil;

function ActiveSurfWindow(aForm:TForm=nil):TFormSurfWindow;
begin
 Result:=nil;
 if Assigned(FormSurfToolsWindowClipDialog) and not Assigned(aForm)
 then aForm:=FormSurfToolsWindowClipDialog.myActiveSurfWindow;
 if FormExists(aForm) and (aForm is TFormSurfWindow)
 then Result:=TFormSurfWindow(aForm);
 if not Assigned(Result) then Result:=Form_SurfWindow.ActiveSurfWindow;
end;

procedure FormSurfToolsWindowClipDialogExecute(aForm:TForm; const aParams:LongString='');
var  apFlags:Integer;
begin
 if Assigned(aForm) then
 if ActiveSurfWindow(aForm).Ok then
 if CanShowModal(FormSurfToolsWindowClipDialog) then
 try
  if not Assigned(FormSurfToolsWindowClipDialog)  then begin
   Application.CreateForm(TFormSurfToolsWindowClipDialog, FormSurfToolsWindowClipDialog);
   FormSurfToolsWindowClipDialog.Master:=@FormSurfToolsWindowClipDialog;
  end;
  if Assigned(FormSurfToolsWindowClipDialog) then begin
   apFlags:=FormSurfToolsWindowClipDialog.ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then LocateFormToCenterOfScreen(FormSurfToolsWindowClipDialog);
   FormSurfToolsWindowClipDialog.myActiveSurfWindow:=ActiveSurfWindow(aForm);
   FormSurfToolsWindowClipDialog.InitBitmap;
   if mrVoice(FormSurfToolsWindowClipDialog.ShowModal)=mrOk then begin
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'FormSurfToolsWindowClipDialogExecute'); 
 end;
end;

procedure TFormSurfToolsWindowClipDialog.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Диалог выбора фрагмента поверхности','Visible surface region selection dialog');
 SmartUpdate(ButtonClose,RusEng('Закрыть','Close'));
 SmartUpdate(ButtonMaxArea,RusEng('Максимальные пределы','Set maximal visible area'));
 CheckBoxInvertX.Caption:=RusEng('Инверсия X','Invert X');
 CheckBoxInvertY.Caption:=RusEng('Инверсия Y','Invert Y');
 Image.Picture.Bitmap.Width:=Image.Width;
 Image.Picture.Bitmap.Height:=Image.Height;
 mySel:=NewXorSelector;
 myLim:=Rect2D(0,0,1,1);
 myNum:=Point2I(2,2);
 Clip:=myLim;
end;

procedure TFormSurfToolsWindowClipDialog.FormDestroy(Sender: TObject);
begin
 Kill(mySel);
end;

procedure TFormSurfToolsWindowClipDialog.FormResize(Sender: TObject);
begin
 InitBitmap;
end;

procedure TFormSurfToolsWindowClipDialog.ImageMouseDown(Sender: TObject;  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
 if Ok and mySel.Start(Image.Canvas,Point2I(x,y))
 then Clip:=Rect2D(LocalToReal(mySel.Selection.A),LocalToReal(mySel.Selection.B));
end;

procedure TFormSurfToolsWindowClipDialog.ImageMouseMove(Sender: TObject; Shift: TShiftState; X, Y: Integer);
begin
 if Ok and mySel.Replace(Image.Canvas,Point2I(x,y))
 then Clip:=Rect2D(LocalToReal(mySel.Selection.A), LocalToReal(mySel.Selection.B));
end;

procedure TFormSurfToolsWindowClipDialog.ImageMouseUp(Sender: TObject;  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
 if Ok and mySel.Stop(Image.Canvas) then
 try
  Clip:=Rect2D(LocalToReal(mySel.Selection.A), LocalToReal(mySel.Selection.B));
  if (abs(RectSizeX(mySel.Selection))+abs(RectSizeY(mySel.Selection))>2) and not RectIsEmpty(myClip)
  then ActiveSurfWindow.Clip:=Clip
  else Clip:=ActiveSurfWindow.Clip;
 except
  on E:Exception do BugReport(E,Self,'ImageMouseUp'); 
 end;
end;

procedure TFormSurfToolsWindowClipDialog.ButtonMaxAreaClick(Sender: TObject);
begin
 if Ok and ActiveSurfWindow.Ok then
 try
  ActiveSurfWindow.Clip:=ActiveSurfWindow.Limits;
  Clip:=ActiveSurfWindow.Clip;
 except
  on E:Exception do BugReport(E,Self,'ButtonMaxAreaClick');
 end;
end;

procedure TFormSurfToolsWindowClipDialog.InitBitmap;
const
 nx = 50;
 ny = 50;
var z1,z2,gamma:double; i,j:integer; Win:TFormSurfWindow; r:TRect2I;
 function ValueToColor(z,z1,z2:Double):TColor;
 var gray:Integer;
 begin
  if z2>z1 then gray:=round(power((z-z1)/(z2-z1),gamma)*$FF) and $FF else gray:=0;
  Result:=RGB(gray,gray,gray);
 end;
begin
 if Ok then
 try
  Image.Picture.Bitmap.Width:=Image.Width;
  Image.Picture.Bitmap.Height:=Image.Height;
  Win:=ActiveSurfWindow;
  if Win.Ok then begin
   myLim:=Win.Limits;
   myNum:=Win.NumPoints;
   Clip:=Win.Clip;
   with ScrollBarGamma do gamma:=power(10,2*Position/(Max-Min));
   z1:=Win[0,0];
   z2:=Win[0,0];
   for i:=0 to Win.NumPoints.X-1 do
   for j:=0 to Win.NumPoints.Y-1 do z1:=min(z1,Win[i,j]);
   for i:=0 to Win.NumPoints.X-1 do
   for j:=0 to Win.NumPoints.Y-1 do z2:=max(z2,Win[i,j]);
   for i:=0 to nx-1 do
   for j:=0 to ny-1 do begin
    r:=Rect2I(Round((i+0)*Image.Picture.Bitmap.Width/nx),
              Round((j+0)*Image.Picture.Bitmap.Height/ny),
              Round((i+1)*Image.Picture.Bitmap.Width/nx),
              Round((j+1)*Image.Picture.Bitmap.Height/ny));
    if not RectIsEmpty(r) then with LocalToIndex(r.a) do begin
     X:=Max(0,Min(myNum.X-1,X));
     Y:=Max(0,Min(myNum.Y-1,Y));
     DrawBar(Image.Picture.Bitmap.Canvas,r,ValueToColor(Win[x,y],z1,z2));
    end;
   end;
  end else begin
   myLim:=Rect2D(0,0,1,1);
   myNum:=Point2I(2,2);
   Clip:=myLim;
  end;
 except
  on E:Exception do BugReport(E,Self,'InitBitmap');
 end;
end;

function TFormSurfToolsWindowClipDialog.GetClip:TRect2D;
begin
 if Ok then Result:=myClip else Result:=Rect2D(0,0,0,0);
end;

procedure TFormSurfToolsWindowClipDialog.SetClip(const aClip:TRect2D);
begin
 if Ok then begin
  myClip:=RectValidate(aClip);
  SmartUpdate(LabelX1, Format('X1=%15.5g',[myClip.A.X]));
  SmartUpdate(LabelY1, Format('Y1=%15.5g',[myClip.A.Y]));
  SmartUpdate(LabelX2, Format('X2=%15.5g',[myClip.B.X]));
  SmartUpdate(LabelY2, Format('Y2=%15.5g',[myClip.B.Y]));
 end;
end;

function TFormSurfToolsWindowClipDialog.LocalToIndex(P:TPoint2I):TPoint2I;
begin
 if Ok then begin
  if CheckBoxInvertX.Checked then P.X:=Image.Picture.Bitmap.Width-1-P.X;
  if CheckBoxInvertY.Checked then P.Y:=Image.Picture.Bitmap.Height-1-P.Y;
  Result.x:=Round(P.X/(Image.Picture.Bitmap.Width-1)*(myNum.x-1));
  Result.y:=Round(P.Y/(Image.Picture.Bitmap.Height-1)*(myNum.y-1));
 end else Result:=Point2I(0,0);
end;

function TFormSurfToolsWindowClipDialog.LocalToReal(P:TPoint2I):TPoint2D;
begin
 if Ok then begin
  if CheckBoxInvertX.Checked then P.X:=Image.Picture.Bitmap.Width-1-P.X;
  if CheckBoxInvertY.Checked then P.Y:=Image.Picture.Bitmap.Height-1-P.Y;
  Result.x:=myLim.a.x+(myLim.b.x-myLim.a.x)*P.x/(Image.Picture.Bitmap.Width-1);
  Result.y:=myLim.a.y+(myLim.b.y-myLim.a.y)*P.y/(Image.Picture.Bitmap.Height-1);
 end else Result:=Point2D(0,0);
end;

procedure TFormSurfToolsWindowClipDialog.OnChangeImage(Sender: TObject);
begin
 if Ok then begin
  InitBitmap;
  Update;
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_surfwindowclipdialog;
begin
end;

procedure Free_form_surfwindowclipdialog;
begin
end;

initialization

 Init_form_surfwindowclipdialog;

finalization

 Free_form_surfwindowclipdialog;

end.

//////////////
// END OF FILE
//////////////

