////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr Zone Fit Zone Viewer.                                          //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231205 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrzonfitzonviewer; // Form Spectr Zone Fit Zone Viewer

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_SpectrZonFitWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_sort, _crw_ef, _crw_ee, _crw_zm,
 _crw_curves, _crw_riff, _crw_calib,
 _crw_daqsys, _crw_daqdev,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSpectrZonFitZonViewer = class(TMasterForm)
    LabelPeaks: TLabel;
    ListBoxPeaks: TListBox;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure ListBoxPeaksClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    Win : TFormSpectrZonFitWindow;
    procedure UpdateControls(Flags:Integer);
  end;

procedure OpenFormSpectrZonFitZonViewer(Win:TFormSpectrZonFitWindow);

implementation

{$R *.lfm}

procedure OpenFormSpectrZonFitZonViewer(Win:TFormSpectrZonFitWindow);
begin
 if (Win is TFormSpectrZonFitWindow) then begin
  Application.CreateForm(TFormSpectrZonFitZonViewer, Win.ZonFit.ZonViewer);
  if Win.ZonFit.ZonViewer.Ok then begin
   (Win.ZonFit.ZonViewer as TFormSpectrZonFitZonViewer).Win:=Win;
   Win.ZonFit.ZonViewer.Master:=@Win.ZonFit.ZonViewer;
   (Win.ZonFit.ZonViewer as TFormSpectrZonFitZonViewer).UpdateControls(ucUpdate+ucShow+ucHome);
  end;
 end;
end;

procedure TFormSpectrZonFitZonViewer.UpdateControls(Flags:Integer);
const FixStr:packed array[Boolean] of Char=(' ','*');
var i,n:Integer; ChFx,AmFx,FwFx:Char; Zone:TZone;
var Ch,Am,Fw,En,Ar,ChEr,FwEr,ArEr,Grnd:Double;
begin
 if Ok then
 if Win.Ok then
 try
  if HasFlags(Flags,ucUpdate) then begin
   Zone:=Win.ZonFit.Roi.Current;
   if Zone.Ok then begin
    Caption:=Format(RusEng('Пики зоны %d - [%s]','Peaks zone %d - [%s]'),[Win.ZonFit.Roi.IndexOfCurrent+1,Win.Caption]);
    LabelPeaks.Caption:=Format(' %2s',      ['#N'])+
                        Format(' %1s%1s%1s',['f','i','x'])+
                        Format(' %7s',      ['Chan'])+
                        Format(' %12s',     ['Ampl'])+
                        Format(' %7s',      ['FWHM'])+
                        Format(' %7s',      ['Energy'])+
                        Format(' %12s',     ['Area'])+
                        Format(' %7s',      ['ChanErr'])+
                        Format(' %7s',      ['AreaErr'])+
                        Format(' %7s',      ['FwhmErr'])+
                        Format(' %8s',      ['Ground']);
    n:=ListBoxPeaks.ItemIndex;
    ListBoxPeaks.Clear;
    for i:=0 to Zone.NumPeaks-1 do begin
     Ch:=Zone.CurrParam.Peak[i].Chan;
     Am:=Zone.CurrParam.Peak[i].Ampl;
     Fw:=Zone.CurrParam.Peak[i].FWHM;
     En:=Win.EnCalibr(Ch);
     Ar:=GaussPeakArea(Am,Fw);
     Grnd:=Zone.EvalGround(Ch,Zone.CurrParam,Zone.TempGrad);
     if (Zone.Covar=nil) then begin
      ArEr:=0;
      ChEr:=0;
      FwEr:=0;
     end else begin
      ChEr:=Zone.ChanError(i);
      ArEr:=Zone.AreaError(i);
      FwEr:=Zone.FWHMError(i);
     end;
     ChFx:=FixStr[Zone.CurrFixed.Peak[i].Chan];
     AmFx:=FixStr[Zone.CurrFixed.Peak[i].Ampl];
     FwFx:=FixStr[Zone.CurrFixed.Peak[i].FWHM];
     ListBoxPeaks.Items.Add(Format(' %2d',      [i+1])+
                            Format('|%1s%1s%1s',[ChFx,AmFx,FwFx])+
                            Format('|%7.1f',    [Ch])+
                            Format('|%12.1f',   [Am])+
                            Format('|%7.3f',    [Fw])+
                            Format('|%7.3f',    [En])+
                            Format('|%12.1f',   [Ar])+
                            Format('|%7.3f',    [ChEr])+
                            Format('|%7.1f',    [ArEr])+
                            Format('|%7.3f',    [FwEr])+
                            Format('|%8.1f',    [Grnd]));
    end;
    ListBoxPeaks.ItemIndex:=min(n,ListBoxPeaks.Items.Count-1);
   end else begin
    ListBoxPeaks.Clear;
    Caption:=Format(RusEng('Пики зоны %d - [%s]','Peaks zone %d - [%s]'),[0,Win.Caption]);
    LabelPeaks.Caption:=RusEng('Нет пиков','No peaks');
   end;
  end;
  if HasFlags(Flags,ucClick) then begin
   if Win.ZonFit.Roi.Current.Ok then begin
    i:=ListBoxPeaks.ItemIndex;
    if (i>=0) and (i<Win.ZonFit.Roi.Current.NumPeaks)
    then Win.Marker:=round(Win.ZonFit.Roi.Current.CurrParam.Peak[i].Chan);
   end;
  end;
  if HasFlags(Flags,ucMarker) then begin
   Zone:=Win.ZonFit.Roi.Current;
   if Zone.Ok then ListBoxPeaks.ItemIndex:=Zone.Nearest(Win.Marker);
  end;
  if HasFlags(Flags,ucHome) then begin
   Top:=0;
   Left:=250;
   Height:=120;
   Width:=Screen.Width-250;
  end;
  if HasFlags(Flags,ucShow) then begin
   Show;
   BringToFront;
   WindowState:=wsNormal;
  end;
  if HasFlags(Flags,ucHide) then begin
   Hide;
  end;
 except
  on E: Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrZonFitZonViewer.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
end;

procedure TFormSpectrZonFitZonViewer.FormClose(Sender: TObject;  var Action: TCloseAction);
begin
 if Ok and Win.Ok then Win.UpdateControls(ucHide);
end;

procedure TFormSpectrZonFitZonViewer.ListBoxPeaksClick(Sender: TObject);
begin
 UpdateControls(ucClick);
 UpdateControls(ucUpdate);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrzonfitzonviewer;
begin
end;

procedure Free_form_spectrzonfitzonviewer;
begin
end;

initialization

 Init_form_spectrzonfitzonviewer;

finalization

 Free_form_spectrzonfitzonviewer;

end.

//////////////
// END OF FILE
//////////////

