////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Spectr Pik Control.                                                   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231203 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_spectrpikcontrol; // Form Spectr Pik Control

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Grids,
 lcltype, lclintf,
 Form_SpectrPikWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_sort,
 _crw_ef, _crw_ee, _crw_zm, _crw_curves, _crw_riff,
 _crw_calib, _crw_daqsys, _crw_daqdev, _crw_peaksearch,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormSpectrPikControl = class(TMasterForm)
    PageControl: TPageControl;
    TabSheetPeakList: TTabSheet;
    TabSheetControls: TTabSheet;
    TabSheetPreset: TTabSheet;    
    StringGridPeaks: TStringGrid;
    ButtonSearch: TButton;
    ButtonClear: TButton;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    ButtonOpen: TButton;
    ButtonSave: TButton;
    ButtonPeakPrev: TButton;
    ButtonPeakNext: TButton;
    GroupBoxView: TGroupBox;
    CheckBoxViewPos: TCheckBox;
    CheckBoxViewForm: TCheckBox;
    CheckBoxViewGround: TCheckBox;
    CheckBoxViewSingle: TCheckBox;
    StringGridPreset: TStringGrid;
    procedure FormCreate(Sender: TObject);
    procedure ButtonSearchClick(Sender: TObject);
    procedure ButtonClearClick(Sender: TObject);
    procedure ButtonOpenClick(Sender: TObject);
    procedure ButtonSaveClick(Sender: TObject);
    procedure ButtonPeakPrevClick(Sender: TObject);
    procedure ButtonPeakNextClick(Sender: TObject);
    procedure CheckBoxViewPosClick(Sender: TObject);
    procedure CheckBoxViewFormClick(Sender: TObject);
    procedure CheckBoxViewGroundClick(Sender: TObject);
    procedure CheckBoxViewSingleClick(Sender: TObject);
    procedure StringGridPeaksClick(Sender: TObject);
    procedure StringGridPresetKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  private
    { Private declarations }
  public
    { Public declarations }
    Win : TFormSpectrPikWindow;
    procedure UpdateControls(Flags:Integer);
  end;

function NewFormSpectrPikControl(Win:TFormSpectrPikWindow):TFormSpectrPikControl;

implementation

{$R *.lfm}

function NewFormSpectrPikControl(Win:TFormSpectrPikWindow):TFormSpectrPikControl;
begin
 Result:=nil;
 if (Win is TFormSpectrPikWindow) then begin
  Application.CreateForm(TFormSpectrPikControl, Result);
  Result.Win:=Win;
  Result.Caption:=Format(RusEng('Поиск пиков - [%s]','Peak search - [%s]'),[Win.Caption]);
 end;
end;

procedure TFormSpectrPikControl.UpdateControls(Flags:Integer);
var P:TPeakInParamsArray absolute PeakInParams;
var i:Integer; R:TGridRect;
begin
 if Ok then
 if Win.Ok then
 try
  if HasFlags(Flags,1) then begin
   StringGridPeaks.DefaultColWidth:=StringGridPeaks.Canvas.TextWidth(Pad('',14,'M'));
   StringGridPeaks.ColWidths[0]:=StringGridPeaks.Canvas.TextWidth(Pad('',7,'M'));
   StringGridPeaks.ColCount:=12;
   StringGridPeaks.RowCount:=1+Win.Peak.NumPeaks;
   if (StringGridPeaks.ColCount>1) then StringGridPeaks.FixedCols:=1;
   if (StringGridPeaks.RowCount>1) then StringGridPeaks.FixedRows:=1;
   StringGridPeaks.Cells[0, 0]:='#N';
   StringGridPeaks.Cells[1, 0]:='Channel';
   StringGridPeaks.Cells[2, 0]:='Energy';
   StringGridPeaks.Cells[3, 0]:='Amplitude';
   StringGridPeaks.Cells[4, 0]:='FWHM';
   StringGridPeaks.Cells[5, 0]:='Area';
   StringGridPeaks.Cells[6, 0]:='Ground';
   StringGridPeaks.Cells[7, 0]:='LeftBound';
   StringGridPeaks.Cells[8, 0]:='LeftGround';
   StringGridPeaks.Cells[9, 0]:='RightBound';
   StringGridPeaks.Cells[10,0]:='RightGround';
   StringGridPeaks.Cells[11,0]:='StatError';
   for i:=0 to Win.Peak.NumPeaks-1 do begin
    StringGridPeaks.Cells[0, 1+i]:=Format('%d',[i]);
    StringGridPeaks.Cells[1, 1+i]:=Format('%.7g',[Win.Peak.Channel[i]]);
    StringGridPeaks.Cells[2, 1+i]:=Format('%.7g',[Win.Peak.Energy[i]]);
    StringGridPeaks.Cells[3, 1+i]:=Format('%.7g',[Win.Peak.Amplitude[i]]);
    StringGridPeaks.Cells[4, 1+i]:=Format('%.7g',[Win.Peak.FWHM[i]]);
    StringGridPeaks.Cells[5, 1+i]:=Format('%.7g',[Win.Peak.Area[i]]);
    StringGridPeaks.Cells[6, 1+i]:=Format('%.7g',[Win.Peak.Ground[i]]);
    StringGridPeaks.Cells[7, 1+i]:=Format('%.7g',[Win.Peak.LeftBound[i]]);
    StringGridPeaks.Cells[8, 1+i]:=Format('%.7g',[Win.Peak.LeftGround[i]]);
    StringGridPeaks.Cells[9, 1+i]:=Format('%.7g',[Win.Peak.RightBound[i]]);
    StringGridPeaks.Cells[10,1+i]:=Format('%.7g',[Win.Peak.RightGround[i]]);
    StringGridPeaks.Cells[11,1+i]:=Format('%.7g',[Win.Peak.StatError[i]]);
   end;
   StringGridPeaks.Update;
   CheckBoxViewPos.Checked:=Win.Peak.isViewPos;
   CheckBoxViewForm.Checked:=Win.Peak.isViewForm;
   CheckBoxViewGround.Checked:=Win.Peak.isViewGround;
   CheckBoxViewSingle.Checked:=Win.Peak.isViewSingle;
   for i:=0 to min(High(P),StringGridPreset.ColCount-1) do begin
    StringGridPreset.Cells[i,0]:=ExtractWord(1+i,PeakInFormat,[';']);
    StringGridPreset.Cells[i,1]:=Format('%.5g',[P[i]]);
   end;
  end;
  if HasFlags(Flags,2) then begin
   R.Left:=1;
   R.Top:=Win.PeakFindNearest(Win.Marker)+1;
   R.Right:=R.Left;
   R.Bottom:=R.Top;
   if R.Top<StringGridPeaks.RowCount then
   if R.Left<StringGridPeaks.ColCount then begin
    StringGridPeaks.Col:=R.Left;
    StringGridPeaks.Row:=R.Top;
    StringGridPeaks.Selection:=R;
    StringGridPeaks.Update;
   end;
  end;
  if HasFlags(Flags,4) then begin
   for i:=0 to min(High(P),StringGridPreset.ColCount-1) do Str2Real(StringGridPreset.Cells[i,1],P[i]);
  end;
  if HasFlags(Flags,$10) then begin
   PageControl.PageIndex:=0;
  end;
  if HasFlags(Flags,$20) then begin
   PageControl.PageIndex:=1;
  end;
  if HasFlags(Flags,$40) then begin
   PageControl.PageIndex:=2;
  end;
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikControl.FormCreate(Sender: TObject);
begin
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 LocateFormToCenterOfScreen(Self);
 Top:=0;
 TabSheetPeakList.Caption:=RusEng('Список пиков','Peak list');
 TabSheetControls.Caption:=RusEng('Команды управления','Controls');
 TabSheetPreset.Caption:=RusEng('Параметры поиска','Preset');
 ButtonSearch.Caption:=RusEng('Поиск','Search');
 ButtonClear.Caption:=RusEng('Очистка','Clear');
 ButtonOpen.Caption:=RusEng('Чтение','Open');
 ButtonSave.Caption:=RusEng('Запись','Save');
 ButtonPeakPrev.Caption:=RusEng('< Пред <','< Prev <');
 ButtonPeakNext.Caption:=RusEng('> След >','> Next >');
 GroupBoxView.Caption:=RusEng('Отображение','View');
 CheckBoxViewPos.Caption:=RusEng('Положение','Position');
 CheckBoxViewForm.Caption:=RusEng('Форма','Form');
 CheckBoxViewGround.Caption:=RusEng('Подложка','Ground');
 CheckBoxViewSingle.Caption:=RusEng('Один','Single');
end;

procedure TFormSpectrPikControl.ButtonSearchClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.PeakSearch;
end;

procedure TFormSpectrPikControl.ButtonClearClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.PeakClear;
end;

procedure TFormSpectrPikControl.ButtonOpenClick(Sender: TObject);
begin
 if Ok then
 if Win.Ok then begin
  OpenDialog.Title:=RusEng('Открыть файл списка пиков','Open peak list file');
  OpenDialog.Filter:=RusEng('Файл списка пиков (*.pik)|*.PIK',
                            'Peak list files   (*.pik)|*.PIK');
  if GuardOpenDialog(OpenDialog).Execute then Win.PeakLoad(OpenDialog.FileName);
 end;
end;

procedure TFormSpectrPikControl.ButtonSaveClick(Sender: TObject);
begin
 if Ok then
 if Win.Ok then begin
  SaveDialog.Title:=RusEng('Сохранить файл списка пиков','Save peak list file');
  SaveDialog.Filter:=RusEng('Файл списка пиков (*.pik)|*.PIK',
                            'Peak list files   (*.pik)|*.PIK');
  if GuardOpenDialog(SaveDialog).Execute then Win.PeakSave(SaveDialog.FileName);
 end;
end;

procedure TFormSpectrPikControl.ButtonPeakPrevClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.PeakPrev;
end;

procedure TFormSpectrPikControl.ButtonPeakNextClick(Sender: TObject);
begin
 if Ok and Win.Ok then Win.PeakNext;
end;

procedure TFormSpectrPikControl.CheckBoxViewPosClick(Sender: TObject);
begin
 if Ok and Win.Ok then
 try
  Win.LockDraw;
  Win.Peak.isViewPos:=CheckBoxViewPos.Checked;
 finally
  Win.UnlockDraw;
 end;
end;

procedure TFormSpectrPikControl.CheckBoxViewFormClick(Sender: TObject);
begin
 if Ok and Win.Ok then
 try
  Win.LockDraw;
  Win.Peak.isViewForm:=CheckBoxViewForm.Checked;
 finally
  Win.UnlockDraw;
 end;
end;

procedure TFormSpectrPikControl.CheckBoxViewGroundClick(Sender: TObject);
begin
 if Ok and Win.Ok then
 try
  Win.LockDraw;
  Win.Peak.isViewGround:=CheckBoxViewGround.Checked;
 finally
  Win.UnlockDraw;
 end;
end;

procedure TFormSpectrPikControl.CheckBoxViewSingleClick(Sender: TObject);
begin
 if Ok and Win.Ok then
 try
  Win.LockDraw;
  Win.Peak.isViewSingle:=CheckBoxViewSingle.Checked;
 finally
  Win.UnlockDraw;
 end;
end;

procedure TFormSpectrPikControl.StringGridPeaksClick(Sender: TObject);
var
 n : Integer;
begin
 if Ok and Win.Ok then
 try
  n:=StringGridPeaks.Row-1;
  if InRange(n,0,Win.Peak.NumPeaks-1)
  then Win.Marker:=Round(Win.Peak.Channel[n]);
 except
  on E:Exception do Daq.Report(E.Message);
 end;
end;

procedure TFormSpectrPikControl.StringGridPresetKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
 if (Key=VK_RETURN) then UpdateControls(4);
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_spectrpikcontrol;
begin
end;

procedure Free_form_spectrpikcontrol;
begin
end;

initialization

 Init_form_spectrpikcontrol;

finalization

 Free_form_spectrpikcontrol;

end.

//////////////
// END OF FILE
//////////////

