////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Curve Tools Run Plugin Dialog.                                        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231214 - Modified for FPC (A.K.)                                         //
// 20240626 - ApplyParams                                                     //
////////////////////////////////////////////////////////////////////////////////

unit form_curvetoolsrunplugindialog; // Form Curve Tools Run Plugin Dialog

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Grids,
 lcltype, lclintf, fileutil,
 Form_CrwDaqSysChild, Form_TextEditor,
 Unit_SystemConsole, Form_CurveWindow,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_proc,
 _crw_ef, _crw_zm, _crw_sort, _crw_ee,
 _crw_sesman, _crw_memo, _crw_flgrd,
 _crw_curves, _crw_crwapi, _crw_crwapiserver,
 _crw_dcc32, _crw_fpcup, _crw_pascalprojects,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormCurveToolsRunPluginDialog }

  TFormCurveToolsRunPluginDialog = class(TMasterForm)
    ActionClone: TAction;
    ActionListButtons: TActionList;
    ActionRun: TAction;
    ActionEdit: TAction;
    ActionDelete: TAction;
    ActionCreate: TAction;
    ActionCompile: TAction;
    ActionCompileAll: TAction;
    ActionClose: TAction;
    ComboBoxProjectType: TComboBox;
    ImageList16: TImageList;
    LabelProjectType: TLabel;
    PanelControls: TPanel;
    PanelManualArgumentsButtons: TPanel;
    GroupBoxManual: TGroupBox;
    MemoManual: TMemo;
    GroupBoxButtons: TGroupBox;
    BitBtnRun: TBitBtn;
    BitBtnEdit: TBitBtn;
    BitBtnCreate: TBitBtn;
    BitBtnDelete: TBitBtn;
    BitBtnClose: TBitBtn;
    BitBtnCompile: TBitBtn;
    BitBtnCompileAll: TBitBtn;
    GroupBoxArguments: TGroupBox;
    PanelProjectType: TPanel;
    StringGridArguments: TStringGrid;
    PanelPlugins: TPanel;
    GroupBoxPlugins: TGroupBox;
    ListBoxPlugins: TListBox;
    TimerCheckPos: TTimer;
    procedure FormCreate(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure ListBoxPluginsClick(Sender: TObject);
    procedure ListBoxPluginsDblClick(Sender: TObject);
    procedure ActionRunExecute(Sender: TObject);
    procedure ActionEditExecute(Sender: TObject);
    procedure ActionDeleteExecute(Sender: TObject);
    procedure ActionCreateExecute(Sender: TObject);
    procedure ActionCompileExecute(Sender: TObject);
    procedure ActionCompileAllExecute(Sender: TObject);
    procedure ActionCloseExecute(Sender: TObject);
    procedure ActionCloneExecute(Sender: TObject);
    procedure ComboBoxProjectTypeChange(Sender: TObject);
    procedure TimerCheckPosTimer(Sender: TObject);
  private
    { Private declarations }
    Compiling : Boolean;
    CloneCapt : LongString;
    CloneCoor : TPoint;
    function  DllExt:LongString;
    function  ProjectExt:LongString;
    function  GetArguments:LongString;
    function  GetCurrentPlugin(FullPath:Boolean=false; Ext:LongString=''):LongString;
    function  IsCurrentPluginDllExists(Mode:Integer):Boolean;
    function  ListPlugins(Pattern:LongString):LongString;
    function  FindPlugin(Pattern:LongString):LongString;
    procedure ExecutePlugin(SrcWin:TFormCurveWindow);
    procedure UpdateControls(Flags:Integer);
    procedure CompileItemsList(const aItemsList:LongString);
    function  DeletePluginProject(Src:LongString):Boolean;
    function  ProtectedProject(Src:LongString):Boolean;
    function  CloneProject(aProject:LongString='';
                           aNewName:LongString=''):LongString;
  public
    { Public declarations }
  end;

procedure ExecuteFormCurveToolsRunPluginDialog(SrcWin:TFormCurveWindow; const aParams:LongString='');

implementation

uses
 Form_CrwDaq,
 Form_DelphiProjectEditor,
 Form_LazarusProjectEditor;

{$R *.lfm}

const
 mrRun     = mrAll+1;
 mrEdit    = mrAll+2;
 mrCreate  = mrAll+3;

procedure ExecuteFormCurveToolsRunPluginDialog(SrcWin:TFormCurveWindow; const aParams:LongString='');
const TheForm:TFormCurveToolsRunPluginDialog=nil;
var apFlags:Integer;
begin
 if CanShowModal(TheForm) then
 try
  if not Assigned(TheForm) then begin
   Application.CreateForm(TFormCurveToolsRunPluginDialog, TheForm);
   TheForm.Master:=@TheForm;
  end;
  if TheForm.Ok then
  with TheForm do begin
   apFlags:=ApplyParams(aParams);
   if not HasFlags(apFlags,apf_FormPos)
   then LocateFormToCenterOfScreen(TheForm);
   UpdateControls(1+2+4); Compiling:=false;
   case ShowModal of
    mrRun:    if (GetCurrentPlugin<>'') then ExecutePlugin(SrcWin);
    mrEdit:   if (GetCurrentPlugin<>'') then FindPascalProjectEditor(GetCurrentPlugin(true),true,true);
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'ExecuteFormCurveToolsRunPluginDialog');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.FormCreate(Sender: TObject);
begin
 inherited;
 Compiling:=false;
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 Caption:=RusEng('Диалог для запуска утилиты обработки данных','Dialog to run data analysis plugin');
 SmartUpdate(GroupBoxPlugins,RusEng('Список утилит:','List of plugins:'));
 SmartUpdate(GroupBoxManual,RusEng('Краткое описание:','Short manual:'));
 SmartUpdate(GroupBoxArguments,RusEng('Список аргументов:','List of arguments:'));
 SmartUpdate(GroupBoxButtons,RusEng('Команды:','Commands:'));
 SmartUpdate(ActionRun,RusEng('Пуск','Run'));
 SmartUpdate(ActionClose,RusEng('Выход','Close'));
 LabelProjectType.Caption:=RusEng('Тип Проекта:','Project Type:');
 ComboBoxProjectType.Items.Text:=ListOfPascalProjectTypes;
 ComboBoxProjectType.ItemIndex:=Min(0,ComboBoxProjectType.Items.Count-1);
 ComboBoxProjectType.Hint:=RusEng('Тип Проекта Pascal.','Pascal Project Type.');
 ActionRun.Hint:=RusEng('Запуск утилиты анализа данных.','Execute data analysis plugin.');
 ActionClose.Hint:=RusEng('Закрыть окно диалога.','Close the dialog window.');
 ActionEdit.Hint:=RusEng('Редактировать текст утилиты.','Edit plugin source file.');
 ActionCreate.Hint:=RusEng('Клонировать, т.е. создать новую утилиту обработки по шаблону.',
                           'Clone, i.e. create new data analysis plugin from sample.');
 ActionCompile.Hint:=RusEng('Компилировать утилиту.','Compile plugin source.');
 ActionCompileAll.Hint:=RusEng('Компилировать все утилиты.','Compile all plugins source.');
 ActionDelete.Hint:=RusEng('Удалить файлы утилиты.','Delete plugin files.');
 CloneCapt:='';
end;

procedure TFormCurveToolsRunPluginDialog.FormResize(Sender: TObject);
begin
 inherited;
 if Ok then
 try
  with StringGridArguments do
  if (ColCount>1) then ColWidths[1]:=Max(135,Width-ColWidths[0]-25);
 except
  on E:Exception do BugReport(E,Self,'FormResize');
 end;
end;

function  TFormCurveToolsRunPluginDialog.DllExt:LongString;
begin
 if IsUnix then Result:='.so' else
 if IsWindows then Result:='.dll' else Result:='';
end;

function  TFormCurveToolsRunPluginDialog.ProjectExt:LongString;
begin
 if not Assigned(Self) then Exit('');
 Result:=TrimDef(ComboBoxProjectType.Text,'.lpr');
end;

function TFormCurveToolsRunPluginDialog.GetArguments:LongString;
var P:TText; r:Integer; Ref,Val:LongString;
begin
 Result:='';
 if Ok then
 try
  P:=NewText;
  try
   for r:=0 to StringGridArguments.RowCount-1 do begin
    Ref:=Trim(StringGridArguments.Cells[0,r]);
    Val:=Trim(StringGridArguments.Cells[1,r]);
    if (Ref<>'') then P.Addln(Format('%s=%s',[Ref,Val]));
   end;
   Result:=P.Text;
  finally
   Kill(P);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetArguments');
 end;
end;

function TFormCurveToolsRunPluginDialog.ListPlugins(Pattern:LongString):LongString;
begin
 Result:='';
 if Ok then
 try
  Pattern:=Trim(Pattern); if (Pattern='') then Exit;
  Result:=FindAllFilesAsText(GetDataAnalysisPluginsFolder,Pattern,true);
 except
  on E:Exception do BugReport(E,Self,'ListPlugins');
 end;
end;

function TFormCurveToolsRunPluginDialog.FindPlugin(Pattern:LongString):LongString;
var List:LongString;
begin
 Result:='';
 if Ok then
 try
  Pattern:=ExtractFileNameExt(Pattern);
  Pattern:=DefaultExtension(Pattern,ProjectExt);
  List:=ListPlugins(Pattern);
  if (WordCount(List,EolnDelims)=1)
  then Result:=UnifyFileAlias(Trim(List),ua_RealPath);
 except
  on E:Exception do BugReport(E,Self,'FindPlugin');
 end;
end;

function TFormCurveToolsRunPluginDialog.GetCurrentPlugin(FullPath:Boolean=false; Ext:LongString=''):LongString;
var Pattern:LongString;
begin
 Result:='';
 if Ok then
 try
  with ListBoxPlugins do
  if (ItemIndex>=0) and (ItemIndex<Items.Count) then Result:=Trim(Items[ItemIndex]);
  if (Length(Result)>0) and FullPath then begin
   Ext:=TrimDef(Ext,ProjectExt);
   Pattern:=ForceExtension(Result,Ext);
   if IsSameText(Ext,DllExt)
   then Pattern:=AdaptDllFileName(Pattern);
   Result:=FindPlugin(Pattern);
  end;
 except
  on E:Exception do BugReport(E,Self,'GetCurrentPlugin');
 end;
end;

function TFormCurveToolsRunPluginDialog.IsCurrentPluginDllExists(Mode:Integer):Boolean;
var DllPath,msg:LongString;
begin
 Result:=false;
 if not Ok then Exit;
 if (GetCurrentPlugin<>'') then
 try
  DllPath:=GetCurrentPlugin(true,DllExt);
  Result:=FileExists(DllPath); msg:='';
  if HasFlags(Mode,1+2) and not Result then begin
   msg:=RusEng('ОШИБКА: ПЛАГИН НЕ СКОМПИЛИРОВАН.','ERROR: PLUGIN IS NOT COMPILED.')+EOL
       +RusEng('Не найден исполняемый файл плагина: ','Not found executable file for plugin: ')+GetCurrentPlugin+EOL
       +RusEng('Скомпилируйте плагин!','Please compile plugin!')+EOL;
   if HasFlags(Mode,1) then begin
    Echo(Trim(msg));
   end;
   if HasFlags(Mode,2) then begin
    MemoManual.Lines.Clear;
    MemoManual.Lines.Add(msg);
    MemoManual.MoveCaretToEnd;
    MemoManual.Update;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'IsCurrentPluginDllExists');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.ExecutePlugin(SrcWin:TFormCurveWindow);
var CrwApi:TCrwApiServer; DllPath:LongString;
var aName,aSource,aBinary:LongString;
begin
 inherited;
 if Ok then
 if (GetCurrentPlugin<>'') then
 try
  if not IsCurrentPluginDllExists(1) then Exit;
  DllPath:=GetCurrentPlugin(true,DllExt);
  if not FileExists(DllPath) then Exit;
  aBinary:=DllPath;
  aName:=GetCurrentPlugin;
  aSource:=GetCurrentPlugin(true);
  CrwApi:=NewCrwApiServer(ForDataAnalysis,aName,aSource,aBinary);
  if Assigned(CrwApi) then
  try
   CrwApi.ExecuteDataAnalysis(DllPath,SrcWin,GetArguments);
  finally
   CrwApi.Free;
  end;
 except
  on E:Exception do BugReport(E,Self,'ExecutePlugin');
 end;
end;

function ListNameComp(List:TStringList; Index1,Index2:Integer):Integer;
 function Fix(const s:LongString):LongString;
 begin
  if (StrFetch(s,1)='_') then Result:='!'+TailStr(s,2) else Result:=s;
 end;
begin
 Result:=CompareText(Fix(List[Index1]),Fix(List[Index2]));
end;

procedure TFormCurveToolsRunPluginDialog.UpdateControls(Flags:Integer);
var P:TText; Lpr,Ref,Val:LongString; i,c,r:Integer;
begin
 if Ok then
 try
  Lpr:=''; Ref:=''; Val:='';
  if HasFlags(Flags,$0001) then begin
   i:=ListBoxPlugins.ItemIndex; Val:='';
   if InRange(i,0,ListBoxPlugins.Items.Count-1) then Val:=ListBoxPlugins.Items[i];
   ListBoxPlugins.Items.Text:=SortTextLines(GetDataAnalysisPluginsList(true,ProjectExt),ListNameComp);
   i:=ListBoxPlugins.Items.IndexOf(Val); if (i>=0) then ListBoxPlugins.ItemIndex:=i;
   with ListBoxPlugins do
   if (ItemIndex<0) or (ItemIndex>Items.Count-1) then ItemIndex:=Min(0,Items.Count-1);
  end;
  if HasFlags(Flags,$0002) then begin
   MemoManual.Text:='';
   Lpr:=GetCurrentPlugin(true);
   if Length(Lpr)>0 then MemoManual.Text:=ExtractTextSection(Lpr,RusEng('[Manual.Rus]','[Manual.Eng]'),efAsIs);
   if MemoManual.Text='' then MemoManual.Text:=RusEng('Без комментариев.','No comments.');
  end;
  if HasFlags(Flags,$0004) then begin
   StringGridArguments.RowCount:=50;
   StringGridArguments.FixedRows:=0;
   StringGridArguments.ColCount:=2;
   StringGridArguments.FixedCols:=1;
   for r:=0 to StringGridArguments.RowCount-1 do
   for c:=0 to StringGridArguments.ColCount-1 do StringGridArguments.Cells[c,r]:='';
   Lpr:=GetCurrentPlugin(true);
   P:=ExtractListSection(Lpr,RusEng('[Arguments.Rus]','[Arguments.Eng]'),efAsIs);
   try
    r:=0;
    for i:=0 to P.Count-1 do begin
     if r>=StringGridArguments.RowCount then break;
     c:=ExtractNameValuePair(P[i],Ref,Val);
     if (c>0) then begin
      if (Ref<>'') then begin
       StringGridArguments.Cells[0,r]:=Ref;
       StringGridArguments.Cells[1,r]:=Val;
       inc(r);
      end;
     end;
    end;
   finally
    Kill(P);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateControls');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.CompileItemsList(const aItemsList:LongString);
var i,m:Integer; aProj:LongString; Items:TText;
const Timeout=30000;
 procedure EnableButtons(aEnable:Boolean);
 begin
  SetEnabledControls(aEnable,[BitBtnRun,BitBtnClose,BitBtnEdit,BitBtnCreate,
                             BitBtnCompile,BitBtnDelete]);
  BitBtnCompileAll.ImageIndex:=IfThen(aEnable,5,6);
 end;
begin
 if Ok then
 try
  Items:=NewText;
  EnableButtons(false);
  try
   if Compiling then Exit;              // No recursion!
   if IsEmptyStr(aItemsList) then Exit; // Nothing to do
   Compiling:=true;
   MemoManual.Lines.Clear;
   MemoManual.Lines.Add(RusEng('Компиляция:','Compile:'));
   MemoManual.Lines.Add(RusEng('===========','========'));
   MemoManual.Update;
   Items.Text:=aItemsList; m:=0;
   for i:=0 to Items.Count-1 do Items[i]:=Trim(Items[i]);
   for i:=0 to Items.Count-1 do m:=Max(m,Length(Items[i]));
   for i:=0 to Items.Count-1 do begin
    aProj:=FindPlugin(ForceExtension(Items[i],ProjectExt));
    if SafeCompilePascalProject(aProj,Timeout) then begin
     MemoManual.Lines.Add(Pad(Items[i],m)+RusEng(' - УСПЕХ.',' - DONE.'));
     MemoManual.Update;
    end else begin
     MemoManual.Lines.Add(Pad(Items[i],m)+RusEng(' - ОТКАЗ.',' - FAIL.'));
     MemoManual.MoveCaretToEnd;
     MemoManual.Update;
     Break;
    end;
    ExecuteSystemConsoleMonitoring;
    SafeApplicationProcessMessages;
    if not Compiling then begin
     MemoManual.Lines.Add(RusEng('ПРЕРВАНО.','CANCEL.'));
     MemoManual.MoveCaretToEnd;
     MemoManual.Update;
     Break;
    end;
    if (i=Items.Count-1) then begin
     MemoManual.Lines.Add(RusEng('ЗАВЕРШЕНО.','COMPLETE.'));
     MemoManual.MoveCaretToEnd;
     MemoManual.Update;
    end;
   end;
  finally
   Compiling:=false;
   EnableButtons(true);
   Kill(Items);
  end;
 except
  on E:Exception do BugReport(E,Self,'CompileItemsList');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.ListBoxPluginsClick(Sender: TObject);
begin
 if Ok then UpdateControls(2+4);
end;

procedure TFormCurveToolsRunPluginDialog.ListBoxPluginsDblClick(Sender: TObject);
begin
 if Ok then ActionRun.Execute;
end;

procedure TFormCurveToolsRunPluginDialog.ActionRunExecute(Sender: TObject);
begin
 if not Ok then Exit;
 if Guard.CheckAction(ga_Guest,ActionRun)<0 then Exit;
 if not IsCurrentPluginDllExists(2) then Exit;
 ModalResult:=mrRun;
end;

procedure TFormCurveToolsRunPluginDialog.ActionEditExecute(Sender: TObject);
begin
 if not Ok then Exit;
 if Guard.CheckAction(ga_Root,ActionEdit)<0 then Exit;
 ModalResult:=mrEdit;
end;

function TFormCurveToolsRunPluginDialog.ProtectedProject(Src:LongString):Boolean;
var FileName:LongString;
begin
 FileName:=Src;
 if IsRelativePath(Src)
 then FileName:=FindPlugin(Src);
 Result:=FileGuard.ProtectedReadOnly(FileName);
 // In addition to FileGuard protect _*.lpr projects.
 Result:=Result or StartsText('_',ExtractFileName(Src));
end;

function TFormCurveToolsRunPluginDialog.DeletePluginProject(Src:LongString):Boolean;
var Dir,Ext,Dst:LongString;
begin
 Result:=false;
 if Ok then
 if IsNonEmptyStr(Src) then
 if not ProtectedProject(Src) then
 try
  Ext:=ExtractFileExt(Src);
  if IsSameText(Ext,'.lpr') then begin
   Dir:=ExtractFileDir(Src);
   if DirExists(Dir)
   then Result:=DeleteDirectory(Dir,false)
   else Result:=true;
   Exit;
  end;
  if IsSameText(Ext,'.dpr') then begin
   Dst:=ForcePath(AddBackSlash(GetDataAnalysisPluginsFolder)+'Backup',Src);
   MkDir(ExtractFilePath(Dst));
   FileRename(Src,Dst);
   FileRename(ForceExtension(Src,DllExt),ForceExtension(Dst,DllExt));
   Exit;
  end;
 except
  on E:Exception do BugReport(E,Self,'DeletePluginProject');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.ActionDeleteExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Root,ActionDelete)<0 then Exit;
 if Ok then
 if (GetCurrentPlugin<>'') then
 if (YesNo(Format(RusEng('Вы действительно хотите удалить'+EOL+'утилиту "%s"?',
                         'Do you really want to delete'+EOL+'plugin "%s"?'),
                         [GetCurrentPlugin]),
                         ControlPosParams(GroupBoxArguments))=mrYes)
 then
 try
  if ProtectedProject(GetCurrentPlugin) then begin
   MemoManual.Lines.Clear;
   MemoManual.Lines.Add(GetCurrentPlugin);
   MemoManual.Lines.Add(RusEng('Этот проект защищен от удаления.',
                               'Can`t remove this protected project.'));
   MemoManual.MoveCaretToEnd;
   MemoManual.Update;
   Exit;
  end;
  DeletePluginProject(GetCurrentPlugin(true));
  UpdateControls(1+2+4);
 except
  on E:Exception do BugReport(E,Self,'ActionDeleteExecute');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.ActionCreateExecute(Sender: TObject);
begin
 if not Ok then Exit;
 if Guard.CheckAction(ga_Root,ActionCreate)<0 then Exit;
 ActionClone.Execute; ModalResult:=mrCreate;
end;

procedure TFormCurveToolsRunPluginDialog.ActionCompileExecute(Sender: TObject);
begin
 if not Ok then Exit;
 if Guard.CheckAction(ga_Root,ActionCompile)<0 then Exit;
 CompileItemsList(GetCurrentPlugin);
end;

procedure TFormCurveToolsRunPluginDialog.ActionCompileAllExecute(Sender: TObject);
begin
 if not Ok then Exit;
 if Guard.CheckAction(ga_Root,ActionCompileAll)<0 then Exit;
 if Compiling
 or (YesNo(RusEng('Вы хотите перекомпилировать все утилиты?'+EOL+
                  'Это займет некоторое время …',
                  'Do you want to compile all plugins?'+EOL+
                  'It will take some time …'),
                  ControlPosParams(GroupBoxArguments))=mrYes)
 then
 try
  CompileItemsList(ListBoxPlugins.Items.Text);
 except
  on E:Exception do BugReport(E,Self,'ActionCompileAllExecute');
 end;
end;

procedure TFormCurveToolsRunPluginDialog.ActionCloseExecute(Sender: TObject);
begin
 if Ok then ModalResult:=mrCancel;
end;

procedure TFormCurveToolsRunPluginDialog.ComboBoxProjectTypeChange(Sender: TObject);
begin
 if Ok then UpdateControls(1+2+4);
end;

procedure TFormCurveToolsRunPluginDialog.ActionCloneExecute(Sender: TObject);
begin
 if not Ok then Exit;
 if Guard.CheckAction(ga_Root,ActionCreate)<0 then Exit;
 CloneProject;
end;

procedure TFormCurveToolsRunPluginDialog.TimerCheckPosTimer(Sender: TObject);
var i:Integer;
begin
 if Ok then
 try
  if (CloneCapt<>'') then
  for i:=0 to Screen.FormCount-1 do
  if IsSameText(Screen.Forms[i].Caption,CloneCapt) then begin
   Screen.Forms[i].Left:=CloneCoor.X;
   Screen.Forms[i].Top:=CloneCoor.Y;
   TimerCheckPos.Enabled:=false;
   CloneCapt:='';
  end;
 except
  on E:Exception do BugReport(E,Self,'TimerCheckPosTimer');
 end;
end;

function TFormCurveToolsRunPluginDialog.CloneProject(aProject:LongString='';
                                                     aNewName:LongString=''):LongString;
var exe,cmd,src,dir,cap,pmt:LongString; idn:String;
begin
 Result:='';
 if not Ok then Exit;
 if IsEmptyStr(aProject) then aProject:=GetCurrentPlugin(true);
 if IsSameText(ExtractFileExt(aProject),ProjectExt) then
 if FileExists(aProject) then
 try
  idn:=Trim(aNewName);
  if IsEmptyStr(idn) then begin
   idn:='new_plugin';
   cap:=SessionManager.SessionHead+': '
       +RusEng('Клонировать утилиту ','Clone plugin ')+' '
       +ExtractFileNameExt(aProject)+' …';
   pmt:=RusEng('Задайте ИМЯ для нового Проекта по шаблону ',
               'Enter NAME for new Project by sample ')
               +ExtractFileNameExt(aProject)+':';
   TimerCheckPos.Enabled:=true; CloneCapt:=cap;
   CloneCoor:=GroupBoxArguments.ClientToScreen(Point(0,0));
   if InputQuery(cap,pmt,idn) then idn:=Trim(idn) else Exit;
   TimerCheckPos.Enabled:=false; CloneCapt:='';
  end;
  idn:=LowerCase(Trim(idn));
  if IsLexeme(idn,lex_Name) then begin
   exe:=Fpcup.IniFileLazClone;
   src:=UnifyFileAlias(aProject);
   dir:=ExtractFileDir(ExtractFileDir(src));
   exe:=MakeRelativePath(exe,ProgName);
   src:=MakeRelativePath(src,ProgName);
   dir:=MakeRelativePath(dir,Progname);
   cmd:=QArg(exe)+' '+QArg(src)+' '+QArg(dir)+' '+idn;
   SendToMainConsole('@run -sw7 -cd ~~ '+cmd+EOL);
  end else begin
   cap:=SessionManager.TitlePidAtHost;
   pmt:=RusEng('Недопустимое имя Проекта: ','Invalid Project name: ')+idn;
   cmd:=Format('@silent @tooltip text "%s: %s" preset stdWarning delay 30000',[cap,pmt]);
   SendToMainConsole(cmd+EOL); Echo(pmt);
  end;
  Result:=AddPathDelim(AddPathDelim(dir)+idn)+idn+ProjectExt;
 except
  on E:Exception do BugReport(E,Self,'ActionCompileAllExecute');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_curvetoolsrunplugindialog;
begin
end;

procedure Free_form_curvetoolsrunplugindialog;
begin
end;

initialization

 Init_form_curvetoolsrunplugindialog;

finalization

 Free_form_curvetoolsrunplugindialog;

end.

//////////////
// END OF FILE
//////////////

