////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form NSIS Window.                                                          //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231201 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit form_nsiswindow; // Form NSIS Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_TextEditor,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard, _crw_task,
 _crw_appforms, _crw_apptools, _crw_apputils;

type
  TFormNsisEditor = class(TFormTextEditor)
    ActionEditCompile: TAction;
    ActionEditHelpNsisApi: TAction;
    MenuEditCompile: TMenuItem;
    MenuEditHelpNsisApi: TMenuItem;
    ToolButtonEditCompileSeparator: TToolButton;
    ToolButtonEditCompile: TToolButton;
    ToolButtonEditHelpNsisApi: TToolButton;
    PopupMenuEditCompile: TMenuItem;
    PopupMenuEditHelpNsisApi: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure ActionEditCompileExecute(Sender: TObject);
    procedure ActionEditHelpNsisApiExecute(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    procedure UpdateCommands; override;
  end;

function NewFormNsisEditor(const aFileName : LongString): TFormNsisEditor;
procedure Kill(var TheObject:TFormNsisEditor); overload;
function GetNsisCompilerPath:LongString;

type
 ENsisEditor = class(ESoftException);

implementation

{$R *.lfm}

function NewFormNsisEditor(const aFileName : LongString): TFormNsisEditor;
begin
 Result:=nil;
 try
  if IsNonEmptyStr(aFileName) then begin
   Application.CreateForm(TFormNsisEditor, Result);
   if Result.Ok then begin
    Result.Open(UnifyFileAlias(aFileName));
    Result.DefaultReadOnly;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'NewForlfmisEditor');
 end;
end;

procedure Kill(var TheObject:TFormNsisEditor); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

function GetNsisCompilerPath:LongString;
begin
 Result:='';
 try
  {$IFDEF WINDOWS}
  Result:=ReadRegistryString(HKEY_LOCAL_MACHINE,
          'SOFTWARE\Classes\NSIS.Script\shell\compile\command','');
  if IsNonEmptyStr(Result)
  then Result:=Trim(StringReplace(
                    StringReplace(Result,'%1','',[rfReplaceAll]),
                                          '"','',[rfReplaceAll]));
  {$ENDIF ~WINDOWS}
 except
  on E:Exception do BugReport(E,nil,'GetNsisCompilerPath');
 end;
end;

procedure TFormNsisEditor.FormCreate(Sender: TObject);
begin
 inherited;
 UpdateMenu(MenuEditCompile,
            RusEng('Компиляция','Compile'),
            RusEng('Компилировать NSIS - скрипт.','Compile NSIS - script.'),
            ShortCut(VK_F9,[ssCtrl]));
 UpdateMenu(MenuEditHelpNsisApi,
            RusEng('Справка по NSIS','Help on NSIS'),
            RusEng('Вызвать справку по NSIS.','Open NSIS help.'),
            ShortCut(VK_F1,[ssCtrl]));
end;

procedure TFormNsisEditor.UpdateCommands;
begin
 inherited;
 try
  ActionEditCompile.Enabled:=(WindowState<>wsMinimized) and not PerformReadOnly;
 except
  on E:Exception do BugReport(E,Self,'UpdateCommands');
 end;
end;

procedure TFormNsisEditor.ActionEditCompileExecute(Sender: TObject);
var CmdLine:LongString;
begin
 if Guard.CheckAction(ga_Root,ActionEditCompile)<0 then Exit;
 if Ok then
 try
  FileSave;
  CmdLine:=GetNsisCompilerPath;
  if IsEmptyStr(CmdLine)
  then RAISE ENsisEditor.Create('NSIS compiler not found. Please install NSIS first.');
  SmartExecute(Format('"%s" "%s"',[CmdLine,PathName]));
 except
  on E:Exception do BugReport(E,Self,'ActionCompileExecute');
 end;
end;

procedure TFormNsisEditor.ActionEditHelpNsisApiExecute(Sender: TObject);
var CmdLine:LongString;
begin
 if Guard.CheckAction(ga_Root,ActionEditHelpNsisApi)<0 then Exit;
 if Ok then
 try
  CmdLine:=GetNsisCompilerPath;
  if IsEmptyStr(CmdLine)
  then RAISE ENsisEditor.Create('NSIS compiler not found. Please install NSIS first.');
  CmdLine:=AddPathDelim(ExtractFilePath(CmdLine))+'nsis.chm';
  if not FileExists(CmdLine) then RAISE ENsisEditor.Create('NSIS help not found.');
  SmartExecute(Format('"%s"',[CmdLine]),SW_SHOW,'open');
 except
  on E:Exception do BugReport(E,Self,'ActionHelpNsisApiExecute');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_nsiswindow;
begin
end;

procedure Free_form_nsiswindow;
begin
end;

initialization

 Init_form_nsiswindow;

finalization

 Free_form_nsiswindow;

end.

//////////////
// END OF FILE
//////////////

