////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form CRW-DAQ System Child.                                                 //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231115 - Modified for FPC (A.K.)                                         //
// 20240601 - DebugLogReport_SdiForm                                          //
// 20240605 - DebugLogReport_DrawView                                         //
// 20240606 - CodePillsEnabled,TakesCodePills                                 //
// 20240614 - IsFormViewable - uses with DrawView                             //
// 20240826 - TSdiManager,SdiMan                                              //
// 20250127 - FixedBoundsTolerance - granularity for FixedBounds              //
// 20250129 - Use TAtomicCounter                                              //
// 20250216 - PendigTriggers: ptrg_XXXX                                       //
////////////////////////////////////////////////////////////////////////////////

unit form_crwdaqsyschild; // Form CRW-DAQ System Child

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, strutils,
 Graphics, Interfaces, LMessages,
 Forms, Controls, ComCtrls, Dialogs,
 lcltype, lclintf, lclproc, typinfo,
 ActnList, Menus, ToolWin, ImgList, Printers, Clipbrd,
 _crw_alloc, _crw_rtc, _crw_str, _crw_ef, _crw_dbglog, _crw_plut,
 _crw_eldraw, _crw_guard, _crw_wmctrl, unit_crwdaq_common,
 _crw_appforms, _crw_apptools, _crw_apputils;

type

  { TFormCrwDaqSysChild }

  TFormCrwDaqSysChild = class(TMasterForm)
    ActionList: TActionList;
    MainMenu: TMainMenu;
    MainPopupMenu: TPopupMenu;
    ToolBar: TToolBar;
    StatusBar: TStatusBar;
    ImageList: TImageList;
    SaveDialog: TSaveDialog;
    ActionFileNewTextEditor: TAction;
    ActionFileNewCurveWindow: TAction;
    ActionFileNewDataAnalysisPlugin: TAction;
    ActionFileNewDataAcquisitionPlugin: TAction;
    ActionFileNewDaqCreator: TAction;
    ActionFileNewDaqConstructor: TAction;
    ActionFileNewProjectFromSample: TAction;
    ActionFileOpen: TAction;
    ActionFileSave: TAction;
    ActionFileSaveAs: TAction;
    ActionFilePrint: TAction;
    ActionFilePrinterSetup: TAction;
    ActionFileExit: TAction;
    ActionViewFixedLeft: TAction;
    ActionViewFixedTop: TAction;
    ActionViewFixedWidth: TAction;
    ActionViewFixedHeight: TAction;
    MenuFile: TMenuItem;
    MenuFileNew: TMenuItem;
    MenuFileNewTextEditor: TMenuItem;
    MenuFileNewCurveWindow: TMenuItem;
    MenuFileNewSeparator1: TMenuItem;
    MenuFileNewDataAnalysisPlugin: TMenuItem;
    MenuFileNewDataAcquisitionPlugin: TMenuItem;
    MenuFileNewSeparator2: TMenuItem;
    MenuFileNewDaqCreator: TMenuItem;
    MenuFileNewDaqConstructor: TMenuItem;
    MenuFileNewProjectFromSample: TMenuItem;
    MenuFileOpen: TMenuItem;
    MenuFileSaveSeparator: TMenuItem;
    MenuFileSave: TMenuItem;
    MenuFileSaveAs: TMenuItem;
    MenuFilePrintSeparator: TMenuItem;
    MenuFilePrint: TMenuItem;
    MenuFilePrinterSetup: TMenuItem;
    MenuFileExitSeparator: TMenuItem;
    MenuFileExit: TMenuItem;
    MenuView: TMenuItem;
    MenuViewFixed: TMenuItem;
    MenuViewFixedLeft: TMenuItem;
    MenuViewFixedTop: TMenuItem;
    MenuViewFixedWidth: TMenuItem;
    MenuViewFixedHeight: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure FormHide(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormActivate(Sender: TObject);
    procedure FormDeactivate(Sender: TObject);
    procedure FormChangeBounds(Sender: TObject);
    procedure FormWindowStateChange(Sender: TObject);
    procedure ActionListUpdate(Action: TBasicAction; var Handled: Boolean);
    procedure ActionFileNewTextEditorExecute(Sender: TObject);
    procedure ActionFileNewCurveWindowExecute(Sender: TObject);
    procedure ActionFileNewDataAnalysisPluginExecute(Sender: TObject);
    procedure ActionFileNewDataAcquisitionPluginExecute(Sender: TObject);
    procedure ActionFileNewDaqCreatorExecute(Sender: TObject);
    procedure ActionFileNewDaqConstructorExecute(Sender: TObject);
    procedure ActionFileNewProjectFromSampleExecute(Sender: TObject);
    procedure ActionFileOpenExecute(Sender: TObject);
    procedure ActionFileSaveExecute(Sender: TObject);
    procedure ActionFileSaveAsExecute(Sender: TObject);
    procedure SaveDialogTypeChange(Sender: TObject);
    procedure ActionFilePrintExecute(Sender: TObject);
    procedure ActionFilePrinterSetupExecute(Sender: TObject);
    procedure ActionFileExitExecute(Sender: TObject);
    procedure ActionViewFixedLeftExecute(Sender: TObject);
    procedure ActionViewFixedTopExecute(Sender: TObject);
    procedure ActionViewFixedWidthExecute(Sender: TObject);
    procedure ActionViewFixedHeightExecute(Sender: TObject);
    procedure StatusBarDblClick(Sender: TObject);
    procedure ToolBarDblClick(Sender: TObject);
  private
    { Private declarations }
    myCloseAction     : TCloseAction;
    myLockCounter     : Cardinal;
    myMonitorTicks    : QWord;
    myMonitorPeriod   : Cardinal;
    myFixed           : record
     Rect             : TRect;
     What             : Integer;
     TimeStamp        : QWord;
     CheckMode        : Integer;
     Interval         : Cardinal;
    end;
    myDaqRef          : Integer;
    function   GetCloseAction:TCloseAction;
    procedure  SetCloseAction(aAction:TCloseAction);
    function   GetMonitorEvent:Boolean;
    function   GetMonitorPeriod:Cardinal;
    procedure  SetMonitorPeriod(aPeriod:Cardinal);
    function   GetFixedRect:TRect;
    function   GetFixedLeft:Integer;
    procedure  SetFixedLeft(L:Integer);
    function   GetFixedTop:Integer;
    procedure  SetFixedTop(T:Integer);
    function   GetFixedWidth:Integer;
    procedure  SetFixedWidth(W:Integer);
    function   GetFixedHeight:Integer;
    procedure  SetFixedHeight(H:Integer);
    function   GetIsFixedLeft:Boolean;
    procedure  SetIsFixedLeft(aFixedLeft:Boolean);
    function   GetFixedWhat:Integer;
    function   GetIsFixedTop:Boolean;
    procedure  SetIsFixedTop(aFixedTop:Boolean);
    function   GetIsFixedWidth:Boolean;
    procedure  SetIsFixedWidth(aFixedWidth:Boolean);
    function   GetIsFixedHeight:Boolean;
    procedure  SetIsFixedHeight(aFixedHeight:Boolean);
    function   GetFixedCheckMode:Integer;
    procedure  SetFixedCheckMode(aMode:Integer);
    function   GetFixedInterval:Cardinal;
    procedure  SetFixedInterval(aInterval:Cardinal);
    function   GetDaqRef:Integer;
    procedure  SetDaqRef(aDaqRef:Integer);
  protected
   myPrintViewKey : Integer;
  protected // CodePills control
    myCodePills : LongWord; // Code Pills Flags to fix known widgetset bugs
    function   NeedsCodePills(aPills:LongWord; aReset:Boolean=true):Boolean;
    function   TakesCodePills(aPills:LongWord):LongWord;
    procedure  PrescribeCodePills(aPills:LongWord);
  protected // CodePills implementation
    function   CodePills_GtkFakeChangeBounds:Boolean; // cpf_GtkFCB
  protected
   procedure   DoUpdateTop(T:Integer); override;
   procedure   DoUpdateLeft(L:Integer); override;
   procedure   DoUpdateWidth(W:Integer); override;
   procedure   DoUpdateHeight(H:Integer); override;
  public
    class var CodePillsEnabled:Boolean;
  public
    procedure  RefreshWindow;
  public
    procedure   AfterConstruction; override;
    procedure   BeforeDestruction; override;
  public
    { Public declarations }
    procedure UpdateCommands; virtual;
    property  CloseAction    : TCloseAction read GetCloseAction    write SetCloseAction;
    property  MonitorEvent   : Boolean      read GetMonitorEvent;
    property  MonitorPeriod  : Cardinal     read GetMonitorPeriod  write SetMonitorPeriod;
    property  FixedRect      : TRect        read GetFixedRect;
    property  FixedWhat      : Integer      read GetFixedWhat;
    property  FixedLeft      : Integer      read GetFixedLeft      write SetFixedLeft;
    property  FixedTop       : Integer      read GetFixedTop       write SetFixedTop;
    property  FixedWidth     : Integer      read GetFixedWidth     write SetFixedWidth;
    property  FixedHeight    : Integer      read GetFixedHeight    write SetFixedHeight;
    property  IsFixedLeft    : Boolean      read GetIsFixedLeft    write SetIsFixedLeft;
    property  IsFixedTop     : Boolean      read GetIsFixedTop     write SetIsFixedTop;
    property  IsFixedWidth   : Boolean      read GetIsFixedWidth   write SetIsFixedWidth;
    property  IsFixedHeight  : Boolean      read GetIsFixedHeight  write SetIsFixedHeight;
    property  FixedCheckMode : Integer      read GetFixedCheckMode write SetFixedCheckMode;
    property  FixedInterval  : Cardinal     read GetFixedInterval  write SetFixedInterval;
    property  DaqRef         : Integer      read GetDaqRef         write SetDaqRef;
    function  IsFormExposed:Boolean;
    function  IsFormViewable:Boolean;
    procedure LockDraw;
    procedure UnlockDraw;
    procedure UnlockDrawHidden;
    procedure StampFixedBounds;
    procedure CheckFixedBounds(Mode:Integer=$1F);
    procedure TimerFixedBounds;
    procedure DrawView; virtual;
    procedure PrintView; virtual;
    procedure FileSave; virtual;
    procedure FileSaveAs; virtual;
    procedure Print; virtual;
    function  GetPrintableImage:TBitmap; virtual;
  protected
    class var myLastActiveCrwChild:TFormCrwDaqSysChild;
  public
    class property LastActiveCrwChild:TFormCrwDaqSysChild read myLastActiveCrwChild;
    class function ActiveCrwChild:TFormCrwDaqSysChild;
  protected
    function  MrgMenu:TMainMenu;
    function  CanMenuMerge:Boolean;
    procedure MenuMerge(aMenu:TMainMenu; aMerge:Boolean);
    procedure CMActivate(var Message:TLMessage); message CM_ACTIVATE;
    procedure CMDeactivate(var Message:TLMessage); message CM_DEACTIVATE;
  public
    function CopyFormBmpToClipboard(Bmp:TBitmap; Verbose:Boolean):TClipboardFormat;
  public // DebugLog reporters
    function  GetBriefFormInfo(const What:LongString; Mode:Integer):LongString;
    procedure DebugLogReport_SdiForm(const What:LongString; Mode:Integer=0);
    procedure DebugLogReport_DrawView(Mode:Integer=0);
  public // Default mode for FixedCheckMode and CheckFixedBounds
    class var CheckFixedBoundsMode : Integer;
    class var FixedBoundsTolerance : TRect;
    class var HidesOnPrint : Integer;
    class var LTWHCopyOnFix : Boolean;
  public                                                // PendingTriggers:
    class function ptrg_FormHide              :Integer; // FormHide
    class function ptrg_FormShow              :Integer; // FormShow
    class function ptrg_FormActivate          :Integer; // FormActivate
    class function ptrg_FormDeactivate        :Integer; // FormDeactivate
    class function ptrg_FormChangeBounds      :Integer; // FormChangeBounds
    class function ptrg_FormWindowStateChange :Integer; // FormWindowStateChange
    class function ptrg_FormClose             :Integer; // FormClose
    class function ptrg_FocusInput            :Integer; // Focus input control
  end;

const                    // CodePills flags:
 cpf_GtkFCB = $00000001; // CodePills_GtkFakeChangeBounds

const                        // HidesOnPrint flags:
 hop_Menu       = $00000001; // Hide Menu
 hop_Toolbar    = $00000002; // Hide Toolbar
 hop_StatusBar  = $00000004; // Hide StatusBar
 hop_ScrollBars = $00000008; // Hide ScrollBarX,ScrollBarY
 hop_DrawView   = $00000010; // DrawView
 hop_ProcMess   = $00000020; // ProcessMessages
 hop_Default    = $0000003F; // Default HidesOnPrint value

const                                     // CheckFixedBounds Mode flags:
 cfbm_L    = $01;                         // Check Left
 cfbm_T    = $02;                         // Check Top
 cfbm_W    = $04;                         // Check Width
 cfbm_H    = $08;                         // Check Height
 cfbm_D    = $10;                         // Check Desktop
 cfbm_LTWH = cfbm_L+cfbm_T+cfbm_W+cfbm_H; // Check Left,Top,Width,Height

const // DebugLogReport Mode
 dlrm_Name        = $00000001;
 dlrm_Caption     = $00000002;
 dlrm_ClassName   = $00000004;
 dlrm_Visible     = $00000008;
 dlrm_WindowState = $00000010;
 dlrm_Bounds      = $00000020;
 dlrm_SdiForm_Default  = dlrm_ClassName+dlrm_Caption+dlrm_Visible+dlrm_WindowState+dlrm_Bounds;
 dlrm_DrawView_Default = dlrm_ClassName+dlrm_Caption+dlrm_Visible+dlrm_WindowState+dlrm_Bounds;

const                          // SdiFlags:
 sf_SdiMain       = $00000001; // Form is SDI Main  Form
 sf_SdiChild      = $00000002; // Form is SDI Child Form of any type
 sf_SdiControl    = $00000004; // Form is SDI Child uses for control
 sf_SdiConsole    = $00000008; // Form is SDI Child TFormConsoleWindow
 sf_SdiCircuit    = $00000010; // Form is SDI Child TFormCircuitWindow
 sf_SdiTabWin     = $00000020; // Form is SDI Child TFormTabWindow
 sf_SdiTextEdit   = $00000040; // Form is SDI Child TFormTextEditor
 sf_SdiSysCalc    = $00000080; // Form is SDI Child Main Console
 sf_SdiCurveWin   = $00000100; // Form is SDI Child TFormCurveWindow
 sf_SdiSurfWin    = $00000200; // Form is SDI Child TFormSurfWindow
 sf_SdiSpectrWin  = $00000400; // Form is SDI Child TFormSpectrWindow
 sf_SdiDaqCtrlDlg = $00000800; // Form is SDI Child TFormDaqControlDialog
 sf_SdiNoRestore  = $10000000; // Not restore form on Application.Restore

const                          // Mode for TSdiManager.SelectChildDialog
 sm_Default       = 0;         // Default
 sm_ZOrder        = 1;         // List in Z order

const                          // TSdiManager.ActivateChild Mode:
 ascm_Refresh     = $00000001; // Call RefreshWindows on Activate (any widgetset)
 ascm_GtkRefresh  = $00000002; // Call RefreshWindows on Activate (only GTK1/2/3)

 ///////////////////////////////////////////////////////////////////////////////
 // TSdiChildList: TSdiManager helper class which contains a list of SDI childs.
 ///////////////////////////////////////////////////////////////////////////////
type
 TSdiChildList = class(TMasterObject)
 private
  mySdiList : TList;
  myMinList : TList;
  function  GetCount:Integer;
  function  GetItems(i:Integer):TFormCrwDaqSysChild;
  procedure SetItems(i:Integer; aItem:TFormCrwDaqSysChild);
 public
  constructor Create;
  destructor  Destroy; override;
 public
  property  Count            : Integer             read GetCount;
  property  Items[i:Integer] : TFormCrwDaqSysChild read GetItems write SetItems; default;
 public
  procedure Add(aItem:TFormCrwDaqSysChild);
  procedure Remove(aItem:TFormCrwDaqSysChild);
  function  IndexOf(aItem:Pointer):Integer;
  function  IsChild(Form:TObject; const Incl:QWord=0; const Excl:QWord=0):Boolean;
 public
  function  Minimized:Integer;
  function  Minimize:Integer;
  function  Restore:Integer;
 public
  // Minimize SdiForms with given SdiFlags: includes Incl, excludes Excl.
  function  MinimizeByFlags(const Incl:QWord; const Excl:QWord=0):Integer;
 end;

type // GUI working mode: SDI, MDI.
 TGuiWorkingMode=(gwm_Default,gwm_SDI,gwm_MDI);

type // Iterator for TSdiManager.ForEachChild
 TForEachSdiChildAction = procedure(Form      : TForm;
                                    Index     : Integer;
                                var Terminate : Boolean;
                                    Custom    : Pointer);

 ///////////////////////////////////////////////////////////////////////////////
 // TSdiManager incapsulates paarameters and functions required for SDI support.
 // SdiMan is the only one TSdiManager instance of TSdiManager.
 // SDI is a Single Document Interface.
 ///////////////////////////////////////////////////////////////////////////////
type
 TSdiManager = class(TMasterObject)
 private
  myChildList : TSdiChildList;
  myActionOnActivateMainForm:TAction;
  function  GetChildList:TSdiChildList;
  function  GetChildCount:Integer;
  function  GetActionOnActivateMainForm:TAction;
  procedure SetActionOnActivateMainForm(aAction:TAction);
 public
  constructor Create;
  destructor  Destroy; override;
 public
  property  ChildList  : TSdiChildList read GetChildList;
  property  ChildCount : Integer       read GetChildCount;
  property  ActionOnActivateMainForm:TAction read GetActionOnActivateMainForm write SetActionOnActivateMainForm;
 public // Check Form is SDI child.
  function  IsChild(Form:TObject; const Incl:QWord=0; const Excl:QWord=0):Boolean;
 public // Iterator to apply Action for each SDI childs.
  function  ForEachChild(Action:TForEachSdiChildAction; Custom:Pointer; Backward:Boolean=false):Integer;
  function  ForEachChildInZOrder(Action:TForEachSdiChildAction; Custom:Pointer; Backward:Boolean=false):Integer;
 public // Find first SDI child in Z order.
  function  FindFirstChildInZOrder(Want,Skip:QWord; Backward:Boolean=false):TFormCrwDaqSysChild;
        // Find topmost SDI child, i.e. first SDI child form in Z order.
        // Parameter Want is flags which should present in Form.SdiFlags.
        // Parameter Skip is flags which must not present in Form.SdiFlags.
        // Note that the topmost SDI child may be invisible or minimized.
 public // Find topmost SDI child.
  function  TopMostChild(const Want:QWord=sf_SdiChild;
                         const Skip:QWord=0):TFormCrwDaqSysChild;
 public // Find active SDI child, i.e. topmost visible SDI child form.
  function  ActiveChild(const Want:QWord=sf_SdiChild;
                       const Skip:QWord=0):TFormCrwDaqSysChild;
 public
  function  UpdateAllChilds:Integer;
  function  SelectChildDialog(const Caption,Title:LongString; Mode:Integer=0; Params:LongString=''):TForm;
  function  FindActiveChild(out aChild:TForm; const Want,Skip:QWord):Boolean;
  function  ActivateChild(aForm:TForm; Mode:Integer=0):Boolean;
 public // Activate next SDI child in Z order.
  function  ActivateChildNext(Step:Integer):Boolean;
 public // Activate Application.MainForm if working in SDI mode.
  function ActivateMainForm:Boolean;
 public // Current GUI working mode: SDI, MDI.
  class var GuiWorkingMode:TGuiWorkingMode;
 public // GUI working in SDI mode?
  class function  IsSdiMode:Boolean; inline;
 public // GUI working in MDI mode?
  class function IsMdiMode:Boolean; inline;
 public
  class var EnableCheckMainFormIsExposed:Boolean;
  class var CursorActivateMainSdiForm:TCursor;
  class var EnableSdiFormMenuMerge:Boolean;
  class var HideChildSdiFormMenu:Boolean;
 end;

 ///////////////////////////////////////////////////////////////////////////////
 // SdiMan is the only one TSdiManager instance.
 ///////////////////////////////////////////////////////////////////////////////
function SdiMan:TSdiManager;

 // DebugLog channel for SDI Forms.
function dlc_SdiForm:Integer; inline;

// DebugLog channel for Form.DrawView.
function dlc_DrawView:Integer; inline;

// DebugLog channel for Form.CheckFixedBounds.
function dlc_FixBound:Integer; inline;

 // Timer procedure to be called at every second
 // to check fixed position/size of SDI children.
procedure Timer_CheckFixedBoundsSdiChildren;

implementation

uses
 Form_CrwDaq,
 Form_TextEditor,
 Form_CurveWindow,
 Form_ListBoxSelection;

{$R *.lfm}

///////////////////
// Utility routines
///////////////////

const
 LockUnlockBalance : TAtomicCounter = nil;

procedure InitSdiCounters;
begin
 LockedInit(LockUnlockBalance);
end;

procedure FreeSdiCounters;
begin
 LockedFree(LockUnlockBalance);
end;

const // DebugLog channels
 dlc_SdiForm_  : Integer = 0;
 dlc_DrawView_ : Integer = 0;
 dlc_FixBound_ : Integer = 0;

function dlc_SdiForm:Integer;
begin
 Result:=dlc_SdiForm_;
end;

function dlc_DrawView:Integer;
begin
 Result:=dlc_DrawView_;
end;

function dlc_FixBound:Integer;
begin
 Result:=dlc_FixBound_;
end;

procedure DoCheckFixedBoundsSdiChildren(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if (Form is TFormCrwDaqSysChild)
 then TFormCrwDaqSysChild(Form).TimerFixedBounds;
end;

procedure Timer_CheckFixedBoundsSdiChildren;
begin
 SdiMan.ForEachChild(DoCheckFixedBoundsSdiChildren,nil);
end;

//////////////////////
// TFormCrwDaqSysChild
//////////////////////

procedure TFormCrwDaqSysChild.AfterConstruction;
begin
 inherited AfterConstruction;
 AddonSdiFlags(sf_SdiChild);
 SdiMan.ChildList.Add(Self);
 FixedInterval:=1000;
 FixedCheckMode:=CheckFixedBoundsMode;
end;

procedure TFormCrwDaqSysChild.BeforeDestruction;
begin
 if (myLastActiveCrwChild=Self) then myLastActiveCrwChild:=nil;
 MenuMerge(MrgMenu,false);
 SdiMan.ChildList.Remove(Self);
 inherited BeforeDestruction;
end;

class function TFormCrwDaqSysChild.ActiveCrwChild:TFormCrwDaqSysChild;
begin
 Result:=nil;
 if Assigned(myLastActiveCrwChild) then
 if SdiMan.IsChild(myLastActiveCrwChild)
 then Result:=myLastActiveCrwChild
 else myLastActiveCrwChild:=nil;
end;

function TFormCrwDaqSysChild.CanMenuMerge:Boolean;
begin
 Result:=false;
 if Assigned(Application) then
 if SdiMan.EnableSdiFormMenuMerge then
 if Assigned(Application.MainForm) then
 if Assigned(Application.MainForm.Menu) then
 if not(csDesigning in ComponentState) and (FormStyle<>fsMDIChild)
 and (Application.MainForm.FormStyle<>fsMDIForm) then Result:=true;
end;

procedure TFormCrwDaqSysChild.MenuMerge(aMenu:TMainMenu; aMerge:Boolean);
begin
 if not Assigned(Self) then Exit;
 if not Assigned(aMenu) then Exit;
 if not Assigned(Application) then Exit;
 if not SdiMan.EnableSdiFormMenuMerge then Exit;
 if not Assigned(Application.MainForm) then Exit;
 if not Assigned(Application.MainForm.Menu) then Exit;
 try
  if aMerge
  then Application.MainForm.Menu.Merge(aMenu)
  else Application.MainForm.Menu.Unmerge(aMenu);
 except
  on E:Exception do BugReport(E,Self,'MenuMerge');
 end;
end;

function TFormCrwDaqSysChild.GetBriefFormInfo(const What:LongString; Mode:Integer):LongString;
begin
 Result:=What;
 if Assigned(Self) and (Mode<>0) then begin
  if HasFlags(Mode,dlrm_Name) then Result:=Format('%s, %s',[Result,Name]);
  if HasFlags(Mode,dlrm_ClassName) then Result:=Format('%s, %s',[Result,ClassName]);
  if HasFlags(Mode,dlrm_Caption) then Result:=Format('%s, %s',[Result,AnsiQuotedStr(Caption,QuoteMark)]);
  if HasFlags(Mode,dlrm_Bounds) then Result:=Format('%s, (%d,%d,%d,%d)',[Result,Left,Top,Width,Height]);
  if HasFlags(Mode,dlrm_Visible) then Result:=Format('%s, %s',[Result,ExtractWord(1+Ord(Visible),'Hidden,Visible',ScanSpaces)]);
  if HasFlags(Mode,dlrm_WindowState) then Result:=Format('%s, %s',[Result,GetEnumName(TypeInfo(TWindowState),Ord(WindowState))]);
  if (What='') then Result:=TrimChars(Result,ScanSpaces,ScanSpaces);
 end else Result:='';
end;

procedure TFormCrwDaqSysChild.DebugLogReport_SdiForm(const What:LongString; Mode:Integer=0);
begin
 if Assigned(Self) and DebugLogEnabled(dlc_SdiForm) then begin
  if (Mode=0) then Mode:=dlrm_SdiForm_Default;
  DebugLog(dlc_SdiForm,GetBriefFormInfo(What,Mode));
 end;
end;

procedure TFormCrwDaqSysChild.DebugLogReport_DrawView(Mode:Integer=0);
begin
 if Assigned(Self) and DebugLogEnabled(dlc_DrawView) then begin
  if (Mode=0) then Mode:=dlrm_DrawView_Default;
  DebugLog(dlc_DrawView,GetBriefFormInfo('',Mode));
 end;
end;

procedure TFormCrwDaqSysChild.CMActivate(var Message:TLMessage);
begin
 DebugLogReport_SdiForm('CMActivate');
 myLastActiveCrwChild:=Self;
 if CanMenuMerge then MenuMerge(MrgMenu,true);
 inherited CMActivate(Message);
end;

procedure TFormCrwDaqSysChild.CMDeactivate(var Message:TLMessage);
begin
 DebugLogReport_SdiForm('CMDeactivate');
 inherited CMDeactivate(Message);
 //if CanMenuMerge then MenuMerge(MrgMenu,false);
 //if (myLastActiveCrwChild=Self) then myLastActiveCrwChild:=nil;
end;

function TFormCrwDaqSysChild.GetCloseAction:TCloseAction;
begin
 if Assigned(Self) then Result:=myCloseAction else Result:=caNone;
end;

procedure TFormCrwDaqSysChild.SetCloseAction(aAction:TCloseAction);
begin
 if Assigned(Self) then myCloseAction:=aAction;
end;

function TFormCrwDaqSysChild.GetMonitorEvent:Boolean;
var CurrTickCount:QWord;
begin
 Result:=false;
 if Assigned(Self) then begin
  CurrTickCount:=GetTickCount64;
  if (CurrTickCount>myMonitorTicks+myMonitorPeriod) then begin
   myMonitorTicks:=CurrTickCount;
   Result:=true;
  end;
 end;
end;

function   TFormCrwDaqSysChild.GetMonitorPeriod:Cardinal;
begin
 if Assigned(Self) then Result:=myMonitorPeriod else Result:=0;
end;

procedure  TFormCrwDaqSysChild.SetMonitorPeriod(aPeriod:Cardinal);
begin
 if Assigned(Self) then myMonitorPeriod:=aPeriod;
end;

function TFormCrwDaqSysChild.GetFixedRect:TRect;
begin
 if Assigned(Self)
 then Result:=myFixed.Rect
 else Result:=Rect(0,0,0,0);
end;

function TFormCrwDaqSysChild.GetFixedLeft:Integer;
begin
 if Assigned(Self)
 then Result:=myFixed.Rect.Left
 else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetFixedLeft(L:Integer);
begin
 if Assigned(Self) then myFixed.Rect.Left:=L;
end;

function TFormCrwDaqSysChild.GetFixedTop:Integer;
begin
 if Assigned(Self)
 then Result:=myFixed.Rect.Top
 else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetFixedTop(T:Integer);
begin
 if Assigned(Self) then myFixed.Rect.Top:=T;
end;

function TFormCrwDaqSysChild.GetFixedWidth:Integer;
begin
 if Assigned(Self)
 then Result:=myFixed.Rect.Width
 else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetFixedWidth(W:Integer);
begin
 if Assigned(Self) then myFixed.Rect.Width:=W;
end;

function TFormCrwDaqSysChild.GetFixedHeight:Integer;
begin
 if Assigned(Self)
 then Result:=myFixed.Rect.Height
 else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetFixedHeight(H:Integer);
begin
 if Assigned(Self) then myFixed.Rect.Height:=H;
end;

function TFormCrwDaqSysChild.GetFixedWhat:Integer;
begin
 if Assigned(Self)
 then Result:=myFixed.What
 else Result:=0;
end;

function TFormCrwDaqSysChild.GetIsFixedLeft:Boolean;
begin
 if Assigned(Self)
 then Result:=HasFlags(myFixed.What,cfbm_L)
 else Result:=False;
end;

procedure TFormCrwDaqSysChild.SetIsFixedLeft(aFixedLeft:Boolean);
begin
 if Assigned(Self) then begin
  ActionViewFixedLeft.Checked:=aFixedLeft;
  LiftFlags(myFixed.What,cfbm_L,aFixedLeft);
  if aFixedLeft and LTWHCopyOnFix then FixedLeft:=Left;
 end;
end;

function TFormCrwDaqSysChild.GetIsFixedTop:Boolean;
begin
 if Assigned(Self)
 then Result:=HasFlags(myFixed.What,cfbm_T)
 else Result:=False;
end;

procedure TFormCrwDaqSysChild.SetIsFixedTop(aFixedTop:Boolean);
begin
 if Assigned(Self) then begin
  ActionViewFixedTop.Checked:=aFixedTop;
  LiftFlags(myFixed.What,cfbm_T,aFixedTop);
  if aFixedTop and LTWHCopyOnFix then FixedTop:=Top;
 end;
end;

function TFormCrwDaqSysChild.GetIsFixedWidth:Boolean;
begin
 if Assigned(Self)
 then Result:=HasFlags(myFixed.What,cfbm_W)
 else Result:=False;
end;

procedure TFormCrwDaqSysChild.SetIsFixedWidth(aFixedWidth:Boolean);
begin
 if Assigned(Self) then begin
  ActionViewFixedWidth.Checked:=aFixedWidth;
  LiftFlags(myFixed.What,cfbm_W,aFixedWidth);
  if aFixedWidth and LTWHCopyOnFix then FixedWidth:=Width;
 end;
end;

function TFormCrwDaqSysChild.GetIsFixedHeight:Boolean;
begin
 if Assigned(Self)
 then Result:=HasFlags(myFixed.What,cfbm_H)
 else Result:=False;
end;

procedure TFormCrwDaqSysChild.SetIsFixedHeight(aFixedHeight:Boolean);
begin
 if Assigned(Self) then begin
  ActionViewFixedHeight.Checked:=aFixedHeight;
  LiftFlags(myFixed.What,cfbm_H,aFixedHeight);
  if aFixedHeight and LTWHCopyOnFix then FixedHeight:=Height;
 end;
end;

function TFormCrwDaqSysChild.GetFixedCheckMode:Integer;
begin
 if Assigned(Self)
 then Result:=myFixed.CheckMode
 else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetFixedCheckMode(aMode:Integer);
begin
 myFixed.CheckMode:=aMode;
end;

function TFormCrwDaqSysChild.GetFixedInterval:Cardinal;
begin
 if Assigned(Self) then Result:=myFixed.Interval else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetFixedInterval(aInterval:Cardinal);
begin
 myFixed.Interval:=aInterval;
end;

function TFormCrwDaqSysChild.GetDaqRef:Integer;
begin
 if Assigned(Self) then Result:=myDaqRef else Result:=0;
end;

procedure TFormCrwDaqSysChild.SetDaqRef(aDaqRef:Integer);
begin
 if Assigned(Self) then myDaqRef:=aDaqRef;
end;

function TFormCrwDaqSysChild.NeedsCodePills(aPills:LongWord; aReset:Boolean=true):Boolean;
begin
 Result:=false;
 if Assigned(Self) and CodePillsEnabled then begin
  Result:=HasFlags(myCodePills,aPills);
  if aReset then LiftFlags(myCodePills,aPills,false);
 end;
end;

function TFormCrwDaqSysChild.TakesCodePills(aPills:LongWord):LongWord;
begin
 Result:=0;
 if Assigned(Self) and CodePillsEnabled then begin
  if HasFlags(aPills,cpf_GtkFCB) then begin
   if CodePills_GtkFakeChangeBounds
   then LiftFlags(Result,cpf_GtkFCB,true);
  end;
 end;
end;

procedure TFormCrwDaqSysChild.PrescribeCodePills(aPills:LongWord);
begin
 if Assigned(Self) and CodePillsEnabled
 then LiftFlags(myCodePills,aPills,true);
end;

 // Gtk widgets required CodePills: fake change bounds,
 // uses to refresh window content after hide/minimize.
function TFormCrwDaqSysChild.CodePills_GtkFakeChangeBounds:Boolean;
var R1,R2:TRect; const dx=1; dy=1;
begin
 Result:=false;
 if Assigned(Self) and CodePillsEnabled then
 if IsUnix and IsWidgetSetName('gtk,gtk2,gtk3') then begin
  DebugLogReport_SdiForm('CodePills_GtkFakeChangeBounds');
  R1:=BoundsRect; R2:=R1; R2.Width:=R2.Width+dx; R2.Height:=R2.Height+dy;
  BoundsRect:=R2; BoundsRect:=R1; // Fake change bounds.
  Result:=true;
 end;
end;

procedure TFormCrwDaqSysChild.DoUpdateTop(T:Integer);
begin
 Top:=T;
 FixedTop:=T;
end;

procedure TFormCrwDaqSysChild.DoUpdateLeft(L:Integer);
begin
 Left:=L;
 FixedLeft:=L;
end;

procedure TFormCrwDaqSysChild.DoUpdateWidth(W:Integer);
begin
 Width:=W;
 FixedWidth:=W;
end;

procedure TFormCrwDaqSysChild.DoUpdateHeight(H:Integer);
begin
 Height:=H;
 FixedHeight:=H;
end;

procedure TFormCrwDaqSysChild.RefreshWindow;
begin
 if Assigned(Self) then begin
  DebugLogReport_SdiForm('RefreshWindow');
  if (TakesCodePills(cpf_GtkFCB)=0) then Refresh;
 end;
end;

function TFormCrwDaqSysChild.MrgMenu:TMainMenu;
begin
 if Assigned(Self)
 then Result:=MainMenu
 else Result:=nil;
end;

procedure TFormCrwDaqSysChild.FormCreate(Sender: TObject);
begin
 myLockCounter:=0;
 myMonitorTicks:=0;
 myMonitorPeriod:=0;
 SetStandardFont(Self);
 SetAllButtonsCursor(Self,crHandPoint);
 PrepareCrwDaqSdiMenu(Self);
 if SdiMan.IsSdiMode then ToolBar.Cursor:=SdiMan.CursorActivateMainSdiForm;
 if SdiMan.IsSdiMode then StatusBar.Cursor:=SdiMan.CursorActivateMainSdiForm;
 UpdateMenu(MenuFile,
            RusEng('Файл','File')+MenuRightSpace,
            RusEng('Меню операций с файлами.','File operations.'),
            0);
 UpdateMenu(MenuFileNew,
            RusEng('Создать','New'),
            RusEng('Меню создания нового окна','Menu to create new window'),
            0);
 UpdateMenu(MenuFileNewTextEditor,
            RusEng('Текстовое окно','Text editor'),
            RusEng('Создать новое окно с текстом','Create text editor'),
            ShortCut(Word('N'),[ssCtrl]));
 UpdateMenu(MenuFileNewCurveWindow,
            RusEng('Окно с кривыми','Curve window'),
            RusEng('Создать новое окно с кривыми','Create window with curves'),
            ShortCut(Word('N'),[ssCtrl,ssAlt]));
 UpdateMenu(MenuFileNewDataAnalysisPlugin,
            RusEng('Утилиту анализа данных','Data analysis plugin'),
            RusEng('Создать новую утилиту анализа данных','Create new data analysis plugin'),
            ShortCut(Word('N'),[ssCtrl,ssShift]));
 UpdateMenu(MenuFileNewDataAcquisitionPlugin,
            RusEng('Утилиту сбора данных','Data acquisition plugin'),
            RusEng('Создать новую утилиту сбора данных','Create new data acquisition plugin'),
            ShortCut(Word('N'),[ssShift,ssAlt]));
 UpdateMenu(MenuFileNewDaqCreator,
            RusEng('Систему сбора данных с нуля - DAQ Creator','Data Acquisition System - DAQ Creator'),
            RusEng('Утилита для создания новой DAQ системы с нуля','Utility to create new DAQ system from scratch'),
            0);
 UpdateMenu(MenuFileNewDaqConstructor,
            RusEng('Элементы DAQ Системы - DAQ Constructor','Elements of DAQ System - DAQ Constructor'),
            RusEng('Утилита для конструирования элементов DAQ системы','Utility to construct elements of DAQ system'),
            0);
 UpdateMenu(MenuFileNewProjectFromSample,
            RusEng('Новый Проект по Образцу','New Project from Sample'),
            RusEng('Создать новый Проект по Образцу.','Create new Project from Sample.'),
            0);
 UpdateMenu(MenuFileOpen,
            RusEng('Открыть …','Open …'),
            RusEng('Открыть файл …','Open file …'),
            ShortCut(Word('O'),[ssCtrl]));
 UpdateMenu(MenuFileSave,
            RusEng('Сохранить','Save'),
            RusEng('Сохранить в файл','Save to file'),
            ShortCut(VK_F2,[]));
 UpdateMenu(MenuFileSaveAs,
            RusEng('Сохранить как …','Save as …'),
            RusEng('Сохранить в новый файл','Save to new file'),
            ShortCut(Word('S'),[ssCtrl,ssShift]));
 UpdateMenu(MenuFilePrint,
            RusEng('Печатать','Print'),
            RusEng('Напечатать содержимое окна','Print window'),
            ShortCut(Word('P'),[ssCtrl]));
 UpdateMenu(MenuFilePrinterSetup,
            RusEng('Уставки печати','Printer setup'),
            RusEng('Выбрать параметры принтера','Setup printer options'),
            ShortCut(Word('P'),[ssCtrl,ssAlt]));
 UpdateMenu(MenuFileExit,
            RusEng('Выход','Exit'),
            RusEng('Выход из программы.','Exit from program.'),
            ShortCut(Word('X'), [ssAlt]));
 UpdateMenu(MenuView,
            RusEng('Вид','View')+MenuRightSpace,
            RusEng('Меню параметров отображения.','View options menu.'),
            0);
 UpdateMenu(MenuViewFixed,
            RusEng('Фиксировать','Fixed'),
            RusEng('Меню для функций фиксации параметров.','Fixed parameters operations.'),
            0);
 UpdateMenu(MenuViewFixedLeft,
            RusEng('Положение X','Location X'),
            RusEng('Фиксировать положение окна по X.','Fixed window location X.'),
            ShortCut(Word('X'),[ssCtrl,ssAlt]));
 UpdateMenu(MenuViewFixedTop,
            RusEng('Положение Y','Location Y'),
            RusEng('Фиксировать положение окна по Y.','Fixed window location Y.'),
            ShortCut(Word('Y'),[ssCtrl,ssAlt]));
 UpdateMenu(MenuViewFixedWidth,
            RusEng('Ширина','Width'),
            RusEng('Фиксировать ширину окна.','Fixed window width.'),
            ShortCut(Word('W'),[ssCtrl,ssAlt]));
 UpdateMenu(MenuViewFixedHeight,
            RusEng('Высота','Height'),
            RusEng('Фиксировать высоту окна.','Fixed window height.'),
            ShortCut(Word('H'),[ssCtrl,ssAlt]));
 CloseAction:=caFree;
 BorderIcons:=BorderIcons-[biMaximize];
 myDaqRef:=0;
end;

procedure TFormCrwDaqSysChild.FormClose(Sender: TObject; var Action: TCloseAction);
begin
 PendingTrigger[ptrg_FormClose]:=true;
 Action:=CloseAction;
end;

procedure TFormCrwDaqSysChild.FormChangeBounds(Sender: TObject);
begin
 if Assigned(Self) then begin
  PendingTrigger[ptrg_FormChangeBounds]:=true;
  DebugLogReport_SdiForm('FormChangeBounds');
  LockDraw;
  UnlockDraw;
  StampFixedBounds;
 end;
end;

procedure TFormCrwDaqSysChild.FormHide(Sender: TObject);
begin
 if Assigned(Self) then begin
  PendingTrigger[ptrg_FormHide]:=true;
  DebugLogReport_SdiForm('FormHide');
  PrescribeCodePills(cpf_GtkFCB);
  inherited;
 end;
end;

procedure TFormCrwDaqSysChild.FormShow(Sender: TObject);
begin
 if Assigned(Self) then begin
  PendingTrigger[ptrg_FormShow]:=true;
  DebugLogReport_SdiForm('FormShow');
  PrescribeCodePills(cpf_GtkFCB);
  inherited;
 end;
end;

procedure TFormCrwDaqSysChild.FormActivate(Sender: TObject);
begin
 if Assigned(Self) then begin
  PendingTrigger[ptrg_FormActivate]:=true;
  DebugLogReport_SdiForm('FormActivate');
  inherited;
  if NeedsCodePills(cpf_GtkFCB) then TakesCodePills(cpf_GtkFCB);
 end;
end;

procedure TFormCrwDaqSysChild.FormDeactivate(Sender: TObject);
begin
 if Assigned(Self) then begin
  PendingTrigger[ptrg_FormDeactivate]:=true;
  DebugLogReport_SdiForm('FormDeactivate');
  PrescribeCodePills(cpf_GtkFCB);
  inherited;
 end;
end;

procedure TFormCrwDaqSysChild.FormWindowStateChange(Sender: TObject);
begin
 if Assigned(Self) then begin
  PendingTrigger[ptrg_FormWindowStateChange]:=true;
  DebugLogReport_SdiForm('FormWindowStateChange');
  inherited;
  LockDraw;
  UnlockDraw;
  StampFixedBounds;
  PrescribeCodePills(cpf_GtkFCB);
 end;
end;

procedure TFormCrwDaqSysChild.UpdateCommands;
begin
 if Assigned(FormCrwDaq) then begin
  FormCrwDaq.ActionFileSave.Enabled:=ActionFileSave.Enabled;
  FormCrwDaq.ActionFileSaveAs.Enabled:=ActionFileSaveAs.Enabled;
  FormCrwDaq.ActionFilePrint.Enabled:=ActionFilePrint.Enabled;
 end;
end;

procedure TFormCrwDaqSysChild.LockDraw;
begin
 if Assigned(Self) then begin
  inc(myLockCounter);
  LockedInc(LockUnlockBalance);
 end;
end;

procedure TFormCrwDaqSysChild.UnlockDraw;
begin
 if Assigned(Self) then begin
  LockedDec(LockUnlockBalance);
  if (myLockCounter>0) then dec(myLockCounter) else myLockCounter:=0;
  if (myLockCounter=0) then DrawView;
 end;
end;

procedure TFormCrwDaqSysChild.UnlockDrawHidden;
begin
 if Assigned(Self) then begin
  LockedDec(LockUnlockBalance);
  if myLockCounter>0 then dec(myLockCounter) else myLockCounter:=0;
 end;
end;

procedure TFormCrwDaqSysChild.StampFixedBounds;
begin
 if Assigned(Self) then myFixed.TimeStamp:=GetTickCount64;
end;

procedure TFormCrwDaqSysChild.CheckFixedBounds(Mode:Integer=$1F);
var BR,FR,NR:TRect; ad,wd,What:Integer; msg:LongString;
begin
 if Assigned(Self) then
 if FormIsExposed(Self) then begin
  if HasFlags(Mode,cfbm_D) and IsUnix and (WmWnd<>0) then begin
   ad:=wmctrl.ActiveDesktop; wd:=WmDesktop;
   if (ad<>wd) then Exit;
  end;
  What:=FixedWhat and Mode;
  if HasFlags(What,cfbm_LTWH) then begin
   BR:=BoundsRect; NR:=BR; FR:=FixedRect;
   if HasFlags(What,cfbm_T) and (NR.Top<>FR.Top) then NR.Top:=FR.Top;
   if HasFlags(What,cfbm_L) and (NR.Left<>FR.Left) then NR.Left:=FR.Left;
   NR.Size:=BR.Size; // NR.Size must be updated after NR.LeftTop changing
   if HasFlags(What,cfbm_W) and (NR.Width<>FR.Width) then NR.Width:=FR.Width;
   if HasFlags(What,cfbm_H) and (NR.Height<>FR.Height) then NR.Height:=FR.Height;
   if not IsAboutTheSame(NR.Size,BR.Size,FixedBoundsTolerance.Size)
   or not IsAboutTheSame(NR.Location,BR.Location,FixedBoundsTolerance.Location) then begin
    if DebugLogEnabled(dlc_FixBound) then begin
     msg:=Format('%s BR(%d,%d,%d,%d) NR(%d,%d,%d,%d), FR(%d,%d,%d,%d)',
                [Caption,BR.Left,BR.Top,BR.Width,BR.Height,
                         NR.Left,NR.Top,NR.Width,NR.Height,
                         FR.Left,FR.Top,FR.Width,FR.Height]);
     DebugLog(dlc_FixBound,msg);
    end;
    BoundsRect:=NR;
   end;
  end;
 end;
end;

//
// TForm.FormCanResize is not implemented in Lazarus.
// So TimerFixedBounds is uses instead.
//
procedure TFormCrwDaqSysChild.TimerFixedBounds;
var Ticks:QWord;
begin
 if Assigned(Self) then begin
  Ticks:=GetTickCount64;
  if (Ticks>=myFixed.TimeStamp+myFixed.Interval) then begin
   CheckFixedBounds(myFixed.CheckMode);
   myFixed.TimeStamp:=Ticks;
  end;
 end;
end;

procedure TFormCrwDaqSysChild.DrawView;
begin
 Update;
end;

function TFormCrwDaqSysChild.CopyFormBmpToClipboard(Bmp:TBitmap; Verbose:Boolean):TClipboardFormat;
var Png:TPortableNetworkGraphic; Msg:LongString;
begin
 Result:=0;
 if Assigned(Bmp) then
 if Assigned(Self) then
 try
  Clipboard.Clear;
  // Windows prefer BMP.
  if IsWindows then begin
   Clipboard.Assign(Bmp);
  end;
  // Unix prefer PNG.
  if IsUnix then begin
   Png:=TPortableNetworkGraphic.Create;
   if Assigned(Png) then
   try
    Png.Assign(Bmp);
    Clipboard.Assign(Png);
   finally
    Png.Free;
   end;
  end;
  Result:=Clipboard.FindPictureFormatID;
  if Verbose then begin
   if (Result<>0) then begin
    Msg:=RusEng('Окно ','Form ')+AnsiQuotedStr(Caption,QuoteMark)
        +RusEng(' скопировано в Буфер Обмена как ',' copied to Clipboard as ')
        +ClipboardFormatToMimeType(Result);
    Echo(Msg);
   end else begin
    Msg:=RusEng('Ошибка копирования изображения в Буфер Обмена.',
                'Error on copy image to Clipboard.');
    Echo(Msg);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'CopyFormBmpToClipboard');
 end;
end;

function TFormCrwDaqSysChild.GetPrintableImage:TBitmap;
begin
 Result:=nil;
 if Assigned(Self) then
 try
  Result:=GetFormImage;
 except
  on E:Exception do BugReport(E,Self,'GetPrintableImage');
 end;
end;

procedure TFormCrwDaqSysChild.PrintView;
var Bmp:TBitmap; Key:Integer; mm:TMainMenu;
var Params:LongString; fsp:TPoint;
begin
 fsp:=ClientToScreen(Point(0,0));
 Params:='@set Panel.Font   Name:PT_Mono\Size:12\Color:Navy\Style:[Bold]'+EOL
        +'@set ListBox.Font Name:PT_Mono\Size:12\Color:Black\Style:[Bold]'+EOL
        +'@set Form.Left '+IntToStr(fsp.x)+' relative Screen'+EOL
        +'@set Form.Top  '+IntToStr(fsp.y)+' relative Screen'+EOL;
 Key:=ListBoxMenu(RusEng('Файл\Печать','File\Print'),
                  RusEng('Как печатать','How to print'),
                  RusEng('Скопировать изображение в Буфер Обмена (цвет.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (серый)'+EOL+
                         'Напечатать на Принтере '+Printer.PrinterName,
                         'Copy bitmap to Clipboard (color)'+EOL+
                         'Copy bitmap to Clipboard (gray)'+EOL+
                         'Print hardcopy to Printer '+Printer.PrinterName),
                         myPrintViewKey,Params);
 if (Key>=0) then
 try
  myPrintViewKey:=Key;
  mm:=Menu;
  try
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=nil;
   if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
   if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
   case Key of
    0,1: begin
          Bmp:=GetPrintableImage;
          if Assigned(Bmp) then
          try
           if (Key=1) then ConvertBmpToBlackAndWhite(Bmp,clWindow,clBtnFace,3);
           CopyFormBmpToClipboard(Bmp,true);
          finally
           Kill(Bmp);
          end;
         end;
    2:   if HasPrintersDialog then Print;
   end;
  finally
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=mm;
  end;
  if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
  if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
 except
  on E:Exception do BugReport(E,Self,'PrintView');
 end;
end;

procedure TFormCrwDaqSysChild.Print;
var bmp:TBitmap; ppw,pph,ax,ay,bx,by,pw,ph:Integer; scale,piw,pih:Double;
var msg:LongString;
begin
 if Assigned(Self) then
 if Assigned(Screen) then
 if Assigned(Printer) then
 try
  bmp:=GetPrintableImage;
  if Assigned(Bmp) then
  try
    Printer.BeginDoc;
    try
     scale:=PrinterPageSettings.Scale;
     ax:=Trunc(CentimeterToInch(PrinterPageSettings.Indent.a.x)*Printer.XDPI);
     ay:=Trunc(CentimeterToInch(PrinterPageSettings.Indent.a.y)*Printer.YDPI);
     bx:=Trunc(CentimeterToInch(PrinterPageSettings.Indent.b.x)*Printer.XDPI);
     by:=Trunc(CentimeterToInch(PrinterPageSettings.Indent.b.y)*Printer.YDPI);
     ppw:=Printer.PageWidth-ax-bx;
     pph:=Printer.PageHeight-ay-by;
     if (ppw<10) or (pph<10) or (scale<1e-3) or (scale>1e3)
     then Raise EEchoException.Create(RusEng('Неверные параметры страницы',
                                             'Invalid page settings'));
     piw:=bmp.Width*(Printer.XDPI/Screen.PixelsPerInch);
     pih:=bmp.Height*(Printer.YDPI/Screen.PixelsPerInch);
     if (PrinterPageSettings.Adjust>0) then begin
      if (piw>ppw) then scale:=Min(scale,ppw/piw);
      if (pih>pph) then scale:=Min(scale,pph/pih);
     end;
     pw:=Trunc(piw*scale); ph:=Trunc(pih*scale);
     Printer.Canvas.StretchDraw(Rect(ax,ay,ax+pw,ay+ph),bmp);
    finally
     Printer.EndDoc;
    end;
    msg:=RusEng('Окно ','Form ')+AnsiQuotedStr(Caption,QuoteMark)
        +RusEng(' отправлено на Принтер: ',' sent to Printer: ')+Printer.PrinterName;
    Echo(msg);
  finally
   Kill(bmp);
  end;
 except
  on E:Exception do BugReport(E,Self,'Print');
 end;
end;

procedure TFormCrwDaqSysChild.FileSave;
begin
end;

procedure TFormCrwDaqSysChild.FileSaveAs;
begin
end;

procedure TFormCrwDaqSysChild.SaveDialogTypeChange(Sender: TObject);
begin
 OpenDialogTypeChangeStdExecute(SaveDialog);
end;

function TFormCrwDaqSysChild.IsFormExposed:Boolean;
begin
 if Assigned(Self) then Result:=FormIsExposed(Self) else Result:=false;
 if Result and SdiMan.EnableCheckMainFormIsExposed then Result:=ApplicationMainFormIsExposed;
end;

function TFormCrwDaqSysChild.IsFormViewable:Boolean;
begin
 Result:=false;
 if Assigned(Self) then begin
  if (WindowState=wsMinimized) then Exit;
  //if not Visible then Exit;
  Result:=true;
 end;
end;

class function TFormCrwDaqSysChild.ptrg_FormHide:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FormShow:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FormActivate:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FormDeactivate:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FormChangeBounds:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FormWindowStateChange:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FormClose:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

class function TFormCrwDaqSysChild.ptrg_FocusInput:Integer;
const Trigger:Integer=0;
begin
 if (Trigger=0) then Trigger:=RegisterPendingTrigger;
 Result:=Trigger;
end;

 {
 *******************************************************************************
 Actions implementation
 *******************************************************************************
 }
procedure TFormCrwDaqSysChild.ActionListUpdate(Action: TBasicAction; var Handled: Boolean);
begin
 Handled:=true;
end;

procedure TFormCrwDaqSysChild.ActionFileNewTextEditorExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewTextEditor.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileNewCurveWindowExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewCurveWindow.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileNewDaqCreatorExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewDaqCreator.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileNewDaqConstructorExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewDaqConstructor.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileNewProjectFromSampleExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewProjectFromSample.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileNewDataAnalysisPluginExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewDataAnalysisPlugin.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileNewDataAcquisitionPluginExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileNewDataAcquisitionPlugin.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileOpenExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileOpen.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileSaveExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionFileSave)<0 then Exit;
 if Ok then FileSave;
end;

procedure TFormCrwDaqSysChild.ActionFileSaveAsExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionFileSaveAs)<0 then Exit;
 if Ok then FileSaveAs;
end;

procedure TFormCrwDaqSysChild.ActionFilePrintExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionFilePrint)<0 then Exit;
 if Ok then PrintView;
end;

procedure TFormCrwDaqSysChild.ActionFilePrinterSetupExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFilePrinterSetup.Execute;
end;

procedure TFormCrwDaqSysChild.ActionFileExitExecute(Sender: TObject);
begin
 if Ok and FormCrwDaq.Ok then FormCrwDaq.ActionFileExit.Execute;
end;

procedure TFormCrwDaqSysChild.ActionViewFixedLeftExecute(Sender: TObject);
begin
 if Ok then begin
  IsFixedLeft:=not IsFixedLeft;
  if IsFixedLeft then FixedLeft:=Left;
 end;
end;

procedure TFormCrwDaqSysChild.ActionViewFixedTopExecute(Sender: TObject);
begin
 if Ok then begin
  IsFixedTop:=not IsFixedTop;
  if IsFixedTop then FixedTop:=Top;
 end;
end;

procedure TFormCrwDaqSysChild.ActionViewFixedWidthExecute(Sender: TObject);
begin
 if Ok then begin
  IsFixedWidth:=not IsFixedWidth;
  if IsFixedWidth then FixedWidth:=Width;
 end;
end;

procedure TFormCrwDaqSysChild.ActionViewFixedHeightExecute(Sender: TObject);
begin
 if Ok then begin
  IsFixedHeight:=not IsFixedHeight;
  if IsFixedHeight then FixedHeight:=Height;
 end;
end;

procedure TFormCrwDaqSysChild.StatusBarDblClick(Sender: TObject);
begin
 inherited;
 SdiMan.ActivateMainForm;
end;

procedure TFormCrwDaqSysChild.ToolBarDblClick(Sender: TObject);
begin
 inherited;
 SdiMan.ActivateMainForm;
end;

////////////////
// TSdiChildList
////////////////

constructor TSdiChildList.Create;
begin
 inherited Create;
 mySdiList:=TList.Create;
 myMinList:=TList.Create;
end;

destructor TSdiChildList.Destroy;
begin
 Kill(myMinList);
 Kill(mySdiList);
 inherited Destroy;
end;

function TSdiChildList.GetCount:Integer;
begin
 if Assigned(Self) and Assigned(mySdiList)
 then Result:=mySdiList.Count
 else Result:=0;
end;

function TSdiChildList.GetItems(i:Integer):TFormCrwDaqSysChild;
var Obj:TObject;
begin
 if Assigned(Self) and Assigned(mySdiList) and (i>=0) and (i<mySdiList.Count)
 then Obj:=mySdiList[i]
 else Obj:=nil;
 if (Obj is TFormCrwDaqSysChild)
 then Result:=TFormCrwDaqSysChild(Obj)
 else Result:=nil;
end;

procedure TSdiChildList.SetItems(i:Integer; aItem:TFormCrwDaqSysChild);
begin
 if Assigned(Self) and Assigned(mySdiList) and (i>=0) and (i<mySdiList.Count)
 then mySdiList[i]:=aItem;
end;

procedure TSdiChildList.Add(aItem:TFormCrwDaqSysChild);
begin
 if Assigned(Self) and Assigned(mySdiList) and Assigned(aItem)
 then mySdiList.Add(aItem);
end;

procedure TSdiChildList.Remove(aItem:TFormCrwDaqSysChild);
begin
 if Assigned(Self) and Assigned(mySdiList) and (mySdiList.IndexOf(aItem)>=0)
 then mySdiList.Remove(aItem);
end;

function TSdiChildList.IndexOf(aItem:Pointer):Integer;
begin
 if Assigned(Self) and Assigned(mySdiList)
 then Result:=mySdiList.IndexOf(aItem)
 else Result:=-1;
end;

function TSdiChildList.IsChild(Form:TObject; const Incl:QWord=0; const Excl:QWord=0):Boolean;
var Flags:QWord;
begin
 if not Assigned(Self) then Exit(false);
 if Assigned(Form) and (IndexOf(Form)>=0)
 then Result:=(Form is TMasterForm)
 else Result:=False;
 if Result and ((Incl<>0) or (Excl<>0)) then begin
  Flags:=TMasterForm(Form).SdiFlags;
  if (Excl<>0) and HasFlags(Flags,Excl) then Exit(False);
  if (Incl<>0) and not HasFlags(Flags,Incl) then Exit(False);
 end;
end;

function TSdiChildList.Minimized:Integer;
begin
 if Assigned(Self) and Assigned(myMinList)
 then Result:=myMinList.Count
 else Result:=0;
end;

procedure SdiMinimizer(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if Assigned(Custom) then
 if SdiMan.IsChild(Form) then
 if (TList(Custom).IndexOf(Form)<0) then
 if (biMinimize in Form.BorderIcons) then
 if (Form.WindowState<>wsMinimized) then begin
  Form.WindowState:=wsMinimized;
  TList(Custom).Add(Form);
 end;
end;

function TSdiChildList.Minimize:Integer;
begin
 Result:=0;
 if Assigned(Self) and Assigned(myMinList) then begin
  SdiMan.ForEachChildInZOrder(SdiMinimizer,myMinList);
  Result:=myMinList.Count;
 end;
end;

function TSdiChildList.Restore:Integer;
var i:Integer; Form:TForm;
begin
 Result:=0;
 if Assigned(Self) and Assigned(myMinList) then
 try
  for i:=myMinList.Count-1 downto 0 do
  if IsChild(myMinList[i]) then begin
   Form:=TForm(myMinList[i]);
   Form.WindowState:=wsNormal;
   Inc(Result);
  end;
  myMinList.Clear;
 except
  on E:Exception do BugReport(E,nil,'Restore');
 end;
end;

type TSmbfRec=record Count:Integer; Incl,Excl:QWord; end;
procedure SdiMinByFlags(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
var MForm:TMasterForm;
begin
 if Assigned(Custom) then
 with TSmbfRec(Custom^) do
 if SdiMan.IsChild(Form) then
 if (biMinimize in Form.BorderIcons) then begin
  MForm:=(Form as TMasterForm);
  if HasFlags(MForm.SdiFlags,Excl) then Exit;
  if not HasFlags(MForm.SdiFlags,Incl) then Exit;
  MForm.WindowState:=wsMinimized;
  if MForm.Visible and (MForm.WmWnd<>0)
  then wmctrl.ShowWindow(MForm.WmWnd,SW_MINIMIZE);
  Inc(Count);
 end;
end;

function TSdiChildList.MinimizeByFlags(const Incl:QWord; const Excl:QWord=0):Integer;
var R:TSmbfRec;
begin
 Result:=0;
 if Assigned(Self) then begin
  R.Count:=0; R.Incl:=Incl; R.Excl:=Excl;
  SdiMan.ForEachChildInZOrder(SdiMinByFlags,@R);
  Result:=R.Count;
 end;
end;

/////////////////////////////
// TSdiManager implementation
/////////////////////////////

constructor TSdiManager.Create;
begin
 inherited Create;
 ChildList.Ok;
end;

destructor TSdiManager.Destroy;
begin
 Kill(TObject(myChildList));
 inherited Destroy;
end;

function TSdiManager.GetChildList:TSdiChildList;
begin
 if not Assigned(Self) then Exit(nil);
 if not Assigned(myChildList) then begin
  myChildList:=TSdiChildList.Create;
  myChildList.Master:=@myChildList;
 end;
 Result:=myChildList;
end;

function TSdiManager.GetChildCount:Integer;
begin
 if not Assigned(Self) then Exit(0);
 Result:=ChildList.Count;
end;

function TSdiManager.GetActionOnActivateMainForm:TAction;
begin
 if Assigned(Self)
 then Result:=myActionOnActivateMainForm
 else Result:=nil;
end;

procedure TSdiManager.SetActionOnActivateMainForm(aAction:TAction);
begin
 if Assigned(Self)
 then myActionOnActivateMainForm:=aAction;
end;

function TSdiManager.IsChild(Form:TObject; const Incl:QWord=0; const Excl:QWord=0):Boolean;
begin
 if not Assigned(Self) then Exit(false);
 Result:=ChildList.IsChild(Form,Incl,Excl);
end;

function TSdiManager.ForEachChild(Action:TForEachSdiChildAction; Custom:Pointer; Backward:Boolean=false):Integer;
var i,Index:Integer; ChildForm:TCustomForm; Terminate:Boolean;
begin
 Result:=0;
 if not Assigned(Self) then Exit(0);
 try
  Index:=0;
  Terminate:=false;
  if (Application is TApplication) then begin
   if Backward then begin
    for i:=ChildList.Count-1 downto 0 do begin
     ChildForm:=ChildList[i];
     if (ChildForm is TForm) then begin
      if Assigned(Action) then Action(TForm(ChildForm),Index,Terminate,Custom);
      if Terminate then Break else Inc(Result);
      Inc(Index);
     end;
    end;
   end else begin
    for i:=0 to ChildList.Count-1 do begin
     ChildForm:=ChildList[i];
     if (ChildForm is TForm) then begin
      if Assigned(Action) then Action(TForm(ChildForm),Index,Terminate,Custom);
      if Terminate then Break else Inc(Result);
      Inc(Index);
     end;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ForEachChild');
 end;
end;

function TSdiManager.ForEachChildInZOrder(Action:TForEachSdiChildAction; Custom:Pointer; Backward:Boolean=false):Integer;
var i,Index:Integer; Form:TCustomForm; Terminate:Boolean;
begin
 Result:=0;
 if not Assigned(Self) then Exit(0);
 try
  Index:=0;
  Terminate:=false;
  if (Application is TApplication) then begin
   if Backward then begin
    for i:=Screen.CustomFormZOrderCount-1 downto 0 do begin
     Form:=Screen.CustomFormsZOrdered[i];
     if (Form is TFormCrwDaqSysChild) then begin
      if Assigned(Action) then Action(TForm(Form),Index,Terminate,Custom);
      if Terminate then Break else Inc(Result);
      Inc(Index);
     end;
    end;
   end else begin
    for i:=0 to Screen.CustomFormZOrderCount-1 do begin
     Form:=Screen.CustomFormsZOrdered[i];
     if (Form is TFormCrwDaqSysChild) then begin
      if Assigned(Action) then Action(TForm(Form),Index,Terminate,Custom);
      if Terminate then Break else Inc(Result);
      Inc(Index);
     end;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ForEachChildInZOrder');
 end;
end;

type TFindFirstSdiRec=packed record Want,Skip:QWord; Found:TForm; end;

procedure SdiSearchAction(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
var SdiFlags:QWord;
begin
 if Assigned(Custom) then
 with TFindFirstSdiRec(Custom^) do
 if (Form is TFormCrwDaqSysChild) then begin
  SdiFlags:=TFormCrwDaqSysChild(Form).SdiFlags;
  if HasFlags(SdiFlags,Want) and not HasFlags(SdiFlags,Skip) then begin
   Found:=Form; Terminate:=true;
  end;
 end;
end;

function TSdiManager.FindFirstChildInZOrder(Want,Skip:QWord; Backward:Boolean=false):TFormCrwDaqSysChild;
var R:TFindFirstSdiRec;
begin
 Result:=nil;
 if not Assigned(Self) then Exit(nil);
 R.Found:=nil; R.Want:=Want; R.Skip:=Skip;
 ForEachChildInZOrder(SdiSearchAction,@R,Backward);
 if (R.Found is TFormCrwDaqSysChild) then Result:=TFormCrwDaqSysChild(R.Found);
end;

function TSdiManager.TopMostChild(const Want:QWord=sf_SdiChild;
                        const Skip:QWord=0):TFormCrwDaqSysChild;
begin
 if not Assigned(Self) then Exit(nil);
 Result:=FindFirstChildInZOrder(Want,Skip);
end;

function TSdiManager.ActiveChild(const Want:QWord=sf_SdiChild;
                        const Skip:QWord=0):TFormCrwDaqSysChild;
var aChild:TForm;
begin
 Result:=nil;
 if not Assigned(Self) then Exit(nil);
 if FindActiveChild(aChild,Want,Skip) then
 if (aChild is TFormCrwDaqSysChild) then Result:=TFormCrwDaqSysChild(aChild);
end;

procedure UpdateTheForm(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 if (Form is TForm) then Form.Update;
end;

function TSdiManager.UpdateAllChilds:Integer;
begin
 if not Assigned(Self) then Exit(0);
 Result:=ForEachChild(UpdateTheForm,nil);
end;

procedure AddToSdiList(Form:TForm; Index:Integer; var Terminate:Boolean; Custom:Pointer);
begin
 TStringList(Custom).AddObject(Form.Caption,Form);
end;

function TSdiManager.SelectChildDialog(const Caption,Title:LongString; Mode:Integer=0; Params:LongString=''):TForm;
var List:TStringList; Index:Integer; aForm:TObject;
begin
 Result:=nil;
 if not Assigned(Self) then Exit(nil);
 try
  List:=TStringList.Create;
  try
   List.Duplicates:=dupAccept;
   if HasFlags(Mode,sm_ZOrder)
   then ForEachChildInZOrder(AddToSdiList,List)
   else ForEachChild(AddToSdiList,List);
   if (List.Count>0) then begin
    Index:=0; aForm:=nil;
    if (ListBoxSelection(Caption,Title,List.Text,Index,Params)=mrOk) then
    if (Index>=0) and (Index<List.Count)
    then aForm:=List.Objects[Index];
    if (aForm is TForm) then Result:=TForm(aForm);
   end;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,Self,'SelectChildDialog');
 end;
end;

function TSdiManager.FindActiveChild(out aChild:TForm; const Want,Skip:QWord):Boolean;
begin
 aChild:=TopMostChild(Want,Skip);
 if not Assigned(aChild) then Exit(false);
 //if (aChild.WindowState=wsMinimized) then Exit(false);
 Result:=aChild.Visible;
end;

function TSdiManager.ActivateChild(aForm:TForm; Mode:Integer=0):Boolean;
var aChild:TFormCrwDaqSysChild;
begin
 Result:=false;
 if not Assigned(Self) then Exit(false);
 try
  if IsChild(aForm) then begin
   if not aForm.Visible then aForm.Show;
   if not aForm.Enabled then aForm.Enabled:=true;
   if (aForm.WindowState<>wsNormal) then aForm.WindowState:=wsNormal;
   aForm.BringToFront;
   if not aForm.Focused then
   if aForm.CanSetFocus then begin
    aForm.SetFocus;
    Result:=true;
   end;
   if (Mode<>0) then
   if (aForm is TFormCrwDaqSysChild) then begin
    aChild:=TFormCrwDaqSysChild(aForm);
    if HasFlags(Mode,ascm_GtkRefresh) then begin
     if IsWidgetSetName('gtk,gtk2,gtk3')
     then LiftFlags(Mode,ascm_Refresh);
    end;
    if HasFlags(Mode,ascm_Refresh) then begin
     aChild.RefreshWindow;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActivateChild');
 end;
end;

function TSdiManager.ActivateChildNext(Step:Integer):Boolean;
var List:TStringList; i,j:Integer; Obj:TObject; aChild:TForm;
begin
 Result:=false;
 if not Assigned(Self) then Exit(false);
 try
  List:=TStringList.Create;
  try
   ForEachChildInZOrder(AddToSdiList,List);
   if (Step>=0) then i:=Step else i:=List.Count+Step;
   if InRange(i,0,List.Count-1) then Obj:=List.Objects[i] else Obj:=nil;
   if IsChild(Obj) then begin
    if (Step>0) then begin
     for j:=List.Count-1 downto i do begin
      aChild:=(List.Objects[j] as TForm);
      if FormIsExposed(aChild)
      then ActivateChild(aChild);
     end;
    end;
    ActivateChild(Obj as TForm);
    Result:=true;
   end;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,Self,'ActivateChildNext');
 end;
end;

function TSdiManager.ActivateMainForm:Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  if IsSdiMode then begin
   Result:=ActivateApplicationMainForm;
   if Assigned(myActionOnActivateMainForm)
   then myActionOnActivateMainForm.Execute;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActivateMainForm');
 end;
end;

class function TSdiManager.IsSdiMode:Boolean;
begin
 Result:=(GuiWorkingMode=gwm_SDI);
end;

class function TSdiManager.IsMdiMode:Boolean;
begin
 Result:=(GuiWorkingMode=gwm_MDI);
end;

/////////////////////////
// TSdiMan implementation
/////////////////////////

const
 TheSdiMan:TSdiManager=nil;

function SdiMan:TSdiManager;
begin
 if not Assigned(TheSdiMan) then begin
  TheSdiMan:=TSdiManager.Create;
  TheSdiMan.Master:=@TheSdiMan;
 end;
 Result:=TheSdiMan;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_crwdaqsyschild;
begin
 InitSdiCounters;
 SdiMan.Ok; // NB: Set GUI mode.
 SdiMan.GuiWorkingMode:=gwm_SDI;
 SdiMan.HideChildSdiFormMenu:=true;
 SdiMan.EnableSdiFormMenuMerge:=true;
 SdiMan.EnableCheckMainFormIsExposed:=false;
 SdiMan.CursorActivateMainSdiForm:=crDefault;
 TFormCrwDaqSysChild.LTWHCopyOnFix:=true;
 TFormCrwDaqSysChild.CodePillsEnabled:=true;
 TFormCrwDaqSysChild.myLastActiveCrwChild:=nil;
 TFormCrwDaqSysChild.CheckFixedBoundsMode:=$1F;
 TFormCrwDaqSysChild.FixedBoundsTolerance:=Rect(3,3,3,3);
 TFormCrwDaqSysChild.HidesOnPrint:=hop_Default;
 TFormCrwDaqSysChild.ptrg_FormHide;
 TFormCrwDaqSysChild.ptrg_FormShow;
 TFormCrwDaqSysChild.ptrg_FormActivate;
 TFormCrwDaqSysChild.ptrg_FormDeactivate;
 TFormCrwDaqSysChild.ptrg_FormChangeBounds;
 TFormCrwDaqSysChild.ptrg_FormWindowStateChange;
 TFormCrwDaqSysChild.ptrg_FormClose;
 TFormCrwDaqSysChild.ptrg_FocusInput;
 dlc_SdiForm_:=RegisterDebugLogChannel('_SdiForm');
 dlc_DrawView_:=RegisterDebugLogChannel('_DrawView');
 dlc_FixBound_:=RegisterDebugLogChannel('_FixBound');
end;

procedure Free_form_crwdaqsyschild;
begin
 ResourceLeakageLog(Format('%-60s = %d',['Balance of TFormCrwDaqSysChild.LockDraw/UnlockDraw',LockedGet(LockUnlockBalance)]));
 ResourceLeakageLog(Format('%-60s = %d',['SdiChildList.Count',SdiMan.ChildList.Count]));
 Kill(TObject(TheSdiMan));
 FreeSdiCounters;
end;

initialization

 Init_form_crwdaqsyschild;

finalization

 Free_form_crwdaqsyschild;

end.

//////////////
// END OF FILE
//////////////

