////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// CRW-DAQ Software Device Script.                                            //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231207 - Modified for FPC (A.K.)                                         //
// 20251210 - @winopen                                                        //
////////////////////////////////////////////////////////////////////////////////

unit _crw_softdevscript; // CRW-DAQ Software Device Script

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow,
 Form_SurfWindow, Form_CircuitWindow,
 Form_ConsoleWindow, Form_TabWindow,
 Form_SpectrWindow, Form_Calculator,
 Form_ListBoxSelection, Form_UartTerminal,
 Form_CalibDialog, Form_DaqEditTagDialog,
 Form_TextEditDialog,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_ee, _crw_pio, _crw_curves,
 _crw_riff, _crw_calib, _crw_couple,
 _crw_daqtags, _crw_daqevnt, _crw_daqsys,
  _crw_daqdev, _crw_softdev, _crw_syscal,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Устройство ScriptDevice с именем name объявляется в файле конфигурации как
  [DeviceList]
  name = Device Software Script
 Устройство служит для общей обработки данных при помощи встроенного интерпретатора.
 Число входов, выходов и калибровок задается в конфигурационном файле.
 Устройство содержит простой интерпретатор для выполнения программ (скриптов) на
 языке условно, названном "Daqsic". Это узко - ориентированный интерпретатор
 для встраивания в устройства DAQ и проведения простых вычислений в процессе
 измерений. В настоящее время формально можно описать Daqsic так:
 1. Набор допустимых символов 'a'..'z', '0'..'9', '.', '_',' ', Tab,'+', '-',
    '*', '/', '%', '^', '(', ')', ',', '=',':', ';', '$'
 2. Поддерживается один тип данных-число (double). Массивы и строки не предусмотрены.
    Поддерживаются переменные, константы, встроенные функции, арифметические
    операторы, скобки любого уровня, метки, условный оператор if..then.. ,
    оператор перехода на метку goto, вызова подпрограммы gosub.
    Циклы можно организовывать через if и goto.
    Распознаются числа в e-формате (типа +1.5E-3), а также шестнадцатиричные
    числа, которые должны начинаться с символа $, например, $AF
 3. Программа Daqsic состоит из последовательно выполняемых строк вида
    Метка: Оператор ;Комментарий
    Наличие метки, комментария и оператора в строке необязательно.
    (То есть могут быть разные комбинации).
    На одной строке может быть только один оператор.
 4. Признаком начала комментария является символ ';'.
    Комментарий при выполнении игнорируется.
 5. Признаком метки является символ ':', причем пробелов между именем метки и ':'
    быть не должно. Имя метки состоит из символов 'a'..'z', '0'..'9', '_' ,
    но первой должна быть буква или '_'. Двух одинаковых меток в программе быть
    не должно.
 6. Поддерживаются переменные. Имя переменной состоит из символов
    'a'..'z', '0'..'9', '_' , но первой должна быть буква или '_'.
    Переменные создаются автоматически при непустом присвоении типа
    ИмяПеременной=Выражение
    Переменные удаляются автоматически при пустом присвоении типа
    ИмяПеременной=
    Переменные могут быть объявлены в конструкции
    var Имя1,Имя2...
    Конструкция var создает указанные переменные со значением 0, если они не
    существовали, но не меняет значений уже существующих переменных.
 7. Условный оператор имеет вид
    if(Условие)then Оператор или
    if Условие then Оператор
    Если выражение Условие имеет значение 0, то Оператор игнорируется.
    Если выражение Условие не равно 0, то Оператор выполняется.
    Оператор может быть выражением или переходом на метку, но не другим
    оператором if
 8. Оператор перехода на метку имеет вид
    goto ИмяМетки
    Интерпретатор найдет строку с меткой ИмяМетки: и передаст управление этой строке.
 9. Выражения могут содержать операторы +,-,*,/,%,^,скобки,ссылки на переменные,
    константы, вызовы функций - все как обычно
 10.Функции различаются наличием после имени скобок с параметрами, разделенными
    запятыми с числом параметров от 0 до 7, например,
    time()
    hypot(x,y)
    Интерпретатор Daqsic содержит функции для  чтения данных со входов устройства,
    записи результата на выходы, доступа к калибровкам а также обработки данных.
    ****************
    Краткое описание
    ****************
    Комментарии:  отделяются символом ;
    Метки:
    Label:        Имя метки и сразу за ней символ :
                  Имена меток начинаются с буквы.
    Операторы:
    goto Label    Передает управление метке Label
    if c then O   Условное выполнение оператора O при условии что выражение c отлично от нуля
    Директивы:
    @voice hello директивы принимают строку как аргумент и делают ее разбор специальным образом.
    Переменные:
    x=10          Создаются автоматически при непустом присвоении.
    x=            Удаляются атоматически при пустом присвоении.
    var x,y
    Константы
    pi            3.14159...
    e             основание натурального логарифма
    false         0
    true          1
    macheps       машинная точность
    _nan          NAN
    _inf          INF
    _minusinf     -INF
    Операторы
    +             сумма
    -             разность
    *             умножение
    /             деление
    %             остаток
    ^             возведение в степень
    Встроенные функции:
    Функции DAQ:
    time()        локальное время DAQ с начала измерений
    timeunits()   единицы измерения локального времени DAQ
    crvlen(c)     длина кривой c (c-ссылка на кривую)
    crvx(c,i)     координата x кривой c в точке номер i
    crvy(c,i)     координата y кривой c в точке номер i
    crvput(c,i,x,y) поместить в кривую c в точке номер i данные x,y
    crvinteg(c,a,b) интеграл кривой c в интервале a..b
    numais()      число аналоговых входов
    numdis()      число цифровых входов
    numaos        число аналоговых выходов
    numdos        число цифровых выходов
    numcals       число калибровок
    refai(n)      ссылка на аналоговый вход n или 0 если его нет
    refdi(n)      ссылка на цифровой вход n или 0 если его нет
    refao(n)      ссылка на аналоговый выход n или 0 если его нет
    refdo(n)      ссылка на цифровой выход n или 0 если его нет
    refcalibr(n)  ссылка на калибровку n или 0 если ее нет
    getai(n,t)    аналоговый вход номер n в момент времени t
    getai_n(n)    число точек,аналоговый вход номер n
    getai_xn(n)   x последней точки,аналоговый вход номер n
    getai_yn(n)   y последней точки,аналоговый вход номер n
    getai_xi(n,i) x точки номер i,аналоговый вход номер n
    getai_yi(n,i) y точки номер i,аналоговый вход номер n
    getdi(n,t)    цифровой вход номер n в момент времени t
    getdi_n(n)    число точек,цифровой вход номер n
    getdi_xn(n)   x последней точки,цифровой вход номер n
    getdi_yn(n)   y последней точки,цифровой вход номер n
    getdi_xi(n,i) x точки номер i,цифровой вход номер n
    getdi_yi(n,i) y точки номер i,цифровой вход номер n
    putev(w,c,t,d0,d1) генерирует событие:(what,chan,time,data0,data1).
                  Возвращает 1 при успехе.
    putao(n,t,d)  генерирует аналоговое событие с номером канала n,
                  временем t и данными d. Возвращает 1 при успехе.
    putdo(n,t,d)  генерирует цифровое событие с номером канала n,
                  временем t и данными d. Возвращает 1 при успехе.
    calibr(n,d,p) возвращает результат калибровочного преобразования
                  n-номер калибровки,d-данные, p-параметр.
    fixerror(n)   регистрирует ошибку номер n
    inportb(n)    ввод байта из порта ввода-вывода n
    inportw(n)    ввод слова из порта ввода-вывода n
    outportb(n,d) вывод байта d в порт ввода-вывода n
    outportw(n,d) вывод слова d в порт ввода-вывода n
    tm_new()         создать новый таймер
    tm_free(a)       удалить таймер a
    tm_addint(a,b)   добавить в таймер a новый интервал b,[ms]
    tm_numint(a)     узнать число интервалов таймера a
    tm_getint(a,b)   узнать длину интервала номер b таймера a,[ms]
    tm_setint(a,b,c) задать длину c,[ms] интервала номер b таймера a
    tm_gettime(a)    узнать время со старта таймера a, [ms]
    tm_start(a)      пуск таймера a
    tm_stop(a)       останов таймера a
    tm_isstart(a)    узнать запущен ли таймер a
    tm_event(a)      генерация событий таймера a
    tm_curint(a)     номер текущего интервала таймера a
    igettag(a)       чтение значения integer тега a
    rgettag(a)       чтение значения real тега a
    isettag(a,b)     запись в integer тег a значения b
    rsettag(a,b)     запись в integer тег a значения b
    typetag(a)       возвращает тип тега a, 1/2/3=integer/real/string
    clicktag()       возвращает тег нажатого сенсора или 0
    clickbutton()    возвращает нажатую кнопку мыши 1/2/4=Left/Right/Middle
    Общие функции:
    deg(x)        перевод радиан в градусы
    rad(x)        перевод градус в радианы
    sin(x)        синус
    cos(x)        косинус
    tan(x)        тангенс
    asin(x)       обратный синус
    acos(x)       обратный косинус
    atan(x)       обратный тангенс
    sinh(x)       гиперболический синус
    cosh(x)       гиперболический косинус
    tanh(x)       гиперболический тангенс
    exp(x)        экспонента
    ln(x)         натуральный логарифм x
    log(n,x)      логарифм x по основанию n
    sqrt(x)       корень
    int(x)        округление в сторону нуля
    trunc(x)      округление в сторону нуля
    frac(x)       дробная часть
    round(x)      округление в сторону ближайшего целого
    floor(x)      округление в сторону нуля
    ceil(x)       округление в сторону INF
    abs(x)        модуль
    hypot(x,y)    sqrt(x^2+y^2)
    rand()        случайное число от 0 до 1
    random(a,b)   случайное число в интервале (a,b)
    sign(x)       знак (+1,0,-1)
    eq(x,y)       равно 1/0 если x=y
    ne(x,y)       равно 1/0 если x<>y
    lt(x,y)       равно 1/0 если x<y
    gt(x,y)       равно 1/0 если x>y
    le(x,y)       равно 1/0 если x<=y
    ge(x,y)       равно 1/0 если x>=y
    max(x,y)      максимальное из x и y
    min(x,y)      минимальное  из x и y
    msecnow()     текущее время в миллисекундах от Рождества Христова
    secnow()      текущее время в секундах      от Рождества Христова
    getticks()    текущее время в тиках таймера BIOS
    or(x,y)       логическое неисключающее или
    xor(x,y)      логическое исключающее или
    and(x,y)      логическое и
    not(x)        логическое отрицание
    bitor(x,y)    побитное арифметическое неисключающее или
    bitxor(x,y)   побитное арифметическое исключающее или
    bitand(x,y)   побитное арифметическое  и
    bitnot(x)     побитное арифметическое отрицание
    getbitmask(n) получить 2^n - то есть бит номер n
    isbit(x,n)    есть ли в x бит номер n
    gamma(n)      гамма-функция: gamma(n) = (n-1)!
    isnan(a)      1 если a=NAN или 0
    isinf(a)      1 если a=INF или 0
    vnew(a)       создаст массив размерности a и вернет ссылку
    vfree(a)      удалит созданный vnew массив заданный ссылкой a
    vsize(a)      вернет размерность массива заданного ссылкой a
    vget(a,b)     вернет элемент массива a с номером b
    vput(a,b,c)   записать в элемент массива a номер b значение c
  Директивы:
    @inittag          @inittag n t - найти или создать тег с именем n и типом t=1..3
    @findtag          @findtag n найти тег по имени n, результат в actiontesult
    @clicksensor      clicksensor name проверяет, нажат ли сенсор с именем name
    @echo             вывод сообщения в консольное окно
    @voice   arg      звуковое сообщение в строке arg
    @action  arg      вызов метода action для списка устройств arg
    @clear   arg      вызов метода clear для списка устройств arg
    @cleardevice arg  вызов метода cleardevice для списка устройств arg
    @start   arg      вызов метода start для списка устройств arg
    @stop    arg      вызов метода stop для списка устройств arg
    @devmsg  dev msg  посылка сообщения msg устройству dev
    @debugout msg     вывод сообщения msg в файл отладки
    @clearcurve arg   очистка кривых из списка arg
    @savecrw          выполняет сохранение кривых в файл CRW
     @savecrw fname winname wintitle winlable curvelist
      fname      - имя файла (можно без пути и без расширения)
      winname    - имя окна под которым оно попадет в crw-архив
      wintitle   - заголовок в верхней части окна
      winlable   - метка в нижней части окна
      curvelist  - список сохраняемых кривых
    @specmarker  win - выдать Marker  спектрометрич. окна win
    @specmarkerl win - выдать MarkerL спектрометрич. окна win
    @specmarkerr win - выдать MarkerR спектрометрич. окна win
    @specroil    win - выдать RoiL    спектрометрич. окна win
    @specroir    win - выдать RoiR    спектрометрич. окна win
    @winopen     win - загрузить/открыть окно win
    @windraw     win - перерисовать окно win
    @winshow     win - показать     окно win
    @winhide     win - спрятать     окно win
    @winselect   win - активизировать окно
    @global expr - вычисляет выражение expr в общем калькуляторе
    В директивах возможна передача значений переменных: если встречается
    конструкция %name, то она заменяется значением переменной name.
  Пример выражений:
    t=time()       ; взять время DAQ
    x=t*60         ; перевести в секунды
    y=sin(2*pi*x)  ; вычислить синус
    putao(0,t,y,1) ; вывести на аналоговый выход 0
    n=100
    @voice %n
 Конфигурация устройства с именем name задается в секции [name].
 Эта секция содержит такие переменные:
    AnalogInputs   = n - задание числа аналоговых входов
    AnalogOutputs  = n - задание числа аналоговых выходов
    DigitalInputs  = n - задание числа цифровых   входов
    DigitalOutputs = n - задание числа цифровых   выходов
    Calibrations   = n - задание числа калибровок
    Calibration#n  = ? - задание калибровки номер n
    DebugMode      = 1/0    - надо ли выводить диагностику ошибок интерпретации в файл отладки
    ScriptSection  = section - задание имени секции с текстом выражений.
 ********************
 Пример конфигурации:
 ********************
 Данный пример генерирует сигналы на трех аналоговых выходах.
    [DeviceList]
    script1 = device software script
    [script1]
    AnalogInputs = 0
    DigitalInputs = 0
    AnalogOutputs = 3
    DigitalOutputs = 0
    Calibrations = 0
    Link AnalogOutput 0 with curve ao0
    Link AnalogOutput 1 with curve ao1
    Link AnalogOutput 2 with curve ao2
    DebugMode = 1
    ScriptSection = script1.text
    [script1.text]
    t=time()
    a0=10*sin(2*pi*t)+0.1*random(-1,1)
    a1=5*cos(2*pi*t)+0.1*random(-1,1)
    a2=-10*sin(2*pi*t*0.1)+0.1*random(-1,1)
    putao(0,t,a0)
    putao(1,t,a1)
    putao(2,t,a2)
 *******************************************************************************
 }
type
 TScriptDevice = class(TSoftwareDevice)
 private
  myEe            : TExpressionEvaluator;
  myTimers        : TObjectStorage;
  myVecList       : TObjectStorage;
  myScriptSection : LongString;
  myDebugMode     : Boolean;
  procedure   RegisterFunctions;
  procedure   RegisterActions;
  function    FindCurve(ref:Double):TCurve;
  function    FindTimer(ref:Double):TIntervalTimer;
  function    FindVector(ref:Double):TDoubleVector;
 public
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  procedure   Handler; override;
  procedure   DeferredCommandProblem(const Cmd:LongString); override;
 end;

implementation

 {
 ****************************
 TScriptDevice implementation
 ****************************
 }
 {
 Создание устройства
 }
constructor TScriptDevice.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceModel('SCRIPT');
 AnalogFifoSize:=64;
 DigitalFifoSize:=64;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=0;
 NumDigitalOutputs:=0;
 NumCalibrations:=0;
 myEe:=NewExpressionEvaluator;
 myEe.Master:=@myEe;
 myEe.Custom:=Self;
 RegisterFunctions;
 RegisterActions;
 myTimers:=NewObjectStorage;
 myVecList:=NewObjectStorage;
 myScriptSection:='';
 myDebugMode:=false;
end;

 {
 Уничтожение устройства
 }
destructor TScriptDevice.Destroy;
begin
 myScriptSection:='';
 Kill(myVecList);
 Kill(myTimers);
 Kill(myEe);
 inherited Destroy;
end;

 {
 Разбор файла конфигурации
 }
procedure TScriptDevice.Config(FileName:LongString);
var i:Integer;
begin
 FileName:=UnifyFileAlias(FileName); i:=0;
 inherited Config(FileName);
 if ReadIniFileInteger(FileName,DevSection,'AnalogInputs%i',i) then NumAnalogInputs:=i;
 if ReadIniFileInteger(FileName,DevSection,'AnalogOutputs%i',i) then NumAnalogOutputs:=i;
 if ReadIniFileInteger(FileName,DevSection,'DigitalInputs%i',i) then NumDigitalInputs:=i;
 if ReadIniFileInteger(FileName,DevSection,'DigitalOutputs%i',i) then NumDigitalOutputs:=i;
 if ReadIniFileInteger(FileName,DevSection,'Calibrations%i',i) then NumCalibrations:=i;
 ReadIniFileBoolean(FileName,DevSection,'DebugMode%b',myDebugMode);
 for i:=0 to NumCalibrations-1 do ReadCalibration(i,FileName,DevSection,'Calibration#'+d2s(i));
 myScriptSection:='';
 if ReadIniFileAlpha(FileName,DevSection,'ScriptSection%a',myScriptSection)
 then myEe.Script:=ExtractTextSection(FileName,'['+myScriptSection+']',efAsIs)
 else myEe.Script:='';
end;

 {
 Создание списка свойств.
 Для потоковой безопасности запрещаем поток сбора данных Polling, а затем снова
 разрешаем.
 }
function TScriptDevice.GetProperty(P:TText):TText;
var t:TText; i:Integer; f:Boolean;
begin
 f:=Polling.Enabled;
 Polling.Enable(false,DefaultDaqTimeOut);
 GetProperty:=inherited GetProperty(P);
 P.Addln('DebugMode = '+d2s(ord(myDebugMode)));
 for i:=0 to NumCalibrations-1 do
 P.AddLn(FormatCalibration('Calibration#'+d2s(i),Calibration[i]));
 P.Addln('ScriptSection = '+myScriptSection);
 if IsNonEmptyStr(myScriptSection) then begin
  P.Addln('['+myScriptSection+']');
  ConcatText(P,myEe.Script);
  P.Addln(RusEng('----- ИСПОЛНЯЕМЫЙ ТЕКСТ -----','----- PRECOMPILED TEXT -----'));
  t:=NewText;
  t.Text:=myEe.PreScript;
  for i:=0 to t.Count-1 do t[i]:=Format('%3d : %s', [i, t[i]]);
  ConcatText(P,t);
  Kill(t);
  P.Addln(RusEng('----- СПИСОК МЕТОК -----','----- LABEL LIST -----'));
  myEe.LabelList.GetText(P);
  P.Addln(RusEng('----- СПИСОК ПЕРЕМЕННЫХ -----','----- VARIABLE LIST -----'));
  myEe.VarList.GetText(P);
  P.Addln(RusEng('----- СПИСОК КОНСТАНТ -----','----- CONSTANT LIST -----'));
  myEe.ConstList.GetText(P);
  P.Addln(RusEng('----- СПИСОК ФУНКЦИЙ -----','----- FUNCTION LIST -----'));
  myEe.FuncList.GetText(P);
  P.Addln(RusEng('----- СПИСОК ДИРЕКТИВ -----','----- DIRECTIVE LIST -----'));
  myEe.ActionList.GetText(P);
 end;
 Polling.Enable(f,DefaultDaqTimeOut);
end;

 {
 Выполнение скрипта, только в потоке Polling.
 }
procedure  TScriptDevice.Handler;
begin
 if (GetCurrentThreadID=Polling.ThreadID) then
 if (myEe.RunScript<>ee_Ok) then begin
  FixError(ecFormulaSyntax);
  if myDebugMode then begin
   DebugOut(stdfDebug,RusEng('Ошибка    : ','Error      : ')+ee_ErrorMessage(myEe.Status)+' - '+StrPas(myEe.ErrorToken));
   DebugOut(stdfDebug,RusEng('Выражение : ','Expression : ')+StrPas(myEe.Buffer));
   DebugOut(stdfDebug,RusEng('Позиция   : ','Position   : ')+d2s(myEe.ErrorPos));
  end;
 end;
end;

procedure TScriptDevice.DeferredCommandProblem(const Cmd:LongString);
begin
 if Ok then
 if myDebugMode then inherited DeferredCommandProblem(Cmd);
end;

 {
 Найти кривую по ссылке.
 }
function TScriptDevice.FindCurve(ref:Double):TCurve;
var Obj:TObject;
begin
 Result:=nil;
 if Assigned(Self) then begin
  Obj:=ObjectRegistry[Round(ref)];
  if (Obj is TCurve)
  then Result:=TCurve(Obj);
 end;
end;

 {
 Найти таймер в списке таймеров.
 }
function TScriptDevice.FindTimer(ref:Double):TIntervalTimer;
var Obj:TObject;
begin
 Result:=nil;
 if Assigned(Self) then begin
  Obj:=ObjectRegistry[Round(ref)];
  if (Obj is TIntervalTimer) and (myTimers.IndexOf(Obj)>=0)
  then Result:=TIntervalTimer(Obj);
 end;
end;

 {
 Найти вектор в списке векторов.
 }
function TScriptDevice.FindVector(ref:Double):TDoubleVector;
var Obj:TObject;
begin
 Result:=nil;
 if Assigned(Self) then begin
  Obj:=ObjectRegistry[Round(ref)];
  if (Obj is TDoubleVector) and (myVecList.IndexOf(Obj)>=0)
  then Result:=TDoubleVector(Obj);
 end;
end;

 {
 Процедура для проверки числа аргументов функции и исключения по ошибке.
 }
procedure iocheck(ee:TExpressionEvaluator; i,j:Integer);
begin
 if (i<>j) then ee.RuntimeError:=ee_User;
end;

 {
 Ссылка на ScriptDevice, связанного с ExpressionEvaluator
 }
function TheScript(ee:TExpressionEvaluator):TScriptDevice;
begin
 Result:=ee.Custom;
end;

 {
 time() - время по часам DAQ
 }
function f_time(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=Daq.Timer.LocalTime;
end;

 {
 timeunits() - единица времени по часам DAQ
 }
function f_timeunits(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=Daq.Timer.LocalTimeUnits;
end;

 {
 tm_new() - создает таймер и возвращает ссылку на него
 }
function f_tm_new(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,0);
 tm:=NewIntervalTimer(tmCyclic,nil);
 TheScript(ee).myTimers.Add(tm);
 Result:=tm.Ref;
end;

 {
 tm_free(tm) - удаляет таймер со ссылкой tm
 }
function f_tm_free(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 Result:=0;
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then begin
  TheScript(ee).myTimers.Remove(tm);
  Result:=1;
 end;
end;

 {
 tm_addint(tm,ms) - добавляет в таймер tm новый интервал ms в миллисекундах
 }
function f_tm_addint(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,2);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then begin
  tm.AddIntervalMs(x[1],tm.NumIntervals+1);
  Result:=1;
 end else Result:=0;
end;

 {
 tm_numint(tm) - возвращает число интервалов таймера
 }
function f_tm_numint(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then Result:=tm.NumIntervals else Result:=0;
end;

 {
 tm_getint(tm,n) - возвращает для таймера tm длину интервала номер n в миллисекундах
 }
function f_tm_getint(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,2);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then Result:=tm.IntervalMs[Round(x[1])] else Result:=0;
end;

 {
 tm_setint(tm,n,ms) - установить для таймера tm длину интервала номер n равной ms миллисекунд
 }
function f_tm_setint(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,3);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then begin
  tm.IntervalMs[Round(x[1])]:=x[2];
  Result:=1;
 end else Result:=0;
end;

 {
 tm_gettime(tm) - возвращает время со старта таймера tm в миллисекундах
 }
function f_tm_gettime(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then Result:=tm.LocalTime else Result:=0;
end;

 {
 tm_start(tm) - пуск таймера tm
 }
function f_tm_start(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then begin
  tm.Start;
  Result:=1;
 end else Result:=0;
end;

 {
 tm_stop(tm) - останов таймера tm
 }
function f_tm_stop(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then begin
  tm.Stop;
  Result:=1;
 end else Result:=0;
end;

 {
 tm_isstart(tm) - узнать, запущен ли таймер tm
 }
function f_tm_isstart(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then Result:=ord(tm.IsStart) else Result:=0;
end;

 {
 tm_event(tm) - генерация событий таймера tm
 }
function f_tm_event(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then Result:=ord(tm.Event) else Result:=0;
end;

 {
 tm_curint(tm) - номер текущего интервала таймера tm
 }
function f_tm_curint(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var tm:TIntervalTimer;
begin
 iocheck(ee,narg,1);
 tm:=TheScript(ee).FindTimer(x[0]);
 if Assigned(tm) then Result:=tm.CurrentInterval else Result:=0;
end;

 {
 crvlen(cref) - длина кривой cref
 }
function f_crvlen(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).FindCurve(x[0]).Count;
end;

 {
 crvx(cref,n) - координата х кривой cref, точки с номером n=1..crvlen(cref)
 }
function f_crvx(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).FindCurve(x[0])[Round(x[1])-1].X;
end;

 {
 crvy(cref,n) - координата х кривой cref, точки с номером n=1..crvlen(cref)
 }
function f_crvy(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).FindCurve(x[0])[Round(x[1])-1].Y;
end;

 {
 crvput(cref,n,x,y) - задать для кривой cref точку номер n равной (x,y), при n=crvlen(cref)+1 вставка точки
 }
function f_crvput(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var c:TCurve; j:Integer;
begin
 iocheck(ee,narg,4);
 c:=TheScript(ee).FindCurve(x[0]);
 j:=Round(x[1])-1;
 if InRange(j,0,c.Count-1) then begin
  c[j]:=Point2D(x[2],x[3]);
  Result:=1;
 end else Result:=0;
end;

 {
 crvinteg(cref,a,b) - определенный интеграл кривой cref в интервале (a,b)
 }
function f_crvinteg(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,3);
 Result:=DaqIntegral(TheScript(ee).FindCurve(x[0]),x[1],x[2]);
end;

 {
 numais() - число аналоговых входов устройства
 }
function f_numais(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).NumAnalogInputs;
end;

 {
 numdis() - число цифровых входов устройства
 }
function f_numdis(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).NumDigitalInputs;
end;

 {
 numaos() - число аналоговых выходов устройства
 }
function f_numaos(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).NumAnalogOutputs;
end;

 {
 numdos() - число цифровых выходов устройства
 }
function f_numdos(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).NumDigitalOutputs;
end;

 {
 numcals() - число калибровок
 }
function f_numcals(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).NumCalibrations;
end;

 {
 refai(n) - ссылка на кривую analog input(n)
 }
function f_refai(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).AnalogInputCurve[Round(x[0])].Ref;
end;

 {
 refdi(n) - ссылка на кривую digital input(n)
 }
function f_refdi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).DigitalInputCurve[Round(x[0])].Ref;
end;

 {
 refao(n) - ссылка на кривую analog output(n)
 }
function f_refao(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).AnalogOutputCurve[Round(x[0])].Ref;
end;

 {
 refdo(n) -  ссылка на кривую digital output(n)
 }
function f_refdo(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).DigitalOutputCurve[Round(x[0])].Ref;
end;

 {
 refcalibr - ссылка на калибровку номер n
 }
function f_refcalibr(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).Calibration[Round(x[0])].Ref;
end;

 {
 getai(n,x) - интерполировать аналоговый вход n в точке x
 }
function f_getai(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).SmoothAnalogInputCurve(Round(x[0]),x[1]);
end;

 {
 getdi(n,x) - интерполировать цифровой вход n в точке x
 }
function f_getdi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var c:TCurve; p:TPoint2D;
begin
 iocheck(ee,narg,2);
 c:=TheScript(ee).DigitalInputCurve[Round(x[0])];
 if (c.Count>0) and (time>=0) then begin
  p:=c.LastPoint;
  if x[1]>=p.x
  then Result:=p.y
  else Result:=c[c.GetIndexAt(x[1])].y;
 end else Result:=0;
end;

 {
 getai_n(n) - число точек кривой на аналоговом входе n
 }
function f_getai_n(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).AnalogInputCurve[Round(x[0])].Count;
end;

 {
 getai_xn(n) - координата x последней точки кривой на аналоговом входе n
 }
function f_getai_xn(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).AnalogInputCurve[Round(x[0])].LastPoint.X;
end;

 {
 getai_yn(n) - координата y последней точки кривой на аналоговом входе n
 }
function f_getai_yn(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).AnalogInputCurve[Round(x[0])].LastPoint.Y;
end;

 {
 getai_xi(n,i) - координата x i-й точки кривой на аналоговом входе n
 }
function f_getai_xi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).AnalogInputCurve[Round(x[0])][Round(x[1])-1].X;
end;

 {
 getai_yi(n,i) - координата y i-й точки кривой на аналоговом входе n
 }
function f_getai_yi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).AnalogInputCurve[Round(x[0])][Round(x[1])-1].Y;
end;

 {
 getdi_n(n) - число точек кривой на цифровом входе n
 }
function f_getdi_n(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).DigitalInputCurve[Round(x[0])].Count;
end;

 {
 getdi_xn(n) - координата x последней точки кривой на цифровом входе n
 }
function f_getdi_xn(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).DigitalInputCurve[Round(x[0])].LastPoint.X;
end;

 {
 getdi_yn(n) - координата y последней точки кривой на цифровом входе n
 }
function f_getdi_yn(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=TheScript(ee).DigitalInputCurve[Round(x[0])].LastPoint.Y;
end;

 {
 getdi_xi(n,i) - координата x i-й точки кривой на цифровом входе n
 }
function f_getdi_xi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).DigitalInputCurve[Round(x[0])][Round(x[1])-1].X;
end;

 {
 getdi_yi(n,i) - координата y i-й точки кривой на цифровом входе n
 }
function f_getdi_yi(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=TheScript(ee).DigitalInputCurve[Round(x[0])][Round(x[1])-1].Y;
end;

 {
 putev(what,chan,time,data0,data1) - помещает в очередь событие what на цифровой
                                     или аналоговый выход chan, время time, данные data0,data1
 }
function f_putev(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,5);
 Result:=ord(TheScript(ee).PutDaqEvent(DaqEvent(Round(x[0]),Round(x[1]),x[2],x[3],x[4])));
end;

 {
 putao(n,time,data) - помещает в очередь событие на аналоговый выход chan,
                      время time, данные data
 }
function f_putao(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,3);
 Result:=ord(TheScript(ee).PutDaqEvent(DaqEvent(evAnalog+evCompress,Round(x[0]),x[1],x[2],0)));
end;

 {
 putdo(n,time,data) - помещает в очередь событие на аналоговый выход chan,
                      время time, данные data
 }
function f_putdo(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,3);
 Result:=ord(TheScript(ee).PutDaqEvent(DaqEvent(evDigital,Round(x[0]),x[1],x[2],0)));
end;

 {
 calibr(n,mv,tc) - получить значение по калибровке n, аргумент mv(термоэдс),
                   параметр tc(температура холодного спая)  
 }
function f_calibr(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,3);
 Result:=TheScript(ee).Transform(Round(x[0]),x[1],x[2]);
end;

 {
 fixerror(n) - фиксация ошибки номер n
 }
function f_fixerror(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=1;
 TheScript(ee).FixError(Round(x[0]));
end;

 {
 inportb(n) - ввод байта из порта n
 }
function f_inportb(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=Port[Round(x[0])];
end;

 {
 inportw(n) - ввод слова из порта n
 }
function f_inportw(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=PortW[Round(x[0])];
end;

 {
 outportb(n,data) - вывод байта data в порт n
 }
function f_outportb(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=1;
 Port[Round(x[0])]:=Round(x[1]);
end;

 {
 outportw(n,data) - вывод слова data в порт n
 }
function f_outportw(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=1;
 PortW[Round(x[0])]:=Round(x[1]);
end;

 {
 vnew(n) - выделение массива из n элементов 0..n-1
 }
function f_vnew(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var v:TDoubleVector;
begin
 iocheck(ee,narg,1);
 v:=NewDoubleVector(Round(x[0]));
 TheScript(ee).myVecList.Add(v);
 Result:=v.Ref;
end;

 {
 vfree(v) - освобождение массива v
 }
function f_vfree(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var v:TDoubleVector;
begin
 iocheck(ee,narg,1);
 v:=TheScript(ee).FindVector(x[0]);
 if Assigned(v) then begin
  TheScript(ee).myVecList.Remove(v);
  Result:=1;
 end else Result:=0;
end;

 {
 vsize(v) - размерность массива v
 }
function f_vsize(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var v:TDoubleVector;
begin
 iocheck(ee,narg,1);
 v:=TheScript(ee).FindVector(x[0]);
 if Assigned(v)
 then Result:=v.Length
 else Result:=0;
end;

 {
 vget(v,i) - чтение массива v[i], i=0..n-1
 }
function f_vget(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var v:TDoubleVector;
begin
 iocheck(ee,narg,2);
 v:=TheScript(ee).FindVector(x[0]);
 if Assigned(v)
 then Result:=v[Round(x[1])]
 else Result:=0;
end;

 {
 vput(v,i,value) - запись в массив v[i]=value, i=0..n-1
 }
function f_vput(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var v:TDoubleVector;
begin
 iocheck(ee,narg,3);
 v:=TheScript(ee).FindVector(x[0]);
 Result:=1;
 if Assigned(v)
 then v[Round(x[1])]:=x[2]
 else Result:=0;
end;

 {
 igettag(i) - чтение integer тега
 }
function f_igettag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=igettag(Round(x[0]));
end;

 {
 rgettag(i) - чтение real тега
 }
function f_rgettag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=rgettag(Round(x[0]));
end;

 {
 isettag(i,value) - запись integer тега
 }
function f_isettag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=ord(isettag(Round(x[0]),Round(x[1])));
end;

 {
 rsettag(i,value) - запись real тега
 }
function f_rsettag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,2);
 Result:=ord(rsettag(Round(x[0]),x[1]));
end;

 {
 typetag(i) - тип тега
 }
function f_typetag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,1);
 Result:=typetag(Round(x[0]));
end;

 {
 clicktag() - тег нажатого сенсора
 }
function f_clicktag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).ClickTag;
end;

 {
 clickbutton() - тег нажатого сенсора
 }
function f_clickbutton(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin
 iocheck(ee,narg,0);
 Result:=TheScript(ee).ClickButton;
end;

 {
 Процедура регистрирует вышеописанные функции для их применения в скриптах.
 }
procedure TScriptDevice.RegisterFunctions;
begin
 myEe.SetFunc('time',       0, f_time,       RusEng('время по часам DAQ','local DAQ time in defined units'));
 myEe.SetFunc('timeunits',  0, f_timeunits,  RusEng('единицы измерения времени по часам DAQ','local DAQ time units'));
 myEe.SetFunc('crvlen',     1, f_crvlen,     RusEng('число точек кривой a','num.points in curve a'));
 myEe.SetFunc('crvx',       2, f_crvx,       RusEng('координата x кривой a в точке b','x-coordinate of curve a, point index b'));
 myEe.SetFunc('crvy',       2, f_crvy,       RusEng('координата y кривой a в точке b','y-coordinate of curve a, point index b'));
 myEe.SetFunc('crvput',     4, f_crvput,     RusEng('поместить в кривую a, в точку b значения c,d','put (c,d) point to curve a, index b'));
 myEe.SetFunc('crvinteg',   3, f_crvinteg,   RusEng('интеграл кривой a, в пределах b..c','curve a integral on (b,c) interval'));
 myEe.SetFunc('numais',     0, f_numais,     RusEng('число аналоговых входов','number of analog inputs'));
 myEe.SetFunc('numdis',     0, f_numdis,     RusEng('число цифровых входов','number of digital inputs'));
 myEe.SetFunc('numaos',     0, f_numaos,     RusEng('число аналоговых выходов','number of analog outputs'));
 myEe.SetFunc('numdos',     0, f_numdos,     RusEng('число цифровых выходов','number of digital outputs'));
 myEe.SetFunc('numcals',    0, f_numcals,    RusEng('число калибровок','number of calibrations'));
 myEe.SetFunc('refai',      1, f_refai,      RusEng('ссылка на аналоговый вход номер a или 0 если его нет','reference of analog input number a or 0'));
 myEe.SetFunc('refdi',      1, f_refdi,      RusEng('ссылка на цифровой вход номер a или 0 если его нет','reference of digital input number a or 0'));
 myEe.SetFunc('refao',      1, f_refao,      RusEng('ссылка на аналоговый выход номер a или 0 если его нет','reference of analog output number a or 0'));
 myEe.SetFunc('refdo',      1, f_refdo,      RusEng('ссылка на цифровой выход номер a или 0 если его нет','reference of analog digital output number a or 0'));
 myEe.SetFunc('refcalibr',  1, f_refcalibr,  RusEng('ссылка на калибровку номер a или 0 если ее нет','reference of calibration number a or 0'));
 myEe.SetFunc('getai',      2, f_getai,      RusEng('аналоговый вход a в момент времени b','analog input number a at time b'));
 myEe.SetFunc('getai_n',    1, f_getai_n,    RusEng('число точек на аналоговом входе a','num.points of analog input a'));
 myEe.SetFunc('getai_xn',   1, f_getai_xn,   RusEng('x последней точки на аналоговом входе a','x-coordinate of last point of analog input a'));
 myEe.SetFunc('getai_yn',   1, f_getai_yn,   RusEng('у последней точки на аналоговом входе a','y-coordinate of last point of analog input a'));
 myEe.SetFunc('getai_xi',   2, f_getai_xi,   RusEng('х-аналоговый вход a, точка номер b','x-coordinate of analog input curve a at point index b'));
 myEe.SetFunc('getai_yi',   2, f_getai_yi,   RusEng('y-аналоговый вход a, точка номер b','y-coordinate of analog input curve a at point index b'));
 myEe.SetFunc('getdi',      2, f_getdi,      RusEng('цифровой вход a в момент времени b','digital input number a at time b'));
 myEe.SetFunc('getdi_n',    1, f_getdi_n,    RusEng('число точек на цифровом входе a','num.points of digital input a'));
 myEe.SetFunc('getdi_xn',   1, f_getdi_xn,   RusEng('x последней точки на цифровом входе a','x-coordinate of last point of digital input a'));
 myEe.SetFunc('getdi_yn',   1, f_getdi_yn,   RusEng('у последней точки на цифровом входе a','y-coordinate of last point of digital input a'));
 myEe.SetFunc('getdi_xi',   2, f_getdi_xi,   RusEng('х-цифровой вход a, точка номер b','x-coordinate of digital input curve a at point index b'));
 myEe.SetFunc('getdi_yi',   2, f_getdi_yi,   RusEng('y-цифровой вход a, точка номер b','y-coordinate of digital input curve a at point index b'));
 myEe.SetFunc('putev',      5, f_putev,      RusEng('сгенерировать событие(what,chan,time,data0,data1)','put event (what,chan,time,data0,data1)'));
 myEe.SetFunc('putao',      3, f_putao,      RusEng('аналоговый вывод, выход a, время b, данные с','put analog event (chan,time,data)'));
 myEe.SetFunc('putdo',      3, f_putdo,      RusEng('цифровой вывод, выход a,время b, данные с','put digital event (chan,time,data)'));
 myEe.SetFunc('calibr',     3, f_calibr,     RusEng('калибровка номер a, данные b, параметр c','calibration number a, argument b, parameter c'));
 myEe.SetFunc('fixerror',   1, f_fixerror,   RusEng('зафиксировать ошибку номер a','fix error number a'));
 myEe.SetFunc('inportb',    1, f_inportb,    RusEng('ввод байта из порта с адресом a','input byte from port a'));
 myEe.SetFunc('inportw',    1, f_inportw,    RusEng('ввод слова из порта с адресом a','input word from port a'));
 myEe.SetFunc('outportb',   2, f_outportb,   RusEng('вывод байта b в порт с адресом a','write byte to port a'));
 myEe.SetFunc('outportw',   2, f_outportw,   RusEng('вывод слова b в порт с адресом a','write word to port a'));
 myEe.SetFunc('tm_new',     0, f_tm_new,     RusEng('создать новый таймер','create new timer, return handle'));
 myEe.SetFunc('tm_free',    1, f_tm_free,    RusEng('удалить таймер a','destroy timer'));
 myEe.SetFunc('tm_addint',  2, f_tm_addint,  RusEng('добавить в таймер a новый интервал b,[ms]','add timer a interval b, ms'));
 myEe.SetFunc('tm_numint',  1, f_tm_numint,  RusEng('узнать число интервалов таймера a','number of timer a intervals'));
 myEe.SetFunc('tm_getint',  2, f_tm_getint,  RusEng('узнать длину интервала номер b таймера a,[ms]','return timer a interval number b in ms'));
 myEe.SetFunc('tm_setint',  3, f_tm_setint,  RusEng('задать длину c,[ms] интервала номер b таймера a','set timer a interval number b in ms'));
 myEe.SetFunc('tm_gettime', 1, f_tm_gettime, RusEng('узнать время со старта таймера a, [ms]','time since timer start in ms'));
 myEe.SetFunc('tm_start',   1, f_tm_start,   RusEng('пуск таймера a','start timer'));
 myEe.SetFunc('tm_stop',    1, f_tm_stop,    RusEng('останов таймера a','stop timer'));
 myEe.SetFunc('tm_isstart', 1, f_tm_isstart, RusEng('узнать запущен ли таймер a','check timer is started'));
 myEe.SetFunc('tm_event',   1, f_tm_event,   RusEng('генерация событий таймера a','return true on timer event'));
 myEe.SetFunc('tm_curint',  1, f_tm_curint,  RusEng('номер текущего интервала таймера a','current timer interval'));
 myEe.SetFunc('vnew',       1, f_vnew,       RusEng('создаст массив размерности a и вернет ссылку','create new array'));
 myEe.SetFunc('vfree',      1, f_vfree,      RusEng('удалит созданный vnew массив заданный ссылкой a','destroy array after vnew'));
 myEe.SetFunc('vsize',      1, f_vsize,      RusEng('вернет размерность массива заданного ссылкой a','dimension of array'));
 myEe.SetFunc('vget',       2, f_vget,       RusEng('вернет элемент массива a с номером b','return array value a[b]'));
 myEe.SetFunc('vput',       3, f_vput,       RusEng('записать в элемент массива a номер b значение c','write array value a[b]=c'));
 myEe.SetFunc('igettag',    1, f_igettag,    RusEng('прочитать integer тег a','read integer tag a'));
 myEe.SetFunc('rgettag',    1, f_rgettag,    RusEng('прочитать real тег a','read real tag a'));
 myEe.SetFunc('isettag',    2, f_isettag,    RusEng('записать в integer тег a значение b','write b to integer tag a'));
 myEe.SetFunc('rsettag',    2, f_rsettag,    RusEng('записать в real тег a значение b','write b to real tag a'));
 myEe.SetFunc('typetag',    1, f_typetag,    RusEng('возвращает тип тега a, 1=integer,2=real,3=string'));
 myEe.SetFunc('clicktag',   0, f_clicktag,   RusEng('возвращает 0 или нажатый тег','0 or clicked sensor tag'));
 myEe.SetFunc('clickbutton',0, f_clickbutton,RusEng('возвращает 0 или нажатую клавишу мыши 1=левая 2=правая 4=средняя','0 or clicked mouse button 1/2/4=Left/Right/Middle'));
end;

const
 arg_delims=[' ',',',#9];

 {
 @inittag name type - создает новый или возвращает существующий тег по имени
                      name с типом type = 1,2,3
 }
function act_inittag(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer;
begin
 if Str2Int(ExtractWord(2,ee.SmartArgs(args),ScanSpaces),i)
 then Result:=inittag(ExtractWord(1,ee.SmartArgs(args),ScanSpaces),i)
 else Result:=0;
end;

 {
 @findtag name - возвращает тег по имени
 }
function act_findtag(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=findtag(Trim(ee.SmartArgs(args)));
end;

 {
 @clicksensor name - проверка, нажат ли сенсор name
 }
function act_clicksensor(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).Ok
 then Result:=Ord(SameText(UnifyAlias(ee.SmartArgs(args)),UnifyAlias(TheScript(ee).ClickSensor)))
 else Result:=0
end;

 {
 @echo msg - выдает строку msg в консольное окно
 }
function act_echo(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=1;
 Daq.ConsoleEcho(ee.SmartArgs(args));
end;

 {
 @debugout msg - выдает строку в файл отладки
 }
function act_debugout(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=1;
 DebugOut(stdfDebug,ee.SmartArgs(args));
end;

 {
 @devmsg     name msg - посылка устройству c именем name сообщения msg с "пробуждением"
 @devsend    name msg - посылка устройству c именем name сообщения msg с "пробуждением"
 @devsendmsg name msg - посылка устройству c именем name сообщения msg с "пробуждением"
 }
function act_devsend(ee:TExpressionEvaluator; const args:LongString):double;
var s,Name,Msg:LongString; Device:TDaqDevice;
begin
 s:=ee.SmartArgs(args);
 Name:=ExtractWord(1,s,arg_delims);
 Device:=FullDaqDeviceList.Find(name);
 if Device.Ok then begin
  Msg:=SkipWords(1,s,arg_delims);
  Result:=Device.HandleMessage(Msg);
 end else Result:=0;
end;

 {
 @devpost    name msg - посылка устройству c именем name сообщения msg без "пробуждения"
 @devpostmsg name msg - посылка устройству c именем name сообщения msg без "пробуждения"
 }
function act_devpost(ee:TExpressionEvaluator; const args:LongString):double;
var s,Name,Msg:LongString; Device:TDaqDevice;
begin
 s:=ee.SmartArgs(args);
 Name:=ExtractWord(1,s,arg_delims);
 Device:=FullDaqDeviceList.Find(name);
 if Device.Ok then begin
  Msg:=SkipWords(1,s,arg_delims);
  Result:=Device.HandleMessage(Msg,hf_Default or hf_SkipAwake);
 end else Result:=0;
end;

 {
 Найти спектрометрическое окно по имени в списке окон DAQ
 }
function FindSpecWin(const aName:LongString):TFormSpectrWindow;
var i:Integer;
begin
 Result:=nil;
 for i:=0 to Daq.SpeWinList.Count-1 do
 with Daq.SpeWinList[i] do if Ok then
 if SameText(UnifyAlias(Caption),UnifyAlias(aName)) then begin
  Result:=Daq.SpeWinList[i];
  break;
 end;
end;

 {
 @specmarker name возвращает в actionresult Marker спетрометрического окна name
 }
function act_specmarker(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=FindSpecWin(ee.SmartArgs(args)).Marker;
end;

 {
 @specmarkerl name возвращает в actionresult MarkerL спетрометрического окна name
 }
function act_specmarkerl(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=FindSpecWin(ee.SmartArgs(args)).MarkerL;
end;

 {
 @specmarkerr name возвращает в actionresult MarkerR спетрометрического окна name
 }
function act_specmarkerr(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=FindSpecWin(ee.SmartArgs(args)).MarkerR;
end;

 {
 @specroil name возвращает в actionresult ROIL спетрометрического окна name
 }
function act_specroil(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=FindSpecWin(ee.SmartArgs(args)).RoiL;
end;

 {
 @specroir name возвращает в actionresult ROIR спетрометрического окна name
 }
function act_specroir(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=FindSpecWin(ee.SmartArgs(args)).RoiR;
end;

 {
 @action list - выполняет метод action для списка устройств list
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_action(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer; Device:TDaqDevice;
begin
 Result:=1;
 for i:=1 to WordCount(ee.SmartArgs(args),arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,ee.SmartArgs(args),arg_delims));
  if Device.Ok and TheScript(ee).PostDeferredCommand('@ACTION '+Device.Name+EOL)
  then // Ok
  else Result:=0;
 end;
end;

 {
 @clear list - выполняет метод clear для списка устройств list
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_clear(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer; Device:TDaqDevice;
begin
 Result:=1;
 for i:=1 to WordCount(ee.SmartArgs(args),arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,ee.SmartArgs(args),arg_delims));
  if Device.Ok and TheScript(ee).PostDeferredCommand('@CLEAR '+Device.Name+EOL)
  then // Ok
  else Result:=0;
 end;
end;

 {
 @cleardevice list - выполняет метод cleardevice для списка устройств list
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_cleardevice(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer; Device:TDaqDevice;
begin
 Result:=1;
 for i:=1 to WordCount(ee.SmartArgs(args),arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,ee.SmartArgs(args),arg_delims));
  if Device.Ok and TheScript(ee).PostDeferredCommand('@CLEARDEVICE '+Device.Name+EOL)
  then // Ok
  else Result:=0;
 end;
end;

 {
 @start list - выполняет метод start для списка устройств list
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_start(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer; Device:TDaqDevice;
begin
 Result:=1;
 for i:=1 to WordCount(ee.SmartArgs(args),arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,ee.SmartArgs(args),arg_delims));
  if Device.Ok and TheScript(ee).PostDeferredCommand('@START '+Device.Name+EOL)
  then // Ok
  else Result:=0;
 end;
end;

 {
 @stop list - выполняет метод stop для списка устройств list
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_stop(ee:TExpressionEvaluator; const args:LongString):double;
var i:Integer; Device:TDaqDevice;
begin
 Result:=1;
 for i:=1 to WordCount(ee.SmartArgs(args),arg_delims) do begin
  Device:=FullDaqDeviceList.Find(ExtractWord(i,ee.SmartArgs(args),arg_delims));
  if Device.Ok and TheScript(ee).PostDeferredCommand('@STOP '+Device.Name+EOL)
  then // Ok
  else Result:=0;
 end;
end;

 {
 @clearcurve list - очищает кривые из списка list
 }
function act_clearcurve(ee:TExpressionEvaluator; const args:LongString):double;
var i,d,HistLen:Integer; Curve:TCurve;
begin
 Result:=1;
 HistLen:=0;
 for i:=1 to WordCount(ee.SmartArgs(args),arg_delims) do
 if Str2Int(ExtractWord(i,ee.SmartArgs(args),arg_delims),d)
 then HistLen:=d
 else begin
  Curve:=Daq.Curves.Find(ExtractWord(i,ee.SmartArgs(args),arg_delims));
  if Curve.Ok
  then DaqClearCurve(Curve,HistLen)
  else Result:=0;
 end;
end;

 {
 @winopen name - открывает окно с именем name
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_winopen(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).PostDeferredCommand('@WINOPEN '+ee.SmartArgs(args)+EOL)
 then Result:=1
 else Result:=0;
end;

 {
 @windraw name - прорисовывает окно с именем name
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_windraw(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).PostDeferredCommand('@WINDRAW '+ee.SmartArgs(args)+EOL)
 then Result:=1
 else Result:=0;
end;

 {
 @winshow name - показывает окно с именем name
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_winshow(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).PostDeferredCommand('@WINSHOW '+ee.SmartArgs(args)+EOL)
 then Result:=1
 else Result:=0;
end;

 {
 @winhide name - сворачивает окно с именем name
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_winhide(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).PostDeferredCommand('@WINHIDE '+ee.SmartArgs(args)+EOL)
 then Result:=1
 else Result:=0;
end;

 {
 @winselect name - показывает окно с именем name
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_winselect(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).PostDeferredCommand('@WINSELECT '+ee.SmartArgs(args)+EOL)
 then Result:=1
 else Result:=0;
end;

 {
 @global expression - вычисляет выражение expression в системном калькуляторе
 }
function act_global(ee:TExpressionEvaluator; const args:LongString):double;
begin
 Result:=SystemCalculator.Eval(ee.SmartArgs(args));
end;

 {
 @savecrw fname winname wintitle winlable curvelist - сохраняет список кривых в файл CRW
  fname      - имя файла (можно относительно конфигфайла и без расширения)
  winname    - имя окна под которым оно попадет в crw-архив
  wintitle   - заголовок в верхней части окна
  winlable   - метка в нижней части окна
  curvelist  - список сохраняемых кривых
 Выполняется в основном потоке VCL, в методе Idle
 }
function act_savecrw(ee:TExpressionEvaluator; const args:LongString):double;
begin
 if TheScript(ee).PostDeferredCommand('@SAVECRW '+ee.SmartArgs(args)+EOL)
 then Result:=1
 else Result:=0;
end;

 {
 Процедура регистрирует вышеописанные директивы для их применения в скриптах.
 }
procedure TScriptDevice.RegisterActions;
begin
 myEe.SetAction('inittag',     act_inittag,     RusEng('найти/создать тег с именем %1 и типом %2=1..3','find/create tag with name %1, type %2=1..3'));
 myEe.SetAction('findtag',     act_findtag,     RusEng('найти тег с именем %1','find tag with name %1'));
 myEe.SetAction('clicksensor', act_clicksensor, RusEng('проверяет, нажат ли сенсор с именем %1','check if sensor with name %1 clicked'));
 myEe.SetAction('echo',        act_echo,        RusEng('вывод сообщения в консольное окно','output message to console'));
 myEe.SetAction('debugout',    act_debugout,    RusEng('вывод сообщения в отладочный файл','output message to debug file'));
 myEe.SetAction('devmsg',      act_devsend,     RusEng('синхронная посылка сообщения %2 устройству %1','send message %2 to device %1'));
 myEe.SetAction('devsend',     act_devsend,     RusEng('синхронная посылка сообщения %2 устройству %1','send message %2 to device %1'));
 myEe.SetAction('devsendmsg',  act_devsend,     RusEng('синхронная посылка сообщения %2 устройству %1','send message %2 to device %1'));
 myEe.SetAction('devpost',     act_devpost,     RusEng('асинхронная посылка сообщения %2 устройству %1','post message %2 to device %1'));
 myEe.SetAction('devpostmsg',  act_devpost,     RusEng('асинхронная посылка сообщения %2 устройству %1','post message %2 to device %1'));
 myEe.SetAction('specmarker',  act_specmarker,  RusEng('выдать Marker спектрометрич. окна %1','get marker of spectrometry window'));
 myEe.SetAction('specmarkerl', act_specmarkerl, RusEng('выдать MarkerL спектрометрич. окна %1','get left marker of spectrometry window %1'));
 myEe.SetAction('specmarkerr', act_specmarkerr, RusEng('выдать MarkerR спектрометрич. окна %1','get right marker of spectrometry window %1'));
 myEe.SetAction('specroil',    act_specroil,    RusEng('выдать ROIL спектрометрич. окна %1','get left ROI marker of spectrometry window %1'));
 myEe.SetAction('specroir',    act_specroir,    RusEng('выдать ROIR спектрометрич. окна %1','get right ROI marker of spectrometry window %1'));
 myEe.SetAction('action',      act_action,      RusEng('вызов action для списка устройств'));
 myEe.SetAction('clear',       act_clear,       RusEng('вызов clear для списка устройств'));
 myEe.SetAction('cleardevice', act_cleardevice, RusEng('вызов cleardevice для списка устройств'));
 myEe.SetAction('start',       act_start,       RusEng('вызов start для списка устройств'));
 myEe.SetAction('stop',        act_stop,        RusEng('вызов stop для списка устройств'));
 myEe.SetAction('clearcurve',  act_clearcurve,  RusEng('очистка списка кривых'));
 myEe.SetAction('winopen',     act_winopen,     RusEng('загружает/открывает окно c заданным именем'));
 myEe.SetAction('windraw',     act_windraw,     RusEng('перерисовывает окно c заданным именем'));
 myEe.SetAction('winshow',     act_winshow,     RusEng('показывает окно c заданным именем'));
 myEe.SetAction('winhide',     act_winhide,     RusEng('прячет окно c заданным именем'));
 myEe.SetAction('winselect',   act_winselect,   RusEng('делает окно c заданным именем активным'));
 myEe.SetAction('global',      act_global,      RusEng('вычисляет выражение в общем калькуляторе'));
 myEe.SetAction('savecrw',     act_savecrw,     RusEng('сохранение данных (списка кривых) в crw-файл'));
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_softdevscript;
begin
end;

procedure Free_crw_softdevscript;
begin
end;

initialization

 Init_crw_softdevscript;

finalization

 Free_crw_softdevscript;

end.

//////////////
// END OF FILE
//////////////

