////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// CRW-DAQ Software Device Bit Set Generator.                                 //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231207 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_softdevbitsetgenerator; // CRW-DAQ Software Device Bit Set Generator

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd, Printers,
 lcltype, lclintf,
 Form_CrwDaqSysChild,
 Form_TextEditor, Form_CurveWindow,
 Form_SurfWindow, Form_CircuitWindow,
 Form_ConsoleWindow, Form_TabWindow,
 Form_SpectrWindow, Form_Calculator,
 Form_ListBoxSelection, Form_UartTerminal,
 Form_CalibDialog, Form_DaqEditTagDialog,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_pio, _crw_curves, _crw_riff,
 _crw_calib, _crw_couple, _crw_daqtags, _crw_daqevnt,
 _crw_daqsys, _crw_daqdev, _crw_softdev,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Объект:       BitSetGeneratorDevice
 Объявление:   Device Software BitSetGenerator
 Назначение:   Служит для генерации сигнала - константы для цифровых и
               аналоговых каналов для целей отладки или управления.
 Входы:        Нет.
 Выходы:       AnalogOutput(0)  - аналоговая константа.
               DigitalOutput(0) - цифровая константа.
 Описание:     При наличии подключенных AnalogOutput(0),DigitalOutput(0)
               генерирует аналоговое или цифровое событие, которое помещает
               в fifo константу для передачи на эти выходы.
               Значение константы можно менять через диалог свойств.
               Кроме того ConstValue,ConstName можно менять через сообщение
               @devmsg devicename ConstValue=0, ConstName=xxx
               Отличается от ConstGenerator тем, что константа задается
               по битам, то есть ориентирована на цифровые сигналы.
 Конфигурация: ConstValue = n
               ConstName = xxx
               По умолчанию ConstValue = 0, ConstName = ConstValue
 *******************************************************************************              
 }
type
 TBitSetGeneratorDevice = class(TSoftwareDevice)
 private
  myConstValue   : LongInt;
  myConstName    : LongString;
  myCloseOnEnter : Boolean;
  myLinkedTag    : Integer;
  myPrevValue    : LongInt;
  function    GetConstValue:LongInt;
  procedure   SetConstValue(aConstValue:LongInt);
  function    GetPrevValue:LongInt;
  function    GetConstName:LongString;
  procedure   SetConstName(const aConstName:LongString);
  function    GetCloseOnEnter:Boolean;
  function    GetLinkedTag:Integer;
 public
  property    ConstValue   : LongInt     read GetConstValue   write SetConstValue;
  property    PrevValue    : LongInt     read GetPrevValue;
  property    ConstName    : LongString  read GetConstName    write SetConstName;
  property    CloseOnEnter : Boolean     read GetCloseOnEnter;
  property    LinkedTag    : Integer     read GetLinkedTag;
 public
  constructor Create(const aName:LongString);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(P:TText):TText; override;
  procedure   Handler; override;
  procedure   Action; override;
  function    NewPropertyDialog:TDaqDevicePropertyDialog; override;
  function    UpdatePropertyDialog:Boolean; override;
  function    HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double; override;
 end;

implementation

uses
 Form_BitSetGeneratorPropertyDialog;

 {
 ***********************************
 BitSetGeneratorDevice implementation
 ***********************************
 }
function TBitSetGeneratorDevice.GetConstValue:LongInt;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myConstValue;
  Unlock;
 end else Result:=0;
end;

procedure TBitSetGeneratorDevice.SetConstValue(aConstValue:LongInt);
begin
 if Assigned(Self) then begin
  Lock;
  myConstValue:=aConstValue;
  Unlock;
 end;
end;

function TBitSetGeneratorDevice.GetPrevValue:LongInt;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myPrevValue;
  myPrevValue:=myConstValue;
  Unlock;
 end else Result:=0;
end;

function TBitSetGeneratorDevice.GetConstName:LongString;
begin
 if Assigned(Self) then begin
  Lock;
  Result:=myConstName;
  Unlock;
 end else Result:='';
end;

procedure TBitSetGeneratorDevice.SetConstName(const aConstName:LongString);
begin
 if Assigned(Self) then begin
  Lock;
  myConstName:=aConstName;
  Unlock;
 end;
end;

function TBitSetGeneratorDevice.GetCloseOnEnter:Boolean;
begin
 if Assigned(Self) then Result:=myCloseOnEnter else Result:=false;
end;

function TBitSetGeneratorDevice.GetLinkedTag:Integer;
begin
 if Assigned(Self) then Result:=myLinkedTag else Result:=0;
end;

constructor TBitSetGeneratorDevice.Create(const aName:LongString);
begin
 inherited Create(aName);
 SetDeviceModel('BITSETGENERATOR');
 AnalogFifoSize:=64;
 DigitalFifoSize:=64;
 NumAnalogInputs:=0;
 NumDigitalInputs:=0;
 NumAnalogOutputs:=1;
 NumDigitalOutputs:=1;
 NumCalibrations:=0;
 myConstValue:=0;
 myPrevValue:=0;
 myConstName:=Name+'_Value';
 myCloseOnEnter:=true;
 myLinkedTag:=0;
end;

destructor TBitSetGeneratorDevice.Destroy;
begin
 myConstName:='';
 inherited Destroy;
end;

procedure TBitSetGeneratorDevice.Config(FileName:LongString);
var TagName:LongString;
begin
 if Ok then begin
  TagName:='';
  FileName:=UnifyFileAlias(FileName);
  inherited Config(FileName);
  ReadIniFileLongInt(FileName,DevSection,'ConstValue%d',myConstValue);
  ReadIniFileAlpha(FileName,DevSection,'ConstName%a',myConstName);
  ReadIniFileBoolean(FileName,DevSection,'CloseOnEnter%b',myCloseOnEnter);
  if ReadIniFileAlpha(FileName,DevSection,'LinkedTag%a',TagName) then begin
   myLinkedTag:=FindTag(TagName);
   case TypeTag(LinkedTag) of
    1:isettag(LinkedTag,ConstValue);
    2:rsettag(LinkedTag,ConstValue);
    3:ssettag(LinkedTag,Format('%d',[ConstValue]));
    else Daq.AddWarning('Device:'+Name+' -> Not found tag '+TagName);
   end;
  end;
 end;
end;

function  TBitSetGeneratorDevice.GetProperty(P:TText):TText;
begin
 GetProperty:=inherited GetProperty(P);
 if Ok then begin
  P.AddLn('ConstValue = '+HexL(ConstValue));
  P.AddLn('ConstName = '+ConstName);
  P.Addln('CloseOnEnter = '+d2s(ord(CloseOnEnter)));
  P.Addln('LinkedTag = '+NameTag(LinkedTag));
 end;
end;

procedure  TBitSetGeneratorDevice.Handler;
begin
 if Assigned(Self) then begin
  if (AnalogOutputCurve[0]<>nil)
  then PutDaqEvent(DaqEvent(evAnalog, 0, Daq.Timer.LocalTime, ConstValue));
  if (DigitalOutputCurve[0]<>nil)
  then PutDaqEvent(DaqEvent(evDigital, 0, Daq.Timer.LocalTime, ConstValue));
  case TypeTag(LinkedTag) of
   1 : iSetTag(LinkedTag,ConstValue);
   2 : rSetTag(LinkedTag,ConstValue);
   3 : sSetTag(LinkedTag,Format('%d',[ConstValue]));
  end;
 end;
end;

procedure TBitSetGeneratorDevice.Action;
begin
 OpenPropertyDialog;
 if (PropertyDialog is TFormBitSetGeneratorPropertyDialog) then
 with TFormBitSetGeneratorPropertyDialog(PropertyDialog) do LocateToMouse;
end;

function TBitSetGeneratorDevice.NewPropertyDialog:TDaqDevicePropertyDialog;
begin
 Result:=NewFormBitSetGeneratorPropertyDialog;
 if (Result is TFormBitSetGeneratorPropertyDialog) then
 with TFormBitSetGeneratorPropertyDialog(Result) do begin
  LinkedDevice:=Self;
  UpdateControls(1+2+4);
  Caption:=Self.Model+': '+Self.Name;
 end;
end;

function TBitSetGeneratorDevice.UpdatePropertyDialog:Boolean;
begin
 Result:=false;
 if Ok then begin
  if (PropertyDialog is TFormBitSetGeneratorPropertyDialog) then
  with TFormBitSetGeneratorPropertyDialog(PropertyDialog) do
  if Visible and (WindowState=wsNormal) then begin
   UpdateControls(4);
   if (PrevValue<>ConstValue) then begin
    UpdateControls(2);
    Result:=true;
   end;
  end;
 end;
end;

function TBitSetGeneratorDevice.HandleMessage(const aMsg:LongString; aFlags:Cardinal=hf_Default):Double;
var d:LongInt; s:LongString; buf:TParsingBuffer;
begin
 Result:=1; d:=0;
 if (ScanVarLongInt(svConfig,StrUpper(StrCopyBuff(buf,aMsg)),'ConstValue%d',d)<>nil)
 then ConstValue:=d;
 s:=ConstName;
 if (ScanVarAlpha(svConfig,StrUpper(StrCopyBuff(buf,aMsg)),'ConstName%a',s)<>nil)
 then if (s<>ConstName) and IsNonEmptyStr(s)
 then ConstName:=s;
 if not HasFlags(aFlags,hf_SkipAwake) then Awake;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_softdevbitsetgenerator;
begin
end;

procedure Free_crw_softdevbitsetgenerator;
begin
end;

initialization

 Init_crw_softdevbitsetgenerator;

finalization

 Free_crw_softdevbitsetgenerator;

end.

//////////////
// END OF FILE
//////////////

