////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2023 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// DAQ DIO-144 Device driver.                                                 //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231204 - Modified for FPC (A.K.)                                         //
////////////////////////////////////////////////////////////////////////////////

unit _crw_daqdio144device; // DAQ DIO-144 Device driver

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf,
 Form_CrwDaqSysChild, Form_TextEditor, Form_CurveWindow, Form_SurfWindow,
 Form_CircuitWindow, Form_ConsoleWindow, Form_TabWindow, Form_SpectrWindow,
 Form_Calculator, Form_ListBoxSelection, Form_UartTerminal, Form_CalibDialog,
 Form_DaqEditTagDialog,
 Unit_SystemConsole,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_fio, _crw_plut,
 _crw_dynar, _crw_snd, _crw_guard,
 _crw_ef, _crw_curves, _crw_riff, _crw_pio,
 _crw_calib, _crw_couple, _crw_polling,
 _crw_daqtags, _crw_daqevnt, _crw_daqsys,
 _crw_daqdev, _crw_i8255, _crw_pcldev,
 _crw_appforms, _crw_apptools, _crw_apputils;

 {
 *******************************************************************************
 Объект:       DIO144Device
 Объявление:   Device PCL DIO-144
               Device PCL DIO-24
               Device PCL PCL-731
               Device PCL DIO-48
 Назначение:   Семейство карт цифрового ввода-вывода на основе чипа i8255.
 Входы:        DigitalInput(...)  - цифровой вывод. Число каналов:
               DIO-144 - до 144
               DIO-24  - до 24
               PCL-731 - до 48
               DIO-48  - до 48
               Входы объединены в группы по 24, которые моделируют чип i8255.
 Выходы:       DigitalOutput(...) - цифровой ввод. Число каналов:
               DIO-144 - 6
               DIO-24  - 1
               PCL-731 - 2
               DIO-48  - 2
               Каждый DigitalOutput представляет собой 24-битное число,
               моделирующее порты чипа i8255.
 Описание:     DIO144Device реализует семейство карт цифрового ввода-вывода
               на основе чипа i8255.
               Карта DIO-24  имеет 1 i8255
               Карта DIO-144 имеет 6 i8255
               Карта PCL-731 имеет 2 i8255
               Карта DIO-48  имеет 2 i8255
               Чип i8255 имеет 3 8-битных порта A,B,C, причем порт C разделен
               на младший полубайт CL и старший CH. Можно считать, что i8255
               представляет 24-битное число, в котором биты делятся на группы
               A  - биты 0..7
               B  - биты 8..15
               CL - биты 16..19
               CH - биты 20..23
               Каждая из этих групп может быть запрограммирована для работы
               на чтение или на запись. Поэтому реальное число входов-выходов
               устройства зависит от конфигурации. Для устройства было принято
               такое соглашение:
               1. Каждый чип i8255 номер n=0..5 проецируется на
                  a)24 входа с номерами n*24..n*24+23
                    A  - DigitalInput(24*n    .. 24*n+7)
                    B  - DigitalInput(24*n+8  .. 24*n+15)
                    CL - DigitalInput(24*n+16 .. 24*n+19)
                    CH - DigitalInput(24*n+20 .. 24*n+23)
                  b)24-битный выход номер n
                    A  - DigitalOutput(n), биты 0..7
                    B  - DigitalOutput(n), биты 8..15
                    CL - DigitalOutput(n), биты 16..19
                    CH - DigitalOutput(n), биты 20..23
               2. Каждый чип имеет свою конфигурацию портов A,B,CL,CH.
               3. Если порт A,B,CL,CH запрограммирован на ввод,
                  то соответствующая группа DigitalInput-ов не используется,
                  а группа битов  DigitalOutput используется.
               4. Если порт A,B,CL,CH запрограммирован на вывод,
                  то соответствующая группа DigitalInput-ов используется,
                  а группа битов  DigitalOutput не используется и равна 0.
 Конфигурация: i8255#n = A B CL CH
               где n-номер чипа i8255
               A  - порт A (биты 0..7)
               B  - порт A (биты 8..15)
               CL - порт C, младший полубайт (биты 16..19)
               CH - порт C, старший полубайт (биты 20..23)
               Если значение A/B/CL/CH равно 1, то порт работает на чтение,
               если 0 - то на запись.
 *******************************************************************************
 }
type
 TDIO144Device = class(TPCLDevice)
 protected
  n8255       : word;
  i8255       : array[0..5] of TI8255;
 public
  constructor Create(const aName:LongString; aBase,an8255:word; const aModel:LongString);
  destructor  Destroy; override;
  procedure   Config(FileName:LongString); override;
  function    GetProperty(TheText:TText):TText; override;
  procedure   Handler; override;
 end;

implementation

 {
 *******************************************************
 TDIO144Device,TDIO24Device,TPCL731Device implementation
 *******************************************************
 }
constructor TDIO144Device.Create(const aName:LongString; aBase,an8255:Word; const aModel:LongString);
var i:Integer;
begin
 inherited Create(aName,aBase);
 SetDeviceModel(aModel);
 n8255:=an8255;
 di_chan_no:=n8255*24;
 do_chan_no:=n8255*24;
 DigitalFifoSize:=n8255*24;
 NumDigitalInputs:=n8255*24;
 NumDigitalOutputs:=n8255;
 for i:=0 to n8255-1 do i8255[i]:=NewI8255(Base+i*4);
end;

destructor  TDIO144Device.Destroy;
var i:Integer;
begin
 for i:=0 to n8255-1 do Kill(i8255[i]);
 inherited Destroy;
end;

procedure TDIO144Device.Config(FileName:LongString);
var i:Integer;
begin
 FileName:=UnifyFileAlias(FileName);
 inherited Config(FileName);
 for i:=0 to n8255-1 do i8255[i].Config(FileName,DevSection,'i8255#'+d2s(i));
end;

function  TDIO144Device.GetProperty(TheText:TText):TText;
var i:Integer;
begin
 GetProperty:=inherited GetProperty(TheText);
 for i:=0 to n8255-1 do
 TheText.Addln('i8255#'+d2s(i)+' = '+
               d2s(ord(HasFlags(i8255[i].Mode,id8255PortA)))+' '+
               d2s(ord(HasFlags(i8255[i].Mode,id8255PortB)))+' '+
               d2s(ord(HasFlags(i8255[i].Mode,id8255PortCLo)))+' '+
               d2s(ord(HasFlags(i8255[i].Mode,id8255PortCHi))));
end;

procedure TDIO144Device.Handler;
var
 i : Integer;
 d : packed array[0..5] of packed record
     case LongInt of
      0: (L : LongInt);
      1: (b : packed array[0..3] of byte;)
     end;
begin
 {ввод}
 for i:=0 to n8255-1 do begin
  d[i].L:=0;
  i8255[i].Read(d[i].b[0]);
  if DigitalOutputCurve[i].Ok
  then PutDaqEvent(DaqEvent(evDigital,i,Daq.Timer.LocalTime,d[i].L));
 end;
 {вывод}
 for i:=0 to n8255-1 do begin
  d[i].L:=ConstructDigitalInputWord(i*24,24);
  i8255[i].Write(d[i].b[0]);
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_crw_daqdio144device;
begin
end;

procedure Free_crw_daqdio144device;
begin
end;

initialization

 Init_crw_daqdio144device;

finalization

 Free_crw_daqdio144device;

end.

//////////////
// END OF FILE
//////////////

