////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2025 Alexey Kuryakin daqgroup@mail.ru under MIT license //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - component CRWLIB.   //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// Form Circuit Window.                                                       //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20231122 - Modified for FPC (A.K.)                                         //
// 20240627 - FormMouseWheel fixed (wrong coordinates calculation)            //
// 20250129 - Use TAtomicCounter                                              //
// 20251212 - FindWantedSize,ApplyWantedSize                                  //
////////////////////////////////////////////////////////////////////////////////

unit form_circuitwindow; // Form Circuit Window

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$WARN 5023 off : Unit "$1" not used in $2}
{$WARN 6058 off : Call to subroutine "$1" marked as inline is not inlined}

interface

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, strutils, math,
 Graphics, Controls, Forms, Dialogs, LMessages,
 ExtCtrls, ComCtrls, StdCtrls, Buttons, Menus,
 ActnList, ToolWin, ImgList, Clipbrd,
 lcltype, lclintf, lclproc, Printers,
 Form_CrwDaqSysChild,
 Form_ListBoxSelection, Form_TextEditDialog,
 _crw_alloc, _crw_fpu, _crw_rtc, _crw_fio, _crw_fifo,
 _crw_str, _crw_eldraw, _crw_plut, _crw_sort, _crw_dynar,
 _crw_snd, _crw_guard, _crw_colors, _crw_fonts,
 _crw_daqtags, _crw_curves, _crw_ee, _crw_ef,
 _crw_appforms, _crw_apptools, _crw_apputils,
 _crw_bmpcache;

 {
 Обьект для создания списка сенсоров - чувствительных областей изображения.
 Сенсор имеет имя, положение в координатах основного поля и набор картинок
 для отображения текущего состояния сенсора. Если указан формат LED.width<>0,
 также рисуется надпись LED.value.
 }
type
 TCircuitSensor = class(TMasterObject)
 private
  myItems              : TObjectStorage;{ Список изображений TSmartBitmap                }
  myName               : LongString;    { Имя сенсора для его идентификации              }
  myPos                : TPoint2I;      { Положение на основном поле изображения         }
  myBookMark           : LongInt;       { Идентификатор текущей картинки в списке        }
  myLedWidth           : Integer;       { Ширина поля или 0 если надпись не отображается }
  myLedDigit           : Integer;       { Цифр после запятой для отображения             }
  myLedValue           : LongString;    { Текущее значение отображаемой величины         }
  myLedDrawn           : LongString;    { Text of LED which really was drawn last time   }
  myLedFormat          : LongString;    { Формат типа %8.3g                              }
  myLedFormatType      : Integer;       { 0/1/2/3=Invalid/Integer/Real/String            }
  myLedFontLink        : LongString;    { Фонт для отображения числа - секция описания   }
  myLedFontPara        : TFontParams;   { Фонт для отображения числа - параметры         }
  myLinkedTag          : Integer;       { Присоединенный тег                             }
  myLinkedTagColor     : Integer;       { Присоединенный тег - Color                     }
  myLinkedTagParam     : Double;        { Присоединенный тег - Param                     }
  myLinkedTagTimer     : Double;        { Присоединенный тег - Timer                     }
  myUsesLinkedTagColor : Boolean;       { Присоединенный тег - Color - используется?     }
  myUsesLinkedTagParam : Boolean;       { Присоединенный тег - Param - используется?     }
  myUsesLinkedTagTimer : Boolean;       { Присоединенный тег - Timer - используется?     }
  myLinkedCurve        : TCurve;        { Присоединенная кривая                          }
  myLinkedValue        : Double;        { Last value of linked tag/curve                 }
  myLinkedObject       : TObject;       { Присоединенный объект                          }
  myTextColor          : TColor;        { Цвет для отображения текста                    }
  myElseMark           : LongInt;       { Идентификатор картинки в списке по умолчанию   }
  myShortCut           : Integer;       { Горячая клавиша для доступа к сенсору          }
  myHintText           : LongString;    { Текст описания (комментария) к сенсору         }
  myHintLine           : LongString;    { Строка описания - первая строка HintText       }
  myToolBarKey         : Integer;       { 0 или клавиша для помещения сенсора в ToolBar  }
  myToolBarItems       : TObjectStorage;{ Список элементов TImage на панели ToolBar      }
  myToolBarOpaque      : LongBool;      { Непрозрачный ToolBar                           }
  myToolBarBorder      : LongBool;      { Наличие рамки ToolBar                          }
  myToolBarSpace       : Integer;       { Свободное пространство внутри рамки            }
  myTagEval            : LongString;    { Формула для вычисления тега по значению        }
  myLedEval            : LongString;    { Формула для вычисления LED  по значению        }
  myPainter            : LongString;    { Скрипт для рисования на сенсоре - полный текст }
  myPainterCall        : LongString;    { Скрипт для рисования на сенсоре - код вызова   }
  myPainterData        : record         { Private data for Painter                       }
   Window,Canvas,Bitmap:Pointer;        { Current painter window, canvas and bitmap      }
   pc,ps,pm,pw         : Integer;       { Current Pen color,style,mode,width             }
   bc,bs               : Integer;       { Current Brush color,style                      }
   cx,cy               : Integer;       { Cursor coordinates                             }
   lx,ly               : Integer;       { LED offset                                     }
   IsSimple            : LongBool;      { Is Painter simple script?                      }
   PolyPoints          : array of TPoint; { Uses by PolyLine,Polygon,PolyBezier          }
  end;                                  {                                                }
  myPendingDraw        : TAtomicCounter;  { Pending draw flag                            }
  myParentForm         : TForm;         { Reference to parent CircuitWindow              }
  myCrcFile            : LongString;    { Source Crc File name                           }
  mySection            : LongString;    { Source Crc File section                        }
  myGuardLevel         : Integer;       { Guard level required to click sensor           }
  function    GetCount:LongInt;
  function    GetItems(i:Integer):TSmartBitmap;
  function    GetGlyph:TSmartBitmap;
  function    GetBounds:TRect2I;
  function    GetMinBounds:TRect2I;
  function    GetMaxBounds:TRect2I;
  function    GetName:LongString;
  function    GetPos:TPoint2I;
  function    GetBookmark:LongInt;
  procedure   SetBookmark(aBookmark:LongInt);
  function    GetLedWidth:Integer;
  procedure   SetLedWidth(aWidth:Integer);
  function    GetLedDigit:Integer;
  procedure   SetLedDigit(aDigit:Integer);
  function    GetLedValue:LongString;
  procedure   SetLedValue(const aValue:LongString);
  function    GetLedDrawn:LongString;
  function    GetLedFormat:LongString;
  procedure   SetLedFormat(const aFormat:LongString);
  function    GetLedFormatType:Integer;
  function    GetLedFontLink:LongString;
  procedure   SetLedFontLink(const aFontLink:LongString);
  function    GetLedFontText:LongString;
  function    GetLedFontColor:TColor;
  function    GetLinkedTag:Integer;
  procedure   SetLinkedTag(aTag:Integer);
  function    GetLinkedTagColor:Integer;
  procedure   SetLinkedTagColor(aColor:Integer);
  function    GetLinkedTagParam:Double;
  procedure   SetLinkedTagParam(aParam:Double);
  function    GetLinkedTagTimer:Double;
  procedure   SetLinkedTagTimer(aTimer:Double);
  function    GetUsesLinkedTagColor:Boolean;
  procedure   SetUsesLinkedTagColor(aUses:Boolean);
  function    GetUsesLinkedTagParam:Boolean;
  procedure   SetUsesLinkedTagParam(aUses:Boolean);
  function    GetUsesLinkedTagTimer:Boolean;
  procedure   SetUsesLinkedTagTimer(aUses:Boolean);
  function    GetLinkedCurve:TCurve;
  procedure   SetLinkedCurve(aCurve:TCurve);
  function    GetLinkedValue:Double;
  procedure   SetLinkedValue(aValue:Double);
  function    GetLinkedObject:TObject;
  procedure   SetLinkedObject(aObject:TObject);
  function    GetTextColor:TColor;
  procedure   SetTextColor(aColor:TColor);
  function    GetElsemark:LongInt;
  procedure   SetElsemark(aElsemark:LongInt);
  function    GetShortCut:TShortCut;
  procedure   SetShortCut(aShortCut:TShortCut);
  function    GetHintText:LongString;
  procedure   SetHintText(const aHintText:LongString);
  function    GetHintLine:LongString;
  function    GetToolBarKey:Integer;
  procedure   SetToolBarKey(aToolBarKey:Integer);
  function    GetToolBarOpaque:Boolean;
  procedure   SetToolBarOpaque(aToolBarOpaque:Boolean);
  function    GetToolBarBorder:Boolean;
  procedure   SetToolBarBorder(aToolBarBorder:Boolean);
  function    GetToolBarSpace:Integer;
  procedure   SetToolBarSpace(aToolBarSpace:Integer);
  function    GetTagEval:LongString;
  procedure   SetTagEval(const aTagEval:LongString);
  function    GetLedEval:LongString;
  procedure   SetLedEval(const aLedEval:LongString);
  function    GetPainter:LongString;
  procedure   SetPainter(const aPainter:LongString);
  function    GetPainterCall:LongString;
  procedure   SetPainterCall(const aPainterCall:LongString);
  function    GetPainterIsSimple:Boolean;
  function    GetPendingDraw:Boolean;
  procedure   SetPendingDraw(aPendingDraw:Boolean);
  function    GetParentForm:TForm;
  procedure   SetParentForm(aParentForm:TForm);
  function    GetSourceCrcFile:LongString;
  function    GetSourceSection:LongString;
  function    GetGuardLevel:Integer;
  procedure   SetGuardLevel(aLevel:Integer);
  procedure   InitToolBarItems;
 public
  constructor Create(const aPos:TPoint2I; const aName:LongString);
  destructor  Destroy; override;
  function    Search(aIdent:LongInt; out aIndex:Integer):Boolean;
  procedure   Add(aGlyph:TSmartBitmap);
  procedure   Draw(Canvas:TCanvas; const Where:TPoint2I);
  procedure   Reload;
 public
  property    Count              : LongInt       read GetCount;
  property    Items[i:Integer]   : TSmartBitmap  read GetItems; default;
  property    Glyph              : TSmartBitmap  read GetGlyph;
  property    Bounds             : TRect2I       read GetBounds;
  property    MinBounds          : TRect2I       read GetMinBounds;
  property    MaxBounds          : TRect2I       read GetMaxBounds;
  property    Name               : LongString    read GetName;
  property    Pos                : TPoint2I      read GetPos;
  property    Bookmark           : LongInt       read GetBookmark           write SetBookmark;
  property    LedWidth           : Integer       read GetLedWidth           write SetLedWidth;
  property    LedDigit           : Integer       read GetLedDigit           write SetLedDigit;
  property    LedValue           : LongString    read GetLedValue           write SetLedValue;
  property    LedDrawn           : LongString    read GetLedDrawn;
  property    LedFormat          : LongString    read GetLedFormat          write SetLedFormat;
  property    LedFormatType      : Integer       read GetLedFormatType;
  property    LedFontLink        : LongString    read GetLedFontLink        write SetLedFontLink;
  property    LedFontText        : LongString    read GetLedFontText;
  property    LedFontColor       : TColor        read GetLedFontColor;
  property    TextColor          : TColor        read GetTextColor          write SetTextColor;
  property    Elsemark           : LongInt       read GetElsemark           write SetElsemark;
  property    ShortCut           : TShortCut     read GetShortCut           write SetShortCut;
  property    HintText           : LongString    read GetHintText           write SetHintText;
  property    HintLine           : LongString    read GetHintLine;
  property    LinkedTag          : Integer       read GetLinkedTag          write SetLinkedTag;
  property    LinkedTagColor     : Integer       read GetLinkedTagColor     write SetLinkedTagColor;
  property    LinkedTagParam     : Double        read GetLinkedTagParam     write SetLinkedTagParam;
  property    LinkedTagTimer     : Double        read GetLinkedTagTimer     write SetLinkedTagTimer;
  property    UsesLinkedTagColor : Boolean       read GetUsesLinkedTagColor write SetUsesLinkedTagColor;
  property    UsesLinkedTagParam : Boolean       read GetUsesLinkedTagParam write SetUsesLinkedTagParam;
  property    UsesLinkedTagTimer : Boolean       read GetUsesLinkedTagTimer write SetUsesLinkedTagTimer;
  property    LinkedCurve        : TCurve        read GetLinkedCurve        write SetLinkedCurve;
  property    LinkedValue        : Double        read GetLinkedValue        write SetLinkedValue;
  property    LinkedObject       : TObject       read GetLinkedObject       write SetLinkedObject;
  property    ToolBarKey         : Integer       read GetToolBarKey         write SetToolBarKey;
  property    ToolBarOpaque      : Boolean       read GetToolBarOpaque      write SetToolBarOpaque;
  property    ToolBarBorder      : Boolean       read GetToolBarBorder      write SetToolBarBorder;
  property    ToolBarSpace       : Integer       read GetToolBarSpace       write SetToolBarSpace;
  property    TagEval            : LongString    read GetTagEval            write SetTagEval;
  property    LedEval            : LongString    read GetLedEval            write SetLedEval;
  property    Painter            : LongString    read GetPainter            write SetPainter;
  property    PainterCall        : LongString    read GetPainterCall        write SetPainterCall;
  property    PainterIsSimple    : Boolean       read GetPainterIsSimple;
  property    PendingDraw        : Boolean       read GetPendingDraw        write SetPendingDraw;
  property    ParentForm         : TForm         read GetParentForm         write SetParentForm;
  property    SourceCrcFile      : LongString    read GetSourceCrcFile;
  property    SourceSection      : LongString    read GetSourceSection;
  property    GuardLevel         : Integer       read GetGuardLevel         write SetGuardLevel;
  function    HasTagEval         : Boolean;
  function    HasLedEval         : Boolean;
  function    HasPainter         : Boolean;
  function    HasLedWidth        : Boolean;
 end;

function NewCircuitSensorFromCrcFile(CrcFile,Section:LongString):TCircuitSensor;

const
  EventWhatNameList='NOTHING,MOUSEDOWN,MOUSEUP,MOUSEMOVE,KEYDOWN,KEYUP,MOUSEWHEEL,MOUSEWHEELDOWN,MOUSEWHEELUP';

type
  TEventWhat = (evNothing,evMouseDown,evMouseUp,evMouseMove,evKeyDown,evKeyUp,evMouseWheel,evMouseWheelDown,evMouseWheelUp);
  TCircuitEvent = record
   What        : TEventWhat;
   Key         : Word;
   Shift       : TShiftState;
   Where       : TPoint2I;
   Wheel       : Integer;
  end;
  TFormCircuitWindow = class;
  TCircuitStrategy   = procedure(Form:TFormCircuitWindow; var Event:TCircuitEvent; Sensor:TCircuitSensor);
  TCircuitMonitor    = procedure(Form:TFormCircuitWindow);

  { TFormCircuitWindow }

  TFormCircuitWindow = class(TFormCrwDaqSysChild)
    PaintBox: TPaintBox;
    ScrollBarX: TScrollBar;
    ScrollBarY: TScrollBar;
    ActionViewToolBarSensor: TAction;
    ActionViewScrollLeft: TAction;
    ActionViewScrollRight: TAction;
    ActionViewScrollUp: TAction;
    ActionViewScrollDown: TAction;
    ActionViewScrollBarShowX: TAction;
    ActionViewScrollBarShowY: TAction;
    ActionViewShowScrollDown: TAction;
    ActionViewShowScrollUp: TAction;
    ActionViewShowScrollRight: TAction;
    ActionViewShowScrollLeft: TAction;
    ActionViewToolBarHeight0: TAction;
    ActionViewToolBarHeight1: TAction;
    ActionViewToolBarHeight2: TAction;
    ActionViewToolBarHeight3: TAction;
    ActionViewToolBarHeight4: TAction;
    ActionViewPainterDebugTools: TAction;
    MenuViewScroll: TMenuItem;
    MenuViewScrollLeft: TMenuItem;
    MenuViewScrollRight: TMenuItem;
    MenuViewScrollUp: TMenuItem;
    MenuViewScrollDown: TMenuItem;
    MenuViewScrollBarShowX: TMenuItem;
    MenuViewScrollBarShowY: TMenuItem;
    MenuViewShowScrollLeft: TMenuItem;
    MenuViewShowScrollDown: TMenuItem;
    MenuViewShowScrollUp: TMenuItem;
    MenuViewShowScrollRight: TMenuItem;
    MenuViewToolBar: TMenuItem;
    MenuViewToolBarHeight0: TMenuItem;
    MenuViewToolBarHeight1: TMenuItem;
    MenuViewToolBarHeight2: TMenuItem;
    MenuViewToolBarHeight3: TMenuItem;
    MenuViewToolBarHeight4: TMenuItem;
    MenuViewPainterDebugTools: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure PaintBoxDblClick(Sender: TObject);
    procedure PaintBoxPaint(Sender: TObject);
    procedure ScrollBarXScroll(Sender: TObject; ScrollCode: TScrollCode; var ScrollPos: Integer);
    procedure ScrollBarYScroll(Sender: TObject; ScrollCode: TScrollCode; var ScrollPos: Integer);
    procedure PaintBoxMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure PaintBoxMouseMove(Sender: TObject; Shift: TShiftState; X,Y: Integer);
    procedure PaintBoxMouseUp(Sender: TObject; Button: TMouseButton;  Shift: TShiftState; X, Y: Integer);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure FormMouseWheel(Sender: TObject; Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure FormMouseWheelDown(Sender: TObject; Shift: TShiftState; MousePos: TPoint; var Handled: Boolean);
    procedure FormMouseWheelUp(Sender: TObject; Shift: TShiftState; MousePos: TPoint; var Handled: Boolean);
    procedure ScrollBarYKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure ScrollBarYKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure ScrollBarXKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure ScrollBarXKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure ActionViewScrollLeftExecute(Sender: TObject);
    procedure ActionViewScrollRightExecute(Sender: TObject);
    procedure ActionViewScrollUpExecute(Sender: TObject);
    procedure ActionViewScrollDownExecute(Sender: TObject);
    procedure ActionViewToolBarSensorExecute(Sender: TObject);
    procedure ActionViewToolBarHeight0Execute(Sender: TObject);
    procedure ActionViewToolBarHeight1Execute(Sender: TObject);
    procedure ActionViewToolBarHeight2Execute(Sender: TObject);
    procedure ActionViewToolBarHeight3Execute(Sender: TObject);
    procedure ActionViewToolBarHeight4Execute(Sender: TObject);
    procedure ActionViewScrollBarShowXExecute(Sender: TObject);
    procedure ActionViewScrollBarShowYExecute(Sender: TObject);
    procedure ActionViewShowScrollDownExecute(Sender: TObject);
    procedure ActionViewShowScrollLeftExecute(Sender: TObject);
    procedure ActionViewShowScrollRightExecute(Sender: TObject);
    procedure ActionViewShowScrollUpExecute(Sender: TObject);
    procedure ActionViewPainterDebugToolsExecute(Sender: TObject);
  private
    { Private declarations }
    myMainBitmap      : TSmartBitmap;
    myItems           : TObjectStorage;
    mySourceCrcFile   : LongString;
    mySourceCfgFile   : LongString;
    mySourceSection   : LongString;
    mySourceName      : LongString;
    myStrategy        : TCircuitStrategy;
    myMonitor         : TCircuitMonitor;
    myHintText        : LongString;
    myHintLine        : LongString;
    myDefToolBarHt    : Integer;
    myDefToolBtnWd    : Integer;
    myDefToolBtnHt    : Integer;
    myEvaluator       : TExpressionEvaluator;
    myStartupScript   : LongString;
    mySimulation      : Boolean;
    myDebugFlags      : Integer;
    myDebugToolsMenu  : Integer;
    myToolbarCommands : TStringList;
    myDecorSize       : TPoint2I;
    myStartSize       : TPoint2I;
    function  GetMainBitmap:TSmartBitmap;
    function  GetCount:Integer;
    function  GetItems(i:Integer):TCircuitSensor;
    function  GetSourceCrcFile:LongString;
    function  GetSourceCfgFile:LongString;
    function  GetSourceSection:LongString;
    function  GetSourceName:LongString;
    function  GetHasStrategy:Boolean;
    procedure SetStrategy(aStrategy:TCircuitStrategy);
    function  GetHasMonitor:Boolean;
    procedure SetMonitor(aMonitor:TCircuitMonitor);
    function  GetHintText:LongString;
    procedure SetHintText(const aHintText:LongString);
    function  GetHintLine:LongString;
    function  GetToolBarHeight:Integer;
    procedure SetToolBarHeight(aToolBarHeight:Integer);
    function  GetScrollBarShowX:Boolean;
    procedure SetScrollBarShowX(aShow:Boolean);
    function  GetScrollBarShowY:Boolean;
    procedure SetScrollBarShowY(aShow:Boolean);
    function  GetEvaluator:TExpressionEvaluator;
    function  GetStartupScript:LongString;
    procedure SetStartupScript(const aStartupScript:LongString);
    function  GetPaintable:Boolean;
    function  GetSimulation:Boolean;
    procedure SetSimulation(aSimulation:Boolean);
    function  GetDebugFlags:Integer;
    procedure SetDebugFlags(aDebugFlags:Integer);
    function  HtZoom(h:Integer):Integer;
  protected
    property  Paintable:Boolean read GetPaintable;
    property  Simulation:Boolean read GetSimulation write SetSimulation;
    property  DebugFlags:Integer read GetDebugFlags write SetDebugFlags;
  public
    { Public declarations }
    property  MainBitmap       : TSmartBitmap     read  GetMainBitmap;
    property  Count            : Integer          read  GetCount;
    property  Items[i:Integer] : TCircuitSensor   read  GetItems; default;
    property  SourceCrcFile    : LongString       read  GetSourceCrcFile;
    property  SourceCfgFile    : LongString       read  GetSourceCfgFile;
    property  SourceSection    : LongString       read  GetSourceSection;
    property  SourceName       : LongString       read  GetSourceName;
    property  HasStrategy      : Boolean          read  GetHasStrategy;
    property  Strategy         : TCircuitStrategy write SetStrategy;
    property  HasMonitor       : Boolean          read  GetHasMonitor;
    property  Monitor          : TCircuitMonitor  write SetMonitor;
    property  HintText         : LongString       read GetHintText write SetHintText;
    property  HintLine         : LongString       read GetHintLine;
    property  ToolBarHeight    : Integer          read GetToolBarHeight write SetToolBarHeight;
    property  ScrollBarShowX   : Boolean          read GetScrollBarShowX write SetScrollBarShowX;
    property  ScrollBarShowY   : Boolean          read GetScrollBarShowY write SetScrollBarShowY;
    property  Evaluator        : TExpressionEvaluator read GetEvaluator;
    property  StartupScript    : LongString       read GetStartupScript  write SetStartupScript;
    function  ImageToPaintBoxClient(const Pos:TPoint2I):TPoint2I;
    function  PaintBoxClientToImage(const Pos:TPoint2I):TPoint2I;
    function  Search(const aName:LongString; out aIndex:Integer):Boolean;
    function  SensorByName(const aName:LongString):TCircuitSensor;
    function  SensorByPos(const Pos:TPoint2I):TCircuitSensor;
    function  SensorByShortCut(ShortCut:TShortCut):TCircuitSensor;
    procedure AddSensor(Sensor:TCircuitSensor);
    procedure UpdateScrollBars;
    procedure UpdateStatus(const Mouse:TPoint2I);
    procedure DrawSensor(Sensor:TCircuitSensor);
    procedure AfterConstruction; override;
    procedure BeforeDestruction; override;
    procedure UpdateCommands; override;
    procedure FileSaveAs; override;
    procedure DrawView; override;
    procedure PrintView; override;
    procedure StartMonitoring;
    procedure StopMonitoring;
    procedure Monitoring;
    procedure UpdateSensor(Sensor:TCircuitSensor; NewValue:Double; NewStrValue:PChar=nil);
    function  SaveToCrcFile(FileName:LongString; RelativePath:Boolean=true):Boolean;
    function  AddToolBarSensorButton(aSensor:TCircuitSensor):Boolean;
    function  AddToolBarActionButton(aAction:TAction):TPanel;
    function  AddToolBarTextPanel(aName,aText,aFont:LongString; aWidth:Integer):TPanel;
    procedure Reload(const Sensor:LongString='All');
    procedure ApplyWinDrawOptions(aOptions:LongString);
    function  ChangePanelState(aName:LongString; how:Char):Boolean;
    procedure UpdateScrollPanels;
    procedure HandlePanelToolbarCommand(aCommand:LongString);
    function  FindWantedSize(const opt:LongString='Mode=7;Px=50;Py=50;Mx=8;My=8;'):TPoint2I;
    procedure ApplyWantedSize;
  protected
    PanelViewScrollLeft:TPanel;
    PanelViewScrollRight:TPanel;
    PanelViewScrollUp:TPanel;
    PanelViewScrollDown:TPanel;
    PanelToolbarHeader:TPanel;
    PanelToolbarLegend:TPanel;
  end;
  TCircuitWindowList = class(TObjectStorage)
  private
    function   GetWindow(i:Integer):TFormCircuitWindow;
    procedure  SetWindow(i:Integer; aWindow:TFormCircuitWindow);
  public
    property   Window[i:Integer]:TFormCircuitWindow read GetWindow write SetWindow; default;
  public
    function   SensorsCount:Integer;
  end;

const
 FullCircuitWindowList : TCircuitWindowList = nil;
 CircuitWindowsMonitor : TCircuitWindowList = nil;
 SafeToolBarSensors    : Boolean            = True;
 DefaultCircuitFont : TFontParams = (
  CharSet : RUSSIAN_CHARSET;
  Color   : clBlack;
  Height  : -13;
  Name    : 'PT Mono';
  Pitch   : fpFixed;
  Style   : [];
 );

function  NewCircuitWindow(MainBitmapFile,CircuitName:LongString):TFormCircuitWindow;
procedure Kill(var TheObject:TFormCircuitWindow); overload;
function  ActiveCircuitWindow:TFormCircuitWindow;

function  NewCircuitWindowFromCrcFile(CrcFile:LongString):TFormCircuitWindow;
function  NewCircuitWindowFromCfgFile(ConfigFile,Section,Name:LongString):TFormCircuitWindow;

function  NewCircuitWindowList(aOwnsObjects : Boolean = true;
                               aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                               aStep        : LongInt = DefaultTObjectStorageStep
                                          ) : TCircuitWindowList;
procedure Kill(var TheObject:TCircuitWindowList); overload;

const
 TCircuitSensorTagEvalErrorCount : Cardinal = 0;
 TCircuitSensorLedEvalErrorCount : Cardinal = 0;
 TCircuitSensorPainterErrorCount : Cardinal = 0;

const
 df_StartupBugEcho = $00000001; // Echo on Startup bug
 df_TagEvalBugEcho = $00000002; // Echo on TagEval bug
 df_LedEvalBugEcho = $00000004; // Echo on LedEval bug
 df_PainterBugEcho = $00000008; // Echo on Painter bug
 df_SummaryBugEcho = df_StartupBugEcho or df_TagEvalBugEcho or df_LedEvalBugEcho or df_PainterBugEcho;
 df_StartupBugFile = $00000010; // File on Startup bug
 df_TagEvalBugFile = $00000020; // File on TagEval bug
 df_LedEvalBugFile = $00000040; // File on LedEval bug
 df_PainterBugFile = $00000080; // File on Painter bug
 df_SummaryBugFile = df_StartupBugFile or df_TagEvalBugFile or df_LedEvalBugFile or df_PainterBugFile;

function SmartCheckBmpFileExists(FileName:LongString):Boolean;

 {
 **********************
 CircuitWindowsProfiler
 **********************
 }
procedure CircuitWindowsProfiler_Clear;
procedure CircuitWindowsProfiler_Start;
procedure CircuitWindowsProfiler_Stop;
procedure CircuitWindowsProfiler_Poll;

var
 CircuitWindowsProfiler : record
  Curr, Prev, Rate    : record
   TickCount          : Int64;
   DrawView           : Int64;
   MakeBitmap         : Int64;
   DrawSensor         : Int64;
   PollSensor         : Int64;
   TagEvalCall        : Int64;
   LedEvalCall        : Int64;
   PainterCall        : Int64;
   PainterApiCall     : Int64;
   PainterApiDraw     : Int64;
   MonitorCall        : Int64;
   MonitorDraw        : Int64;
  end;
  RateFactor          : Double;
 end;

implementation

{$R *.lfm}

 {
 **********************
 CircuitWindowsProfiler
 **********************
 }
procedure CircuitWindowsProfiler_Clear;
begin
 FillChar(CircuitWindowsProfiler, SizeOf(CircuitWindowsProfiler), 0);
end;

procedure CircuitWindowsProfiler_Start;
begin
 CircuitWindowsProfiler_Clear;
 SecondActions.Add(CircuitWindowsProfiler_Poll);
end;

procedure CircuitWindowsProfiler_Stop;
begin
 SecondActions.Remove(CircuitWindowsProfiler_Poll);
 CircuitWindowsProfiler_Clear;
end;

procedure CircuitWindowsProfiler_Poll;
begin
 with CircuitWindowsProfiler do begin
  Curr.TickCount     := GetTickCount64;
  Rate.TickCount     := Curr.TickCount-Prev.TickCount;
  RateFactor:=IfThen(Rate.TickCount=0,0.0,1000.0/Rate.TickCount);
  Rate.DrawView       := Curr.DrawView       - Prev.DrawView;
  Rate.MakeBitmap     := Curr.MakeBitmap     - Prev.MakeBitmap;
  Rate.DrawSensor     := Curr.DrawSensor     - Prev.DrawSensor;
  Rate.PollSensor     := Curr.PollSensor     - Prev.PollSensor;
  Rate.TagEvalCall    := Curr.TagEvalCall    - Prev.TagEvalCall;
  Rate.LedEvalCall    := Curr.LedEvalCall    - Prev.LedEvalCall;
  Rate.PainterCall    := Curr.PainterCall    - Prev.PainterCall;
  Rate.PainterApiCall := Curr.PainterApiCall - Prev.PainterApiCall;
  Rate.PainterApiDraw := Curr.PainterApiDraw - Prev.PainterApiDraw;
  Rate.MonitorCall    := Curr.MonitorCall    - Prev.MonitorCall;
  Rate.MonitorDraw    := Curr.MonitorDraw    - Prev.MonitorDraw;
  Prev:=Curr;
 end;
end;

 {
 *********************************
 TCircuitWindowList implementation
 *********************************
 }
function   TCircuitWindowList.GetWindow(i:Integer):TFormCircuitWindow;
begin
 Result:=TFormCircuitWindow(Items[i]);
end;

procedure  TCircuitWindowList.SetWindow(i:Integer; aWindow:TFormCircuitWindow);
begin
 Items[i]:=aWindow;
end;

function   TCircuitWindowList.SensorsCount:Integer;
var i:Integer;
begin
 Result:=0;
 if Assigned(Self) then
 try
  for i:=Count-1 downto 0 do Inc(Result,Self[i].Count);
 except
  on E:Exception do BugReport(E,Self,'SensorsCount');
 end;
end;

function  NewCircuitWindowList(aOwnsObjects : Boolean = true;
                               aCapacity    : LongInt = DefaultTObjectStorageCapacity;
                               aStep        : LongInt = DefaultTObjectStorageStep
                                          ) : TCircuitWindowList;
begin
 Result:=TCircuitWindowList.Create(aOwnsObjects,aCapacity,aStep);
end;

procedure Kill(var TheObject:TCircuitWindowList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end; 
end;

 {
 *******************************************************************************
 General purpose routines implementation
 *******************************************************************************
 }
procedure DefaultCircuitMonitor(Form:TFormCircuitWindow);
const LastTick:QWord=0; Delay=100;
var i:Integer;
begin
 if (GetTickCount64>=LastTick+Delay) then begin
  for i:=0 to Form.Count-1 do
  if (Form[i].Count>1)
  then Form.UpdateSensor(Form[i],Random(Form[i].Count))
  else if Form[i].HasLedWidth then Form.UpdateSensor(Form[i],Random);
  LastTick:=GetTickCount64;
 end;
end;

procedure DefaultCircuitStrategy(Form:TFormCircuitWindow; var Event:TCircuitEvent; Sensor:TCircuitSensor);
begin
 if Form is TFormCircuitWindow then with Event do
 case What of
  evMouseDown:
   if Sensor is TCircuitSensor
   then Echo(RusEng('Нажат сенсор ','Pressed sensor ')+Sensor.Name)
   else Echo(RusEng('Нажата мышь','Mouse down')+' X='+d2s(Where.X,4)+', Y='+d2s(Where.Y,4)+' Btn='+d2s(Key));
  evKeyDown:
   begin
    Echo(RusEng('Нажата клавиша ','Pressed key ')+d2s(ShortCut(Key,Shift))+' ($'+HexW(ShortCut(Key,Shift))+')');
    Echo(RusEng('Мышь','Mouse')+' X='+d2s(Where.X,4)+', Y='+d2s(Where.Y,4));
    if Key=VK_RETURN then begin
     Form.Monitor:=DefaultCircuitMonitor;
     Form.StartMonitoring;
    end;
    if Key=VK_ESCAPE then begin
     Form.Monitor:=nil;
     Form.StopMonitoring;
    end;
   end;
 end;
end;

function NewCircuitWindow(MainBitmapFile,CircuitName:LongString):TFormCircuitWindow;
begin
 Result:=nil;
 MainBitmapFile:=UnifyFileAlias(MainBitmapFile,ua_FileLow);
 if SmartCheckBmpFileExists(MainBitmapFile) and IsNonEmptyStr(CircuitName) then begin
  Application.CreateForm(TFormCircuitWindow,Result);
  if Result.Ok then begin
   Result.myMainBitmap:=NewSmartBitmap(MainBitmapFile);
   if Result.myMainBitmap.Ok then begin
    Result.myMainBitmap.Master:=@Result.myMainBitmap;
    if IsEmptyStr(CircuitName)
    then Result.Caption:=ExtractFileName(MainBitmapFile)
    else Result.Caption:=CircuitName;
    Result.UpdateScrollBars;
    Result.Strategy:=DefaultCircuitStrategy;
   end else Kill(Result);
  end else Kill(Result);
 end;
end;

procedure Kill(var TheObject:TFormCircuitWindow); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E,nil,'Kill');
 end;
end;

function ActiveCircuitWindow:TFormCircuitWindow;
var Child:TForm;
begin
 Result:=nil;
 try
  if SdiMan.FindActiveChild(Child,sf_SdiChild,sf_SdiControl) then
  if (Child is TFormCircuitWindow) then Result:=TFormCircuitWindow(Child);
 except
  on E:Exception do BugReport(E,nil,'ActiveCircuitWindow');
 end;
end;

procedure PrintPainterError(DebugFlags:Integer; const Data:LongString);
begin
 if Data<>'' then
 if DebugFlags<>0 then
 try
  if (DebugFlags and df_SummaryBugEcho <> 0) then Echo(Data);
  if (DebugFlags and df_SummaryBugFile <> 0) then DebugOut(stdfDebug,Data);
 except
  on E:Exception do BugReport(E,nil,'PrintPainterError');
 end;
end;

function FormatPainterError(Form:TForm; Sensor:TCircuitSensor; ee:TExpressionEvaluator; ms:Double; const Hint:LongString):LongString;
begin
 Result:='';
 try
  if ms>0 then Result:=StdDateTimePrompt;
  Result:=Result+'Error ';
  if Form is TForm then Result:=Result+'in Form «'+Form.Caption+'» ';
  if Sensor is TCircuitSensor then Result:=Result+'in Sensor «'+Sensor.Name+'» ';
  if Hint<>'' then Result:=Result+'hint «'+Hint+'» ';
  if Result<>'' then Result:=Result+EOL;
  if ee is TExpressionEvaluator then
  Result:=Result+
          ee_ErrorMessage(ee.Status)+'.'+EOL+
          StrPas(ee.Buffer)+EOL+
          CharStr(ee.ErrorPos+1)+'^'+EOL+
          CharStr(ee.ErrorPos+1-Length(ee.ErrorToken))+StrPas(ee.ErrorToken)+EOL;
 except
  on E:Exception do BugReport(E,nil,'FormatPainterError');
 end;
end;

function PanelToolbarXCommand(Sender:TObject):LongString;
begin
 if (Sender is TPanel)
 then Result:=(Sender as TPanel).Name+'.Command'
 else Result:='';
end;

procedure ReadToolbarHeader(Win:TFormCircuitWindow; CrcFile:LongString);
var FP:TFontParams; s,sWidth,sTitle,sFont,sCmnd,sHint,tbc:LongString; i:Integer;
 procedure ParseLine(s:LongString; out sWidth,sTitle,sFont,sCmnd,sHint:LongString);
 begin
  sWidth := ExtractPhrase(1,s,ScanSpaces);
  sTitle := ExtractPhrase(2,s,ScanSpaces);
  sFont  := ExtractPhrase(3,s,ScanSpaces);
  sCmnd  := ExtractPhrase(4,s,ScanSpaces);
  sHint  := ExtractPhrase(5,s,ScanSpaces);
 end;
begin
 s:=''; sWidth:=''; sTitle:=''; sFont:=''; sCmnd:=''; sHint:=''; tbc:='';
 if Assigned(Win) then begin
  if Assigned(Win.PanelToolbarHeader) then
  if ReadIniFileString(CrcFile,'[Circuit]','ToolBarHeader%s',s,efConfigNC) then begin
   ParseLine(s,sWidth,sTitle,sFont,sCmnd,sHint);
   i:=StrToIntDef(sWidth,-1);
   if (i>=0) then Win.PanelToolbarHeader.Width:=Max(4,i);
   Win.PanelToolbarHeader.Caption:=sTitle;
   if ReadBufferedFont(FP,sFont,True,StandardFont)
   then RestoreFont(Win.PanelToolbarHeader.Font,FP);
   tbc:=PanelToolbarXCommand(Win.PanelToolbarHeader);
   if IsNonEmptyStr(sCmnd) then Win.myToolbarCommands.Values[tbc]:=Trim(sCmnd);
   if IsNonEmptyStr(sHint) then Win.PanelToolbarHeader.Hint:=Trim(sHint);
  end else Win.PanelToolbarHeader.Caption:='';
  if ReadIniFileString(CrcFile,'[Circuit]','ToolBarLegend%s',s,efConfigNC) then begin
   ParseLine(s,sWidth,sTitle,sFont,sCmnd,sHint);
   i:=StrToIntDef(sWidth,-1);
   if (i>=0) and (PhraseCount(s,ScanSpaces)>1) then begin
    Win.PanelToolbarLegend:=Win.AddToolBarTextPanel('PanelToolbarLegend',sTitle,sFont,i);
    if Assigned(Win.PanelToolbarLegend) then begin
     tbc:=PanelToolbarXCommand(Win.PanelToolbarLegend);
     if IsNonEmptyStr(sCmnd) then Win.myToolbarCommands.Values[tbc]:=Trim(sCmnd);
     if IsNonEmptyStr(sHint) then Win.PanelToolbarLegend.Hint:=Trim(sHint);
    end;
   end;
  end;
 end;
end;

function NewCircuitWindowFromCrcFile(CrcFile:LongString):TFormCircuitWindow;
var i:Integer; s,Hint,Path:LongString; Sensor:TCircuitSensor; SensorList:TText;
begin
 Result:=nil;
 CrcFile:=UnifyFileAlias(CrcFile,ua_FileLow); Path:=''; Hint:='';
 if ReadIniFilePath(CrcFile,'[Circuit]','GeneralMap',ExtractFilePath(CrcFile),Path) then begin
  Path:=Dequote_or_URL_Decode(Path);
  s:=ExtractFileName(Path);
  ReadIniFileAlpha(CrcFile,'[Circuit]','Name%a',s);
  Result:=NewCircuitWindow(Path,s);
  if Result.Ok then begin
   if ReadIniFileString(CrcFile,'[Circuit]','Hint%s',Hint,efConfigNC)
   then Result.HintText:=Dequote_or_URL_Decode(Hint);
   Result.mySourceCrcFile:=CrcFile;
   Result.StartupScript:=ExtractTextSectionByPrefix(CrcFile,'[Circuit]','StartupScript',efConfigNC,svConfig,true);
   if Length(Result.StartupScript)>0 then begin
    Result.Evaluator.Script:=Result.StartupScript;
    if Result.Evaluator.RunScript<>ee_OK then begin
     Inc(TCircuitSensorPainterErrorCount);
     if (Result.DebugFlags and (df_StartupBugEcho or df_StartupBugFile) <> 0)
     then PrintPainterError(Result.DebugFlags,FormatPainterError(Result,nil,Result.Evaluator,mSecNow,'fail Startup Evaluate'));
    end;
    Result.Evaluator.Script:='';
   end;
   Result.Update;
   try
    Result.LockDraw;
    {чтение списка сенсоров}
    SensorList:=ExtractEnumWordList(CrcFile,'[SensorList]','Sensor',efConfig);
    try
     if Assigned(SensorList) and (SensorList.Count>0) then begin
      for i:=0 to SensorList.Count-1 do begin
       Sensor:=NewCircuitSensorFromCrcFile(CrcFile,'['+SensorList[i]+']');
       if Sensor.Ok
       then Result.AddSensor(Sensor)
       else begin
        DebugOut(stdfDebug,'Could not read sensor '+SensorList[i]+', file '+CrcFile);
        continue;
       end;
      end;
     end else begin
      for i:=1 to MaxInt do begin
       Sensor:=NewCircuitSensorFromCrcFile(CrcFile,'[Sensor#'+d2s(i)+']');
       if Sensor.Ok
       then Result.AddSensor(Sensor)
       else break;
      end;
     end;
    finally
     Kill(SensorList);
    end;
   finally
    Result.UnlockDraw;
   end;
   if ReadIniFileInteger(CrcFile,'[Circuit]','ToolBarHeight%i',i)
   then Result.ToolBarHeight:=i;
   SendToMainConsole('@silent @integrity load.crc '+Result.SourceCrcFile+EOL);
   ReadToolbarHeader(Result,CrcFile);
   Result.UpdateScrollPanels;
   Result.ApplyWantedSize;
  end;
 end;
end;

function NewCircuitWindowFromCfgFile(ConfigFile,Section,Name:LongString):TFormCircuitWindow;
var CrcFile,Caption,Hint:LongString; i:Integer;
begin
 Result:=nil;
 ConfigFile:=UnifyFileAlias(ConfigFile,ua_FileLow); CrcFile:=''; Caption:=''; Hint:=''; i:=0;
 if ReadIniFilePath(ConfigFile,Section,Name,ExtractFilePath(ConfigFile),CrcFile) then begin
  Result:=NewCircuitWindowFromCrcFile(CrcFile);
  if Result.Ok then begin
   Result.mySourceCfgFile:=ConfigFile;
   Result.mySourceSection:=Section;
   Result.mySourceName:=Name;
   Result.Caption:=RemoveBrackets(Section);
   if ReadIniFileAlpha(CrcFile,'[Circuit]','Name%a',Caption) then Result.Caption:=Caption;
   if ReadIniFileAlpha(ConfigFile,Section,'Name%a',Caption) then Result.Caption:=Caption;
   if ReadIniFileString(ConfigFile,Section,'Hint%s',Hint,efConfig and not efCaseMask)
   then Result.HintText:=Dequote_or_URL_Decode(Hint);
   if ReadIniFileInteger(ConfigFile,Section,'ToolBarHeight%i',i) then Result.ToolBarHeight:=i;
  end else Kill(Result);
 end;
end;

function DetectLedFormatType(const S:LongString):Integer;
var i,t:Integer; c,p:Char;
begin
 t:=0; p:=#0;
 for i:=1 to Length(S) do begin
  c:=System.UpCase(S[i]);
  if (p='%') then begin
   case c of
    '%'                      : p:=#0;                  // %% found, skip it
    '0'..'9','.','-',':','*' : ;                       // format parameters
    'D','U','X'              : begin t:=1; break; end; // Integer
    'E','F','G','N','M'      : begin t:=2; break; end; // Real
    'S'                      : begin t:=3; break; end; // String
    else                       break;                  // Invalid format
   end;
  end else begin
   if (c='%') then p:=c;
  end;
 end;
 Result:=t;
end;

 {
 *******************************************************************************
 TCircuitSensor implementation
 *******************************************************************************
 }
function TCircuitSensor.GetCount:LongInt;
begin
 if Assigned(Self) then Result:=myItems.Count else Result:=0;
end;

function TCircuitSensor.GetItems(i:Integer):TSmartBitmap;
begin
 if Assigned(Self) then Result:=TSmartBitmap(myItems[i]) else Result:=nil;
end;

function TCircuitSensor.GetGlyph:TSmartBitmap;
var aIndex:Integer;
begin
 if Search(BookMark,aIndex) then Result:=Items[aIndex] else
 if Search(ElseMark,aIndex) then Result:=Items[aIndex] else Result:=nil;
end;

function TCircuitSensor.GetBounds:TRect2I;
begin
 with Pos do Result:=Glyph.GetRect(X,Y);
end;

function TCircuitSensor.GetMinBounds:TRect2I;
var i:Integer;
begin
 with Pos do begin
  Result:=Glyph.GetRect(X,Y);
  if Count>0 then for i:=0 to Count-1 do
  Result:=RectIntersection(Result,Items[i].GetRect(X,Y));
 end;
end;

function TCircuitSensor.GetMaxBounds:TRect2I;
var i:Integer;
begin
 with Pos do begin
  Result:=Glyph.GetRect(X,Y);
  if Count>0 then for i:=0 to Count-1 do
  Result:=RectUnion(Result,Items[i].GetRect(X,Y));
 end;
end;

function TCircuitSensor.GetName:LongString;
begin
 if Assigned(Self) then Result:=myName else Result:='';
end;

function TCircuitSensor.GetPos:TPoint2I;
begin
 if Assigned(Self) then Result:=myPos else Result:=Point2I(0,0);
end;

function TCircuitSensor.GetBookmark:LongInt;
begin
 if Assigned(Self) then Result:=myBookmark else Result:=0;
end;

procedure TCircuitSensor.SetBookmark(aBookmark:LongInt);
begin
 if Assigned(Self) then myBookmark:=aBookmark;
end;

function TCircuitSensor.GetLedWidth:Integer;
begin
 if Assigned(Self) then Result:=myLedWidth else Result:=0;
end;

procedure TCircuitSensor.SetLedWidth(aWidth:Integer);
begin
 if Assigned(Self) then myLedWidth:=aWidth;
end;

function TCircuitSensor.GetLedDigit:Integer;
begin
 if Assigned(Self) then Result:=myLedDigit else Result:=0;
end;

procedure TCircuitSensor.SetLedDigit(aDigit:Integer);
begin
 if Assigned(Self) then myLedDigit:=aDigit;
end;

function TCircuitSensor.GetLedValue:LongString;
begin
 if Assigned(Self) then Result:=myLedValue else Result:='';
end;

procedure TCircuitSensor.SetLedValue(const aValue:LongString);
begin
 if Assigned(Self) then myLedValue:=aValue;
end;

function TCircuitSensor.GetLedDrawn:LongString;
begin
 if Assigned(Self) then Result:=myLedDrawn else Result:='';
end;

function TCircuitSensor.GetLedFormat:LongString;
begin
 if Assigned(Self) then Result:=myLedFormat else Result:='';
end;

procedure TCircuitSensor.SetLedFormat(const aFormat:LongString);
begin
 if Assigned(Self) then begin myLedFormat:=aFormat; myLedFormatType:=DetectLedFormatType(aFormat); end;
end;

function TCircuitSensor.GetLedFormatType:Integer;
begin
 if Assigned(Self) then Result:=myLedFormatType else Result:=0;
end;

function TCircuitSensor.GetLedFontLink:LongString;
begin
 if Assigned(Self) then Result:=myLedFontLink else Result:='';
end;

procedure TCircuitSensor.SetLedFontLink(const aFontLink:LongString);
begin
 if Assigned(Self) then myLedFontLink:=aFontLink;
end;

function TCircuitSensor.GetLedFontText:LongString;
begin
 if Assigned(Self) then Result:=GetFontAsText(myLedFontPara) else Result:='';
end;

function TCircuitSensor.GetLedFontColor:TColor;
begin
 if Assigned(Self) then Result:=myLedFontPara.Color else Result:=clBlack;
end;

function TCircuitSensor.GetLinkedTag:Integer;
begin
 if Assigned(Self) then Result:=myLinkedTag else Result:=0;
end;

procedure TCircuitSensor.SetLinkedTag(aTag:Integer);
begin
 if Assigned(Self) then myLinkedTag:=aTag;
end;

function TCircuitSensor.GetLinkedTagColor:Integer;
begin
 if Assigned(Self) then Result:=myLinkedTagColor else Result:=0;
end;

procedure TCircuitSensor.SetLinkedTagColor(aColor:Integer);
begin
 if Assigned(Self) then myLinkedTagColor:=aColor;
end;

function TCircuitSensor.GetLinkedTagParam:Double;
begin
 if Assigned(Self) then Result:=myLinkedTagParam else Result:=0;
end;

procedure TCircuitSensor.SetLinkedTagParam(aParam:Double);
begin
 if Assigned(Self) then myLinkedTagParam:=aParam;
end;

function TCircuitSensor.GetLinkedTagTimer:Double;
begin
 if Assigned(Self) then Result:=myLinkedTagTimer else Result:=0;
end;

procedure TCircuitSensor.SetLinkedTagTimer(aTimer:Double);
begin
 if Assigned(Self) then myLinkedTagTimer:=aTimer;
end;

function TCircuitSensor.GetUsesLinkedTagColor:Boolean;
begin
 if Assigned(Self) then Result:=myUsesLinkedTagColor else Result:=false;
end;

procedure TCircuitSensor.SetUsesLinkedTagColor(aUses:Boolean);
begin
 if Assigned(Self) then myUsesLinkedTagColor:=aUses;
end;

function TCircuitSensor.GetUsesLinkedTagParam:Boolean;
begin
 if Assigned(Self) then Result:=myUsesLinkedTagParam else Result:=false;
end;

procedure TCircuitSensor.SetUsesLinkedTagParam(aUses:Boolean);
begin
 if Assigned(Self) then myUsesLinkedTagParam:=aUses;
end;

function TCircuitSensor.GetUsesLinkedTagTimer:Boolean;
begin
 if Assigned(Self) then Result:=myUsesLinkedTagTimer else Result:=false;
end;

procedure TCircuitSensor.SetUsesLinkedTagTimer(aUses:Boolean);
begin
 if Assigned(Self) then myUsesLinkedTagTimer:=aUses;
end;

function TCircuitSensor.GetLinkedCurve:TCurve;
begin
 if Assigned(Self) then Result:=myLinkedCurve else Result:=nil;
end;

procedure TCircuitSensor.SetLinkedCurve(aCurve:TCurve);
begin
 if Assigned(Self) then myLinkedCurve:=aCurve;
end;

function TCircuitSensor.GetLinkedValue:Double;
begin
 if Assigned(Self) then Result:=myLinkedValue else Result:=0;
end;

procedure TCircuitSensor.SetLinkedValue(aValue:Double);
begin
 if Assigned(Self) then myLinkedValue:=aValue;
end;

function TCircuitSensor.GetLinkedObject:TObject;
begin
 if Assigned(Self) then Result:=myLinkedObject else Result:=nil;
end;

procedure TCircuitSensor.SetLinkedObject(aObject:TObject);
begin
 if Assigned(Self) then myLinkedObject:=aObject;
end;

function TCircuitSensor.GetTextColor:TColor;
begin
 if Assigned(Self) then Result:=myTextColor else Result:=0;
end;

procedure TCircuitSensor.SetTextColor(aColor:TColor);
begin
 if Assigned(Self) then myTextColor:=aColor;
end;

function TCircuitSensor.GetElsemark:LongInt;
begin
 if Assigned(Self) then Result:=myElsemark else Result:=0;
end;

procedure TCircuitSensor.SetElsemark(aElsemark:LongInt);
begin
 if Assigned(Self) then myElsemark:=aElsemark;
end;

function TCircuitSensor.GetShortCut:TShortCut;
begin
 if Assigned(Self) then Result:=myShortCut else Result:=0;
end;

procedure TCircuitSensor.SetShortCut(aShortCut:TShortCut);
begin
 if Assigned(Self) then myShortCut:=aShortCut;
end;

function TCircuitSensor.GetHintText:LongString;
begin
 if Assigned(Self) then Result:=myHintText else Result:='';
end;

procedure TCircuitSensor.SetHintText(const aHintText:LongString);
begin
 if Assigned(Self) then begin
  myHintText:=aHintText;
  myHintLine:=aHintText;
  if PosEol(myHintLine)>0
  then myHintLine:=Copy(myHintLine,1,PosEol(myHintLine)-1);
 end;
end;

function TCircuitSensor.GetHintLine:LongString;
begin
 if Assigned(Self) then Result:=myHintLine else Result:='';
end;

function TCircuitSensor.GetToolBarKey:Integer;
begin
 if Assigned(Self) then Result:=myToolBarKey else Result:=0;
end;

procedure TCircuitSensor.SetToolBarKey(aToolBarKey:Integer);
begin
 if Assigned(Self) then
 try
  myToolBarKey:=aToolBarKey;
  if myToolBarKey<>0 then if myToolBarItems=nil then InitToolBarItems;
 except
  on E:Exception do BugReport(E,Self,'SetToolBarKey');
 end;
end;

procedure TCircuitSensor.InitToolBarItems;
begin
 if Assigned(Self) then
 if not Assigned(myToolBarItems) then
 myToolBarItems:=NewObjectStorage(false,1,8);
end;

function TCircuitSensor.GetToolBarOpaque:Boolean;
begin
 if Assigned(Self) then Result:=myToolBarOpaque else Result:=false;
end;

procedure TCircuitSensor.SetToolBarOpaque(aToolBarOpaque:Boolean);
begin
 if Assigned(Self) then myToolBarOpaque:=aToolBarOpaque;
end;

function TCircuitSensor.GetToolBarBorder:Boolean;
begin
 if Assigned(Self) then Result:=myToolBarBorder else Result:=false;
end;

procedure TCircuitSensor.SetToolBarBorder(aToolBarBorder:Boolean);
begin
 if Assigned(Self) then myToolBarBorder:=aToolBarBorder;
end;

function TCircuitSensor.GetToolBarSpace:Integer;
begin
 if Assigned(Self) then Result:=myToolBarSpace else Result:=0;
end;

procedure TCircuitSensor.SetToolBarSpace(aToolBarSpace:Integer);
begin
 if Assigned(Self) then myToolBarSpace:=aToolBarSpace;
end;

function TCircuitSensor.GetTagEval:LongString;
begin
 if Assigned(Self) then Result:=myTagEval else Result:='';
end;

procedure TCircuitSensor.SetTagEval(const aTagEval:LongString);
begin
 if Assigned(Self) then myTagEval:=aTagEval;
end;

function TCircuitSensor.HasTagEval:Boolean;
begin
 if Assigned(Self) then Result:=Length(myTagEval)>0 else Result:=false;
end;

function TCircuitSensor.GetLedEval:LongString;
begin
 if Assigned(Self) then Result:=myLedEval else Result:='';
end;

procedure TCircuitSensor.SetLedEval(const aLedEval:LongString);
begin
 if Assigned(Self) then myLedEval:=aLedEval;
end;

function TCircuitSensor.HasLedEval:Boolean;
begin
 if Assigned(Self) then Result:=Length(myLedEval)>0 else Result:=false;
end;

function TCircuitSensor.GetPainter:LongString;
begin
 if Assigned(Self) then Result:=myPainter else Result:='';
end;

procedure TCircuitSensor.SetPainter(const aPainter:LongString);
begin
 if Assigned(Self) then begin
  myPainterData.IsSimple:=IsSimpleScript(aPainter);
  myPainter:=aPainter;
 end;
end;

function TCircuitSensor.GetPainterCall:LongString;
begin
 if Assigned(Self) then Result:=myPainterCall else Result:='';
end;

procedure TCircuitSensor.SetPainterCall(const aPainterCall:LongString);
begin
 if Assigned(Self) then myPainterCall:=aPainterCall;
end;

function TCircuitSensor.GetPainterIsSimple:Boolean;
begin
 if Assigned(Self) then Result:=myPainterData.IsSimple else Result:=false;
end;

function TCircuitSensor.HasPainter:Boolean;
begin
 if Assigned(Self) then Result:=Length(myPainter)>0 else Result:=false;
end;

function TCircuitSensor.HasLedWidth:Boolean;
begin
 if Assigned(Self) then Result:=myLedWidth<>0 else Result:=false;
end;

function TCircuitSensor.GetPendingDraw:Boolean;
begin
 if Assigned(Self)
 then Result:=LockedExchange(myPendingDraw,0)<>0
 else Result:=false;
end;

procedure TCircuitSensor.SetPendingDraw(aPendingDraw:Boolean);
begin
 if Assigned(Self) then begin
  if aPendingDraw then LockedInc(myPendingDraw);
 end;
end;

function TCircuitSensor.GetParentForm:TForm;
begin
 if Assigned(Self) then Result:=myParentForm else Result:=nil;
end;

procedure TCircuitSensor.SetParentForm(aParentForm:TForm);
begin
 if Assigned(Self) then myParentForm:=aParentForm;
end;

function TCircuitSensor.GetSourceCrcFile:LongString;
begin
 if Assigned(Self) then Result:=myCrcFile else Result:='';
end;

function TCircuitSensor.GetSourceSection:LongString;
begin
 if Assigned(Self) then Result:=mySection else Result:='';
end;

function TCircuitSensor.GetGuardLevel:Integer;
begin
 if Assigned(Self) then Result:=myGuardLevel else Result:=ga_Lock;
end;

procedure TCircuitSensor.SetGuardLevel(aLevel:Integer);
begin
 if (Self=nil) then Exit;
 if TGuard.IsValidLevel(aLevel) then myGuardLevel:=aLevel;
end;

constructor TCircuitSensor.Create(const aPos:TPoint2I; const aName:LongString);
begin
 inherited Create;
 BmpCache.Capture;
 LockedInit(myPendingDraw);
 myItems:=NewObjectStorage(true);
 myName:=UnifyAlias(aName);
 myPos:=aPos;
 myBookMark:=0;
 myElseMark:=0;
 myLedWidth:=0;
 myLedDigit:=0;
 myLedValue:='0';
 myLedDrawn:='';
 myLedFormat:='';
 myLedFormatType:=0;
 myLedFontLink:='';
 myLedFontPara:=DefaultCircuitFont;
 myLinkedTag:=0;
 myLinkedTagColor:=0;
 myLinkedTagParam:=0;
 myLinkedTagTimer:=0;
 myUsesLinkedTagColor:=false;
 myUsesLinkedTagParam:=false;
 myUsesLinkedTagTimer:=false;
 myLinkedCurve:=nil;
 myLinkedObject:=nil;
 myTextColor:=DefaultCircuitFont.Color;
 myShortCut:=0;
 myHintText:='';
 myHintLine:='';
 myToolBarKey:=0;
 myToolBarItems:=nil;
 myToolBarOpaque:=false;
 myToolBarBorder:=true;
 myToolBarSpace:=2;
 myTagEval:='';
 myLedEval:='';
 myPainter:='';
 myPainterCall:='';
 myPainterData.Window:=nil;
 myPainterData.Canvas:=nil;
 myPainterData.Bitmap:=nil;
 myPainterData.pc:=Ord(clBlack);
 myPainterData.ps:=Ord(psSolid);
 myPainterData.pm:=Ord(pmCopy);
 myPainterData.pw:=1;
 myPainterData.bc:=Ord(clBlack);
 myPainterData.bs:=Ord(bsSolid);
 myPainterData.cx:=0;
 myPainterData.cy:=0;
 myPainterData.lx:=0;
 myPainterData.ly:=0;
 myPainterData.IsSimple:=true;
 SetLength(myPainterData.PolyPoints,0);
 myParentForm:=nil;
 myCrcFile:='';
 mySection:='';
 myGuardLevel:=Guard.GetActionLevel('TFormCircuitWindow.ClickSensor',ga_Guest);
end;

destructor TCircuitSensor.Destroy;
begin
 Kill(myItems);
 myName:='';
 myLedValue:='';
 myLedDrawn:='';
 myLedFormat:='';
 myLedFontLink:='';
 myHintText:='';
 myHintLine:='';
 myTagEval:='';
 myLedEval:='';
 myPainter:='';
 myPainterCall:='';
 SetLength(myPainterData.PolyPoints,0);
 myCrcFile:='';
 mySection:='';
 Kill(myToolBarItems);
 LockedFree(myPendingDraw);
 BmpCache.Uncapture;
 inherited Destroy;
end;

function TCircuitSensor.Search(aIdent:LongInt; out aIndex:Integer):Boolean;
var Left,Right,Middle,Comparison:Integer;
begin
 Result:=False;
 Left:=0;
 if Assigned(Self) then begin
  Right:=Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) shr 1;
   Comparison:=Items[Middle].Ident-aIdent;
   if Comparison<0 then Left:=Middle+1 else begin
    Right:=Middle-1;
    if Comparison=0 then begin
     Result:=True;
     Left:=Middle;
    end;
   end;
  end;
 end;
 aIndex:=Left;
end;

procedure TCircuitSensor.Add(aGlyph:TSmartBitmap);
var aIndex:Integer;
begin
 if Ok and (aGlyph is TSmartBitmap) then begin
  if Search(aGlyph.Ident,aIndex)
  then myItems[aIndex]:=aGlyph
  else myItems.Insert(aIndex,aGlyph);
  if not Search(BookMark,aIndex) then BookMark:=aGlyph.Ident;
 end;
end;

procedure TCircuitSensor.Draw(Canvas:TCanvas; const Where:TPoint2I);
const
 Recursion : Integer = 0;
var
 WindowState : TWindowState;
 Str         : LongString;
 aGlyph      : TSmartBitmap;
 TextRect    : TRect2I;
 GlyphRect   : TRect2I;
 Bitmap      : TBitmap;
 procedure RunPainter(Canvas:TCanvas; Bitmap:TBitmap);
 var
  ParentWin : TFormCircuitWindow;
  Evaluator : TExpressionEvaluator;
 begin
  try
   if Length(LedFontLink)>0
   then RestoreFont(Bitmap.Canvas.Font,myLedFontPara)
   else Bitmap.Canvas.Font.Assign(Canvas.Font);
   TObject(ParentWin):=myParentForm;
   if Assigned(ParentWin) then
   if (TObject(ParentWin) is TFormCircuitWindow) then begin
    Evaluator:=ParentWin.Evaluator;
    if Assigned(Evaluator) then begin
     myPainterData.Window:=ParentWin;
     myPainterData.Canvas:=Canvas;
     myPainterData.Bitmap:=Bitmap;
     myPainterData.lx:=0;
     myPainterData.ly:=0;
     Evaluator.Custom:=Self;
     Inc(CircuitWindowsProfiler.Curr.PainterCall);
     if Evaluator.SetValue('v',Self.LinkedValue) then begin
      if PainterIsSimple then begin
       if EvaluateSimpleScript(Evaluator,Painter)<>ee_OK then begin
        Inc(TCircuitSensorPainterErrorCount);
        if (ParentWin.DebugFlags and (df_PainterBugEcho or df_PainterBugFile) <> 0)
        then PrintPainterError(ParentWin.DebugFlags,FormatPainterError(ParentWin,Self,Evaluator,mSecNow,'fail Painter(v) Evaluate'));
       end;
      end else begin
       Evaluator.Script:=Painter;
       if Evaluator.RunScript<>ee_OK then begin
        Inc(TCircuitSensorPainterErrorCount);
        if (ParentWin.DebugFlags and (df_PainterBugEcho or df_PainterBugFile) <> 0)
        then PrintPainterError(ParentWin.DebugFlags,FormatPainterError(ParentWin,Self,Evaluator,mSecNow,'fail Painter(v) Evaluate'));
       end;
       Evaluator.Script:='';
      end;
     end else begin
      Inc(TCircuitSensorPainterErrorCount);
      if (ParentWin.DebugFlags and (df_PainterBugEcho or df_PainterBugFile) <> 0)
      then PrintPainterError(ParentWin.DebugFlags,FormatPainterError(ParentWin,Self,nil,mSecNow,'fail Painter(v) SetValue'));
     end;
     Evaluator.Custom:=nil;
     myPainterData.Window:=nil;
     myPainterData.Canvas:=nil;
     myPainterData.Bitmap:=nil;
     SetLength(myPainterData.PolyPoints,0);
    end;
   end;
  except
   on E:Exception do BugReport(E,Self,'RunPainter');
  end;
 end;
begin
 if Ok then
 if Assigned(Canvas) then
 try
  Inc(Recursion);
  try
   if Recursion=1 then begin
    aGlyph:=Glyph;
    if aGlyph.ExposedInsideOf(Canvas, Where.X, Where.Y) then begin
     Bitmap:=aGlyph.CreateBitmap;
     try
      Inc(CircuitWindowsProfiler.Curr.MakeBitmap);
      if Assigned(myParentForm) then WindowState:=myParentForm.WindowState else WindowState:=wsNormal;
      if Bitmap is TBitmap then begin
       Str:='';
       if HasLedWidth then begin
        if LedWidth>0
        then Str:=Pad(LedValue,LedWidth)
        else Str:=LeftPad(LedValue,Abs(LedWidth));
       end else Str:=aGlyph.Title;
       if HasPainter then RunPainter(Canvas,Bitmap);
       if (WindowState<>wsMinimized) then begin
        if Length(Str)>0 then begin
         if Length(LedFontLink)>0
         then RestoreFont(Bitmap.Canvas.Font,myLedFontPara)
         else Bitmap.Canvas.Font.Assign(Canvas.Font);
         GlyphRect:=Rect2I(0, 0, Bitmap.Width, Bitmap.Height);
         TextRect:=Rect2I(0, 0, Bitmap.Canvas.TextWidth(Str),Bitmap.Canvas.TextHeight(Str));
         RectMove(TextRect, RectCenterX(GlyphRect) - RectCenterX(TextRect),
                            RectCenterY(GlyphRect) - RectCenterY(TextRect));
         if ((myPainterData.lx<>0) or (myPainterData.ly<>0)) and HasPainter
         then RectMove(TextRect,myPainterData.lx,myPainterData.ly);
         DrawText(Bitmap.Canvas, TextRect.A, Str, TextColor, clWhite, bsClear);
        end;
        Canvas.Draw(Where.X, Where.Y, Bitmap);
        Inc(CircuitWindowsProfiler.Curr.DrawSensor);
       end;
       myLedDrawn:=Str;
      end;
     finally
      Kill(Bitmap);
     end;
    end;
   end;
  finally
   Dec(Recursion);
  end;
 except
  on E:Exception do BugReport(E,Self,'Draw');
 end;
end;

procedure TCircuitSensor.Reload;
var
 i : Integer;
begin
 if Ok then
 try
  for i:=0 to Count-1 do  with Items[i] do
  if FileExists(Source) then LoadFromFile(Source);
 except
  on E:Exception do BugReport(E,Self,'Reload');
 end;
end;

function NewCircuitSensorFromCrcFile(CrcFile,Section:LongString):TCircuitSensor;
var aPos:TPoint2I; aName,aTitl,aEval:LongString; aOpaque,aBorder:Boolean;
var aLed:packed record w,d:LongInt; v,f,p:PureString; end;
var aTag:packed record d:LongInt; s:PureString; end;
var i,aKey,aSpace:Integer;
begin
 Result:=nil;
 aPos:=Point2I(0,0);
 CrcFile:=UnifyFileAlias(CrcFile,ua_FileLow);
 if ReadIniFileRecord(CrcFile,Section,'Pos%i;%i',aPos) then
 try
  aName:=''; aTitl:=''; aEval:='';
  SafeFillChar(aTag,SizeOf(aTag),0);
  SafeFillChar(aLed,SizeOf(aLed),0);
  aKey:=0; aSpace:=0; aOpaque:=false; aBorder:=false;
  if not ReadIniFileAlpha(CrcFile,Section,'Name%a',aName)
  then aName:=RemoveBrackets(Section);
  if IsNonEmptyStr(aName) then Result:=TCircuitSensor.Create(aPos,aName);
  i:=1;
  if Result.Ok then
  while ReadIniFileRecord(CrcFile,Section,'Tag#'+d2s(i)+'%d;%s',aTag,efConfigNC) do begin
   aName:=UnifyFileAlias(SmartFileRef(ExtractFirstParamUrl(aTag.s,QuoteMark,ScanSpaces),'.bmp',CrcFile),ua_FileLow);
   aTitl:=ExtractFirstParamUrl(SkipFirstParam(aTag.s,QuoteMark,ScanSpaces),QuoteMark,ScanSpaces);
   if SmartCheckBmpFileExists(aName)
   then Result.Add(NewSmartBitmap(aName,nil,aTag.d,aTitl))
   else SendToMainConsole('@silent @integrity DAQ:CrcReadBmp:Failure '+aName+EOL);
   inc(i);
  end;
  if Result.Count>0 then begin
   Result.myCrcFile:=Trim(CrcFile); Result.mySection:=Trim(Section);
   if ReadIniFileLongInt(CrcFile,Section,'Tag%d',aTag.d) then begin
    Result.BookMark:=aTag.d;
    Result.ElseMark:=aTag.d;
   end;
   aLed.w:=0;
   aLed.d:=0;
   aLed.v:='0';
   aLed.f:='*';
   aLed.p:='*';
   ReadIniFileRecord(CrcFile,Section,'LED%d;%d;%a;%a;%a',aLed,efConfigNC);
   Result.LedWidth:=aLed.w;
   Result.LedDigit:=aLed.d;
   Result.LedValue:=aLed.v;
   if Pos('%',aLed.f)>0 then Result.LedFormat:=aLed.f;
   Result.LedFontLink:='';
   Result.myLedFontPara:=DefaultCircuitFont;
   if IsNonEmptyStr(aLed.p) and not SameText(aLed.p,'*') then begin
    if IsSectionName(aLed.p) then begin
     if ReadIniFileFont(Result.myLedFontPara,CrcFile,UnifySection(aLed.p))
     then Result.LedFontLink:=aLed.p else Result.LedFontLink:='';
    end else begin
     if ReadBufferedFont(Result.myLedFontPara,aLed.p,true,StandardFont)
     then Result.LedFontLink:=aLed.p else Result.LedFontLink:='';
    end;
    if (Result.LedFontLink='') then Result.myLedFontPara:=DefaultCircuitFont;
    if (Result.LedFontLink<>'') then Result.TextColor:=Result.LedFontColor;
   end;
   aTag.s:='';
   if ReadIniFileString(CrcFile,Section,'Hint%s',aTag.s,efConfig and not efCaseMask)
   then Result.HintText:=Dequote_or_URL_Decode(aTag.s);
   if Result.LedWidth=0 then
   if ReadIniFileInteger(CrcFile,Section,'ToolBarKey%i',aKey)
   then Result.ToolBarKey:=aKey;
   if Result.ToolBarKey<>0 then begin
    if ReadIniFileBoolean(CrcFile,Section,'ToolBarOpaque%b',aOpaque)
    then Result.ToolBarOpaque:=aOpaque;
    if ReadIniFileBoolean(CrcFile,Section,'ToolBarBorder%b',aBorder)
    then Result.ToolBarBorder:=aBorder;
    if ReadIniFileInteger(CrcFile,Section,'ToolBarSpace%i',aSpace)
    then Result.ToolBarSpace:=aSpace;
   end;
   if ReadIniFileString(CrcFile,Section,'TagEval(v)%s',aEval,efConfigNC)
   then Result.TagEval:=Trim(aEval);
   if ReadIniFileString(CrcFile,Section,'LedEval(v)%s',aEval,efConfigNC)
   then Result.LedEval:=Trim(aEval);
   Result.PainterCall:=ExtractTextSectionByPrefix(CrcFile,Section,'Painter(v)',efConfigNC,svConfig,false);
   Result.Painter:=ExtractTextSectionByPrefix(CrcFile,Section,'Painter(v)',efConfigNC,svConfig,true);
   if ReadIniFileAlpha(CrcFile,Section,'GuardLevel%a',aName)
   then Result.GuardLevel:=Guard.NameToLevel(aName,Result.GuardLevel);
  end else Kill(TObject(Result));
 except
  on E:Exception do BugReport(E,nil,'NewCircuitSensorFromCrcFile');
 end;
end;

function SmartCheckBmpFileExists(FileName:LongString):Boolean;
var TextWidth,FontSize,BarWidth,BarHeight,ColorDepth,FillColor:Integer;
var bmp:TBitmap; BaseName,FontName:LongString;
const ColorDepthSet=[1,4,8,15,16,24];
begin
 Result:=false;
 if IsEmptyStr(FileName) then exit;
 FileName:=UnifyFileAlias(FileName,ua_FileLow);
 if FileExists(FileName) then begin Result:=true; exit; end;
 if SameText(ExtractFileExt(FileName),'.bmp') then
 try
  BaseName:=ExtractBaseName(FileName);
  // Handle barbmp_width_height_depth_color.bmp
  // For example, barbmp_120_20_4_white.bmp
  if SameText(ExtractWord(1,BaseName,['_']),'barbmp') then
  if Str2Int(ExtractWord(2,BaseName,['_']),BarWidth) then if (BarWidth>0) then
  if Str2Int(ExtractWord(3,BaseName,['_']),BarHeight) then if (BarHeight>0) then
  if Str2Int(ExtractWord(4,BaseName,['_']),ColorDepth) then if (ColorDepth in ColorDepthSet) then begin
   FillColor:=_crw_colors.StringToColor(ExtractWord(5,BaseName,['_']));
   if (FillColor<>clNone) then begin
    bmp:=CreateBarBmp(BarWidth,BarHeight,ColorDepth,FillColor);
    if Assigned(bmp) then
    try
     bmp.SaveToFile(FileName);
     SendToMainConsole('@silent @integrity DAQ:CrcReadBmp:Created '+FileName+EOL);
     if FileExists(FileName) then Result:=true;
    finally
     FreeAndNil(bmp);
    end;
   end;
  end;
  // Handle ledbmp_textwidth_fontsize_depth_color_font.bmp
  // For example, ledbmp_8_10_4_aqua_ptmono.bmp
  if SameText(ExtractWord(1,BaseName,['_']),'ledbmp') then
  if Str2Int(ExtractWord(2,BaseName,['_']),TextWidth) then if (TextWidth>0) then
  if Str2Int(ExtractWord(3,BaseName,['_']),FontSize) then if (FontSize>0) then
  if Str2Int(ExtractWord(4,BaseName,['_']),ColorDepth) then if (ColorDepth in ColorDepthSet) then begin
   FillColor:=_crw_colors.StringToColor(ExtractWord(5,BaseName,['_']));
   FontName:=FindFontNameByAlias(ExtractWord(6,BaseName,['_']));
   if (FillColor<>clNone) and (FontName<>'') then begin
    bmp:=CreateLedBmp(TextWidth,FontSize,ColorDepth,FillColor,FontName);
    if Assigned(bmp) then
    try
     bmp.SaveToFile(FileName);
     SendToMainConsole('@silent @integrity DAQ:CrcReadBmp:Created '+FileName+EOL);
     if FileExists(FileName) then Result:=true;
    finally
     FreeAndNil(bmp);
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'SmartCheckBmpFileExists');
 end;
end;

 {
 ***********************************************************************************************************************
 Sensor Painter routines
 sensorwidth()                                      - get sensor width
 sensorheight()                                     - get sensor height
 linkedvalue()                                      - linked tag/curve value, must be equal to v
 linkedcurve()                                      - linked curve reference
 linkedtag()                                        - linked tag reference
 linkedtagcolor()                                   - gettagcolor(linkedtag)
 linkedtagparam()                                   - gettagparam(linkedtag)
 linkedtagtimer()                                   - gettagtimer(linkedtag)
 bookmark()                                         - sensor bookmark, i.e. current picture tag
 ledwidth()                                         - LED field width
 leddigit()                                         - LED digits after dot
 moveled(dx,dy)                                     - move LED to (dx,dy) px
 setcursor(x,y)                                     - set text cursor position
 setpen(pencolor,penstyle,penmode,penwidth)         - set pen style to draw lines
 setbrush(brushcolor,brushstyle)                    - set brush style to fill bars
 drawpoint(x1,y1,pointcolor,pointstyle)             - draw a point with given marker
 drawline(x1,y1,x2,y2)                              - draw a line with current pen
 drawarrow(x1,y1,x2,y2,d1,d2)                       - draw a line with arraw ends d1,d2
 drawrect(x1,y1,x2,y2)                              - draw a rect (filled rectangle)
 drawbox(x1,y1,x2,y2)                               - draw a box  (filled rectangle)
 drawbar(x1,y1,x2,y2)                               - draw a bar  (filled rectangle)
 drawroundrect(x1,y1,x2,y2,rx,ry)                   - draw a rounded rect (filled rounded rectangle)
 drawroundbox(x1,y1,x2,y2,rx,ry)                    - draw a rounded rect (filled rounded rectangle)
 drawroundbar(x1,y1,x2,y2,rx,ry)                    - draw a rounded rect (filled rounded rectangle)
 drawellipse(x1,y1,x2,y2)                           - draw a filled ellipse
 drawpie(x1,y1,x2,y2,x3,y3,x4,y4)                   - draw a filled ellipse pie
 drawarc(x1,y1,x2,y2,x3,y3,x4,y4)                   - draw a ellipse arc
 drawchord(x1,y1,x2,y2,x3,y3,x4,y4)                 - draw a filled ellipse chord
 drawpoly(x,y,pl)                                   - draw polygonal figure of (x,y) array, pl is operation
 setsimulation(m)                                   - set simulation mode m=0/1=normal/simulation (just for testing)
 setdebugflags(f)                                   - set DebugFlags (just for debug):
                                                      bit 0/1/2/3: echo on Startup/TagEval/LedEval/Painter errors
                                                      bit 4/5/6/7: log to Temp\Debug.out file on Startup/TagEval/LedEval/Painter errors
 @print msg                                         - draw a text (msg) at cursor
 @textwidth msg                                     - return text width
 @textheight msg                                    - return text height
 @font charset/color/height/size/name/pitch/style value  - assign value to font property like @font name PT Mono
 where:
  color=[R,G,B]=[clBlack,clMaroon,clGreen,clOlive,clNavy,clPurple,clTeal,clGray,clSilver,clRed,clLime,clYellow,clBlue,clFuchsia,clAqua,clLtGray,clDkGray,clWhite]
  penstyle=[0..6]=[psSolid,psDash,psDot,psDashDot,psDashDotDot,psClear,psInsideFrame]
  penmode=[0..15]=[pmBlack,pmWhite,pmNop,pmNot,pmCopy,pmNotCopy,pmMergePenNot,pmMaskPenNot,pmMergeNotPen,pmMaskNotPen,pmMerge,pmNotMerge,pmMask,pmNotMask,pmXor,pmNotXor]
  penwidth=line width
  barcolor=color of bar filling
  barstyle=[0..7]=[bsSolid,bsClear,bsHorizontal,bsVertical,bsFDiagonal,bsBDiagonal,bsCross,bsDiagCross]
  charset=[ANSI_CHARSET, DEFAULT_CHARSET, SYMBOL_CHARSET, MAC_CHARSET, SHIFTJIS_CHARSET, HANGEUL_CHARSET,
           JOHAB_CHARSET, GB2312_CHARSET, CHINESEBIG5_CHARSET, GREEK_CHARSET, TURKISH_CHARSET, HEBREW_CHARSET,
           ARABIC_CHARSET, BALTIC_CHARSET, RUSSIAN_CHARSET, THAI_CHARSET, EASTEUROPE_CHARSET, OEM_CHARSET]
  font style=[0..15]=4 bits [fsBold,fsItalic,fsUnderline,fsStrikeOut]
  font pitch=[0..2]=[fpDefault,fpVariable,fpFixed]
  pl=[0..5]=plNone,plClear,plAddPoint,plPolyLine,plPolygon,plPolyBezier
 ***********************************************************************************************************************
 }
type
 TPainterObjects = record
  Sensor : TCircuitSensor;
  Window : TFormCircuitWindow;
  Bitmap : TBitmap;
  Canvas : TCanvas;
 end;

function GetPainterObjects(ee:TExpressionEvaluator; out PainterObjects:TPainterObjects):Boolean;
begin
 with PainterObjects do begin
  Sensor:=ee.Custom;
  if Assigned(Sensor) and not (TObject(Sensor) is TCircuitSensor) then Sensor:=nil;
  if Assigned(Sensor) then begin
   Window:=Sensor.myPainterData.Window;
   Bitmap:=Sensor.myPainterData.Bitmap;
   Canvas:=Sensor.myPainterData.Canvas;
  end else begin
   Window:=nil;
   Bitmap:=nil;
   Canvas:=nil;
  end;
  Inc(CircuitWindowsProfiler.Curr.PainterApiCall);
  Result:=Assigned(Sensor) and Assigned(Window) and Assigned(Bitmap) and Assigned(Canvas);
 end;
end;

function _sensorwidth(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Bitmap.Width;
  end;
 except
  on E:Exception do BugReport(E,nil,'_sensorwidth');
 end;
end;

function _sensorheight(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Bitmap.Height;
  end;
 except
  on E:Exception do BugReport(E,nil,'_sensorheight');
 end;
end;

function _linkedvalue(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Sensor.LinkedValue;
  end;
 except
  on E:Exception do BugReport(E,nil,'_linkedvalue');
 end;
end;

function _linkedcurve(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Sensor.LinkedCurve.Ref;
  end;
 except
  on E:Exception do BugReport(E,nil,'_linkedcurve');
 end;
end;

function _linkedtag(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Sensor.LinkedTag;
  end;
 except
  on E:Exception do BugReport(E,nil,'_linkedtag');
 end;
end;

function _linkedtagcolor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.UsesLinkedTagColor:=true;
   Result:=Sensor.LinkedTagColor;
  end;
 except
  on E:Exception do BugReport(E,nil,'_linkedtagcolor');
 end;
end;

function _linkedtagparam(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.UsesLinkedTagParam:=true;
   Result:=Sensor.LinkedTagParam;
  end;
 except
  on E:Exception do BugReport(E,nil,'_linkedtagparam');
 end;
end;

function _linkedtagtimer(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.UsesLinkedTagTimer:=true;
   Result:=Sensor.LinkedTagTimer;
  end;
 except
  on E:Exception do BugReport(E,nil,'_linkedtagtimer');
 end;
end;

function _bookmark(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Sensor.Bookmark;
  end;
 except
  on E:Exception do BugReport(E,nil,'_bookmark');
 end;
end;

function _ledwidth(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Sensor.LedWidth;
  end;
 except
  on E:Exception do BugReport(E,nil,'_ledwidth');
 end;
end;

function _leddigit(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Result:=Sensor.LedDigit;
  end;
 except
  on E:Exception do BugReport(E,nil,'_leddigit');
 end;
end;

function _moveled(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.myPainterData.lx:=round(x[0]);
   Sensor.myPainterData.ly:=round(x[1]);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_moveled');
 end;
end;

function _setcursor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.myPainterData.cx:=round(x[0]);
   Sensor.myPainterData.cy:=round(x[1]);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_setcursor');
 end;
end;

function _setpen(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.myPainterData.pc:=round(x[0]);
   Sensor.myPainterData.ps:=round(x[1]);
   Sensor.myPainterData.pm:=round(x[2]);
   Sensor.myPainterData.pw:=round(x[3]);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_setpen');
 end;
end;

function _setbrush(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   Sensor.myPainterData.bc:=round(x[0]);
   Sensor.myPainterData.bs:=round(x[1]);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_setbrush');
 end;
end;

function _drawpoint(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,pc,ps:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); pc:=round(x[2]); ps:=round(x[3]);
   if Window.Paintable then
   DrawPointMarker(Bitmap.Canvas,Point2I(x1,y1),TColor(pc),ps);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawpoint');
 end;
end;

function _drawline(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   if Window.Paintable then
   DrawLine(Bitmap.Canvas,Point2I(x1,y1),Point2i(x2,y2),
            TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
            TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawline');
 end;
end;

function _drawarrow(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2,d1,d2:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   d1:=round(x[4]); d2:=round(x[5]);
   if Window.Paintable then
   DrawArrow(Bitmap.Canvas,Point2I(x1,y1),Point2i(x2,y2),
             TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
             TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,d1,d2);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawarrow');
 end;
end;

function _drawrect(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   if Window.Paintable then
   DrawCustomRect(Bitmap.Canvas,Rect2I(x1,y1,x2,y2),
                  TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                  TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,
                  TColor(Sensor.myPainterData.bc),TBrushStyle(Sensor.myPainterData.bs));
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawrect');
 end;
end;

function _drawroundrect(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2,rx,ry:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   rx:=round(x[4]); ry:=round(x[5]);
   if Window.Paintable then
   DrawRoundRect(Bitmap.Canvas,Rect2I(x1,y1,x2,y2),Point2I(rx,ry),
                  TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                  TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,
                  TColor(Sensor.myPainterData.bc),TBrushStyle(Sensor.myPainterData.bs));
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawroundrect');
 end;
end;

function _drawellipse(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   if Window.Paintable then
   DrawEllipse(Bitmap.Canvas,Rect2I(x1,y1,x2,y2),
               TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
               TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,
               TColor(Sensor.myPainterData.bc),TBrushStyle(Sensor.myPainterData.bs));
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawellipse');
 end;
end;

function _drawpie(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2,x3,y3,x4,y4:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   x3:=round(x[4]); y3:=round(x[5]); x4:=round(x[6]); y4:=round(x[7]);
   if Window.Paintable then
   DrawEllipsePie(Bitmap.Canvas,Rect2I(x1,y1,x2,y2),Rect2I(x3,y3,x4,y4),
                  TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                  TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,
                  TColor(Sensor.myPainterData.bc),TBrushStyle(Sensor.myPainterData.bs));
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawpie');
 end;
end;

function _drawarc(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2,x3,y3,x4,y4:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   x3:=round(x[4]); y3:=round(x[5]); x4:=round(x[6]); y4:=round(x[7]);
   if Window.Paintable then
   DrawEllipseArc(Bitmap.Canvas,Rect2I(x1,y1,x2,y2),Rect2I(x3,y3,x4,y4),
                  TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                  TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw);
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawarc');
 end;
end;

function _drawchord(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,x2,y2,x3,y3,x4,y4:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); x2:=round(x[2]); y2:=round(x[3]);
   x3:=round(x[4]); y3:=round(x[5]); x4:=round(x[6]); y4:=round(x[7]);
   if Window.Paintable then
   DrawEllipseChord(Bitmap.Canvas,Rect2I(x1,y1,x2,y2),Rect2I(x3,y3,x4,y4),
                    TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                    TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,
                    TColor(Sensor.myPainterData.bc),TBrushStyle(Sensor.myPainterData.bs));
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawchord');
 end;
end;

function _drawpoly(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var PainterObjects:TPainterObjects;
var x1,y1,pl,len:Integer;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   x1:=round(x[0]); y1:=round(x[1]); pl:=round(x[2]);
   if Window.Paintable then
   case pl of
    0: // Nothing to do
    begin end;
    1: // Clear polygonal points array
    SetLength(Sensor.myPainterData.PolyPoints,0);
    2: // Add point to polygonal array
    begin
     len:=Length(Sensor.myPainterData.PolyPoints);
     SetLength(Sensor.myPainterData.PolyPoints,len+1);
     Sensor.myPainterData.PolyPoints[len]:=Point(x1,y1);
    end;
    3: // Draw polygonal line
    begin
     len:=Length(Sensor.myPainterData.PolyPoints);
     if len<2 then exit;
     DrawPolyLine(Bitmap.Canvas,Sensor.myPainterData.PolyPoints,
                  TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                  TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw);
    end;
    4: // Draw filled polygonal figure
    begin
     len:=Length(Sensor.myPainterData.PolyPoints);
     if len<3 then exit;
     DrawPolygon(Bitmap.Canvas,Sensor.myPainterData.PolyPoints,
                 TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                 TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw,
                 TColor(Sensor.myPainterData.bc),TBrushStyle(Sensor.myPainterData.bs));
    end;
    5: // Draw Bezier spline figure
    begin
     len:=Length(Sensor.myPainterData.PolyPoints);
     if len<4 then exit;
     DrawPolyBezier(Bitmap.Canvas,Sensor.myPainterData.PolyPoints,
                    TColor(Sensor.myPainterData.pc),TPenStyle(Sensor.myPainterData.ps),
                    TPenMode(Sensor.myPainterData.pm),Sensor.myPainterData.pw);
    end;
    else exit;
   end;
   Result:=1;
  end;
 except
  on E:Exception do BugReport(E,nil,'_drawpoly');
 end;
end;

function _setsimulation(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var Window:TFormCircuitWindow;
begin
 Result:=0;
 try
  TObject(Window):=ee.Parent;
  if TObject(Window) is TFormCircuitWindow then begin
   Result:=Ord(Window.Simulation);
   Window.Simulation:=(x[0]<>0);
  end;
 except
  on E:Exception do BugReport(E,nil,'_setsimulation');
 end;
end;

function _setdebugflags(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
var Window:TFormCircuitWindow;
begin
 Result:=0;
 try
  TObject(Window):=ee.Parent;
  if TObject(Window) is TFormCircuitWindow then begin
   Result:=Window.DebugFlags;
   if not IsNanOrInf(x[0]) then Window.DebugFlags:=Round(x[0]);
  end;
 except
  on E:Exception do BugReport(E,nil,'_setdebugflags');
 end;
end;

function act_print(ee:TExpressionEvaluator; const args:LongString):double;
var PainterObjects:TPainterObjects;
var line:LongString;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   line:=ee.SmartArgs(args);
   if Length(line)>0 then
   if Window.Paintable then
   DrawText(Bitmap.Canvas, Point2I(Sensor.myPainterData.cx,Sensor.myPainterData.cy), line,
            Bitmap.Canvas.Font.Color, Sensor.myPainterData.bc, TBrushStyle(Sensor.myPainterData.bs));
   Result:=Length(line);
  end;
 except
  on E:Exception do BugReport(E,nil,'act_print');
 end;
end;

function act_textwidth(ee:TExpressionEvaluator; const args:LongString):double;
var PainterObjects:TPainterObjects;
var line:LongString;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   line:=ee.SmartArgs(args);
   Result:=Bitmap.Canvas.TextWidth(line);
  end;
 except
  on E:Exception do BugReport(E,nil,'act_textwidth');
 end;
end;

function act_textheight(ee:TExpressionEvaluator; const args:LongString):double;
var PainterObjects:TPainterObjects;
var line:LongString;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   line:=ee.SmartArgs(args);
   Result:=Bitmap.Canvas.TextHeight(line);
  end;
 except
  on E:Exception do BugReport(E,nil,'act_textheight');
 end;
end;

function act_font(ee:TExpressionEvaluator; const args:LongString):double;
var PainterObjects:TPainterObjects;
var line,w1,wt:LongString; i:Integer; fs:TFontStyles;
begin
 Result:=0;
 try
  if GetPainterObjects(ee,PainterObjects) then with PainterObjects do begin
   line:=ee.SmartArgs(args);
   w1:=ExtractWord(1,line,ScanSpaces);
   wt:=Trim(SkipWords(1,line,ScanSpaces));
   if IsSameText(w1,'Charset') then begin
    Result:=Bitmap.Canvas.Font.Charset;
    if Str2Int(wt,i) then Bitmap.Canvas.Font.Charset:=i;
   end else
   if IsSameText(w1,'Color') then begin
    Result:=Bitmap.Canvas.Font.Color;
    if Str2Int(wt,i) then Bitmap.Canvas.Font.Color:=i;
   end else
   if IsSameText(w1,'Height') then begin
    Result:=Bitmap.Canvas.Font.Height;
    if Str2Int(wt,i) then Bitmap.Canvas.Font.Height:=i;
   end else
   if IsSameText(w1,'Size') then begin
    Result:=Bitmap.Canvas.Font.Size;
    if Str2Int(wt,i) then Bitmap.Canvas.Font.Size:=i;
   end else
   if IsSameText(w1,'Name') then begin
    Result:=Length(Bitmap.Canvas.Font.Name);
    if IsNonEmptyStr(wt) then Bitmap.Canvas.Font.Name:=Trim(wt);
   end else
   if IsSameText(w1,'Pitch') then begin
    Result:=Ord(Bitmap.Canvas.Font.Pitch);
    if Str2Int(wt,i) then Bitmap.Canvas.Font.Pitch:=TFontPitch(i);
   end else
   if IsSameText(w1,'Style') then begin
    Result:=0;
    if (fsBold      in Bitmap.Canvas.Font.Style) then Result:=Result+1;
    if (fsItalic    in Bitmap.Canvas.Font.Style) then Result:=Result+2;
    if (fsUnderline in Bitmap.Canvas.Font.Style) then Result:=Result+4;
    if (fsStrikeout in Bitmap.Canvas.Font.Style) then Result:=Result+8;
    if Str2Int(wt,i) then begin
     fs:=[];
     if (i and 1) <> 0 then fs:=fs+[fsBold];
     if (i and 2) <> 0 then fs:=fs+[fsItalic];
     if (i and 4) <> 0 then fs:=fs+[fsUnderline];
     if (i and 8) <> 0 then fs:=fs+[fsStrikeout];
     Bitmap.Canvas.Font.Style:=fs;
    end;
   end;
  end;
 except
  on E:Exception do BugReport(E,nil,'act_font');
 end;
end;

procedure InitPainter(ee:TExpressionEvaluator);
 procedure SetConst(ee:TExpressionEvaluator; value:Double; const name:LongString);
 begin
  ee.ConstList.SetValue(PChar(name),value);
 end;
 procedure AddSomeKnownColors;
 var i:Integer; name:String; code:TColor;
 begin
  for i:=0 to KnownColorsCount-1 do begin
   name:=KnownColorsName(i);
   if (StrFetch(name,1)='_') then begin
    code:=KnownColorsCode(i);    // Add both colors as
    SetConst(ee,code,'cl'+name); // _3DFace and 3DFace
    SetConst(ee,code,'cl'+Copy(name,2,Length(name)-1));
   end;
  end;
 end;
begin
 AddSomeKnownColors;
 SetConst(ee, clBlack,                  'clBlack');
 SetConst(ee, clMaroon,                 'clMaroon');
 SetConst(ee, clGreen,                  'clGreen');
 SetConst(ee, clOlive,                  'clOlive');
 SetConst(ee, clNavy,                   'clNavy');
 SetConst(ee, clPurple,                 'clPurple');
 SetConst(ee, clTeal,                   'clTeal');
 SetConst(ee, clGray,                   'clGray');
 SetConst(ee, clSilver,                 'clSilver');
 SetConst(ee, clRed,                    'clRed');
 SetConst(ee, clLime,                   'clLime');
 SetConst(ee, clYellow,                 'clYellow');
 SetConst(ee, clBlue,                   'clBlue');
 SetConst(ee, clFuchsia,                'clFuchsia');
 SetConst(ee, clAqua,                   'clAqua');
 SetConst(ee, clLtGray,                 'clLtGray');
 SetConst(ee, clDkGray,                 'clDkGray');
 SetConst(ee, clWhite,                  'clWhite');
 SetConst(ee, clNone,                   'clNone');
 SetConst(ee, clDefault,                'clDefault');
 SetConst(ee, Ord(psSolid),             'psSolid');
 SetConst(ee, Ord(psDash),              'psDash');
 SetConst(ee, Ord(psDot),               'psDot');
 SetConst(ee, Ord(psDashDot),           'psDashDot');
 SetConst(ee, Ord(psDashDotDot),        'psDashDotDot');
 SetConst(ee, Ord(psClear),             'psClear');
 SetConst(ee, Ord(psInsideFrame),       'psInsideFrame');
 SetConst(ee, Ord(pmBlack),             'pmBlack');
 SetConst(ee, Ord(pmWhite),             'pmWhite');
 SetConst(ee, Ord(pmNop),               'pmNop');
 SetConst(ee, Ord(pmNot),               'pmNot');
 SetConst(ee, Ord(pmCopy),              'pmCopy');
 SetConst(ee, Ord(pmNotCopy),           'pmNotCopy');
 SetConst(ee, Ord(pmMergePenNot),       'pmMergePenNot');
 SetConst(ee, Ord(pmMaskPenNot),        'pmMaskPenNot');
 SetConst(ee, Ord(pmMergeNotPen),       'pmMergeNotPen');
 SetConst(ee, Ord(pmMaskNotPen),        'pmMaskNotPen');
 SetConst(ee, Ord(pmMerge),             'pmMerge');
 SetConst(ee, Ord(pmNotMerge),          'pmNotMerge');
 SetConst(ee, Ord(pmMask),              'pmMask');
 SetConst(ee, Ord(pmNotMask),           'pmNotMask');
 SetConst(ee, Ord(pmXor),               'pmXor');
 SetConst(ee, Ord(pmNotXor),            'pmNotXor');
 SetConst(ee, Ord(bsSolid),             'bsSolid');
 SetConst(ee, Ord(bsClear),             'bsClear');
 SetConst(ee, Ord(bsHorizontal),        'bsHorizontal');
 SetConst(ee, Ord(bsVertical),          'bsVertical');
 SetConst(ee, Ord(bsFDiagonal),         'bsFDiagonal');
 SetConst(ee, Ord(bsBDiagonal),         'bsBDiagonal');
 SetConst(ee, Ord(bsCross),             'bsCross');
 SetConst(ee, Ord(bsDiagCross),         'bsDiagCross');
 SetConst(ee, Ord(ANSI_CHARSET),        'ANSI_CHARSET');
 SetConst(ee, Ord(DEFAULT_CHARSET),     'DEFAULT_CHARSET');
 SetConst(ee, Ord(SYMBOL_CHARSET),      'SYMBOL_CHARSET');
 SetConst(ee, Ord(MAC_CHARSET),         'MAC_CHARSET');
 SetConst(ee, Ord(SHIFTJIS_CHARSET),    'SHIFTJIS_CHARSET');
 SetConst(ee, Ord(HANGEUL_CHARSET),     'HANGEUL_CHARSET');
 SetConst(ee, Ord(JOHAB_CHARSET),       'JOHAB_CHARSET');
 SetConst(ee, Ord(GB2312_CHARSET),      'GB2312_CHARSET');
 SetConst(ee, Ord(CHINESEBIG5_CHARSET), 'CHINESEBIG5_CHARSET');
 SetConst(ee, Ord(GREEK_CHARSET),       'GREEK_CHARSET');
 SetConst(ee, Ord(TURKISH_CHARSET),     'TURKISH_CHARSET');
 SetConst(ee, Ord(HEBREW_CHARSET),      'HEBREW_CHARSET');
 SetConst(ee, Ord(ARABIC_CHARSET),      'ARABIC_CHARSET');
 SetConst(ee, Ord(BALTIC_CHARSET),      'BALTIC_CHARSET');
 SetConst(ee, Ord(RUSSIAN_CHARSET),     'RUSSIAN_CHARSET');
 SetConst(ee, Ord(THAI_CHARSET),        'THAI_CHARSET');
 SetConst(ee, Ord(EASTEUROPE_CHARSET),  'EASTEUROPE_CHARSET');
 SetConst(ee, Ord(OEM_CHARSET),         'OEM_CHARSET');
 SetConst(ee, Ord(DEFAULT_PITCH),       'DEFAULT_PITCH');
 SetConst(ee, Ord(FIXED_PITCH),         'FIXED_PITCH');
 SetConst(ee, Ord(VARIABLE_PITCH),      'VARIABLE_PITCH');
 SetConst(ee, Ord(fpDefault),           'fpDefault');
 SetConst(ee, Ord(fpVariable),          'fpVariable');
 SetConst(ee, Ord(fpFixed),             'fpFixed');
 SetConst(ee, Ord(fsBold),              'fsBold');
 SetConst(ee, Ord(fsItalic),            'fsItalic');
 SetConst(ee, Ord(fsUnderline),         'fsUnderline');
 SetConst(ee, Ord(fsStrikeOut),         'fsStrikeOut');
 SetConst(ee, 0,                        'plNone');
 SetConst(ee, 1,                        'plClear');
 SetConst(ee, 2,                        'plAddPoint');
 SetConst(ee, 3,                        'plPolyLine');
 SetConst(ee, 4,                        'plPolygon');
 SetConst(ee, 5,                        'plPolyBezier');
 ee.SetFunc('sensorwidth',    0, _sensorwidth,    RusEng('ширина сенсора',                   'sensor width'));
 ee.SetFunc('sensorheight',   0, _sensorheight,   RusEng('высота сенсора',                   'sensor height'));
 ee.SetFunc('linkedvalue',    0, _linkedvalue,    RusEng('данные сенсора (v)',               'sensor linked value (v)'));
 ee.SetFunc('linkedcurve',    0, _linkedcurve,    RusEng('ссылка на кривую',                 'sensor linked curve reference'));
 ee.SetFunc('linkedtag',      0, _linkedtag,      RusEng('ссылка на тэг',                    'sensor linked tag reference'));
 ee.SetFunc('linkedtagcolor', 0, _linkedtagcolor, RusEng('GetTagColor(LinkedTag())',         'GetTagColor(LinkedTag())'));
 ee.SetFunc('linkedtagparam', 0, _linkedtagparam, RusEng('GetTagParam(LinkedTag())',         'GetTagParam(LinkedTag())'));
 ee.SetFunc('linkedtagtimer', 0, _linkedtagtimer, RusEng('GetTagTimer(LinkedTag())',         'GetTagTimer(LinkedTag())'));
 ee.SetFunc('bookmark',       0, _bookmark,       RusEng('идентификатор картинки',           'sensor bitmap tag (bookmark)'));
 ee.SetFunc('ledwidth',       0, _ledwidth,       RusEng('ширина поля LED',                  'sensor LED Width'));
 ee.SetFunc('leddigit',       0, _leddigit,       RusEng('точность поля LED',                'sensor LED Digits after dot'));
 ee.SetFunc('moveled',        2, _moveled,        RusEng('сдвиг надписи LED',                'sensor LED move to (a,b) px'));
 ee.SetFunc('setcursor',      2, _setcursor,      RusEng('задает курсор текста',             'set cursor (a,b), i.e. text position'));
 ee.SetFunc('setpen',         4, _setpen,         RusEng('задает стиль линий',               'set pen, i.e. line color,style,mode,width'));
 ee.SetFunc('setbrush',       2, _setbrush,       RusEng('задает стиль заполнения',          'set brush, i.e. fill color,style'));
 ee.SetFunc('drawpoint',      4, _drawpoint,      RusEng('рисует точку (маркер)',            'draw point (marker)'));
 ee.SetFunc('drawline',       4, _drawline,       RusEng('рисует отрезок прямой',            'draw line'));
 ee.SetFunc('drawarrow',      6, _drawarrow,      RusEng('рисует прямую стрелку',            'draw arrow line'));
 ee.SetFunc('drawrect',       4, _drawrect,       RusEng('рисует прямоугольник',             'draw bar, i.e. filled rectangle'));
 ee.SetFunc('drawbox',        4, _drawrect,       RusEng('рисует прямоугольник',             'draw bar, i.e. filled rectangle'));
 ee.SetFunc('drawbar',        4, _drawrect,       RusEng('рисует прямоугольник',             'draw bar, i.e. filled rectangle'));
 ee.SetFunc('drawroundrect',  6, _drawroundrect,  RusEng('рисует скругленный прямоугольник', 'draw rounded rectangle'));
 ee.SetFunc('drawroundbox',   6, _drawroundrect,  RusEng('рисует скругленный прямоугольник', 'draw rounded rectangle'));
 ee.SetFunc('drawroundbar',   6, _drawroundrect,  RusEng('рисует скругленный прямоугольник', 'draw rounded rectangle'));
 ee.SetFunc('drawellipse',    4, _drawellipse,    RusEng('рисует эллипс',                    'draw filled ellipse'));
 ee.SetFunc('drawpie',        8, _drawpie,        RusEng('рисует сегмент эллипса',           'draw filled ellipse pie'));
 ee.SetFunc('drawarc',        8, _drawarc,        RusEng('рисует дугу эллипса',              'draw ellipse arc'));
 ee.SetFunc('drawchord',      8, _drawchord,      RusEng('рисует хорду эллипса',             'draw filled ellipse chord'));
 ee.SetFunc('drawpoly',       3, _drawpoly,       RusEng('рисует полигональную фигуру',      'draw polygonal figure'));
 ee.SetFunc('setsimulation',  1, _setsimulation,  RusEng('включает режим симуляции',         'simulation mode a=0/1=normal/simulation'));
 ee.SetFunc('setdebugflags',  1, _setdebugflags,  RusEng('задает флаги отладки DebugFlags',  'set DebugFlags for debugging'));
 ee.SetAction('print',        act_print,          RusEng('рисует текст',                     'print text at cursor'));
 ee.SetAction('textwidth',    act_textwidth,      RusEng('ширина текста в пикселях',         'text width in pixels'));
 ee.SetAction('textheight',   act_textheight,     RusEng('высота текста в пикселях',         'text height in pixels'));
 ee.SetAction('font',         act_font,           RusEng('уставка параметров фонта',         'font parameters preset'));
end;

 {
 *******************************************************************************
 TFormCircuitWindow implementation
 *******************************************************************************
 }
function TFormCircuitWindow.GetMainBitmap:TSmartBitmap;
begin
 if Assigned(Self) then Result:=myMainBitmap else Result:=nil;
end;

function TFormCircuitWindow.GetCount:Integer;
begin
 if Assigned(Self) then Result:=myItems.Count else Result:=0;
end;

function TFormCircuitWindow.GetItems(i:Integer):TCircuitSensor;
begin
 if Assigned(Self) then Result:=TCircuitSensor(myItems[i]) else Result:=nil;
end;

function TFormCircuitWindow.GetSourceCrcFile:LongString;
begin
 if Ok then Result:=mySourceCrcFile else Result:='';
end;

function TFormCircuitWindow.GetSourceCfgFile:LongString;
begin
 if Ok then Result:=mySourceCfgFile else Result:='';
end;

function TFormCircuitWindow.GetSourceSection:LongString;
begin
 if Ok then Result:=mySourceSection else Result:='';
end;

function TFormCircuitWindow.GetSourceName:LongString;
begin
 if Ok then Result:=mySourceName else Result:='';
end;

function TFormCircuitWindow.GetHasStrategy:Boolean;
begin
 if Assigned(Self) then Result:=Assigned(myStrategy) else Result:=false;
end;

procedure TFormCircuitWindow.SetStrategy(aStrategy:TCircuitStrategy);
begin
 if Assigned(Self) then myStrategy:=aStrategy;
end;

function TFormCircuitWindow.GetHasMonitor:Boolean;
begin
 if Assigned(Self) then Result:=Assigned(myMonitor) else Result:=false;
end;

procedure TFormCircuitWindow.SetMonitor(aMonitor:TCircuitMonitor);
begin
 if Assigned(Self) then myMonitor:=aMonitor;
end;

function TFormCircuitWindow.GetHintText:LongString;
begin
 if Assigned(Self) then Result:=myHintText else Result:='';
end;

procedure TFormCircuitWindow.SetHintText(const aHintText:LongString);
begin
 if Assigned(Self) then begin
  myHintText:=aHintText;
  myHintLine:=aHintText;
  if PosEol(myHintLine)>0
  then myHintLine:=Copy(myHintLine,1,PosEol(myHintLine)-1);
 end;
end;

function TFormCircuitWindow.GetHintLine:LongString;
begin
 if Assigned(Self) then Result:=myHintLine else Result:='';
end;

function TFormCircuitWindow.GetToolBarHeight:Integer;
begin
 if Assigned(Self)
 then if Assigned(ToolBar)
      then Result:=ToolBar.Height
      else Result:=0
 else Result:=0;
end;

function TFormCircuitWindow.GetPaintable:Boolean;
begin
 if Assigned(Self) then Result:=(WindowState<>wsMinimized) and not mySimulation else Result:=false;
 if Result then Inc(CircuitWindowsProfiler.Curr.PainterApiDraw);
end;

function TFormCircuitWindow.GetSimulation:Boolean;
begin
 if Assigned(Self) then Result:=mySimulation else Result:=false;
end;

procedure TFormCircuitWindow.SetSimulation(aSimulation:Boolean);
begin
 if Assigned(Self) then mySimulation:=aSimulation;
end;

function TFormCircuitWindow.GetDebugFlags:Integer;
begin
 if Assigned(Self) then Result:=myDebugFlags else Result:=0;
end;

procedure TFormCircuitWindow.SetDebugFlags(aDebugFlags:Integer);
begin
 if Assigned(Self) then myDebugFlags:=aDebugFlags;
end;

procedure DoUpdateToolBarButtons(Component:TComponent; Index:Integer;  var Terminate:Boolean; Custom:Pointer);
begin
 if Component is TPanel then
 with Component as TPanel do
 if Parent=Custom then begin
  if IsEmptyStr(Caption) then Width:=TToolBar(Custom).ButtonWidth;
  Height:=TToolBar(Custom).ButtonHeight;
 end;
end;

function TFormCircuitWindow.HtZoom(h:Integer):Integer;
const k=1.0; var dih,dif:Integer;
begin
 dih:=(myDefToolBtnHt div 8)*8;
 if (dih in [16,24,32,48]) then begin
  dif:=(myDefToolBarHt-dih);
  if h in [1..9]
  then Result:=Round(dih*(1+(h-1)*k))+dif
  else Result:=h;
  if (Result>=myDefToolBarHt) then Exit;
 end;
 if h in [1..9]
 then Result:=Round(myDefToolBarHt*(1+(h-1)*k))
 else Result:=h;
end;

procedure TFormCircuitWindow.SetToolBarHeight(aToolBarHeight:Integer);
var ToolBar_Height,hMin,hMax:Integer;
begin
 if Assigned(Self) then
 if Assigned(ToolBar) then
 try
  hMin:=myDefToolBarHt;
  hMax:=myDefToolBarHt*9;
  ToolBar_Height:=HtZoom(aToolBarHeight);
  ToolBar.Height:=EnsureRange(ToolBar_Height,hMin,hMax);
  ToolBar.ButtonHeight:=ToolBar.Height-myDefToolBarHt+myDefToolBtnHt;
  ToolBar.ButtonWidth:=ToolBar.ButtonHeight+myDefToolBtnWd-myDefToolBtnHt;
  ForEachComponent(Self,DoUpdateToolBarButtons,ToolBar);
  ToolBar.Visible:=aToolBarHeight>0;
  UpdateScrollBars;
 except
  on E:Exception do BugReport(E,Self,'SetToolBarHeight');
 end;
end;

function TFormCircuitWindow.GetScrollBarShowX:Boolean;
begin
 if Assigned(Self) then Result:=ScrollBarX.Visible else Result:=False;
end;

procedure TFormCircuitWindow.SetScrollBarShowX(aShow:Boolean);
begin
 if Assigned(Self) then begin
  ActionViewScrollBarShowX.Checked:=aShow;
  ScrollBarX.Visible:=aShow;
 end;
end;

function TFormCircuitWindow.GetScrollBarShowY:Boolean;
begin
 if Assigned(Self) then Result:=ScrollBarY.Visible else Result:=False;
end;

procedure TFormCircuitWindow.SetScrollBarShowY(aShow:Boolean);
begin
 if Assigned(Self) then begin
  ActionViewScrollBarShowY.Checked:=aShow;
  ScrollBarY.Visible:=aShow;
 end;
end;

function TFormCircuitWindow.GetEvaluator:TExpressionEvaluator;
begin
 if Assigned(Self) then Result:=myEvaluator else Result:=nil;
end;

function TFormCircuitWindow.GetStartupScript:LongString;
begin
 if Assigned(Self) then Result:=myStartupScript else Result:='';
end;

procedure TFormCircuitWindow.SetStartupScript(const aStartupScript:LongString);
begin
 if Assigned(Self) then myStartupScript:=aStartupScript;
end;

function TFormCircuitWindow.ImageToPaintBoxClient(const Pos:TPoint2I):TPoint2I;
begin
 Result:=Pos;
 if Assigned(Self) then PointMove(Result, -ScrollBarX.Position, -ScrollBarY.Position);
end;

function TFormCircuitWindow.PaintBoxClientToImage(const Pos:TPoint2I):TPoint2I;
begin
 Result:=Pos;
 if Assigned(Self) then PointMove(Result, +ScrollBarX.Position, +ScrollBarY.Position);
end;

function CompareNames(const Name1,Name2:LongString):Integer;
begin
 Result:=Sign(CompareText(Name1,Name2));
 Exit; // Skip obsolete version
 if Name1<Name2 then Result:=-1 else
 if Name1>Name2 then Result:=+1 else Result:=0;
end;

function TFormCircuitWindow.Search(const aName:LongString; out aIndex:Integer):Boolean;
var Left,Right,Middle,Comparison:Integer;
begin
 Result:=False;
 Left:=0;
 if Assigned(Self) then begin
  Right:=Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) shr 1;
   Comparison:=CompareNames(UnifyAlias(Items[Middle].Name),UnifyAlias(aName));
   if Comparison<0 then Left:=Middle+1 else begin
    Right:=Middle-1;
    if Comparison=0 then begin
     Result:=True;
     Left:=Middle;
    end;
   end;
  end;
 end;
 aIndex:=Left;
end;

function TFormCircuitWindow.SensorByName(const aName:LongString):TCircuitSensor;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then Result:=Items[aIndex] else Result:=nil;
end;

procedure TFormCircuitWindow.AddSensor(Sensor:TCircuitSensor);
var aIndex:Integer;
begin
 if Ok and (Sensor is TCircuitSensor) then begin
  try
   LockDraw;
   if Search(Sensor.Name,aIndex)
   then myItems[aIndex]:=Sensor
   else myItems.Insert(aIndex,Sensor);
   if Sensor.ToolBarKey<>0 then AddToolBarSensorButton(Sensor);
   Sensor.ParentForm:=Self;
  finally
   UnlockDraw;
  end;
 end;
end;

function TFormCircuitWindow.SensorByPos(const Pos:TPoint2I):TCircuitSensor;
var i:Integer;
begin
 Result:=nil;
 for i:=0 to Count-1 do
 if RectContainsPoint(Items[i].Bounds,Pos) then begin
  Result:=Items[i];
  break;
 end;
end;

function TFormCircuitWindow.SensorByShortCut(ShortCut:TShortCut):TCircuitSensor;
var i:Integer;
begin
 Result:=nil;
 if ShortCut<>0 then
 for i:=0 to Count-1 do
 if Items[i].ShortCut=ShortCut then begin
  Result:=Items[i];
  break;
 end;
end;

procedure TFormCircuitWindow.UpdateScrollBars;
const Inner = 2;
begin
 if Ok then
 try
  ScrollBarX.Min:=0;
  ScrollBarX.Max:=Max(0,MainBitmap.Width-PaintBox.Width+Inner);
  ScrollBarX.SmallChange:=1;
  ScrollBarX.LargeChange:=(ScrollBarX.Max-ScrollBarX.Min) div
                       Max(1,MainBitmap.Width div Max(1,3*PaintBox.Width div 8));
  ScrollBarX.PageSize:=(ScrollBarX.Max-ScrollBarX.Min) div
                       Max(1,MainBitmap.Width div Max(1,3*PaintBox.Width div 4));
  ScrollBarY.Min:=0;
  ScrollBarY.Max:=Max(0,MainBitmap.Height-PaintBox.Height+Inner);
  ScrollBarY.SmallChange:=1;
  ScrollBarY.LargeChange:=(ScrollBarY.Max-ScrollBarY.Min) div
                       Max(1,MainBitmap.Height div Max(1,3*PaintBox.Height div 8));
  ScrollBarY.PageSize:=(ScrollBarY.Max-ScrollBarY.Min) div
                       Max(1,MainBitmap.Height div Max(1,3*PaintBox.Height div 4));
 except
  on E:Exception do BugReport(E,Self,'UpdateScrollBars');
 end;
end;

procedure TFormCircuitWindow.UpdateStatus(const Mouse:TPoint2I);
var Sensor:TCircuitSensor; cr:TCursor;
begin
 if Ok then
 try
  Sensor:=SensorByPos(PaintBoxClientToImage(Mouse));
  if Sensor.Ok then begin
   cr:=crHandPoint;
   if PaintBox.Cursor<>cr then PaintBox.Cursor:=cr;
   if Length(Sensor.HintLine)>0
   then StatusBar.SimpleText:=' '+Sensor.HintLine
   else StatusBar.SimpleText:=' Sensor='+Sensor.Name;
  end else begin
   if SdiMan.IsSdiMode
   then cr:=SdiMan.CursorActivateMainSdiForm
   else cr:=crArrow;
   if PaintBox.Cursor<>cr then PaintBox.Cursor:=cr;
   if Length(HintLine)>0
   then StatusBar.SimpleText:=' '+HintLine
   else StatusBar.SimpleText:=' ';
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateStatus');
 end;
end;

procedure CheckToolBarButtons(aComponent:TComponent; aSensor:TCircuitSensor);
var i,j,k:Integer;
begin
 if Assigned(aComponent) and Assigned(aSensor) then
 for i:=aSensor.myToolBarItems.Count-1 downto 0 do begin
  k:=-1;
  for j:=aComponent.ComponentCount-1 downto 0 do
  if aComponent.Components[j]=aSensor.myToolBarItems[i] then begin k:=j; Break; end;
  if k<0 then aSensor.myToolBarItems.Delete(i);
 end;
end;

procedure FixPixelFormat(Picture:TPicture);
begin
 if Assigned(Picture) then begin
  {$IFnDEF FPC}
  // In FPC PixelFormat resets bitmap, so don`t use it
  if not (Picture.Bitmap.PixelFormat in [pf4Bit,pf8bit])
  then Picture.Bitmap.PixelFormat:=pf8bit;
  {$ENDIF}
 end;
end;

procedure DoDrawToolBarButton(Index:LongInt; const aObject:TObject; var Terminate:Boolean; Custom:Pointer);
var Bitmap:TBitmap; aColor:TColor;
begin
 Bitmap:=nil;
 if aObject is TImage then with aObject as TImage do
 if (Tag<>0) and (Tag=TCircuitSensor(Custom).Ref) then
 try
  Bitmap:=TCircuitSensor(Custom).Glyph.CreateBitmap;
  aColor:=Bitmap.TransparentColor;
  TCircuitSensor(Custom).Draw(Bitmap.Canvas,Point2I(0,0));
  if Bitmap.Transparent then Bitmap.TransparentColor:=aColor;
  Picture.Bitmap.Assign(Bitmap);
  FixPixelFormat(Picture);
 finally
  Kill(Bitmap);
 end;
end;

procedure TFormCircuitWindow.DrawSensor(Sensor:TCircuitSensor);
begin
 if Ok and Sensor.Ok then
 try
  Sensor.Draw(PaintBox.Canvas,ImageToPaintBoxClient(Sensor.Pos));
  if Sensor.ToolBarKey<>0 then begin
   if SafeToolBarSensors then CheckToolBarButtons(Self,Sensor);
   Sensor.myToolBarItems.ForEach(DoDrawToolBarButton,Sensor);
  end;
 except
  on E:Exception do BugReport(E,Self,'DrawSensor');
 end;
end;

procedure TFormCircuitWindow.UpdateCommands;
var Exposed,RootLevel:Boolean;
begin
 inherited UpdateCommands;
 Exposed:=FormIsExposed(Self);
 RootLevel:=(Guard.Level=ga_Root);
 SetEnabledActions(false,[ActionFileSave]);
 SetEnabledActions(Exposed, [ActionFileSaveAs,
                             ActionFilePrint]);
 SetEnabledActions(Exposed and RootLevel,[ActionViewPainterDebugTools]);
end;

procedure TFormCircuitWindow.FileSaveAs;
var Ext,FName:LongString;
begin
 if Ok then
 try
  SaveDialog.Title:=RusEng('Файл\Сохранить','File\Save');
  SaveDialog.Filter:=RusEng('Мнемосхемы         (*.crc)|*.crc|'+
                            'Все остальные файлы  (*.*)|*.*|',
                            'Circuit files (*.crc)|*.crc|'+
                            'All other files (*.*)|*.*|');
  SaveDialog.FileName:=SourceCrcFile;
  if IsEmptyStr(SaveDialog.FileName)
  then OpenDialogSelectType(SaveDialog,'*.crc')
  else OpenDialogSelectType(SaveDialog,SaveDialog.FileName);
  if GuardOpenDialog(SaveDialog).Execute then begin
   FName:=UnifyFileAlias(SaveDialog.FileName);
   Ext:=UnifyAlias(SaveDialog.DefaultExt);
   if IsEmptyStr(Ext) or IsWildCard(Ext)
   then FName:=UnifyFileAlias(FName,ua_FileLow)
   else FName:=UnifyFileAlias(ForceExtension(FName,Ext),ua_FileLow);
   SaveToCrcFile(FName);
  end;
 except
  on E:Exception do BugReport(E,Self,'FileSaveAs');
 end;
end;

procedure TFormCircuitWindow.Reload(const Sensor:LongString='All');
var
 i : Integer;
begin
 if Ok then
 try
  LockDraw;
  try
   for i:=0 to Count-1 do
   if IsSameText(Sensor,'All') or IsSameText(Sensor,Items[i].Name) then Items[i].Reload;
  finally
   UnlockDraw;
  end;
 except
  on E:Exception do BugReport(E,Self,'Reload');
 end;
end;

procedure TFormCircuitWindow.DrawView;
var R:TRect2I; i:Integer;
begin
 if Ok then
 if IsFormViewable then
 try
  DebugLogReport_DrawView;
  Inc(CircuitWindowsProfiler.Curr.DrawView);
  with ImageToPaintBoxClient(Point2I(0,0)) do MainBitmap.Draw(PaintBox.Canvas, X, Y);
  with ImageToPaintBoxClient(Point2I(MainBitmap.Width,MainBitmap.Height)) do begin
   R:=Rect2I(X,0,PaintBox.Width,PaintBox.Height);
   if not RectIsEmpty(R) then DrawBar(PaintBox.Canvas,R,PaintBox.Color);
   R:=Rect2I(0,Y,X,PaintBox.Height);
   if not RectIsEmpty(R) then DrawBar(PaintBox.Canvas,R,PaintBox.Color);
  end;
  for i:=0 to Count-1 do DrawSensor(Items[i]);
 except
  on E:Exception do BugReport(E,Self,'DrawView');
 end;
end;

procedure TFormCircuitWindow.PrintView;
var Bmp:TBitmap; Key:Integer; mm:TMainMenu; tb,sb,sx,sy:Boolean;
var Params:LongString; fsp:TPoint;
begin
 fsp:=PaintBox.ClientToScreen(Point(0,0));
 Params:='@set Panel.Font   Name:PT_Mono\Size:12\Color:Navy\Style:[Bold]'+EOL
        +'@set ListBox.Font Name:PT_Mono\Size:12\Color:Black\Style:[Bold]'+EOL
        +'@set Form.Left '+IntToStr(fsp.x)+' relative Screen'+EOL
        +'@set Form.Top  '+IntToStr(fsp.y)+' relative Screen'+EOL;
 Key:=ListBoxMenu(RusEng('Файл\Печать','File\Print'),
                  RusEng('Как печатать','How to print'),
                  RusEng('Скопировать изображение в Буфер Обмена (цвет.)'+EOL+
                         'Скопировать изображение в Буфер Обмена (серый)'+EOL+
                         'Напечатать на Принтере '+Printer.PrinterName,
                         'Copy bitmap to Clipboard (color)'+EOL+
                         'Copy bitmap to Clipboard (gray)'+EOL+
                         'Print hardcopy on Printer '+Printer.PrinterName),
                         myPrintViewKey,Params);
 if (Key>=0) then
 try
  myPrintViewKey:=Key;
  tb:=ToolBar.Visible;
  sb:=StatusBar.Visible;
  sx:=ScrollBarX.Visible;
  sy:=ScrollBarY.Visible;
  mm:=Menu;
  try
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=False;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarX.Visible:=False;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarY.Visible:=False;
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=nil;
   if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
   if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
   case Key of
    0,1: begin
          Bmp:=GetPrintableImage;
          if Assigned(Bmp) then
          try
           if (Key=1) then ConvertBmpToBlackAndWhite(Bmp,clWindow,clBtnFace,3);
           CopyFormBmpToClipboard(Bmp,true);
          finally
           Kill(Bmp);
          end;
         end;
    2:   if HasPrintersDialog then Print;
   end;
  finally
   if HasFlags(HidesOnPrint,hop_Menu) then Menu:=mm;
   if HasFlags(HidesOnPrint,hop_ToolBar) then ToolBar.Visible:=tb;
   if HasFlags(HidesOnPrint,hop_StatusBar) then StatusBar.Visible:=sb;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarX.Visible:=sx;
   if HasFlags(HidesOnPrint,hop_ScrollBars) then ScrollBarY.Visible:=sy;
  end;
  if HasFlags(HidesOnPrint,hop_DrawView) then DrawView;
  if HasFlags(HidesOnPrint,hop_ProcMess) then SafeApplicationProcessMessages;
 except
  on E:Exception do BugReport(E,Self,'PrintView');
 end;
end;

procedure CircuitWindowMonitor(Index:LongInt; const aObject:TObject; var Terminate:Boolean; CustomData:Pointer);
begin
 if aObject is TFormCircuitWindow then
 with aObject as TFormCircuitWindow do if MonitorEvent then Monitoring;
end;

procedure CircuitWindowsMonitoring;
begin
 CircuitWindowsMonitor.ForEach(CircuitWindowMonitor,nil);
end;

procedure TFormCircuitWindow.StartMonitoring;
begin
 if Ok and CircuitWindowsMonitor.Ok then begin
  if CircuitWindowsMonitor.IndexOf(Self) < 0 then begin
   CircuitWindowsMonitor.Add(Self);
   Tick55Actions.Add(CircuitWindowsMonitoring);
  end;
 end;
end;

procedure TFormCircuitWindow.StopMonitoring;
begin
 if Ok and CircuitWindowsMonitor.Ok then begin
  if CircuitWindowsMonitor.IndexOf(Self) >= 0 then begin
   CircuitWindowsMonitor.Remove(Self);
   if CircuitWindowsMonitor.Count=0 then
   Tick55Actions.Remove(CircuitWindowsMonitoring);
  end;
 end;
end;

procedure TFormCircuitWindow.Monitoring;
begin
 Inc(CircuitWindowsProfiler.Curr.MonitorCall);
 if HasMonitor then myMonitor(Self) else Exit;
 Inc(CircuitWindowsProfiler.Curr.MonitorDraw);
end;

procedure TFormCircuitWindow.UpdateSensor(Sensor:TCircuitSensor; NewValue:Double; NewStrValue:PChar=nil);
var
 R1,R2     : TRect2I;
 BookMark  : Integer;
 LedStr    : LongString;
 LedVal    : Double;
 Changed   : Boolean;
 Updated   : Boolean;
 LinkedTag : Integer;
 NewColor  : Integer;
 NewParam  : Double;
 NewTimer  : Double;
begin
 if Ok and Sensor.Ok then
 try
  Inc(CircuitWindowsProfiler.Curr.PollSensor);
  Updated:=false;
  Changed:=(Sensor.LinkedValue<>NewValue);
  if Changed then Sensor.LinkedValue:=NewValue;
  LinkedTag:=Sensor.LinkedTag;
  if (LinkedTag<>0) then begin
   if Sensor.UsesLinkedTagColor then begin
    NewColor:=GetTagColor(LinkedTag);
    if (Sensor.LinkedTagColor<>NewColor) then begin
     Sensor.LinkedTagColor:=NewColor;
     Changed:=true;
    end;
   end;
   if Sensor.UsesLinkedTagParam then begin
    NewParam:=GetTagParam(LinkedTag);
    if (Sensor.LinkedTagParam<>NewParam) then begin
     Sensor.LinkedTagParam:=NewParam;
     Changed:=true;
    end;
   end;
   if Sensor.UsesLinkedTagTimer then begin
    NewTimer:=GetTagTimer(LinkedTag);
    if (Sensor.LinkedTagTimer<>NewTimer) then begin
     Sensor.LinkedTagTimer:=NewTimer;
     Changed:=true;
    end;
   end;
  end;
  // Evaluate BookMark - new value of picture Tag
  if Sensor.Count=1 then BookMark:=Sensor.Bookmark else
  if Sensor.HasTagEval then begin
   Inc(CircuitWindowsProfiler.Curr.TagEvalCall);
   if not Evaluator.SetValue('v',NewValue) then begin
    Inc(TCircuitSensorTagEvalErrorCount);
    BookMark:=Sensor.Bookmark;
    if (Self.DebugFlags and (df_TagEvalBugEcho or df_TagEvalBugFile) <> 0)
    then PrintPainterError(Self.DebugFlags,FormatPainterError(Self,Sensor,nil,mSecNow,'fail TagEval(v) SetValue'));
   end else
   if Evaluator.EvaluateExpression(PChar(Sensor.TagEval))<>ee_OK then begin
    Inc(TCircuitSensorTagEvalErrorCount);
    BookMark:=Sensor.Bookmark;
    if (Self.DebugFlags and (df_TagEvalBugEcho or df_TagEvalBugFile) <> 0)
    then PrintPainterError(Self.DebugFlags,FormatPainterError(Self,Sensor,Evaluator,mSecNow,'fail TagEval(v) Evaluate'));
   end else BookMark:=round(Evaluator.Answer);
  end else begin
   if Sensor.HasLedWidth
   then BookMark:=Sensor.Bookmark
   else BookMark:=round(NewValue);
  end;
  // Evaluate LedStr - new LED string
  if Sensor.HasLedWidth then begin
   if Sensor.HasLedEval then begin
    Inc(CircuitWindowsProfiler.Curr.LedEvalCall);
    if not Evaluator.SetValue('v',NewValue) then begin
     Inc(TCircuitSensorLedEvalErrorCount);
     LedVal:=_NaN;
     if (Self.DebugFlags and (df_LedEvalBugEcho or df_LedEvalBugFile) <> 0)
     then PrintPainterError(Self.DebugFlags,FormatPainterError(Self,Sensor,nil,mSecNow,'fail LedEval(v) SetValue'));
    end else
    if Evaluator.EvaluateExpression(PChar(Sensor.LedEval))<>ee_OK then begin
     Inc(TCircuitSensorLedEvalErrorCount);
     LedVal:=_NaN;
     if (Self.DebugFlags and (df_LedEvalBugEcho or df_LedEvalBugFile) <> 0)
     then PrintPainterError(Self.DebugFlags,FormatPainterError(Self,Sensor,Evaluator,mSecNow,'fail LedEval(v) Evaluate'));
    end else LedVal:=Evaluator.Answer;
   end else LedVal:=NewValue;
   if Length(Sensor.LedFormat)>0 then
   try
    if NewStrValue <> nil
    then LedStr:=_crw_str.Format(Sensor.LedFormat,[NewStrValue],True) else
    if (Sensor.LedFormatType=1)
    then LedStr:=_crw_str.Format(Sensor.LedFormat,[Round(LedVal)],True)
    else LedStr:=_crw_str.Format(Sensor.LedFormat,[LedVal],True);
   except
    on E:Exception do begin
     Sensor.LedFormat:='';
     BugReport(E,Self,'Invalid sensor format.');
    end;
   end else
   if Assigned(NewStrValue)
   then LedStr:=Format('%*s',[Sensor.LedWidth,NewStrValue])
   else LedStr:=FormatG(LedVal,Sensor.LedWidth,Sensor.LedDigit);
  end else LedStr:='';
  // Draw on Tag or LED change
  if (Sensor.BookMark<>Bookmark) then begin
   if Sensor.HasLedWidth then Sensor.LedValue:=LedStr;
   R1:=Sensor.Bounds;
   Sensor.BookMark:=BookMark;
   R2:=Sensor.Bounds;
   if RectIsEqual(R2,RectUnion(R1,R2))
   then DrawSensor(Sensor)
   else DrawView;
   Updated:=true;
  end else
  if Sensor.HasLedWidth then
  if (Sensor.LedValue<>LedStr) then begin
   Sensor.LedValue:=LedStr;
   DrawSensor(Sensor);
   Updated:=true;
  end;
  if Sensor.HasPainter then
  if (Changed and not Updated) or Sensor.PendingDraw then begin
   DrawSensor(Sensor);
  end;
 except
  on E:Exception do BugReport(E,Self,'UpdateSensor');
 end;
end;

function TFormCircuitWindow.SaveToCrcFile(FileName:LongString; RelativePath:Boolean=true):Boolean;
 function AdjustPath(const Path:LongString):LongString;
 begin
  if RelativePath
  then Result:=SmartFileRel(Path,FileName)
  else Result:=FExpand(Path);
 end;
 procedure WriteTextWithPrefix(var F:System.Text; const aPrefix:LongString; const aText:LongString);
 var List:TText; i:Integer;
 begin
  if Length(aText)>0 then begin
   List:=NewText;
   try
    List.Text:=aText;
    for i:=0 to List.Count-1 do writeln(F,aPrefix,List[i]);
   finally
    Kill(List);
   end;
  end;
 end;
var
 i     : Integer;
 j     : Integer;
 IOR   : Integer;
 F     : System.Text;
 Fonts : TStringList;
begin
 Result:=false;
 FileName:=UnifyFileAlias(FileName,ua_FileLow);
 if Ok and MainBitmap.Ok then
 try
  IOR:=IOResult;
  System.Assign(F,FileName);
  Fonts:=TStringList.Create;
  try
   Fonts.Sorted:=True;
   Fonts.Duplicates:=dupIgnore;
   System.Rewrite(F);
   writeln(F,'[Circuit]');
   writeln(F,'GeneralMap = '+AnsiQuotedIfNeed(AdjustPath(MainBitmap.Source)));
   if Length(Self.HintText)>0 then
   writeln(F,'Hint = ',AnsiQuotedIfNeed(Self.HintText));
   if ToolBarHeight<>0 then
   writeln(F,'ToolBarHeight = ',ToolBarHeight);
   if Length(Self.StartupScript)>0 then WriteTextWithPrefix(F,'StartupScript = ',Self.StartupScript);
   writeln(F,'[]');
   for i:=0 to Self.Count-1 do if Self[i].Ok then begin
    writeln(F,'[SensorList]');
    writeln(F,'Sensor = ',Self[i].Name);
    writeln(F,'[',Self[i].Name,']');
    writeln(F,'Pos = ',Self[i].Pos.X,', ',Self[i].Pos.Y);
    writeln(F,'Tag = ',Self[i].ElseMark);
    if Self[i].HasTagEval then writeln(F,'TagEval(v) = ',Self[i].TagEval);
    if Self[i].HasLedEval then writeln(F,'LedEval(v) = ',Self[i].LedEval);
    if Self[i].HasPainter then WriteTextWithPrefix(F,'Painter(v) = ',Self[i].Painter);
    write(F,'LED = ',Self[i].LedWidth,', ',Self[i].LedDigit,', ',Self[i].LedValue);
    if Length(Self[i].LedFormat)>0
    then write(F,', ',Self[i].LedFormat)
    else write(F,', *');
    if Length(Self[i].LedFontLink)>0
    then write(F,', ',Self[i].LedFontLink)
    else write(F,', *');
    writeln(F);
    for j:=0 to Self[i].Count-1 do
    if Self[i][j].Ok then begin
     write(F,'Tag#',j+1,' = ',Self[i][j].Ident,', ',AnsiQuotedIfNeed(AdjustPath(Self[i][j].Source)));
     if IsNonEmptyStr(Self[i][j].Title) then write(F,', ',AnsiQuotedIfNeed(Self[i][j].Title));
     writeln(F);
    end;
    if Length(Self[i].HintText)>0 then
    writeln(F,'Hint = ',AnsiQuotedIfNeed(Self[i].HintText));
    if Self[i].ToolBarKey<>0 then
    writeln(F,'ToolBarKey = ',Self[i].ToolBarKey);
    writeln(F,'GuardLevel = ',Guard.LevelToName(Self[i].GuardLevel));
    writeln(F,'[]');
    if Length(Self[i].LedFontLink)>0 then
    if IsSectionName(Self[i].LedFontLink) then
    if Fonts.IndexOf(Self[i].LedFontLink)<0 then begin
     writeln(F,UnifySection(Self[i].LedFontLink));
     writeln(F,Trim(Self[i].LedFontText));
     writeln(F,'[]');
     Fonts.Add(Self[i].LedFontLink);
    end;
   end;
  finally
   System.Close(F);
   Result:=(IOResult=0);
   SetInOutRes(IOR);
   Kill(Fonts);
  end;
  if Result then SendToMainConsole('@silent @integrity save.crc '+FileName+EOL);
 except
  on E:Exception do BugReport(E,Self,'SaveToCrcFile');
 end
end;

function IsSensorTranparent(aSensor:TCircuitSensor):Boolean;
begin
 if IsWidgetSetName('gtk,gtk2,gtk3')
 then Result:=not aSensor.ToolBarOpaque
 else Result:=false;
end;

function TFormCircuitWindow.AddToolBarSensorButton(aSensor:TCircuitSensor):Boolean;
const
 MaxBmpWidth  = 128;
 MaxBmpHeight = 128;
var
 Panel      : TPanel;
 Image      : TImage;
 Bitmap     : TBitmap;
 PanelName  : LongString;
 ImageName  : LongString;
 ToolButton : TToolButton;
begin
 Result:=False;
 if Assigned(Self) then
 if Assigned(ToolBar) then
 try
  Bitmap:=Nil;
  if not Assigned(aSensor) then begin
   ToolButton:=TToolButton.Create(Self);
   ToolButton.Style:=tbsSeparator;
   ToolButton.Left:=Screen.Width;
   ToolButton.Parent:=ToolBar;
   ToolButton.Width:=8;
   Result:=True;
  end else
  if aSensor.Count>0 then
  if aSensor.Glyph.Ok then
  if RectSizeX(aSensor.MaxBounds)<=MaxBmpWidth then
  if RectSizeY(aSensor.MaxBounds)<=MaxBmpHeight then begin
   PanelName:=Format('PanelSensor%d',[aSensor.Ref]);
   ImageName:=Format('ImageSensor%d',[aSensor.Ref]);
   if not Assigned(FindComponent(PanelName)) then
   if not Assigned(FindComponent(ImageName)) then
   try
    Panel:=TPanel.Create(Self);
    Panel.OnClick:=ActionViewToolBarSensor.OnExecute;
    Panel.Height:=ToolBar.ButtonHeight;
    Panel.Width:=ToolBar.ButtonWidth;
    Panel.Hint:=aSensor.HintLine;
    if aSensor.ToolBarBorder then Panel.BorderStyle:=bsSingle else Panel.BorderStyle:=bsNone;
    Panel.BorderWidth:=Max(0,Min(aSensor.ToolBarSpace,Panel.Height div 4));
    Panel.BevelInner:=bvNone;
    Panel.BevelOuter:=bvNone;
    Panel.Cursor:=crHandPoint;
    Panel.Left:=Screen.Width;
    Panel.Tag:=aSensor.Ref;
    Panel.Name:=PanelName;
    Panel.Parent:=ToolBar;
    Panel.Caption:='';
    Image:=TImage.Create(Self);
    if aSensor.myToolBarItems=nil
    then aSensor.InitToolBarItems;
    aSensor.myToolBarItems.Add(Image);
    Bitmap:=aSensor.Glyph.CreateBitmap;
    Image.Picture.Bitmap.Assign(Bitmap);
    Image.OnClick:=ActionViewToolBarSensor.OnExecute;
    FixPixelFormat(Image.Picture);
    Image.Hint:=aSensor.HintLine;
    Image.Cursor:=crHandPoint;
    Image.Transparent:=IsSensorTranparent(aSensor);
    Image.Tag:=aSensor.Ref;
    Image.Name:=ImageName;
    Image.Align:=alClient;
    Image.Stretch:=True;
    Image.Parent:=Panel;
    Result:=True;
   finally
    Kill(Bitmap);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'AddToolBarSensorButton');
 end
end;

function TFormCircuitWindow.AddToolBarActionButton(aAction:TAction):TPanel;
var
 Panel      : TPanel;
 Image      : TImage;
 PanelName  : LongString;
 ImageName  : LongString;
 ToolButton : TToolButton;
begin
 Result:=nil;
 if Assigned(Self) then
 if Assigned(ToolBar) then
 try
  if not Assigned(aAction) then begin
   ToolButton:=TToolButton.Create(Self);
   ToolButton.Style:=tbsSeparator;
   ToolButton.Left:=Screen.Width;
   ToolButton.Parent:=ToolBar;
   ToolButton.Width:=8;
   Result:=nil;
  end else
  if Assigned(ImageList) then
  if aAction.ImageIndex>=0 then
  if aAction.ImageIndex<ImageList.Count then begin
   if IsSameText(Copy(aAction.Name,1,6),'Action') then begin
    PanelName:='Panel'+Copy(aAction.Name,7,Length(aAction.Name)-6);
    ImageName:='Image'+Copy(aAction.Name,7,Length(aAction.Name)-6);
   end else begin
    PanelName:='Panel'+aAction.Name;
    ImageName:='Image'+aAction.Name;
   end;
   if not Assigned(FindComponent(PanelName)) then
   if not Assigned(FindComponent(ImageName)) then begin
    Panel:=TPanel.Create(Self);
    Panel.Height:=ToolBar.ButtonHeight;
    Panel.Width:=ToolBar.ButtonWidth;
    Panel.OnClick:=aAction.OnExecute;
    Panel.BorderStyle:=bsSingle;
    Panel.Cursor:=crHandPoint;
    Panel.Hint:=aAction.Hint;
    Panel.Left:=Screen.Width;
    Panel.Name:=PanelName;
    Panel.Parent:=ToolBar;
    Panel.Caption:='';
    Image:=TImage.Create(Self);
    ImageList.GetBitmap(aAction.ImageIndex,Image.Picture.Bitmap);
    FixPixelFormat(Image.Picture);
    Image.OnClick:=aAction.OnExecute;
    Image.Cursor:=crHandPoint;
    Image.Hint:=aAction.Hint;
    Image.Transparent:=True;
    Image.Name:=ImageName;
    Image.Align:=alClient;
    Image.Stretch:=True;
    Image.Parent:=Panel;
    Result:=Panel;
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'AddToolBarActionButton');
 end
end;

function TFormCircuitWindow.AddToolBarTextPanel(aName,aText,aFont:LongString; aWidth:Integer):TPanel;
var FP:TFontParams;
begin
 Result:=nil;
 if Assigned(Self) then
 if Assigned(ToolBar) then
 try
  aName:=Trim(aName);
  if IsLexeme(aName,lex_Name) then
  if not Assigned(FindComponent(aName)) then begin
   Result:=TPanel.Create(Self);
   Result.Height:=ToolBar.ButtonHeight;
   Result.Width:=Max(aWidth,4);
   Result.OnClick:=ActionViewToolBarSensor.OnExecute;
   Result.BorderStyle:=bsSingle;
   Result.Cursor:=crHandPoint;
   Result.Hint:='';
   Result.Left:=Screen.Width;
   Result.Name:=aName;
   Result.Parent:=ToolBar;
   Result.Caption:=aText;
   if IsNonEmptyStr(aFont) then
   if ReadBufferedFont(FP,aFont,True,StandardFont) then begin
    RestoreFont(Result.Font,FP);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'AddToolBarStaticButton');
 end
end;

procedure TFormCircuitWindow.FormCreate(Sender: TObject);
begin
 inherited;
 BmpCache.Capture;
 RestoreFont(Font,DefaultCircuitFont);
 if Assigned(Application.MainForm) then
 StatusBar.Font.Assign(Application.MainForm.Font);
 myMainBitmap:=nil;
 myItems:=NewObjectStorage(true);
 mySourceCrcFile:='';
 mySourceCfgFile:='';
 mySourceSection:='';
 mySourceName:='';
 myStrategy:=nil;
 myMonitor:=nil;
 ToolBar.Hide;
 UpdateScrollBars;
 UpdateStatus(Point2I(-1,-1));
 myHintText:='';
 myHintLine:='';
 if Assigned(ToolBar) then begin
  myDefToolBarHt:=ToolBar.Height;
  myDefToolBtnWd:=ToolBar.ButtonWidth;
  myDefToolBtnHt:=ToolBar.ButtonHeight;
 end;
 myEvaluator:=NewExpressionEvaluator;
 myEvaluator.Master:=@myEvaluator;
 myEvaluator.Parent:=Self;
 InitPainter(myEvaluator);
 myStartupScript:='';
 mySimulation:=false;
 myDebugFlags:=0;
 myToolbarCommands:=TStringList.Create;
 UpdateMenu(MenuViewScroll,
            RusEng('Сдвиг изображения','Scroll image'),
            RusEng('Функции сдвига изображения.','View scroll operations.'),
            0);
 UpdateMenu(MenuViewShowScrollLeft,
            RusEng('Показать: Сдвиг ВЛЕВО','Show: Scroll LEFT'),
            RusEng('Показать: Кнопку Сдвиг изображения ВЛЕВО.','Show: Button Scroll image LEFT.'),
            0);
 UpdateMenu(MenuViewShowScrollRight,
            RusEng('Показать: Сдвиг ВПРАВО','Show: Scroll RIGHT'),
            RusEng('Показать: Кнопку Сдвиг изображения ВПРАВО.','Show: Button Scroll image RIGHT.'),
            0);
 UpdateMenu(MenuViewShowScrollUp,
            RusEng('Показать: Сдвиг ВВЕРХ','Show: Scroll UP'),
            RusEng('Показать: Кнопку Сдвиг изображения ВВЕРХ.','Show: Button Scroll image UP.'),
            0);
 UpdateMenu(MenuViewShowScrollDown,
            RusEng('Показать: Сдвиг ВНИЗ','Show: Scroll DOWN'),
            RusEng('Показать: Кнопку Сдвиг изображения ВНИЗ.','Show: Button Scroll image DOWN.'),
            0);
 UpdateMenu(MenuViewScrollLeft,
            RusEng('ВЛЕВО','LEFT'),
            RusEng('Сдвиг изображения ВЛЕВО.','Scroll image LEFT.'),
            ShortCut(VK_LEFT,[ssShift]));
 UpdateMenu(MenuViewScrollRight,
            RusEng('ВПРАВО','RIGHT'),
            RusEng('Сдвиг изображения ВПРАВО.','Scroll image RIGHT.'),
            ShortCut(VK_RIGHT,[ssShift]));
 UpdateMenu(MenuViewScrollUp,
            RusEng('ВВЕРХ','UP'),
            RusEng('Сдвиг изображения ВВЕРХ.','Scroll image UP.'),
            ShortCut(VK_UP,[ssShift]));
 UpdateMenu(MenuViewScrollDown,
            RusEng('ВНИЗ','DOWN'),
            RusEng('Сдвиг изображения ВНИЗ.','Scroll image DOWN.'),
            ShortCut(VK_DOWN,[ssShift]));
 UpdateMenu(MenuViewScrollBarShowX,
            RusEng('Показать: == Скроллер','Show: == Scroller'),
            RusEng('Показать: Скроллер по горизонтали.','Show: horizontal Scroller.'),
            TextToShortCut('Ctrl+Alt+Z'));
 UpdateMenu(MenuViewScrollBarShowY,
            RusEng('Показать: || Скроллер','Show: || Scroller'),
            RusEng('Показать: Скроллер по вертикали.','Show: vertical Scroller.'),
            TextToShortCut('Ctrl+Alt+V'));
 UpdateMenu(MenuViewToolBar,
            RusEng('Панель команд ToolBar','ToolBar'),
            RusEng('Управление панелью команд.','View ToolBar operations.'),
            0);
 UpdateMenu(MenuViewToolBarHeight0,
            RusEng('Высота панели 0','ToolBar Height 0'),
            RusEng('Задать высоту панели команд 0.','Set ToolBar Height 0.'),
            TextToShortcut('Ctrl+Alt+0'));
 UpdateMenu(MenuViewToolBarHeight1,
            RusEng('Высота панели 1','ToolBar Height 1'),
            RusEng('Задать высоту панели команд 1.','Set ToolBar Height 1.'),
            TextToShortcut('Ctrl+Alt+1'));
 UpdateMenu(MenuViewToolBarHeight2,
            RusEng('Высота панели 2','ToolBar Height 2'),
            RusEng('Задать высоту панели команд 2.','Set ToolBar Height 2.'),
            TextToShortcut('Ctrl+Alt+2'));
 UpdateMenu(MenuViewToolBarHeight3,
            RusEng('Высота панели 3','ToolBar Height 3'),
            RusEng('Задать высоту панели команд 3.','Set ToolBar Height 3.'),
            TextToShortcut('Ctrl+Alt+3'));
 UpdateMenu(MenuViewToolBarHeight4,
            RusEng('Высота панели 4','ToolBar Height 4'),
            RusEng('Задать высоту панели команд 4.','Set ToolBar Height 4.'),
            TextToShortcut('Ctrl+Alt+4'));
 UpdateMenu(MenuViewPainterDebugTools,
            RusEng('Отладка Painter','Painter Debug'),
            RusEng('Инструменты отладки сценариев Painter.','Tools to debug Painter scripts.'),
            0);
 PanelToolbarHeader:=AddToolBarTextPanel('PanelToolbarHeader','','',0);
 PanelViewScrollLeft:=AddToolBarActionButton(ActionViewScrollLeft);
 PanelViewScrollRight:=AddToolBarActionButton(ActionViewScrollRight);
 PanelViewScrollUp:=AddToolBarActionButton(ActionViewScrollUp);
 PanelViewScrollDown:=AddToolBarActionButton(ActionViewScrollDown);
 AddToolBarActionButton(Nil);
end;

procedure TFormCircuitWindow.FormDestroy(Sender: TObject);
begin
 mySourceCrcFile:='';
 mySourceCfgFile:='';
 mySourceSection:='';
 mySourceName:='';
 myHintText:='';
 myHintLine:='';
 Kill(myMainBitmap);
 Kill(myItems);
 Kill(myEvaluator);
 myStartupScript:='';
 Kill(myToolbarCommands);
 BmpCache.Uncapture;
 inherited;
end;

procedure TFormCircuitWindow.AfterConstruction;
begin
 inherited AfterConstruction;
 FullCircuitWindowList.Add(Self);
 AddonSdiFlags(sf_SdiCircuit);
end;

procedure TFormCircuitWindow.BeforeDestruction;
var i:Integer;
begin
 StopMonitoring;
 FullCircuitWindowList.Remove(Self);
 for i:=0 to Count-1 do Items[i].ParentForm:=nil;
 myEvaluator.Parent:=nil;
 inherited BeforeDestruction;
end;


procedure TFormCircuitWindow.FormResize(Sender: TObject);
begin
 inherited;
 UpdateScrollBars;
end;

function TFormCircuitWindow.ChangePanelState(aName:LongString; how:Char):Boolean;
var c:TComponent; p:TPanel;
begin
 Result:=False;
 if Ok then
 try
  c:=FindComponent(aName);
  if (c is TPanel) then p:=TPanel(c) else p:=nil;
  if Assigned(p) then begin
   case how of
    '+': p.Visible:=True;
    '-': p.Visible:=False;
    '^': p.Visible:=not p.Visible;
   end;
   Result:=p.Visible;
  end;
 except
  on E:Exception do BugReport(E,Self,'ChangePanelState');
 end;
end;

procedure TFormCircuitWindow.ActionViewShowScrollLeftExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewShowScrollLeft)<0 then Exit;
 MenuViewShowScrollLeft.Checked:=ChangePanelState('PanelViewScrollLeft','^');
end;

procedure TFormCircuitWindow.ActionViewShowScrollRightExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewShowScrollRight)<0 then Exit;
 MenuViewShowScrollRight.Checked:=ChangePanelState('PanelViewScrollRight','^');
end;

procedure TFormCircuitWindow.ActionViewShowScrollUpExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewShowScrollUp)<0 then Exit;
 MenuViewShowScrollUp.Checked:=ChangePanelState('PanelViewScrollUp','^');
end;

procedure TFormCircuitWindow.ActionViewShowScrollDownExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewShowScrollDown)<0 then Exit;
 MenuViewShowScrollDown.Checked:=ChangePanelState('PanelViewScrollDown','^');
end;

procedure TFormCircuitWindow.PaintBoxPaint(Sender: TObject);
begin
 inherited;
 try
  LockDraw;
 finally
  UnlockDraw;
 end;
end;

procedure TFormCircuitWindow.ScrollBarXScroll(Sender: TObject; ScrollCode: TScrollCode; var ScrollPos: Integer);
begin
 inherited;
 if ScrollCode <> scTrack then begin
  try
   LockDraw;
  finally
   UnlockDraw;
  end;
 end;
end;

procedure TFormCircuitWindow.ScrollBarYScroll(Sender: TObject;  ScrollCode: TScrollCode; var ScrollPos: Integer);
begin
 inherited;
 if ScrollCode <> scTrack then begin
  try
   LockDraw;
  finally
   UnlockDraw;
  end;
 end;
end;

procedure TFormCircuitWindow.PaintBoxDblClick(Sender: TObject);
var Cursor:TPoint; Where:TPoint2I; Sensor:TCircuitSensor;
begin
 inherited;
 Cursor:=Mouse.CursorPos;
 Cursor:=PaintBox.ScreenToClient(Cursor);
 Where:=PaintBoxClientToImage(Point2I(Cursor));
 Sensor:=SensorByPos(Where);
 if (Sensor=nil) then SdiMan.ActivateMainForm;
end;

procedure TFormCircuitWindow.PaintBoxMouseDown(Sender: TObject;  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var Event:TCircuitEvent; Sensor:TCircuitSensor;
begin
 inherited;
 if HasStrategy then begin
  Event.What:=evMouseDown;
  case Button of
   mbLeft   : Event.Key:=VK_LBUTTON;
   mbRight  : Event.Key:=VK_RBUTTON;
   mbMiddle : Event.Key:=VK_MBUTTON;
   else       Event.Key:=0;
  end;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(X,Y));
  Event.Wheel:=0;
  Sensor:=SensorByPos(Event.Where);
  myStrategy(Self,Event,Sensor);
 end;
end;

procedure TFormCircuitWindow.PaintBoxMouseMove(Sender: TObject;  Shift: TShiftState; X, Y: Integer);
var Event:TCircuitEvent; Sensor:TCircuitSensor;
begin
 inherited;
 UpdateStatus(Point2I(X,Y));
 if HasStrategy then begin
  Event.What:=evMouseMove;
  Event.Key:=0;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(X,Y));
  Event.Wheel:=0;
  Sensor:=SensorByPos(Event.Where);
  myStrategy(Self,Event,Sensor);
 end;
end;

procedure TFormCircuitWindow.PaintBoxMouseUp(Sender: TObject;  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var Event:TCircuitEvent; Sensor:TCircuitSensor;
begin
 inherited;
 if HasStrategy then begin
  Event.What:=evMouseUp;
  case Button of
   mbLeft   : Event.Key:=VK_LBUTTON;
   mbRight  : Event.Key:=VK_RBUTTON;
   mbMiddle : Event.Key:=VK_MBUTTON;
   else       Event.Key:=0;
  end;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(X,Y));
  Event.Wheel:=0;
  Sensor:=SensorByPos(Event.Where);
  myStrategy(Self,Event,Sensor);
 end;
end;

procedure TFormCircuitWindow.FormKeyDown(Sender: TObject; var Key: Word;  Shift: TShiftState);
var Event:TCircuitEvent; Sensor:TCircuitSensor;
begin
 inherited;
 if HasStrategy then begin
  Event.What:=evKeyDown;
  Event.Key:=Key;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(PaintBox.ScreenToClient(Mouse.CursorPos)));
  Event.Wheel:=0;
  Sensor:=SensorByShortCut(ShortCut(Event.Key,Event.Shift));
  myStrategy(Self,Event,Sensor);
  if Sensor.Ok then Key:=0;
 end;
end;

procedure TFormCircuitWindow.FormKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
var Event:TCircuitEvent; Sensor:TCircuitSensor;
begin
 inherited;
 if HasStrategy then begin
  Event.What:=evKeyUp;
  Event.Key:=Key;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(PaintBox.ScreenToClient(Mouse.CursorPos)));
  Event.Wheel:=0;
  Sensor:=SensorByShortCut(ShortCut(Event.Key,Event.Shift));
  myStrategy(Self,Event,Sensor);
  if Sensor.Ok then Key:=0;
 end;
end;

procedure TFormCircuitWindow.FormMouseWheel(Sender: TObject;  Shift: TShiftState; WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
var Event:TCircuitEvent; Sensor:TCircuitSensor; sMousePos,cMousePos:TPoint;
begin
 sMousePos:=ClientToScreen(MousePos);
 cMousePos:=PaintBox.ScreenToClient(sMousePos);
 if IsControlContainsScreenPos(PaintBox,sMousePos) then
 if HasStrategy then begin
  Event.What:=evMouseWheel;
  Event.Key:=0;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(cMousePos));
  Event.Wheel:=WheelDelta;
  Sensor:=SensorByPos(Event.Where);
  myStrategy(Self,Event,Sensor);
  Handled:=true;
  Exit;
 end;
 inherited;
end;

procedure TFormCircuitWindow.FormMouseWheelDown(Sender: TObject; Shift: TShiftState; MousePos: TPoint; var Handled: Boolean);
var Event:TCircuitEvent; Sensor:TCircuitSensor; sMousePos,cMousePos:TPoint;
begin
 sMousePos:=ClientToScreen(MousePos);
 cMousePos:=PaintBox.ScreenToClient(sMousePos);
 if IsControlContainsScreenPos(PaintBox,sMousePos) then
 if HasStrategy then begin
  Event.What:=evMouseWheelDown;
  Event.Key:=0;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(cMousePos));
  Event.Wheel:=0;
  Sensor:=SensorByPos(Event.Where);
  myStrategy(Self,Event,Sensor);
  Handled:=true;
  Exit;
 end;
 inherited;
end;

procedure TFormCircuitWindow.FormMouseWheelUp(Sender: TObject;  Shift: TShiftState; MousePos: TPoint; var Handled: Boolean);
var Event:TCircuitEvent; Sensor:TCircuitSensor; sMousePos,cMousePos:TPoint;
begin
 sMousePos:=ClientToScreen(MousePos);
 cMousePos:=PaintBox.ScreenToClient(sMousePos);
 if IsControlContainsScreenPos(PaintBox,sMousePos) then
 if HasStrategy then begin
  Event.What:=evMouseWheelUp;
  Event.Key:=0;
  Event.Shift:=Shift;
  Event.Where:=PaintBoxClientToImage(Point2I(cMousePos));
  Event.Wheel:=0;
  Sensor:=SensorByPos(Event.Where);
  myStrategy(Self,Event,Sensor);
  Handled:=true;
  Exit;
 end;
 inherited;
end;

procedure TFormCircuitWindow.ScrollBarYKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
 inherited;
 FormKeyDown(Sender, Key, Shift);
end;

procedure TFormCircuitWindow.ScrollBarYKeyUp(Sender: TObject;  var Key: Word; Shift: TShiftState);
begin
 inherited;
 FormKeyUp(Sender, Key, Shift);
end;

procedure TFormCircuitWindow.ScrollBarXKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
 inherited;
 FormKeyDown(Sender, Key, Shift);
end;

procedure TFormCircuitWindow.ScrollBarXKeyUp(Sender: TObject;  var Key: Word; Shift: TShiftState);
begin
 inherited;
 FormKeyUp(Sender, Key, Shift);
end;

procedure ApplyScrolling(ScrollBar:TScrollBar; Factor:Integer);
begin
 if Assigned(ScrollBar) then
 try
  ScrollBar.Position:=Max(ScrollBar.Min,Min(ScrollBar.Max,
                          ScrollBar.Position+Factor*ScrollBar.PageSize));
 except
  on E:Exception do BugReport(E,nil,'ApplyScrolling');
 end;
end;

procedure TFormCircuitWindow.ActionViewScrollLeftExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewScrollLeft)<0 then Exit;
 inherited;
 try
  LockDraw;
  ApplyScrolling(ScrollBarX,-1);
 finally
  UnlockDraw;
 end;
end;

procedure TFormCircuitWindow.ActionViewScrollRightExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewScrollRight)<0 then Exit;
 inherited;
 try
  LockDraw;
  ApplyScrolling(ScrollBarX,+1);
 finally
  UnlockDraw;
 end;
end;

procedure TFormCircuitWindow.ActionViewScrollUpExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewScrollUp)<0 then Exit;
 inherited;
 try
  LockDraw;
  ApplyScrolling(ScrollBarY,-1);
 finally
  UnlockDraw;
 end;
end;

procedure TFormCircuitWindow.ActionViewScrollDownExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewScrollDown)<0 then Exit;
 inherited;
 try
  LockDraw;
  ApplyScrolling(ScrollBarY,+1);
 finally
  UnlockDraw;
 end;
end;

procedure TFormCircuitWindow.ActionViewToolBarSensorExecute(Sender: TObject);
var
 Event  : TCircuitEvent;
 Sensor : TCircuitSensor;
begin
 inherited;
 if Assigned(Sender) then
 try
  if HasStrategy then
  if Sender is TComponent then
  if (Sender as TComponent).Tag<>0 then
  if (Sender is TPanel) or (Sender is TImage) then begin
   TObject(Sensor):=ObjectRegistry[(Sender as TComponent).Tag];
   if TObject(Sensor) is TCircuitSensor then
   if myItems.IndexOf(Sensor)>=0 then
   if Sensor.ToolBarKey<>0 then begin
    if Sensor.ToolBarKey in [VK_LBUTTON,VK_RBUTTON,VK_MBUTTON]
    then Event.What:=evMouseDown else Event.What:=evKeyDown;
    Event.Key:=Sensor.ToolBarKey;
    Event.Shift:=[];
    if GetKeyState(VK_CONTROL) < 0 then Include(Event.Shift, ssCtrl);
    if GetKeyState(VK_SHIFT) < 0 then Include(Event.Shift, ssShift);
    if GetKeyState(VK_MENU) < 0 then Include(Event.Shift, ssAlt);
    Event.Where:=RectCenter(Sensor.Bounds);
    Event.Wheel:=0;
    myStrategy(Self,Event,Sensor);
    Exit;
   end;
  end;
  if Assigned(Sender) then
  if (Sender=PanelToolbarHeader)
  or (Sender=PanelToolbarLegend) then
  if Assigned(myToolbarCommands) and HasStrategy then begin
   HandlePanelToolbarCommand(PanelToolbarXCommand(Sender));
   Exit;
  end;
 except
  on E:Exception do BugReport(E,Self,'ActionViewToolBarSensorExecute');
 end;
end;

procedure TFormCircuitWindow.ActionViewToolBarHeight0Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewToolBarHeight0)<0 then Exit;
 inherited;
 ToolBarHeight:=0;
end;

procedure TFormCircuitWindow.ActionViewToolBarHeight1Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewToolBarHeight1)<0 then Exit;
 inherited;
 ToolBarHeight:=1;
end;

procedure TFormCircuitWindow.ActionViewToolBarHeight2Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewToolBarHeight2)<0 then Exit;
 inherited;
 ToolBarHeight:=2;
end;

procedure TFormCircuitWindow.ActionViewToolBarHeight3Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewToolBarHeight3)<0 then Exit;
 inherited;
 ToolBarHeight:=3;
end;

procedure TFormCircuitWindow.ActionViewToolBarHeight4Execute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewToolBarHeight4)<0 then Exit;
 inherited;
 ToolBarHeight:=4;
end;

procedure TFormCircuitWindow.ActionViewScrollBarShowXExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewScrollBarShowX)<0 then Exit;
 inherited;
 ScrollBarShowX:=not ScrollBarShowX;
end;

procedure TFormCircuitWindow.ActionViewScrollBarShowYExecute(Sender: TObject);
begin
 if Guard.CheckAction(ga_Guest,ActionViewScrollBarShowY)<0 then Exit;
 inherited;
 ScrollBarShowY:=not ScrollBarShowY;
end;

procedure TFormCircuitWindow.ActionViewPainterDebugToolsExecute(Sender: TObject);
var m:Integer;
 procedure EditDebugFlags;
 var s:LongString;
 begin
  try
   s:=ListBoxMultiSelection(Caption+' -> Painter Debug Flags',
       RusEng('Флаги отладочного вывода','Flags of Debug output'),
       RusEng('По ошибкам Startup - вывод в консоль',  'On Startup errors - print to console')+EOL+
       RusEng('По ошибкам TagEval - вывод в консоль',  'On TagEval errors - print to console')+EOL+
       RusEng('По ошибкам LedEval - вывод в консоль',  'On LedEval errors - print to console')+EOL+
       RusEng('По ошибкам Painter - вывод в консоль',  'On Painter errors - print to console')+EOL+
       RusEng('По ошибкам Startup - вывод в Temp\Debug.out','On Startup errors - print to Temp\Debug.out')+EOL+
       RusEng('По ошибкам TagEval - вывод в Temp\Debug.out','On TagEval errors - print to Temp\Debug.out')+EOL+
       RusEng('По ошибкам LedEval - вывод в Temp\Debug.out','On LedEval errors - print to Temp\Debug.out')+EOL+
       RusEng('По ошибкам Painter - вывод в Temp\Debug.out','On Painter errors - print to Temp\Debug.out')+EOL,
       false,true,0,DebugFlags);
   DebugFlags:=MultiSelectionIndexesToInt(s);
  except
   on E:Exception do BugReport(E,Self,'EditDebugFlags');
  end;
 end;
 procedure PainterDebugListView;
 var s:LongString;
 begin
  s:='@silent @run lister  Temp\debug.out';
  ReadIniFileString(SysIniFile,'[DaqSys]','PainterDebugListView%s',s,efConfigNC);
  if IsNonEmptyStr(s) then SendToMainConsole(Trim(s)+EOL);
 end;
 procedure PainterDebugTailView;
 var s:LongString;
 begin
  s:='@silent @run wintail Temp\debug.out';
  ReadIniFileString(SysIniFile,'[DaqSys]','PainterDebugTailView%s',s,efConfigNC);
  if IsNonEmptyStr(s) then SendToMainConsole(Trim(s)+EOL);
 end;
 procedure ViewPainterState;
 var p:TText;
 begin
  p:=NewText;
  try
   p.Addln('[Painter]');
   p.Addln('CircuitWindow = '+Caption);
   p.Addln('CurrentTime = '+StdDateTimeStr(mSecNow));
   p.Addln('[]');
   p.Addln('');
   p.AddLn('[Variables]'); Evaluator.VarList.GetText(p);    p.Addln('[]'); p.AddLn('');
   p.AddLn('[Constants]'); Evaluator.ConstList.GetText(p);  p.Addln('[]'); p.AddLn('');
   p.AddLn('[Functions]'); Evaluator.FuncList.GetText(p);   p.Addln('[]'); p.AddLn('');
   p.AddLn('[Actions]');   Evaluator.ActionList.GetText(p); p.Addln('[]'); p.AddLn('');
   TextEditDialog(Caption+' -> Painter State',
                  RusEng('Состояние Painter','Painter State'),
                  p);
  finally
   Kill(p);
  end;
 end;
begin
 if Guard.CheckAction(ga_Guest,ActionViewPainterDebugTools)<0 then Exit;
 inherited;
 try
  m:=ListBoxMenu(Caption+' -> Painter Debug Tools',
                 RusEng('Отладочные инструменты для сценариев Painter','Painter Debug Tools'),
                 RusEng('Задать флаги отладочного вывода (SetDebugFlags)','Set debugging output flags (SetDebugFlags)')+EOL+
                 RusEng('Просмотр текста в temp\debug.out','View Text of temp\debug.out')+EOL+
                 RusEng('Следить за файлом temp\debug.out','View Tail of temp\debug.out')+EOL+
                 RusEng('Посмотреть состояние Painter','View status of Painter')+EOL,
                 myDebugToolsMenu);
  case m of
   0: EditDebugFlags;
   1: PainterDebugListView;
   2: PainterDebugTailView;
   3: ViewPainterState;
  end;
  if m>=0 then myDebugToolsMenu:=m;
 except
  on E:Exception do BugReport(E,Self,'ActionViewPainterDebugToolsExecute');
 end;
end;

procedure TFormCircuitWindow.ApplyWinDrawOptions(aOptions:LongString);
const OptList='+ViewScrollLeft,-ViewScrollLeft,'   // 1,2
             +'+ViewScrollRight,-ViewScrollRight,' // 3,4
             +'+ViewScrollUp,-ViewScrollUp,'       // 5,6
             +'+ViewScrollDown,-ViewScrollDown';   // 7,8
var iw,nop:Integer; opt:LongString; cop:Char;
begin
 if Ok then
 try
  for iw:=1 to WordCount(aOptions,ScanSpaces) do begin
   opt:=ExtractWord(iw,aOptions,ScanSpaces);
   nop:=WordIndex(opt,OptList,ScanSpaces);
   cop:=StrFetch(opt,1);
   case nop of
    1,2: MenuViewShowScrollLeft.Checked:=ChangePanelState('PanelViewScrollLeft',cop);
    3,4: MenuViewShowScrollRight.Checked:=ChangePanelState('PanelViewScrollRight',cop);
    5,6: MenuViewShowScrollUp.Checked:=ChangePanelState('PanelViewScrollUp',cop);
    7,8: MenuViewShowScrollDown.Checked:=ChangePanelState('PanelViewScrollDown',cop);
   end;
  end;
 except
  on E:Exception do BugReport(E,Self,'ApplyWinDrawOptions');
 end;
end;

procedure TFormCircuitWindow.UpdateScrollPanels;
begin
 if Ok then begin
  MenuViewShowScrollLeft.Checked:=ChangePanelState('PanelViewScrollLeft','?');
  MenuViewShowScrollRight.Checked:=ChangePanelState('PanelViewScrollRight','?');
  MenuViewShowScrollUp.Checked:=ChangePanelState('PanelViewScrollUp','?');
  MenuViewShowScrollDown.Checked:=ChangePanelState('PanelViewScrollDown','?');
  if Assigned(PanelToolbarHeader) then begin
   if (PanelToolbarHeader.Caption='')
   then PanelToolbarHeader.Visible:=False
   else PanelToolbarHeader.Visible:=True;
  end;
  if Assigned(PanelToolbarLegend) then begin
   if (PanelToolbarLegend.Caption='')
   then PanelToolbarLegend.Visible:=False
   else PanelToolbarLegend.Visible:=True;
  end;
 end;
end;

procedure TFormCircuitWindow.HandlePanelToolbarCommand(aCommand:LongString);
var cmd:LongString;
begin
 if Ok then begin
  aCommand:=Trim(aCommand);
  if IsEmptyStr(aCommand) then Exit;
  if not Assigned(myToolbarCommands) then Exit;
  cmd:=Trim(myToolbarCommands.Values[aCommand]);
  if not IsLexeme(cmd,lex_AtCmnd) then Exit;
  SendToMainConsole(cmd+EOL);
 end;
end;

 {
 Calculate windows size by sensors.
 Argument opt contains Mode,Px,Py,Mx,My as cookies.
 Mode = 1 = Bit0 = include window decorations (caption,toolbar,etc).
 Mode = 2 = Bit1 = include Mx,My spaces over raw size to have nice view.
 Mode = 4 = Bit2 = ensure size is in Screen.WorkArea with Px,Py margins.
 Note: ignore toolbar sensors with ToolBarKey<>0 or negative Pos.
 }
function TFormCircuitWindow.FindWantedSize(const opt:LongString='Mode=7;Px=50;Py=50;Mx=8;My=8;'):TPoint2I;
var i,Mode:Integer; Item:TCircuitSensor; R:TRect2I; P,M,D:TPoint2I;
 function IsValidPos(const P:TPoint2I):Boolean;
 begin
  Result:=(P.x>=0) and (P.y>=0);
 end;
begin
 Result:=Default(TPoint2I);
 if Ok then
 try
  Mode:=StrToIntDef(CookieScan(opt,'Mode',Ord(';')),0);
  P.x:=StrToIntDef(CookieScan(opt,'Px',Ord(';')),0);
  P.y:=StrToIntDef(CookieScan(opt,'Py',Ord(';')),0);
  M.x:=StrToIntDef(CookieScan(opt,'Mx',Ord(';')),0);
  M.y:=StrToIntDef(CookieScan(opt,'My',Ord(';')),0);
  for i:=0 to Count-1 do begin
   Item:=Items[i];
   if Item.Ok then
   if IsValidPos(Item.Pos) then
   if (Item.ToolBarKey=0) then begin
    R:=Item.MaxBounds;
    Result.x:=Max(Result.x,R.b.x);
    Result.y:=Max(Result.y,R.b.y);
   end;
  end;
  if HasFlags(Mode,1) then begin
   if (myDecorSize.x<=0) or (myDecorSize.y<=0) then begin
    myDecorSize.x:=Width-PaintBox.Width;
    myDecorSize.y:=Height-PaintBox.Height;
   end;
   D.x:=myDecorSize.x;
   D.y:=myDecorSize.y;
   PointMove(Result,D.x,D.y);
  end;
  if HasFlags(Mode,2) then begin
   PointMove(Result,M.x,M.y);
  end;
  if HasFlags(Mode,4) then begin
   Result.x:=EnsureRange(Result.x,p.x,Screen.WorkAreaWidth-p.x);
   Result.y:=EnsureRange(Result.y,p.y,Screen.WorkAreaHeight-p.y);
  end;
  if (myStartSize.x<=0) or (myStartSize.y<=0) then begin
   myStartSize:=Point2I(Width,Height);
  end;
 except
  on E:Exception do BugReport(E,Self,'FindWantedSize');
 end;
end;

procedure TFormCircuitWindow.ApplyWantedSize;
var P1,P2:TPoint2I;
begin
 if Ok then
 try
  P1:=FindWantedSize;
  if P1.x>myStartSize.x then Width:=P1.x;
  if P1.y>myStartSize.y then Height:=P1.y;
  P2:=FindWantedSize('Mode=1;Px=0;Py=0;Mx=0;My=0;');
  if (P2.x<=Width) and (P2.y<=Height) then begin
   ApplyWinDrawOptions('-ViewScrollLeft,-ViewScrollRight');
   ApplyWinDrawOptions('-ViewScrollUp,-ViewScrollDown');
   ScrollBarShowX:=False;
   ScrollBarShowY:=False;
  end;
 except
  on E:Exception do BugReport(E,Self,'ApplyWantedSize');
 end;
end;

///////////////////////////////////////
// Unit initialization and finalization
///////////////////////////////////////

procedure Init_form_circuitwindow;
begin
 CircuitWindowsProfiler_Clear;
 FullCircuitWindowList:=NewCircuitWindowList(false);
 FullCircuitWindowList.Master:=@FullCircuitWindowList;
 CircuitWindowsMonitor:=NewCircuitWindowList(false);
 CircuitWindowsMonitor.Master:=@CircuitWindowsMonitor;
end;

procedure Free_form_circuitwindow;
begin
 ResourceLeakageLog(Format('%-60s = %d',['FullCircuitWindowList.Count', FullCircuitWindowList.Count]));
 ResourceLeakageLog(Format('%-60s = %d',['CircuitWindowsMonitor.Count', CircuitWindowsMonitor.Count]));
 Kill(FullCircuitWindowList);
 Kill(CircuitWindowsMonitor);
end;

initialization

 Init_form_circuitwindow;

finalization

 Free_form_circuitwindow;

end.

//////////////
// END OF FILE
//////////////


