import socket
import sys
import time
import tkinter as tk
from tkinter import ttk, scrolledtext
import threading

class ScannerApp:
    def __init__(self, root):
        self.root = root
        self.root.title("DCON TCP SCANNER")
        self.root.geometry("800x600")
        
        # Параметры подключения
        self.ip_var = tk.StringVar()
        self.port_var = tk.StringVar()
        
        # Если переданы аргументы командной строки, используем их
        if len(sys.argv) == 3:
            self.ip_var.set(sys.argv[1])
            self.port_var.set(sys.argv[2])
        
        # Создаем интерфейс
        self.create_widgets()
        
    def create_widgets(self):
        # Фрейм для ввода параметров
        input_frame = ttk.Frame(self.root)
        input_frame.pack(padx=10, pady=10, fill='x')
        
        # IP адрес
        ttk.Label(input_frame, text="IP адрес:").grid(row=0, column=0, padx=5, pady=5, sticky='e')
        ip_entry = ttk.Entry(input_frame, textvariable=self.ip_var, width=20)
        ip_entry.grid(row=0, column=1, padx=5, pady=5)
        
        # Порт
        ttk.Label(input_frame, text="Порт:").grid(row=0, column=2, padx=5, pady=5, sticky='e')
        port_entry = ttk.Entry(input_frame, textvariable=self.port_var, width=10)
        port_entry.grid(row=0, column=3, padx=5, pady=5)
        
        # Кнопка запуска
        start_button = ttk.Button(input_frame, text="Начать сканирование", command=self.start_scan)
        start_button.grid(row=0, column=4, padx=10, pady=5)
        
        # Фрейм для вывода логов
        log_frame = ttk.Frame(self.root)
        log_frame.pack(padx=10, pady=10, fill='both', expand=True)
        
        # Заголовок для логов
        ttk.Label(log_frame, text="Лог процесса:").pack(anchor='w')
        
        # Text widget для логов с прокруткой
        self.log_text = scrolledtext.ScrolledText(log_frame, height=15, width=80)
        self.log_text.pack(padx=5, pady=5, fill='both', expand=True)
        
        # Фрейм для результатов
        results_frame = ttk.Frame(self.root)
        results_frame.pack(padx=10, pady=(0, 10), fill='both', expand=True)
        
        # Заголовок для результатов
        ttk.Label(results_frame, text="Результаты:").pack(anchor='w')
        
        # Text widget для результатов с прокруткой
        self.results_text = scrolledtext.ScrolledText(results_frame, height=8, width=80)
        self.results_text.pack(padx=5, pady=5, fill='both', expand=True)
        
    def log_message(self, message):
        """Добавление сообщения в лог"""
        self.log_text.insert(tk.END, f"{message}\n")
        self.log_text.see(tk.END)
        self.root.update_idletasks()
        
    def results_message(self, message):
        """Добавление сообщения в результаты"""
        self.results_text.insert(tk.END, f"{message}\n")
        self.results_text.see(tk.END)
        self.root.update_idletasks()
        
    def calculate_crc(self, data):
        """Расчет контрольной суммы"""
        checksum = 0
        for byte in data:
            checksum += byte
        return checksum & 0xFF
        
    def start_scan(self):
        # Получаем параметры
        ip = self.ip_var.get().strip()
        port_str = self.port_var.get().strip()
        
        if not ip:
            self.log_message("Пожалуйста, введите IP адрес")
            return
            
        if not port_str:
            self.log_message("Пожалуйста, введите номер порта")
            return
            
        try:
            port = int(port_str)
        except ValueError:
            self.log_message("Некорректный номер порта")
            return
            
        # Очищаем предыдущие результаты
        self.log_text.delete(1.0, tk.END)
        self.results_text.delete(1.0, tk.END)
        
        # Запускаем сканирование в отдельном потоке
        scan_thread = threading.Thread(
            target=self.scan_addresses,
            args=(ip, port)
        )
        scan_thread.daemon = True
        scan_thread.start()
        
    def scan_addresses(self, ip, port):
        """Основная функция сканирования"""
        try:
            self.log_message(f"Подключение к {ip}:{port}")
            
            # Создаем сокет
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(0.1)  # Таймаут 0.1 секунды
            sock.connect((ip, port))
            self.log_message("Подключение успешно установлено")
            
        except Exception as e:
            self.log_message(f"Ошибка подключения: {e}")
            return
            
        try:
            # Циклическая отправка команд (0x00 - 0xFF)
            for i in range(256):
                cmd_data = f"${i:02X}M"
                
                try:
                    # Запрос с контрольной суммой
                    self.log_message(f"Опрос адреса ${i:02X}...")
                    
                    data_for_crc = cmd_data.encode('utf-8')
                    crc_value = self.calculate_crc(data_for_crc)
                    crc_hex = f"{crc_value:02X}"
                    full_cmd = cmd_data + crc_hex + "\r"
                    
                    sock.send(full_cmd.encode('utf-8'))
                    
                    # Читаем ответ (с контрольной суммой)
                    response1 = sock.recv(1024)
                    if response1:
                        resp_str1 = response1.decode('utf-8', errors='ignore').strip()
                        
                        # Убираем CRC из ответа для отображения
                        processed_response = self.remove_crc_from_response(resp_str1)
                        
                        result_msg = f"Адрес ${i:02X}: {processed_response} с контрольной суммой"
                        self.results_message(result_msg)
                        
                        # Выводим в лог полный ответ для анализа
                        self.log_message(f"Ответ от адреса ${i:02X}: {resp_str1}")
                    else:
                        self.log_message(f"Нет ответа от адреса ${i:02X}")
                    
                except socket.timeout:
                    self.log_message(f"Таймаут при работе с адресом ${i:02X}")
                except Exception as e:
                    error_msg = f"Ошибка при работе с адресом ${i:02X}: {e}"
                    self.log_message(error_msg)
                
                time.sleep(0.1)  # Задержка между командами (100 мс)
                
        except Exception as e:
            self.log_message(f"Ошибка в процессе сканирования: {e}")
        finally:
            sock.close()
            self.log_message("Сканирование завершено")
    
    def remove_crc_from_response(self, response):
        """
        Убираем CRC из ответа
        Например: "!037053D97" -> "!037053"
        Предполагаем, что CRC - это последние 2 символа
        """
        if len(response) >= 2:
            # Проверяем формат ответа и убираем последние 2 символа (CRC)
            return response[:-2]
        return response

def main():
    root = tk.Tk()
    app = ScannerApp(root)
    root.mainloop()

if __name__ == "__main__":
    main()