 {
 ****************************************************************************
 CRW32 project
 Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2001, <kouriakine@mail.ru>
 Communication ports.
 Modifications:
  20011101 - Creation
  20011209 - First tested release
  20020302 - bug fixed, I hope
  20030330 - Struggle for safety (add some try/except checks)...
             TUart.ReadConfig added.
  20040709 - Uses \\.\COMn instead of COM. Problems with COMn, n>9 fixed.
  20050221 - DcbFlags
  20061212 - EnumComPorts
  20210827 - MaxPortNum is now upto 255; Uart.Poll optimization.
 ****************************************************************************
 }

unit _uart; { Universal Asyncronouse Receiver / Transmitter }

{$I _sysdef}

interface

uses
 sysutils, windows, classes, math, _alloc, _fifo, _rtc, _str, _fio, _polling;

 {
 *******************************************************************************
 .
          (COM-)
         RS-232/422/485.
         RTS,CTS,DTR,RING & etc,  
       3- RS-232 (Rx,Tx,Gnd) 
  RS-485      .       
          .
 *******************************************************************************
  .
       .   I/O   
  ,       FIFO.  
        /,  
   FIFO     FIFO ,    .
         ,     
   ,   ,  ,      ,
     .  ,   CPU  
     0 (  Windows 2000  0).
       I/O        
  -,    . ,  Windows 2000 
     20 ms   (   ), 
         50     (50 ).
 *******************************************************************************
 .
          uart,   
    .     
  .    ,   
  uart.Polling.Enable(..).
          n   uart[n].  
     uart[n].Open(...).   uart[n].Active  true.
   -   uart[n].WriteStr(s), uart[n].ReadStr.
    RxCount,TxCount,RxSpace,TxSpace,RxTotal,TxTotal  
   ,   FIFO,     FIFO,   
       .  RxClear,TxClear
   -    .
           Host -  RS-485 
     -. RequestAvailable   
     (   ,  
     ). RaiseRequest  , 
    ,    .   
       AnswerRequest.   AnswerRequest
   rs_NoRequest -  ,   , rs_Wait - 
   , rs_Answer -  , rs_TimeOut -   ,
  rs_Error -  .
  ,       - .
  ModalAnswerRequest      
      .   CRW16.
              uart[n].Close.
     ,  . -    , uart  
       .     
  ,    .     , 
    .    uart,  
  ,           ,
   -  .      
   Win32,  ,      .
 *******************************************************************************
 .
            , 
       .     INI-:
  uart[1].OpenConfig('example.cfg','[SerialPort-COM1]')
   [SerialPort-COM1]
   Port = COM1        ;  COM-
   BaudRate = 115200  ;  , 
   Parity = NONE      ;    - NONE, ODD, EVEN , MARK, SPACE
   DataBits = 8       ;   
   StopBits = 1       ;  - 1, 1.5, 2
   XonXoff = false    ;    XON/XOFF
   BufSize = 8        ;   fifo  
        GetProperties. 
 *******************************************************************************
  .
  if uart[1].Open(9600,NOPARITY,8,ONESTOPBIT,false,4096) then begin
   Echo('Open COM1.');
   uart[1].WriteStr('Send this string to COM1.');
   Echo('Read from COM1:'+uart[1].ReadStr);
   Echo('Rx FIFO contains '+d2s(uart[1].RxCount)+' bytes.');
   Echo('Rx FIFO free space '+d2s(uart[1].RxSpace)+' bytes.');
   Echo('Received '+d2s(uart[1].RxTotal)+' bytes.');
   uart[1].Close;
  end;
 *******************************************************************************
   :
  COM2 (25pin):
  2 : TXD   Transmit Data                 
  3 : RXD   Receive Data                  
  4 : RTS   Request To Send                 
  5 : CTS   Clear To Send                  
  6 : DSR   Data Set Ready                  
  7 : GND   GrouND                        ()
  8 : RLSD  Received Line Signal Detect     
  20: DTR   Data Terminal Ready            
  22: RI    Ring Indicator
  COM1 (9pin):
  2 :RXD
  3 :TXD
  5 :GND
    2-    -
   PC    - PC     GND - GND   RXD - TXD  TXD - RXD
   PC 25 - PC 25   7  -  7    3  -  2     2  -  3
   PC  9 - PC 25   5  -  7    2  -  2     3  -  3
    PC-ADAM4520 :
   PC     - ADAM 4050   GND - GND  TXD - TXD    RXD - RXD 
   25 pin - 9 pin        7  -  5    2  -  3      3  -  2  
  (  - RTS    RTS-RTS:4-7)
 *******************************************************************************
 }

const
 MaxPortNum         = 250;       { Max number of ports, [1..255]. }
 RxBufferSize       = 1024;      { Size of buffer for Rx transfer }
 TxBufferSize       = 1024;      { Size of buffer for Tx transfer }
 RxQueueSize        = 4096;      { Size of Rx queue buffer }
 TxQueueSize        = 4096;      { Size of Tx queue buffer }
 UartFifoGrowFactor = 2;         { Grow factor for FIFO }
 UartFifoGrowLimit  = 1024*128;  { Max size of FIFO buffer }

type
 TPortNum           = 1..MaxPortNum;
 TParity            = NOPARITY..SPACEPARITY;   {NOPARITY,ODDPARITY,EVENPARITY,MARKPARITY,SPACEPARITY}
 TDataBits          = 5..8;
 TStopBits          = ONESTOPBIT..TWOSTOPBITS; {ONESTOPBIT,ONE5STOPBITS,TWOSTOPBITS}
 TUartErrors        = (ue_TOTAL,        { All errors. }
                       ue_RXOVER,       { Receive Queue overflow. }
                       ue_OVERRUN,      { Receive Overrun Error. }
                       ue_RXPARITY,     { Receive Parity Error. }
                       ue_FRAME,        { Receive Framing error. }
                       ue_BREAK,        { Break Detected. }
                       ue_TXFULL,       { TX Queue is full. }
                       ue_IOE,          { An I/O error occurred during communications with the device. }
                       ue_MODE,         { The requested mode unsupported, or file handle is invalid.}
                       ue_RxFIFO_OVER,  { Receiver fifo overflow. }
                       ue_TxFIFO_OVER,  { Transmitter fifo overflow. }
                       ue_UNEXPECTED);  { Unexpected error. }
 TRequestStatus     = (rs_NoRequest,    {       }
                       rs_WaitAnswer,   {     }
                       rs_Answer,       {      }
                       rs_TimeOut,      {       }
                       rs_Error);       { -    }
 {
 *******************************************************************************
 Warning:
  TUartPort is public component of TUart class and never should be use alone.
  Only TUart may create or destroy TUartPort object!
  TUartPort and should not be create or destroy in other way!
 *******************************************************************************
 }
type
 TUart = class;
 TUartPort = class(TLatch)
 private
  myUart      : TUart;
  myPortNum   : TPortNum;
  myPortName  : packed array[0..31] of char;
  myFileName  : packed array[0..31] of char;
  myCommFile  : THandle;
  myRxFifo    : TFifo;
  myTxFifo    : TFifo;
  myRxTotal   : Int64;
  myTxTotal   : Int64;
  myRxPending : Boolean;
  myTxPending : Boolean;
  myDCB       : TDCB;
  myCOMMPROP  : COMMPROP;
  myTIMEOUTS  : COMMTIMEOUTS;
  myRxOverlap : TOverlapped;
  myTxOverlap : TOverlapped;
  myRxBuffer  : packed array[0..RxBufferSize-1] of char;
  myTxBuffer  : packed array[0..TxBufferSize-1] of char;
  myErrors    : packed array[TUartErrors] of DWORD;
  myRequest   : packed record
   Inquiry    : ShortString;
   Answer     : ShortString;
   Timer      : TIntervalTimer;
   Delimiter  : Char;
  end;
  function    GetActive:Boolean;
  function    GetError(ErrorCode:TUartErrors):DWORD;
  function    GetRxCount:Integer;
  function    GetTxCount:Integer;
  function    GetRxSpace:Integer;
  function    GetTxSpace:Integer;
  function    GetRxTotal:Int64;
  function    GetTxTotal:Int64;
 protected
  procedure   FixHardwareError(lpError:DWORD);
  procedure   FixError(ErrorCode:TUartErrors);
  procedure   Poll;
 public { internally use only }
  constructor Create(aUart:TUart; aPortNum:TPortNum);
  destructor  Destroy; override;
 public { open / close port }
  function    Open(aBaudrate : LongInt   = 9600;
                   aParity   : TParity   = NOPARITY;
                   aDataBits : TDataBits = 8;
                   aStopBits : TStopBits = ONESTOPBIT;
                   aXonXoff  : Boolean   = false;
                   aFifoSize : LongInt   = 4096;
                   aDcbFlags : LongInt   = 0 ) : Boolean;
  function    OpenConfig(const IniFile,Section:ShortString):Boolean;
  function    Ctrl(const param:LongString):LongString;
  procedure   Close;
 public { general input / output routines }
  function    WriteBuf(const Data:PChar; Count:Cardinal):Cardinal;
  function    ReadBuf(var Data:PChar; MaxCount:Cardinal):Cardinal;
  function    WriteStr(const Data:ShortString):Boolean;
  function    ReadStr(MaxCount:Byte=255):ShortString;
 public { miscellaneous routines }
  procedure   RxClear;
  procedure   TxClear;
  function    GetProperties(TheText:TText):TText;
 public { RS-485 host routines in style "request-answer" }
  function    RequestAvailable:Boolean;
  function    RaiseRequest(const aInquiry   : ShortString;
                                 aTimeOutMs : double;
                                 aDelimiter : Char = ASCII_CR):Boolean;
  function    AnswerRequest(var Inquiry     : ShortString;
                            var Answer      : ShortString;
                            var InquiryTime : double ) : TRequestStatus;
  procedure   ClearRequest;
  function    ModalAnswerRequest(const aInquiry   : ShortString;
                                       aTimeOutMs : double;
                                       aDelimiter : Char = ASCII_CR):ShortString;
 public { properties }
  property    Active:Boolean read GetActive;
  property    Errors[ErrorCode:TUartErrors]:DWORD read GetError;
  property    RxCount:Integer read GetRxCount;
  property    TxCount:Integer read GetTxCount;
  property    RxSpace:Integer read GetRxSpace;
  property    TxSpace:Integer read GetTxSpace;
  property    RxTotal:Int64   read GetRxTotal;
  property    TxTotal:Int64   read GetTxTotal;
 end;
 {
 *******************************************************************************
 Genegal communication ports class.
 Uart is one and only available TUart class object.
 *******************************************************************************
 Example:
  if uart[1].Open then begin                       // try open port COM1
   uart[1].WriteStr('Send this string to COM1.');  // write string to port
   Echo('Read from COM1:'+uart[1].ReadStr);        // read string from port
   uart[1].Close;                                  // close port
  end;                                             //
 *******************************************************************************
 }
 TUart = class(TMasterObject)
 private
  myPorts   : packed array[1..MaxPortNum] of TUartPort;
  myLatch   : TRtlCriticalSection;
  myPollStr : ShortString;
  myPollMap : TByteSet;
  myPolling : TPolling;
  function    GetPort(Num:TPortNum):TUartPort;
  function    GetPolling:TPolling;
  function    GetPollMap:TByteSet;
  procedure   SetPollMap(const Map:TByteSet);
  procedure   GetPollStr(var Str:ShortString);
  procedure   EnablePolling(Num:Integer; Enable:boolean);
  procedure   Poll;
 protected
  property    PollMap : TByteSet read GetPollMap write SetPollMap;
 public
  constructor Create(aDelay    : Integer         = 1;
                     aPriority : TThreadPriority = tpHighest);
  procedure   AfterConstruction; override;
  procedure   BeforeDestruction; override;
  destructor  Destroy; override;
  function    GetProperties(TheText : TText;
                      const Prefix  : ShortString = 'SerialPort-'):TText;
  procedure   ReadConfig(const IniFile,Section:ShortString);
 public
  property    Port[Num:TPortNum] : TUartPort read GetPort; default;
  property    Polling            : TPolling  read GetPolling;
 end;

 {
 *******************************************************************************
 Uart is one and only available TUart class object.
 ALL working with communication ports we have to do via uart object.
 Never create or kill any other TUart objects!
 *******************************************************************************
 }
const
 uart : TUart = nil;

 {
 *******************************************************************************
 Check sum utilites.
 Check sums uses to make data transfer more robust and stable. This set of
 check sum functions is similar with Advantec ADAM-4000 seria, I-7000 seria
 devices & etc. This devices have data package format DATA+CS+CR, where
 DATA is data chars, CS is check sum of DATA as 2-chars HEX string, CR is char(13).
 For example, package '$08MD9'+CR is '$08M' data, 'D9' checksum, and CR.
 FindCheckSum    -    ,     
                     .
 CatCheckSum     -         HEX - .
 CatCheckSumCR   -       CR  
                     HEX - .
 TruncCheckSum   -        
                     .    , 
                    .
 TruncCheckSumCR -        
                     .    , 
                    .     CR,  .
 *******************************************************************************
 }
function FindCheckSum(Data:PChar; Length:Integer):Byte;
function CatCheckSum(const DataStr:ShortString):ShortString;
function CatCheckSumCR(const DataStr:ShortString; CR:Char=ASCII_CR):ShortString;
function TruncCheckSum(const DataStr:ShortString):ShortString;
function TruncCheckSumCR(const DataStr:ShortString; CR:Char=ASCII_CR):ShortString;

 {
 EnumComPorts returns CR,LF separated list of available COM ports.
 }
function EnumComPorts(const Delim:LongString=CRLF):LongString;

implementation

 {
 ************************
 TUartPort implementation
 ************************
 }
const                      { DCB flag bits offset:           }
 fBinary           = 0;    { :1  binary mode, no EOF check   }
 fParity           = 1;    { :1  enable parity checking      }
 fOutxCtsFlow      = 2;    { :1  CTS output flow control     }
 fOutxDsrFlow      = 3;    { :1  DSR output flow control     }
 fDtrControl       = 4;    { :2  DTR flow control type       }
 fDsrSensitivity   = 6;    { :1  DSR sensitivity             }
 fTXContinueOnXoff = 7;    { :1  XOFF continues Tx           }
 fOutX             = 8;    { :1  XON/XOFF out flow control   }
 fInX              = 9;    { :1  XON/XOFF in flow control    }
 fErrorChar        = 10;   { :1  enable error replacement    }
 fNull             = 11;   { :1  enable null stripping       }
 fRtsControl       = 12;   { :2  RTS flow control            }
 fAbortOnError     = 14;   { :1  abort reads/writes on error }
 fDummy2           = 15;   { :17 reserved                    }
 PURGE_Rx          = PURGE_RXABORT or PURGE_RXCLEAR;
 PURGE_Tx          = PURGE_TXABORT or PURGE_TXCLEAR;
 PURGE_RxTx        = PURGE_Rx or PURGE_Tx;

const
 fdcbBinary           = $0001;   // binary mode, no EOF check
 fdcbParity           = $0002;   // enable parity checking
 fdcbCtsFlow          = $0004;   // CTS output flow control
 fdcbDsrFlow          = $0008;   // DSR output flow control
 fdcbDtrEnable        = $0010;   // DTR flow control type: software
 fdcbDtrHandshake     = $0020;   // DTR flow control type: hardware
 fdcbDsrEnable        = $0040;   // DSR sensitivity
 fdcbTXContinueOnXoff = $0080;   // XOFF continues Tx
 fdcbOutXEnable       = $0100;   // XON/XOFF out flow control
 fdcbInXEnable        = $0200;   // XON/XOFF in flow control
 fdcbErrorChar        = $0400;   // enable error replacement
 fdcbNullStripped     = $0800;   // enable null stripping
 fdcbRtsEnable        = $1000;   // RTS flow control:software
 fdcbRtsHandshake     = $2000;   // RTS flow control:hardware
 fdcbRtsToggle        = $3000;   // RTS flow control:
 fdcbAbortOnError     = $4000;   // abort reads/writes on error
 fdcbHardControl      = fdcbBinary OR fdcbDtrEnable OR fdcbDsrFlow OR fdcbCtsFlow OR fdcbRTSToggle;
 fdcbSoftControl      = fdcbBinary OR fdcbOutXEnable OR fdcbInXEnable;
 fdcbNoneControl      = fdcbBinary;

constructor TUartPort.Create(aUart:TUart; aPortNum:TPortNum);
begin
 inherited Create;
 Lock;
 try
  myUart:=aUart;
  myPortNum:=aPortNum;
  StrPCopy(myPortName,'COM'+d2s(myPortNum));
  StrPCopy(myFileName,'\\.\COM'+d2s(myPortNum));
  myCommFile:=INVALID_HANDLE_VALUE;
  myRxFifo:=nil;
  myTxFifo:=nil;
  myRxTotal:=0;
  myTxTotal:=0;
  myRxPending:=false;
  myTxPending:=false;
  SafeFillChar(myDCB,sizeof(myDCB),0);
  SafeFillChar(myCOMMPROP,sizeof(myCOMMPROP),0);
  SafeFillChar(myTIMEOUTS,sizeof(myTIMEOUTS),0);
  SafeFillChar(myRxOverlap,sizeof(myRxOverlap),0);
  SafeFillChar(myTxOverlap,sizeof(myTxOverlap),0);
  SafeFillChar(myRxBuffer,sizeof(myRxBuffer),0);
  SafeFillChar(myTxBuffer,sizeof(myTxBuffer),0);
  SafeFillChar(myErrors,sizeof(myErrors),0);
  myRequest.Inquiry:='';
  myRequest.Answer:='';
  myRequest.Timer:=nil;
  myRequest.Delimiter:=ASCII_CR;
 finally
  Unlock;
 end;
end;

destructor  TUartPort.Destroy;
begin
 try
  Lock;
  try
   myUart.EnablePolling(myPortNum,false);
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
 Close;
 inherited Destroy;
end;

function TUartPort.GetActive:Boolean;
begin
 Result:=false;
 if Assigned(Self) then begin
  Lock;
  Result:=(myCommFile <> INVALID_HANDLE_VALUE);
  Unlock;
 end;
end;

function TUartPort.GetError(ErrorCode:TUartErrors):DWORD;
begin
 Result:=0;
 if Assigned(Self) and (ErrorCode>=Low(myErrors)) and (ErrorCode<=High(myErrors)) then begin
  Lock;
  Result:=myErrors[ErrorCode];
  Unlock;
 end;
end;

function TUartPort.GetRxCount:Integer;
begin
 Result:=0;
 if Assigned(Self) then begin
  Lock;
  Result:=myRxFifo.Count;
  Unlock;
 end;
end;

function TUartPort.GetTxCount:Integer;
begin
 Result:=0;
 if Assigned(Self) then begin
  Lock;
  Result:=myTxFifo.Count;
  Unlock;
 end;
end;

function TUartPort.GetRxSpace:Integer;
begin
 Result:=0;
 if Assigned(Self) then begin
  Lock;
  Result:=myRxFifo.Space;
  Unlock;
 end;
end;

function TUartPort.GetTxSpace:Integer;
begin
 Result:=0;
 if Assigned(Self) then begin
  Lock;
  Result:=myTxFifo.Space;
  Unlock;
 end;
end;

function TUartPort.GetRxTotal:Int64;
begin
 Result:=0;
 if Assigned(Self) then begin
  Lock;
  Result:=myRxTotal;
  Unlock;
 end;
end;

function TUartPort.GetTxTotal:Int64;
begin
 Result:=0;
 if Assigned(Self) then begin
  Lock;
  Result:=myTxTotal;
  Unlock;
 end;
end;

procedure TUartPort.FixHardwareError(lpError:DWORD);
begin
 if lpError<>0 then begin
  if lpError and CE_RXOVER   <> 0 then FixError(ue_RXOVER);
  if lpError and CE_OVERRUN  <> 0 then FixError(ue_OVERRUN);
  if lpError and CE_RXPARITY <> 0 then FixError(ue_RXPARITY);
  if lpError and CE_FRAME    <> 0 then FixError(ue_FRAME);
  if lpError and CE_BREAK    <> 0 then FixError(ue_BREAK);
  if lpError and CE_TXFULL   <> 0 then FixError(ue_TXFULL);
  if lpError and CE_IOE      <> 0 then FixError(ue_IOE);
  if lpError and CE_MODE     <> 0 then FixError(ue_MODE);
 end;
end;

procedure TUartPort.FixError(ErrorCode:TUartErrors);
begin
 if Assigned(Self) and (ErrorCode>=Low(myErrors)) and (ErrorCode<=High(myErrors)) then begin
  Lock;
  inc(myErrors[ue_TOTAL]);
  inc(myErrors[ErrorCode]);
  Unlock;
 end;
end;

function TUartPort.Open(aBaudrate : LongInt   = 9600;
                        aParity   : TParity   = NOPARITY;
                        aDataBits : TDataBits = 8;
                        aStopBits : TStopBits = ONESTOPBIT;
                        aXonXoff  : Boolean   = false;
                        aFifoSize : LongInt   = 4096;
                        aDcbFlags : LongInt   = 0 ) : Boolean;
 function SetupCommPortParameters:Boolean;
 var
  RxQ,TxQ  : Cardinal;
 begin
  Result:=false;
  {
  now check file
  }
  if myCommFile = INVALID_HANDLE_VALUE then exit;                          { must be valid handle! }
  if GetFileType(myCommFile)<>FILE_TYPE_CHAR then exit;                    { must be char file! }
  {
  now setup Rx & Tx queue size
  }
  RxQ:=RxQueueSize;                                                        { set Rx queue size }
  TxQ:=TxQueueSize;                                                        { set Tx queue size }
  if not GetCommProperties(myCommFile,myCOMMPROP) then exit;               { readback comm properties }
  if myCOMMPROP.dwMaxRxQueue>0 then RxQ:=min(RxQ,myCOMMPROP.dwMaxRxQueue); { check and correct Rx queue size }
  if myCOMMPROP.dwMaxTxQueue>0 then TxQ:=min(TxQ,myCOMMPROP.dwMaxTxQueue); { check and correct Tx queue size }
  if not SetupComm(myCommFile, RxQ, TxQ) then exit;                        { setup Rx and Tx queue size to device }
  {
  now setup DCB struct
  }
  myDCB.DCBlength:=sizeof(myDCB);                                          { size of DCB struct }
  myDCB.BaudRate:=aBaudRate;                                               { current baud rate  }
  myDCB.Flags:=0;                                                          { clear Flags bits   }
  myDCB.Flags:=myDCB.Flags or (DWORD(1) shl fBinary);                      { binary mode, no EOF check }
  if aParity<>NOPARITY then                                                { if uses parity check then }
  myDCB.Flags:=myDCB.Flags or (DWORD(1) shl fParity);                      { enable parity checking }
  myDCB.Flags:=myDCB.Flags or (DWORD(1) shl fAbortOnError);                { abort reads/writes on error }
  if aXonXoff then begin
   myDCB.Flags:=myDCB.Flags or (DWORD(1) shl fTXContinueOnXoff);           { XOFF continues Tx }
   myDCB.Flags:=myDCB.Flags or (DWORD(1) shl fOutX);                       { XON/XOFF out flow control }
   myDCB.Flags:=myDCB.Flags or (DWORD(1) shl fInX);                        { XON/XOFF in flow control }
  end;
  aDcbFlags:=aDcbFlags and not (fdcbBinary or fdcbParity or fdcbAbortOnError);
  myDCB.Flags:=myDCB.Flags or aDcbFlags;                                   { setup other Flags bits   }
  myDCB.wReserved:=0;                                                      { not currently used }
  myDCB.XonLim:=(RxQ div 4)*1;                                             { transmit XON threshold }
  myDCB.XoffLim:=(RxQ div 4)*3;                                            { transmit XOFF threshold }
  myDCB.ByteSize:=aDataBits;                                               { 4-8=number of bits/byte }
  myDCB.Parity:=aParity;                                                   { 0-4=no,odd,even,mark,space }
  myDCB.StopBits:=aStopBits;                                               { 0,1,2 = 1, 1.5, 2 }
  myDCB.XonChar:=ASCII_XON;                                                { Tx and Rx XON character }
  myDCB.XoffChar:=ASCII_XOFF;                                              { Tx and Rx XOFF character }
  myDCB.ErrorChar:=ASCII_NUL;                                              { error replacement character }
  myDCB.EofChar:=ASCII_FS;                                                 { end of input character }
  myDCB.EvtChar:=ASCII_CR;                                                 { received event character }
  myDCB.wReserved1:=0;                                                     { reserved; do not use }
  if not SetCommState(myCommFile, myDCB) then exit;                        { setup DCB to device }
  {
  now set timeouts
  }
  myTIMEOUTS.ReadIntervalTimeout:=MAXDWORD;                                { means that do'nt use Rx timouts }
  myTIMEOUTS.ReadTotalTimeoutMultiplier:=0;
  myTIMEOUTS.ReadTotalTimeoutConstant:=0;
  myTIMEOUTS.WriteTotalTimeoutMultiplier:=0;                               { means that do'nt use Tx timouts }
  myTIMEOUTS.WriteTotalTimeoutConstant:=0;
  if not SetCommTimeouts(myCommFile, myTIMEOUTS) then exit;                { now setup timeouts to device }
  {
  now set events mask
  }
  if not SetCommMask(myCommFile, 0) then exit;                             { does not use any events }
  {
  now readback all params and purge queues
  }
  if not GetCommProperties(myCommFile,myCOMMPROP) then exit;               { readback comm properties }
  if myCOMMPROP.dwCurrentRxQueue=0 then myCOMMPROP.dwCurrentRxQueue:=RxQ;  { correct if 0 (no access) }
  if myCOMMPROP.dwCurrentTxQueue=0 then myCOMMPROP.dwCurrentTxQueue:=TxQ;  { correct if 0 (no access) }
  if not GetCommState(myCommFile,myDCB) then exit;                         { readback DCB }
  if not GetCommTimeouts(myCommFile, myTIMEOUTS) then exit;                { readback timeouts }
  if not PurgeComm(myCommFile, PURGE_RxTx) then exit;                      { clear device Rx and Tx buffers }
  Result:=true;
 end;
begin
 Result:=false;
 if Assigned(Self) then
 try
  Lock;
  try
   Close;
   myCommFile:=CreateFile( myFileName,                      { pointer to name of the file }
                           GENERIC_READ or GENERIC_WRITE,   { access (read-write) mode }
                           0,                               { share mode - exclusive access }
                           nil,                             { no security attrs }
                           OPEN_EXISTING,                   { how to create }
                           FILE_FLAG_OVERLAPPED,            { use overlapped I/O }
                           0 );                             { template file - not used }
   if myCommFile <> INVALID_HANDLE_VALUE then begin
    if SetupCommPortParameters then begin
     myRxFifo:=NewFifo(aFifoSize); myRxFifo.GrowFactor:=UartFifoGrowFactor; myRxFifo.GrowLimit:=UartFifoGrowLimit;
     myTxFifo:=NewFifo(aFifoSize); myTxFifo.GrowFactor:=UartFifoGrowFactor; myTxFifo.GrowLimit:=UartFifoGrowLimit;
     myRequest.Timer:=NewIntervalTimer;
     myRequest.Timer.AddIntervalMs(200,1);
     myUart.EnablePolling(myPortNum,true);
     Result:=true;
    end else Close;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TUartPort.Close;
begin
 if Assigned(Self) then
 try
  Lock;
  try
   myUart.EnablePolling(myPortNum,false);
   if myCommFile <> INVALID_HANDLE_VALUE then begin
    PurgeComm(myCommFile, PURGE_RxTx);
    CloseHandle(myCommFile);
    myCommFile:=INVALID_HANDLE_VALUE;
   end;
   Kill(myRxFifo);
   Kill(myTxFifo);
   myRxTotal:=0;
   myTxTotal:=0;
   myRxPending:=false;
   myTxPending:=false;
   SafeFillChar(myDCB,sizeof(myDCB),0);
   SafeFillChar(myCOMMPROP,sizeof(myCOMMPROP),0);
   SafeFillChar(myTIMEOUTS,sizeof(myTIMEOUTS),0);
   SafeFillChar(myRxOverlap,sizeof(myRxOverlap),0);
   SafeFillChar(myTxOverlap,sizeof(myTxOverlap),0);
   SafeFillChar(myRxBuffer,sizeof(myRxBuffer),0);
   SafeFillChar(myTxBuffer,sizeof(myTxBuffer),0);
   SafeFillChar(myErrors,sizeof(myErrors),0);
   myRequest.Inquiry:='';
   myRequest.Answer:='';
   Kill(myRequest.Timer);
   myRequest.Delimiter:=ASCII_CR;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TUartPort.Poll;
var
 RxCounter  : DWORD;
 TxCounter  : DWORD;
 RxBlock    : DWORD;
 TxBlock    : DWORD;
 lpErrors   : DWORD;
 lpStat     : ComStat;
begin
 if Assigned(Self) then
 try
  Lock;
  try
   if myCommFile <> INVALID_HANDLE_VALUE then begin
    {
    Read from COMM
    }
    if myRxPending then begin
     { pending I/O completion, check overlapped I/O status, no wait }
     if GetOverlappedResult(myCommFile,myRxOverlap,RxCounter,FALSE) then begin
      { I/O operation complete, try to put FIFO and clear pending flag }
      if myRxFifo.Put(@myRxBuffer,RxCounter)=LongInt(RxCounter)
      then inc(myRxTotal,RxCounter)
      else FixError(ue_RxFIFO_OVER);
      myRxPending:=false;
     end else begin
      if GetLastError=ERROR_IO_INCOMPLETE then begin
       { expected result, do nothing, wait I/O completion }
      end else begin
       { error, fix it, clear queue and error flag, break pending I/O }
       if ClearCommError(myCommFile,lpErrors,nil) then FixHardwareError(lpErrors) else lpErrors:=0;
       PurgeComm(myCommFile,PURGE_Rx);
       FixError(ue_UNEXPECTED);
       myRxPending:=false;
      end;
     end;
    end else begin
     { no pending I/O, check receiver queue and read if not empty }
     if ClearCommError(myCommFile,lpErrors,@lpStat) then begin
      FixHardwareError(lpErrors);
      RxBlock:=min(min(lpStat.cbInQue,sizeof(myRxBuffer)),myRxFifo.Space);
      if RxBlock>0 then begin
       if ReadFile(myCommFile,myRxBuffer,RxBlock,RxCounter,@myRxOverlap) then begin
        { read complete immediatly, try to put FIFO and clear pending flag }
        if myRxFifo.Put(@myRxBuffer,RxCounter)=LongInt(RxCounter)
        then inc(myRxTotal,RxCounter)
        else FixError(ue_RxFIFO_OVER);
        myRxPending:=false;
       end else begin
        if GetLastError=ERROR_IO_PENDING then begin
         { expected result, overlapped read, set pending flag }
         myRxPending:=true;
        end else begin
         { error }
         if ClearCommError(myCommFile,lpErrors,nil) then FixHardwareError(lpErrors) else lpErrors:=0;
         PurgeComm(myCommFile,PURGE_Rx);
         FixError(ue_UNEXPECTED);
         myRxPending:=false;
        end;
       end;
      end;
     end else FixError(ue_UNEXPECTED);
    end;
    {
    Write to COMM
    }
    if myTxPending then begin
     { pending I/O completion, check overlapped I/O status, no wait }
     if GetOverlappedResult(myCommFile,myTxOverlap,TxCounter,FALSE) then begin
      { I/O operation complete, fix it and clear pending flag }
      inc(myTxTotal,TxCounter);
      myTxPending:=false;
     end else begin
      if GetLastError=ERROR_IO_INCOMPLETE then begin
       { expected result, do nothing, wait I/O completion }
      end else begin
       { error, fix it, clear queue and error flag, break pending I/O }
       if ClearCommError(myCommFile,lpErrors,nil) then FixHardwareError(lpErrors) else lpErrors:=0;
       PurgeComm(myCommFile,PURGE_Tx);
       FixError(ue_UNEXPECTED);
       myTxPending:=false;
      end;
     end;
    end else begin
     if ClearCommError(myCommFile,lpErrors,@lpStat) then begin
      FixHardwareError(lpErrors);
      TxBlock:=min(min(myTxFifo.Count,sizeof(myTxBuffer)),myCOMMPROP.dwCurrentTxQueue-lpStat.cbOutQue);
      if TxBlock>0 then
      if myTxFifo.Get(@myTxBuffer,TxBlock)=LongInt(TxBlock) then begin
       if WriteFile(myCommFile,myTxBuffer,TxBlock,TxCounter,@myTxOverlap) then begin
        {write immediatly, fix it and clear pending flag}
        inc(myTxTotal,TxCounter);
        myTxPending:=false;
       end else begin
        if GetLastError=ERROR_IO_PENDING then begin
         { expected result, set pending flag }
         myTxPending:=true;
        end else begin
         { error }
         if ClearCommError(myCommFile,lpErrors,nil) then FixHardwareError(lpErrors) else lpErrors:=0;
         PurgeComm(myCommFile,PURGE_Tx);
         FixError(ue_UNEXPECTED);
         myTxPending:=false;
        end;
       end;
      end else FixError(ue_UNEXPECTED);;
     end else FixError(ue_UNEXPECTED);
    end;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.WriteBuf(const Data:PChar; Count:Cardinal):Cardinal;
begin
 Result:=0;
 if Assigned(Self) then
 try
  Lock;
  try
   Result:=myTxFifo.Put(@Data[0],Count);
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.ReadBuf(var Data:PChar; MaxCount:Cardinal):Cardinal;
begin
 Result:=0;
 if Assigned(Self) then
 try
  Lock;
  try
   Result:=myRxFifo.Get(@Data[0],MaxCount);
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.WriteStr(const Data:ShortString):Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  Lock;
  try
   Result:=myTxFifo.Put(@Data[1],Length(Data))=Length(Data);
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.ReadStr(MaxCount:Byte=255):ShortString;
begin
 Result:='';
 if Assigned(Self) then
 try
  Lock;
  try
   Result[0]:=Chr(myRxFifo.Get(@Result[1],MaxCount));
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TUartPort.RxClear;
begin
 if Assigned(Self) then
 try
  Lock;
  try
   if myCommFile <> INVALID_HANDLE_VALUE then begin
    myRxFifo.Clear;
    PurgeComm(myCommFile,PURGE_Rx);
    myRxPending:=false;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TUartPort.TxClear;
begin
 if Assigned(Self) then
 try
  Lock;
  try
   if myCommFile <> INVALID_HANDLE_VALUE then begin
    myTxFifo.Clear;
    PurgeComm(myCommFile,PURGE_Tx);
    myTxPending:=false;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function  TUartPort.RequestAvailable:Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  Lock;
  try
   Result:=(myCommFile <> INVALID_HANDLE_VALUE) and Assigned(myRequest.Timer) and not myRequest.Timer.isStart;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.RaiseRequest(const aInquiry   : ShortString;
                                      aTimeOutMs : double;
                                      aDelimiter : Char = ASCII_CR):Boolean;
begin
 Result:=false;
 if Assigned(Self) then
 try
  Lock;
  try
   if RequestAvailable then begin
    RxClear;
    TxClear;
    if WriteStr(aInquiry) then begin
     myRequest.Inquiry:=aInquiry;
     myRequest.Answer:='';
     myRequest.Timer.IntervalMs[0]:=aTimeOutMs;
     myRequest.Timer.Start;
     myRequest.Delimiter:=aDelimiter;
     Result:=true;
    end else begin
     myRequest.Inquiry:='';
     myRequest.Answer:='';
     myRequest.Delimiter:=ASCII_CR;
    end;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.AnswerRequest(var Inquiry     : ShortString;
                                 var Answer      : ShortString;
                                 var InquiryTime : double ) : TRequestStatus;
var
 DelimPos : integer;
begin
 Result:=rs_Error;
 if Assigned(Self) then
 try
  Lock;
  try
   if (myCommFile <> INVALID_HANDLE_VALUE) and Assigned(myRequest.Timer) then begin
    {  }
    if myRequest.Timer.isStart then begin
     myRequest.Answer:=myRequest.Answer+ReadStr;
     {   ,      }
     DelimPos:=pos(myRequest.Delimiter,myRequest.Answer);
     if DelimPos>0 then begin
      Result:=rs_Answer;
      Inquiry:=myRequest.Inquiry;
      Answer:=Copy(myRequest.Answer,1,DelimPos-1);
      InquiryTime:=myRequest.Timer.StartTime;
      ClearRequest;
     end
     {   -   ,   }
     else begin
      {      }
      if myRequest.Timer.Event then begin
       Result:=rs_TimeOut;
       Inquiry:=myRequest.Inquiry;
       Answer:='';
       InquiryTime:=myRequest.Timer.StartTime;
       ClearRequest;
      end
      {    -  }
      else begin
       Result:=rs_WaitAnswer;
       Inquiry:=myRequest.Inquiry;
       Answer:='';
       InquiryTime:=myRequest.Timer.StartTime;
      end;
     end;
    end
    {  }
    else begin
     Result:=rs_NoRequest;
     Inquiry:='';
     Answer:='';
     InquiryTime:=0;
    end;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TUartPort.ClearRequest;
begin
 if Assigned(Self) then
 try
  Lock;
  try
   RxClear;
   TxClear;
   myRequest.Inquiry:='';
   myRequest.Answer:='';
   myRequest.Timer.Stop;
   myRequest.Delimiter:=ASCII_CR;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.ModalAnswerRequest(const aInquiry   : ShortString;
                                            aTimeOutMs : double;
                                            aDelimiter : Char = ASCII_CR):ShortString;
var
 Inquiry  : ShortString;
 Answer   : ShortString;
 TimeMsec : double;
begin
 Result:='';
 if RaiseRequest(aInquiry,aTimeOutMs,aDelimiter) then begin
  while true do
  case AnswerRequest(Inquiry,Answer,TimeMsec) of
   rs_NoRequest  : break;
   rs_WaitAnswer : Sleep(1);
   rs_Answer     : begin Result:=Answer; break; end;
   rs_TimeOut    : break;
   rs_Error      : break;
   else break;
  end;
 end;
end;

function TUartPort.GetProperties(TheText:TText):TText;
 procedure AddVal(const msg:ShortString; val:LongInt);
 begin
  if Val<>0 then TheText.Addln(msg+' : '+d2s(val));
 end;
begin
 Result:=TheText;
 if Assigned(Self) then
 try
  Lock;
  try
   if Active then begin
    TheText.AddLn('Port = '+StrPas(myPortName));
    TheText.Addln('BaudRate = '+d2s(myDCB.BaudRate));
    case myDCB.Parity of
     NOPARITY    : TheText.Addln('Parity = NONE');
     ODDPARITY   : TheText.Addln('Parity = ODD');
     EVENPARITY  : TheText.Addln('Parity = EVEN');
     MARKPARITY  : TheText.Addln('Parity = MARK');
     SPACEPARITY : TheText.Addln('Parity = SPACE');
    end;
    TheText.Addln('DataBits = '+d2s(myDCB.ByteSize));
    case myDCB.StopBits of
     ONESTOPBIT   : TheText.Addln('StopBits = 1');
     ONE5STOPBITS : TheText.Addln('StopBits = 1.5');
     TWOSTOPBITS  : TheText.Addln('StopBits = 2');
    end;
    TheText.Addln('XonXoff = '+d2s(ord(myDCB.Flags and ((DWORD(1) shl fInX)+(DWORD(1) shl fOutX))<>0)));
    TheText.Addln('BufSize = '+d2s(myRxFifo.Size div 1024)+' KB');
    TheText.Addln('DcbFlags= $'+HexW(myDcb.Flags));
    TheText.Addln('TRAFFIC INFORMATION:');
    TheText.Addln(' Rx total, bytes    : '+d2s(RxTotal));
    TheText.Addln(' Tx total, bytes    : '+d2s(TxTotal));
    TheText.Addln(' Rx contains, bytes : '+d2s(RxCount));
    TheText.Addln(' Tx contains, bytes : '+d2s(TxCount));
    if Errors[ue_Total]<>0 then begin
     TheText.AddLn('ERRORS DETECTED:');
     AddVal('Total errors    ',Errors[ue_TOTAL]);
     AddVal('Rx overflow     ',Errors[ue_RXOVER]);
     AddVal('Rx overrun      ',Errors[ue_OVERRUN]);
     AddVal('Parity          ',Errors[ue_RXPARITY]);
     AddVal('Framing         ',Errors[ue_FRAME]);
     AddVal('Break           ',Errors[ue_BREAK]);
     AddVal('Tx overflow     ',Errors[ue_TXFULL]);
     AddVal('I/O error       ',Errors[ue_IOE]);
     AddVal('ode unsupported',Errors[ue_MODE]);
     AddVal('Rx FIFO overflow',Errors[ue_RxFIFO_OVER]);
     AddVal('Tx FIFO overflow',Errors[ue_TxFIFO_OVER]);
     AddVal('Unexpected error',Errors[ue_UNEXPECTED]);
    end;
   end else begin
    TheText.Addln('Port is not accessible.');
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.OpenConfig(const IniFile,Section:ShortString):Boolean;
var
 BaudRate  : LongInt;
 Parity    : TParity;
 DataBits  : LongInt;
 StopBits  : LongInt;
 XonXoff   : Boolean;
 BufSize   : LongInt;
 fStopBits : Double;
 s         : ShortString;
 DcbFlags  : LongInt;
begin
 Result:=false;
 if Assigned(Self) then
 try
  Lock;
  try
   if ReadIniFileVariable(IniFile,Section,'Port%a',s) then
   if UnifyAlias(s)=UnifyAlias(StrPas(myPortName)) then begin
    if not ReadIniFileVariable(IniFile,Section,'BaudRate%d',BaudRate) then BaudRate:=9600;
    if not ReadIniFileVariable(IniFile,Section,'Parity%a',s) then s:='NOPARITY';
    s:=UnifyAlias(s);
    if (s='NONE')  or (s='NOPARITY')    or (s='N') then Parity:=NOPARITY    else
    if (s='ODD')   or (s='ODDPARITY')   or (s='O') then Parity:=ODDPARITY   else
    if (s='EVEN')  or (s='EVENPARITY')  or (s='E') then Parity:=EVENPARITY  else
    if (s='MARK')  or (s='MARKPARITY')  or (s='M') then Parity:=MARKPARITY  else
    if (s='SPACE') or (s='SPACEPARITY') or (s='S') then Parity:=SPACEPARITY else Parity:=NOPARITY;
    if not ReadIniFileVariable(IniFile,Section,'DataBits%d',DataBits) then DataBits:=8;
    if ReadIniFileVariable(IniFile,Section,'StopBits%f',fStopBits) then begin
     if fStopBits=1.0 then StopBits:=ONESTOPBIT   else
     if fStopBits=1.5 then StopBits:=ONE5STOPBITS else
     if fStopBits=2.0 then StopBits:=TWOSTOPBITS  else StopBits:=ONESTOPBIT;
    end else StopBits:=ONESTOPBIT;
    if not ReadIniFileVariable(IniFile,Section,'XonXoff%b',XonXoff) then XonXoff:=false;
    if not ReadIniFileVariable(IniFile,Section,'BufSize%d',BufSize) then BufSize:=4;
    if not ReadIniFileVariable(IniFile,Section,'DcbFlags%d',DcbFlags) then DcbFlags:=0;
    Result:=Open(BaudRate,Parity,DataBits,StopBits,XonXoff,BufSize*1024,DcbFlags);
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

function TUartPort.Ctrl(const param:LongString):LongString;
var p,n:Integer; sn,sv:LongString; r:Double; SaveDcb:TDCB;
 function UpdateDcb:Boolean;
 begin
  Result:=false;
  if not SetCommState(myCommFile,myDCB) then begin myDCB:=SaveDcb; Exit; end;
  if not GetCommState(myCommFile,myDCB) then begin myDCB:=SaveDcb; Exit; end;
  Result:=true;
 end;
begin
 Result:='?';
 if Assigned(Self) then
 if Active then
 try
  Lock;
  try
   SaveDcb:=myDCB;
   p:=System.Pos('=',param);
   if p=0 then begin
    sn:=SysUtils.Trim(param);
    sv:='';
   end else begin
    sn:=SysUtils.Trim(System.Copy(param,1,p-1));
    sv:=SysUtils.Trim(System.Copy(param,p+1,length(param)-p));
   end;
   if SameText(sn,'BaudRate') then begin
    if (p>0) and Str2Int(sv,n) then begin
     myDCB.BaudRate:=n;
     if not UpdateDcb then Exit;
    end;
    Result:=d2s(myDCB.BaudRate);
    Exit;
   end;
   if SameText(sn,'Parity') then begin
    if (p>0) then begin
     sv:=UpCaseStr(Trim(sv));
     if (sv='NONE')  or (sv='NOPARITY')    or (sv='N') then n:=NOPARITY    else
     if (sv='ODD')   or (sv='ODDPARITY')   or (sv='O') then n:=ODDPARITY   else
     if (sv='EVEN')  or (sv='EVENPARITY')  or (sv='E') then n:=EVENPARITY  else
     if (sv='MARK')  or (sv='MARKPARITY')  or (sv='M') then n:=MARKPARITY  else
     if (sv='SPACE') or (sv='SPACEPARITY') or (sv='S') then n:=SPACEPARITY else n:=-1;
    end else n:=-1;
    if (p>0) and (n>=0) then begin
     myDCB.Parity:=n;
     if not UpdateDcb then Exit;
    end;
    case myDCB.Parity of
     NOPARITY    : Result:='NONE';
     ODDPARITY   : Result:='ODD';
     EVENPARITY  : Result:='EVEN';
     MARKPARITY  : Result:='MARK';
     SPACEPARITY : Result:='SPACE';
    end;
    Exit;
   end;
   if SameText(sn,'DataBits') then begin
    if (p>0) and Str2Int(sv,n) then begin
     myDCB.ByteSize:=n;
     if not UpdateDcb then Exit;
    end;
    Result:=d2s(myDCB.ByteSize);
    Exit;
   end;
   if SameText(sn,'StopBits') then begin
    if (p>0) and Str2Real(sv,r) then begin
     if r=1.0 then myDCB.StopBits:=ONESTOPBIT   else
     if r=1.5 then myDCB.StopBits:=ONE5STOPBITS else
     if r=2.0 then myDCB.StopBits:=TWOSTOPBITS;
     if not UpdateDcb then Exit;
    end;
    case myDCB.StopBits of
     ONESTOPBIT   : Result:='1';
     ONE5STOPBITS : Result:='1.5';
     TWOSTOPBITS  : Result:='2';
    end;
    Exit;
   end;
   if SameText(sn,'DcbFlags') then begin
    if (p>0) and Str2Int(sv,n) then begin
     myDCB.Flags:=n;
     if not UpdateDcb then Exit;
    end;
    Result:='$'+HexW(myDCB.Flags);
    Exit;
   end;
  finally
   Unlock;
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

 {
 ********************
 TUart implementation
 ********************
 }
procedure UartPollAction(aPolling:TPolling; var Terminate:Boolean);
begin
 if aPolling.LinkObject is TUart
 then (aPolling.LinkObject as TUart).Poll
 else Terminate:=true;
end;

constructor TUart.Create(aDelay    : Integer         = 1;
                         aPriority : TThreadPriority = tpHighest);
var
 i : integer;
begin
 Kill(TObject(uart));
 inherited Create;
 InitializeCriticalSection(myLatch);
 myPollStr:='';
 myPollMap:=[];
 for i:=Low(myPorts) to High(myPorts) do begin
  myPorts[i]:=TUartPort.Create(Self,i);
  myPorts[i].Master:=myPorts[i];
 end;
 myPolling:=NewPolling(UartPollAction, aDelay, aPriority, false, 'System.Uart');
 myPolling.Master:=myPolling;
 myPolling.LinkObject:=Self;
end;

procedure TUart.AfterConstruction;
begin
 inherited AfterConstruction;
 uart:=Self;
 uart.Master:=uart;
 Polling.Enable(true);
end;

procedure TUart.BeforeDestruction;
begin
 uart:=nil;
 inherited BeforeDestruction;
end;

destructor TUart.Destroy;
var
 i : Integer;
begin
 PollMap:=[];
 myPolling.Enable(false);
 Kill(myPolling);
 for i:=Low(myPorts) to High(myPorts) do Kill(TObject(myPorts[i]));
 DeleteCriticalSection(myLatch);
 inherited Destroy;
end;

function TUart.GetPort(Num:TPortNum):TUartPort;
begin
 if Assigned(Self) and (Num>=Low(myPorts)) and (Num<=High(myPorts))
 then Result:=myPorts[Num]
 else Result:=nil;
end;

function TUart.GetPolling:TPolling;
begin
 if Assigned(Self) then Result:=myPolling else Result:=nil;
end;

function TUart.GetPollMap:TByteSet;
begin
 if (Self=nil) then begin Result:=[]; Exit; end;
 EnterCriticalSection(myLatch);
 try
  Result:=myPollMap;
 finally
  LeaveCriticalSection(myLatch);
 end;
end;

procedure TUart.SetPollMap(const Map:TByteSet);
var PortNum:Integer;
begin
 if (Self=nil) then Exit;
 EnterCriticalSection(myLatch);
 try
  myPollMap:=Map;
  myPollStr:='';
  for PortNum:=Low(myPorts) to High(myPorts) do
  if (PortNum in myPollMap) then myPollStr:=myPollStr+Chr(PortNum);
 finally
  LeaveCriticalSection(myLatch);
 end;
end;

procedure TUart.GetPollStr(var Str:ShortString);
begin
 if (Self=nil) then Exit;
 EnterCriticalSection(myLatch);
 try
  Str:=myPollStr;
 finally
  LeaveCriticalSection(myLatch);
 end;
end;

procedure TUart.EnablePolling(Num:Integer; Enable:boolean);
var Map:TByteSet;
begin
 if (Self=nil) then Exit;
 EnterCriticalSection(myLatch);
 try
  Map:=PollMap;
  if Enable then Include(Map,Num) else Exclude(Map,Num);
  PollMap:=Map;
 finally
  LeaveCriticalSection(myLatch);
 end;
end;

procedure TUart.Poll;
var i,PortNum:Integer; Str:ShortString;
begin
 if (Self=nil) then Exit;
 Str:=''; GetPollStr(Str);
 for i:=1 to Length(Str) do begin
  PortNum:=Ord(Str[i]);
  if (PortNum>=Low(myPorts)) and (PortNum<=High(myPorts)) then myPorts[PortNum].Poll;
 end;
end;

function TUart.GetProperties(TheText : TText;
                       const Prefix  : ShortString = 'SerialPort-'):TText;
var PortNum:TPortNum;
begin
 Result:=TheText;
 if Assigned(Self) then begin
  TheText.Addln('[System]');
  TheText.AddLn(Format('UartPolling = %d, %s',[Polling.Delay,GetPriorityName(Polling.Priority)]));
  for PortNum:=Low(myPorts) to High(myPorts) do
  if Port[PortNum].Active then begin
   TheText.Addln('['+Prefix+'COM'+d2s(PortNum)+']');
   Port[PortNum].GetProperties(TheText);
  end;
 end;
end;

procedure TUart.ReadConfig(const IniFile,Section:ShortString);
var
 D : Integer;
 P : TThreadPriority;
begin
 if Assigned(Self) then begin
  if ReadIniFilePolling(IniFile,Section,'UartPolling',D,P) then begin
   Polling.Delay:=D;
   Polling.Priority:=P;
  end;
 end;
end;

 {
 *******************************************************************************
 Check sum utilites implementation.
 *******************************************************************************
 }
function FindCheckSum(Data:PChar; Length:Integer):byte;
var i:integer;
begin
 Result:=0;
 for i:=0 to Length-1 do inc(Result,ord(Data[i]));
end;

function CatCheckSum(const DataStr:ShortString):ShortString;
begin
 Result:=DataStr+HexB(FindCheckSum(@DataStr[1],Length(DataStr)));
end;

function CatCheckSumCR(const DataStr:ShortString; CR:Char=ASCII_CR):ShortString;
var Len:integer;
begin
 Len:=Length(DataStr);
 if DataStr[Len]=CR
 then Result:=Copy(DataStr,1,Len-1)+HexB(FindCheckSum(@DataStr[1],Len-1))+CR
 else Result:=DataStr+HexB(FindCheckSum(@DataStr[1],Len));
end;

function TruncCheckSum(const DataStr:ShortString):ShortString;
var Len:integer;
begin
 Result:='';
 Len:=Length(DataStr);
 if (Len>2) and (HexB(FindCheckSum(@DataStr[1],Len-2))=Copy(DataStr,Len-1,2))
 then Result:=Copy(DataStr,1,Len-2);
end;

function TruncCheckSumCR(const DataStr:ShortString; CR:Char=ASCII_CR):ShortString;
var Len:integer;
begin
 Result:='';
 Len:=Length(DataStr);
 if DataStr[Len]=CR then Dec(Len);
 if (Len>2) and (HexB(FindCheckSum(@DataStr[1],Len-2))=Copy(DataStr,Len-1,2))
 then Result:=Copy(DataStr,1,Len-2);
end;

function EnumComPorts(const Delim:LongString=CRLF):LongString;
var
 KeyHandle: HKEY;
 Ports: TStringList;
 ErrCode, Index: Integer;
 ValueName, Data: LongString;
 ValueLen, DataLen, ValueType: DWORD;
begin
 Result:='';
 try
  if RegOpenKeyEx(HKEY_LOCAL_MACHINE,'HARDWARE\DEVICEMAP\SERIALCOMM',0,KEY_READ,
     KeyHandle)=ERROR_SUCCESS
  then begin
   Ports:=TStringList.Create;
   try
    Index:=0;
    repeat
     DataLen:=256;
     ValueLen:=256;
     SetLength(Data,DataLen);
     SetLength(ValueName,ValueLen);
     ErrCode:=RegEnumValue(KeyHandle, Index, PChar(ValueName), ValueLen,
                           nil, @ValueType, Windows.PByte(PChar(Data)), @DataLen);
     if ErrCode=ERROR_SUCCESS then begin
      SetLength(Data,DataLen-1);
      Data:=Trim(Data);
      Ports.Add(Data);
      Inc(Index);
     end else Break;
    until ErrCode<>ERROR_SUCCESS;
    Ports.Sort;
    Result:=SysUtils.Trim(Ports.Text);
    if (Length(Delim)>0) and (Delim<>CRLF)
    then Result:=StringReplace(Result,CRLF,Delim,[rfReplaceAll]);
   finally
    RegCloseKey(KeyHandle);
    Ports.Free;
   end;
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

initialization

 uart:=TUart.Create;

finalization

 Kill(TObject(uart));

end.

