 {
 ****************************************************************************
 CRW32 project
 Copyright (C) by Kuryakin Alexey, Sarov, Russia, 2001, <kouriakine@mail.ru>
 Expression evaluator - simple interpreter for calculations.
 Modifications:
 20011215 - Creation (uses CRW16) & test
 20011216 - smartargs
 20011219 - TeeFunction,ee_MaxFuncArgs, test Ok
 20020131 - now actions start from @ or \, so @voice or \voice equals.
 20020202 - change Search methods, inline QuickSearch code in place
 20030326 - Struggle for safety (add some try/except checks)...
 20030328 - Replace TObjectList to TObjectStorage.
 20060303 - Case sensitive, x=@cmd assignment
 20060318 - @mmtimer
 20060325 - @async
 20170818 - getpid, getppid, cpu_count, pidaffinity
 20170926 - rgb,cga2rgb
 20171010 - IsSimpleScript, EvaluateSimpleScript
 20171017 - Optimization + string hashing makes EE *2 times faster
 20171026 - Parent property
 20171116 - Hash become Cardinal, Search & CalcHash modified.
 20171204 - Uses _HASH library and Hasher property, ee_DefaultHasher.
 20180831 - add constant processid
 20190906 - add mksecnow(),cpu_start(),cpu_clock()
 ****************************************************************************
 }

unit _ee; { Expression Evaluator }

{$I _sysdef}

interface

uses
 sysutils, windows, classes, contnrs, math, winsock, _alloc, _fpu, _rtc, _ef, _zm, _str,
 _fio, _snd, _fifo, _dynar, _mmt, _task, _plut, _hash;

const
 ee_MaxVarsLen  = 31;  //    
 ee_MaxNoteLen  = 127; //   /
 ee_MaxToksLen  = 255; //    
 ee_MaxFuncArgs = 15;  //    
 ee_MaxStack    = 255; //     gosub

const                  // Default hash function
 ee_DefaultHasher : THash32Function = Hash32_RS;

type
 {
 Forward declaration
 }
 TExpressionEvaluator = class;
 {
 List of buffers allocated by Allocate/Reallocate/Deallocate calls.
 }
 TeeBufferList = class(TList)
 private
  FOwnsBuffers: Boolean;
 protected
  procedure Notify(Ptr:Pointer; Action:TListNotification); override;
 public
  constructor Create; overload;
  constructor Create(aOwnsBuffers:Boolean); overload;
 public
  property OwnsBuffers:Boolean read FOwnsBuffers write FOwnsBuffers;
 end;
 {
      TExpressionEvaluator
 }
 PeeVariableItem = ^TeeVariableItem;
 TeeVariableItem = packed record // Buffer to store variable
  Hash    : Cardinal;            // Hash for fast search
  Value   : Double;              // Value of variable
  Name    : record end;          // Name is zstring
 end;
 {
        TExpressionEvaluator
 }
 TeeVariableItemList = class(TMasterObject)
 private
  myList       : TeeBufferList;
  myHasher     : THash32Function;
  myParent     : TExpressionEvaluator;
  myCollisions : Cardinal;
  function    GetCount:Integer;
  function    GetCollisions:Cardinal;
  function    GetHasher:THash32Function;
  function    GetItems(aIndex:Integer):PeeVariableItem;
  function    GetHashs(aIndex:Integer):Cardinal;
  function    GetNames(aIndex:Integer):PChar;
  function    GetValues(aIndex:Integer):Double;
  procedure   SetValues(aIndex:Integer; aValue:Double);
 public
  constructor Create(aParent:TExpressionEvaluator);
  destructor  Destroy; override;
 public
  function    ClearVar(aName:PChar):Boolean;
  function    FindVar(aName:PChar):PeeVariableItem;
  function    Search(aKey:PChar; var aIndex:Integer):Boolean;
  function    GetValue(aName:PChar; var aValue:Double):Boolean;
  function    SetValue(aName:PChar; const aValue:Double):Boolean;
  function    GetText(aText:TText):TText;
  procedure   Clear;
 public
  property    Count:Integer                         read GetCount;
  property    Hasher:THash32Function                read GetHasher;
  property    Collisions:Cardinal                   read GetCollisions;
  property    Items[aIndex:Integer]:PeeVariableItem read GetItems; default;
  property    Hashs[aIndex:Integer]:Cardinal        read GetHashs;
  property    Names[aIndex:Integer]:PChar           read GetNames;
  property    Values[aIndex:Integer]:Double         read GetValues write SetValues;
 end;
 {
         
 }
 TeeFunction     = function(ee:TExpressionEvaluator; const Arg:array of Double; Narg:Integer):Double;
 PeeFunctionItem = ^TeeFunctionItem;
 TeeFunctionItem = packed record // Buffer to store functions
  Hash    : Cardinal;            // Hash for fast search
  Narg    : Integer;             // Number of arguments
  Func    : TeeFunction;         // Function pointer
  Name    : record end;          // Name is zstring
  Note    : record end;          // Note is zstring
 end;
 {
        TExpressionEvaluator
 }
 TeeFunctionItemList = class(TMasterObject)
 private
  myList       : TeeBufferList;
  myHasher     : THash32Function;
  myParent     : TExpressionEvaluator;
  myCollisions : Cardinal;
  function    GetCount:Integer;
  function    GetCollisions:Cardinal;
  function    GetHasher:THash32Function;
  function    GetItems(aIndex:Integer):PeeFunctionItem;
  function    GetHashs(aIndex:Integer):Cardinal;
  function    GetNames(aIndex:Integer):PChar;
  function    GetNotes(aIndex:Integer):PChar;
  function    GetNargs(aIndex:Integer):Integer;
 public
  constructor Create(aParent:TExpressionEvaluator);
  destructor  Destroy; override;
 public
  function    ClearFunc(aName:PChar):Boolean;
  function    FindFunc(aName:PChar):PeeFunctionItem;
  function    Search(aKey:PChar; var aIndex:Integer):Boolean;
  function    SetFunc(aName:PChar; aNarg:Integer; aFunc:TeeFunction; aNote:PChar):Boolean;
  function    GetText(aText:TText):TText;
  procedure   Clear;
 public
  property    Count:Integer                         read GetCount;
  property    Hasher:THash32Function                read GetHasher;
  property    Collisions:Cardinal                   read GetCollisions;
  property    Items[aIndex:Integer]:PeeFunctionItem read GetItems; default;
  property    Hashs[aIndex:Integer]:Cardinal        read GetHashs;
  property    Names[aIndex:Integer]:PChar           read GetNames;
  property    Notes[aIndex:Integer]:PChar           read GetNotes;
  property    Nargs[aIndex:Integer]:Integer         read GetNargs;
 end;
 {
        
  -  ,     
   -,   
 }
 TeeAction     = function(ee:TExpressionEvaluator; const aArgs:ShortString):Double;
 PeeActionItem = ^TeeActionItem;
 TeeActionItem = packed record // Buffer to store actions
  Hash   : Cardinal;           // Hash for fast search
  Action : TeeAction;          // Action pointer
  Name   : record end;         // Name is zstring
  Note   : record end;         // Note is zstring
 end;
 {
         
     
 }
 TeeActionItemList = class(TMasterObject)
 private
  myList       : TeeBufferList;
  myHasher     : THash32Function;
  myParent     : TExpressionEvaluator;
  myCollisions : Cardinal;
  function    GetCount:Integer;
  function    GetCollisions:Cardinal;
  function    GetHasher:THash32Function;
  function    GetItems(aIndex:Integer):PeeActionItem;
  function    GetHashs(aIndex:Integer):Cardinal;
  function    GetNames(aIndex:Integer):PChar;
  function    GetNotes(aIndex:Integer):PChar;
 public
  constructor Create(aParent:TExpressionEvaluator);
  destructor  Destroy; override;
 public
  function    ClearAction(aName:PChar):Boolean;
  function    FindAction(aName:PChar):PeeActionItem;
  function    Search(aKey:PChar; var aIndex:Integer):Boolean;
  function    SetAction(aName:PChar; aAction:TeeAction; aNote:PChar):Boolean;
  function    GetText(aText:TText):TText;
  procedure   Clear;
 public
  property    Count:Integer                       read GetCount;
  property    Hasher:THash32Function              read GetHasher;
  property    Collisions:Cardinal                 read GetCollisions;
  property    Items[aIndex:Integer]:PeeActionItem read GetItems; default;
  property    Hashs[aIndex:Integer]:Cardinal      read GetHashs;
  property    Names[aIndex:Integer]:PChar         read GetNames;
  property    Notes[aIndex:Integer]:PChar         read GetNotes;
 end;
 {
 *******************************************************************************
  TExpressionEvaluator   .
  :
  ee:=TExpressionEvaluator.Create;
  if ee.EvaluateExpression('sin(pi/6)*exp(1)+2')=ee_Ok
  then Echo(ee.Answer) else Echo('error '+ee.Status);
  ee.Script:='x=2'+CRLF+'y=2'+CRLF+'z=x*y';
  if ee.RunScript=ee_Ok then Echo(ee.Answer) else Echo('error '+ee.Status);
  Kill(ee);
 *******************************************************************************
       ,   
    TLatch,   Lock/Unlock   
   .         .
 *******************************************************************************
      ,  ,
  ()  .   Script  ,   
 RunScript    , ,  :
    var n,fact
    n=6
    fact=1
    loop:
     gosub mult
     n=n-1
    if gt(n,1) then goto loop
    exit
    mult:
     fact=fact*n
    return
        PreScript.
     -  @  \,      
  - - , ,
  @voice hello world
        ,  
   .    
  .       .
        
          
  .
       actionresult.
   +,-,*,/,^,
    e- ( 1.5E3)   
 ,      $, , $AF
    deg(x)           
    rad(x)           
    sin(x)        
    cos(x)        
    tan(x)        
    asin(x)        
    acos(x)        
    atan(x)        
    sinh(x)        
    cosh(x)        
    tanh(x)        
    exp(x)        
    ln(x)           x
    log(n,x)       x   n
    sqrt(x)       
    int(x)           
    trunc(x)         
    frac(x)        
    round(x)          
    floor(x)         
    ceil(x)          INF
    abs(x)        
    hypot(x,y)    sqrt(x^2+y^2)
    rand()           0  1
    random(a,b)       (a,b)
    sign(x)        (+1,0,-1)
    eq(x,y)        1/0  x=y
    ne(x,y)        1/0  x<>y
    lt(x,y)        1/0  x<y
    gt(x,y)        1/0  x>y
    le(x,y)        1/0  x<=y
    ge(x,y)        1/0  x>=y
    max(x,y)        x  y
    min(x,y)         x  y
    msecnow()           
    secnow()                 
    getticks()         BIOS
    mksecnow()        
    or(x,y)         
    xor(x,y)        
    and(x,y)       
    not(x)         
    bitor(x,y)       
    bitxor(x,y)      
    bitand(x,y)      
    bitnot(x)       
    getbitmask(n)  2^n
    isbit(x,n)       x   n
    gamma(n)      -: gamma(n) = (n-1)!
    isnan(a)      1  a=NAN  0
    isinf(a)      1  a=INF  0
    getpid()      get current process id
    getppid()     get parent process id
    cpu_count()   get CPU count
    cpu_start()   start CPU clocks measure
    cpu_clock()   get CPU clocks since cpu_start
    pidaffinity(p,m) set affinity mask m to process p
  
  1+1
  10*(x=5)                 <== ,   C,    !
  ((1/3)*sin(45))^2
  X=50
  Y=100
  z=hypot(x,y)*rand()
 *******************************************************************************
 }
 EExpressionEvaluator = class(ESoftException);
 TExpressionEvaluator = class(TLatch)
 private
  myVarList    : TeeVariableItemList;               { }
  myConstList  : TeeVariableItemList;               { }
  myLabelList  : TeeVariableItemList;               { }
  myFuncList   : TeeFunctionItemList;               { }
  myActionList : TeeActionItemList;                 { }
  myScript     : TText;                             { }
  myPreScript  : TText;                             {  }
  myPreStatus  : Integer;                           {  }
  myPreErrMsg  : array[0..ee_MaxToksLen] of Char;   {  }
  myStatus     : Integer;                           {  }
  myAnswer     : Double;                            { }
  myMayPrint   : Boolean;                           { }
  myExpression : PChar;                             { }
  myTokenType  : Integer;                           {  }
  myBuffer     : array[0..ee_MaxToksLen] of Char;   { }
  myToken      : array[0..ee_MaxToksLen] of Char;   {  }
  myErrorToken : array[0..ee_MaxToksLen] of Char;   {,  }
  myErrorPos   : Integer;                           { }
  myCustom     : Pointer;                           {   }
  myParent     : TObject;                           {   }
  myLineBuf    : array[0..255] of Char;             { }
  myStack      : array[0..ee_MaxStack-1] of Integer;{  gosub}
  myHasher     : THash32Function;                   { }
  myCpuBase    : Int64;                             {cpu_start,cpu_clock}
  function    GetVarList:TeeVariableItemList;
  function    GetConstList:TeeVariableItemList;
  function    GetLabelList:TeeVariableItemList;
  function    GetFuncList:TeeFunctionItemList;
  function    GetActionList:TeeActionItemList;
  function    GetStatus:Integer;
  function    GetAnswer:Double;
  function    GetMayPrint:Boolean;
  function    GetExpression:PChar;
  function    GetBuffer:PChar;
  function    GetToken:PChar;
  function    GetErrorToken:PChar;
  function    GetErrorPos:Integer;
  function    GetCustom:Pointer;
  procedure   SetCustom(aCustom:Pointer);
  function    GetParent:TObject;
  procedure   SetParent(aParent:TObject);
  procedure   RaiseError(n:Integer);                { }
  procedure   Parse;                                { }
  function    Level1(var r:Double):Boolean;         {  }
  procedure   Level2(var r:Double);
  procedure   Level3(var r:Double);
  procedure   Level4(var r:Double);
  procedure   Level5(var r:Double);
  procedure   Level6(var r:Double);
  function    ExecuteScript(aScript:TText):Integer; {  }
  function    GetScript:LongString;                 {   }
  procedure   SetScript(aLines:LongString);         {   ,  }
  function    GetPreScript:LongString;              { }
  function    GetHasher:THash32Function;
 public
  constructor Create(aHasher:THash32Function=nil);
  destructor  Destroy;override;
  function    SetConst(const aName:ShortString; aValue:Double):Boolean;
  function    SetFunc(const aName:ShortString; aNarg:Integer; aFunc:TeeFunction; const aNote:ShortString):Boolean;
  function    SetAction(const aName:ShortString; aAction:TeeAction; const aNote:ShortString):Boolean;
 public
  property    VarList:TeeVariableItemList   read GetVarList;                 { }
  property    ConstList:TeeVariableItemList read GetConstList;               { }
  property    LabelList:TeeVariableItemList read GetLabelList;               { }
  property    FuncList:TeeFunctionItemList  read GetFuncList;                { }
  property    ActionList:TeeActionItemList  read GetActionList;              { }
  property    Status:Integer                read GetStatus;                  {  }
  property    Answer:Double                 read GetAnswer;                  { }
  property    MayPrint:Boolean              read GetMayPrint;                { }
  property    Expression:PChar              read GetExpression;              { }
  property    Buffer:PChar                  read GetBuffer;                  {  }
  property    Token:PChar                   read GetToken;                   { }
  property    ErrorToken:PChar              read GetErrorToken;              {    }
  property    ErrorPos:Integer              read GetErrorPos;                {   }
  property    Script:LongString             read GetScript write SetScript;  {   }
  property    PreScript:LongString          read GetPreScript;               {  }
  property    Custom:Pointer                read GetCustom write SetCustom;  {   }
  property    Parent:TObject                read GetParent write SetParent;  {   }
  property    RuntimeError:Integer                         write RaiseError; { }
  property    Hasher:THash32Function        read GetHasher;                  { }
 public
  procedure   Clear;                                                       {  }
  function    ClearVar(aName:PChar):Boolean;                               { }
  function    DeclareVar(aName:PChar):Boolean;                             { }
  function    GetValue(aName:PChar; var aValue:Double):Boolean;            { }
  function    SetValue(aName:PChar; const aValue:Double):Boolean;          { }
  function    EvaluateExpression(e:PChar;ForbidEmpty:Boolean=true):Integer;{ }
  function    EvaluateLine(e:PChar):Integer;                               {  ( )}
  function    RunScript:Integer;                                           {  Script}
  procedure   SaveVars(const IniFile,Section:ShortString);                 {   }
  procedure   RestoreVars(const IniFile,Section:ShortString);              {  }
  function    SmartArgs(const Args:ShortString):ShortString;               {      %var  }
 end;

function  NewExpressionEvaluator:TExpressionEvaluator;
procedure Kill(var TheObject:TExpressionEvaluator); overload;

procedure Kill(var TheObject:TeeBufferList); overload;

function NewTeeVariableItem(aName:PChar; aValue:Double; aHasher:THash32Function):PeeVariableItem;
function NewTeeFunctionItem(aName:PChar; aNarg:Integer; aFunc:TeeFunction; aNote:PChar; aHasher:THash32Function):PeeFunctionItem;
function NewTeeActionItem(aName:PChar; aAction:TeeAction; aNote:PChar; aHasher:THash32Function):PeeActionItem;

 {
 ,  ,  
 }
const
 ee_Ok          = 0;        {   }
 ee_Syntax      = 1;        {   }
 ee_UnBalanced  = 2;        {   }
 ee_DivZero     = 3;        {    }
 ee_Unknown     = 4;        {     }
 ee_MaxVars     = 5;        {     }
 ee_BadFunc     = 6;        {   }
 ee_NumArgs     = 7;        {     }
 ee_NoArgs      = 8;        {    }
 ee_Empty       = 9;        {   }
 ee_Label       = 10;       {   }
 ee_VarDecl     = 11;       {    }
 ee_If          = 12;       {    if }
 ee_NilRef      = 13;       {   (nil) }
 ee_BadReturn   = 14;       { Return  GoSub }
 ee_StackOver   = 15;       {    GoSub }
 ee_Exception   = 16;       {     }
 ee_LineTooLong = 17;       {     }
 ee_User        = 18;       {  }

function ee_ErrorMessage(ErrorCode:Integer):ShortString;

 // Check if Script is simple, i.e. can be executed line by line, without precompilation.
function IsSimpleScript(const Script:LongString):Boolean;

 // Evaluate Script text line by line.
function EvaluateSimpleScript(ee:TExpressionEvaluator; const Script:LongString):Integer;

 // Expression Evaluator Benchmark to eastimate performance.
function ExpressionEvaluatorBenchmark(n:Integer=1000000):LongString;

 // Hash collisions counter
const
 ee_HashCollisions : Int64 = 0;

implementation

uses _syscal;

 {
    
 }
const
 id_Error    = 0;  {       }
 id_Variable = 1;  {    }
 id_Delimer  = 2;  {   }
 id_Number   = 3;  {         }

 {
   
 }
const
 WhiteChars  = [' ',#9];                                  {-}
 AlphaChars  = ['a'..'z','A'..'Z','0'..'9','_'];          {}
 DelimChars  = ['+','-','*','/','%','^','(',')',',','=']; {P}
 ArgDelims   = [' ',',',#9,CR];                           {P }
 ActionChars = ['@','\'];                                 { }
 UpChar      = '`';                                       {Uppercase formatter}
 DefaultFmt  = '%g';                                      {Default format}
 DigitChars  = ['0'..'9','.'];                            {Chars uses as format params}
 FormtChars  = DigitChars + ['f','g','e'];                {Chars uses as format specifiers}

 {
 *****************************************
    .
 *****************************************
 }
function deg(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=x[0]*180/pi; end;

function rad(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=x[0]*pi/180; end;

function sin(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.sin(x[0]); end;

function cos(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.cos(x[0]); end;

function tan(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.tan(x[0]); end;

function asin(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.arcsin(x[0]); end;

function acos(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.arccos(x[0]); end;

function atan(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.arctan(x[0]); end;

function sinh(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.sinh(x[0]); end;

function cosh(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.cosh(x[0]); end;

function tanh(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.tanh(x[0]); end;

function exp(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.exp(x[0]); end;

function ln(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.ln(x[0]); end;

function log(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.logn(x[0],x[1]); end;

function sqrt(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.sqrt(x[0]); end;

function int(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.int(x[0]); end;

function trunc(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.trunc(x[0]); end;

function frac(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.frac(x[0]); end;

function round(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.round(x[0]); end;

function floor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.floor(x[0]); end;

function ceil(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.ceil(x[0]); end;

function abs(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.abs(x[0]); end;

function hypot(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.hypot(x[0],x[1]); end;

function rand(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.random; end;

function random(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=x[0]+(x[1]-x[0])*system.random; end;

function sign(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_ef.sign(x[0]); end;

function eq(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(x[0]=x[1]); end;

function ne(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(x[0]<>x[1]); end;

function lt(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(x[0]<x[1]); end;

function gt(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(x[0]>x[1]); end;

function le(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(x[0]<=x[1]); end;

function ge(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(x[0]>=x[1]); end;

function max(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.max(x[0],x[1]); end;

function min(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=math.min(x[0],x[1]); end;

function msecnow(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.msecnow; end;

function secnow(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.msecnow*0.001; end;

function getticks(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=windows.GetTickCount/55; end;

function mksecnow(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.mksecnow; end;

function getclockres(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.GetClockResolution(System.Round(x[0]))/FileTimesPerMSec; end;

function setclockres(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.SetClockResolution(System.Round(x[0]*FileTimesPerMSec))/FileTimesPerMSec; end;

function bor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord((x[0]<>0) or (x[1]<>0)); end;

function bxor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord((x[0]<>0) xor (x[1]<>0)); end;

function band(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord((x[0]<>0) and (x[1]<>0)); end;

function bnot(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(not(x[0]<>0)); end;

function aor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.round(x[0]) or system.round(x[1]); end;

function axor(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.round(x[0]) xor system.round(x[1]); end;

function aand(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=system.round(x[0]) and system.round(x[1]); end;

function anot(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=not system.round(x[0]); end;

function getbitmask(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=DWORD(1) shl system.round(x[0]); end;

function isbit(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(system.round(x[0]) and (DWORD(1) shl system.round(x[1]))<>0); end;

function gamma(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_ef.gamma(x[0]); end;

function isnan(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(_fpu.isnan(x[0])); end;

function isinf(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=ord(_fpu.isinf(x[0])); end;

function getpid(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=Windows.GetCurrentProcessId; end;

function getppid(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_task.FindParentProcessId; end;

function cpu_count(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_task.cpu_count; end;

function cpu_start(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin ee.myCpuBase:=_rtc.ReadTimeStampCounter; Result:=Ord(ee.myCpuBase<>0); end;

function cpu_clock(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.ReadTimeStampCounter-ee.myCpuBase; end;

function pidaffinity(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_task.PidAffinity(system.round(x[0]),system.round(x[1])); end;

function rgb(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_plut.RGB(system.round(x[0]),system.round(x[1]),system.round(x[2])); end;

function cga2rgb(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_plut.CgaToRGBColor(system.round(x[0])); end;

function htonl(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=winsock.htonl(system.round(x[0])); end;

function ntohl(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=winsock.ntohl(system.round(x[0])); end;

function htons(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=winsock.htons(system.round(x[0])); end;

function ntohs(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=winsock.ntohs(system.round(x[0])); end;

function msecrangemin(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.MSecRangeMin; end;

function msecrangemax(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=_rtc.MSecRangeMax; end;

function guilanguage(ee:TExpressionEvaluator; const x:array of Double; narg:Integer):Double;
begin Result:=Ord(_str.Language); end;

 { @voice   }
function act_voice(ee:TExpressionEvaluator; const args:ShortString):double;
begin
 if UsesBlaster then begin
  Result:=1;
  Voice(ee.SmartArgs(args));
 end else Result:=0;
end;

 { @async    FIFO  }
function act_async(ee:TExpressionEvaluator; const args:ShortString):double;
begin
 Result:=0;
 try
  if Length(args)>0 then
  Result:=ord(SystemCalculator.Fifo.PutText(args+CRLF))*Length(args);
 except
  on E:Exception do BugReport(E);
 end;
end;

 { @global     }
function act_global(ee:TExpressionEvaluator; const args:ShortString):double;
var p:PChar; arg:array[0..255] of Char;
begin
 Result:=0;
 try
  p:=StrPCopy(arg,ee.SmartArgs(args)); while p^ in WhiteChars do inc(p);
  if (StrLIComp('@async',p,6)=0) and not (p[6] in AlphaChars) then begin
   inc(p,6); while p^ in WhiteChars do inc(p);
   Result:=act_async(ee,StrPas(p));
   Exit;
  end;
  Result:=SystemCalculator.Eval(StrPas(arg));
 except
  on E:Exception do BugReport(E);
 end;
end;

 { @system     }
function act_system(ee:TExpressionEvaluator; const args:ShortString):double;
var p:PChar; arg:array[0..255] of Char;
begin
 Result:=0;
 try
  p:=StrPCopy(arg,args); while p^ in WhiteChars do inc(p);
  if (StrLIComp('@async',p,6)=0) and not (p[6] in AlphaChars) then begin
   inc(p,6); while p^ in WhiteChars do inc(p);
   Result:=act_async(ee,StrPas(p));
   Exit;
  end;
  Result:=SystemCalculator.Eval(args);
 except
  on E:Exception do BugReport(E);
 end;
end;

 { @echo    }
function act_echo(ee:TExpressionEvaluator; const args:ShortString):double;
var s:ShortString;
begin
 s:=ee.SmartArgs(args);
 Result:=Length(s);
 Echo(s);
end;

 { @mmtimer \   }
function act_mmtimer(ee:TExpressionEvaluator; const args:ShortString):double;
var s:ShortString; n:Integer;
begin
 Result:=0;
 try
  s:=ee.SmartArgs(args);
  if Str2Int(Trim(s),n) then mmTimer.Period:=n;
  Result:=mmTimer.Period;
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
 ******************************************
      
 ******************************************
 }
function StrOk(Str:PChar):Boolean;
begin
 if Assigned(Str) then Result:=(Str[0]<>#0) else Result:=false;
end;

function SortText(const aText:LongString):LongString;
var List:TStringList;
begin
 Result:='';
 try
  List:=TStringList.Create;
  try
   List.Text:=aText;
   List.Sort;
   Result:=List.Text;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
 **************************
 TeeBufferList implementation
 **************************
 }
constructor TeeBufferList.Create;
begin
 inherited Create;
 FOwnsBuffers:=True;
end;

constructor TeeBufferList.Create(aOwnsBuffers:Boolean);
begin
 inherited Create;
 FOwnsBuffers:=aOwnsBuffers;
end;

procedure TeeBufferList.Notify(Ptr: Pointer; Action: TListNotification);
begin
 if OwnsBuffers and (Action=lnDeleted) then Deallocate(Ptr);
 inherited Notify(Ptr,Action);
end;

procedure Kill(var TheObject:TeeBufferList); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
 ******************************
 TeeVariableItem implementation
 ******************************
 }
function NewTeeVariableItem(aName:PChar; aValue:Double; aHasher:THash32Function):PeeVariableItem;
var NameLen:Integer;
begin
 Result:=nil;
 try
  NameLen:=StrLLen(aName,ee_MaxVarsLen);
  if (NameLen>0)
  then Result:=Allocate(sizeof(Result^)+NameLen+1);
  if Assigned(Result) then begin
   Result.Hash:=0;
   Result.Value:=aValue;
   StrLower(StrLCopy(PChar(@Result.Name),aName,NameLen));
   if Assigned(aHasher) then Result.Hash:=aHasher(PChar(@Result.Name),NameLen);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
 **********************************
 TeeVariableItemList implementation
 **********************************
 }
constructor TeeVariableItemList.Create(aParent:TExpressionEvaluator);
begin
 inherited Create;
 Exceptions:=false;
 myList:=TeeBufferList.Create;
 myHasher:=aParent.Hasher;
 myParent:=aParent;
end;

destructor TeeVariableItemList.Destroy;
begin
 Kill(myList);
 inherited Destroy;
end;

function TeeVariableItemList.GetCount:Integer;
begin
 if Assigned(Self) then Result:=myList.Count else Result:=0;
end;

function TeeVariableItemList.GetCollisions:Cardinal;
begin
 if Assigned(Self) then Result:=myCollisions else Result:=0;
end;

function TeeVariableItemList.GetHasher:THash32Function;
begin
 if Assigned(Self) then Result:=myHasher else Result:=nil;
end;

function TeeVariableItemList.GetItems(aIndex:Integer):PeeVariableItem;
var Item:PeeVariableItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 Result:=Item;
end;

function TeeVariableItemList.GetHashs(aIndex:Integer):Cardinal;
var Item:PeeVariableItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=Item.Hash
 else Result:=0;
end;

function TeeVariableItemList.GetNames(aIndex:Integer):PChar;
var Item:PeeVariableItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=PChar(@Item.Name)
 else Result:=nil;
end;

function TeeVariableItemList.GetValues(aIndex:Integer):Double;
var Item:PeeVariableItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=Item.Value
 else Result:=0;
end;

procedure TeeVariableItemList.SetValues(aIndex:Integer; aValue:Double);
var Item:PeeVariableItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Item.Value:=aValue;
end;

function TeeVariableItemList.Search(aKey:PChar; var aIndex:Integer):Boolean;
var Left,Right,Middle,Comparison:Integer; ItemList:PPointerList;
    MiddleItem:PeeVariableItem; MiddleKey:PChar; aHash:Cardinal;
    KeyLen:Integer; KeyBuf:array[0..ee_MaxVarsLen] of char;
begin
 Left:=0;
 Result:=False;
 if Assigned(Self) then
 if Assigned(aKey) then
 if (aKey[0]<>#0) then begin
  KeyLen:=0;
  while (aKey[KeyLen]<>#0) and (KeyLen<ee_MaxVarsLen) do begin
   KeyBuf[KeyLen]:=LoCaseTable[aKey[KeyLen]];
   inc(KeyLen);
  end;
  KeyBuf[KeyLen]:=#0;
//Code above is fast equivalent of call:
//KeyLen:=StrLen(StrLower(StrLCopy(KeyBuf,aKey,ee_MaxVarsLen)));
  if Assigned(myHasher)
  then aHash:=myHasher(KeyBuf,KeyLen)
  else aHash:=0;
  ItemList:=myList.List;
  Right:=myList.Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) shr 1;
   MiddleItem:=ItemList[Middle];
   if Assigned(myHasher) and Assigned(MiddleItem) then begin
    if MiddleItem.Hash>aHash then Comparison:=+1 else
    if MiddleItem.Hash<aHash then Comparison:=-1 else Comparison:=0;
   end else Comparison:=0;
   if Comparison=0 then begin
    if Assigned(MiddleItem)
    then MiddleKey:=PChar(@MiddleItem.Name)
    else MiddleKey:=nil;
    Comparison:=StrComp(MiddleKey,KeyBuf);
    if (Comparison<>0) and Assigned(myHasher) then begin
     Inc(ee_HashCollisions);
     Inc(myCollisions);
    end;
   end;
   if Comparison<0 then Left:=Middle+1 else begin
    Right:=Middle-1;
    if Comparison=0 then begin
     Result:=True;
     Left:=Middle;
    end;
   end;
  end;
 end;
 aIndex:=Left;
end;

function TeeVariableItemList.FindVar(aName:PChar):PeeVariableItem;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then Result:=Items[aIndex] else Result:=nil;
end;

function TeeVariableItemList.ClearVar(aName:PChar):Boolean;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then begin
  myList.Delete(aIndex);
  Result:=true;
 end else Result:=false;
end;

function TeeVariableItemList.GetValue(aName:PChar; var aValue:Double):Boolean;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then begin
  aValue:=Values[aIndex];
  Result:=true;
 end else begin
  aValue:=0;
  Result:=false;
 end;
end;

function TeeVariableItemList.SetValue(aName:PChar; const aValue:Double):Boolean;
var aIndex:Integer; aItem:PeeVariableItem;
begin
 Result:=false;
 if Search(aName,aIndex) then begin
  Values[aIndex]:=aValue;
  Result:=true;
 end else if Assigned(Self) and StrOk(aName) then
 try
  aItem:=NewTeeVariableItem(aName,aValue,myHasher);
  if Assigned(aItem) then myList.Insert(aIndex,aItem);
  Result:=Assigned(aItem);
 except
  on E:Exception do begin
   ErrorReport(E);
  end;
 end;
end;

function TeeVariableItemList.GetText(aText:TText):TText;
var MaxLen,i:Integer; List:TText;
begin
 Result:=aText;
 if Assigned(Self) and Assigned(aText) then
 try
  List:=NewText;
  try
   MaxLen:=0;
   for i:=0 to Count-1 do MaxLen:=math.max(MaxLen,StrLen(Names[i]));
   for i:=0 to Count-1 do List.Addln(Format('%-*s = %g',[MaxLen,Names[i],Values[i]]));
   if Assigned(myHasher) then List.Text:=SortText(List.Text);
   aText.Concat(List);
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TeeVariableItemList.Clear;
begin
 if Assigned(Self) then myList.Clear;
end;

 {
 ****************************
 TFunctionItem implementation
 ****************************
 }
function NewTeeFunctionItem(aName:PChar; aNarg:Integer; aFunc:TeeFunction; aNote:PChar; aHasher:THash32Function):PeeFunctionItem;
var NameLen,NoteLen:Integer;
begin
 Result:=nil;
 try
  NameLen:=StrLLen(aName,ee_MaxVarsLen);
  NoteLen:=StrLLen(aNote,ee_MaxNoteLen);
  if (NameLen>0) and (NoteLen>0) and Assigned(aFunc) and (aNarg in [0..ee_MaxFuncArgs])
  then Result:=Allocate(sizeof(Result^)+NameLen+1+NoteLen+1);
  if Assigned(Result) then begin
   Result.Hash:=0;
   Result.Narg:=aNarg;
   Result.Func:=aFunc;
   StrLower(StrLCopy(PChar(@Result.Name),aName,NameLen));
   StrLCopy(PChar(@Result.Name)+NameLen+1,aNote,NoteLen);
   if Assigned(aHasher) then Result.Hash:=aHasher(PChar(@Result.Name),NameLen); 
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function TeeFunctionItemEval(const Item:PeeFunctionItem; ee:TExpressionEvaluator; const Arg:array of Double):Double;
begin
 if Assigned(Item) and Assigned(Item.Func) and Assigned(ee)
 then Result:=Item.Func(ee,Arg,Item.Narg)
 else Result:=0;
end;

 {
 ********************************
 TFunctionItemList implementation
 ********************************
 }
constructor TeeFunctionItemList.Create(aParent:TExpressionEvaluator);
begin
 inherited Create;
 Exceptions:=false;
 myList:=TeeBufferList.Create;
 myHasher:=aParent.Hasher;
 myParent:=aParent;
end;

destructor TeeFunctionItemList.Destroy;
begin
 Kill(myList);
 inherited Destroy;
end;

function TeeFunctionItemList.GetCount:Integer;
begin
 if Assigned(Self) then Result:=myList.Count else Result:=0;
end;

function TeeFunctionItemList.GetCollisions:Cardinal;
begin
 if Assigned(Self) then Result:=myCollisions else Result:=0;
end;

function TeeFunctionItemList.GetHasher:THash32Function;
begin
 if Assigned(Self) then Result:=myHasher else Result:=nil;
end;

function TeeFunctionItemList.GetItems(aIndex:Integer):PeeFunctionItem;
var Item:PeeFunctionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 Result:=Item;
end;

function TeeFunctionItemList.GetHashs(aIndex:Integer):Cardinal;
var Item:PeeFunctionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=Item.Hash
 else Result:=0;
end;

function TeeFunctionItemList.GetNames(aIndex:Integer):PChar;
var Item:PeeFunctionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=PChar(@Item.Name)
 else Result:=nil;
end;

function TeeFunctionItemList.GetNotes(aIndex:Integer):PChar;
var Item:PeeFunctionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=PChar(@Item.Name)+StrLen(PChar(@Item.Name))+1
 else Result:=nil;
end;

function TeeFunctionItemList.GetNargs(aIndex:Integer):Integer;
var Item:PeeFunctionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=Item.Narg
 else Result:=0;
end;

function TeeFunctionItemList.Search(aKey:PChar; var aIndex:Integer):Boolean;
var Left,Right,Middle,Comparison:Integer; ItemList:PPointerList;
    MiddleItem:PeeFunctionItem; MiddleKey:PChar; aHash:Cardinal;
    KeyLen:Integer; KeyBuf:array[0..ee_MaxVarsLen] of char;
begin
 Left:=0;
 Result:=False;
 if Assigned(Self) then
 if Assigned(aKey) then
 if (aKey[0]<>#0) then begin
  KeyLen:=0;
  while (aKey[KeyLen]<>#0) and (KeyLen<ee_MaxVarsLen) do begin
   KeyBuf[KeyLen]:=LoCaseTable[aKey[KeyLen]];
   inc(KeyLen);
  end;
  KeyBuf[KeyLen]:=#0;
//Code above is fast equivalent of call:
//KeyLen:=StrLen(StrLower(StrLCopy(KeyBuf,aKey,ee_MaxVarsLen)));
  if Assigned(myHasher)
  then aHash:=myHasher(KeyBuf,KeyLen)
  else aHash:=0;
  ItemList:=myList.List;
  Right:=myList.Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) shr 1;
   MiddleItem:=ItemList[Middle];
   if Assigned(myHasher) and Assigned(MiddleItem) then begin
    if MiddleItem.Hash>aHash then Comparison:=+1 else
    if MiddleItem.Hash<aHash then Comparison:=-1 else Comparison:=0;
   end else Comparison:=0;
   if Comparison=0 then begin
    if Assigned(MiddleItem)
    then MiddleKey:=PChar(@MiddleItem.Name)
    else MiddleKey:=nil;
    Comparison:=StrComp(MiddleKey,KeyBuf);
    if (Comparison<>0) and Assigned(myHasher) then begin
     Inc(ee_HashCollisions);
     Inc(myCollisions);
    end;
   end;
   if Comparison<0 then Left:=Middle+1 else begin
    Right:=Middle-1;
    if Comparison=0 then begin
     Result:=True;
     Left:=Middle;
    end;
   end;
  end;
 end;
 aIndex:=Left;
end;

function TeeFunctionItemList.FindFunc(aName:PChar):PeeFunctionItem;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then Result:=Items[aIndex] else Result:=nil;
end;

function TeeFunctionItemList.ClearFunc(aName:PChar):Boolean;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then begin
  myList.Delete(aIndex);
  Result:=true;
 end else Result:=false;
end;

function TeeFunctionItemList.SetFunc(aName:PChar; aNarg:Integer; aFunc:TeeFunction; aNote:PChar):Boolean;
var aIndex:Integer; aItem:PeeFunctionItem;
begin
 Result:=false;
 if Assigned(Self) and StrOk(aName) then begin
  if Search(aName,aIndex) then myList.Delete(aIndex);
  try
   aItem:=NewTeeFunctionItem(aName,aNarg,aFunc,aNote,myHasher);
   if Assigned(aItem) then myList.Insert(aIndex,aItem);
   Result:=Assigned(aItem);
  except
   on E:Exception do begin
    ErrorReport(E);
   end;
  end;
 end;
end;

function TeeFunctionItemList.GetText(aText:TText):TText;
var s:ShortString; c:Char; MaxLen,i,j:Integer; List:TText;
begin
 Result:=aText;
 if Assigned(Self) and Assigned(aText) then
 try
  List:=NewText;
  try
   MaxLen:=0;
   for i:=0 to Count-1 do MaxLen:=math.max(MaxLen,StrLen(Names[i])+2*Nargs[i]+2);
   for i:=0 to Count-1 do begin
    s:=StrPas(Names[i]);
    s:=s+'(';
    c:='a';
    for j:=1 to Nargs[i] do begin
     s:=s+c;
     if j<Nargs[i] then s:=s+',';
     inc(c);
    end;
    s:=s+')';
    List.Addln(Format('%-*s %s',[MaxLen,s,Notes[i]]));
   end;
   if Assigned(myHasher) then List.Text:=SortText(List.Text);
   aText.Concat(List);
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end;
end;

procedure TeeFunctionItemList.Clear;
begin
 if Assigned(Self) then myList.Clear;
end;

 {
 **************************
 TActionItem implementation
 **************************
 }
function NewTeeActionItem(aName:PChar; aAction:TeeAction; aNote:PChar; aHasher:THash32Function):PeeActionItem;
var NameLen,NoteLen:Integer;
begin
 Result:=nil;
 try
  NameLen:=StrLLen(aName,ee_MaxVarsLen);
  NoteLen:=StrLLen(aNote,ee_MaxNoteLen);
  if (NameLen>0) and (NoteLen>0) and Assigned(aAction)
  then Result:=Allocate(sizeof(Result^)+NameLen+1+NoteLen+1);
  if Assigned(Result) then begin
   Result.Hash:=0;
   Result.Action:=aAction;
   StrLower(StrLCopy(PChar(@Result.Name),aName,NameLen));
   StrLCopy(PChar(@Result.Name)+NameLen+1,aNote,NoteLen);
   if Assigned(aHasher) then Result.Hash:=aHasher(PChar(@Result.Name),NameLen); 
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

function TeeActionItemExecute(Item:PeeActionItem; ee:TExpressionEvaluator; const aNarg:ShortString):Double;
begin
 if Assigned(Item) and Assigned(Item.Action) and Assigned(ee)
 then Result:=Item.Action(ee,aNarg)
 else Result:=0;
end;

 {
 ******************************
 TActionItemList implementation
 ******************************
 }
constructor TeeActionItemList.Create(aParent:TExpressionEvaluator);
begin
 inherited Create;
 Exceptions:=false;
 myList:=TeeBufferList.Create;
 myHasher:=aParent.Hasher;
 myParent:=aParent;
end;

destructor TeeActionItemList.Destroy;
begin
 Kill(myList);
 inherited Destroy;
end;

function TeeActionItemList.GetCount:Integer;
begin
 if Assigned(Self) then Result:=myList.Count else Result:=0;
end;

function TeeActionItemList.GetCollisions:Cardinal;
begin
 if Assigned(Self) then Result:=myCollisions else Result:=0;
end;

function TeeActionItemList.GetHasher:THash32Function;
begin
 if Assigned(Self) then Result:=myHasher else Result:=nil;
end;

function TeeActionItemList.GetItems(aIndex:Integer):PeeActionItem;
var Item:PeeActionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 Result:=Item;
end;

function TeeActionItemList.GetHashs(aIndex:Integer):Cardinal;
var Item:PeeActionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=Item.Hash
 else Result:=0;
end;

function TeeActionItemList.GetNames(aIndex:Integer):PChar;
var Item:PeeActionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=PChar(@Item.Name)
 else Result:=nil;
end;

function TeeActionItemList.GetNotes(aIndex:Integer):PChar;
var Item:PeeActionItem;
begin
 if Assigned(Self) and (Cardinal(aIndex)<Cardinal(myList.Count))
 then Item:=myList.List[aIndex]
 else Item:=nil;
 if Assigned(Item)
 then Result:=PChar(@Item.Name)+StrLen(PChar(@Item.Name))+1
 else Result:=nil;
end;

function TeeActionItemList.Search(aKey:PChar; var aIndex:Integer):Boolean;
var Left,Right,Middle,Comparison:Integer; ItemList:PPointerList;
    MiddleItem:PeeActionItem; MiddleKey:PChar; aHash:Cardinal;
    KeyLen:Integer; KeyBuf:array[0..ee_MaxVarsLen] of char;
begin
 Left:=0;
 Result:=False;
 if Assigned(Self) then
 if Assigned(aKey) then
 if (aKey[0]<>#0) then begin
  KeyLen:=0;
  while (aKey[KeyLen]<>#0) and (KeyLen<ee_MaxVarsLen) do begin
   KeyBuf[KeyLen]:=LoCaseTable[aKey[KeyLen]];
   inc(KeyLen);
  end;
  KeyBuf[KeyLen]:=#0;
//Code above is fast equivalent of call:
//KeyLen:=StrLen(StrLower(StrLCopy(KeyBuf,aKey,ee_MaxVarsLen)));
  if Assigned(myHasher)
  then aHash:=myHasher(KeyBuf,KeyLen)
  else aHash:=0;
  ItemList:=myList.List;
  Right:=myList.Count-1;
  while Left<=Right do begin
   Middle:=(Left+Right) shr 1;
   MiddleItem:=ItemList[Middle];
   if Assigned(myHasher) and Assigned(MiddleItem) then begin
    if MiddleItem.Hash>aHash then Comparison:=+1 else
    if MiddleItem.Hash<aHash then Comparison:=-1 else Comparison:=0;
   end else Comparison:=0;
   if Comparison=0 then begin
    if Assigned(MiddleItem)
    then MiddleKey:=PChar(@MiddleItem.Name)
    else MiddleKey:=nil;
    Comparison:=StrComp(MiddleKey,KeyBuf);
    if (Comparison<>0) and Assigned(myHasher) then begin
     Inc(ee_HashCollisions);
     Inc(myCollisions);
    end;
   end;
   if Comparison<0 then Left:=Middle+1 else begin
    Right:=Middle-1;
    if Comparison=0 then begin
     Result:=True;
     Left:=Middle;
    end;
   end;
  end;
 end;
 aIndex:=Left;
end;

function TeeActionItemList.FindAction(aName:PChar):PeeActionItem;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then Result:=Items[aIndex] else Result:=nil;
end;

function TeeActionItemList.ClearAction(aName:PChar):Boolean;
var aIndex:Integer;
begin
 if Search(aName,aIndex) then begin
  myList.Delete(aIndex);
  Result:=true;
 end else Result:=false;
end;

function TeeActionItemList.SetAction(aName:PChar; aAction:TeeAction; aNote:PChar):Boolean;
var aIndex:Integer; aItem:PeeActionItem;
begin
 Result:=false;
 if Assigned(Self) and StrOk(aName) then begin
  if Search(aName,aIndex) then myList.Delete(aIndex);
  try
   aItem:=NewTeeActionItem(aName,aAction,aNote,myHasher);
   if Assigned(aItem) then myList.Insert(aIndex,aItem);
   Result:=Assigned(aItem);
  except
   on E:Exception do begin
    ErrorReport(E);
   end; 
  end;
 end;
end;

procedure TeeActionItemList.Clear;
begin
 if Assigned(Self) then myList.Clear;
end;

function TeeActionItemList.GetText(aText:TText):TText;
var MaxLen,i:Integer; List:TText;
begin
 Result:=aText;
 if Assigned(Self) and Assigned(aText) then
 try
  List:=NewText;
  try
   MaxLen:=0;
   for i:=0 to Count-1 do MaxLen:=math.max(MaxLen,StrLen(Names[i]));
   for i:=0 to Count-1 do List.Addln(Format('@%-*s - %s',[MaxLen+1,Names[i],Notes[i]]));
   if Assigned(myHasher) then List.Text:=SortText(List.Text);
   aText.Concat(List);
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E,Self);
 end; 
end;

 {
 ***********************************
 TExpressionEvaluator implementation
 ***********************************
 }
constructor TExpressionEvaluator.Create(aHasher:THash32Function);
begin
 inherited Create;
 Exceptions:=false;
 if not Assigned(aHasher) then aHasher:=ee_DefaultHasher;
 myHasher:=aHasher;
 { }
 myVarList:=TeeVariableItemList.Create(Self);   myVarList.Master:=myVarList;
 myConstList:=TeeVariableItemList.Create(Self); myConstList.Master:=myConstList;
 myLabelList:=TeeVariableItemList.Create(Self); myLabelList.Master:=myLabelList;
 myFuncList:=TeeFunctionItemList.Create(Self);  myFuncList.Master:=myFuncList;
 myActionList:=TeeActionItemList.Create(Self);  myActionList.Master:=myActionList;
 myScript:=NewText;                             myScript.Master:=myScript;
 myPreScript:=NewText;                          myPreScript.Master:=myPreScript;
 myPreStatus:=ee_Ok;
 myPreErrMsg[0]:=#0;
 myCustom:=nil;
 myParent:=nil;
 { }
 Clear;
 { }
 SetConst('pi',          3.14159265358979323846);
 SetConst('e',           2.71828182845904523536);
 SetConst('false',       0);
 SetConst('true',        1);
 SetConst('macheps',     macheps);
 SetConst('_nan',        _nan);
 SetConst('_inf',        _plusinf);
 SetConst('_minusinf',   _minusinf);
 SetConst('minshortint', Low(ShortInt));
 SetConst('maxshortint', High(ShortInt));
 SetConst('minsmallint', Low(SmallInt));
 SetConst('maxsmallint', High(SmallInt));
 SetConst('minlongint',  Low(LongInt));
 SetConst('maxlongint',  High(LongInt));
 SetConst('minint',      Low(Integer));
 SetConst('maxint',      High(Integer));
 SetConst('maxbyte',     High(Byte));
 SetConst('maxword',     High(Word));
 SetConst('maxdword',    High(DWORD));
 SetConst('maxlongword', High(LongWord));
 SetConst('minsingle',   MinSingle);
 SetConst('maxsingle',   MaxSingle);
 SetConst('mindouble',   MinDouble);
 SetConst('maxdouble',   MaxDouble);
 SetConst('processid',   GetCurrentProcessId);
 SetConst('isunix',      Ord(IsUnix));
 SetConst('islinux',     Ord(IsLinux));
 SetConst('iswindows',   Ord(IsWindows));
 SetConst('lng_unknown', Ord(lng_Unknown));
 SetConst('lng_english', Ord(lng_English));
 SetConst('lng_russian', Ord(lng_Russian));
 SetConst('cp_utf7',     CP_UTF7);
 SetConst('cp_utf8',     CP_UTF8);
 SetConst('cp_none',     CP_NONE);
 { }
 SetFunc('sin',          1, sin,          RusEng('',                           'sine'));
 SetFunc('cos',          1, cos,          RusEng('',                         'cosine'));
 SetFunc('tan',          1, tan,          RusEng('',                         'tangent'));
 SetFunc('asin',         1, asin,         RusEng(' ',                  'inverted sine'));
 SetFunc('acos',         1, acos,         RusEng(' ',                'inverted cosine'));
 SetFunc('atan',         1, atan,         RusEng(' ',                'inverted tangent'));
 SetFunc('sinh',         1, sinh,         RusEng(' ',           'sine hyperbolic'));
 SetFunc('cosh',         1, cosh,         RusEng(' ',         'cosine hyperbolic'));
 SetFunc('tanh',         1, tanh,         RusEng(' ',         'tangent hyperbolic'));
 SetFunc('exp',          1, exp,          RusEng(' ',          'napierian exponent'));
 SetFunc('ln',           1, ln,           RusEng(' ',            'napierian logarithm'));
 SetFunc('log',          2, log,          RusEng(' b   a',       'logarithm of b with base a'));
 SetFunc('sqrt',         1, sqrt,         RusEng('',                          'radical'));
 SetFunc('int',          1, int,          RusEng('   0',          'truncate to 0'));
 SetFunc('trunc',        1, trunc,        RusEng('   0',          'truncate to 0'));
 SetFunc('frac',         1, frac,         RusEng('  ',             'fractional part'));
 SetFunc('round',        1, round,        RusEng('   ', 'round to nearest integer'));
 SetFunc('floor',        1, floor,        RusEng('   0',          'truncate to 0'));
 SetFunc('ceil',         1, ceil,         RusEng('   INF',        'truncate to INF'));
 SetFunc('abs',          1, abs,          RusEng('',                          'absolute value'));
 SetFunc('deg',          1, deg,          RusEng('   ',        'radians to degrees'));
 SetFunc('rad',          1, rad,          RusEng('   ',      'degrees to radians'));
 SetFunc('hypot',        2, hypot,        RusEng('sqrt(a^2+b^2)',                   'sqrt(a^2+b^2)'));
 SetFunc('rand',         0, rand,         RusEng('   0  1',       'random value from 0 to 1'));
 SetFunc('random',       2, random,       RusEng('   a  b',       'random value from a to b'));
 SetFunc('sign',         1, sign,         RusEng(' : -1,0,1',           'signum: -1,0,1'));
 SetFunc('eq',           2, eq,           RusEng('1  a=b  0',                '1 if a=b,  0 otherwise'));
 SetFunc('ne',           2, ne,           RusEng('1  a<>b  0',               '1 if a<>b, 0 otherwise'));
 SetFunc('lt',           2, lt,           RusEng('1  a<b  0',                '1 if a<b,  0 otherwise'));
 SetFunc('gt',           2, gt,           RusEng('1  a>b  0',                '1 if a>b,  0 otherwise'));
 SetFunc('le',           2, le,           RusEng('1  a<=b  0',               '1 if a<=b, 0 otherwise'));
 SetFunc('ge',           2, ge,           RusEng('1  a>=b  0',               '1 if a>=b, 0 otherwise'));
 SetFunc('max',          2, max,          RusEng('  a,b',             'max of a,b'));
 SetFunc('min',          2, min,          RusEng('  a,b',              'min of a,b'));
 SetFunc('msecnow',      0, msecnow,      RusEng('  .  ..',       'time in msec from Xmas'));
 SetFunc('secnow',       0, secnow,       RusEng('  .  ..',            'time in sec from Xmas'));
 SetFunc('getticks',     0, getticks,     RusEng(' BIOS  ',              'time in BIOS ticks'));
 SetFunc('mksecnow',     0, mksecnow,     RusEng('  .  ',     'time in mksec from start'));
 SetFunc('getclockres',  1, getclockres,  RusEng('    ms',  'get timer resolution in ms'));
 SetFunc('setclockres',  1, setclockres,  RusEng('    ms',  'set timer resolution in ms'));
 SetFunc('or',           2, bor,          RusEng('  ',    'logical OR'));
 SetFunc('xor',          2, bxor,         RusEng('  ',      'logical XOR'));
 SetFunc('and',          2, band,         RusEng(' ',                    'logical AND'));
 SetFunc('not',          1, bnot,         RusEng(' ',             'logical inversion'));
 SetFunc('bitor',        2, aor,          RusEng('  ',      'arithmetical OR'));
 SetFunc('bitxor',       2, axor,         RusEng('  ',        'arithmetical XOR'));
 SetFunc('bitand',       2, aand,         RusEng(' ',                      'arithmetical AND'));
 SetFunc('bitnot',       1, anot,         RusEng(' ',               'arithmetical inversion'));
 SetFunc('getbitmask',   1, getbitmask,   RusEng(' 2^a',                       '2^int(a)'));
 SetFunc('isbit',        2, isbit,        RusEng('1   a   b  0',     '1 if a''s bit b <> 0, 0 otherwise'));
 SetFunc('gamma',        1, gamma,        RusEng('-: gamma(a)=(a-1)!',  'gamma(a)=(a-1)!'));
 SetFunc('isnan',        1, isnan,        RusEng('1  =NAN  0',              '1 if =NAN, 0 otherwise'));
 SetFunc('isinf',        1, isinf,        RusEng('1  =INF  0',              '1 if =INF, 0 otherwise'));
 SetFunc('getpid',       0, getpid,       RusEng('  ',         'get current process ID'));
 SetFunc('getppid',      0, getppid,      RusEng('  ',    'get parent process ID'));
 SetFunc('cpu_count',    0, cpu_count,    RusEng('  ',       'get number of processors'));
 SetFunc('cpu_start',    0, cpu_start,    RusEng('   CPU',   'start CPU clock counter measure'));
 SetFunc('cpu_clock',    0, cpu_clock,    RusEng(' CPU   cpu_start', 'CPU clock counter since cpu_start'));
 SetFunc('pidaffinity',  2, pidaffinity,  RusEng('    CPU',  'set process affinity mask'));
 SetFunc('rgb',          3, rgb,          RusEng('  (R,G,B)=(a,b,c)',  'compose true color (R,G,B)=(a,b,c)'));
 SetFunc('cga2rgb',      1, cga2rgb,      RusEng('RGB  CGA(a)',                 'RGB true color of CGA(a)'));
 SetFunc('htonl',        1, htonl,        RusEng('host to net long (a)',            'host to net long (a)'));
 SetFunc('ntohl',        1, ntohl,        RusEng('net to host long (a)',            'net to host long (a)'));
 SetFunc('htons',        1, htons,        RusEng('host to net short (a)',           'host to net short (a)'));
 SetFunc('ntohs',        1, ntohs,        RusEng('net to host short (a)',           'net to host short (a)'));
 SetFunc('msecrangemin', 0, msecrangemin, RusEng('  msecnow',         'low range of msecnow'));
 SetFunc('msecrangemax', 0, msecrangemax, RusEng('  msecnow',        'high range of msecnow'));
 SetFunc('guilanguage',  0, guilanguage,  RusEng('  GUI','GUI language code')+' (lng_unknown,lng_english,lng_russian)');
 SetAction('voice',      act_voice,       RusEng('  wav ','play sound wav files'));
 SetAction('async',      act_async,       RusEng('   ( ).','Asynchronous command (thru queue).'));
 SetAction('global',     act_global,      RusEng('    ','Evaluate expression in system calculator'));
 SetAction('system',     act_system,      RusEng('    ','Evaluate expression in system calculator'));
 SetAction('echo',       act_echo,        RusEng('   ','Output message to console'));
 SetAction('mmtimer',    act_mmtimer,     RusEng('\   , [ms].','Get\set multimedia timer period, [ms].'));
end;

 {
   
 }
destructor TExpressionEvaluator.Destroy;
begin
 Kill(TObject(myVarList));
 Kill(TObject(myConstList));
 Kill(TObject(myLabelList));
 Kill(TObject(myFuncList));
 Kill(TObject(myActionList));
 Kill(myScript);
 Kill(myPreScript);
 inherited Destroy;
end;

function TExpressionEvaluator.SetConst(const aName:ShortString; aValue:Double):Boolean;
var buf:array[0..ee_MaxToksLen] of Char;
begin
 if Assigned(Self)
 then Result:=ConstList.SetValue(StrPCopy(buf,aName),aValue)
 else Result:=false;
end;

function TExpressionEvaluator.SetFunc(const aName:ShortString; aNarg:Integer; aFunc:TeeFunction; const aNote:ShortString):Boolean;
var buf1,buf2:array[0..ee_MaxToksLen] of Char;
begin
 if Assigned(Self)
 then Result:=FuncList.SetFunc(StrPCopy(buf1,aName), aNarg, aFunc, StrPCopy(buf2,aNote))
 else Result:=false;
end;

function TExpressionEvaluator.SetAction(const aName:ShortString; aAction:TeeAction; const aNote:ShortString):Boolean;
var buf1,buf2:array[0..ee_MaxToksLen] of Char;
begin
 if Assigned(Self)
 then Result:=ActionList.SetAction(StrPCopy(buf1,aName), aAction, StrPCopy(buf2,aNote))
 else Result:=false;
end;

function TExpressionEvaluator.GetVarList:TeeVariableItemList;
begin
 if Assigned(Self) then Result:=myVarList else Result:=nil;
end;

function TExpressionEvaluator.GetConstList:TeeVariableItemList;
begin
 if Assigned(Self) then Result:=myConstList else Result:=nil;
end;

function TExpressionEvaluator.GetLabelList:TeeVariableItemList;
begin
 if Assigned(Self) then Result:=myLabelList else Result:=nil;
end;

function TExpressionEvaluator.GetFuncList:TeeFunctionItemList;
begin
 if Assigned(Self) then Result:=myFuncList else Result:=nil;
end;

function TExpressionEvaluator.GetActionList:TeeActionItemList;
begin
 if Assigned(Self) then Result:=myActionList else Result:=nil;
end;

function TExpressionEvaluator.GetStatus:Integer;
begin
 if Assigned(Self) then Result:=myStatus else Result:=ee_NilRef;
end;

function TExpressionEvaluator.GetAnswer:Double;
begin
 if Assigned(Self) then Result:=myAnswer else Result:=0;
end;

function TExpressionEvaluator.GetMayPrint:Boolean;
begin
 if Assigned(Self) then Result:=myMayPrint else Result:=false;
end;

function TExpressionEvaluator.GetExpression:PChar;
begin
 if Assigned(Self) then Result:=myExpression else Result:=nil;
end;

function TExpressionEvaluator.GetBuffer:PChar;
begin
 if Assigned(Self) then Result:=myBuffer else Result:=nil;
end;

function TExpressionEvaluator.GetToken:PChar;
begin
 if Assigned(Self) then Result:=myToken else Result:=nil;
end;

function TExpressionEvaluator.GetErrorToken:PChar;
begin
 if Assigned(Self) then Result:=myErrorToken else Result:=nil;
end;

function TExpressionEvaluator.GetErrorPos:Integer;
begin
 if Assigned(Self) then Result:=myErrorPos else Result:=0;
end;

function TExpressionEvaluator.GetCustom:Pointer;
begin
 if Assigned(Self) then Result:=myCustom else Result:=nil;
end;

procedure TExpressionEvaluator.SetCustom(aCustom:Pointer);
begin
 if Assigned(Self) then myCustom:=aCustom;
end;

function TExpressionEvaluator.GetParent:TObject;
begin
 if Assigned(Self) then Result:=myParent else Result:=nil;
end;

procedure TExpressionEvaluator.SetParent(aParent:TObject);
begin
 if Assigned(Self) then myParent:=aParent;
end;

function TExpressionEvaluator.GetHasher:THash32Function;
begin
 if Assigned(Self) then Result:=myHasher else Result:=nil;
end;

 {
      
 }
procedure TExpressionEvaluator.Clear;
begin
 if Assigned(Self) then begin
  myStatus:=ee_Ok;
  myAnswer:=0;
  myMayPrint:=false;
  myExpression:=@myBuffer;
  myTokenType:=id_Error;
  myBuffer[0]:=#0;
  myToken[0]:=#0;
  myErrorToken[0]:=#0;
  myErrorPos:=0;
 end;
end;

 {
       n
 }
procedure TExpressionEvaluator.RaiseError(n:Integer);
begin
 if Assigned(Self) then begin
  myErrorPos:=math.max(0,LongWord(myExpression)-LongWord(@myBuffer)-1);
  StrLCopy(myErrorToken,myToken,ee_MaxToksLen);
  myStatus:=n;
  RAISE EExpressionEvaluator.CreateFmt('Expression evaluator exception: "%s".',
                                       [ee_ErrorMessage(myStatus)]);
 end; 
end;

 {
  ,  false,    
 }
function TExpressionEvaluator.ClearVar(aName:PChar):Boolean;
begin
 if Assigned(Self) then Result:=myVarList.ClearVar(aName) else Result:=false;
end;

 {
   name ,   .
    ,     .
  true     name    .
 }
function TExpressionEvaluator.DeclareVar(aName:PChar):Boolean;
var aValue:Double;
begin
 if Assigned(Self) then begin
  Result:=true;
  if not myVarList.GetValue(aName,aValue) then begin
   aValue:=0;
   Result:=myVarList.SetValue(aName,aValue);
  end;
 end else Result:=false;
end;

 {
     true   
     .
 }
function TExpressionEvaluator.GetValue(aName:PChar; var aValue:Double):Boolean;
begin
 if Assigned(Self)
 then Result:=myVarList.GetValue(aName,aValue) or myConstList.GetValue(aName,aValue)
 else Result:=false;
end;

 {
     name     
 ,    .
 }
function TExpressionEvaluator.SetValue(aName:PChar; const aValue:Double):Boolean;
begin
 if Assigned(Self) then Result:=myVarList.SetValue(aName,aValue) else Result:=false;
end;

 {
     Expression      
 ,   .     Token.
    ,  .
       .
 pression        ,
       .
 }
procedure TExpressionEvaluator.Parse;
var t:PChar;
begin
 if Assigned(Self) then begin
  myTokenType:=id_Error;
  t:=myToken;
  while myExpression^ in WhiteChars do inc(myExpression);
  if myExpression^ in (DelimChars+ActionChars) then begin
   myTokenType:=id_Delimer;
   t^:=myExpression^;
   inc(t);
   inc(myExpression);
  end
  else if myExpression^ in ['0'..'9','.','$'] then begin
   myTokenType:=id_Number;
   if myExpression^='$' then begin
    t^:=myExpression^;
    inc(t);
    inc(myExpression);
    while myExpression^ in ['0'..'9','a'..'f','A'..'F'] do begin
     t^:=myExpression^;
     inc(t);
     inc(myExpression);
    end;
   end else begin
    while myExpression^ in ['0'..'9','.'] do begin
     t^:=myExpression^;
     inc(t);
     inc(myExpression);
    end;
    if myExpression^ in ['e','E'] then begin
     t^:=myExpression^;
     inc(t);
     inc(myExpression);
     if myExpression^ in ['+','-'] then begin
      t^:=myExpression^;
      inc(t);
      inc(myExpression);
     end;
     while myExpression^ in ['0'..'9'] do begin
      t^:=myExpression^;
      inc(t);
      inc(myExpression);
     end;
    end;
   end;
  end
  else if myExpression^ in AlphaChars then begin
   myTokenType:=id_Variable;
   while myExpression^ in AlphaChars do begin
    t^:=myExpression^;
    inc(t);
    inc(myExpression);
   end;
  end
  else if myExpression^<>#0 then begin
   t^:=myExpression^;
   inc(t);
   inc(myExpression);
   RaiseError(ee_Syntax);
  end;
  t^:=#0;
  while myExpression^ in WhiteChars do inc(myExpression);
 end;
end;

 {
    ,  var.
  true      (   ).
 }
function TExpressionEvaluator.Level1(var r:Double):Boolean;
var t:array[0..ee_MaxVarsLen] of Char;
begin
 Result:=false;
 if Assigned(Self) then begin
  Result:=(myToken[0] in ActionChars);
  if myTokenType=id_Variable then
  if StrIComp('var',myToken)=0 then begin
   while true do begin
    Parse;
    if (myTokenType<>id_Variable) or (myToken[0]=#0) or not DeclareVar(myToken)
    then RaiseError(ee_VarDecl);
    Parse;
    case myToken[0] of
     #0  : break;
     ',' : continue;
     else RaiseError(ee_VarDecl);
    end;
   end;
   Result:=true;
   exit;
  end else
  if myExpression^='=' then begin
   StrLCopy(t,myToken,ee_MaxVarsLen);
   Parse;
   Parse;
   if myToken[0]=#0 then begin
    ClearVar(t);
    Result:=true;
    exit;
   end;
   Level2(r);
   if not SetValue(t,r) then begin
    Result:=false;
    RaiseError(ee_MaxVars);
    exit;
   end;
   Result:=true;
   exit;
  end;
  Level2(r);
 end;
end;

 {
     +  -.
 }
procedure TExpressionEvaluator.Level2(var r:Double);
var t:Double; c:Char;
begin
 if Assigned(Self) then begin
  t:=0;
  Level3(r);
  while myToken[0] in ['+','-'] do begin
   c:=myToken[0];
   Parse;
   Level3(t);
   case c of
    '+':r:=r+t;
    '-':r:=r-t;
   end;
  end;
 end;
end;


 {
    *, /, %.
            
  "" - NAN,INF.
 }
procedure TExpressionEvaluator.Level3(var r:Double);
var t:Double; c:Char;
begin
 if Assigned(Self) then begin
  Level4(r);
  while myToken[0] in ['*','/','%'] do begin
   c:=myToken[0];
   Parse;
   Level4(t);
   {if (t=0) and (c<>'*') then RaiseError(ee_DivZero);}
   case c of
    '*': r:=r*t;
    '/': r:=r/t;
    '%': r:=system.frac(r/t)*t;
   end;
  end;
 end;
end;

 {
    " ".
 }
procedure TExpressionEvaluator.Level4(var r:Double);
var t:Double;
begin
 if Assigned(Self) then begin
  Level5(r);
  if myToken[0]='^' then begin
   Parse;
   Level5(t);
   r:=Power(r,t);
  end;
 end;
end;


 {
     +  -.
 }
procedure TExpressionEvaluator.Level5(var r:Double);
var c:Char;
begin
 if Assigned(Self) then begin
  c:=#0;
  if myToken[0] in ['+','-'] then begin
   c:=myToken[0];
   Parse;
  end;
  Level6(r);
  if c='-' then r:=-r;
 end;
end;

 {
   , , , .
 }
procedure TExpressionEvaluator.Level6(var r:Double);
var n:Integer;  a:array[0..ee_MaxFuncArgs] of Double; Func:PeeFunctionItem;
    Action:PeeActionItem;
begin
 if Assigned(Self) then begin
  if myToken[0] in ActionChars then begin
   Parse;
   if myToken[0]=#0 then RaiseError(ee_Syntax);
   if myTokenType<>id_Variable then RaiseError(ee_Syntax);
   Action:=myActionList.FindAction(myToken);
   if not Assigned(Action) then RaiseError(ee_Syntax);
   while myExpression^ in WhiteChars do inc(myExpression);
   r:=TeeActionItemExecute(Action,Self,StrPas(myExpression));
   myExpression:=StrEnd(myExpression);
   SetValue('actionresult',r);
   Parse;
  end else
  if myToken[0]='(' then begin
   Parse;
   if myToken[0]=')' then RaiseError(ee_NoArgs);
   Level1(r);
   if myToken[0]<>')'then RaiseError(ee_UnBalanced);
   Parse;
  end else begin
   if myTokenType=id_Number then begin
    if not atof(myToken,r) then RaiseError(ee_Syntax);
    Parse;
   end
   else if myTokenType=id_Variable then begin
    if myExpression^='(' then begin
     Func:=myFuncList.FindFunc(myToken);
     if Assigned(Func) then begin
      Parse;
      n:=0;
      repeat
       Parse;
       case myToken[0] of
        ')':break;
        ',':RaiseError(ee_NoArgs);
       end;
       a[n]:=0;
       Level1(a[n]);
       inc(n);
      until  (n>ee_MaxFuncArgs) or (myToken[0]<>',') or (Func.Narg=n);
      if myToken[0]<>')' then RaiseError(ee_Unbalanced);
      Parse;
      if n<>Func.Narg then begin
       StrLCopy(myToken,PChar(@Func.Name),ee_MaxVarsLen);
       RaiseError(ee_NumArgs);
      end;
      r:=TeeFunctionItemEval(Func,Self,a);
      exit;
     end else RaiseError(ee_BadFunc);
    end
    else if not GetValue(myToken,r) then RaiseError(ee_Unknown);
    Parse;
   end else RaiseError(ee_Syntax);
  end;
 end;
end;

 {
    e      myResult.
  a=true,    
  ee_Ok,      .
    ,  ForbidEmpty=true.
 }
function TExpressionEvaluator.EvaluateExpression(e:PChar;ForbidEmpty:Boolean):Integer;
begin
 Result:=ee_NilRef;
 if Assigned(Self) then
 try
  Clear;                                              { }
  try                                                 { }
   if not Assigned(e) then RaiseError(ee_Empty);      { nil }
   while e^ in WhiteChars do inc(e);                  {  }
   myExpression:=StrLCopy(myBuffer,e,ee_MaxToksLen);  {   }
   Parse;                                             { }
   if myToken[0]=#0 then begin                        {  ,}
    if ForbidEmpty then RaiseError(ee_Empty);         { -}
    Result:=myStatus;                                 {   }
    exit;
   end;
   myMayPrint:=not Level1(myAnswer);                  { }
   if myToken[0]<>#0 then RaiseError(ee_Syntax);      {  -}
   Result:=myStatus;                                  { }
  except                                              {   }
   on EExpressionEvaluator do begin
    myMayPrint:=true;                                 { }
    Result:=myStatus;                                 { }
   end;
   else RAISE;
  end;
 except
  on E:Exception do begin
   myErrorPos:=0;
   myStatus:=ee_Exception;
   StrLCopy(myErrorToken,myToken,ee_MaxToksLen);
   ErrorReport(E);
   myMayPrint:=true;
   Result:=myStatus;
  end;
 end;
end;

 {
    e.
    EvaluateExpression    -   .
 }
function TExpressionEvaluator.EvaluateLine(e:PChar):Integer;
begin
 Result:=EvaluateExpression(e,false);
end;

 {
     Daqsic
 }
function TExpressionEvaluator.GetScript:LongString;
begin
 if Assigned(Self) then Result:=myScript.Text else Result:='';
end;

 {
       
         
   
 1. 
 2.  
 3.  
 3.        
 }
procedure TExpressionEvaluator.SetScript(aLines:LongString);
var i,p,n:Integer; s,lab:ShortString; v:Double;
 function MayBeLabel(const s:ShortString;p:Integer):Boolean;
 var i:Integer;
 begin
  MayBeLabel:=false;
  if p>1 then begin
   i:=1;
   while (i<p) and (s[i] in WhiteChars) do inc(i);
   if i<p then begin
    while (i<p) and (s[i] in AlphaChars) do inc(i);
    MayBeLabel:=(i=p);
   end;
  end;
 end;
begin
 if Assigned(Self) then
 try
  myScript.Count:=0;                       {  }
  myPreScript.Count:=0;                    {  }
  myLabelList.Clear;                       {  }
  myScript.Text:=aLines;                   {   }
  myPreStatus:=ee_Ok;                      {   }
  myPreErrMsg[0]:=#0;                      {  }
  n:=0;                                    {   }
  for i:=0 to myScript.Count-1 do begin    {    }
   s:=myScript[i];                         {   }
   p:=pos(';',s);                          {  }
   if p>0 then delete(s,p,length(s)-p+1);  {    }
   s:=TrimChars(s,WhiteChars,WhiteChars);  {  }
   if s='' then continue;                  {  }
   p:=pos(':',s);                          {   }
   if MayBeLabel(s,p) then begin           {   }
    lab:=copy(s,1,p-1);                    {  }
    delete(s,1,p);                         {    }
    s:=TrimChars(s,WhiteChars,WhiteChars); {  }
    try                                    {  }
     myExpression:=StrPCopy(myBuffer,lab); { . }
     Parse;                                { }
     if (myTokenType<>id_Variable) or      {   }
        (myToken[0]=#0) or                 {   }
        (myExpression[0]<>#0) or           {    }
        (myLabelList.GetValue(myToken,v))  {   }
     then RaiseError(ee_Label);            {   }
     v:=n;                                 {   }
     if not myLabelList.SetValue(myToken,v){    }
     then RaiseError(ee_Label);            { }
    except                                 {   :}
     on EExpressionEvaluator do begin      {  }
      myPreScript.Count:=0;                {  }
      myLabelList.Clear;                   {  }
      myPreStatus:=myStatus;               {   }
      StrPCopy(myPreErrMsg,myScript[i]);   {  }
      exit;                                {}
     end;
     else RAISE;                           { }
    end;
   end;
   myPreScript.Addln(s);                   {   }
   inc(n);                                 {     }
  end;
 except
  on E:Exception do begin
   myPreScript.Count:=0;                   {  }
   myLabelList.Clear;                      {  }
   myPreStatus:=ee_Exception;              {   }
   StrPCopy(myPreErrMsg,E.Message);        {  }
   ErrorReport(E);
  end;
 end;
end;

 {
     - ,  
 }
function TExpressionEvaluator.GetPreScript:LongString;
begin
 if Assigned(Self) then Result:=myPreScript.Text else Result:='';
end;

 {
        
   DAQsic (Basic  DAQ).
  -     
 DAQ       .
       Daqsic :
  1.    'a'..'z', '0'..'9', '.', '_', ' ', Tab,
     '+', '-', '*', '/', '%', '^', '(', ')', ',', '=', ':',';'
  2.    - (Double).
         .
      , ,  , 
     ,   , ,   if..then.. ,
         goto.
         if  goto.
  3.  Daqsic      
      :  ;
      ,      .
     (     ).
            .
  4.      ';'.
        .
  5.     ':',    
       ':'   .     
     'a'..'z', '0'..'9', '_' ,       '_'.
            .
  6.  .     
     'a'..'z', '0'..'9', '_' ,       '_'.
           
     =
           
     =
          
     var 1,2...
      var      0, 
       ,      
     .
  7.    
     if()then  
     if  then 
          0,   .
          0,   .
  8.      
     goto 
          :   
      .
  9.     +,-,*,/,%,^,,
      , ,   -   
  10.       ,
           0  7, ,
     time()
     hypot(x,y)
 }
function TExpressionEvaluator.RunScript:Integer;
begin
 Result:=ee_NilRef;
 if Assigned(Self) then
 try
  Clear;                                  { }
  if myPreStatus<>ee_Ok then begin        {  }
   myStatus:=myPreStatus;                 {    }
   StrCopy(myBuffer,myPreErrMsg);         { }
   myMayPrint:=true;                      { }
   Result:=myStatus;                      { }
  end else begin                          {   ,}
   Result:=ExecuteScript(myPreScript);    {  }
  end;
 except
  on E:Exception do begin
   myErrorPos:=0;
   myStatus:=ee_Exception;
   StrLCopy(myErrorToken,myToken,ee_MaxToksLen);
   ErrorReport(E);
   myMayPrint:=true;
   Result:=myStatus;
  end;
 end;
end;

 {
    
   
    
      
        
        
   ,  ,     ..
 }
function TExpressionEvaluator.ExecuteScript(aScript:TText):Integer;
label ExtractLine,CheckIf,CheckGoto,CheckGoSub,CheckReturn,CheckExit,
      EvalExpression,NextLine;
var LineNum,StackPtr,isOk:Integer; Line:ShortString; Cur,P:PChar; nl:Double;
begin
 Result:=ee_NilRef;
 if Assigned(Self) then
 try
  Result:=ee_Ok;
  Clear;                                   {  }
  if Assigned(aScript) then begin
   LineNum:=0;                             {  }
   StackPtr:=Low(myStack);                 {  }
   while LineNum<aScript.Count do begin
    {
      
    }
   ExtractLine:
    Line:=aScript[LineNum];                {  }
    if Line='' then goto NextLine;         {,   }
    Cur:=StrPCopy(myLineBuf,Line);         {   }
    while Cur^ in WhiteChars do inc(Cur);  {  }
    if Cur^=#0 then goto NextLine;         {,   }
    {
     if   "if  then ",  -
     (=false, =true).
      -  "if "  "if("   .
    }
   CheckIf:
    if (StrLIComp(Cur,'if',2)=0) and (Cur[2] in (WhiteChars+['('])) then begin
     P:=StrIPos(Cur,'then');
     if (P<>nil) and (P[4] in (WhiteChars+[#0])) then begin
      P^:=#0;                              {    then}
      isOk:=EvaluateExpression(Cur+2);     {   if}
      if (isOk=ee_Ok) then begin           { ?}
       if myAnswer=0                       {}
       then goto NextLine                  {  false,   }
       else begin                          {  then:}
        Cur:=P+4;                          {    then}
        while Cur^ in WhiteChars do inc(Cur); {  }
        if Cur[0]=#0 then goto NextLine;   {,   }
        goto CheckIf;                      {  if..then..if...}
       end;
      end else begin                       {  }
       myMayPrint:=true;
       Result:=isOk;                       {  }
       break;
      end;
     end else begin                        {   if..then}
      Clear;
      StrCopy(myErrorToken,StrCopy(myToken,StrCopy(myBuffer,Cur)));
      myStatus:=ee_If;
      myMayPrint:=true;
      Result:=myStatus;
      break;
     end;
    end;
    {
     goto   "goto Label",  Label- 
      "Label:"  ,  .
    }
   CheckGoto:
    if (StrLIComp(Cur,'goto',4)=0) and (Cur[4] in WhiteChars) then begin
     Clear;
     try
      myExpression:=StrCopy(myBuffer,Cur+4);     {    goto}
      Parse;                                     { , }
      if (myTokenType<>id_Variable) or           {   }
         (myToken[0]=#0) or                      {   }
         (myExpression[0]<>#0) or                {   !}
         (not myLabelList.GetValue(myToken,nl))  {     }
      then RaiseError(ee_Label);                 {  }
      LineNum:=system.round(nl);                 { -}
      continue;
     except
      on EExpressionEvaluator do begin
       myMayPrint:=true;
       Result:=myStatus;
       break;
      end;
      else RAISE;
     end;
    end;
    {
     gosub   "gosub Label",  Label- 
      "Label:"  ,  .
    }
   CheckGoSub:
    if (StrLIComp(Cur,'gosub',5)=0) and (Cur[5] in WhiteChars) then begin
     Clear;
     try
      myExpression:=StrCopy(myBuffer,Cur+5);     {    gosub}
      Parse;                                     { , }
      if (myTokenType<>id_Variable) or           {   }
         (myToken[0]=#0) or                      {   }
         (myExpression[0]<>#0) or                {   !}
         (not myLabelList.GetValue(myToken,nl))  {     }
      then RaiseError(ee_Label);                 {  }
      if StackPtr>=High(myStack)                 { }
      then RaiseError(ee_StackOver);             { }
      myStack[StackPtr]:=LineNum+1;              {  }
      inc(StackPtr);                             {  }
      LineNum:=system.round(nl);                 { -}
      continue;
     except
      on E:EExpressionEvaluator do begin
       myMayPrint:=true;
       Result:=myStatus;
       break;
      end;
      else RAISE;
     end;
    end;
    {
     return       gosub
    }
   CheckReturn:
    if (StrLIComp(Cur,'return',6)=0) and (Cur[6] in (WhiteChars+[#0])) then begin
     if StackPtr<=Low(myStack) then begin        {    }
      Clear;
      StrCopy(myErrorToken,StrCopy(myToken,StrCopy(myBuffer,Cur)));
      myStatus:=ee_BadReturn;
      myMayPrint:=true;
      Result:=myStatus;
      break;
     end;
     dec(StackPtr);                              {   }
     LineNum:=myStack[StackPtr];                 {   }
     continue;
    end;
    {
     exit    
    }
   CheckExit:
    if (StrLIComp(Cur,'exit',4)=0) and (Cur[4] in (WhiteChars+[#0])) then begin
     Clear;
     myStatus:=ee_Ok;
     myMayPrint:=false;
     Result:=myStatus;
     break;
    end;
    {
       
    }
   EvalExpression:
    isOk:=EvaluateLine(Cur);
    Result:=isOk;
    if isOk<>ee_Ok then break;
    {
        
    }
   NextLine:
    inc(LineNum);
   end;
  end;
 except
  on E:Exception do begin
   myErrorPos:=0;
   myStatus:=ee_Exception;
   StrLCopy(myErrorToken,myToken,ee_MaxToksLen);
   ErrorReport(E);
   myMayPrint:=true;
   Result:=myStatus;
  end;
 end;
end;

procedure TExpressionEvaluator.SaveVars(const IniFile,Section:ShortString);
var f:text; P:TText;
begin
 if Assigned(Self) and (not IsEmptyStr(IniFile)) and (not IsEmptyStr(Section)) then
 try
  SetInOutRes(0);
  system.Assign(f,IniFile);
  system.Rewrite(f);
  try
   writeln(f,Section);
   P:=VarList.GetText(NewText);
   try
    P.Write(f);
   finally
    Kill(P);
   end;
  finally
   system.Close(f);
   SetInOutRes(0);
  end;
 except
  on E:Exception do ErrorReport(E);
 end;
end;

procedure TExpressionEvaluator.RestoreVars(const IniFile,Section:ShortString);
var P:TText;
begin
 if (not IsEmptyStr(IniFile)) and (not IsEmptyStr(Section)) and FileExists(IniFile) then
 try
  P:=ExtractListSection(IniFile,Section,efAsIs);
  try
   ExecuteScript(P);
  finally
   Kill(P);
  end;
 except
  on E:Exception do ErrorReport(E);
 end;
end;

 ///////////////////////////////////////////////////////////////////////////////
 // Special conversions for interpreter output
 // %Var        - insert Format('%g',[Var])
 // %3.4f%Var   - insert Format{'%3.4f',[Var])
 // a`          - convert a to A
 ///////////////////////////////////////////////////////////////////////////////
function TExpressionEvaluator.SmartArgs(const Args:ShortString):ShortString;
var
 v   : Double;
 i,j : Integer;
 Fmt : ShortString;
 Buf : packed array[0..255] of Char;
begin
 Result:=TrimChars(Args,WhiteChars,WhiteChars);
 if Assigned(Self) and not IsEmptyStr(Result) then
 try
  Fmt:=DefaultFmt;
  i:=1;
  while i<=Length(Result) do begin
   if Result[i]='%' then begin
    j:=i+1;
    if (j<=Length(Result)) and (Result[j] in DigitChars) then begin
     while (j<=Length(Result)) and (Result[j] in FormtChars) do Inc(j);
     Fmt:=System.Copy(Result,i,j-i);
     System.Delete(Result,i,j-i);
     Continue;
    end else begin
     while (j<=Length(Result)) and (Result[j] in AlphaChars) do Inc(j);
     if GetValue(StrPCopy(Buf,System.Copy(Result,i+1,j-i-1)),v) then begin
      System.Delete(Result,i,j-i);
      System.Insert(Format(Fmt,[v]),Result,i);
     end else i:=j;
     Fmt:=DefaultFmt;
     Continue;
    end;
   end;
   Inc(i);
  end;
  if UpChar>#0 then begin
   for i:=2 to Length(Result) do
   if Result[i]=UpChar then Result[i-1]:=UpCase(Result[i-1]);
   Result:=SysUtils.StringReplace(Result,UpChar,'',[rfReplaceAll]);
  end;
 except
  on E:Exception do ErrorReport(E);
 end;
end;

function  NewExpressionEvaluator:TExpressionEvaluator;
begin
 Result:=nil;
 try
  Result:=TExpressionEvaluator.Create;
 except
  on E:Exception do BugReport(E);
 end;
end;

procedure Kill(var TheObject:TExpressionEvaluator); overload;
begin
 try
  FreeAndNil(TheObject);
 except
  on E:Exception do BugReport(E);
 end;
end;

 {
       
 }
function ee_ErrorMessage(ErrorCode:Integer):ShortString;
begin
 case ErrorCode of
  ee_Ok          : Result:=RusEng('  ','Operation complete success');
  ee_Syntax      : Result:=RusEng(' ','Syntax error');
  ee_UnBalanced  : Result:=RusEng(' ','Unbalanced parenthesis');
  ee_DivZero     : Result:=RusEng('  ','Division by zero');
  ee_Unknown     : Result:=RusEng(' ','Unknown variable');
  ee_MaxVars     : Result:=RusEng('  ','Maximum variables exceeded');
  ee_BadFunc     : Result:=RusEng(' ','Unrecognised funtion');
  ee_NumArgs     : Result:=RusEng('   ','Wrong number of arguments to funtion');
  ee_NoArgs      : Result:=RusEng('  ','Missing an argument');
  ee_Empty       : Result:=RusEng(' ','Empty expression');
  ee_Label       : Result:=RusEng(' ','Invalid label');
  ee_VarDecl     : Result:=RusEng('  ','Variable declare error');
  ee_If          : Result:=RusEng('  if','Error in If operator');
  ee_NilRef      : Result:=RusEng('  NIL','NIL object reference');
  ee_BadReturn   : Result:=RusEng(' Return  GoSub!','Return without GoSub!');
  ee_StackOver   : Result:=RusEng('  !','GoSub stack overflow!');
  ee_Exception   : Result:=RusEng('   ','Exception during script execution');
  ee_LineTooLong : Result:=RusEng('   ','Input line string too long');
  ee_User        : Result:=RusEng(' ','User defined error');
  else             Result:=RusEng(' ','Unknown error');
 end;
end;
 
 // Check if Script is simple, i.e. can be executed line by line, without precompilation.
function IsSimpleScript(const Script:LongString):Boolean;
var List:TText; i,j,wc:Integer; Line,ws:ShortString;
const Delims=WhiteChars+DelimChars+ArgDelims;
begin
 Result:=true;
 if Length(Script)>0 then
 try
  Result:=false;
  List:=NewText;
  try
   List.Text:=Script;                    // Convert to text
   for i:=0 to List.Count-1 do begin     // For each line:
    Line:=Trim(List[i]);                 // Get line
    if Length(Line)=0 then continue;     // Skip empty lines
    wc:=WordCount(Line,Delims);          // Find number of words
    for j:=1 to wc do begin              // For each word:
     ws:=ExtractWord(j,Line,Delims);     // Extract word
     if Pos('@',ws)=1 then break;        // Skip @actions
     if Pos('\',ws)=1 then break;        // Skip \actions
     if Pos(':',ws)>0 then exit;         // Found Lable:
     if SameText(ws,'if') then exit;     // Found If
     if SameText(ws,'then') then exit;   // Found Then
     if SameText(ws,'goto') then exit;   // Found GoTo
     if SameText(ws,'exit') then exit;   // Found Exit
     if SameText(ws,'gosub') then exit;  // Found GoSub
     if SameText(ws,'return') then exit; // Found Return
    end;
   end;
   Result:=true;
  finally
   Kill(List);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

 // Evaluate Script text line by line.
function EvaluateSimpleScript(ee:TExpressionEvaluator; const Script:LongString):Integer;
var Line:array[0..ee_MaxToksLen] of Char; i,LinePos,Count:Integer;
begin
 Result:=ee_NilRef;
 if Assigned(ee) then
 try
  Result:=ee_OK;
  LinePos:=0; Count:=0;
  for i:=1 to Length(Script) do begin
   if (Script[i] in [ASCII_CR,ASCII_LF,ASCII_NUL]) then begin
    if (LinePos>0) and (Count>0) then begin
     if Count>ee_MaxToksLen
     then Result:=ee_LineTooLong
     else Result:=ee.EvaluateExpression(StrLCopy(Line,@(Script[LinePos]),Count),false);
    end;
    LinePos:=0; Count:=0;
    if Result<>ee_OK then break;
   end else begin
    if LinePos=0 then LinePos:=i;
    inc(Count);
   end;
  end;
  if (LinePos>0) and (Count>0) then begin
   if Count>ee_MaxToksLen
   then Result:=ee_LineTooLong
   else Result:=ee.EvaluateExpression(StrLCopy(Line,@(Script[LinePos]),Count),false);
  end;
 except
  on E:Exception do BugReport(E);
 end;
end;

 // Expression Evaluator Benchmark to eastimate performance.
function ExpressionEvaluatorBenchmark(n:Integer=1000000):LongString;
var ee:TExpressionEvaluator; expr:LongString; i:Integer; ms:Double;
begin
 Result:='Expression Evaluator Benchmark:'+CRLF;
 expr:='gt(v,alarm1)+gt(v,alarm2)';
 ee:=NewExpressionEvaluator;
 try
  ms:=_rtc.mSecNow;
  for i:=0 to n-1 do begin
   ee.SetValue('v',system.frac(ms*0.01));
   ee.SetValue('alarm1',0.5);
   ee.SetValue('alarm2',0.7);
   if ee.EvaluateLine(PChar(expr))<>0 then break;
  end;
  ms:=_rtc.mSecNow-ms;
  Result:=Result+Format('Expression = %s',[expr])+CRLF;
  Result:=Result+Format('Total loop = %d',[i])+CRLF;
  Result:=Result+Format('Total time = %g ms',[ms])+CRLF;
  Result:=Result+Format('Oper. time = %3g mks/op',[ms*1000/n])+CRLF;
 finally
  Kill(ee);
 end;
end;

initialization

finalization

 ResourceLeakageLog(Format('%-60s = %d',['Expression Evaluator Hash Collisions',ee_HashCollisions]));

end.
