////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Численное интегрирование кривой.
Интегрирование кривой производится по методу трапеций.
Это дискретный аналог первообразной функции.
[]
[Manual.Eng]
This utility evaluate curve integral using trapezium rule.
That is discrete analog of function integration.
[]
[Arguments.Rus]
Имя окна = Результат:"RDMS_Filter"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
[]
[Arguments.Eng]
Caption = Result:"RDMS_Filter"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
[]
}

library rdms_filter;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

const
 NMax = 20;
 Tmax = 1*60;        // Max period to get points
 Qmax = 30*60*0.001; // 30 min with activity 0.001 mkKu/L
 Qpor = 0.80;        // Threshold
var
 qnum : Integer = 0;
 qbuf : array[0..NMax-1] of TPoint2d;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNotSortedX + cfDuplicatesX + cfNanInf;

 AMP_KVA=-750;
 K1_AMP1=2.0E-5;
 Volume=1;

var
 p : TPoint2d;
 i,j : Integer;
 t,dt,u_mv,i_na,mkku,ai,fi,qi,ti,t0,tm : Double;
 c1, c2, cFirst, cLast : Integer;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  for c1:=cFirst to cLast do begin
   if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для интегрирования!',
                                           'Curve %d is not convenient for integration!'),[c1]));
   //
   // Get OA without smoothing
   //
   c2:=CreateCurve(twin);
   CurveColor[c2]:=$FF0000;
   CurveStyle[c2]:=$10;
   for i:=0 to CurveLength[c1]-1 do begin
    p:=CurvePoint[c1,i];
    t:=p.x;
    u_mv:=p.y;
    i_na:=K1_AMP1*u_mv;
    mkku:=AMP_KVA*i_na/Volume;
    CurveAddPoint(c2,Point2d(t,mkku));
   end;
   //
   // Get OA with smoothing
   //
   t0:=0;
   ti:=0;
   fi:=0;
   qi:=-1;
   tm:=Tmax;
   qnum:=0;
   c2:=CreateCurve(twin);
   CurveColor[c2]:=0;
   CurveStyle[c2]:=$10;
   for i:=0 to CurveLength[c1]-1 do begin
    //
    // Get "measured" data point, t=time, u_ch=voltage
    //
    p:=CurvePoint[c1,i];
    t:=p.x;
    u_mv:=p.y;
    //
    // Convert voltage to i_na=current[nA], mkku=activity[mkKu/L]
    //
    i_na:=K1_AMP1*u_mv;
    mkku:=AMP_KVA*i_na/Volume;
    //
    // if first point, remember it
    //
    if qi<0 then begin
     CurveAddPoint(c2,Point2d(t,mkku));
     fi:=0;
     qi:=0;
     ti:=t;
     t0:=t;
     continue;
    end;
    //
    // Accumulate integrals
    //
    fi:=fi+mkku*(t-ti);
    qi:=qi+abs(mkku)*(t-ti);
    ti:=t;
    dt:=t-t0;
    if (qi>QMax) or (dt>tm) then begin
     ai:=fi/dt;
     qi:=qi/dt;
     tm:=Min(Tmax,Qmax/qi);
     if (qi>QPor) then begin
      qnum:=0;
     end else begin
      Move(qbuf[0],qbuf[1],sizeof(qbuf)-sizeof(qbuf[0]));
      qbuf[0]:=Point2d(dt,fi);
      qnum:=Min(qnum+1,NMax);
      fi:=0;
      dt:=0;
      for j:=0 to qnum-1 do begin
       fi:=fi+qbuf[j].y;
       dt:=dt+qbuf[j].x;
      end;
      ai:=fi/dt;
     end;
     CurveAddPoint(c2,Point2d(ti,ai));
     fi:=0;
     qi:=0;
     t0:=t;
    end;
   end;
   //
   //
   //
   if CurveFlags(c2,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Ошибка обработки кривой %d !',
                                           'Error on curve %d processing!'),[c1]));
   Refreshment(100);
  end;
  //if SelectedCurve[swin]>0 then SelectedCurve[twin]:=1 else SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;			
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
 
