////////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2001-2024 DaqGroup daqgroup@mail.ru under MIT license        //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// This file is part of the CRW-DAQ project by DaqGroup - addon user plugin.  //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Purpose:                                                                   //
// crwdaq data analysis plugin.                                               //
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// History:                                                                   //
// 20241030 - Sample created by A.K.                                          //
// 20241104 - Translated from DPR source by A.K.                              //
////////////////////////////////////////////////////////////////////////////////

{
[Manual.Rus]
Эта утилита служит для построения полинома МНК по точкам кривой в области 
между правой и левой вертикальной границей "Области интересов" ROI.
"Область интересов" должна быть задана в окне - источнике перед вызовом утилиты
при помощи двух маркеров ROI. Надо также задать степень полинома, а также центр c,
масштаб s, начало L, и конец R, выраженные через границы ROI (a,b).
[]
[Manual.Eng]
This plugin uses to find polynom by curve points between
left and right "Region Of Interest", ROI, markers.
Set "Region Of Interest" before call plugin.
Uses ROI markers in source data window.
[]
[Arguments.Rus]
Имя окна = Результат:"Полином кривой ROI X"
Заголовок = ^CЗаголовок^N^L  Y
Легенда = ^RX  ^N^CЛегенда
Степень = 1
ЧислоТочек = 100
Центр = c = (a + b) / 2
Масштаб = s = (b - a)
Начало = L = a - (b - a) / 2
Конец = R = b + (b - a) / 2
[]
[Arguments.Eng]
Caption = Result:"Polynom of curve ROI X"
Title = ^CTitle^N^L  Y
Legend = ^RX  ^N^CLegend
Power = 1
NumPoints = 100
Center = c = (a + b) / 2
Scale = s = (b - a)
Left = L = a - (a + b) / 2
Right = R = b + (a + b) / 2
[]
}

library _curve_polynom_roi_x;

{$I _crw_sysdef.inc}

{$I _crw_sysmode.inc}

{$R *.res}

uses
 //////////////////////////////////////////////////////
 {$I _crw_uses_first.inc} // NB: MUST BE FIRST USES !!!
 //////////////////////////////////////////////////////
 sysutils, classes, math, graphics,
 _crw_crwapi;

/////////////////////////////////////////////////////////
{$I _crw_plugin_declare.inc} // Declare CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// function CRWDAQ_PLUGIN(CrwApi:ICrwApi):Integer;     //
/////////////////////////////////////////////////////////
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 CheckFlags = cfInvalid + cfNoData + cfTooSmall + cfNanInf;
var
 p : TPoint2D;
 Roi : TRect2D;
 Interpreter : IScriptInterpreter;
 Centr, Scale, Left, Right : Double;
 i, n, Power, c1, c2, cFirst, cLast, NumPoints : Integer;
 Poly : IPolynomWrap;
 px, py : PDoubleArray;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  VerifyPluginDate(CrwApiEdition,Version);
  VerifyPluginType(Target,ForDataAnalysis);
  if not WindowExists(swin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - источник!',
                                   'Source window not found!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('Не найдено окно - приемник!',
                                   'Target window not found!'));
  if CurvesCount[swin]=0 
  then Raise EDanApi.Create(RusEng('Нет данных для обработки!',
                                   'No input data curves found!'));
  Roi:=WindowRoi[swin];
  if isNAN(Roi.A.X+Roi.A.Y)
  then Raise EDanApi.Create(RusEng('Левый маркер РОИ не определен!',
                                   'Left ROI marker is not defined!'));
  if isNAN(Roi.B.X+Roi.B.Y)
  then Raise EDanApi.Create(RusEng('Правый маркер РОИ не определен!',
                                   'Right ROI marker is not defined!'));
  Roi:=Rect2DValidate(Roi);
  Roi.A.Y:=_MinusInf;
  Roi.B.Y:=_PlusInf;
  WindowRoi[twin]:=WindowRoi[swin];
  WindowCaption[twin]:=GetArgumentAsString(RusEng('Имя окна','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('Заголовок','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('Легенда','Legend'));
  if not GetArgumentAsInteger(RusEng('Степень','Power'),Power) or (Power<0) or (Power>9) or
     not GetArgumentAsInteger(RusEng('ЧислоТочек','NumPoints'),NumPoints) or (NumPoints<2)
  then Raise EDanApi.Create(RusEng('Ошибка задания аргументов!','Invalid input arguments!'));
  Interpreter:=CreateScriptInterpreter;
  try
   Interpreter.Script:=Interpreter.Script+GetArgumentAsString(RusEng('Центр','Center'))+CRLF;
   Interpreter.Script:=Interpreter.Script+GetArgumentAsString(RusEng('Масштаб','Scale'))+CRLF;
   Interpreter.Script:=Interpreter.Script+GetArgumentAsString(RusEng('Начало','Left'))+CRLF;
   Interpreter.Script:=Interpreter.Script+GetArgumentAsString(RusEng('Конец','Right'))+CRLF;
   if Interpreter.SetValue('a',Roi.A.X) and
      Interpreter.SetValue('b',Roi.B.X) and
      (Interpreter.RunScript=0)         and
      Interpreter.GetValue('c',Centr)   and
      Interpreter.GetValue('s',Scale)   and
      Interpreter.GetValue('l',Left)    and
      Interpreter.GetValue('r',Right)   and
      not isNAN(Centr)                  and
      not isNAN(Scale)                  and
      not isINF(Left)                   and
      not isINF(Right)
   then // Ok
   else Raise EDanApi.Create(RusEng('Ошибка интерпретации!','Interpreter error!'));;
  finally
   Kill(Interpreter);
  end;
  if SelectedCurve[swin]>0 then begin
   cFirst:=SelectedCurve[swin];
   cLast:=SelectedCurve[swin];
  end else begin
   if YesNo(RusEng('Выполнить для всех кривых в окне?',
                   'Execute for all curves in window?'))<>mrYes
   then Raise EDanApi.Create(RusEng('Пользователь прервал!','User break!'));
   cFirst:=1;
   cLast:=CurvesCount[swin];
  end;
  for c1:=cFirst to cLast do begin
   if CurveFlags(c1,0,1E-10,3) and CheckFlags <> 0
   then Raise EDanApi.Create(Format(RusEng('Кривая %d непригодна для фитирования!',
                                           'Curve %d is not convenient for fitting!'),[c1]));
   n:=0;
   px:=nil;
   py:=nil;
   Poly:=CreatePolynom(Power,Centr,Scale);
   try
    px:=Allocate(CurveLength[c1]*sizeof(px[0]));
    py:=Allocate(CurveLength[c1]*sizeof(py[0]));
    if (px=nil) or (py=nil) then Raise EDanApi.Create(RusEng('Нет памяти!','Out of memory!'));
    for i:=0 to CurveLength[c1]-1 do begin
     if Rect2dContainsPoint(Roi,CurvePoint[c1,i]) then begin
      px[n]:=CurvePoint[c1,i].x;
      py[n]:=CurvePoint[c1,i].y;
      inc(n);
     end;
     Refreshment(100);
    end;
    if not Poly.Find(px[0],py[0],n) 
    then Raise EDanApi.Create(Format(RusEng('Ошибка фитирования кривой %d!','Error fitting %d!'),[c1]));
    writeln(Format('Curve:%s',[CurveName[c1]]));
    writeln(Format('Power = %d',[Poly.Power]));
    writeln(Format('Center = %g',[Poly.Center]));
    writeln(Format('Scale = %g',[Poly.Scale]));
    writeln(Format('Polynom coefficients:',[]));
    for i:=0 to Poly.Power do writeln(Format(' c[%d] = %g', [i, Poly[i]]));
    c2:=CreateCurve(twin);
    CurveAssign(c2,c1);
    c2:=CreateCurve(twin,'Fit:'+CurveName[c1],clBlue,$1F);
    CurveLength[c2]:=NumPoints;
    for i:=0 to NumPoints-1 do begin
     p.x:=Left+(Right-Left)*i/(NumPoints-1);
     p.y:=Poly.Get(p.x);
     CurvePoint[c2,i]:=p;
     Refreshment(100);
    end;
   finally
    Deallocate(Pointer(px));
    Deallocate(Pointer(py));
    Kill(Poly);
   end;
  end;
  SelectedCurve[twin]:=0;
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0;
   Echo(PluginName+RusEng(': ОШИБКА!',': ERROR!')); 
   if UsesBlaster then Voice('EXCEPTION');
   Echo(E.Message); Error(E.Message);
   Result:=-1;
  end;
 end;
end;

/////////////////////////////////////////////////////////
{$I _crw_plugin_exports.inc} // Exports CRWDAQ_PLUGIN. //
/////////////////////////////////////////////////////////
// exports CRWDAQ_PLUGIN name CRWDAQ_PLUGIN_ID;        //
/////////////////////////////////////////////////////////
begin
end.

//////////////
// END OF FILE
//////////////
 
