{
[Manual.Rus]
      RTD, .
 ,     6651-2009.
   n    (a,b)  C.
 (L,H)     RTD .
 ppd           n.
 lug       (L,H)   (a,b).
 RTD  : PT100_385, PT100_391, CU100_428, CU100_426, NI100_617.
[]
[Manual.Eng]
This plugin calculates RTD resistance, Ohm.
Used formulas of RTD is corresponds to GOST 6651-2009.
Number of points n and temperature range (a,b) in C should be specified.
Variables (L,H) will contain the nominal range of specified RTD.
Variable ppd specifies the number of points per degree to calculate n.
Variable lug specifies extension outside nominal range to calculate (a,b).
RTD type is one of: PT100_385, PT100_391, CU100_428, CU100_426, NI100_617.
[]
[Arguments.Rus]
           = RTD resistance table by GOST 6651-2009 polynoms
         = ^CResistance^N^L  Ohm
           = ^RC  ^N^CTemperature
 RTD           = PT100_385, PT100_391, CU100_428, CU100_426, NI100_617
Script#1          = ppd=1
Script#2          = lug=0
Script#3          = a=Round(L-lug)
Script#4          = b=Round(H+lug)
Script#5          = n=(b-a)*ppd+1
Script#6          =
Script#7          =
Script#8          =
Script#9          =
[]
[Arguments.Eng]
Caption           = RTD resistance table by GOST 6651-2009 polynoms
Title             = ^CResistance^N^L  Ohm
Legend            = ^RC  ^N^CTemperature
RTD Type          = PT100_385, PT100_391, CU100_428, CU100_426, NI100_617
Script#1          = ppd=1
Script#2          = lug=0
Script#3          = a=Round(L-lug)
Script#4          = b=Round(H+lug)
Script#5          = n=(b-a)*ppd+1
Script#6          =
Script#7          =
Script#8          =
Script#9          =
[]
}
LIBRARY _GOST_TAB_RTD;

{$I _sysdef}

uses ShareMem, SysUtils, Windows, Math, Classes, _CrwApi;

type Real = Double;

function StrFix(x:Real;w,d:Integer):LongString;
begin
 Result:=Format('%*.*f',[w,d,x]);
end;

function _NaN:Real;
begin
 _NaN:=0.0/0.0;
end;

function isNaN(x:Real):Boolean;
begin
 isNaN:=(x<>x);
end;

function isInf(x:Real):Boolean;
begin
 isInf:=(x<-MaxDouble) or (x>+MaxDouble);
end;

var
 restemperdet_cnt : Real;    { RTD call counter      }
 restemperdet_eps : Real;    { RTD calc epsilon      }
 restemperdet_tol : Real;    { RTD calc tolerance    }
 restemperdet_big : Real;    { RTD big range value   }
 restemperdet_del : Real;    { RTD line ext. delta   }
 restemperdet_fac : Real;    { RTD srch grow factor  }
 restemperdet_met : Integer; { RTD calc method       }
 restemperdet_mit : Integer; { RTD max.iterations    }

 //////////////////////////////////////////////////////////////////////////////
 // RTD and ThermoCouple calculation unit: convert mV to C and vise versa.
 //////////////////////////////////////////////////////////////////////////////
 // Reference list:
 // 1) GOST R 8.585-2001                              - thermocouple tables
 // 2) http://srdata.nist.gov/its90/download/all.tab  - thermocouple tables
 // 3) GOST R 50431-92                                - thermocouple tables
 // 4) GOST 6651-2009                                 - RTD tables+formulas
 // 5) GOST R 8.625-2006                              - RTD tables+formulas
 // 6) GOST 6651-94                                   - RTD tables+formulas
 // Note: GOST R 8.585-2001, GOST 6651-2009 are most actual, others is obsolete
 //////////////////////////////////////////////////////////////////////////////
 //
 //////////////////////////////////////////////////////////////////////////////
 // function restemperdet_ohm(t:Real; idn:Integer):Real;
 // Return:
 //  RTD resistance, Ohms. Return NaN on invalid idn.
 // Arguments:
 //  t    - temperature, C.
 //  idn  - RTD type identifier number:
 //   385 = RTD Pt100 platinum  R0=100 alfa=0.00385
 //   391 = RTD Pt100 platinum  R0=100 alfa=0.00391
 //   428 = RTD Cu100 copper       R0=100 alfa=0.00428
 //   426 = RTD Cu100 copper       R0=100 alfa=0.00426
 //   617 = RTD Ni100 nikel      R0=100 alfa=0.00617
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_ohm(t:Real; idn:Integer):Real;
 var ohm:Real;
  // R0AiBjCkDn(t,L,H,..) define polynom function
  // use linear extrapolation outside (L,H) range
  // A,B,C,D: coefficients taken from GOST tables
  function R0A1B2(t,L,R,R0,A,B:Real):Real;
  var x,s,f,q:Real;
  begin
   x:=max(L,min(R,t)); s:=t-x;
   f:=R0*(1+x*(A+x*B));
   if s<>0 then begin // linear extrapolation
    q:=R0*(A+x*2*B);
    f:=f+q*s;
   end;
   R0A1B2:=f;
  end;
  function R0A1B2C4D3(t,L,R,R0,A,B,C,D:Real):Real;
  var x,s,f,q:Real;
  begin
   x:=max(L,min(R,t)); s:=t-x;
   f:=R0*(1+x*(A+x*(B+x*C*(x+D))));
   if s<>0 then begin // linear extrapolation
    q:=R0*(A+x*(2*B+x*C*(4*x+3*D)));
    f:=f+q*s;
   end;
   R0A1B2C4D3:=f;
  end;
  function R0A1B2C3D1(t,L,R,R0,A,B,C,D:Real):Real;
  var x,s,f,q:Real;
  begin
   x:=max(L,min(R,t)); s:=t-x;
   f:=R0*(1+x*(A+B*D+x*(B+C*x)));
   if s<>0 then begin // linear extrapolation
    q:=R0*(A+B*D+x*(2*B+C*3*x));
    f:=f+q*s;
   end;
   R0A1B2C3D1:=f;
  end;
  function R0A1B2C3D2(t,L,R,R0,A,B,C,D:Real):Real;
  var x,s,f,q:Real;
  begin
   x:=max(L,min(R,t)); s:=t-x;
   f:=R0*(1+x*(A+x*(B+C*(x+D))));
   if s<>0 then begin // linear extrapolation
    q:=R0*(A+x*(2*B+C*(3*x+2*D)));
    f:=f+q*s;
   end;
   R0A1B2C3D2:=f;
  end;
 begin
  if (idn=385) then begin
   if t < 0.000
   then ohm:=R0A1B2C4D3(t, -200.000,   0.000, 100.0, 3.9083E-3, -5.775E-7,  -4.183E-12, -100.00)
   else ohm:=R0A1B2(    t,    0.000, 850.000, 100.0, 3.9083E-3, -5.775E-7);
  end else
  if (idn=391) then begin
   if t < 0.000
   then ohm:=R0A1B2C4D3(t, -200.000,   0.000, 100.0, 3.9690E-3, -5.841E-7,  -4.330E-12, -100.00)
   else ohm:=R0A1B2(    t,    0.000, 850.000, 100.0, 3.9690E-3, -5.841E-7);
  end else
  if (idn=428) then begin
   if t < 0.000
   then ohm:=R0A1B2C3D1(t, -180.000,   0.000, 100.0, 4.28E-3,   -6.2032E-7, 8.5154E-10, 6.7)
   else ohm:=R0A1B2(    t,    0.000, 200.000, 100.0, 4.28E-3,   0);
  end else
  if (idn=426) then begin
   ohm:=R0A1B2(         t,  -50.000, 200.000, 100.0, 4.26E-3,   0);
  end else
  if (idn=617) then begin
   if t < 100.000
   then ohm:=R0A1B2(    t,  -60.000, 100.000, 100.0, 5.4963E-3,  6.7556E-6)
   else ohm:=R0A1B2C3D2(t,  100.000, 180.000, 100.0, 5.4963E-3,  6.7556E-6, 9.2004E-09, -100.00);
  end else
  ohm:=_NaN;
  restemperdet_cnt:=restemperdet_cnt+1;
  restemperdet_ohm:=ohm;
 end;
 //////////////////////////////////////////////////////////////////////////////
 // function restemperdet_lot(idn:Integer):Real;
 // Return:
 //  RTD low temperature range, C. Return NaN on invalid idn.
 // Arguments:
 //  idn - RTD type identifier number (see restemperdet_ohm).
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_lot(idn:Integer):Real;
 var t:Real;
 begin
  if (idn=385) then t:=-200 else
  if (idn=391) then t:=-200 else
  if (idn=428) then t:=-180 else
  if (idn=426) then t:=-50  else
  if (idn=617) then t:=-60  else
  t:=_NaN;
  restemperdet_lot:=t;
 end; 
 //////////////////////////////////////////////////////////////////////////////
 // function restemperdet_hit(idn:Integer):Real;
 // Return:
 //  RTD high temperature range, C. Return NaN on invalid idn.
 // Arguments:
 //  idn - RTD type identifier number (see restemperdet_ohm).
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_hit(idn:Integer):Real;
 var t:Real;
 begin
  if (idn=385) then t:=850 else
  if (idn=391) then t:=850 else
  if (idn=428) then t:=200 else
  if (idn=426) then t:=200 else
  if (idn=617) then t:=180 else
  t:=_NaN;
  restemperdet_hit:=t;
 end;
 //////////////////////////////////////////////////////////////////////////////
 // function restemperdet_idn(ids:String):Integer;
 // Return:
 //  RTD type identification number. Return NaN on invalid ids.
 // Arguments:
 //  ids - RTD type identifier string (see restemperdet_ohm).
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_idn(ids:String):Integer;
 var idn:Integer;
 begin
  ids:=Trim(ids);
  if (ids='PT100_385') then idn:=385 else
  if (ids='PT100_391') then idn:=391 else
  if (ids='CU100_428') then idn:=428 else
  if (ids='CU100_426') then idn:=426 else
  if (ids='NI100_617') then idn:=617 else
  idn:=0;
  restemperdet_idn:=idn;
 end;
 //////////////////////////////////////////////////////////////////////////////
 // function restemperdet_ids(idn:Integer):String;
 // Return:
 //  RTD type identifier string. Return empty string on invalid idn.
 // Arguments:
 //  idn - RTD type identifier number (see restemperdet_ohm).
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_ids(idn:Integer):String;
 var ids:String;
 begin
  if (idn=385) then ids:='PT100_385' else
  if (idn=391) then ids:='PT100_391' else
  if (idn=428) then ids:='CU100_428' else
  if (idn=426) then ids:='CU100_426' else
  if (idn=617) then ids:='NI100_617' else
  ids:='';
  restemperdet_ids:=ids; ids:='';
 end;
 //////////////////////////////////////////////////////////////////////////////
 // function restemperdet_ttc(ohm:Real; idn:Integer):Real;
 // Return:
 //  RTD temperature, C. Return NAN on error or invalid idn.
 // Arguments:
 //  ohm - RTD resistance, ohms.
 //  idn - RTD type: 385,391, 428, 426, 617
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_ttc(ohm:Real; idn:Integer):Real;
 var a,b,c,d,e,fa,fb,fc,eps,tol,tol1,xm,p,q,r,s,big,del,fac:Real;
  itn,met,mit:Integer; Terminate,Failure:Boolean;
  function FSign(a,b:Real):Real;
  begin
   if b<=0 then FSign:=-abs(a) else FSign:=abs(a);
  end;
  function Sign(x:Real):Integer;
  begin
   if x<0 then Sign:=-1 else if x>0 then Sign:=+1 else Sign:=0;
  end;
  function F(t:Real):Real;
  var goal:Real;
  begin
   goal:=restemperdet_ohm(t,idn)-ohm;
   if isNaN(goal) then Failure:=True;
   if isInf(goal) then Failure:=True;
   F:=goal;
  end;
 begin
  c:=0; itn:=0;
  Failure:=False; Terminate:=False;
  met:=restemperdet_met; if met<=0 then met:=1;     // zeroin method
  mit:=restemperdet_mit; if mit<=0 then mit:=128;   // max.iteration
  eps:=restemperdet_eps; if eps<=0 then eps:=1E-15; // mach. epsilon
  tol:=restemperdet_tol; if tol<=0 then tol:=1E-14; // the tolerance
  big:=restemperdet_big; if big<=0 then big:=1E+15; // big range
  del:=restemperdet_del; if del<=0 then del:=0;     // delta
  fac:=restemperdet_fac; if fac<=1 then fac:=1;     // factor
  if not Failure then begin
   a:=restemperdet_lot(idn); b:=restemperdet_hit(idn);
   if a>b then begin c:=a; a:=b; b:=c; end;
   fa:=F(a); fb:=F(b);
   if del>0 then begin // Use linear extrapolation outside (a,b)
    if fa<fb then begin
     if Sign(fa)>0 then begin c:=a-fa*del/(fa-F(a-del)); Terminate:=true; end else
     if Sign(fb)<0 then begin c:=b-fb*del/(F(b+del)-fb); Terminate:=true; end;
    end else
    if fa>fb then begin
     if Sign(fb)>0 then begin c:=b-fb*del/(F(b+del)-fb); Terminate:=true; end else
     if Sign(fa)<0 then begin c:=a-fa*del/(fa-F(a-del)); Terminate:=true; end;
    end;
   end else
   if Sign(fa)*Sign(fb)>0 then begin
    d:=b-a;
    if Sign(fa)*Sign(fb-fa)>0 then if fac<=1 then a:=-big else begin
     while (itn<mit) and (abs(a-d)<big) and not Failure do begin
      if Sign(F(a-d))*Sign(fb)<=0 then begin a:=a-d; itn:=mit; end;
      d:=d*2; itn:=itn+1;
     end;
    end;
    if Sign(fb)*Sign(fb-fa)<0 then if fac<=1 then b:=+big else begin
     while (itn<mit) and (abs(b+d)<big) and not Failure do begin
      if Sign(fa)*Sign(F(b+d))<=0 then begin b:=b+d; itn:=mit; end;
      d:=d*2; itn:=itn+1;
     end;
    end;
    itn:=0;
   end;
   if not Terminate then
   if met=1 then begin // Method: ZeroIn
    fa:=F(a);
    if fa=0.0 then c:=a else begin
     fb:=F(b);
     if fb=0.0 then c:=b else begin
      c:=a; fc:=fa; d:=b-a; e:=d;
      if Sign(fa)*Sign(fb)>0 then Failure:=True;
      while not Terminate and not Failure do begin
       if abs(fc)<abs(fb) then begin a:=b; b:=c; c:=a; fa:=fb; fb:=fc; fc:=fa; end;
       tol1:=2.0*eps*abs(b)+0.5*tol; xm:=0.5*(c-b);
       if (abs(xm)<=tol1) or (fb=0.0) or (itn>=mit) then Terminate:=True else begin
        if (abs(e)<tol1) or (abs(fa)<=abs(fb)) then begin
         d:=xm; e:=d;
        end else begin
         if a<>c then begin
          q:=fa/fc; r:=fb/fc; s:=fb/fa; p:=s*(2.0*xm*q*(q-r)-(b-a)*(r-1.0)); q:=(q-1.0)*(r-1.0)*(s-1.0);
         end else begin
          s:=fb/fa; p:=2.0*xm*s; q:=1.0-s;
         end;
         if p>0.0 then q:=-q; p:=abs(p);
         if (2.0*p>=3.0*xm*q-abs(tol1*q)) or (p>=abs(e*q*0.5)) then begin
          d:=xm; e:=d;
         end else begin
          e:=d; d:=p/q;
         end;
        end;
        a:=b; fa:=fb;
        if abs(d)> tol1 then b:=b+d else b:=b+FSign(tol1,xm);
        fb:=F(b); if fb*(fc/abs(fc))>0 then begin c:=a; fc:=fa; d:=b-a; e:=d; end;
       end;
       itn:=itn+1;
      end;
      c:=b;
     end;
    end;
   end else begin // Method: Bisection
    fa:=F(a);
    if fa=0.0 then c:=a else begin
     fb:=F(b);
     if fb=0.0 then c:=b else begin
      if Sign(fa)*Sign(fb)>0 then Failure:=True;
      while not Terminate and not Failure do begin
       c:=a+(b-a)*0.5; fc:=F(c);
       if (fc=0.0) then Terminate:=True else
       if (itn>=mit) then Terminate:=True else
       if (abs(b-a)<tol) then Terminate:=True else
       if Sign(fc)*Sign(fa)<0 then begin b:=c; fb:=fc; end else
       if Sign(fc)*Sign(fb)<0 then begin a:=c; fa:=fc; end else Failure:=True;
       itn:=itn+1;
      end;
     end;
    end;
   end;
  end;
  if Failure then c:=_NaN;
  restemperdet_ttc:=c;
 end;
 //////////////////////////////////////////////////////////////////////////////
 // Test RTD conversions; return error count; 0 means OK.
 //////////////////////////////////////////////////////////////////////////////
 function restemperdet_tst:Integer;
 var errors:Integer;
  procedure TestOhm(t:Real; idn:Integer; s:String);
  var ohm,ttc:Real; fine:Boolean; q:String;
  begin
   fine:=true;
   ohm:=restemperdet_ohm(t,idn);
   ttc:=restemperdet_ttc(ohm,idn);
   q:=StrFix(ohm,1,2);
   if q='-0.00' then q:='0.00';
   if s<>'' then fine:=(q=s); q:='';
   if (abs(t-ttc)>1E-6) then fine:=false;
   writeln('Test ',restemperdet_ids(idn):2,':  t=',t:4:0,'  ohm=',ohm:7:3,'  ttc=',ttc:9:3,'  ',fine);
   if not fine then errors:=errors+1;
  end;
 begin
  errors:=0;
  TestOhm( -203,  385,   '');
  TestOhm( -200,  385,   '18.52');
  TestOhm(  -50,  385,   '80.31');
  TestOhm(    0,  385,  '100.00');
  TestOhm(   50,  385,  '119.40');
  TestOhm(  100,  385,  '138.51');
  TestOhm(  150,  385,  '157.33');
  TestOhm(  850,  385,  '390.48');
  TestOhm(  853,  385,  '');
  TestOhm( -203,  391,   '');
  TestOhm( -200,  391,   '17.24');
  TestOhm(  -50,  391,   '80.00');
  TestOhm(    0,  391,  '100.00');
  TestOhm(   50,  391,  '119.70');
  TestOhm(  100,  391,  '139.11');
  TestOhm(  150,  391,  '158.22');
  TestOhm(  850,  391,  '395.16');
  TestOhm(  853,  391,  '');
  TestOhm(  -53,  426,   '');
  TestOhm(  -50,  426,   '78.70');
  TestOhm(  -20,  426,   '91.48');
  TestOhm(    0,  426,  '100.00');
  TestOhm(   50,  426,  '121.30');
  TestOhm(  100,  426,  '142.60');
  TestOhm(  150,  426,  '163.90');
  TestOhm(  200,  426,  '185.20');
  TestOhm(  203,  426,  '');
  TestOhm( -183,  428,   '');
  TestOhm( -180,  428,   '20.53');
  TestOhm(  -50,  428,   '78.46');
  TestOhm(    0,  428,  '100.00');
  TestOhm(   50,  428,  '121.40');
  TestOhm(  100,  428,  '142.80');
  TestOhm(  150,  428,  '164.20');
  TestOhm(  200,  428,  '185.60');
  TestOhm(  203,  428,  '');
  TestOhm(  -73,  617,   '');
  TestOhm(  -60,  617,   '69.45');
  TestOhm(  -50,  617,   '74.21');
  TestOhm(    0,  617,  '100.00');
  TestOhm(   50,  617,  '129.17');
  TestOhm(  100,  617,  '161.72');
  TestOhm(  150,  617,  '198.68');
  TestOhm(  180,  617,  '223.21');
  TestOhm(  183,  617,  '');
  restemperdet_tst:=errors;
 end;
 //////////////////////////////////////////////////////////////////////////////


function CRW32_PLUGIN(CrwApi:TCrwApi):Integer; StdCall;
const
 swin = +1; // Source window reference
 twin = -1; // Target window reference
 cwin =  0; // Clipboard window reference
 MaxNumPoints = 1024*1024*16;
var
 p : TPoint2D;
 s, types : ShortString;
 i, j, idn, crv, ncount : Integer;
 a, b, n, l, h, emax, tmax : Double;
 Interpreter : TScriptInterpreter;
 procedure Refreshment(Delta:Integer);
 const LastTicks : Cardinal = 0;
 begin
  if LastTicks=0 then LastTicks:=GetTickCount;
  if abs(GetTickCount-LastTicks) > Delta then begin
   with CrwApi,GuiApi do begin ApplicationProcessMessages; UpdateSystemConsole; end;
   LastTicks:=GetTickCount;
  end;
 end;
begin
 Result:=0;					
 with CrwApi,SysApi,GuiApi,DanApi do
 try
  RedirectStdIn(Input);				
  RedirectStdOut(Output);
  if Target <> ForDataAnalysis 
  then Raise EDanApi.Create(RusEng('  Target!',
                                   'Invalid Target!'));
  if not WindowExists(twin) 
  then Raise EDanApi.Create(RusEng('   - !',
                                   'Target window not found!'));
  WindowCaption[twin]:=GetArgumentAsString(RusEng(' ','Caption'));
  WindowTitle[twin]:=GetArgumentAsString(RusEng('','Title'));
  WindowLegend[twin]:=GetArgumentAsString(RusEng('','Legend'));
  types:=GetArgumentAsString(RusEng(' RTD','RTD Type'));
  if WordCount(types,ScanSpaces)=0
  then Raise EDanApi.Create(RusEng('   !',
                                   'RTD type is not specified!'));
  Interpreter:=CreateScriptInterpreter;
  try
   ncount:=0;
   writeln(restemperdet_tst);
   restemperdet_eps:=MachEps*1;
   restemperdet_tol:=MachEps*10;
   restemperdet_big:=1E+15;
   restemperdet_del:=1;
   restemperdet_fac:=2;
   for i:=1 to WordCount(types,ScanSpaces) do begin
    emax:=0; tmax:=0;
    idn:=restemperdet_idn(ExtractWord(i,types,ScanSpaces));
    l:=restemperdet_lot(idn); if isNAN(l) then continue;
    h:=restemperdet_hit(idn); if isNAN(h) then continue;
    if idn=0 then continue;
    for j:=1 to 9 do begin
     s:=GetArgumentAsString('Script#'+IntToStr(j));
     if s<>'' then Interpreter.Script:=Interpreter.Script+s+CRLF;
    end;
    if Interpreter.SetValue('l',l)    and
       Interpreter.SetValue('h',h)    and
       (Interpreter.RunScript=0)      and
       Interpreter.GetValue('a',a)    and
       Interpreter.GetValue('b',b)    and
       Interpreter.GetValue('n',n)    and
       not isNAN(a) and not isINF(a)  and
       not isNAN(b) and not isINF(b)  and
       not isNAN(n) and not isINF(n)  and
       (a<b) and (Round(n)>=1) and (Round(n)<=MaxNumPoints)
    then {Ok}
    else Raise EDanApi.Create(RusEng(' !','Interpreter error!'));
    crv:=CreateCurve(twin,'GOST_RTD_'+restemperdet_ids(idn),0,$10,'RTD type '+restemperdet_ids(idn));
    CurveLength[crv]:=Round(n);
    for j:=0 to CurveLength[crv]-1 do begin
     p.x:=a+j*(b-a)/(CurveLength[crv]-1);
     p.y:=restemperdet_ohm(p.x,idn);
     if (idn<>Ord('B')) or (p.x>60) then begin // Skip type B at t<60
      emax:=max(emax,abs(p.y-restemperdet_ohm(restemperdet_ttc(p.y,idn),idn)));
      tmax:=max(tmax,abs(p.x-restemperdet_ttc(p.y,idn)));
      inc(ncount,2);
     end;
     if not isNAN(p.x) and not isNAN(p.y) and
        not isINF(p.x) and not isINF(p.y)
     then CurvePoint[crv,j]:=p
     else Raise EDanApi.Create(RusEng(' !','Calculation error!'));
     Refreshment(100);
    end;
    writeln('Type ',restemperdet_ids(idn):2,'  emax=',emax,'  tmax=',tmax,'  test=',tmax+emax<1E-6);
    if ncount>0 then writeln(restemperdet_cnt/ncount:1:0,' iterations per restemperdet_ttc');
   end;
  finally
   Interpreter.Free;
  end;
 except
  on E:Exception do begin
   if WindowExists(twin) then CurvesCount[twin]:=0; 
   if UsesBlaster then Voice('EXCEPTION');
   Error(E.Message);
   Echo(E.Message);
   Result:=-1;
  end;
 end;
end;

exports CRW32_PLUGIN name CRW32_PLUGIN_ID;
begin
end.
 
