////////////////////////////////////////////////////////////////////////////////////////////////////
// Physics of Vapor unit: vapor pressure, humidity calculations etc
////////////////////////////////////////////////////////////////////////////////////////////////////
// Reference list:
// 1)  ГОСТ Р 8.811-2012.
//     Таблицы психрометрические. Построение, содержание, расчетные соотношения.
//     http://meganorm.ru/Data2/1/4293780/4293780569.pdf
//     http://www.gostexpert.ru/gost/gost-8.811-2012
// 2)  WMO-2008.
//     Guide to Meteorological Instruments and Methods of Observation. WMO-No.8 2008.
//     http://www.wmo.int/pages/prog/www/IMOP/CIMO-Guide.html
//     http://library.wmo.int/pmb_ged/wmo_8_en-2012.pdf
// 3)  NPL-1996.
//     A Guide to the Measurement of Humidity. The Institute of Measurement and Control,1996,London.
//     http://content.rotronic-usa.com/rs/rotronicinstrumentcorp/images/NPL%20Guide%20to%20Humidity.pdf
// 4)  Holger Voemel.
//     http://cires.colorado.edu/~voemel/home.html
//     http://cires.colorado.edu/~voemel/vp.html
//     http://cires.colorado.edu/~voemel/vp.pro
// 5)  Buck-1981.
//     Buck, A.L., New equations for computing vapor pressure and enhancement factor,
//     J. Appl. Meteorol., 20, 1527-1532, 1981.
//     http://www.public.iastate.edu/~bkh/teaching/505/arden_buck_sat.pdf
//     http://journals.ametsoc.org/doi/pdf/10.1175/1520-0450%281981%29020%3C1527%3ANEFCVP%3E2.0.CO%3B2
// 6)  Alduchov-1996.
//     Alduchov A, Eskrige R. Improved Magnus Form Approximation of Saturation Vapor Pressure.
//     Journal of applied meteorology, 1996, Vol.35, 601-609.
//     http://journals.ametsoc.org/doi/pdf/10.1175/1520-0450%281996%29035%3C0601%3AIMFAOS%3E2.0.CO%3B2
// 7)  Marti-1993.
//     Marti, J. and K Mauersberger, A survey and new measurements of ice vapor pressure
//     at temperatures between 170 and 250 K, GRL 20, 363-366, 1993.
//     https://www.researchgate.net/publication/23814668
// 8)  Hardy-1998.
//     Hardy B. ITS-90 formulations for vapor pressure, frostpoint temperature, dewpoint temperature,
//     and enhancement factors in the range –100 to +100 C
//     The Proceedings of the Third International Symposium on Humidity & Moisture, Teddington,
//     London, England, April 1998
//     http://www.thunderscientific.com/tech_info/reflibrary/its90formulas.pdf
// 9)  Davis-2008.
//     Davis R S et al. Revised formula for the density of moist air (CIPM-2007).
//     Metrologia 45 (2008) 149–155.
//     http://wahiduddin.net/calc/calc_refs/CIPM-2007.pdf
// 10) Davis-1992.
//     Davis R S. Equation for the determination of the density of moist air (1981/91).
//     Metrologia 29 (1992) 67–70.
//     http://www.nist.gov/calibrations/upload/metv29i1p67-2.pdf
// 11) Orlando-1999.
//     Orlando A.F. Calculation of humidity parameters and uncertainties using different
//     formulations and softwares.
//     http://www.inmetro.gov.br/producaointelectual/obras_intelectuais/226_obraIntelectual.pdf
// 12) Sonntag-1994.
//     Sonntag, D., Advancements in the field of hygrometry, Meteorol. Z., N. F., 3, 51-66, 1994.
// 13) Sonntag-1990.
//     Sonntag, D, “Important new values of the physical constants of 1986, vapour pressure
//     formulations based on the ITS-90 and psychrometer formulae”, Zeitschrift fur Meteorologie,
//     1990, 40(5), 340-344.
// 14) Bolton-1980. 
//     Bolton, D., The computation of equivalent potential temperature, Monthly Weather Report,
//     108, 1046-1053, 1980. equation (10).
// 15) www.easycalculation.com
//     https://www.easycalculation.com/formulas/relative-humidity-dew-point.html
//     https://www.easycalculation.com/weather/dewpoint-wetbulb-calculator.php
// 16) planetcalc.ru
//     http://planetcalc.ru/2161/
// 17) Wexler-1977.
//     Wexler, A., Vapor pressure formulation for ice, Journal of Research of the National
//     Bureau of Standards-A. 81A, 5-20, 1977.
//     http://nvlpubs.nist.gov/nistpubs/jres/081/1/V81.N01.A02.pdf
// 18) Wexler-1983.
//     Hyland, R. W. and A. Wexler, Formulations for the Thermodynamic Properties of the
//     saturated Phases of H2O from 173.15K to 473.15K, ASHRAE Trans, 89(2A), 500-519, 1983.
//     http://www.caee.utexas.edu/prof/Novoselac/classes/ARE383/Handouts/F01_06SI.pdf
// 19) Buck-2012.
//     Buck Research, Model CR-1A Hygrometer Operating Manual, May 2012.
//     http://www.hygrometers.com/wp-content/uploads/CR-1A-users-manual-2009-12.pdf
// 20) Murray-1967.
//     Murray, F.W., On the computation of saturation vapor pressure, J. Appl.Meteorol., 6, 203-204, 1967.
//     http://journals.ametsoc.org/doi/pdf/10.1175/1520-0450%281967%29006%3C0203%3AOTCOSV%3E2.0.CO%3B2
// 21) Fukuta-2003.
//     Fukuta N. Gramada C.M.,Vapor pressure measurement of supercooled water, J.Atmos.Sci.,60,1871-1875,2003.
//     http://homework.sdmesa.edu/mgramada/JAS2003Vol60.pdf
// 22) Wagner-2002.
//     Wagner W. and A. Pruss (2002), The IAPWS formulation 1995 for the thermodynamic properties of
//     ordinary water substance for general and scientific use, J.Phys.Chem.Ref.Data,31(2),387-535.
//     http://www.teos-10.org/pubs/Wagner_and_Pruss_2002.pdf
//     http://www.thermophysics.ru/pdf_doc/IAPWS_1995.pdf
// 23) Noppel-2002.
//     Noppel M et al. An improved parameterization for sulfuric acid–water nucleation rates
//     for tropospheric and stratospheric conditions. J.Geophys.Res.(2002),107,doi:10.1029/2002JD002184.
// 24) Koop-2005.
//     Murphy and Koop, Review of the vapour pressure of ice and supercooled water for atmospheric
//     applications, Q. J. R. Meteorol. Soc (2005), 131, pp. 1539-1565.
//     http://onlinelibrary.wiley.com/doi/10.1256/qj.04.94/pdf
// 25) Wiederhold-1997.
//     Pieter R. Wiederhold. Water Vapor Measurement: Methods and Instrumentation. CRC Press. 1997.
//     http://bookzz.org/book/941466/c12405   http://booksee.org/book/613798
//     http://bookre.org/reader?file=1137566
// 26) http://mc-computing.com/Science_Facts/Water_Vapor/
// 27) Osborne-1934.
//     Nathan S. Osborne and Cyril H. Meyers
//     A FORMULA AND TABLES FOR THE PRESSURE OF SATURATED WATER VAPOR IN THE RANGE 0 TO 374 C
//     U.S. Department of Commerce National Bureau of Standards. RESEARCH PAPER RP691.
//     Part of Journal of Research of the Rational Bureau of Standards, vol. 13, July 1934
//     http://nvlpubs.nist.gov/nistpubs/jres/13/jresv13n1p1_a2b.pdf
////////////////////////////////////////////////////////////////////////////////////////////////////
// Other humidity calculators:
//  http://www.humcal.com/index.php
//  http://www.michell.com/us/calculator/
//  http://respirometry.org/calculator/water-vapor-calculators
//  http://www.conservationphysics.org/atmcalc/atmoclc1.php
//  http://www.humidity-calculator-online.com/
//  http://www.humidity-calculator-online.com/fileadmin/Humidity_calculator/EEHumidityCalculator.swf
////////////////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////////////////////////
// Return enhancement factor to calculate saturated vapor pressure in moist air.
// Vapor pressure factor of moist air vs pressure p [mBar] & temperature t [°C].
// es is ideal sapuration vapor pressure [mBar]. For pure phase uses factor = 1.
// mFactor (mFactor=-6..6,0=default) specify which formula uses for calculation.
// mFactor=0 means 1 (pure phase), mFactor<0 means ice, mFactor>0 means water.
////////////////////////////////////////////////////////////////////////////////////////////////////
function vapor_efactor(p,t,es:Real; mFactor:Integer):Real;
const
 Ice = -1; // Uses for phase ice (frost)
 Dew = +1; // Uses for phase dew (water)
var factor:Real; Phase:Integer;
 //
 // 1) WMO-2008. Annex 4.B.
 //
 procedure Dew_WMO;
 begin
  factor:=1.0016+3.15E-6*p-0.074/p;
 end;
 procedure Ice_WMO;
 begin
  factor:=1.0016+3.15E-6*p-0.074/p;
 end;
 //
 // 1) Buck-1981.
 //
 procedure Dew_Buck;
 begin
  factor:=1.0007+3.46E-6*p;
 end;
 procedure Ice_Buck;
 begin
  factor:=1.0003+4.18E-6*p;
 end;
 //
 // 1) Alduchov-1996.
 //
 procedure Dew_Alduchov;
 begin
  factor:=1.00071*Exp(4.5E-6*p);
 end;
 procedure Ice_Alduchov;
 begin
  factor:=0.99882*Exp(8.0E-6*p);
 end;
 //
 // 1) Hardy-1998.
 // 2) www.humidity-calculator-online.com (uses IPTS-90)
 //
 function Hardy(p,t,es,a0,a1,a2,a3,b0,b1,b2,b3:Real):Real;
 var f,a,b:Real;
 begin
  a:=a0+(a1+(a2+a3*t)*t)*t;
  b:=b0+(b1+(b2+b3*t)*t)*t; b:=Exp(b);
  f:=Exp(a*(1-es/p)+b*(p/es-1));
  Hardy:=f;
 end;
 procedure Dew_Hardy68; // IPTS-68
 begin
  if t<0
  // IPTS-68 [°C] For Water –50 to 0°C
  then factor:=Hardy(p,t,es,  3.62183E-4,  2.60553E-5,  3.86501E-7,  3.82449E-9,
                             -1.07604E+1,  6.39725E-2, -2.63416E-4,  1.67254E-6)
  // IPTS-68 [°C] For Water 0 to 100°C
  else factor:=Hardy(p,t,es,  3.53624E-4,  2.93228E-5,  2.61474E-7,  8.57538E-9,
                             -1.07588E+1,  6.32529E-2, -2.53591E-4,  6.33784E-7);
 end;
 procedure Ice_Hardy68; // IPTS-68
 begin
  // IPTS-68 [°C] For Ice –100 to 0°C
  factor:=Hardy(p,t,es,  3.64449E-4,  2.93631E-5,  4.88635E-7,  4.36543E-9,
                        -1.07271E+1,  7.61989E-2, -1.74771E-4,  2.46721E-6);
 end;
 procedure Dew_Hardy90; // IPTS-90
 begin
  if t<0
  // IPTS-90 [°C] For Water –50 to 0°C
  then factor:=Hardy(p,t,es,  3.62183E-4,  2.6061244E-5,  3.8667770E-7, 3.8268958E-9,
                             -1.07604E+1,  6.3987441E-2, -2.6351566E-4, 1.6725084E-6)
  // IPTS-90 [°C] For Water 0 to 100°C
  else factor:=Hardy(p,t,es,  3.53624E-4,  2.9328363E-5,  2.6168979E-7, 8.5813609E-9,
                             -1.07588E+1,  6.3268134E-2, -2.5368934E-4, 6.3405286E-7);
 end;
 procedure Ice_Hardy90; // IPTS-90
 begin
  // IPTS-90 [°C] For Ice –100 to 0°C
  factor:=Hardy(p,t,es,  3.64449E-4,  2.9367585E-5,  4.8874766E-7,  4.3669918E-9,
                        -1.07271E+1,  7.6215115E-2, -1.7490155E-4,  2.4668279E-6);
 end;
 //
 // Define Default formula there.
 //
 procedure Dew_Default; begin Dew_WMO; end;
 procedure Ice_Default; begin Ice_WMO; end;
begin
 factor:=1;       // Uses for pure phase
 if mFactor<0     // Sign of mFactor means Phase
 then Phase:=Ice  // mFactor<0 means ice (frost)
 else Phase:=Dew; // mFactor>0 means dew (water)
 if p>0 then
 if es>0 then
 if es<=p then
 if Abs(mFactor) >=1  then
 if Abs(mFactor) <=5  then
 case Abs(mFactor) of
  1:  if Phase=Ice then Ice_Default  else Dew_Default;
  2:  if Phase=Ice then Ice_WMO      else Dew_WMO;
  3:  if Phase=Ice then Ice_Buck     else Dew_Buck;
  4:  if Phase=Ice then Ice_Alduchov else Dew_Alduchov;
  5:  if Phase=Ice then Ice_Hardy68  else Dew_Hardy68;
  6:  if Phase=Ice then Ice_Hardy90  else Dew_Hardy90;
 end;
 vapor_efactor:=max(factor,1);
end;

////////////////////////////////////////////////////////////////////////////////////////////////////
// Return saturated vapor pressure [mBar] for given vapor temperature t [°C].
// pa is air pressure [mBar]. Use pa=0 for pure phase, or pa>0 for moist air.
// nFormula (1..25,default=1) specifies one of popular formulas found in www.
// Sign of nFormula is phase: nFormula>0 means pure water, nFormula<0 is ice.
////////////////////////////////////////////////////////////////////////////////////////////////////
function vapor_pressure(t,pa:Real; nFormula,mFactor:Integer):Real;
const
 TZ         = 273.15;   // Celsius [°C] zero, [K]
 T0         = 273.16;   // Triple water point temperature, [K]
 Ts         = 373.16;   // Steam  water point temperature, [K]
 ei0        = 6.1071;   // Saturation pressure at triple point, ice
 ews        = 1013.246; // Saturation pressure at steam point temperature, normal atmosphere
 Ice        = -1;       // Uses for phase ice (frost)
 Dew        = +1;       // Uses for phase dew (water)
var TK,Psat:Real; Phase:Integer;
 //
 // Magnus 1844 formula. See Buck-1981.
 //
 function Magnus(t,a,b,c:Real):Real;
 begin
  Magnus:=a*Exp(b*t/(t+c));
 end;
 function Magnus10(t,a,b,c:Real):Real;
 begin
  Magnus10:=a*Power(10,b*t/(t+c));
 end;
 //
 // Bogel 1979 formula. See Buck-1981.
 //
 function Bogel(t,a,b,c,d:Real):Real;
 begin
  Bogel:=a*Exp((b-t/d)*t/(t+c));
 end;
 //
 // See Wexler-1983, Koop-2005, Sonntag-1990, Sonntag-1994, Holger Vomel, Orlando-1999.
 // It's very common function, includes many other formulas.
 //
 function HylandWexler(t,g0,g1,g2,g3,g4,g5,g6,g7:Real):Real;
 begin
  HylandWexler:=Exp((g0/t+g1)/t+g2+(g3+(g4+(g5+g6*t)*t)*t)*t+g7*Ln(t));
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.19, Приложение И (обязательное), формула (И.1), диапазон (-60,100)°C.
 //
 procedure Dew_GOST2012I1;
 begin
  Psat:=Power(10,10.79574*(1-T0/TK)-5.02800*Log(10,TK/T0)+1.50475E-4*(1-Power(10,-8.2969*(TK/T0-1)))
             +0.42873E-3*(Power(10,4.76955*(1-T0/TK))-1)+0.78614);
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.19, Приложение И (обязательное), формула (И.2), диапазон (-90,0)°C.
 //
 procedure Ice_GOST2012I2;
 begin
  Psat:=Power(10,-9.09685*(T0/TK-1)-3.56654*Log(10,T0/TK)+0.87682*(1-TK/T0)+0.78614);
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.19, Приложение И (обязательное), формула (И.3), диапазон (-45,60)°C.
 // 2) WMO-2008. Annex 4.B. Water (–45 to 60°C) (pure phase).
 // 3) NPL-1996. Page 53, Formula (7).
 // 4) planetcalc.ru
 // 5) Holger Voemel.
 // 6) Sonntag-1990.
 //
 procedure Dew_GOST2012I3;
 begin
  Psat:=Magnus(t, 6.112, 17.62, 243.12);
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.19, Приложение И (обязательное), формула (И.4), диапазон (-60,0)°C.
 // 2) WMO-2008. Annex 4.B. Ice (–65 to 0°C) (pure phase).
 // 3) NPL-1996. Page 53, Formula (8).
 //
 procedure Ice_GOST2012I4;
 begin
  Psat:=Magnus(t, 6.112, 22.46, 272.62);
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.24, Приложение К (справочное), формула (К.1), диапазон (-60,100)°C.
 //
 procedure Dew_GOST2012K1;
 begin
  Psat:=HylandWexler(TK, 0, -6096.9385, 16.635794, -2.711193E-2, 1.673952E-5, 0, 0, 2.433502);
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.24, Приложение К (справочное), формула (К.2), диапазон (-90,0)°C.
 //
 procedure Ice_GOST2012K2;
 begin
  Psat:=HylandWexler(TK, 0, -6024.5282, 24.7219, 1.0613868E-2, -1.3198825E-5, 0, 0, -0.49382577);
 end;
 //
 // 1) ГОСТ Р 8.811-2012,стр.24, Приложение М (справочное), формула (М.2), диапазон (-30,50)°C.
 //
 procedure Dew_GOST2012M2;
 begin
  Psat:=Magnus(t, 6.1121, 17.5043, 241.2);
 end;
 //
 // 1) ГОСТ Р 8.811-2012, стр.24, Приложение М (справочное), формула (М.2), диапазон (-60,0)°C.
 //
 procedure Ice_GOST2012M2;
 begin
  Psat:=Magnus(t, 6.1121, 22.4893, 272.881);
 end;
 //
 // 1) Bolton-1980. equation (10).
 // 2) www.easycalculation.com
 //
 procedure Dew_Bolton;
 begin
  Psat:=Magnus(t, 6.112, 17.67, 243.50);
 end;
 //
 // Modification of Magnus formula from Tim Brice
 // 1) http://www.srh.noaa.gov/epz/?n=wxcalc_vaporpressure
 //    http://www.srh.noaa.gov/images/epz/wxcalc/vaporPressure.pdf
 // 2) http://www.shodor.org/os411/courses/_master/tools/calculators/satmixratio/
 //
 procedure Dew_Brice;
 begin
  Psat:=Magnus10(t, 6.11, 7.5, 237.3);
 end;
 //
 // Modification of Magnus formula
 // 1) http://www.csgnetwork.com/vaporpressurecalc.html
 //    https://www.easycalculation.com/weather/water-vapor-pressure.php
 //    http://www.mmsn.org/tools/pressure-conversion-tools.html#VP
 //    http://www.4wx.com/wxcalc/vaporpressure.html
 //
 procedure Dew_wxCalc;
 begin
  Psat:=Magnus10(t, 6.11, 7.5, 237.7);
 end;
 //
 // 1) Marti-1993.
 //
 procedure Ice_Marti;
 begin
  Psat:=Power(10,-2663.5/TK+12.537)/100;
 end;
 //
 // 2) Davis-1992.
 // 1) Davis-2008.
 //
 procedure Dew_CIPM81;
 begin
  Psat:=HylandWexler(TK, 0, -6.3536311E3, 34.04926034, -1.9509874E-2, 1.2811805E-5, 0, 0, 0)/100;
 end;
 procedure Dew_CIPM91;
 begin
  Psat:=HylandWexler(TK, 0, -6.3431645E3, 33.93711047, -1.9121316E-2, 1.2378847E-5, 0, 0, 0)/100;
 end;
 //
 // 1) Wexler-1977.
 // 2) Hardy-1998.
 // 3) Koop-2005.
 //
 procedure Dew_Wexler68; // IPTS-68 temperature scale
 begin
  Psat:=HylandWexler(TK, -2.9912729E3, -6.0170128E3, 1.887643854E1, -2.8354721E-2, 1.7838301E-5,
              -8.4150417E-10, 4.4412543E-13, 2.858487)/100;
 end;
 procedure Dew_Wexler90; // IPTS-90 temperature scale
 begin
  Psat:=HylandWexler(TK, -2.8365744E3, -6.028076559E3, 1.954263612E1, -2.737830188E-2, 1.6261698E-5,
              7.0229056E-10,  -1.8680009E-13, 2.7150305)/100;
 end;
 //
 // 1) Wexler-1983.
 //
 procedure Dew_HylandWexler;
 begin
  Psat:=HylandWexler(TK, 0, -0.58002206E4, 0.13914993E1, -0.48640239E-1, 0.41764768E-4,
              -0.14452093E-7, 0, 0.65459673E1)/100;
 end;
 procedure Ice_HylandWexler;
 begin
  Psat:=HylandWexler(TK, 0, -0.56745359E4, 0.63925247E1, -0.96778430E-2, 0.62215701E-6,
              +0.20747825E-8, -0.94840240E-12, 0.41635019E1)/100;
 end;
 //
 // 1) Buck-1981.
 // 2) Wiederhold-1997.
 //
 procedure Dew_Buck;
 begin
  Psat:=Magnus(t, 6.1121, 17.502, 240.97);
 end;
 procedure Ice_Buck;
 begin
  Psat:=Magnus(t, 6.1115, 22.452, 272.55);
 end;
 //
 // Bucks vapor pressure formulation based on Bogel formula
 // 1) Buck-1981, Buck-2012.
 // 2) https://en.wikipedia.org/wiki/Arden_Buck_equation
 //
 procedure Dew_Buck2;
 begin
  Psat:=Bogel(t, 6.1121, 18.678, 257.14, 234.5);
 end;
 procedure Ice_Buck2;
 begin
  Psat:=Bogel(t, 6.1115, 23.036, 279.82, 333.7);
 end;
 //
 // Goff Gratch formulation
 // 1) Smithsonian Meteorological Tables, 5th edition, p. 350, 1984
 //    From original source: Goff and Gratch (1946), p. 107.
 // 2) https://en.wikipedia.org/wiki/Goff%E2%80%93Gratch_equation
 // 3) Murray-1967.
 //
 procedure Dew_GoffGratch;
 begin
  Psat:=Power(10,-7.90298*(Ts/TK-1)+5.02808*Log(10,Ts/TK)-1.3816E-7*(Power(10,11.344*(1-TK/Ts))-1)
             +8.1328E-3*(Power(10,-3.49149*(Ts/TK-1))-1)+Log(10,ews));
 end;
 procedure Ice_GoffGratch;
 begin
  Psat:=Power(10,-9.09718*(T0/TK-1)-3.56654*Log(10,T0/TK)+0.876793*(1-TK/T0)+Log(10,ei0));
 end;
 //
 // 1) Murray-1967.
 //
 procedure Dew_MagnusTetens;
 begin
  Psat:=Power(10,7.5*t/(t+237.3)+0.7858);
 end;
 procedure Ice_MagnusTetens;
 begin
  Psat:=Power(10,9.5*t/(265.5+t)+0.7858);
 end;
 procedure Dew_Murray;
 begin
  Psat:=6.1078*Exp(17.2693882*(TK-T0)/(TK-35.86));  
 end;
 procedure Ice_Murray;
 begin
  Psat:=6.1078*Exp(21.8745584*(TK-T0)/(TK-7.66)); 
 end;
 //
 // 1) Koop-2005.
 //
 procedure Dew_MurphyKoop;
 begin
  Psat:=Exp(54.842763-6763.22/TK-4.210*Ln(TK)+0.000367*TK
           +TANH(0.0415*(TK-218.8))*(53.878-1331.22/TK-9.44523*Ln(TK)+0.014025*TK))/100;
 end;
 procedure Ice_MurphyKoop;
 begin
  Psat:=HylandWexler(TK, 0, -5723.265, 9.550426, -0.00728332, 0, 0, 0, 3.53068)/100;
 end;
 //
 // 1) Sonntag-1994.
 //
 procedure Dew_Sonntag;
 begin
  Psat:=HylandWexler(TK, 0, -6096.9385, 16.635794, -2.711193E-2, 1.673952E-5, 0, 0, 2.433502);
 end;
 procedure Ice_Sonntag;
 begin
  Psat:=HylandWexler(TK, 0, -6024.5282, 24.721994, 1.0613868E-2, -1.3198825E-5, 0, 0, -0.49382577);
 end;
 //
 // WMO formulation, which is very similar to Goff Gratch
 // Intended WMO formulation, originally published by Goff (1957)
 // incorrectly referenced by WMO technical regulations, WMO-NO 49, Vol I, General Meteorological
 // Standards and Recommended Practices, App. A, Corrigendum Aug 2000.
 // and incorrectly referenced by WMO technical regulations, WMO-NO 49, Vol I, General Meteorological
 // Standards and Recommended Practices, App. A, 1988.
 // 1) WMO technical regulations, WMO-NO 49, Vol I, General Meteorological Standards and Recommended
 //    Practices, Aug 2000, App. A.
 //
 procedure Dew_WMO;
 begin
  Psat:=Power(10,10.79574*(1-T0/TK)-5.02800*Log(10,TK/T0)+1.50475E-4*(1-Power(10,-8.2969*(TK/T0-1)))
             +0.42873E-3*(Power(10,+4.76955*(1-T0/TK))-1)+0.78614);
 end;
 procedure Ice_WMO;
 begin
  Psat:=Power(10,-9.09685*(T0/TK-1)-3.56654*Log(10,T0/TK)+0.87682*(1-TK/T0)+0.78614);
 end;
 //
 // 1) Noppel-2002.
 //
 procedure Dew_Preining;
 begin
  Psat:=HylandWexler(TK, 0, -7235.424651, 77.34491296, 5.7113E-3, 0, 0, 0, -8.2)/100;
 end;
 //
 // 1) Alduchov-1996. Range (-40,50),(-80,0).
 //
 procedure Dew_Alduchov;
 begin
  Psat:=Magnus(t, 6.1094, 17.625, 243.04);
 end;
 procedure Ice_Alduchov;
 begin
  Psat:=Magnus(t, 6.1121, 22.587, 273.86);
 end;
 //
 // 1) Wagner-2002.
 // This is the 'official' formulation from the International Association for the Properties of Water and Steam
 // The valid range of this formulation is 273.16 <= t <= 647.096 K and is based on the ITS90 temperature scale.
 //
 procedure Dew_IAPWS;
 var Tc,Pc,nu,a1,a2,a3,a4,a5,a6:Real;
 begin
  Tc:=647.096;  // Temperature at the critical point, [K]
  Pc:=22.064E4; // Vapor pressure at the critical point, [hPa]
  nu:=(1-TK/Tc);
  a1:=-7.85951783; a2:=1.84408259; a3:=-11.7866497; a4:=22.6807411; a5:=-15.9618719; a6:=1.80122502;
  Psat:=Pc*Exp(Tc/TK*(a1*nu+a2*Power(nu,1.5)+a3*Power(nu,3)+a4*Power(nu,3.5)+a5*Power(nu,4)+a6*Power(nu,7.5)));
 end;
 //
 // 1) Fukuta-2003.
 // This paper does not give a vapor pressure formulation, but rather a correction over the Smithsonian Tables.
 // Thus calculate the table value first, then use the correciton to get to the measured value.
 //
 procedure Dew_Fukuta;
 var x:Real;
 begin
  Dew_GoffGratch; x:=t+19;
  Psat:=Psat*(0.9992+(7.113E-4+(-1.847E-4+(1.189E-5+(1.130E-7-1.743E-8*x)*x)*x)*x)*x);
 end;
 //
 // Define Default formula there.
 //
 procedure Dew_Default; begin Dew_GOST2012I1; end;
 procedure Ice_Default; begin Ice_GOST2012I2; end;
begin
 Psat:=_NaN;      // Not defined yet
 TK:=TZ+t;        // Celsius to Kelvins
 if nFormula<0    // Sign of nFormula means Phase
 then Phase:=Ice  // nFormula<0 means ice (frost)
 else Phase:=Dew; // nFormula>0 means dew (water)
 if Abs(nFormula)<1  then nFormula:=Phase;
 if Abs(nFormula)>25 then nFormula:=Phase;
 case Abs(nFormula) of
  1:  if Phase=Ice then Ice_Default      else Dew_Default;
  2:  if Phase=Ice then Ice_GOST2012I2   else Dew_GOST2012I1;
  3:  if Phase=Ice then Ice_GOST2012I4   else Dew_GOST2012I3;
  4:  if Phase=Ice then Ice_GOST2012K2   else Dew_GOST2012K1;
  5:  if Phase=Ice then Ice_GOST2012M2   else Dew_GOST2012M2;
  6:  if Phase=Ice then Ice_GoffGratch   else Dew_GoffGratch;
  7:  if Phase=Ice then Ice_HylandWexler else Dew_HylandWexler;
  8:  if Phase=Ice then Ice_Default      else Dew_Wexler68;
  9:  if Phase=Ice then Ice_Default      else Dew_Wexler90;
  10: if Phase=Ice then Ice_Sonntag      else Dew_Sonntag;
  11: if Phase=Ice then Ice_WMO          else Dew_WMO;
  12: if Phase=Ice then Ice_Buck         else Dew_Buck;
  13: if Phase=Ice then Ice_Buck2        else Dew_Buck2;
  14: if Phase=Ice then Ice_MagnusTetens else Dew_MagnusTetens;
  15: if Phase=Ice then Ice_Murray       else Dew_Murray;
  16: if Phase=Ice then Ice_MurphyKoop   else Dew_MurphyKoop;
  17: if Phase=Ice then Ice_Marti        else Dew_Default;
  18: if Phase=Ice then Ice_Default      else Dew_Bolton;
  19: if Phase=Ice then Ice_Default      else Dew_Brice;
  20: if Phase=Ice then Ice_Default      else Dew_wxCalc;
  21: if Phase=Ice then Ice_Default      else Dew_CIPM81;
  22: if Phase=Ice then Ice_Default      else Dew_CIPM91;
  23: if Phase=Ice then Ice_Default      else Dew_Preining;
  24: if Phase=Ice then Ice_Alduchov     else Dew_Alduchov;
  25: if Phase=Ice then Ice_Default      else Dew_IAPWS;
 end;
 vapor_pressure:=Psat*vapor_efactor(pa,t,Psat,mFactor);
end;
//
// Return relative humidity [%RH] by given dewpoint temperature td [°C] and air temperature ta [°C].
// pa is air pressure, [mBar].
//
function dewpoint_to_rh(td,ta,pa:Real; nFormula,mFactor:Integer):Real;
begin
 dewpoint_to_rh:=100*vapor_pressure(td,pa,nFormula,mFactor)/vapor_pressure(ta,pa,nFormula,mFactor);
end;
//
// Return volume moisture [PPMv] by given dewpoint temperature td [°C] and air pressure pa [mBar].
//
function dewpoint_to_ppmv(td,pa:Real; nFormula,mFactor:Integer):Real;
var pw:Real;
begin
 pw:=vapor_pressure(td,pa,nFormula,mFactor);
 if pa<=0 then pa:=atm_to_mbar(1);
 dewpoint_to_ppmv:=1e6*pw/(pa-pw);
end;
//
// Calculate which formula to use for vapor pressure,factor evaluation.
// n,m is wanted nFormula,mFactor formula, ta is air temperature, [°C].
// Function return n,m with correct sign to calculate vapor pressure or
// vapor factor depending on current air phase: dew=water or ice=frost.
// This function is useful because calculation of vapor pressure needed
// different formula depending on dew/ice phase. Usage looks like that:
// p:=vapor_pressure(t, pa, vapor_nFormula(n,ta), vapor_mFactor(m,ta));
// where:  t=vapor temperature, pa,ta=moist air pressure & temperature,
// n,m-wanted formula for pressure (n) and enhanced factor (m).
//
function vapor_nFormula(n:Integer; ta:Real):Integer;
const Ice=-1; Dew=+1;
var Phase:Integer;
begin
 if ta<0 then Phase:=Ice else Phase:=Dew;
 vapor_nFormula:=Phase*iMax(1,abs(n)*Ord(abs(n)<=25));
end;
function vapor_mFactor(m:Integer; ta:Real):Integer;
const Ice=-1; Dew=+1;
var Phase:Integer;
begin
 if ta<0 then Phase:=Ice else Phase:=Dew;
 vapor_mFactor:=Phase*abs(m)*Ord(abs(m)<=6);
end;
//
// Check computations by data given in literature.
//
procedure CheckHumidityCalc;
 procedure Check1(t,ival,wval:Real);
 const wid=12; dig=7;
 var ical,wcal,wdif,wrel,idif,irel,ippm,wppm:Real; n:Integer;
 begin
  Write('N':2,' ');
  Write('Cels'      :wid,'  ');
  Write('wval'      :wid,' ');
  Write('wcal'      :wid,' ');
  Write('wdif'      :wid,' ');
  Write('wrel,%'    :wid,' ');
  Write('wrel,ppm'  :wid,'   ');
  Write('ival'      :wid,' ');
  Write('ical'      :wid,' ');
  Write('idif'      :wid,' ');
  Write('irel,%'    :wid,' ');
  Write('irel,ppm'  :wid,'   ');
  Writeln;
  for n:=1 to 25 do begin
   ical:=vapor_pressure(t,1013.25,-n,0); idif:=ical-ival; irel:=100*abs(idif/ival); ippm:=irel*1E4;
   wcal:=vapor_pressure(t,1013.25,+n,0); wdif:=wcal-wval; wrel:=100*abs(wdif/wval); wppm:=wrel*1E4;
   Write(n:2,' ');
   Write(StrFix(t,wid,dig),'   ');
   Write(StrFix(wval,wid,dig),' ');
   Write(StrFix(wcal,wid,dig),' ');
   Write(StrFix(wdif,wid,dig),' ');
   Write(StrFix(wrel,wid,dig),' ');
   Write(StrFix(wppm,wid,0),'   ');
   Write(StrFix(ival,wid,dig),' ');
   Write(StrFix(ical,wid,dig),' ');
   Write(StrFix(idif,wid,dig),' ');
   Write(StrFix(irel,wid,dig),' ');
   Write(StrFix(ippm,wid,0),'   ');
   Writeln;
  end;
 end;
 procedure Check2(TK,iPa,wPa:Real);
 begin
  Check1(TK-273.15,iPa/100,wPa/100);
 end;
begin
 Writeln('Humidity calculation check. Time:',GetDateTime(mSecNow));
 {
 // ГОСТ Р 8.811-2012. Таблица Е.1.
 Check1( -79,    0.000644,   _NaN     );
 Check1( -60,    _NaN,       0.0190   );
 Check1( -2,     5.177201,   _NaN     );
 Check1( -1,     5.626739,   5.6821   );
 Check1(  0,     _NaN,       6.1121   );
 Check1(  1,     _NaN,       6.5708   );
 }
 {
 // Wexler-1977.
 Check1( -60,    1.08203E-2, _Nan    );
 Check1( -50,    3.94017E-2, _Nan    );
 Check1( -40,    0.128486,   _Nan    );
 Check1( -30,    0.380238,   _Nan    );
 Check1( -20,    1.03276,    _Nan    );
 Check1( -10,    2.59922,    _Nan    );
 Check1(   0,    6.11153,    _Nan    );
 }
 {
 // Murray-1967.
 Check1( -50,    0.03935,    0.06356 );
 Check1( -40,    0.1283,     0.1891  );
 Check1( -30,    0.3798,     0.5088  );
 Check1( -20,    1.032,      1.2540  );
 Check1( -10,    2.59922,    2.8627  );
 Check1(   0,    6.107,      6.1078  );
 Check1(  10,     _NaN,      12.272  );
 Check1(  20,     _NaN,      23.373  );
 Check1(  30,     _NaN,      42.430  );
 Check1(  40,     _NaN,      73.777  );
 Check1(  50,     _NaN,      123.40  );
 }
 {
 // Koop-2005. Appendix C. Table C1.
 Check2( 150,    6.106E-6,   1.562E-5 );
 Check2( 180,    0.0053975,  0.011239 );
 Check2( 210,    0.70202,    1.2335   );
 Check2( 240,    27.272,     37.667   );
 Check2( 273.15, 611.154,    611.213  );
 Check2( 273.16, 611.657,    611.657  );
 Check2( 300,    _NaN,       3536.8   );
 }
 {
 // Wiederhold-1997. Page 307. Table 13.6.
 Check1( 100,     _NaN,      1014.19 );
 Check1(  80,     _NaN,      476.95  );
 Check1(  60,     _NaN,      200.65  );
 Check1(  40,     _NaN,      74.21   );
 Check1(  20,     _NaN,      23.49   );
 Check1(  10,     _NaN,      12.33   );
 Check1(   0,     6.14,      6.14    );
 Check1( -10,     2.61,      _NaN    );
 Check1( -20,     1.03,      _NaN    );
 Check1( -40,     0.129,     _NaN    );
 Check1( -60,      0.011,    _NaN   );
 Check1( -80,     0.0006,    _NaN    );
 }
 {
 // http://mc-computing.com/Science_Facts/Water_Vapor/JPL_paper.html
 // GOST2012I1/WMO formula show best result; GOS2012TK1/Sonntag looks good too
 Check1(   0,     6.108,     6.108   );
 Check1(   0.01,  6.11657,   6.11657 );
 Check1(   10,    _NaN,      12.27   );
 Check1(   15,    _NaN,      17.04   );
 Check1(   20,    _NaN,      23.37   );
 Check1(   30,    _NaN,      42.43   );
 Check1(   40,    _NaN,      73.78   );
 }
 {
 // http://mc-computing.com/Science_Facts/Water_Vapor/JPL_paper.html
 // just to check GOST2012I1/WMO formula
 Check1(   0,     6.10695,   6.10695 );
 Check1(  10,     _NaN,      12.27089);
 Check1(  15,     _NaN,      17.04204);
 Check1(  20,     _NaN,      23.37080);
 Check1(  30,     _NaN,      42.42726);
 Check1(  40,     _NaN,      73.77329);
 }
 {
 // http://www.jma.go.jp/jma/jma-eng/jma-center/ric/material/1_Lecture_Notes/CP3-Humidity.pdf
 Check1( -30,     0.38,      _NaN    );
 Check1( -25,     0.63,      _NaN    );
 Check1( -20,     1.03,      _NaN    );
 Check1( -15,     1.65,      _NaN    );
 Check1( -10,     2.60,      _NaN    );
 Check1(  -5,     4.01,      _NaN    );
 Check1(   0,     6.11,      6.11    );
 Check1(   5,     _NaN,      8.72    );
 Check1(   10,    _NaN,      12.27   );
 Check1(   15,    _NaN,      17.04   );
 Check1(   20,    _NaN,      23.37   );
 Check1(   25,    _NaN,      31.67   );
 Check1(   30,    _NaN,      42.43   );
 }
 {
 // Osborne-1934.
 // HylandWexler show best result for t>200; for t<200 all formulas looks good enough
 Check1(  -5,     _NaN,      atm_to_mbar(0.004162));
 Check1(   0,     _NaN,      atm_to_mbar(0.006027));
 Check1(  50,     _NaN,      atm_to_mbar(0.12170));
 Check1( 100,     _NaN,      atm_to_mbar(1));
 Check1( 150,     _NaN,      atm_to_mbar(4.6975));
 Check1( 200,     _NaN,      atm_to_mbar(15.347));
 Check1( 250,     _NaN,      atm_to_mbar(39.256));
 Check1( 300,     _NaN,      atm_to_mbar(84.793));
 Check1( 350,     _NaN,      atm_to_mbar(163.20));
 Check1( 374,     _NaN,      atm_to_mbar(217.98));
 }
 {
 // http://www.michell.com/de/calculator/
 // just to check GOST2012K2         formula for ice (t<0)
 // just to check GOS2012TK1/Sonntag formula for dew (t<>0)
 Check1( 373.15,  _NaN,      247640);
 Check1( 350,     _NaN,      182810);
 Check1( 300,     _NaN,      90533);
 Check1( 250,     _NaN,      40647);
 Check1( 200,     _NaN,      15652);
 Check1( 150,     _NaN,      4766.6);
 Check1( 100,     _NaN,      1014.2);
 Check1(  50,     _NaN,      123.53);
 Check1(   0,     6.1121,    6.1121);
 Check1( -10,     2.5989,    2.8652);
 Check1( -20,     1.0324,    1.2559);
 Check1( -30,     0.37999,   0.51032);
 Check1( -40,     1.2837E-1, 1.9033E-1);
 Check1( -50,     3.9358E-2, 6.4388E-2);
 Check1( -60,     1.0807E-2, 1.9484E-2);
 Check1( -70,     2.6154E-3, 5.1867E-3);
 Check1( -80,     5.4725E-4, 1.1903E-3);
 Check1( -90,     9.6705E-5, 2.2981E-4);
 Check1(-100,     1.4021E-5, 3.6216E-5);
 }
end;
//
// Test humidity calculations.
//
procedure TestHumidityCalc(arg:String);
const wid=12; dig=4; m=0;
var n:Integer; td,ta,pa:Real;
begin
 td:=rValDef(ExtractWord(1,arg),10);
 ta:=rValDef(ExtractWord(2,arg),21);
 pa:=rValDef(ExtractWord(3,arg),atm_to_mbar(1));
 Writeln('******************************************************');
 Writeln('** Humidity calculation test. Time:',GetDateTime(mSecNow));
 Writeln('** T_dew[°C]=',Str(td),',  T_[air],°C=',Str(ta)+',  P_air[mBar]='+Str(pa));
 Writeln('******************************************************');
 Write('n':2,' ');
 Write('Dew,mBar':wid,' ');
 Write('Ice,mBar':wid,' ');
 Write('Moist,mBar':wid,' ');
 Write('%RH':wid,' ');
 Write('PPMv':wid,' ');
 Writeln;
 for n:=1 to 25 do begin
  Write(n:2,' ');
  Write(StrFix(vapor_pressure(td,pa,+n,m),wid,dig),' ');
  Write(StrFix(vapor_pressure(td,pa,-n,m),wid,dig),' ');
  Write(StrFix(vapor_pressure(ta,pa,n,m),wid,dig),' ');
  Write(StrFix(dewpoint_to_rh(td,ta,pa,n,m),wid,dig),' ');
  Write(StrFix(dewpoint_to_ppmv(td,pa,n,m),wid,dig),' ');
  Writeln;
 end;
end;
