////////////////////////////////////////////////////////////////////////////////////////////////////
// Physical units convertion routines
////////////////////////////////////////////////////////////////////////////////////////////////////
// Reference list:
// 1) ГОСТ 8.417-2002.
//    Государственная система обеспечения единства измерений. Единицы величин.
//    http://meganorm.ru/Data2/1/4294845/4294845105.pdf
//    http://gostexpert.ru/gost/gost-8.417-2002
// 2) BIPM-8-2006.
//    International Bureau of Weights and Measures (BIPM). The International System of Units (SI).
//    8th edition  2006.
//    http://www.bipm.org/utils/common/pdf/si_brochure_8_en.pdf
// 3) Taylor-2008.
//    Ambler Thompson, Barry N. Taylor. Guide for the Use of the International System of Units (SI).
//    NIST Special Publication 811. 2008 Edition.
//    http://www.nist.gov/customcf/get_pdf.cfm?pub_id=200349
////////////////////////////////////////////////////////////////////////////////////////////////////

//
// Convert pressure from standard atmospheres [atm] to millibars [mBar] = [hPa] and back.
// Note that 1 atm correspond to standard air pressure, 1013.25 mBar, 760 mmHg.
// 1 mBar = 1 hPa = 0.001 Bar = 100 Pa.
//
function atm_to_mbar(atm:Real):Real;
begin
 atm_to_mbar:=atm*1013.25;
end;
function mbar_to_atm(mbar:Real):Real;
begin
 mbar_to_atm:=mbar/1013.25;
end;

//
// Convert pressure from [mmHg] to [mBar] and back.
// 1 mmHg = 1 Torr.
//
function mmhg_to_mbar(mmhg:Real):Real;
begin
 mmhg_to_mbar:=mmhg/760*1013.25;
end;
function mbar_to_mmhg(mbar:Real):Real;
begin
 mbar_to_mmhg:=mbar*760/1013.25;
end;

//
// Convert pressure from [psi] to [mBar] and back.
// 1psi = 1 lbs = 6894.757293Pa = 68.94757293mBar.
// Known as pound-force per square inch, lbf/in^2.
// http://wiki.uz.cm/wiki/Pounds_per_square_inch
//
function psi_to_mbar(psi:Real):Real;
begin
 psi_to_mbar:=psi*68.94757293;
end;
function mbar_to_psi(mbar:Real):Real;
begin
 mbar_to_psi:=mbar/68.94757293;
end;

//
// Convert Celsius [°C] to Kelvins [K] and back.
//
function cels_to_kelv(cels:Real):Real;
begin
 cels_to_kelv:=cels+273.15;
end;
function kelv_to_cels(kelv:Real):Real;
begin
 kelv_to_cels:=kelv-273.15;
end;

//
// Convert Celsius [°C] to Fahrenheit [°F] and back.
// (0,100)°C = (32,212)°F.
//
function cels_to_fahr(cels:Real):Real;
begin
 cels_to_fahr:=cels*1.8+32;
end;
function fahr_to_cels(fahr:Real):Real;
begin
 fahr_to_cels:=(fahr-32)/1.8;
end;
