 {
 Functions for NetModbus.
 function modbus_get_tag(tag:Integer;def:Real):Real;
 function modbus_set_tag(tag:Integer; val:Real):Boolean;
 function modbus_inc_tag(tag:Integer; inc:Real):Boolean;
 procedure modbus_fixerror(tag,code:Integer; msg:String);
 function modbus_swap_int16(w:Integer):Integer;
 function modbus_swap2(dat:String; SwapMode:Integer):String;
 function modbus_swap4(dat:String; SwapMode:Integer):String;
 function modbus_swap8(dat:String; SwapMode:Integer):String;
 function modbus_dump_int16(dat:Integer; SwapMode:Integer):String;
 function modbus_dump_int32(dat:Integer; SwapMode:Integer):String;
 function modbus_dump_float(dat:Real; SwapMode:Integer):String;
 function modbus_dump_double(dat:Real; SwapMode:Integer):String;
 function modbus_take_int16(dat:String; SwapMode:Integer):Integer;
 function modbus_take_int32(dat:String; SwapMode:Integer):Integer;
 function modbus_take_float(dat:String; SwapMode:Integer):Real;
 function modbus_take_double(dat:String; SwapMode:Integer):Real;
 function modbus_is_except(fid:Integer):Boolean;
 function modbus_un_except(fid:Integer):Integer;
 function modbus_as_except(fid:Integer; isOn:Boolean):Integer;
 function modbus_addr_ok(addr:Integer):Boolean;
 function modbus_func_ok(func:Integer):Boolean;
 function modbus_prot_ok(prot:Integer):Boolean;
 function modbus_prot_name(prot:Integer):String;
 function modbus_prot_alias(prot:Integer):String;
 function modbus_prot_byname(name:String):Integer;
 function modbus_calc_lrc(buf:String):Integer;
 function modbus_calc_crc(buf:String):Integer;
 function modbus_join_lrc(dat:String):String;
 function modbus_join_crc(dat:String):String;
 function modbus_encode_ip(tid,pid,uid,fid:Integer; dat:String):String;
 function modbus_decode_ip(adu:String; var tid,pid,uid,fid:Integer; var dat:String):Integer;
 function modbus_encode_rtu(uid,fid:Integer; dat:String):String;
 function modbus_decode_rtu(adu:String; var uid,fid:Integer; var dat:String):Integer;
 function modbus_encode_ascii(uid,fid:Integer; dat:String):String;
 function modbus_decode_ascii(adu:String; var uid,fid:Integer; var dat:String):Integer;
 function modbus_encode_adu(tid,pid,uid,fid:Integer; dat:String):String;
 function modbus_decode_adu(adu:String; prot:Integer; var tid,pid,uid,fid:Integer; var dat:String):Integer;
 function modbus_encode_pdu(dir:Char; fid:Integer; saddr,quant:Integer; raw:String):String;
 function modbus_decode_pdu(dir:Char; fid:Integer; dat:String; var saddr,quant:Integer; var raw:String):Integer;
 function modbus_ext_byte(raw:String; byteoffs:Integer):Integer;
 function modbus_ext_int16(raw:String; regoffs,SwapMode:Integer):Integer;
 function modbus_ext_int32(raw:String; regoffs,SwapMode:Integer):Integer;
 function modbus_ext_uint32(raw:String; regoffs,SwapMode:Integer):Real;
 function modbus_ext_float(raw:String; regoffs,SwapMode:Integer):Real;
 function modbus_ext_uint64(raw:String; regoffs,SwapMode:Integer):Real;
 function modbus_ext_coil(raw:String; regoffs:Integer):Integer;
 function modbus_proxy_poll(cmd:String; ref,cid,tot,port,uid,fid:Integer; dat:String):String;
 function modbus_proxy_nice(cmd:String; ref,cid,tim,port,uid,fid:Integer; dat:String; wid:Integer):String;
 function modbus_proxy_reply(cmd,arg:String; var ref,cid,tim,port,uid,fid:Integer; var dat:String):Boolean;
 function modbus_errmsg(errno:Integer):String;
 procedure ClearNetModbus;
 procedure InitNetModbus;
 procedure FreeNetModbus;
 procedure PollNetModbus;
 }
 //
 // Get integer or real tag value or default.
 //
 function modbus_get_tag(tag:Integer; def:Real):Real;
 begin
  if TypeTag(tag)=1 then modbus_get_tag:=iGetTag(tag) else
  if TypeTag(tag)=2 then modbus_get_tag:=rGetTag(tag) else
  modbus_get_tag:=def;
 end;
 //
 // Set integer or real tag value.
 //
 function modbus_set_tag(tag:Integer; val:Real):Boolean;
 begin
  if TypeTag(tag)=1 then modbus_set_tag:=iSetTag(tag,Round(val)) else
  if TypeTag(tag)=2 then modbus_set_tag:=rSetTag(tag,val) else
  modbus_set_tag:=False;
 end;
 //
 // Increment integer or real tag value. Uses for error counting.
 //
 function modbus_inc_tag(tag:Integer; inc:Real):Boolean;
 begin
  if TypeTag(tag)=1 then modbus_inc_tag:=iSetTag(tag,iGetTag(tag)+Round(inc)) else
  if TypeTag(tag)=2 then modbus_inc_tag:=rSetTag(tag,rGetTag(tag)+inc) else
  modbus_inc_tag:=False;
 end;
 //
 // Fix modbus error: increment tag & code & print message.
 //
 procedure modbus_fixerror(tag,code:Integer; msg:String);
 begin
  if not modbus_inc_tag(tag,1) then
  if code<>0 then bNul(FixError(code));
  if Length(msg)>0 then Problem(msg);
 end;
 //
 // Swap two bytes in 16-bit word.
 //
 function modbus_swap_int16(w:Integer):Integer;
 begin
  modbus_swap_int16:=Ord(Chr(w))*256+Ord(Chr(w div 256));
 end;
 //
 // Swap bytes, words, dwords (correspond to SwapMode bit 0,1,2).
 //
 function modbus_swap2(dat:String; SwapMode:Integer):String;
 var s,t : array[1..2] of Char;
 begin
  s:=dat;
  case iAnd(SwapMode,1) of
   0: begin t[1]:=s[1]; t[2]:=s[2]; end;
   1: begin t[1]:=s[2]; t[2]:=s[1]; end;
  end;
  modbus_swap2:=t;
 end;
 function modbus_swap4(dat:String; SwapMode:Integer):String;
 var s,t : array[1..4] of Char;
 begin
  s:=dat;
  case iAnd(SwapMode,3) of
   0: begin t[1]:=s[1]; t[2]:=s[2]; t[3]:=s[3]; t[4]:=s[4]; end;
   1: begin t[1]:=s[2]; t[2]:=s[1]; t[3]:=s[4]; t[4]:=s[3]; end;
   2: begin t[1]:=s[3]; t[2]:=s[4]; t[3]:=s[1]; t[4]:=s[2]; end;
   3: begin t[1]:=s[4]; t[2]:=s[3]; t[3]:=s[2]; t[4]:=s[1]; end;
  end;
  modbus_swap4:=t;
 end;
 function modbus_swap8(dat:String; SwapMode:Integer):String;
 var s,t : array[1..8] of Char;
 begin
  s:=dat;
  case iAnd(SwapMode,7) of
   0: begin t[1]:=s[1]; t[2]:=s[2]; t[3]:=s[3]; t[4]:=s[4]; t[5]:=s[5]; t[6]:=s[6]; t[7]:=s[7]; t[8]:=s[8]; end;
   1: begin t[1]:=s[2]; t[2]:=s[1]; t[3]:=s[4]; t[4]:=s[3]; t[5]:=s[6]; t[6]:=s[5]; t[7]:=s[8]; t[8]:=s[7]; end;
   2: begin t[1]:=s[3]; t[2]:=s[4]; t[3]:=s[1]; t[4]:=s[2]; t[5]:=s[7]; t[6]:=s[8]; t[7]:=s[5]; t[8]:=s[6]; end;
   3: begin t[1]:=s[4]; t[2]:=s[3]; t[3]:=s[2]; t[4]:=s[1]; t[5]:=s[8]; t[6]:=s[7]; t[7]:=s[6]; t[8]:=s[5]; end;
   4: begin t[1]:=s[5]; t[2]:=s[6]; t[3]:=s[7]; t[4]:=s[8]; t[5]:=s[1]; t[6]:=s[2]; t[7]:=s[3]; t[8]:=s[4]; end;
   5: begin t[1]:=s[6]; t[2]:=s[5]; t[3]:=s[8]; t[4]:=s[7]; t[5]:=s[2]; t[6]:=s[1]; t[7]:=s[4]; t[8]:=s[3]; end;
   6: begin t[1]:=s[7]; t[2]:=s[8]; t[3]:=s[5]; t[4]:=s[6]; t[5]:=s[3]; t[6]:=s[4]; t[7]:=s[1]; t[8]:=s[2]; end;
   7: begin t[1]:=s[8]; t[2]:=s[7]; t[3]:=s[6]; t[4]:=s[5]; t[5]:=s[4]; t[6]:=s[3]; t[7]:=s[2]; t[8]:=s[1]; end;
  end;
  modbus_swap8:=t;
 end;
 //
 // Get data dump as int16, with byte reordering.
 //
 function modbus_dump_int16(dat:Integer; SwapMode:Integer):String;
 begin
  modbus_dump_int16:=modbus_swap2(Dump(dat),SwapMode);
 end;
 //
 // Get data dump as int32, with byte reordering.
 //
 function modbus_dump_int32(dat:Integer; SwapMode:Integer):String;
 begin
  modbus_dump_int32:=modbus_swap4(Dump(dat),SwapMode); 
 end;
 //
 // Get data dump as float, with byte reordering.
 //
 function modbus_dump_float(dat:Real; SwapMode:Integer):String;
 begin
  modbus_dump_float:=modbus_swap4(DumpF(dat),SwapMode); 
 end;
 //
 // Get data dump as double, with byte reordering.
 //
 function modbus_dump_double(dat:Real; SwapMode:Integer):String;
 begin
  modbus_dump_double:=modbus_swap8(Dump(dat),SwapMode);
 end;
 //
 // Take data from dump as int16, with byte reordering.
 //
 function modbus_take_int16(dat:String; SwapMode:Integer):Integer;
 begin
  modbus_take_int16:=Dump2i(modbus_swap2(dat,SwapMode));
 end;
 //
 // Take data from dump as int32, with byte reordering.
 //
 function modbus_take_int32(dat:String; SwapMode:Integer):Integer;
 begin
  modbus_take_int32:=Dump2i(modbus_swap4(dat,SwapMode));
 end;
 //
 // Take data from dump as float, with byte reordering.
 //
 function modbus_take_float(dat:String; SwapMode:Integer):Real;
 begin
  modbus_take_float:=Dump2f(modbus_swap4(dat,SwapMode)); 
 end;
 //
 // Take data dump as double, with byte reordering.
 //
 function modbus_take_double(dat:String; SwapMode:Integer):Real;
 begin
  modbus_take_double:=Dump2r(modbus_swap8(dat,SwapMode)); 
 end;
 //
 // Return true only if modbus function id (fid) contains exception bit.
 //
 function modbus_is_except(fid:Integer):Boolean;
 begin
  modbus_is_except:=HasFlags(fid,modbus_er_Except);
 end;
 //
 // Unset exception bit, i.e. return modbus function id (fid) without exception bit.
 //
 function modbus_un_except(fid:Integer):Integer;
 begin
  modbus_un_except:=iAnd(fid,iNot(modbus_er_Except));
 end;
 //
 // Assign exception bit, i.e. return modbus function id (fid) with exception bit set ON.
 //
 function modbus_as_except(fid:Integer; isOn:Boolean):Integer;
 begin
  if isOn
  then modbus_as_except:=iOr(fid,modbus_er_Except)
  else modbus_as_except:=iAnd(fid,iNot(modbus_er_Except));
 end;
 //
 // Check if device address is OK.
 //
 function modbus_addr_ok(addr:Integer):Boolean;
 begin
  modbus_addr_ok:=((addr>=modbus_MinUnitId) and (addr<=modbus_MaxUnitId));
 end;
 //
 // Check func is good to use: 1,2,3,4,5,6,15,16.
 //
 function modbus_func_ok(func:Integer):Boolean;
 begin
  func:=iAnd(func,iNot(modbus_er_Except));
  if (func=modbus_fn_ReadCS) then modbus_func_ok:=true else
  if (func=modbus_fn_ReadIS) then modbus_func_ok:=true else
  if (func=modbus_fn_ReadHR) then modbus_func_ok:=true else
  if (func=modbus_fn_ReadIR) then modbus_func_ok:=true else
  if (func=modbus_fn_WritSC) then modbus_func_ok:=true else
  if (func=modbus_fn_WritSR) then modbus_func_ok:=true else
  if (func=modbus_fn_WritMC) then modbus_func_ok:=true else
  if (func=modbus_fn_WritMR) then modbus_func_ok:=true else
  modbus_func_ok:=false;
 end;
 //
 // Check if MODBUS protocol is OK.
 //
 function modbus_prot_ok(prot:Integer):Boolean;
 begin
  modbus_prot_ok:=((prot>=modbus_pr_IP) and (prot<=modbus_pr_ASCII));
 end;
 //
 // Get MODBUS protocol name.
 //
 function modbus_prot_name(prot:Integer):String;
 begin
  if modbus_prot_ok(prot)
  then modbus_prot_name:=ExtractWord(prot,modbus_pr_NAMES)
  else modbus_prot_name:='';
 end;
 //
 // Get MODBUS protocol alias.
 //
 function modbus_prot_alias(prot:Integer):String;
 begin
  if modbus_prot_ok(prot)
  then modbus_prot_alias:=ExtractWord(prot,modbus_pr_ALIAS)
  else modbus_prot_alias:='';
 end;
 //
 // Get MODBUS protocol by name or alias.
 //
 function modbus_prot_byname(name:String):Integer;
 var prot:Integer;
 begin
  prot:=WordIndex(Trim(name),modbus_pr_NAMES);
  if not modbus_prot_ok(prot) then prot:=WordIndex(Trim(name),modbus_pr_ALIAS);
  if modbus_prot_ok(prot) then modbus_prot_byname:=prot else modbus_prot_byname:=0;  
 end;
 //
 // Calculate MODBUS/ASCII LRC checksum.
 //
 function modbus_calc_lrc(buf:String):Integer;
 var sum,i:Integer;
 begin
  if hid_NetModbusLrc<0 then begin
   sum:=0;
   for i:=1 to Length(buf) do sum:=sum+Ord(buf[i]);
   sum:=Ord(Chr(256-Ord(Chr(sum))));
  end else sum:=HashIndexOf(buf,0,hid_NetModbusLrc);
  modbus_calc_lrc:=sum;
 end;
 //
 // Calculate MODBUS/RTU CRC checksum.
 //
 function modbus_calc_crc(buf:String):Integer;
 const hFFFF=65535; hA001=40961;
 var sum,i,j:Integer;
 begin
  if hid_NetModbusCrc<0 then begin
   sum:=hFFFF;
   for i:=1 to Length(buf) do begin
    sum:=iXor(sum,Ord(buf[i]));
    for j:=1 to 8 do if Odd(sum)
    then sum:=iXor(sum div 2,hA001)
    else sum:=sum div 2;
   end;
  end else sum:=HashIndexOf(buf,0,hid_NetModbusCrc);
  modbus_calc_crc:=sum;
 end;
 //
 // Join LRC to Data tail, return dat+LRC(dat).
 //
 function modbus_join_lrc(dat:String):String;
 begin
  modbus_join_lrc:=dat+Chr(modbus_calc_lrc(dat));
 end;
 //
 // Join CRC16 to Data tail, return dat+CRC16(dat).
 //
 function modbus_join_crc(dat:String):String;
 begin
  modbus_join_crc:=dat+modbus_dump_int16(modbus_calc_crc(dat),0);
 end;
 //
 // Encode MODBUS IP message.
 //
 function modbus_encode_ip(tid,pid,uid,fid:Integer; dat:String):String;
 var head:array[1..8] of Char; DatLen:Integer; headStr:String;
 begin
  modbus_errno:=modbus_er_OK;
  if Length(dat)=0 then modbus_errno:=modbus_er_BADLEN else
  if not modbus_addr_ok(uid) then modbus_errno:=modbus_er_BADUID else
  if not modbus_func_ok(fid) then modbus_errno:=modbus_er_ILLFUN;
  DatLen:=2+Length(dat);          // Length of uid+fid+dat
  head[1]:=Chr(tid div 256);      // MBAP Header Transaction Id (Hi)
  head[2]:=Chr(tid);              // MBAP Header Transaction Id (Lo)
  head[3]:=Chr(pid div 256);      // MBAP Header Protocol Id (Hi)
  head[4]:=Chr(pid);              // MBAP Header Protocol Id (Lo)
  head[5]:=Chr(DatLen div 256);   // MBAP Header Data Length (Hi)
  head[6]:=Chr(DatLen);           // MBAP Header Data Length (Lo)
  head[7]:=Chr(uid);              // MBAP Header Unit Id
  head[8]:=Chr(fid);              // PDU  Function Id
  headStr:=head;                  // Convert to string
  modbus_encode_ip:=headStr+dat;  // ADU = MBAP Header + fid + dat
  headStr:='';                    //       <- 7 byte-> + <- PDU ->
 end;
 //
 // Decode MODBUS IP message. Return Length(PDU=fid+dat) or 0.
 //
 function modbus_decode_ip(adu:String; var tid,pid,uid,fid:Integer; var dat:String):Integer;
 var head:array[1..8] of Char; leng,DatLen,PduLen:Integer;
 begin
  modbus_errno:=modbus_er_OK;                      // Clear error code
  PduLen:=0; tid:=0;pid:=0;uid:=0;fid:=0;dat:='';  // Zero all output variables
  leng:=Length(adu);                               // Expected ADU length >= 8.
  if leng>=8 then begin                            // MBAP[7]+PDU[1+dat] at least
   head:=adu;                                      // Decode MBAP & PDU header
   tid:=Ord(head[2])+256*Ord(head[1]);             // MBAP Header - Transaction identifier
   pid:=Ord(head[4])+256*Ord(head[3]);             // MBAP Header - Protocol identifier
   DatLen:=Ord(head[6])+256*Ord(head[5]);          // MBAP Header - Length of uid+PDU
   uid:=Ord(head[7]);                              // MBAP Header - Unit identifier
   fid:=Ord(head[8]);                              // PDU  Function code
   PduLen:=DatLen-1;                               // PDU  Length
   if PduLen<=0               then begin PduLen:=0; modbus_errno:=modbus_er_ILLVAL; end else
   if leng<PduLen+7           then begin PduLen:=0; modbus_errno:=modbus_er_ILLVAL; end else
   if not modbus_func_ok(fid) then begin PduLen:=0; modbus_errno:=modbus_er_ILLFUN; end else
   if not modbus_addr_ok(uid) then begin PduLen:=0; modbus_errno:=modbus_er_BADUID; end;
  end else modbus_errno:=modbus_er_BADLEN;         // Error if length too small
  if PduLen>1 then dat:=Copy(adu,9,PduLen-1);      // Extract raw data block, skip MBAP[7]+fid
  modbus_decode_ip:=PduLen;                        // Length of PDU=fid+dat or 0
 end;
 //
 // Encode MODBUS RTU message: uid+fid+dat+CRC16.
 //
 function modbus_encode_rtu(uid,fid:Integer; dat:String):String;
 begin
  modbus_errno:=modbus_er_OK;
  if Length(dat)=0 then modbus_errno:=modbus_er_BADLEN else
  if not modbus_addr_ok(uid) then modbus_errno:=modbus_er_BADUID else
  if not modbus_func_ok(fid) then modbus_errno:=modbus_er_ILLFUN;
  modbus_encode_rtu:=modbus_join_crc(Chr(uid)+(Chr(fid)+dat));
 end;
 //
 // Decode MODBUS RTU message. Return Length(PDU=fid+dat) or 0.
 //
 function modbus_decode_rtu(adu:String; var uid,fid:Integer; var dat:String):Integer;
 var head:array[1..2] of Char; PduLen,leng,crc1,crc2:Integer;
 begin
  modbus_errno:=modbus_er_OK;                         // Clear error code
  PduLen:=0; dat:=''; uid:=0; fid:=0;                 // Zero all output variables
  leng:=Length(adu);                                  // Expected ADU length >=4
  if leng>=4 then begin                               // uid+fid+CRC16 at least
   head:=adu;                                         // Decode ADU header:
   uid:=Ord(head[1]);                                 // Expect Unit identifier
   fid:=Ord(head[2]);                                 // Expect Function code
   crc1:=modbus_calc_crc(Copy(adu,1,leng-2));         // Actual CRC
   crc2:=modbus_take_int16(Copy(adu,leng-1,2),0);     // Stored CRC
   if (crc1=crc2) then PduLen:=leng-3;                // Length of PDU=fid+dat, skip uid     and CRC16
   if PduLen>1 then dat:=Copy(adu,3,PduLen-1);        // Extract data block,    skip uid+fid and CRC16
   if crc1<>crc2              then begin PduLen:=0; modbus_errno:=modbus_er_BADCRC; end else
   if not modbus_addr_ok(uid) then begin PduLen:=0; modbus_errno:=modbus_er_BADUID; end else
   if not modbus_func_ok(fid) then begin PduLen:=0; modbus_errno:=modbus_er_ILLFUN; end;
  end else modbus_errno:=modbus_er_BADLEN;            // Error if length too small
  modbus_decode_rtu:=PduLen;                          // Length of PDU=fid+dat or 0 
 end;
 //
 // Encode MODBUS ASCII message.
 //
 function modbus_encode_ascii(uid,fid:Integer; dat:String):String;
 begin
  modbus_errno:=modbus_er_OK;
  if Length(dat)=0 then modbus_errno:=modbus_er_BADLEN else
  if not modbus_addr_ok(uid) then modbus_errno:=modbus_er_BADUID else
  if not modbus_func_ok(fid) then modbus_errno:=modbus_er_ILLFUN;
  modbus_encode_ascii:=':'+Hex_Encode(modbus_join_lrc(Chr(uid)+(Chr(fid)+dat)))+CRLF;
 end;
 //
 // Decode MODBUS RTU message. Return Length(PDU=fid+dat) or 0.
 //
 function modbus_decode_ascii(adu:String; var uid,fid:Integer; var dat:String):Integer;
 var head:array[1..2] of Char; PduLen,leng,lrc1,lrc2,pb,pe:Integer;
 begin
  modbus_errno:=modbus_er_OK;                          // Clear error code
  PduLen:=0; uid:=0; fid:=0; dat:='';                  // Zero all output variables
  pb:=Pos(':',adu); pe:=Pos(CRLF,adu);                 // Find ADU begin/end marker
  if (pb>0) and (pe>pb) then begin                     // If begin/end marker found
   adu:=Hex_Decode(Copy(adu,pb+1,pe-pb-1));            // Decode message HEX -> BIN
   leng:=Length(adu);                                  // Expected ADU length >= 3.
   if leng>=3 then begin                               // uid+fid+LRC at least
    head:=adu;                                         // Decode ADU header:
    uid:=Ord(head[1]);                                 // Expect Unit identifier
    fid:=Ord(head[2]);                                 // Expect Function code
    lrc1:=modbus_calc_lrc(Copy(adu,1,leng-1));         // Actual LRC
    lrc2:=Dump2i(Copy(adu,leng,1));                    // Stored LRC
    if (lrc1=lrc2) then PduLen:=leng-2;                // Length of PDU=fid+dat, skip uid     and LRC
    if PduLen>1 then dat:=Copy(adu,3,PduLen-1);        // Extract data block,    skip uid+fid and LRC
    if lrc1<>lrc2              then begin PduLen:=0; modbus_errno:=modbus_er_BADCRC; end else
    if not modbus_addr_ok(uid) then begin PduLen:=0; modbus_errno:=modbus_er_BADUID; end else
    if not modbus_func_ok(fid) then begin PduLen:=0; modbus_errno:=modbus_er_ILLFUN; end;
   end else modbus_errno:=modbus_er_BADLEN;            // Error if length too small
  end else modbus_errno:=modbus_er_BADLEN;             // Error no start/stop marker
  modbus_decode_ascii:=PduLen;                         // Length of PDU=fid+dat or 0 
 end;
 //
 // Encode modbus ADU message for given transaction (tid), protocol (pid), unit uid, function fid and data (dat).
 // Arguments:
 //  tid  - (in) transaction id
 //  pid  - (in) protocol id, should be 1
 //  uid  - (in) unit id, i.e. modbus module address, 1..247
 //  fid  - (in) modbus function id, 1,2,3,4,5,6,15,16
 //  dat  - (in) PDU data; PDU=(fid+dat)
 // Return ADU (application data unit) message ready to send.
 // Also sets modbus_errno value on errors.
 //
 function modbus_encode_adu(tid,pid,uid,fid:Integer; dat:String):String;
 begin
  if pid=modbus_pr_IP    then modbus_encode_adu:=modbus_encode_ip(tid,0,uid,fid,dat) else
  if pid=modbus_pr_RTU   then modbus_encode_adu:=modbus_encode_rtu(uid,fid,dat) else
  if pid=modbus_pr_ASCII then modbus_encode_adu:=modbus_encode_ascii(uid,fid,dat) else
  modbus_encode_adu:='';
 end;
 //
 // Decode Application Data Unit (adu) with given protocol (prot).
 // Arguments:
 //  adu  - (in)  message to decode
 //  prot - (in)  protocol 1/2/3=IP/RTU/ASCII
 //  tid  - (out) transaction id
 //  pid  - (out) protocol id, expected to be 1
 //  uid  - (out) unit id, i.e. modbus module address, 1..247
 //  fid  - (out) modbus function id, 1,2,3,4,5,6,15,16
 //  dat  - (out) PDU data; PDU=(fid+dat)
 // Return positive PDU (protocol data unit) length, or zero/negative error code on errors.
 // If positive value returned, assign tid,pid,uid,fid,dat to transaction,protocol,unit,function,data.
 // Also sets modbus_errno value on errors.
 //
 function modbus_decode_adu(adu:String; prot:Integer; var tid,pid,uid,fid:Integer; var dat:String):Integer;
 begin
  tid:=0; pid:=0; uid:=0; fid:=0; dat:='';
  if (prot=modbus_pr_IP)    then modbus_decode_adu:=modbus_decode_ip(adu,tid,pid,uid,fid,dat) else
  if (prot=modbus_pr_RTU)   then modbus_decode_adu:=modbus_decode_rtu(adu,uid,fid,dat) else
  if (prot=modbus_pr_ASCII) then modbus_decode_adu:=modbus_decode_ascii(adu,uid,fid,dat) else
  modbus_decode_adu:=-modbus_er_ILLVAL;
 end;
 //
 // Encode PDU, i.e. calculate dat string; PDU=(fid+dat).
 // Arguments:
 //  dir   = (in)     I/O  direction: 'R' = Request, 'A' = Answer (response).
 //  fid   = (in)     modbus function identifier
 //  saddr = (in)     start address                      if required or zero
 //  quant = (in)     quantity of coils/inputs/registers if required or zero
 //  raw   = (in)     raw data bytes (coils/registers)   if required or empty
 // Return : PDU data block (dat); PDU=(fid+dat) or empty string on any errors.
 // Also sets modbus_errno value on errors.
 //
 function modbus_encode_pdu(dir:Char; fid:Integer; saddr,quant:Integer; raw:String):String;
 var dat:String; count:Integer;
 begin
  modbus_errno:=modbus_er_OK;
  dat:=''; count:=Length(raw);
  if (dir='R') or (dir='r') then begin // Request
   if (fid=modbus_fn_ReadCS) then begin
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxCSAddr)) and ((1<=quant) and (quant<=modbus_MaxCSGet))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadIS) then begin
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxISAddr)) and ((1<=quant) and (quant<=modbus_MaxISGet))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadHR) then begin
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxHRAddr)) and ((1<=quant) and (quant<=modbus_MaxHRGet))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadIR) then begin
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxIRAddr)) and ((1<=quant) and (quant<=modbus_MaxIRGet))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSC) then begin
    if ((0<=saddr) and (saddr<=modbus_MaxCSAddr)) and ((quant=modbus_sc_off) or (quant=modbus_sc_on))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSR) then begin
    if ((0<=saddr) and (saddr<=modbus_MaxHRAddr)) and ((0<=quant) and (quant<=modbus_MaxWord))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMC) then begin
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxCSAddr)) and
       ((1<=quant) and (quant<=modbus_MaxCSPut)) and (count=((quant+7) div 8))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)+Chr(count)+raw
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMR) then begin
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxHRAddr)) and
       ((1<=quant) and (quant<=modbus_MaxHRPut)) and (count=quant*2)
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)+Chr(count)+raw
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   modbus_errno:=modbus_er_ILLFUN;
  end else
  if (dir='A') or (dir='a') then begin // Answer
   if modbus_is_except(fid) then begin
    fid:=modbus_un_except(fid);
    if (fid=modbus_fn_ReadCS) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_ReadIS) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_ReadHR) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_ReadIR) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_WritSC) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_WritSR) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_WritMC) then dat:=Copy(raw,1,1) else
    if (fid=modbus_fn_WritMR) then dat:=Copy(raw,1,1) else
    modbus_errno:=modbus_er_ILLFUN;
   end else
   if (fid=modbus_fn_ReadCS) then begin
    if (1<=count) and (count<=modbus_MaxCSGet div 8)
    then dat:=Chr(count)+raw
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadIS) then begin
    if (1<=count) and (count<=modbus_MaxISGet div 8)
    then dat:=Chr(count)+raw
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadHR) then begin
    if (1<=count) and (count<=modbus_MaxHRGet*2)
    then dat:=Chr(count)+raw
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadIR) then begin
    if (1<=count) and (count<=modbus_MaxIRGet*2)
    then dat:=Chr(count)+raw
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSC) then begin
    if (0<=saddr) and (saddr<=modbus_MaxWord) and ((quant=modbus_sc_off) or (quant=modbus_sc_on))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSR) then begin
    if ((0<=saddr) and (saddr<=modbus_MaxWord)) and ((0<=quant) and (quant<=modbus_MaxWord))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMC) then begin
    if ((0<=saddr) and (saddr<=modbus_MaxWord)) and ((1<=quant) and (quant<=modbus_MaxCSPut))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMR) then begin
    if ((0<=saddr) and (saddr<=modbus_MaxWord)) and ((1<=quant) and (quant<=modbus_MaxHRPut))
    then dat:=modbus_dump_int16(saddr,modbus_sw_normal)+modbus_dump_int16(quant,modbus_sw_normal)
    else modbus_errno:=modbus_er_ILLVAL;
   end else
   modbus_errno:=modbus_er_ILLFUN;
  end;
  modbus_encode_pdu:=dat;
 end;
 //
 // Decode PDU=(fid+dat) header.
 // Arguments:
 //  dir   = (in)     I/O  direction: 'R' = Request, 'A' = Answer (response).
 //  fid   = (in)     modbus function identifier
 //  dat   = (in)     PDU data; PDU=Chr(fid)+dat
 //  saddr = (in/out) start address                      if required or 0
 //  quant = (in/out) quantity of coils/inputs/registers if required or 0
 //  raw   = (out)    raw data bytes (coils/registers)   if required or empty
 // Return : Positive PDU length or Negative/Zero error code.
 //          Expected to be: count=length(raw) if returned positive result.
 //          Expected to be: PDU length = Length(Chr(fid)+dat) = Length(dat)+1.
 // Also sets modbus_errno value on errors.
 //
 function modbus_decode_pdu(dir:Char; fid:Integer; dat:String; var saddr,quant:Integer; var raw:String):Integer;
 var pdulen,datlen,count:Integer; header:array[1..5] of Char;
 begin
  modbus_errno:=modbus_er_OK;
  pdulen:=0; datlen:=Length(dat);
  header:=dat; count:=0; raw:='';
  if (dir='R') or (dir='r') then begin // PDU = Request
   if (fid=modbus_fn_ReadCS) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxCSAddr)) and ((1<=quant) and (quant<=modbus_MaxCSGet))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadIS) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxISAddr)) and ((1<=quant) and (quant<=modbus_MaxISGet))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadHR) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxHRAddr)) and ((1<=quant) and (quant<=modbus_MaxHRGet))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_ReadIR) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxIRAddr)) and ((1<=quant) and (quant<=modbus_MaxIRGet))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSC) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr<=modbus_MaxCSAddr)) and ((quant=modbus_sc_off) or (quant=modbus_sc_on))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSR) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr<=modbus_MaxHRAddr)) and ((0<=quant) and (quant<=modbus_MaxWord))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMC) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]); count:=Ord(header[5]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxCSAddr)) and
       ((1<=quant) and (quant<=modbus_MaxCSPut)) and (count=((quant+7) div 8))
    then pdulen:=6+count else pdulen:=-modbus_er_ILLVAL;
    if pdulen>0 then raw:=Copy(dat,6,count) else count:=0;
   end else
   if (fid=modbus_fn_WritMR) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]); count:=Ord(header[5]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxHRAddr)) and
       ((1<=quant) and (quant<=modbus_MaxHRPut)) and (count=quant*2)
    then pdulen:=6+count else pdulen:=-modbus_er_ILLVAL;
    if pdulen>0 then raw:=Copy(dat,6,count) else count:=0;
   end else
   pdulen:=-modbus_er_ILLFUN;
  end else
  if (dir='A') or (dir='a') then begin // PDU = Answer, i.e. Response
   if modbus_is_except(fid) then begin
    fid:=modbus_un_except(fid);
    if (fid=modbus_fn_ReadCS) then pdulen:=2 else
    if (fid=modbus_fn_ReadIS) then pdulen:=2 else
    if (fid=modbus_fn_ReadHR) then pdulen:=2 else
    if (fid=modbus_fn_ReadIR) then pdulen:=2 else
    if (fid=modbus_fn_WritSC) then pdulen:=2 else
    if (fid=modbus_fn_WritSR) then pdulen:=2 else
    if (fid=modbus_fn_WritMC) then pdulen:=2 else
    if (fid=modbus_fn_WritMR) then pdulen:=2 else
    pdulen:=-modbus_er_ILLFUN;
    if pdulen>0 then count:=pdulen-1;
    if count>0 then raw:=Copy(dat,1,count);
   end else
   if (fid=modbus_fn_ReadCS) then begin
    count:=Ord(header[1]);
    if (1<=count) and (count<=modbus_MaxCSGet div 8)
    then pdulen:=2+count else pdulen:=-modbus_er_ILLVAL;
    if pdulen>0 then raw:=Copy(dat,2,count) else count:=0;
   end else
   if (fid=modbus_fn_ReadIS) then begin
    count:=Ord(header[1]);
    if (1<=count) and (count<=modbus_MaxISGet div 8)
    then pdulen:=2+count else pdulen:=-modbus_er_ILLVAL;
    if pdulen>0 then raw:=Copy(dat,2,count) else count:=0;
   end else
   if (fid=modbus_fn_ReadHR) then begin
    count:=Ord(header[1]);
    if (1<=count) and (count<=modbus_MaxHRGet*2)
    then pdulen:=2+count else pdulen:=-modbus_er_ILLVAL;
    if pdulen>0 then raw:=Copy(dat,2,count) else count:=0;
   end else
   if (fid=modbus_fn_ReadIR) then begin
    count:=Ord(header[1]);
    if (1<=count) and (count<=modbus_MaxIRGet*2)
    then pdulen:=2+count else pdulen:=-modbus_er_ILLVAL;
    if pdulen>0 then raw:=Copy(dat,2,count) else count:=0;
   end else
   if (fid=modbus_fn_WritSC) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if (0<=saddr) and (saddr<=modbus_MaxCSAddr) and ((quant=modbus_sc_off) or (quant=modbus_sc_on))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritSR) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr<=modbus_MaxHRAddr)) and ((0<=quant) and (quant<=modbus_MaxWord))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMC) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxCSAddr)) and ((1<=quant) and (quant<=modbus_MaxCSPut))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   if (fid=modbus_fn_WritMR) then begin
    saddr:=Ord(header[1])*256+Ord(header[2]); quant:=Ord(header[3])*256+Ord(header[4]);
    if ((0<=saddr) and (saddr+quant-1<=modbus_MaxHRAddr)) and ((1<=quant) and (quant<=modbus_MaxHRPut))
    then pdulen:=5 else pdulen:=-modbus_er_ILLVAL;
   end else
   pdulen:=-modbus_er_ILLFUN;
  end else
  pdulen:=-modbus_er_ILLVAL;
  if pdulen>datlen+1 then pdulen:=-modbus_er_ILLVAL;
  if count<>length(raw) then pdulen:=-modbus_er_ILLVAL;
  if pdulen<0 then modbus_errno:=Abs(pdulen) else
  if pdulen=0 then modbus_errno:=modbus_er_ILLVAL;
  modbus_decode_pdu:=pdulen;
 end;
 //
 // Extract data from raw buffer which decoded by modbus_decode_pdu.
 // raw      - string contains raw data from modbys_decode_pdu
 // regoffs  - zero based register offset of data register/coil to extract
 // byteoffs - zero based byte offset of data to extract
 // SwapMode - data byte/word/dword swap mode
 //
 function modbus_ext_byte(raw:String; byteoffs:Integer):Integer;
 var index:Integer;
 begin
  index:=1+byteoffs;
  if (1<=index) and (index<=Length(raw))
  then modbus_ext_byte:=Ord(raw[index]) // dump2i(Copy(raw,index,1))
  else modbus_ext_byte:=0;
 end;
 function modbus_ext_int16(raw:String; regoffs,SwapMode:Integer):Integer;
 var index:Integer;
 begin
  index:=1+regoffs*2;
  if (1<=index) and (index<=Length(raw)-1)
  then modbus_ext_int16:=modbus_take_int16(Copy(raw,index,2),SwapMode)
  else modbus_ext_int16:=0;
 end;
 function modbus_ext_int32(raw:String; regoffs,SwapMode:Integer):Integer;
 var index:Integer;
 begin
  index:=1+regoffs*2;
  if (1<=index) and (index<=Length(raw)-3)
  then modbus_ext_int32:=modbus_take_int32(Copy(raw,index,4),SwapMode)
  else modbus_ext_int32:=0;
 end;
 function modbus_ext_uint32(raw:String; regoffs,SwapMode:Integer):Real;
 var v:Real;
 begin
  v:=modbus_ext_int32(raw,regoffs,SwapMode);
  if v<0 then v:=v+65536.0;
  modbus_ext_uint32:=v;
 end;
 function modbus_ext_float(raw:String; regoffs,SwapMode:Integer):Real;
 var index:Integer;
 begin
  index:=1+regoffs*2;
  if (1<=index) and (index<=Length(raw)-3)
  then modbus_ext_float:=modbus_take_float(Copy(raw,index,4),SwapMode)
  else modbus_ext_float:=0;
 end;
 function modbus_ext_double(raw:String; regoffs,SwapMode:Integer):Real;
 var index:Integer;
 begin
  index:=1+regoffs*2;
  if (1<=index) and (index<=Length(raw)-7)
  then modbus_ext_double:=modbus_take_double(Copy(raw,index,8),SwapMode)
  else modbus_ext_double:=0;
 end;
 function modbus_ext_uint64(raw:String; regoffs,SwapMode:Integer):Real;
 var index:Integer; v:Real;
 begin
  index:=1+regoffs*2;
  if (1<=index) and (index<=Length(raw)-7) then begin
   raw:=modbus_swap8(Copy(raw,index,8),SwapMode);
   v:=modbus_ext_uint32(raw,0,modbus_sw_native)+
      modbus_ext_uint32(raw,2,modbus_sw_native)*4294967296.0;
  end else v:=0;
  modbus_ext_uint64:=v;
 end;
 function modbus_ext_coil(raw:String; regoffs:Integer):Integer;
 var index,bitnum,coil:Integer;
 begin
  coil:=0;
  if regoffs>=0 then begin
   index:=1+(regoffs div 8); bitnum:=(regoffs mod 8);
   if length(raw)>=index then coil:=Ord(IsBit(Ord(raw[index]),bitnum));
  end;
  modbus_ext_coil:=coil;
 end;
 //
 // Encode command to send to &ModbusProxy or reply to driver.
 // cmd  - (in)  command name, expected @Modbus.Poll, @Modbus.Reply, @Modbus.Refuse, @Modbus.Timeout or empty string.
 // ref  - (in)  sender device reference, usually driver's devMySelf
 // cid  - (in)  command id, any user defined number
 // tot  - (in)  timeout, ms
 // port - (in)  logical port
 // uid  - (in)  unit id
 // fid  - (in)  function id
 // dat  - (in)  PDU data; PDU=(fid+dat)
 // Return formatted data ready to send to &ModbusProxy.
 //
 function modbus_proxy_poll(cmd:String; ref,cid,tot,port,uid,fid:Integer; dat:String):String;
 begin
  if Length(cmd)=0 then cmd:='@Modbus.Poll';
  modbus_proxy_poll:=cmd+' '+Str(ref)+' '+Str(cid)+' '+Str(tot)+' '
                           +Str(port)+' '+Str(uid)+' '+Str(fid)+' $$'+Hex_Encode(dat);
 end;
 //
 // Nice (human-readable) format to print &ModbusProxy messages.
 // If wid>0, use $HEX_ENCODE(dat)...; if wid<0 use Trim(dat)...
 //
 function modbus_proxy_nice(cmd:String; ref,cid,tim,port,uid,fid:Integer; dat:String; wid:Integer):String;
 begin
  if wid<0 then if length(dat)>abs(wid)
  then dat:=Copy(dat,1,abs(wid))+'...';
  if wid>0 then if length(dat)<=wid
  then dat:='$$'+Hex_Encode(dat)
  else dat:='$$'+Hex_Encode(Copy(dat,1,wid div 2))+'...';
  modbus_proxy_nice:=cmd+' '+RefInfo(ref,'Name')+' '+Str(cid)+' '+Str(tim)
                        +' '+Str(port)+' '+Str(uid)+' '+Str(fid)+' '+dat;
 end;
 //
 // Decode Reply/Timeout/Refuse message received from &ModbusProxy.
 // cmd  - (in)  Should be @Modbus.Poll or @Modbus.Timeout or @Modbus.Refuse
 // arg  - (in)  command argument list[7]: ref cid tim port uid fid $$dat
 // ref  - (out) sender device reference, expected &ModbusProxy
 // cid  - (out) command id, any user defined number
 // tim  - (out) response time, ms
 // port - (out) logical port
 // uid  - (out) unit id
 // fid  - (out) function id
 // dat  - (out) PDU data; PDU=(fid+dat)
 // Return true if message looks acceptable to use.
 // Also sets modbus_errno value on errors.
 //
 function modbus_proxy_reply(cmd,arg:String; var ref,cid,tim,port,uid,fid:Integer; var dat:String):Boolean;
 var p,cmdid:Integer;
 begin
  ref:=0;
  cmdid:=HashList_GetLink(StdIn_CmdHashTab,cmd);
  if (cmdid=cmd_NetModbusPoll) then begin
   p:=pos(' $$',arg);
   if p>0 then begin
    dat:=Copy(arg,p+1);
    arg:=Copy(arg,1,p-1);
    ref:=Val(ExtractWord(1,arg));  if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
    cid:=Val(ExtractWord(2,arg));  if ref=0 then modbus_errno:=modbus_er_BADREF;
    tim:=Val(ExtractWord(3,arg));  if tim<=0 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
    port:=Val(ExtractWord(4,arg)); if port<1 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
    uid:=Val(ExtractWord(5,arg));  if not modbus_addr_ok(uid) then begin ref:=0; modbus_errno:=modbus_er_BADUID; end;
    fid:=Val(ExtractWord(6,arg));  if not modbus_func_ok(fid) then begin ref:=0; modbus_errno:=modbus_er_ILLFUN; end;
    dat:=Hex_Decode(dat);          if Length(dat)=0 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
   end;
  end else
  if (cmdid=cmd_NetModbusReply) then begin
   p:=pos(' $$',arg);
   if p>0 then begin
    dat:=Copy(arg,p+1);
    arg:=Copy(arg,1,p-1);
    ref:=Val(ExtractWord(1,arg));  if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
    cid:=Val(ExtractWord(2,arg));  if ref=0 then modbus_errno:=modbus_er_BADREF;
    tim:=Val(ExtractWord(3,arg));  if tim<0 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
    port:=Val(ExtractWord(4,arg)); if port<1 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
    uid:=Val(ExtractWord(5,arg));  if not modbus_addr_ok(uid) then begin ref:=0; modbus_errno:=modbus_er_BADUID; end;
    fid:=Val(ExtractWord(6,arg));  if not modbus_func_ok(fid) then begin ref:=0; modbus_errno:=modbus_er_ILLFUN; end;
    dat:=Hex_Decode(dat);          if Length(dat)=0 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
   end;
  end else
  if (cmdid=cmd_NetModbusTimeout) then begin
   p:=pos(' $$',arg);
   if p>0 then begin
    dat:=Copy(arg,p+1);
    arg:=Copy(arg,1,p-1);
    ref:=Val(ExtractWord(1,arg));  if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
    cid:=Val(ExtractWord(2,arg));  if ref=0 then modbus_errno:=modbus_er_BADREF;
    tim:=Val(ExtractWord(3,arg));  if tim<0 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
    port:=Val(ExtractWord(4,arg)); if port<1 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
    uid:=Val(ExtractWord(5,arg));  if not modbus_addr_ok(uid) then begin ref:=0; modbus_errno:=modbus_er_BADUID; end;
    fid:=Val(ExtractWord(6,arg));  if not modbus_func_ok(fid) then begin ref:=0; modbus_errno:=modbus_er_ILLFUN; end;
    dat:=Hex_Decode(dat);          if Length(dat)=0 then begin ref:=0; modbus_errno:=modbus_er_ILLVAL; end;
   end;
  end else
  if (cmdid=cmd_NetModbusRefuse) then begin
   ref:=Val(ExtractWord(1,arg));   if ref=0 then ref:=RefFind('Device '+ExtractWord(1,arg));
   cid:=Val(ExtractWord(2,arg));   if ref=0 then modbus_errno:=modbus_er_BADREF;
   tim:=Val(ExtractWord(3,arg));
   port:=Val(ExtractWord(4,arg));
   uid:=Val(ExtractWord(5,arg));
   fid:=Val(ExtractWord(6,arg));
   dat:=Trim(SkipWords(6,arg));
  end else
  modbus_errno:=modbus_er_BADCMD;
  if not IsRefDevice(ref) then begin ref:=0; modbus_errno:=modbus_er_BADREF; end;
  if ref=0  then begin cid:=0; tim:=0; port:=0; uid:=0; fid:=0; dat:=''; end;
  modbus_proxy_reply:=(ref<>0);
 end;
 //
 // Get Modbus error message by error code (errno).
 //
 function modbus_errmsg(errno:Integer):String;
 begin
  if errno=modbus_er_OK     then modbus_errmsg:='SUCCESS('+Str(errno)+')' else
  if errno=modbus_er_ILLFUN then modbus_errmsg:='ILLEGAL_FUNCTION('+Str(errno)+')' else
  if errno=modbus_er_ILLADR then modbus_errmsg:='ILLEGAL_ADDRESS('+Str(errno)+')' else
  if errno=modbus_er_ILLVAL then modbus_errmsg:='ILLEGAL_VALUE('+Str(errno)+')' else
  if errno=modbus_er_BADLEN then modbus_errmsg:='BAD_LENGTH('+Str(errno)+')' else
  if errno=modbus_er_BADCRC then modbus_errmsg:='BAD_CRC('+Str(errno)+')' else
  if errno=modbus_er_BADUID then modbus_errmsg:='BAD_UNIT('+Str(errno)+')' else
  if errno=modbus_er_BADCMD then modbus_errmsg:='BAD_COMMAND('+Str(errno)+')' else
  if errno=modbus_er_BADREF then modbus_errmsg:='BAD_REFERENCE('+Str(errno)+')' else
  modbus_errmsg:='FAILURE('+Str(errno)+')'; 
 end;
 //
 // Clear NetModbus.
 //
 procedure ClearNetModbus;
 begin
 end;
 //
 // Initialize NetModbus.
 //
 procedure InitNetModbus;
  procedure InitModbusLrcCrc;
  const testData='0123456789';
  var checkLrc,checkCrc:Integer;
  begin
   hid_NetModbusLrc:=-1; checkLrc:=modbus_calc_lrc(testData);
   hid_NetModbusCrc:=-1; checkCrc:=modbus_calc_crc(testData);
   hid_NetModbusLrc:=iValDef(ParamStr('HasherCode ModbusLrc'),-1);
   hid_NetModbusCrc:=iValDef(ParamStr('HasherCode Crc16Modbus'),-1);
   if hid_NetModbusLrc>=0 then if HashIndexOf(testData,0,hid_NetModbusLrc)<>checkLrc then hid_NetModbusLrc:=-1;
   if hid_NetModbusCrc>=0 then if HashIndexOf(testData,0,hid_NetModbusCrc)<>checkCrc then hid_NetModbusCrc:=-1;
   if hid_NetModbusLrc>=0 then Success('HasherCode ModbusLrc   '+Str(hid_NetModbusLrc));
   if hid_NetModbusCrc>=0 then Success('HasherCode Crc16Modbus '+Str(hid_NetModbusCrc));
  end;
 begin
  modbus_errno:=modbus_er_OK;
  modbus_MaxCSAddr:=modbus_MaxWord;
  modbus_MaxISAddr:=modbus_MaxWord;
  modbus_MaxHRAddr:=modbus_MaxWord;
  modbus_MaxIRAddr:=modbus_MaxWord;
  InitDevice(devModbusProxy, ModbusProxy, 1);
  cmd_NetModbusPoll    := RegisterStdInCmd('@Modbus.Poll',    '');
  cmd_NetModbusReply   := RegisterStdInCmd('@Modbus.Reply',   '');
  cmd_NetModbusTimeout := RegisterStdInCmd('@Modbus.Timeout', '');
  cmd_NetModbusRefuse  := RegisterStdInCmd('@Modbus.Refuse',  '');
  InitModbusLrcCrc;
 end;
 //
 // Finalize NetModbus.
 //
 procedure FreeNetModbus;
 begin
 end;
 //
 // Poll NetModbus.
 //
 procedure PollNetModbus;
 begin
 end;
 
